/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "gridloc.h"
#include "shared.h"

/*--------------------------------------------------------------------*/

/*  Position_to_Grid_Locator()
 *
 *  Function to calculate a grid locator give lon/lat
 */

  void
Position_to_Grid_Locator( double lon, double lat, char *grid_loc )
{
  double temp;

  /* Adjust longitude and latitude to references for the   */
  /* Maidenhead grid locator system and roundup to 1/2 sec */
  lon += 180.0 + 1.3888888888E-4;
  lat +=  90.0 + 1.3888888888E-4;

  /* Calculate first letter of field */
  temp = lon / 20.0;
  grid_loc[0] = (char)((int)temp + 'A');

  /* Calculate first number of square */
  lon -= floor(temp) * 20.0;
  temp = lon / 2.0;
  grid_loc[2] = (char)((int)temp + '0');

  /* Calculate first letter of sub-square */
  lon -= floor(temp) * 2.0;
  temp = lon * 12.0;
  grid_loc[4] = (char)((int)temp + 'A');

  /* Calculate second letter of field */
  temp = lat / 10.0;
  grid_loc[1] = (char)((int)temp + 'A');

  /* Calculate second number of square */
  lat -= floor(temp) * 10.0;
  temp = lat / 1.0;
  grid_loc[3] = (char)((int)temp + '0');

  /* Calculate second letter of sub-square */
  lat -= floor(temp) * 1.0;
  temp = lat * 24.0;
  grid_loc[5] = (char)((int)temp + 'A');

  grid_loc[6] = '\0';

} /* End of Position_to_Grid_Locator() */

/*--------------------------------------------------------------------*/

/*  Grid_Locator_to_Position()
 *
 *  Function to calculate longitude/latitude
 *  given a position's grid locator
 */

  void
Grid_Locator_to_Position( char *grid_loc, double *lon, double *lat )
{
  *lon  = (grid_loc[0] - 'A') * 20.0;
  *lon += (grid_loc[2] - '0') * 2.0;
  *lon += (grid_loc[4] - 'A') / 12.0;
  *lon -= 180.0;

  *lat  = (grid_loc[1] - 'A') * 10.0;
  *lat += (grid_loc[3] - '0') * 1.0;
  *lat += (grid_loc[5] - 'A') / 24.0;
  *lat -= 90.0;

} /* End of Grid_Locator_to_Position() */

/*--------------------------------------------------------------------*/

/*  Locator_Valid()
 *
 *  Validates a grid locator
 */

  gboolean
Locator_Valid( char *locator )
{
  int len, /* Length of user-entered data  */
	  idx;

  /* Ignore blank fields */
  len = (int)strlen( locator );
  if( len == 0 )
	return( TRUE );

  /* Capitalize letters */
  for( idx = 0; idx < len; idx++ )
  {
	if( (locator[idx] > 0x60) && (locator[idx] < 0x7b) )
	  locator[idx] -= 0x20;

	/* Validate grid locator */
	switch( idx )
	{
	  case 0: case 1: /* First letters */
		if( (locator[idx] < 'A') || (locator[idx] > 'S') )
		  return( /* FALSE */TRUE );
		break;

	  case 2: case 3: /* Middle numbers */
		if( (locator[idx] < '0') || (locator[idx] > '9') )
		  return( FALSE );
		break;

	  case 4: case 5: /* Last letters */
		if( (locator[idx] < 'A') || (locator[idx] > 'X') )
		  return( FALSE );

	} /* switch( len ) */

  } /* for( idx = 0; idx < len; idx++ ) */

  return( len == 6 );

} /* End of Locator_Valid()*/

/*--------------------------------------------------------------------*/

