/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "earthtrack.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Arc_Cos()
 *
 *  This function implements the arccosine function,
 *  returning a value between 0 and 2xpi.

  static double
Arc_Cos( double x, double y )
{

  double
	result = 0.0,
		   fraction;

  fraction = x / y;

  if( (x > 0.0) && (y > 0.0) )
	result = acos( fraction );

  if( (x < 0.0) && (y > 0.0) )
	result = acos(fraction);

  if( (x < 0.0) && (y < 0.0) )
	result = pi + acos(fraction);

  if( (x > 0.0) && (y < 0.0) )
	result = pi + acos( fraction );

  return result;

} End of Arc_Cos() */

/*------------------------------------------------------------------------*/

/*  Visibility_to_Color()
 *
 *  This function generates a character string based on the
 *  visibility information returned by xsatcom.  It is used
 *  to set the colors of the markers and range circles
 *  plotted by xearth and xplanet.
 */
#define COLOR_SIZE	7

  static void
Visibility_to_Color( satellite_status_t *sat_status, char *color )
{
  /* strlen+1 is needed to allow null termination */
  if( isSatFlagClear(sat_status, SAT_ECLIPSED) )
	Strlcpy( color, "cyan", COLOR_SIZE );
  else
	Strlcpy( color, "blue", COLOR_SIZE );

  if( sat_status->elev > 0.0 )
	Strlcpy( color, "yellow", COLOR_SIZE );

} /* End of Visibility_to_Color() */

/*------------------------------------------------------------------------*/

/*  Range_Lat_Lon()
 *
 *  Calculates lat/long co-ordinates for a given
 *  azimuth angle in the satellite's range circle

  static void
Range_Lat_Lon(
	satellite_status_t *sat_status,
	int azim,
	double *range_lat,
	double *range_lon )
{
  double
	ssp_lat,   Sub-satellite point latitude
	ssp_lon,   Sub-satellite point longitude
	beta,      Footprint factor in calculations
	dazim,     Azimuth angle in range circle
	denom,
	numer;

  Convert to radians
  ssp_lat = Radians( sat_status->ssplat );
  ssp_lon = Radians( sat_status->ssplon );
  beta = (0.5 * sat_status->ftprnt) / rc_data.radiusearthkm;

  dazim = Radians( (double)azim );
  *range_lat = asin( sin(ssp_lat) * cos(beta) +
	  cos(dazim)  * sin(beta) * cos(ssp_lat) );
  numer = cos(beta) - ( sin(ssp_lat) * sin(*range_lat) );
  denom = cos(ssp_lat) * cos(*range_lat);

  if( azim == 0 && (beta > pio2 - ssp_lat) )
	*range_lon = ssp_lon + pi;
  else if( azim == 180 && (beta > pio2 + ssp_lat) )
	*range_lon = ssp_lon + pi;
  else if( fabs(numer / denom) > 1.0 )
	*range_lon = ssp_lon;
  else
  {
	if( (180 - azim) >= 0)
	  *range_lon = ssp_lon - Arc_Cos( numer, denom );
	else
	  *range_lon = ssp_lon + Arc_Cos( numer, denom );
  }

  while( *range_lon < 0.0 )
	*range_lon += twopi;

  while( *range_lon > twopi )
	*range_lon -= twopi;

  *range_lat = Degrees( *range_lat );
  *range_lon = Degrees( *range_lon );

} End of Range_Lat_Lon() */

/*------------------------------------------------------------------------*/

/*  Display_Satellite
 *
 *  Displays satellite tracks using xplanet
 */

  int
Display_Satellite( satellite_status_t *sat_status )
{
  char new_color[COLOR_SIZE];	/* Color string for xplanet */
  static char old_color[COLOR_SIZE] = "";
  FILE *sats_fd;

  /* Determine color for xplanet to use */
  Visibility_to_Color( sat_status, new_color );

  /* Don't change display if satellite status has not changed */
  if( strcmp(old_color, new_color) == 0 )
	return( 0 );

  /* Open xplanet satellites file descriptor */
  sats_fd = fopen( rc_data.xplanet_satfile, "w" );
  if( sats_fd == NULL )
  {
	perror( "xplanet satellites file" );
	ClearFlag( XPLANET_PLOT );
	ClearFlag( MULTISAT_PLOT );
	Error_Dialog( _("Failed to open xplanet satellites file"), FATAL );
	return(-1);
  }

  /* Enable plotting of all sats in xsatcom.sat */
  if( isFlagSet(MULTISAT_PLOT) )
  {
	int idx;

	/* Enter all satellites to xplanet satellite file */
	for( idx = 0; idx < num_sets; idx++ )
	{
	  /* Don't include current satellite */
	  if( (idx != set_idx) || isFlagClear(XPLANET_PLOT) )
		fprintf( sats_fd,
			"%d color=lightgrey altcirc=0 trail={orbit,-5,10,2}\n",
			element_sets[idx].satnum );
	}

  } /* if( isFlagSet(MULTISAT_PLOT) ) */

  /* Enter current satellite */
  if( isFlagSet(XPLANET_PLOT) )
  {
	fprintf( sats_fd,
		"%d color=%s altcirc=0 trail={orbit,-5,10,2}\n",
		element_sets[set_idx].satnum, new_color );
  }

  fclose( sats_fd );

  /* Start xplanet */
  if( xplanet == XPLANET_STOPPED )
  {
	int error = 0;

	error = system( rc_data.xplanet_command );

	/* On error delete xplanet's files */
	if( error )
	{
	  unlink( rc_data.xplanet_gcarcs );
	  unlink( rc_data.xplanet_marker );
	  unlink( rc_data.xplanet_config );
	  unlink( rc_data.xplanet_satfile );
	  xplanet = XPLANET_STOPPED;
	}
	else xplanet = XPLANET_RUNNING;
  }

  return( 0 );

} /* End of Display_Satellite() */

/*------------------------------------------------------------------------*/

/* Config_Xplanet()
 *
 * Writes an xplanet config file for xsatcom
 */

  int
Config_Xplanet( observer_status_t *obs_status )
{
  /* xplanet file descriptors */
  FILE *config_fd, *marker_fd;

  if( isFlagClear(XPLANET_PLOT) && isFlagClear(MULTISAT_PLOT) )
	return(0);

  /* Open xplanet marker file descriptor */
  marker_fd = fopen( rc_data.xplanet_marker, "w" );
  if( marker_fd == NULL )
  {
	perror( "xplanet marker file" );
	ClearFlag( XPLANET_PLOT );
	ClearFlag( MULTISAT_PLOT );
	Error_Dialog( _("Failed to open xplanet marker file"), NONFATAL );
	return( -1 );
  }

  /* Enter observer position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s\"\n",
	  Degrees(obs_status->obs_geodetic.lat),
	  Degrees(obs_status->obs_geodetic.lon),
	  obs_status->loc_name );
  fclose( marker_fd );

  /* Open xplanet config file descriptor */
  config_fd = fopen( rc_data.xplanet_config, "w" );
  if( config_fd == NULL )
  {
	perror( "xplanet config file" );
	ClearFlag( XPLANET_PLOT );
	ClearFlag( MULTISAT_PLOT );
	Error_Dialog( _("Failed to open xplanet config file"), FATAL );
	return(-1);
  }

  /* Write the xplanet config file */
  fprintf( config_fd,
	  "# xplanet config file produced by xsatom\n"
	  "shade=30\n"
	  "text_color={255,0,0}\n"
	  "twilight=6\n"
	  "marker_file=%s\n"
	  "satellite_file=%s",
	  rc_data.xplanet_marker,
	  rc_data.xplanet_satfile );
  fclose( config_fd );

  return( 0 );

} /* Config_Xplanet() */

/*------------------------------------------------------------------------*/

