/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "main.h"
#include "shared.h"

/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;


  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Process command line options */
  while( (option = getopt(argc, argv, "hv") ) != -1 )
    switch( option )
    {
      case 'h' : /* Print usage and exit */
        Usage();
        return(0);

      case 'v' : /* Print version */
        puts( PACKAGE_STRING );
        return(0);

      default: /* Print usage and exit */
        Usage();
        exit(-1);

    } /* End of switch( option ) */

  /* Init GTK */
  gtk_init (&argc, &argv);

  /* Create main window and menu */
  Strlcpy( rc_data.xsatcom_glade,
      getenv("HOME"), sizeof(rc_data.xsatcom_glade) );
  Strlcat( rc_data.xsatcom_glade,
      "/.xsatcom/xsatcom.glade", sizeof(rc_data.xsatcom_glade) );

  /* Check for the Glade config file */
  FILE *fp = fopen( rc_data.xsatcom_glade, "r" );
  if( fp == NULL )
  {
    fprintf( stderr, "xsatcom: cannot open xsatcom Glade GUI Description file.\n" );
    perror( rc_data.xsatcom_glade );
    fprintf( stderr, "xsatcom: trying to create xsatcom config directory "
                     "from the installation prefix file tree.\n" );

    /* Find the binary's path (location in file system) */
    char exe_path[256], file_path[288];
    
    /* Read the file path to xsatcom executable */
    size_t len = sizeof( exe_path );
    int bytes = (int)readlink( "/proc/self/exe", exe_path, len );
    if( bytes <= 0 )
    {
      fprintf( stderr, "xsatcom: cannot read xsatcom binary's location.\n" );
      perror( "/proc/self/exe" );
      exit( -1 );
    }

    /* Remove "/bin/xsatcom" from the path with room for termination */
    bytes -= sizeof( "/bin/xsatcom" ) - 1;
    if( bytes < 1 )
    {
      fprintf( stderr, "xsatcom: cannot create file path to examples/xsatcom.\n" );
      exit( -1 );
    }

    /* Create file path to xsatcom examples directory */
    exe_path[bytes] = '\0';
    Strlcpy( file_path, exe_path, sizeof(file_path) );
    Strlcat( file_path, "/share/examples/xsatcom", sizeof(file_path) );
    fprintf( stderr, "xsatcom: creating xsatcom config directory from: %s\n", file_path );

    /* Create system command to copy examples/xsatcom to ~/.xsatcom */
    char syscmnd[512];
    Strlcpy( syscmnd, "cp -r ", sizeof(syscmnd) );
    Strlcat( syscmnd, file_path, sizeof(syscmnd) );
    Strlcat( syscmnd, " ", sizeof(syscmnd) );
    Strlcat( syscmnd, getenv("HOME"), sizeof(syscmnd) );
    Strlcat( syscmnd, "/.xsatcom",   sizeof(syscmnd) );
    int ret = system( syscmnd );
    if( ret == -1 )
    {
      fprintf( stderr,"xsatcom: cannot create xsatcom's working directory.\n" );
      perror( file_path );
      exit( -1 );
    }
  } /* if( fp == NULL ) */
  else fclose( fp );

  satellite_track = create_satellite_track( &satellite_track_builder );
  gtk_window_set_title( GTK_WINDOW(satellite_track), PACKAGE_STRING );

  /* Set labels to CAT and Rotor Enable buttons */
  {
    GtkWidget *button;

    button = Builder_Get_Object( satellite_track_builder, "rotor_control" );
    rot_status = gtk_label_new (_("Enable Rotors"));
    gtk_widget_show (rot_status);
    gtk_container_add (GTK_CONTAINER (button), rot_status);
    gtk_label_set_justify (GTK_LABEL (rot_status), GTK_JUSTIFY_LEFT);
    g_object_set_data_full (G_OBJECT (satellite_track), "rot_status",
        g_object_ref (rot_status), (GDestroyNotify) g_object_unref);

    button = Builder_Get_Object( satellite_track_builder, "tcvr_cat" );
    cat_status = gtk_label_new (_("Enable CAT"));
    gtk_widget_show (cat_status);
    gtk_container_add (GTK_CONTAINER (button), cat_status);
    gtk_label_set_justify (GTK_LABEL (cat_status), GTK_JUSTIFY_LEFT);
    g_object_set_data_full (G_OBJECT (satellite_track), "cat_status",
        g_object_ref (cat_status), (GDestroyNotify) g_object_unref);

    button = Builder_Get_Object( satellite_track_builder, "doppler_corr" );
    dop_status = gtk_label_new (_("Doppler"));
    gtk_widget_show (dop_status);
    gtk_container_add (GTK_CONTAINER (button), dop_status);
    gtk_label_set_justify (GTK_LABEL (dop_status), GTK_JUSTIFY_LEFT);
    g_object_set_data_full (G_OBJECT (satellite_track), "dop_status",
        g_object_ref (dop_status), (GDestroyNotify) g_object_unref);

    button = Builder_Get_Object( satellite_track_builder, "txrx_track" );
    trk_status = gtk_label_new (_("Tx/Rx Track"));
    gtk_widget_show (trk_status);
    gtk_container_add (GTK_CONTAINER (button), trk_status);
    gtk_label_set_justify (GTK_LABEL (trk_status), GTK_JUSTIFY_LEFT);
    g_object_set_data_full (G_OBJECT (satellite_track), "trk_status",
        g_object_ref (trk_status), (GDestroyNotify) g_object_unref);
  }

  gtk_widget_show (satellite_track);

  /* Load observer location data from xsatcom.obs */
  g_idle_add( Load_Config, NULL );

  gtk_main ();

  return 0;
}

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */
static void sig_handler( int signal ) __attribute__ ((noreturn));
static void sig_handler( int signal )
{
  /* Cleanup before exit */
  Cleanup();

  /* Exit according to signal */
  fprintf( stderr, "\n" );
  switch( signal )
  {
    case SIGINT :
      fprintf( stderr, "%s\n",  _("xsatcom: Exiting via User Interrupt") );
      break;

    case SIGSEGV :
      fprintf( stderr, "%s\n",  _("xsatcom: Segmentation Fault") );
      break;

    case SIGFPE :
      fprintf( stderr, "%s\n",  _("xsatcom: Floating Point Exception") );
      break;

    case SIGABRT :
      fprintf( stderr, "%s\n",  _("xsatcom: Abort Signal received") );
      break;

    case SIGTERM :
      fprintf( stderr, "%s\n",  _("xsatcom: Termination Request received") );
  }

  exit( signal );

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

