/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "input.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Convert_Satellite_Data()
 *
 *  Converts the strings in a raw two-line element
 *  set to their intended numerical values. Processes
 *  these values as needed by the SGP4 routines */

  static void
Convert_Satellite_Data( char *tle_set, element_set_t *elem_set )
{
  char buff[15];
  double temp;

  /** Decode Card 1 **/
  /* Satellite's catalogue number */
  Strlcpy( buff, &tle_set[2], 6 );
  elem_set->satnum = atoi(buff);

  /* International Designator for satellite */
  Strlcpy( elem_set->idesg, &tle_set[9], 9 );
  if( elem_set->idesg[7] == ' ' ) elem_set->idesg[7] = '\0';
  if( elem_set->idesg[6] == ' ' ) elem_set->idesg[6] = '\0';

  /* Satellite's epoch */
  Strlcpy( buff, &tle_set[18], 15 );
  elem_set->jdsatepoch = Atof(buff);

  /* Satellite's First Time Derivative */
  Strlcpy( buff, &tle_set[33], 11 );
  elem_set->ndot = Atof(buff);

  /* Satellite's Second Time Derivative */
  Strlcpy( buff, &tle_set[44], 2 );
  buff[1] = '.';
  Strlcpy( &buff[2], &tle_set[45], 6 );
  buff[7] = 'E';
  Strlcpy( &buff[8], &tle_set[50], 3 );
  elem_set->nddot = Atof(buff);

  /* Satellite's bstar drag term */
  Strlcpy( buff, &tle_set[53], 2 );
  buff[1] = '.';
  Strlcpy( &buff[2], &tle_set[54], 6 );
  buff[7] = 'E';
  Strlcpy( &buff[8], &tle_set[59], 3 );
  elem_set->bstar = Atof(buff);

  /* Element Number */
  Strlcpy( buff, &tle_set[64], 5 );
  elem_set->elset = atoi(buff);

  /** Decode Card 2 **/
  /* Satellite's Orbital Inclination (degrees) */
  Strlcpy( buff, &tle_set[77], 9 );
  elem_set->inclo = Atof(buff);

  /* Satellite's RAAN (degrees) */
  Strlcpy( buff, &tle_set[86], 9 );
  elem_set->nodeo = Atof(buff);

  /* Satellite's Orbital Eccentricity */
  buff[0] = '.';
  Strlcpy( &buff[1], &tle_set[95], 8 );
  elem_set->ecco = Atof(buff);

  /* Satellite's Argument of Perigee (degrees) */
  Strlcpy( buff, &tle_set[103], 9 );
  elem_set->argpo = Atof(buff);

  /* Satellite's Mean Anomaly of Orbit (degrees) */
  Strlcpy( buff, &tle_set[112], 9 );
  elem_set->mo = Atof(buff);

  /* Satellite's Mean Motion (rev/day) */
  Strlcpy( buff, &tle_set[121], 11 );
  elem_set->no = Atof(buff);

  /* Satellite's Revolution number at epoch */
  Strlcpy( buff, &tle_set[132], 6 );
  elem_set->revnum = Atof(buff);

  /* Preprocess tle set */
  elem_set->nodeo = Radians( elem_set->nodeo );
  elem_set->argpo = Radians( elem_set->argpo );
  elem_set->mo    = Radians( elem_set->mo );
  elem_set->inclo = Radians( elem_set->inclo );

  temp = twopi/xmnpda/xmnpda;

  elem_set->no    *= temp*xmnpda;
  elem_set->ndot  *= temp;
  elem_set->nddot *= temp/xmnpda;

  // ---------------- initialize the orbit at sgp4epoch -------------------
  sgp4init( elem_set->satnum, elem_set->jdsatepoch-2433281.5, elem_set );

} /* Convert_Satellite_Data */

/*------------------------------------------------------------------*/

/*  Checksum_Good()
 *
 *  Calculates the checksum mod 10 of a line from a TLE set and
 *  returns 1 if it compares with checksum in column 68, else 0.
 *  tle_set is a character string holding the two lines read
 *  from a text file containing NASA format Keplerian elements.
 */

  static int
Checksum_Good( char *tle_set )
{
  int i, check_digit, value, checksum = 0;

  for(i = 0; i < 68; i++)
  {
    if( (tle_set[i] >= '0') && (tle_set[i] <= '9') )
      value = tle_set[i] - '0';
    else if( tle_set[i] == '-' )
      value = 1;
    else
      value = 0;

    checksum += value;

  } /* End for(i = 0; i < 68; i++) */

  checksum %= 10;
  check_digit = tle_set[68] - '0';

  return( checksum == check_digit );
} /* Function Checksums_Good */

/*------------------------------------------------------------------*/

/*  Good_Elements()
 *
 *  Carries out various checks on a TLE set to verify its validity.
 *  tle_set is a character string holding the two lines read
 *  from a text file containing NASA format Keplerian elements.
 */

  static gboolean
Good_Elements( char *tle_set )
{
  /* Verify checksum of both lines of a TLE set */
  if( !Checksum_Good(&tle_set[0]) || !Checksum_Good(&tle_set[69]) )
    return( FALSE );

  /* Check the line number of each line */
  if( (tle_set[0] != '1') || (tle_set[69] != '2') )
    return( FALSE );

  /* Verify that Satellite Number is same in both lines */
  if( strncmp(&tle_set[2], &tle_set[71], 5) != 0 )
    return( FALSE );

  /* Check that various elements are in the right place */
  if( ((tle_set[ 23] != '.') && (tle_set[ 23] != ',')) ||
      ((tle_set[ 34] != '.') && (tle_set[ 34] != ',')) ||
      ((tle_set[ 80] != '.') && (tle_set[ 80] != ',')) ||
      ((tle_set[ 89] != '.') && (tle_set[ 89] != ',')) ||
      ((tle_set[106] != '.') && (tle_set[106] != ',')) ||
      ((tle_set[115] != '.') && (tle_set[115] != ',')) ||
      ((tle_set[123] != '.') && (tle_set[123] != ',')) ||
      ((strncmp(&tle_set[61], " 0 ", 3) != 0)) )
    return( FALSE );

  return( TRUE );

} /* Function Good_Elements */

/*------------------------------------------------------------------*/

/*  Load_Database_Tle_Data()
 *
 *  Loads the TLE sets of all satellites in the xsatcom.sat
 *  database file into buffer, checking for validity.
 */
  char *
Load_Database_Tle_Data( void )
{
  int
    chr,       /* Character read from file    */
    tle_ret,   /* xsatcom.tle file read return*/
    sat_ret,   /* xsatcom.sat file read return*/
    dup_flg,   /* Duplicate entry flag        */
    tle_lines, /* Number of lines in TLE file */
    num_ld,    /* Number of TLE sets loaded   */
    num_sats,  /* Number of satellites in use */
    icp,       /* Index for comparison loops  */
    idxa,      /* Index for loops etc         */
    idxb;      /* Index for loops etc         */

  char
    home_dir[25],  /* Home directory name string */
    tle_fpath[64], /* File path to xsatcom.tle   */
    sat_fpath[64], /* File path to xsatcom.sat   */
    tle_set[139],  /* Two lines of a TLE set     */
    line[81],      /* Line buff for Load_Line()  */
    name[NAME_SIZE];  /* Satellite name buffer   */

  FILE
    *tle_file,  /* Keplerian TLE database */
    *sat_file;  /* Satellites database    */


  /* Setup 'home' directory path */
  Strlcpy( home_dir, getenv("HOME"), sizeof(home_dir) );

  /* Setup file path to xsatcom.tle (~/.xsatcom/xsatcom.tle) */
  snprintf( tle_fpath, sizeof(tle_fpath),
      "%s/.xsatcom/xsatcom.tle", home_dir );

  /* Open TLE database file */
  tle_file = fopen( tle_fpath, "r" );
  if( tle_file == NULL )
    return( Abort_On_Error(-17) );

  /* Count number of lines in xsatcom.tle */
  tle_lines = 0;
  while( (chr = fgetc(tle_file)) != EOF )
    if( chr == LF ) tle_lines++;

  /* Check that number of lines is %3 and not 0 */
  if( (tle_lines % 3) || (tle_lines == 0) )
  {
    fclose( tle_file );
    return( Abort_On_Error(-2) );
  }

  /* Setup file path to xsatcom.sat (~/.xsatcom/xsatcom.sat) */
  snprintf( sat_fpath, sizeof(sat_fpath),
      "%s/.xsatcom/xsatcom.sat", home_dir );

  /* Open satellite database file */
  sat_file = fopen( sat_fpath, "r" );
  if( sat_file == NULL )
  {
    fclose( tle_file );
    return( Abort_On_Error(-18) );
  }

  /*** Load TLE data from xsatcom.tle for sat names in xsatcom.sat ***/

  /* Count number of satellites in xsatcom.sat */
  num_sats = 0;
  do
  {
    sat_ret = Load_Line( line, sat_file, 80, "xsatcom.sat", FALSE );
    if( sat_ret == ERROR )
    {
      fclose( tle_file );
      return( Abort_On_Error(-1) );
    }

    icp = 1;
    if( line[0] == '[' )
    {
      while( (line[icp++] != ']') && (icp < 81) );
      if( icp < 81 ) num_sats++;
    }
  }
  while( sat_ret != EOF_NORM );

  /* Abort if no satellite names found */
  if( num_sats == 0 )
  {
    fclose( tle_file );
    fclose( sat_file );
    return( Abort_On_Error(-12) );
  }

  /* Allocate memory to TLE database buffer */
  mem_realloc( (void **)&element_sets,
      sizeof(element_set_t) * (size_t)num_sats,
      _("for element_sets in Load_Database_Tle_Data()") );
  if( element_sets == NULL )
  {
    fclose( tle_file );
    return( "!OK" );
  }

  /* Look for a satellite name in xsatcom.sat */
  /* and load its TLE set from xsatcom.tle    */
  rewind( sat_file );
  num_ld  = 0;
  tle_ret = 0;
  for( idxa = 0; idxa < num_sats; idxa++ )
  {
    do
    {
      /* Read a line, break at EOF */
      sat_ret = Load_Line( line, sat_file, 80, "xsatcom.sat", FALSE );
      if( sat_ret == ERROR )
      {
        fclose( tle_file );
        return( Abort_On_Error(-1) );
      }

      /* Look for a satellite name in [...] */
      icp = 1;
      name[0] = '\0';
      if( line[0] == '[' )
      {
        while( (line[icp++] != ']') && (icp < 81) );

        /* Read satellite name */
        if( icp < 81 )
        {
          line[icp-1] = '\0'; /* Kill ']' */
          /* Truncate name to 24 chars + null */
          Strlcpy( name, &line[1], sizeof(name) );
        }
        else
        {
          fclose( tle_file );
          return( Abort_On_Error(-22) );
        }

      } /* End of if( line[0] == '[' ) */
    } /* End of do. Continue if ']' not found in line */
    while( ((icp == 1) || (icp == 81)) && (sat_ret != EOF_NORM) );

    /* Find a TLE entry for the satellite name  */
    /* that was loaded from the xsatcom.sat file */
    rewind(tle_file);
    for( idxb = 0; ( (idxb < tle_lines) && (tle_ret == SUCCESS) ); idxb++ )
    {
      tle_ret = Load_Line( line, tle_file, 80, "xsatcom.tle", TRUE );
      if( (tle_ret == EOF) || (tle_ret == ERROR) )
        return( Abort_On_Error(-1) );
      if( tle_ret == EOF_NORM )
      {
        fclose( tle_file );
        return( Abort_On_Error(-1) );
      }

      if( strncmp(line, name, strlen(name)) == 0 )
        break;
    }

    /* Abort if no matching name found by end of file */
    if( idxb == tle_lines )
    {
      static char error[110];
      size_t s = sizeof(error);

      fclose( tle_file );
      Strlcpy( error, name, s );
      Strlcat( error, ": ", s );
      Strlcat( error, Abort_On_Error(-13), s );
      return( error );
    }

    /* Read in first line of TLE set */
    tle_ret = Load_Line( tle_set, tle_file, 69, "xsatcom.tle", TRUE );
    if( (tle_ret == EOF) || (tle_ret == ERROR) )
      return( Abort_On_Error(-1) );
    if( tle_ret == EOF_NORM )
    {
      fclose( tle_file );
      return( Abort_On_Error(-1) );
    }

    /* Read in second line of TLE set */
    tle_ret = Load_Line( &tle_set[69], tle_file, 69, "xsatcom.tle", TRUE );
    if( (tle_ret == EOF) || (tle_ret == ERROR) )
      return( Abort_On_Error(-1) );
    if(  tle_ret == EOF_NORM )
    {
      fclose( tle_file );
      return( Abort_On_Error(-1) );
    }

    /* Check for duplicate TLE set before entering to buffer */
    dup_flg = 0;
    for( icp = 0; icp < num_ld; icp++ )
      if( strncmp(element_sets[icp].name,
            name, strlen(name)) == 0 )
        dup_flg = 1;

    /* Enter TLE set to buffer if not duplicate */
    if( !dup_flg )
    {
      /* Enter name in TLE structure */
      Strlcpy( element_sets[num_ld].name, name,
          sizeof(element_sets[num_ld].name) );

      /* Check TLE set and Abort -3 if not valid */
      if( !Good_Elements(tle_set) )
      {
        fclose( sat_file );
        fprintf( stderr,
            _("Invalid TLE set for satellite \"%s\"\n"),
            name );
        return( Abort_On_Error(-3) );
      }

      /* Convert the TLE set to orbital elements */
      Convert_Satellite_Data( tle_set, &element_sets[num_ld] );

      /* Convert satellite's epoch time to Julian  */
      element_sets[num_ld].jdsatepoch =
        Julian_Date_of_Epoch(element_sets[num_ld].jdsatepoch);

      num_ld++;

    } /* End of if( !dup_flg ) */

  } /* End of for( idxa = 0; idxa < num_sats; idxa++ ) */

  num_sets = num_ld;

  fclose( tle_file );
  fclose( sat_file );

  return( "OK!" );

} /* End of Load_Database_Tle_Data() */

/*------------------------------------------------------------------*/

/*  Load_Tle_Set()
 *
 *  Loads the TLE set of a given satellite
 */

  void
Load_Tle_Set( char *sat_name, element_set_t *sat_elem_set )
{
  /* index for loops etc */
  int idx;

  /* Clear satellite name string */
  Strlcpy( sat_elem_set->name, "", 2 );

  /* Eliminate ~ at end of name, if any */
  idx = (int)strlen( sat_name );
  if( idx > NAME_SIZE - 2 )
    sat_name[NAME_SIZE - 2] = '\0';

  /* Look for a matching satellite name in database */
  for( idx = 0; idx < num_sets; idx++ )
    if( strncmp(element_sets[idx].name,
          sat_name, strlen(sat_name)) == 0 )
    {
      /* Load TLE set on match */
      *sat_elem_set = element_sets[idx];
      set_idx = idx;
      break;
    }

  /* If no matching name, load 1st TLE set as default */
  if( strcmp(sat_elem_set->name, "") == 0 )
  {
    *sat_elem_set = element_sets[0];
    set_idx = 0;
  }

  Config_Xplanet( &observer_status );

} /* End of Load_Tle_Set() */

/*------------------------------------------------------------------*/

/*  Read_Transponder_Data()
 *
 *  Reads satellite's transponder modes including uplink
 *  and downlink frequencies, tcvr's corresponding
 *  transmit and receive frequencies and beacons
 */

  char *
Read_Transponder_Data( char *sat_name )
{
  char
    home_dir[25],  /* Home directory name string */
    sat_fpath[64], /* File path to xsatcom.sat    */
    line[81];      /* Line buffer for Load_Line  */

  int
    idx,  /* Index for loops */
    ret;  /* Load_Line() return value */

  /* Satellite database */
  FILE *sat_file;


  /* Allocate memory to transponder mode data buffer */
  if( satellite_mode == NULL )
    mem_alloc( (void **)&satellite_mode, sizeof(satellite_mode_t),
        _("for satellite_mode in Read_Transponder_Data()") );
  if( satellite_mode == NULL ) return( "!OK" );

  /* Setup 'home' directory path */
  Strlcpy( home_dir, getenv("HOME"), sizeof(home_dir) );

  /* Setup file path to xsatcom.tle (~/.xsatcom/xsatcom.tle) */
  snprintf( sat_fpath, sizeof(sat_fpath),
      "%s/.xsatcom/xsatcom.sat", home_dir );

  /* Open satellite database file */
  sat_file = fopen( sat_fpath, "r" );
  if( sat_file == NULL )
    return( Abort_On_Error(-18) );

  /* Limit sat name comparison to first */
  /* NAME_SIZE-2 characters to avoid '~'  */
  idx = (int)strlen( sat_name );
  if( idx > NAME_SIZE - 2 ) idx = NAME_SIZE - 2;

  /* Find the satellite's data block */
  do
  {
    /* Break if sat_name found in database */
    ret = Load_Line( line, sat_file, 80, "xsatcom.sat", FALSE );
    if( ret == ERROR ) return( Abort_On_Error(-1) );
    if( strncmp( &line[1], sat_name, (unsigned long)idx ) == 0 )
      break;
  }
  while( ret != EOF_NORM );
  num_modes = 0;

  /* Look for transponder mode and frequency data.  */
  /* Frequencies are specified in float MHz and are */
  /* rounded to Hz units as used by CAT control.    */
  /* Read next line in xsatcom.sat */
  ret = Load_Line( line, sat_file, 80, "xsatcom.sat", FALSE );
  if( ret == ERROR )
    return( Abort_On_Error(-1) );

  while( TRUE )
  {
    /* Clear satellite mode flags and name */
    satellite_mode[num_modes].flags = NO_TRANSPONDER;
    Strlcpy( satellite_mode[num_modes].mode_name,   "No Mode!",
        sizeof(satellite_mode[num_modes].mode_name) );
    satellite_mode[num_modes].tone_freq = 0;

    /* Look for a 'Mode' entry, return if none */
    if( strncmp(line, "Mode", 4) == 0 )
    {
      /* Remove leading spaces from mode name */
      int len = (int)strlen(line);
      for( idx = 4; idx < len; idx++)
        if( (line[idx] != ' ') ) break;
    }
    else
    {
      fclose( sat_file );
      return( "OK!" );
    }

    /* No mode name error */
    if( (strlen(&line[idx]) == 0) || (ret == EOF_NORM) )
    {
      fclose( sat_file );
      return( Abort_On_Error(-20) );
    }

    /* Enter mode name in buffer */
    if( idx > 70 ) idx = 70; /* Avoid line buf overrun */
    Strlcpy( satellite_mode[num_modes].mode_name, &line[idx],
        sizeof(satellite_mode[num_modes].mode_name) );

    /* Read satellite's transponder data if available */
    /* Read next line in xsatcom.sat */
    ret = Load_Line( line, sat_file, 80, "xsatcom.sat", FALSE );
    if( ret == ERROR )
      return( Abort_On_Error(-1) );

    /* Clear satellite mode flags and name */
    while( TRUE )
    {
      /* Abort if no Mode data entry due to EOF or "End" */
      if( ret == EOF_NORM )
      {
        if( satellite_mode[num_modes].flags == NO_TRANSPONDER )
        {
          fclose( sat_file );
          return( Abort_On_Error(-14) );
        }
        else
        {
          fclose( sat_file );
          return( "OK!" );
        }
      }

      /* Look for 'Uplink' entry, read freq and mode */
      if( strncmp(line, "Uplink", 6) == 0 )
      {
        /* Get Tx freq and Mode */
        satellite_mode[num_modes].uplink_freq =
          (int)( Atof(&line[6]) * 1.0E6 + 0.5E-6 );
        satellite_mode[num_modes].tx_mode = Read_Operating_Mode( line );
        satellite_mode[num_modes].flags |= UPLINK_DATA;
        if( !(satellite_mode[num_modes].flags & TXFREQ_DATA) )
          satellite_mode[num_modes].tx_freq =
            satellite_mode[num_modes].uplink_freq;

      } /* if(strcmp(line, "Transmit") == 0) */

      /* Look for CTCSS tone entry */
      if( strncmp(line, "Tone", 4) == 0 )
        satellite_mode[num_modes].tone_freq = (int)(10.0 * Atof(&line[4]));

      /* Look for separate Tx freq which may not be same as uplink freq */
      if( strncmp(line, "Transmit", 8) == 0 )
      {
        satellite_mode[num_modes].tx_freq =
          (int)( Atof(&line[8]) * 1.0E6 + 0.5E-6 );
        satellite_mode[num_modes].flags |= TXFREQ_DATA;
      }

      /* Look for a "Downlink" entry, read freq and mode */
      if( strncmp(line, "Downlink", 8) == 0 )
      {
        satellite_mode[num_modes].dnlink_freq =
          (int)( Atof(&line[8]) * 1.0E6 + 0.5E-6 );
        satellite_mode[num_modes].rx_mode =
          Read_Operating_Mode( line );
        satellite_mode[num_modes].flags |= DNLINK_DATA;
        if( !(satellite_mode[num_modes].flags & RXFREQ_DATA) )
          satellite_mode[num_modes].rx_freq =
            satellite_mode[num_modes].dnlink_freq;
      } /* if(strcmp(line, "Downlink") == 0) */

      /* Look for 'Receive' entry, read freq */
      if( strncmp(line, "Receive", 7) == 0 )
      {
        satellite_mode[num_modes].rx_freq =
          (int)( Atof(&line[7]) * 1.0E6 + 0.5E-6 );
        satellite_mode[num_modes].flags |= RXFREQ_DATA;
      }

      /* Look for a 'Passband' entry */
      if( strncmp(line, "Passband", 8) == 0 )
      {
        /* Look for passband type entry */
        if( strstr(line, "invert") != NULL )
          satellite_mode[num_modes].flags &= ~PASSBAND_NORM;
        else if( strstr(line, "normal") != NULL )
          satellite_mode[num_modes].flags |= PASSBAND_NORM;
        else
        {
          fclose( sat_file );
          return( Abort_On_Error(-15) );
        }
      } /* if( strcmp(line, "Passband") == 0 ) */
      else
        satellite_mode[num_modes].flags |= PASSBAND_NORM;

      /* Read beacon frequency */
      if( strncmp(line, "Beacon", 6) == 0 )
      {
        satellite_mode[num_modes].bcn_freq =
          (int)( Atof(&line[6]) * 1.0E6 + 0.5E-6 );
        satellite_mode[num_modes].bcn_mode =
          Read_Operating_Mode( line );
        if( !(satellite_mode[num_modes].flags & RXFREQ_DATA) )
          satellite_mode[num_modes].rx_freq =
            satellite_mode[num_modes].bcn_freq;
        satellite_mode[num_modes].flags |= BEACON_DATA;
      } /* if( strncmp(line, "Beacon", 6) == 0 ) */

      /* Break if no more Mode data (Found next Mode or Name) */
      ret = Load_Line(line, sat_file, 80, "xsatcom.sat", FALSE);
      if( ret == ERROR )
        return( Abort_On_Error(-11) );
      if( (strncmp(line, "Mode", 4) == 0) ||
          (line[0] == '[') ||
          (ret == EOF_NORM) )
        break;

    } /* while( TRUE ) */
    num_modes++;

    /* Next satellite name is found */
    if( (line[0] == '[') || (ret == EOF_NORM) )
      break;

    /* Reallocate modes buffer for more data */
    mem_realloc( (void **)&satellite_mode,
        (size_t)(num_modes+1) * sizeof(satellite_mode_t),
        _("for satellite_mode in Read_Transponder_Data()") );
    if( satellite_mode == NULL ) return( "!OK" );

  } /* End of while( TRUE ) */

  fclose( sat_file );
  return( "OK!" );

} /* End of Read_Transponder_Data() */

/*------------------------------------------------------------------*/

/*  Read_Operating_Mode()
 *
 *  Reads the uplink/downlink operating mode
 *  specified in the satellite database file
 */

  unsigned char
Read_Operating_Mode( char *line )
{
  int
    ln_idx,
    md_idx;

  size_t
    ln_len,
    md_len;

  /* Mode strings (LSB, USB etc) */
  char *mode_names[NUM_MODE_CODES+1] = { TCVR_MODE_NAMES };

  /* Go to end of line and move back to first non-white char (space or tab) */
  ln_idx = (int)strlen( line );
  while( ln_idx-- >= 0 )
    if( (line[ln_idx] != ' ') && (line[ln_idx] != HT) )
      break;
  line[ ln_idx+1 ] = '\0'; /* Terminate line at end of mode string */

  /* Move back to first white char (space or tab) */
  while( ln_idx-- >= 0 )
    if( (line[ln_idx] == ' ') || (line[ln_idx] == HT) )
      break;
  ln_idx++; /* Go to first char of Mode string */

  /* Look for a match of Mode string with a Mode name */
  for( md_idx = 0; md_idx < NUM_MODE_CODES; md_idx++ )
  {
    /* Compare string length first */
    ln_len = strlen( &line[ln_idx] );
    md_len = strlen( mode_names[md_idx] );
    if( (ln_len == md_len) &&
        (strncmp(&line[ln_idx], mode_names[md_idx], md_len) == 0) )
      return( (unsigned char)md_idx );
  }

  return( NO_TCVR_MODE );
} /* End of Read_Operating_Mode() */

/*------------------------------------------------------------------*/

/*  Aos_Possible()
 *
 *  This function returns a 1 if the satellite of interest
 *  can ever rise above the horizon of the ground station
 */

  static int
Aos_Possible( element_set_t *sat_elem_set, observer_status_t  *obs_status )
{
  double incl, sma, apogee;

  incl = sat_elem_set->inclo;
  if (incl >= pio2)
    incl = pi-incl;

  sma = 331.25*exp( log(1./sat_elem_set->no*twopi) * tothrd );
  apogee = sma*(1. + sat_elem_set->ecco) - rc_data.radiusearthkm;

  if( (acos(rc_data.radiusearthkm/(apogee+rc_data.radiusearthkm))+incl) >
      fabs(obs_status->obs_geodetic.lat) )
    return 1;
  else
    return 0;

} /* End of Aos_Possible() */

/*------------------------------------------------------------------*/

/*  Satellite_Decayed()
 *
 *  This function returns a 1 if it appears that the
 *  satellite of interest has decayed at time 'julian_utc'
 */

  static int
Satellite_Decayed( element_set_t *sat_elem_set, double julian_utc )
{
  double
    drag,
    test;

  drag = ( sat_elem_set->ndot == 0. ? 1.E-8 : sat_elem_set->ndot ) * xmnpda;
  test = sat_elem_set->jdsatepoch + ( (0.072722052-sat_elem_set->no)/(10.0*fabs(drag)) );

  if( test < julian_utc )
    return 1;
  else
    return 0;

} /* End of Satellite_Decayed() */

/*------------------------------------------------------------------*/

/*  Satellite_Geostationary()
 *
 *  This function returns a 1 if the satellite of
 *  interest appears to be in a geostationary orbit
 */

  static int
Satellite_Geostationary( element_set_t *sat_elem_set )
{
  if( fabs(sat_elem_set->no - 0.0043751) < 1.3E-6 )
    return 1;
  else
    return 0;

} /* End of Satellite_Geostationary() */

/*------------------------------------------------------------------*/

/*  Set_Acessibility_Flags()
 *
 *  Checks whether satellite can be accessed and sets flags
 */

  void
Set_Accessibility_Flags(
    element_set_t *sat_elem_set,
    observer_status_t  *obs_status,
    satellite_status_t *sat_status )
{
  struct tm
    utc,      /* Calendar date and time (UTC)   */
    localtm;  /* Calendar date and time (local) */

  double julian_utc;

  /* Get UTC calendar and convert to Julian */
  Calendar_Time_Now( &utc, &localtm );
  julian_utc = Julian_Date( &utc );

  if( Aos_Possible(sat_elem_set, obs_status) )
    Clear_SatFlag( sat_status, SAT_INACCESSIBLE );
  else
    Set_SatFlag( sat_status, SAT_INACCESSIBLE | NO_AOS_LOS );

  if( Satellite_Geostationary(sat_elem_set) )
    Set_SatFlag( sat_status, SAT_GEOSTATIONARY | NO_AOS_LOS );
  else
    Clear_SatFlag( sat_status, SAT_GEOSTATIONARY );

  if( Satellite_Decayed(sat_elem_set, julian_utc) )
    Set_SatFlag( sat_status, SAT_DECAYED | NO_AOS_LOS );
  else
    Clear_SatFlag( sat_status, SAT_DECAYED );

} /* End of Set_Acessibility_Flags() */

/*------------------------------------------------------------------*/

/*  New_Status()
 *
 *  Does the pre-processing needed after
 *  a change of satellite or observer
 */

  void
New_Status(
    gboolean new_sat,
    element_set_t *sat_elem_set,
    satellite_status_t *sat_status,
    observer_status_t  *obs_status )
{
  /* Julian UTC */
  double julian_utc;

  kinetics_t
    obs_kinetics,   /* Observer position/velocity vectors */
    solar_kinetics; /*    Solar position/velocity vectors */

  /* Do following on sat change only */
  if( new_sat )
  {
    /* Copy satellite name into status structure */
    Strlcpy( sat_status->name,
        sat_elem_set->name,
        sizeof(sat_status->name) );

    /* If name is longer than NAME_SIZE-2, make ~ the last char. */
    if( strlen(sat_elem_set->name) > NAME_SIZE - 2 )
    {
      sat_status->name[NAME_SIZE - 2] = '~';
      sat_status->name[NAME_SIZE - 1] = '\0';
    }

    /* Clear all satellite flags */
    Clear_SatFlag( sat_status, ALL_FLAGS );

    /* Set the rotors' tracking resolution according to orbit */
    if( sat_elem_set->method == 'n' )
    {
      Clear_SatFlag( sat_status, DEEP_SPACE );
      sat_status->resol = (char)rc_data.low_resol;
    }
    else
    {
      Set_SatFlag( sat_status, DEEP_SPACE );
      if( Satellite_Geostationary(sat_elem_set) )
        sat_status->resol = (char)rc_data.high_resol;
      else
        sat_status->resol = (char)rc_data.med_resol;
    }

    /* Clear elevation status, needed to */
    /* determine horizon crossing events */
    sat_status->elev = 0.;

    /* Check whether satellite can ever have an AOS/LOS */
    Set_Accessibility_Flags( sat_elem_set, obs_status, sat_status );

    return;

  } /* if( new_sat ) */

  /* Do following if there is an observer change */

  /* Get UTC calendar and convert to Julian */
  Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
  julian_utc = Julian_Date( &obs_status->utc );

  /* Calculate topocentric position of sun */
  Calculate_Solar_Position( julian_utc, &solar_kinetics );

  /* Calculate solar position rel to observer */
  Calculate_User_PosVel( julian_utc, obs_status, &obs_kinetics );
  Calculate_Observer(
      &solar_kinetics, obs_status,
      &obs_kinetics, &obs_status->solar_set );

  /* Calculate lunar azimuth and elevation */
  Calculate_Moon_Position( julian_utc, obs_status, &obs_kinetics );

  /* Clear elevation status, needed to */
  /* determine horizon crossing events */
  sat_status->elev = 0.;

  /* Check whether satellite can ever have an AOS/LOS */
  Set_Accessibility_Flags( sat_elem_set, obs_status, sat_status );

  Config_Xplanet( obs_status );

} /* End of New_Status() */

/*------------------------------------------------------------------*/

/*  Find_New_Satellite()
 *
 *  Scrolls up/down in the TLE buffer to find a new satellite.
 *  The new satellite status is then displayed in main screen
 */

  void
Find_New_Satellite(
    int which,
    element_set_t *sat_elem_set,
    satellite_status_t *sat_status,
    observer_status_t  *obs_status )
{
  switch( which ) /* How to search */
  {
    case FORWD_SATELLITE : /* Find next satellite */
      /* Input next TLE set from database */
      set_idx = ( ++set_idx == num_sets ? 0 : set_idx );
      *sat_elem_set = element_sets[set_idx];
      break;

    case BACK_SATELLITE : /* Find previous satellite */
      /* Input previous TLE set from database */
      set_idx = ( --set_idx < 0 ? num_sets-1 : set_idx );
      *sat_elem_set = element_sets[set_idx];
      break;

    case FIRST_SATELLITE : /* Find first satellite */
      /* Input a TLE set from database */
      *sat_elem_set = element_sets[0];
      set_idx = 0;
      break;

    case LAST_SATELLITE : /* Find last satellite */
      /* Input a TLE set from database */
      *sat_elem_set = element_sets[num_sets-1];
      set_idx = num_sets-1;

  } /* End of switch( which ) */

  /* Preprocess new satellite data */
  New_Status( TRUE, sat_elem_set, sat_status, obs_status );

} /* End of Find_New_Satellite() */

/*------------------------------------------------------------------*/

/*  Load_Observer_Data()
 *
 *  Loads a block of observer location data from xsatcom.obs file.
 */

  char *
Load_Observer_Data( void )
{
  int
    ret, /* Load_Line() return value */
    isr, /* Index used in searches   */
    idx; /* Index for various loops  */

  char
    home_dir[25],    /* Home directory name */
    obs_fpath[64],   /* File path to xsatcom.obs */
    name[NAME_SIZE], /* Location name buffer */
    line[81];        /* Line buff for Load_Line */

  /* To convert longitude to +- 180 deg. */
  double lon;

  FILE *obs_file;

  /* Setup 'home' directory path */
  Strlcpy( home_dir, getenv("HOME"), sizeof(home_dir) );

  /* Setup file path to xsatcom.tle (~/.xsatcom/xsatcom.tle) */
  snprintf( obs_fpath, sizeof(obs_fpath),
      "%s/.xsatcom/xsatcom.obs", home_dir );

  /* Open observer file */
  obs_file = fopen( obs_fpath, "r" );
  if( obs_file == NULL )
    return( Abort_On_Error(-19) );

  /* Count the number of location entries */
  num_obs = 0;

  /* Abort if EOF on first read from file */
  ret = Load_Line( line, obs_file, 80, "xsatcom.obs", TRUE );
  if( (ret == EOF) || (ret == ERROR) )
    return( Abort_On_Error(-4) );

  /* Count number of locations */
  while( ret != EOF_NORM )
  {
    isr = 1;
    if( line[0] == '[' )
    {
      while( (line[isr++] != ']') && (isr < 81) );
      if( isr < 81 )
        num_obs++;
      else
      {
        fclose( obs_file );
        return( Abort_On_Error(-6) );
      }
    }

    ret = Load_Line(line, obs_file, 80, "xsatcom.obs", FALSE);
    if( ret == ERROR )
      return( Abort_On_Error(-4) );
  } /* End of while( eof != EOF ) */

  /* Abort if no location data found */
  if( num_obs == 0 )
  {
    fclose( obs_file );
    return( Abort_On_Error(-5) );
  }

  /* Allocate memory to observer database */
  mem_alloc( (void **)&observer_data,
      sizeof(observer_status_t) * (size_t)num_obs,
      _("for observer_data in Load_Observer_Data()") );
  if( observer_data == NULL ) return( "!OK" );

  /* Read a line from xsatcom.obs */
  rewind( obs_file );
  ret = Load_Line( line, obs_file, 80, "xsatcom.obs", FALSE );
  if( ret != SUCCESS )
    return( Abort_On_Error(-4) );

  /* Read observer data into buffer */
  for( idx = 0; idx < num_obs; idx++ )
  {
    /* Look for a data block name */
    do
    {
      isr = 1;
      /* Look for a data block name in [...] */
      if( line[0] == '[' )
      {
        while( (line[isr++] != ']') && (isr < 80) );

        /* Enter location name */
        if( isr < 80 )
        {
          line[isr-1] = '\0'; /* Kill ']' */

          /* If name > NAME_SIZE-1 char, make ~ the last char. */
          if( strlen(&line[1]) > NAME_SIZE - 1 )
          {
            line[NAME_SIZE - 2] = '~';
            line[NAME_SIZE - 1] = '\0';
          }
          Strlcpy( observer_data[idx].loc_name, &line[1],
              sizeof(observer_data[idx].loc_name) );

        } /* End of if( isr < 81 ) */

      } /* End of if( line[0] == '[' ) */

      /* Read a line, return at EOF */
      ret = Load_Line( line, obs_file, 80, "xsatcom.obs", FALSE );
      if( ret == EOF_NORM )
        break;
      if( ret != SUCCESS )
        return( Abort_On_Error(-1) );

    } /* End of do */
    /* Break if ']' found in line */
    while( (isr == 1) || (isr == 81) );

          ClearFlag( LOC_ENTERED | POSITION_ENTERED );

          /* Set default values for missing items */
          observer_data[idx].obs_geodetic.lon = 0.;
          observer_data[idx].obs_geodetic.lat = 0.;
          observer_data[idx].obs_geodetic.hgt = 0.0;
          observer_data[idx].grid_loc[0] = '\0';

          /* Read items from location data block till next block name */
          while( (line[0] != '[') )
          {
            if( strncmp(line, "lon", 3) == 0 ) /* Read lon. */
            {
              SetFlag( POSITION_ENTERED );
              observer_data[idx].obs_geodetic.lon =
                Radians( Atof(&line[3]) );
            }
            else if( strncmp(line, "lat", 3) == 0 ) /* Read lat. */
            {
              SetFlag( POSITION_ENTERED );
              observer_data[idx].obs_geodetic.lat =
                Radians( Atof(&line[3]) );
            }
            else if( strncmp(line, "height", 6) == 0 ) /* Read height */
              observer_data[idx].obs_geodetic.hgt =
                Atof(&line[6])/1000.0;
            else if( strncmp(line, "QTHloc", 6) == 0 ) /* Read QTH loc. */
            {
              SetFlag( LOC_ENTERED );
              isr = 6;

              /* Reject spaces after 'QTHloc' identifier */
              while( (line[isr] == ' ') && (isr < 79) )
                isr++;
              Strlcpy( observer_data[idx].grid_loc, &line[isr],
                  sizeof(observer_data[idx].grid_loc) );
            }

            /* Read a line, return at EOF */
            ret = Load_Line( line, obs_file, 80, "xsatcom.obs", FALSE );
            if( ret == EOF_NORM )
              break;
            if( ret != SUCCESS )
              return( Abort_On_Error(-1) );

          } /* End of while( line[0] != '[' ) */

          /* If Grid Locator and Position data are specified, */
          /* counter-check Grid Locator and exit on mismatch  */
          if( isFlagSet(LOC_ENTERED) &&
              isFlagSet(POSITION_ENTERED) )
          {
            lon = Degrees( observer_data[idx].obs_geodetic.lon );
            if( lon > 180 ) lon -= 360;

            Position_to_Grid_Locator(
                lon,
                Degrees( observer_data[idx].obs_geodetic.lat ),
                name );
            /* Abort if Grid Locator does not agree with position */
            if( strncmp( observer_data[idx].grid_loc, name, 6) != 0 )
            {
              fprintf( stderr,
                  _("xsatcom: Grid Locator mismatch with observer location: %s %s\n"),
                  observer_data[idx].grid_loc, name );
              return( Abort_On_Error(-7) );
            }
          }
          /* If only Grid Locator is specified, calculate Position */
          else if( isFlagSet(LOC_ENTERED) &&
              isFlagClear(POSITION_ENTERED) )
          {
            Grid_Locator_to_Position( observer_data[idx].grid_loc,
                &observer_data[idx].obs_geodetic.lon,
                &observer_data[idx].obs_geodetic.lat );
            observer_data[idx].obs_geodetic.lon =
              Radians(observer_data[idx].obs_geodetic.lon);
            observer_data[idx].obs_geodetic.lat =
              Radians(observer_data[idx].obs_geodetic.lat);
          }
          /* If only Position is specified, calculate Grid Locator */
          else if( isFlagSet(POSITION_ENTERED) &&
              isFlagClear(LOC_ENTERED) )
            Position_to_Grid_Locator( Degrees(observer_data[idx].obs_geodetic.lon),
                Degrees(observer_data[idx].obs_geodetic.lat),
                observer_data[idx].grid_loc );

          /* Do some sanity checking on data, abort on error */
          if( (fabs(observer_data[idx].obs_geodetic.lon) > twopi) ||
              (fabs(observer_data[idx].obs_geodetic.lat) >  pio2) ||
              (observer_data[idx].obs_geodetic.hgt > 15.0) ||
              (observer_data[idx].obs_geodetic.hgt < -0.5) )
            return( Abort_On_Error(-6) );

  } /* End of for( idx = 0; idx < num_obs; idx++ ) */

  fclose( obs_file );

  return( "OK!" );

} /* End of Load_Observer_Data() */

/*------------------------------------------------------------------*/

/* Load_Observer_Set()
 *
 *  Loads an observer's location data
 */

  void
Load_Observer_Set( char *loc_name, observer_status_t *obs_status )
{
  /* index for loops etc */
  int idx;

  /* Clear location name string */
  Strlcpy( obs_status->loc_name, "",
      sizeof(obs_status->loc_name) );

  /* Look for a matching observer name in database */
  for( idx = 0; idx < num_obs; idx++ )
  {
    if( strncmp(observer_data[idx].loc_name, loc_name, strlen(loc_name)) == 0 )
    {
      /* Load TLE set on match */
      *obs_status = observer_data[idx];
      obs_idx = idx;
      break;
    }
  }

  /* If no matching name, load 1st observer set as default */
  if( strcmp(obs_status->loc_name, "") == 0 )
  {
    *obs_status = observer_data[0];
    obs_idx = 0;
  }

} /* End of Load_Observer_Set() */

/*------------------------------------------------------------------*/

/*  Save_Location()
 *
 *  Saves user-entered location data in ~/.xsatcom/xsatcom.obs
 */

  void
Save_Location( observer_status_t *obs_status )
{
  char
    obs_fpath[64], /* Path to xsatcom.obs file */
    home_dir[25],  /* Path to home directory */
    conv[11];      /* Conversions buffer */
  size_t s = sizeof(conv);

  FILE *obs_file;

  /* Setup 'home' directory path */
  Strlcpy( home_dir, getenv("HOME"), sizeof(home_dir) );

  /* Setup file path to xsatcom.tle (~/.xsatcom/xsatcom.tle) */
  snprintf( obs_fpath, sizeof(obs_fpath),
      "%s/.xsatcom/xsatcom.obs", home_dir );

  /* Open observer file */
  obs_file = fopen( obs_fpath, "a" );
  if( obs_file == NULL )
  {
    Error_Dialog( Abort_On_Error(-4), FATAL );
    return;
  }

  fputs( "[", obs_file );
  fputs( (*obs_status).loc_name, obs_file );
  fputs( "]", obs_file );

  snprintf( conv, s, "%10.5f",
      Degrees(obs_status->obs_geodetic.lon) );
  fputs( "\nlon  ", obs_file );
  fputs( conv, obs_file );

  snprintf( conv, s, "%10.5f",
      Degrees(obs_status->obs_geodetic.lat) );
  fputs( "\nlat  ", obs_file );
  fputs( conv, obs_file );

  snprintf( conv, s, "%d",
      (int)(obs_status->obs_geodetic.hgt*1000.0+0.5) );
  fputs( "\nheight  ", obs_file );
  fputs( conv, obs_file );

  fputs( "\nQTHloc  ", obs_file );
  fputs( obs_status->grid_loc, obs_file );

  fclose( obs_file );

} /* End of Save_Location() */

/*------------------------------------------------------------------*/

