/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "gui.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Show_Observer_Status()
 *
 *  Function to display the observer's location,
 *  UTC time and solar/lunar azimuth/elevation
 */

  void
Show_Observer_Status(
    GtkBuilder *builder,
    int active_window,
    observer_status_t *obs_status )
{
  GtkEntry *txt_entry;

  char
    temp[16],
    ftime[13]; /* Formatted time string */

    /* Create the object id number as string */
    snprintf( temp, sizeof(temp), "obs_loc%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    gtk_entry_set_text( txt_entry, obs_status->loc_name );

    snprintf( temp, sizeof(temp), "obs_lon%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 10, "%+09.4f", Degrees(obs_status->obs_geodetic.lon) );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "obs_lat%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 9, "%+08.4f", Degrees(obs_status->obs_geodetic.lat) );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "obs_height%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 8, "%d", (int)(obs_status->obs_geodetic.hgt*1000.0+0.5) );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "date_loc%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    strftime( ftime, 13, "%d %b %Y", &obs_status->localtm );
    gtk_entry_set_text( txt_entry, ftime );

    snprintf( temp, sizeof(temp), "time_loc%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    strftime( ftime, 13, "%H:%M:%S", &obs_status->localtm );
    gtk_entry_set_text( txt_entry, ftime );

    snprintf( temp, sizeof(temp), "date_utc%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    strftime( ftime, 13, "%d %b %Y", &obs_status->utc );
    gtk_entry_set_text( txt_entry, ftime );

    snprintf( temp, sizeof(temp), "time_utc%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    strftime( ftime, 13, "%H:%M:%S", &obs_status->utc );
    gtk_entry_set_text( txt_entry, ftime );

    snprintf( temp, sizeof(temp), "sun_azel%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 16, "%06.3f/%+06.3f",
        Degrees(obs_status->solar_set.x),
        Degrees(obs_status->solar_set.y) );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "moon_range%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 8, "%d", (int)(obs_status->lunar_set.z+0.5) );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "moon_azim%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 8, "%06.3f", obs_status->lunar_set.x );
    gtk_entry_set_text( txt_entry, temp );

    snprintf( temp, sizeof(temp), "moon_elev%d", active_window );
    txt_entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
    snprintf( temp, 8, "%+06.3f", obs_status->lunar_set.y );
    gtk_entry_set_text( txt_entry, temp );

} /* End of Show_Observer_Status() */

/*------------------------------------------------------------------------*/

/*  Show_Satellite_Data()
 *
 *  Displays basic info of current satellite
 */

  void
Show_Satellite_Data(
    GtkBuilder *builder,
    int active_window,
    satellite_status_t *sat_status )
{
  GtkEntry *entry;
  GtkLabel *label;

  char temp[16];

  /* Enter satellite data */
  snprintf( temp, sizeof(temp), "sat_name%d", active_window );
  entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
  gtk_entry_set_text( entry, sat_status->name );

  snprintf( temp, sizeof(temp), "sat_azim%d", active_window );
  entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
  snprintf( temp, 7, "%06.2f", sat_status->azim );
  gtk_entry_set_text( entry, temp );

  snprintf( temp, sizeof(temp), "sat_elev%d", active_window );
  entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
  snprintf( temp, 7, "%+06.2f", sat_status->elev );
  gtk_entry_set_text( entry, temp );

  snprintf( temp, sizeof(temp), "sat_range%d", active_window );
  entry = GTK_ENTRY( Builder_Get_Object(builder, temp) );
  snprintf( temp, 8, "%d", (int)sat_status->range );
  gtk_entry_set_text( entry, temp );

  /* Set elevation label */
  snprintf( temp, sizeof(temp), "elev_label%d", active_window );
  label = GTK_LABEL( Builder_Get_Object(builder, temp) );
  if( sat_status->elev < 0. )
    gtk_label_set_markup( label, PASS_NOACCESS );
  else
    gtk_label_set_markup( label, PASS_ACCESS );

} /* Show_Satellite_Data() */

/*------------------------------------------------------------------------*/

/*  Show_Multiobserver_Status()
 *
 *  Screens multi-location observer status
 */

  void
Show_Multiobserver_Status(
    GtkTreeView *multiloc_treeview,
    GtkTreeStore *store,
    satellite_status_t *mxsat_status )
{
  /* Tree model iterators */
  GtkTreeIter iter1, iter2;
  static GtkTreePath *path = NULL;

  /* Index for loops etc */
  int idx;

  /* Formatted time and date strings */
  char
    fdat[12],
    futc[9],
    ftime[12],
    temp1[26],
    temp2[26],
    temp3[13],
    temp4[13];

    size_t
      s1 = sizeof( temp1 ),
         s2 = sizeof( temp2 ),
         s3 = sizeof( temp3 ),
         s4 = sizeof( temp4 );

    double
      julian_utc,   /* Julian UTC */
      home_aos,     /* Sat AOS for home location */
      rem_aos,      /* Sat AOS for remt location */
      overlap_from, /* AOS/LOS overlap from time */
      overlap_to;   /* AOS/LOS overlap to time   */

    /* Conversion between julian and struct tm time */
    struct tm date_time;


    /* Display multi-observer status */
    gtk_tree_model_get_iter_first( GTK_TREE_MODEL(store), &iter1 );
    path = gtk_tree_path_new_first();

    /* Do tree view */
    for( idx = 0; idx < num_obs; idx++ )
    {
      if( gtk_tree_view_row_expanded(multiloc_treeview, path) )
      {
        snprintf( temp1, 8, "%+07.2f", Degrees(observer_data[idx].obs_geodetic.lon) );
        snprintf( temp2, 7, "%+06.2f", Degrees(observer_data[idx].obs_geodetic.lat) );
        snprintf( temp3, 8, "%d",
            (int)(observer_data[idx].obs_geodetic.hgt * 1000.0 + 0.5));

        gtk_tree_model_iter_nth_child( GTK_TREE_MODEL(store), &iter2, &iter1, 0);
        gtk_tree_store_set(
            store, &iter2,
            DATA1_COLUMN, observer_data[idx].grid_loc,
            DATA2_COLUMN, temp1,
            DATA3_COLUMN, temp2,
            DATA4_COLUMN, temp3,
            -1 );

        strftime( fdat, 12, "%d/%b/%Y", &observer_data[0].utc );
        strftime( futc,  9, "%H:%M:%S", &observer_data[0].utc );
        snprintf( temp1, 12, "%05.1f/%+05.1f",
            Degrees(observer_data[idx].solar_set.x),
            Degrees(observer_data[idx].solar_set.y) );
        snprintf( temp2, 12, "%05.1f/%+05.1f",
            observer_data[idx].lunar_set.x,
            observer_data[idx].lunar_set.y );

        gtk_tree_model_iter_nth_child( GTK_TREE_MODEL(store), &iter2, &iter1, 1);
        gtk_tree_store_set
          ( store, &iter2,
            DATA1_COLUMN, fdat,
            DATA2_COLUMN, futc,
            DATA3_COLUMN, temp1,
            DATA4_COLUMN, temp2,
            -1 );

        snprintf( temp1, 7, "%06.2f",  mxsat_status[idx].azim );
        snprintf( temp2, 7, "%+06.2f", mxsat_status[idx].elev );
        snprintf( temp3, 8, "%d", (int)mxsat_status[idx].range );

        gtk_tree_model_iter_nth_child( GTK_TREE_MODEL(store), &iter2, &iter1, 2);
        gtk_tree_store_set(
            store, &iter2,
            DATA1_COLUMN, mxsat_status[0].name,
            DATA2_COLUMN, temp1,
            DATA3_COLUMN, temp2,
            DATA4_COLUMN, temp3,
            -1 );

        /* Display AOS/LOS according to relevant flags */
        if( isSatFlagSet(&mxsat_status[idx], SAT_GEOSTATIONARY) )
        {
          Strlcpy( temp1, "Geostatic", s1 );
          Strlcpy( temp2, "Geostatic", s2 );
        }
        else if( isSatFlagSet(&mxsat_status[idx], NO_AOS_LOS) )
        {
          Strlcpy( temp1, "No AOS", s1 );
          Strlcpy( temp2, "No LOS", s2 );
        }
        else
        {
          Date_Time( mxsat_status[idx].aos, &date_time );
          strftime( temp1, 12, "%b %d ", &date_time );
          strftime( ftime, 12, "%H:%M",  &date_time );
          Strlcat( temp1, ftime, s1 );

          Date_Time( mxsat_status[idx].los, &date_time );
          strftime( temp2, 12, "%b %d ", &date_time );
          strftime( ftime, 12, "%H:%M",  &date_time );
          Strlcat( temp2, ftime, s2 );
        }

        if( isSatFlagClear(&mxsat_status[idx], NO_AOS_LOS) )
        {
          julian_utc = Julian_Date( &observer_data[0].utc );

          /* If aos>los, then sat is visible and aos is set to utc */
          if( mxsat_status[0].aos >  mxsat_status[0].los )
            home_aos = julian_utc;
          else
            home_aos = mxsat_status[0].aos;

          if( mxsat_status[idx].aos >  mxsat_status[idx].los )
            rem_aos = julian_utc;
          else
            rem_aos = mxsat_status[idx].aos;

          /* Furthest aos and nearest los are    */
          /* the start/end of visibility overlap */
          if( home_aos >  rem_aos )
            overlap_from = home_aos;
          else
            overlap_from = rem_aos;

          if( mxsat_status[0].los < mxsat_status[idx].los )
            overlap_to = mxsat_status[0].los;
          else
            overlap_to = mxsat_status[idx].los;

          /* If start of overlap time nearer than end */
          /* then there is overlap, so display times  */
          if( overlap_from < overlap_to )
          {
            Date_Time( overlap_from, &date_time );
            strftime( temp3, 13, "%b %d %H:%M", &date_time );

            Date_Time( overlap_to, &date_time );
            strftime( temp4, 13, "%b %d %H:%M", &date_time );
          }
          else
          {
            Strlcpy( temp3, "No Overlap", s3 );
            Strlcpy( temp4, "No Overlap", s4 );
          }
        }
        else
        {
          Strlcpy( temp3, "No Overlap", s3 );
          Strlcpy( temp4, "No Overlap", s4 );
        }

        gtk_tree_model_iter_nth_child(
            GTK_TREE_MODEL(store), &iter2, &iter1, 3 );
        gtk_tree_store_set(
            store, &iter2,
            DATA1_COLUMN, temp1,
            DATA2_COLUMN, temp2,
            DATA3_COLUMN, temp3,
            DATA4_COLUMN, temp4,
            -1 );

      } /* if( gtk_tree_view_row_expanded(multiloc_treeview, path) ) */

      gtk_tree_model_iter_next( GTK_TREE_MODEL(store), &iter1 );
      gtk_tree_path_next( path );

    } /* End of for( idx = 1; idx < num_obs; idx++ ) */

    gtk_tree_path_free( path );

} /* End of Show_Multiobserver_Status() */

/*------------------------------------------------------------------------*/

/*  Show_Satellite_Status()
 *
 *  Function to display status data of a satellite
 */

  void
Show_Satellite_Status( satellite_status_t  *sat_status )
{
  /* Used for setting entries and labels */
  GtkEntry *txt_entry;
  GtkLabel *label;

  char temp[8];

  /* Formatted time string */
  char ftime[13];

  /* Conversion between julian and struct tm time */
  struct tm date_time;


  /* Show satellite name and tracking model */
  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_name") );
  gtk_entry_set_text( txt_entry, sat_status->name );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_model") );
  if( isSatFlagSet(sat_status, DEEP_SPACE) )
    gtk_entry_set_text( txt_entry, "NORAD: SDP4" );
  else
    gtk_entry_set_text( txt_entry, "NORAD: SGP4" );

  /* Check and implement AOS/LOS flags */
  if( isSatFlagSet(sat_status, NO_AOS_LOS) )
  {
    if( isSatFlagSet(sat_status, SAT_INACCESSIBLE) )
    {
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_aos") );
      gtk_entry_set_text( txt_entry, "No AOS/LOS" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_los") );
      gtk_entry_set_text( txt_entry, "No AOS/LOS" );
    }
    else if( isSatFlagSet(sat_status, SAT_DECAYED) )
    {
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_aos") );
      gtk_entry_set_text( txt_entry, "DECAYED?" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_los") );
      gtk_entry_set_text( txt_entry, "DECAYED?" );
    }
    else if( isSatFlagSet(sat_status, SAT_GEOSTATIONARY) )
    {
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_aos") );
      gtk_entry_set_text( txt_entry, "GEOSTATIC" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_los") );
      gtk_entry_set_text( txt_entry, "GEOSTATIC" );
    }

  } /* if( isSatFlagSet(sat_status, NO_AOS_LOS) ) */
  else
  {
    Date_Time( sat_status->aos, &date_time );
    strftime( ftime, 13, "%b %d %H:%M", &date_time );
    txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_aos") );
    gtk_entry_set_text( txt_entry, ftime );

    Date_Time( sat_status->los, &date_time );
    strftime( ftime, 13, "%b %d %H:%M", &date_time );
    txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_los") );
    gtk_entry_set_text( txt_entry, ftime );

  } /* if( isSatFlagSet(sat_status, NO_AOS_LOS) ) */

  /* Set illumination status */
  label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "illum_label") );
  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_eclst") );
  if( isSatFlagSet(sat_status, SAT_ECLIPSED) )
  {
    gtk_label_set_markup( label, SAT_ECLIPSE );
    gtk_entry_set_text( txt_entry, "Eclipsed" );
  }
  else
  {
    gtk_label_set_markup( label, SAT_ILLUMIN );
    gtk_entry_set_text( txt_entry, "In Sunlight" );
  }

  /* Set azim, elev, range, phase and doppler */
  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_azim") );
  snprintf( temp, 7, "%06.2f", sat_status->azim );
  gtk_entry_set_text( txt_entry, temp );

  label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "elev_label") );
  if( sat_status->elev < 0.0 )
    gtk_label_set_markup( label, SAT_NOACCESS );
  else
    gtk_label_set_markup( label, SAT_ACCESS );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_elev") );
  snprintf( temp, 7, "%+06.2f", sat_status->elev );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_range") );
  snprintf( temp, 8, "%d", (int)sat_status->range );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_phase") );
  snprintf( temp, 7, "%06.2f", sat_status->phs256 );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_doppler") );
  snprintf( temp, 7, "%0.3f", sat_status->doppler * 1.0E6 );
  gtk_entry_set_text( txt_entry, temp );

  /* Set long, latd, altd, footprint and rev num */
  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_lon") );
  snprintf( temp, 7, "%06.2f", sat_status->ssplon );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_lat") );
  snprintf( temp, 7, "%+06.2f", sat_status->ssplat );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_alt") );
  snprintf( temp, 8, "%d", (int)sat_status->alt );
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_ftprnt") );
  snprintf( temp, 8, "%d", (int)sat_status->ftprnt);
  gtk_entry_set_text( txt_entry, temp );

  txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "sat_revnum") );
  snprintf( temp, 8, "%d", (int)sat_status->revnum );
  gtk_entry_set_text( txt_entry, temp );

} /* End of Show_Satellite_Status() */

/*------------------------------------------------------------------------*/

/*  Show_Multisatellite_Status()
 *
 *  Displays muti-satellite orbital data and selects
 *  an individual satellite for single-tracking
 */

  void
Show_Multisatellite_Status(
    GtkTreeView *multisat_treeview,
    satellite_status_t *multisat_status )
{
  GtkListStore *store;
  GtkTreeIter iter;

  int
    rank, /* Satellite's rank in the display   */
    idx;  /* Index for loops etc  */

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Data for the tree view */
  char
    name[NAME_SIZE],
    azim[8],
    elev[8],
    range[8],
    phase[6],
    aos[21],
    los[21];

    gboolean access;


    /* Create list store of satellite data */
    store = gtk_list_store_new ( NLIST_COLUMNS,
        G_TYPE_STRING,
        G_TYPE_BOOLEAN,
        G_TYPE_STRING,
        G_TYPE_STRING,
        G_TYPE_STRING,
        G_TYPE_STRING,
        G_TYPE_STRING,
        G_TYPE_STRING );

    /* Display multi-satellite status */
    for( rank = 0; rank < num_sets; rank++ )
      for( idx = 0; idx < num_sets; idx++ )
      {
        if( rank == (multisat_status[idx].flags & SAT_RANK_FLAGS) )
        {
          Strlcpy( name, multisat_status[idx].name, sizeof(name) );
          access = multisat_status[idx].elev > 0.;
          snprintf(  azim, 8,  "%06.2f",
              multisat_status[idx].azim + 0.005 );
          snprintf(  elev, 8, "%+06.2f",
              multisat_status[idx].elev + 0.005 );
          snprintf( range, 8,      "%d",
              (int)(multisat_status[idx].range + 0.5) );
          snprintf( phase, 6,  "%05.1f",
              multisat_status[idx].phs256 + 0.05 );

          /* Don't display AOS/LOS if sat is inaccessible or geostationary */
          if( isSatFlagSet(&multisat_status[idx], SAT_GEOSTATIONARY) )
          {
            Strlcpy( aos, "  Geostatic ", sizeof(aos) );
            Strlcpy( los, "  Geostatic ", sizeof(los) );
          }
          else if( isSatFlagSet(&multisat_status[idx], NO_AOS_LOS) )
          {
            Strlcpy( aos, " No AOS/LOS ", sizeof(aos) );
            Strlcpy( los, " No AOS/LOS ", sizeof(los) );
          }
          else
          {
            Date_Time( multisat_status[idx].aos, &date_time );
            strftime( aos, 21, "%d/%b/%Y %H:%M:%S", &date_time );
            Date_Time( multisat_status[idx].los, &date_time );
            strftime( los, 21, "%d/%b/%Y %H:%M:%S", &date_time );

          } /*if(isSatFlagSet(&multisat_status[idx],SAT_GEOSTATIONARY))*/

          /* Acquire an iterator */
          gtk_list_store_append( store, &iter );

          gtk_list_store_set( store, &iter,
              NAME_COLUMN,   name,
              ACCESS_COLUMN, access,
              AZIM_COLUMN,   azim,
              ELEV_COLUMN,   elev,
              RANGE_COLUMN,  range,
              PHASE_COLUMN,  phase,
              AOS_COLUMN,    aos,
              LOS_COLUMN,    los,
              -1 );
          break;
        }

      } /* End of for( idx = 0; idx < num_sets; idx++ ) */

    gtk_tree_view_set_model( multisat_treeview, GTK_TREE_MODEL(store) );
    g_object_unref( store );

} /* End of Show_Multisatellite_Status() */

/*------------------------------------------------------------------------*/

/*  Show_Pass_Predict()
 *
 *  Displays satellite pass predictions
 */

  void
Show_Pass_Predict( cairo_t *cr,
    GtkBuilder *builder,
    satellite_status_t *sat_status,
    pass_data_t *pass_data, int pnum )
{
  /* Pass pred label */
  GtkLabel *label;

  /* Width and height of drawing area */
  gint width, height;

  /* Pango layout for plot data */
  PangoLayout *layout;

  /* Plotting variables */
  int
    tab_date, tab_4f, tab_5f, tab_plot, tab_1ch,
    plot_x, plot_y, tab_y, idx;

  /* For plotting graphs */
  GdkPoint
    *azim_points = NULL,
    *elev_points = NULL;

  /* Max and min values of azim/elev needed in plots */
  double
    max_elev,
    min_elev,
    max_azim,
    min_azim;

  /* Range of azim/elev data (max-min) */
  double
    rng_elev,
    rng_azim;

  /* Conversion between julian and struct tm time */
  struct tm date_time;
  char temp[64];


  /* Allocate plotting points */
  size_t mem = sizeof( GdkPoint ) * (unsigned long)pnum;
  mem_alloc( (void **)&azim_points, mem,
      _("for azim_points in Show_Pass_Predict()") );
  if( azim_points == NULL ) return;
  mem_alloc( (void **)&elev_points, mem,
      _("for elev_points in Show_Pass_Predict()") );
  if( elev_points == NULL ) return;

  /* Get size of drawing area */
  GtkAllocation alloc;
  gtk_widget_get_allocation( pass_drawingarea, &alloc );
  width  = alloc.width;
  height = alloc.height;

  /* Cairo context */
  cairo_set_source_rgb( cr, BLACK );
  cairo_rectangle( cr,
      0.0, 0.0,
      (gdouble)width,
      (gdouble)height );
  cairo_fill( cr );

  /* Create pango layout for date/time and get tab 1 size */
  layout = gtk_widget_create_pango_layout( pass_drawingarea, "" );
  pango_layout_set_markup( layout, "<span font=\"Mono\">00/00/00 00:00:00 </span>", -1 );
  pango_layout_get_pixel_size( layout, &tab_date, &tab_y );

  /* Create pango layout for numerical format (04.1f) and get tab_4f size */
  pango_layout_set_markup( layout, "<span font=\"Mono\">00.0 </span>", -1 );
  pango_layout_get_pixel_size( layout, &tab_4f, &tab_y );
  tab_1ch = tab_4f / 5; /* One char tab */

  /* Create pango layout for numerical format (05.1f) and get tab_5f size */
  pango_layout_set_markup( layout, "<span font=\"Mono\">000.0 </span>", -1 );
  pango_layout_get_pixel_size( layout, &tab_5f, &tab_y );

  /* Available width for azim and elev graphs */
  tab_plot = (width - tab_date - 3 * tab_5f - tab_4f) / 2;

  /* Get the needed widgets */
  label = GTK_LABEL( Builder_Get_Object(builder, "pass_pred_label") );

  /* Set pass predict viewer label */
  if( isSatFlagSet(sat_status, SAT_GEOSTATIONARY) )
    gtk_label_set_markup( label,
        "<span foreground=\"darkred\" >No AOS/LOS "
        "Predictions: Satellite Geostationary</span>" );
  else if( isSatFlagSet(sat_status, NO_AOS_LOS) )
    gtk_label_set_markup( label,
        "<span foreground=\"darkred\" >No AOS/LOS "
        "Predictions: Satellite Inaccessible</span>" );
  else if( isFlagSet(VISIBLE_PREDICT) )
    gtk_label_set_markup( label,
        "<span foreground=\"darkblue\" >"
        "Visible Pass Predictions</span>" );
  else
    gtk_label_set_markup( label,
        "<span foreground=\"darkgreen\" >"
        "Accessible Pass Predictions</span>" );

  /* If there is AOS/LOS data available */
  if( pnum != 0 )
  {
    /* Draw plot titles */
    pango_layout_set_markup( layout,
        "<span font=\"Mono\">  Date     UTC    Elevation Plot</span>", -1 );
    plot_x = plot_y = 4;
    cairo_set_source_rgb( cr, GREY );
    cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
    pango_cairo_show_layout( cr, layout );

    plot_x += tab_date + tab_4f + tab_plot;
    pango_layout_set_markup( layout, "<span font=\"Mono\">Azimuth Plot</span>", -1 );
    cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
    pango_cairo_show_layout( cr, layout );

    pango_layout_set_markup( layout, "<span font=\"Mono\">Long   Latd</span>", -1 );
    plot_x += tab_5f + tab_plot;
    cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
    pango_cairo_show_layout( cr, layout );

    /* Find max and min values of azim/elev */
    max_elev = min_elev = pass_data[0].elev;
    max_azim = min_azim = pass_data[0].azim;
    for( idx = 1; idx < pnum; idx++ )
    {
      if( pass_data[idx].elev > max_elev )
        max_elev = pass_data[idx].elev;
      if( pass_data[idx].elev < min_elev )
        min_elev = pass_data[idx].elev;

      if( pass_data[idx].azim > max_azim )
        max_azim = pass_data[idx].azim;
      if( pass_data[idx].azim < min_azim )
        min_azim = pass_data[idx].azim;
    }

    /* Calculate azim/elev range of values */
    rng_elev = max_elev - min_elev;
    rng_azim = max_azim - min_azim;

    /* Plot pass predicition */
    plot_y = tab_y + 4;
    for( idx = 0; idx < pnum; idx++ )
    {
      plot_x = 4;

      /* Display date/time */
      Date_Time( pass_data[idx].julian_utc, &date_time );
      strftime( temp, 45, "<span font=\"Mono\">%d/%m/%y %H:%M:%S</span>", &date_time );
      pango_layout_set_markup( layout, temp, -1 );
      cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
      pango_cairo_show_layout( cr, layout );

      /* Print elevation values */
      plot_x += tab_date;
      if( pass_data[idx].elev < 0.0 )
        pass_data[idx].elev = 0.0;
      snprintf( temp, 31, "<span font=\"Mono\">%04.1f</span>", pass_data[idx].elev );
      pango_layout_set_markup( layout, temp, -1 );
      cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
      pango_cairo_show_layout( cr, layout );

      /* Set graph points for elevation */
      plot_x += tab_4f;
      elev_points[idx].x = plot_x + (int)( (double)(tab_plot - tab_1ch)*
          (pass_data[idx].elev - min_elev) / rng_elev - 0.5 );
      elev_points[idx].y = plot_y + tab_y / 2;

      /* Print azimuth values */
      plot_x += tab_plot;
      snprintf( temp, 32, "<span font=\"Mono\">%05.1f</span>", pass_data[idx].azim );
      pango_layout_set_markup( layout, temp, -1 );
      cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
      pango_cairo_show_layout( cr, layout );

      /* Set graph points for azimuth */
      plot_x += tab_5f;
      azim_points[idx].x = plot_x + (int)( (double)(tab_plot - tab_1ch) *
          (pass_data[idx].azim-min_azim)/rng_azim - 0.5 );
      azim_points[idx].y = plot_y + tab_y / 2;

      /* Print phase, long, lat */
      plot_x += tab_plot;
      snprintf( temp, 32, "<span font=\"Mono\">%05.1f</span>", pass_data[idx].lon );
      pango_layout_set_markup( layout, temp, -1 );
      cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
      pango_cairo_show_layout( cr, layout );

      plot_x += tab_5f;
      snprintf( temp, 32, "<span font=\"Mono\">%+05.1f</span>", pass_data[idx].lat );
      pango_layout_set_markup( layout, temp, -1 );
      cairo_move_to( cr, (gdouble)plot_x, (gdouble)plot_y );
      pango_cairo_show_layout( cr, layout );

      /* Draw elev point according to sat status */
      if( isFlagSet(VISIBLE_PREDICT) )
      {
        if( pass_data[idx].flags & SAT_VISIBLE )
          cairo_set_source_rgb( cr, CYAN );
        else
          cairo_set_source_rgb( cr, BLUE );
      }
      else cairo_set_source_rgb( cr, GREEN );

      cairo_rectangle( cr,
          (gdouble)(elev_points[idx].x - 2),
          (gdouble)(elev_points[idx].y - 2),
          5.0, 5.0 );
      cairo_fill( cr );

      /* Draw azim point */
      cairo_set_source_rgb( cr, YELLOW );
      cairo_rectangle( cr,
          (gdouble)(azim_points[idx].x - 2),
          (gdouble)(azim_points[idx].y - 2),
          5.0, 5.0 );
      cairo_fill( cr );

      cairo_set_source_rgb( cr, GREY );
      plot_y += tab_y;

    } /* End of for( idx = 0; idx < num; idx++ ) */
  } /* End of if( num != 0) */

  /* Draw the graphs */
  Cairo_Draw_Lines( cr, elev_points, pnum );
  Cairo_Draw_Lines( cr, azim_points, pnum );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );

  g_object_unref( layout );
  free_ptr( (void **)&azim_points );
  free_ptr( (void **)&elev_points );

} /* End of Show_Pass_Predict() */

/*------------------------------------------------------------------------*/

/*  Show_Illumination_Predict()
 *
 *  Displays satellite pass predictions
 */

  void
Show_Illumination_Predict( cairo_t *cr, double *jdate, int *illum )
{
  /* Index for loops etc */
  int idx;

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Formatted time string */
  char title[51];

  /* Illumination values and day */
  double illm[30], day[30];

  /* Plot box rect */
  GdkRectangle rect;

  /* Draw plotting frame */
  Date_Time( jdate[0], &date_time );
  strftime( title, 51, "Illumination Predictions from %d/%m/%Y", &date_time );
  Draw_Plotting_Frame( cr, illum_drawingarea, &rect, title, 11, 30 );

  /* Prepare % illumination data */
  for( idx = 0; idx < 30; idx++ )
  {
    day[idx]  = (double)(idx+1);
    illm[idx] = (double)illum[idx]/14.4;
  } /* End of for( idx = 0; idx < 14; idx++ ) */

  /* Draw illumination graph */
  Draw_Graph( cr, illum_drawingarea, &rect, illm, day,
      100.0, 0.0, 1.0, 30.0, 30 );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );

} /* End of Show_Illumination_Predict() */

/*------------------------------------------------------------------------*/

/*  Show_Rotor_Control()
 *
 *  Function to display Az/El rotor status and send
 *  directional demands to the GS-232 controller
 */

  void
Show_Rotor_Control( double azim, double elev, int resol )
{
  /* The text entry widgets of rotor control */
  GtkEntry *az_dem, *az_pos, *el_dem, *el_pos;

  /* Tracking label */
  GtkLabel *label;

  int
    azim_dir,   /* Azimuth direction   */
    elev_dir,   /* Elevation direction */
    azim_error, /* Azimuth error flag  */
    elev_error; /* Elevation error flag*/

  static int
    azim_dem,  /* Azimuth demand   */
    elev_dem;  /* Elevation demand */

  char temp[4];


  /* Read current rotor directions. If serial port   */
  /* read fails, then default azim/elev position = 0 */
  Rotor_Read_Direction( &azim_dir, &elev_dir );
  if( isFlagSet(ROTOR_SERIAL_FAIL) )
    azim_dir = elev_dir = 0;

  /* Initialize Azimuth and Elevation demand */
  if( isFlagClear(CONTROL_INIT) &&
      isFlagClear(ROTOR_SERIAL_FAIL) )
  {
    SetFlag( CONTROL_INIT );
    azim_dem = azim_dir;
    elev_dem = elev_dir;
  }

  /* Calculate new Az/El demand if rotors are not running */
  if( isFlagClear(ROTORS_RUNNING) )
  {
    /* Calculate Az/El demand, rounded to tracking resolution */
    azim_dem = (int)( azim + resol/2.0 );
    elev_dem = (int)( elev + resol/2.0 );
    azim_dem = ( azim_dem / resol ) * resol;
    elev_dem = ( elev_dem / resol ) * resol;

    /* Set limits to azimuth and elevation demand just in case */
    azim_dem = ((azim_dem < 0) ? 0 : ((azim_dem > 360) ? 360 : azim_dem));
    elev_dem = ((elev_dem < 0) ? 0 : ((elev_dem > 180) ? 180 : elev_dem));

    /* If reverse tracking is enabled, the elevation is modified to */
    /* 180-elev and azimuth to azim-180. This helps with overhead   */
    /* and North-crossing passes to avoid wide changes of azimuth   */
    if( isFlagSet(REVERSE_TRACK) )
    {
      elev_dem = 180 - elev_dem;
      azim_dem -= 180;
      azim_dem = ( (azim_dem < 0) ? azim_dem+360 : azim_dem );
    }

  } /* End if( isFlagClear(ROTORS_RUNNING) ) */

  /* Set direction error flags if demand and   */
  /* actual position differ by more than 2 deg */
  azim_error = abs( azim_dem - azim_dir );
  azim_error =
    ((azim_error >  2) && (azim_error < 358)) ||
    ((azim_dem ==   0) && (azim_dir > 358) && (azim_dir < 362)) ||
    ((azim_dem == 360) && (azim_dir >  -2) && (azim_dir <   2));

  elev_error = abs( elev_dem - elev_dir );
  elev_error = ( elev_error > 2 );

  /* If azim or elev error, set the relevant flag */
  if( azim_error || elev_error )
    SetFlag( DIRECTION_ERROR );
  else
  {
    if( isFlagClear(TRACKING_ENABLED) )
    {
      Close_Rotor_Serial();
      ClearFlag( ROTORS_ENABLED );
    }

    ClearFlag( DIRECTION_ERROR | ROTORS_RUNNING );
  }

  /* If direction error and rotors not already running, */
  /* send a new position demand to the rotor controller */
  if( isFlagSet(DIRECTION_ERROR) &&
      isFlagClear(ROTORS_RUNNING) )
  {
    SetFlag( ROTORS_RUNNING );
    Rotor_Send_Direction( azim_dem, elev_dem );
  }

  az_dem = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rot_azdem") );
  az_pos = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rot_azim") );
  el_dem = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rot_eldem") );
  el_pos = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rot_elev") );
  label  = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "rot_status") );

  /* Display current operation */
  if( isFlagSet(TRACK_MANUAL) )
    gtk_label_set_markup( label, MANUAL_TRACKING );
  else if( isFlagSet(TRACK_MOON) )
    gtk_label_set_markup( label, MOON_TRACKING );
  else if( isFlagSet(TRACK_SUN) )
    gtk_label_set_markup( label, SUN_TRACKING );
  else if( isFlagSet(TRACK_POLAR) )
    gtk_label_set_markup( label, POLAR_TRACKING );
  else if( isFlagSet(PARK_ROTORS) )
    gtk_label_set_markup( label, ROTORS_PARKED );
  else if( isFlagSet(TRACKING_ENABLED) )
  {
    if( isFlagSet(REVERSE_TRACK) )
      gtk_label_set_markup( label, REV_TRACKING );
    else
      gtk_label_set_markup( label, NORM_TRACKING );
  }
  else if( isFlagSet(ROTORS_RUNNING) )
    gtk_label_set_markup( label, NORM_TRACKING );

  /* Display Running button if such */
  if( azim_error || elev_error )
  {
    if( isFlagSet(ROTOR_SERIAL_FAIL) )
      gtk_label_set_markup( label, SERIAL_FAIL );
    else
      gtk_label_set_markup( label, ROT_RUNNING );
  }

  /* Display 'Stop' if no direction error */
  if( !azim_error )
    /* Display "Stop" for Azim demand */
    gtk_entry_set_text( az_dem, " Stop " );
  else
  {
    /* Display Azim demand */
    snprintf( temp, 4, "%03d", azim_dem );
    gtk_entry_set_text( az_dem, temp);
  }

  /* Display "Stop" for Elev demand */
  if( !elev_error )
    gtk_entry_set_text( el_dem, " Stop " );
  else
  {
    /* Display Elev demand */
    snprintf( temp, 3, "%02d", elev_dem );
    gtk_entry_set_text( el_dem, temp);
  }

  /* If serial port read fail flag is set, show an alarm */
  if( isFlagSet(ROTOR_SERIAL_FAIL) )
  {
    /* Show actual directions */
    gtk_entry_set_text( az_pos, "!Data" );
    gtk_entry_set_text( el_pos, "!Data" );
  }
  else
  {
    /* Show actual directions */
    snprintf( temp, 4, "%03d", azim_dir );
    gtk_entry_set_text( az_pos, temp);

    snprintf( temp, 3, "%02d", elev_dir );
    gtk_entry_set_text( el_pos, temp);
  }

} /* End of Show_Rotor_Control() */

/*------------------------------------------------------------------------*/

/*  Show_CAT_Control()
 *
 *  Function to display the Transceiver CAT window
 */

  void
Show_CAT_Control( transceiver_status_t *tcvr_status )
{
  /* Mode strings (LSB, USB etc) */
  char *mode_names[NUM_MODE_CODES+1] = { TCVR_MODE_NAMES };
  char temp[32];


  /* Show CAT data if serial port read OK */
  if( isFlagSet(CAT_SETUP) )
  {
    int
      mhz,  /* MHz, KHz, Hz parts of tx/rx frequency */
      khz,
      hz;

    GtkEntry *txt_entry;

    /* Show Rx data if Downlink specified */
    if( tcvr_status->flags & DNLINK_DATA )
    {
      /* Rx mode of operation */
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_mode") );
      gtk_entry_set_text( txt_entry, mode_names[tcvr_status->rx_mode] );

      /* Rx frequency */
      mhz =  tcvr_status->rx_freq / 1000000;
      khz = (tcvr_status->rx_freq - mhz * 1000000) / 1000;
      hz  = (tcvr_status->rx_freq - mhz * 1000000 - khz * 1000);

      if( hz < 0 ) hz = 0;
      snprintf( temp, sizeof(temp), "%d,%03d,%03d", mhz, khz, hz );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_freq") );
      gtk_entry_set_text( txt_entry, temp );

      /* Downlink Frequency */
      mhz =  tcvr_status->dnlink_freq / 1000000;
      khz = (tcvr_status->dnlink_freq - mhz * 1000000) / 1000;
      hz  = (tcvr_status->dnlink_freq - mhz * 1000000 - khz * 1000);

      if( hz < 0 ) hz = 0;
      snprintf( temp, sizeof(temp), "%d,%03d,%03d", mhz, khz, hz );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_freq") );
      gtk_entry_set_text( txt_entry, temp );

      /* Downlink path loss */
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_ploss") );
      snprintf( temp, sizeof(temp), "%d", tcvr_status->rx_ploss);
      gtk_entry_set_text( txt_entry, temp );

      /* Rx S-meter */
      {
        GtkProgressBar *pbar =
          GTK_PROGRESS_BAR( Builder_Get_Object(satellite_track_builder, "smeter_pbar") );
        int s;

        /* Try to approximate the S meter amateur convention */
        s = tcvr_status->rx_smeter;
        if( s > 9 ) /* S-meter > S9, present as S9+XXdb */
        {
          s = 9;
          unsigned int db = (unsigned int)(tcvr_status->rx_smeter - s) * 5;
          snprintf( temp, sizeof(temp), "S%d+%udB", s, db );
        }
        else snprintf( temp, sizeof(temp), "S%d", tcvr_status->rx_smeter );

        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_smeter") );
        gtk_entry_set_text( txt_entry, temp );
        if( tcvr_status->rx_smeter > 20 ) tcvr_status->rx_smeter = 20;
        if( tcvr_status->rx_smeter < 0 ) tcvr_status->rx_smeter = 0;
        gtk_progress_bar_set_fraction(
            pbar, (gdouble)tcvr_status->rx_smeter / 20.0 );
      }

      if( isFlagClear(DOPPLER_DISABLED) )
      {
        /* Rx doppler shift */
        snprintf( temp, sizeof(temp), "%d", tcvr_status->rx_doppler );
        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_doppler") );
        gtk_entry_set_text( txt_entry, temp );
      }
      else
      {
        /* No Rx doppler shift */
        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_doppler") );
        gtk_entry_set_text( txt_entry, "Disabled" );
      }

    } /* if( tcvr_status->flags & DNLINK_DATA ) */
    else /* Blank Rx status display */
    {
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_mode") );
      gtk_entry_set_text( txt_entry, " - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_freq") );
      gtk_entry_set_text( txt_entry, "  - - - - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_freq") );
      gtk_entry_set_text( txt_entry, "  - - - - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_ploss") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "rx_doppler") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "dnlink_smeter") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      GtkProgressBar *pbar =
        GTK_PROGRESS_BAR( Builder_Get_Object(satellite_track_builder, "smeter_pbar") );
      gtk_progress_bar_set_fraction( pbar, 0.0 );
    }

    /* Show Tx data if Uplink specified */
    if( tcvr_status->flags & UPLINK_DATA )
    {
      /* Tx mode of operation */
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_mode") );
      gtk_entry_set_text( txt_entry, mode_names[tcvr_status->tx_mode] );

      /* Tx frequency */
      mhz =  tcvr_status->tx_freq / 1000000;
      khz = (tcvr_status->tx_freq - mhz * 1000000) / 1000;
      hz  = (tcvr_status->tx_freq - mhz * 1000000 - khz * 1000);

      if( hz < 0 ) hz = 0;
      snprintf( temp, sizeof(temp), "%d,%03d,%03d", mhz, khz, hz );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_freq") );
      gtk_entry_set_text( txt_entry, temp );

      /* Uplink Frequency */
      mhz =  tcvr_status->uplink_freq / 1000000;
      khz = (tcvr_status->uplink_freq - mhz * 1000000) / 1000;
      hz  = (tcvr_status->uplink_freq - mhz * 1000000 - khz * 1000);

      if( hz < 0 ) hz = 0;
      snprintf( temp, sizeof(temp), "%d,%03d,%03d", mhz, khz, hz );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_freq") );
      gtk_entry_set_text( txt_entry, temp );

      /* Tx path loss */
      snprintf( temp, sizeof(temp), "%d", tcvr_status->tx_ploss );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_ploss") );
      gtk_entry_set_text( txt_entry, temp );

      /* Tx PO/ALC */
      {
        GtkProgressBar *pbar =
          GTK_PROGRESS_BAR( Builder_Get_Object(satellite_track_builder, "poalc_pbar") );

        snprintf( temp, sizeof(temp), "%d", tcvr_status->tx_power );
        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_poalc") );
        gtk_entry_set_text( txt_entry, temp );
        if( tcvr_status->tx_power > 100 ) tcvr_status->tx_power = 100;
        if( tcvr_status->tx_power < 0 ) tcvr_status->tx_power = 0;
        gtk_progress_bar_set_fraction(
            pbar, (gdouble)tcvr_status->tx_power / 100.0 );
      }

      if( isFlagClear(DOPPLER_DISABLED) )
      {
        /* Tx doppler shift */
        snprintf( temp, sizeof(temp), "%d", tcvr_status->tx_doppler );
        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_doppler") );
        gtk_entry_set_text( txt_entry, temp );
      }
      else
      {
        /* No Tx doppler shift */
        txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_doppler") );
        gtk_entry_set_text( txt_entry, "Disabled" );
      }

    } /* if( tcvr_status->flags & UPLINK_DATA ) */
    else /* Blank Tx status display */
    {
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_mode") );
      gtk_entry_set_text( txt_entry, " - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_freq") );
      gtk_entry_set_text( txt_entry, "  - - - - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_freq") );
      gtk_entry_set_text( txt_entry, "  - - - - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_ploss") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "uplink_poalc") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      txt_entry = GTK_ENTRY( Builder_Get_Object(satellite_track_builder, "tx_doppler") );
      gtk_entry_set_text( txt_entry, "  - - - -" );
      GtkProgressBar *pbar =
        GTK_PROGRESS_BAR( Builder_Get_Object(satellite_track_builder, "poalc_pbar") );
      gtk_progress_bar_set_fraction( pbar, 0.0 );
    }

    gtk_label_set_markup( GTK_LABEL(cat_status), CAT_ENABLE );
  } /* if( isFlagClear(TCVR_SERIAL_FAIL) ) */
  else gtk_label_set_markup( GTK_LABEL(cat_status), SERIAL_FAIL );

} /* End of Show_CAT_Control() */

/*------------------------------------------------------------------------*/

/*  Show_Offset_Calibration()
 *
 *  Function to display the Offset calibration window
 */

  gboolean
Show_Offset_Calibration( gpointer data )
{
  static GtkWidget
    *azim_scale,
    *elev_scale;

  GtkBuilder *builder;

  char
    strn[28], /* String for reading data from Controller    */
    buff[6];  /* Buffer for cloning data from input strings */
  size_t s = sizeof(buff);

  /* First call flag */
  static int first_call = TRUE;


  /* Initialize */
  if( first_call )
  {
    /* Request offset calibration data from Controller */
    if( !Open_Rotor_Serial() ) return( FALSE );
    Rotor_Send_String( "O2" );
    ClearFlag( ROTOR_SERIAL_FAIL );

    /* Create widgets */
    calibrate = create_calibration( &builder );
    gtk_widget_show( calibrate );
    azim_scale = Builder_Get_Object(builder, "azim_scale" );
    elev_scale = Builder_Get_Object(builder, "elev_scale" );
    gtk_label_set_text( GTK_LABEL(
          Builder_Get_Object(builder, "cal_label")), "Offset Calibration" );

    first_call = FALSE;
    g_object_unref( builder );

    return( TRUE );
  } /* if( first_call ) */

  /* Read in data from controller. After sending an O2 command, the */
  /* GS232 controller sends offset data in the following format:    */
  /* AZ+llll=+rrrr EL+llll=+rrrr where llll and rrrr are 4-digit    */
  /* integer numbers that must be equal for zero offset calibration.*/

  /* Input Az-El offset calibration data */
  Rotor_Read_String( strn, 28 );

  /* Finish when no serial port data */
  if( isFlagClear(ROTOR_SERIAL_FAIL) )
  {
    int
      ileft,  /* Left value of offset equation  */
      iright, /* Right value of offset equation */
      ierror; /* Offset calibration error */

    /* Clone left-side number of Az data */
    Strlcpy( buff, &strn[2], s );
    ileft = atoi( buff );

    /* Clone right-side number of Az data */
    Strlcpy( buff, &strn[8], s );
    iright = atoi( buff );

    /* Calculate and limit calibration error to +- 10 */
    ierror = iright - ileft;
    if( ierror >  10 )
      ierror =  10;
    if( ierror < -10 )
      ierror = -10;

    /* Display error */
    gtk_range_set_value( GTK_RANGE(azim_scale), (gdouble)ierror );

    /* Clone left-side number of El data */
    Strlcpy( buff, &strn[16], s );
    ileft = atoi( buff );

    /* Clone right-side number of El data */
    Strlcpy( buff, &strn[22], s );
    iright = atoi( buff );

    /* Calculate and limit calibration error to +- 10 */
    ierror = iright - ileft;
    if( ierror >  10 )
      ierror =  10;
    if( ierror < -10 )
      ierror = -10;

    /* Display error */
    gtk_range_set_value( GTK_RANGE(elev_scale), (gdouble)ierror );

    return( TRUE );
  }

  /* Close window if no data from rotor controller */
  gtk_widget_destroy( calibrate );
  ClearFlag( ROTOR_SERIAL_FAIL );
  Close_Rotor_Serial();
  first_call = TRUE;

  Set_Mode( SINGLE_SAT );

  return( FALSE );

} /* End of Show_Offset_Calibration() */

/*------------------------------------------------------------------------*/

/*  Show_Fullscale_Calibration()
 *
 *  Function to display the Full-scale calibration window
 */

  gboolean
Show_Fullscale_Calibration( gpointer data )
{
  static GtkWidget
    *azim_scale,
    *elev_scale;

  GtkBuilder *builder;

  char
    strn[15], /* String for reading data from Controller */
    buff[6];  /* Buffer for cloning data from input strings */
  size_t s = sizeof(buff);

  /* First call flag */
  static int first_call = TRUE;


  /* Initialize */
  if( first_call )
  {
    /* Request fullscale calibration data from Controller */
    if( !Open_Rotor_Serial() ) return( FALSE );
    Rotor_Send_String( "F2" );
    ClearFlag( ROTOR_SERIAL_FAIL );

    /* Create widgets */
    calibrate = create_calibration( &builder);
    gtk_widget_show( calibrate );
    azim_scale = Builder_Get_Object(builder, "azim_scale" );
    elev_scale = Builder_Get_Object(builder, "elev_scale" );
    gtk_label_set_text( GTK_LABEL(
          Builder_Get_Object(builder, "cal_label")), "Full-Scale Calibration" );

    first_call = FALSE;
    g_object_unref( builder );

    return( TRUE );
  } /* if( first_call ) */

  /* Read in data from controller. After sending an F2 command, the */
  /* GS232 controller sends fullscale data in the following format: */
  /* +aaaa+eeee where aaaa is Azimuth and eeee is Elevation.        */

  /* Input Az-El fullscale calibration data */
  Rotor_Read_String( strn, 15 );

  /* Finish when no serial port data */
  if( isFlagClear(ROTOR_SERIAL_FAIL) )
  {
    int
      azim,  /* Azimuth value   */
      elev,  /* Elevation value */
      error; /* Fullscale calibration error */

    /* Clone Azimuth value */
    Strlcpy( buff, strn, s );
    azim = atoi( buff ) - rc_data.azim_offset;
    if( azim < 180 )
      azim += 360;

    /* Calculate and limit calibration error to +- 10 */
    error = azim - 360;
    if( error >  10 )
      error =  10;
    if( error < -10 )
      error = -10;

    /* Display error */
    gtk_range_set_value( GTK_RANGE(azim_scale), (gdouble)error );

    /* Clone Elevation value */
    Strlcpy( buff, &strn[5], s );
    elev = atoi( buff );

    /* Calculate and limit calibration error to +- 10 */
    error = elev - 180;
    if( error >  10 )
      error =  10;
    if( error < -10 )
      error = -10;

    /* Display error */
    gtk_range_set_value( GTK_RANGE(elev_scale), (gdouble)error );
    return( TRUE );
  }

  /* Close window if no data from rotor controller */
  gtk_widget_destroy( calibrate );
  ClearFlag( ROTOR_SERIAL_FAIL );
  Close_Rotor_Serial();
  first_call = TRUE;

  Set_Mode( SINGLE_SAT );

  return( FALSE );

} /* End of Show_Fullscale_Calibration() */

/*-----------------------------------------------------------------------*/

/* Set_Doppler_Label()
 *
 * Sets the color (markup) of the Doppler Correction labels
 */

  void
Set_Doppler_Label( gchar *markup )
{
  GtkLabel *label;

  label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "rx_dplr") );
  gtk_label_set_markup( label, markup );
  label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "tx_dplr") );
  gtk_label_set_markup( label, markup );
  gtk_label_set_markup( GTK_LABEL(dop_status), markup );
}

/*------------------------------------------------------------------*/

