/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "cat.h"
#include "shared.h"

/* Original serial port options */
static struct termios serial_old_options;

/* Serial port File descriptor */
static int serial_fd = 0;

static char
/* Command codes for Yaesu FT847 CAT */
  FT847_CAT_ON[]      = { (char)0x00, 0, 0, 0, (char)0x00 }, /*  Enable CAT */
  FT847_CAT_OFF[]     = { (char)0x00, 0, 0, 0, (char)0x80 }, /* Disable CAT */
  FT847_SATMD_ON[]    = { (char)0x00, 0, 0, 0, (char)0x4E }, /* Satellite Mode On  */
  FT847_SATMD_OFF[]   = { (char)0x00, 0, 0, 0, (char)0x8E }, /* Satellite Mode Off */
  FT847_STRX_VFO_G[]  = { (char)0x00, 0, 0, 0, (char)0x13 }, /* Get Satellite Rx VFO freq and mode */
  FT847_STRX_VFO_S[]  = { (char)0x00, 0, 0, 0, (char)0x11 }, /* Set Satellite Rx VFO freq */
  FT847_STTX_VFO_G[]  = { (char)0x00, 0, 0, 0, (char)0x23 }, /* Get Satellite Tx VFO freq and mode */
  FT847_STTX_VFO_S[]  = { (char)0x00, 0, 0, 0, (char)0x21 }, /* Set Satellite Tx VFO freq */
  FT847_STRX_MODE_S[] = { (char)0x00, 0, 0, 0, (char)0x17 }, /* Set Satellite Rx Mode */
  FT847_STTX_MODE_S[] = { (char)0x00, 0, 0, 0, (char)0x27 }, /* Set Satellite Tx Mode */
  FT847_CTCSS_ON[]    = { (char)0x4A, 0, 0, 0, (char)0x2A }, /* Set CTCSS Encoder ON  */
  FT847_CTCSS_OFF[]   = { (char)0x8A, 0, 0, 0, (char)0x2A }, /* Set CTCSS Encoder OFF */

  /* Command codes for Yaesu FT857 CAT */
  FT857_SPLIT_ON[]    = { (char)0x00, 0, 0, 0, (char)0x02 }, /* Set FT857 Split On    */
  FT857_SPLIT_OFF[]   = { (char)0x00, 0, 0, 0, (char)0x82 }, /* Set FT857 Split Off   */
  FT857_TOGGLE_AB[]   = { (char)0x00, 0, 0, 0, (char)0x81 }, /* Toggle FT857 VFO A/B  */
  FT857_CTCSS_S[]     = { (char)0x00, 0, 0, 0, (char)0x0B }, /* Set CTCSS frequency   */
  FT857_CTCSS_ON[]    = { (char)0x00, 0, 0, 0, (char)0x0A }, /* Set CTCSS Encoder ON  */
  FT857_CTCSS_OFF[]   = { (char)0x00, 0, 0, 0, (char)0x0A }, /* Set CTCSS Encoder OFF */

  /* Command codes for Yaesu FT847 and FT857 CAT */
  FT8x7_PTT_OFF[]     = { (char)0x00, 0, 0, 0, (char)0x88 }, /* PTT Off (receive) */
  FT8x7_RX_STATUS_G[] = { (char)0x00, 0, 0, 0, (char)0xE7 }, /* Request RX status  */
  FT8x7_TX_STATUS_G[] = { (char)0x00, 0, 0, 0, (char)0xF7 }, /* Request TX status  */
  FT8x7_MAIN_VFO_G[]  = { (char)0x00, 0, 0, 0, (char)0x03 }, /* Get Main VFO freq and mode */
  FT8x7_MAIN_VFO_S[]  = { (char)0x00, 0, 0, 0, (char)0x01 }, /* Set Main VFO freq */
  FT8x7_MAIN_MODE_S[] = { (char)0x00, 0, 0, 0, (char)0x07 }, /* Set Main Mode */

  /* Command codes for Elecraft K3 */
  K3_K31M_S[] = "K31;",             /* Set K3 CAT mode */
  K3_K31M_G[] = "K3;",              /* Get K3 CAT mode */
  K3_VFOA_S[] = "FAnnnnnnnnnnn;",   /* Set VFO A frequency */
  K3_VFOA_G[] = "FA;",              /* Get VFO A frequency */
  K3_VFOB_S[] = "FBnnnnnnnnnnn;",   /* Set VFO B frequency */
  K3_VFOB_G[] = "FB;",              /* Get VFO B frequency */
  K3_SBRX_S[] = "SBn;",             /* Set Sub Receiver enable  */
  K3_SBRX_G[] = "SB;",              /* Get Sub Receiver enabled */
  K3_TXRX_S[] = "RX;",              /* Set PTT Off (receive mode)  */
  K3_MODA_S[] = "MDn;",             /* Set Transceiver mode (Main) */
  K3_MODA_G[] = "MD;",              /* Get Transceiver mode (Main) */
  K3_MODB_S[] = "MD$n;",            /* Set Transceiver mode (Sub)  */
  K3_MODB_G[] = "MD$;",             /* Get Transceiver mode (Sub)  */
  K3_POWR_S[] = "PCnnn;",           /* Get TX power output */
  K3_POWR_G[] = "PC;",              /* Get TX power output */
  K3_SMTA_S[] = "SMnnnn;",          /* Get K3 Main Rx S-meter */
  K3_SMTA_G[] = "SM;",              /* Get K3 Main Rx S-meter */
  K3_SMTB_S[] = "SM$nnnn;",         /* Get K3 Sub Rx S-meter  */
  K3_SMTB_G[] = "SM$;",             /* Get K3 Sub Rx S-meter  */
  K3_FBWA_G[] = "BW;",              /* Get K3 Main Rx Filter Bandwidth */
  K3_FBWB_G[] = "BW$;",             /* Get K3 Sub  Rx Filter Bandwidth */
  K3_FBWA_S[] = "BWnnnn;",          /* Set K3 Main Rx Filter Bandwidth */
  K3_FBWB_S[] = "BW$nnnn;";         /* Set K3 Sub  Rx Filter Bandwidth */

/*-------------------------------------------------------------------------*/

/* Cancel_CAT()
 *
 * Cancels Transceiver CAT (on error)
 */

  static void
Cancel_CAT( void )
{
  GtkToggleButton *tg =
    GTK_TOGGLE_BUTTON( Builder_Get_Object(satellite_track_builder, "tcvr_cat") );
  sleep(1);
  gtk_toggle_button_set_active( tg, FALSE );
  Error_Dialog( "Failed to Establish Transceiver CAT", NONFATAL );
  ClearFlag( CAT_SETUP );
  ClearFlag( TCVR_SERIAL_TEST );
}

/*-------------------------------------------------------------------------*/

/*  Read_Tcvr_Serial()
 *
 *  Reading Data from the Tcvr's Serial Port
 */

  static void
Read_Tcvr_Serial( char *data, size_t len )
{
  int nbytes = 0, cnt = 0;

  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return;

  /* Check for data to be available */
  while( ++cnt < MAX_SERIAL_RETRIES )
  {
    ioctl( serial_fd, FIONREAD, &nbytes);
    if( nbytes == (int)len ) break;
    usleep(50000);
  }

  /* Error condition */
  if( cnt >= MAX_SERIAL_RETRIES )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xsatcom: read(): incorrect number of bytes available:"
        " %d/%d after %d tries\n", nbytes, (int)len, cnt );
    data[0] = '\0';
    Cancel_CAT();
    return;
  }

  /* Clear data buffer and read from serial */
  size_t ret = (size_t)read( serial_fd, data, len );

  /* Error condition */
  if( ret != len )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xsatcom: read(): return value wrong: %d/%d\n",
        (int)ret, (int)len );
    data[0] = '\0';
    Cancel_CAT();
    return;
  }

  return;
} /* End of Read_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Tcvr_Serial()
 *
 *  Writes a command to the Tranceiver
 */

  static void
Write_Tcvr_Serial( char *data, size_t len )
{
  int cnt = 0;
  size_t ret = 0;

  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return;

  /* Flush serial and write command, retrying if needed */
  tcflush( serial_fd, TCIOFLUSH );
  while( ++cnt < MAX_SERIAL_RETRIES )
  {
    ret = (size_t)write( serial_fd, data, len );
    if( ret == len ) break;
    tcflush( serial_fd, TCIOFLUSH );
    usleep(50000);
  }

  /* Error condition */
  if( cnt >= MAX_SERIAL_RETRIES )
  {
    tcflush( serial_fd, TCIOFLUSH );
    fprintf( stderr,
        "xsatcom: write(): incorrect number of bytes written:"
        " %d/%d after %d tries\n", (int)ret, (int)len, cnt );
    Cancel_CAT();
    return;
  }

  /* Give time to CAT to do its thing */
  usleep(5000);
  return;

} /* End of Write_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/* K3_Get_Data()
 *
 * Gets data from the rig for a given Get command
 */
  static gboolean
K3_Get_Data( char *get, char *data )
{
  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return( FALSE );

  /* Clear data buffer */
  size_t len = strlen( data );
  memset( data, '?', len );

  /* Request data from K3, print message on serial port error */
  Write_Tcvr_Serial( get, strlen(get) );
  Read_Tcvr_Serial( data, strlen(data) );
  data[ len ] = '\0';

  /* Print error message if received data header is wrong */
  if( strncmp(get, data, 2 ) != 0 ) /* Leave out */
  {
    fprintf( stderr,
        "xsatcom: K3_Get_Data(): strncmp() failed:"
        " get: %s  data: %s\n", get, data );
    Cancel_CAT();
    return( FALSE );
  }

  return( TRUE );
} /* K3_Get_Data( char *get, char *data ) */

/*-------------------------------------------------------------------------*/

/* K3_Original_Status()
 *
 * Gets or sets the original status
 * (freq, mode etc) of the transceiver
 */
  static gboolean
K3_Original_Status( int action )
{
  static int
    vfo_A, vfo_B, mode_A, mode_B, flbw_A, flbw_B, duplex;

  if( isFlagClear(TCVR_SERIAL_TEST) &&
      isFlagClear(CAT_SETUP) )
    return( FALSE );

  /* Get Status of transceiver */
  if( action == GET_TCVR_STATUS )
  {
    /* Get sub-receiver status */
    if( !K3_Get_Data(K3_SBRX_G, K3_SBRX_S) )
      return( FALSE );
    if( strncmp(K3_SBRX_S, "SB1;", sizeof(K3_SBRX_S)) == 0 )
      duplex = RIG_DUPLEX;
    else
      duplex = RIG_SIMPLEX;

    /* Get Main freq, mode, bandwidth */
    if( !K3_Get_Data(K3_VFOA_G, K3_VFOA_S) ) return( FALSE );
    if( !K3_Get_Data(K3_MODA_G, K3_MODA_S) ) return( FALSE );
    if( !K3_Get_Data(K3_FBWA_G, K3_FBWA_S) ) return( FALSE );
    vfo_A  = atoi( K3_VFOA_S + 2 );
    mode_A = atoi( K3_MODA_S + 2 );
    flbw_A = atoi( K3_FBWA_S + 2 );

    /* Get Sub Rx freq, mode, bandwidth if active */
    if( duplex == RIG_DUPLEX )
    {
      if( !K3_Get_Data(K3_VFOB_G, K3_VFOB_S) ) return( FALSE );
      if( !K3_Get_Data(K3_MODB_G, K3_MODB_S) ) return( FALSE );
      if( !K3_Get_Data(K3_FBWB_G, K3_FBWB_S) )
        return( FALSE );
      vfo_B  = atoi( K3_VFOB_S + 2 );
      mode_B = atoi( K3_MODB_S + 3 );
      flbw_B = atoi( K3_FBWB_S + 3 );
    }
  } /* if( action == GET_TCVR_STATUS ) */

  /* Set Status of transceiver */
  if( action == SET_TCVR_STATUS )
  {
    /* Set sub-receiver status */
    snprintf( K3_SBRX_S, sizeof(K3_SBRX_S), "SB%d;", duplex );
    Write_Tcvr_Serial( K3_SBRX_S, sizeof(K3_SBRX_S)-1 );
    sleep(1);

    /* Set Main Rx freq, mode, bandwidth */
    snprintf( K3_VFOA_S, sizeof(K3_VFOA_S), "FA%011d;", vfo_A );
    Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
    snprintf( K3_MODA_S, sizeof(K3_MODA_S), "MD%d;", mode_A );
    Write_Tcvr_Serial( K3_MODA_S, sizeof(K3_MODA_S)-1 );
    Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
    snprintf( K3_FBWA_S, sizeof(K3_FBWA_S), "BW%04d;", flbw_A );
    Write_Tcvr_Serial( K3_FBWA_S, sizeof(K3_FBWA_S)-1 );

    /* Set Sub Rx freq, mode, bandwidth if active */
    if( duplex == RIG_DUPLEX )
    {
      snprintf( K3_VFOB_S, sizeof(K3_VFOB_S), "FB%011d;", vfo_B );
      Write_Tcvr_Serial( K3_VFOB_S, sizeof(K3_VFOB_S)-1 );
      snprintf( K3_MODB_S, sizeof(K3_MODB_S), "MD$%d;", mode_B );
      Write_Tcvr_Serial( K3_MODB_S, sizeof(K3_MODB_S)-1 );
      Write_Tcvr_Serial( K3_VFOB_S, sizeof(K3_VFOB_S)-1 );
      snprintf( K3_FBWB_S, sizeof(K3_FBWB_S), "BW$%04d;", flbw_B );
      Write_Tcvr_Serial( K3_FBWB_S, sizeof(K3_FBWB_S)-1 );
    }
  } /* if( action == SET_TCVR_STATUS ) */

  return( TRUE );
} /* K3_Original_Status() */

/*-------------------------------------------------------------------------*/

/*  Open_Tcvr_Serial()
 *
 *  Opens Tcvr's Serial Port device, returns the file
 *  descriptor serial_fd on success or exits on error
 */

  gboolean
Open_Tcvr_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;
  char mesg[MESG_SIZE];
  size_t s = sizeof(mesg);

  /* For testing serial port */
  char test[5];


  /* Abort if serial already open */
  if( isFlagSet(CAT_SETUP) ) return( TRUE );
  if( rc_data.tcvr_type == NONE )
  {
    Cancel_CAT();
    return( TRUE );
  }

  /* Open CAT serial port, exit on error */
  ClearFlag( TCVR_SERIAL_TEST );
  serial_fd = open( rc_data.cat_serial, O_RDWR | O_NOCTTY );
  if( serial_fd <= 0 )
  {
    perror( rc_data.cat_serial );
    snprintf( mesg, s,
        _("Failed to open %s\n"\
          "CAT not enabled"), rc_data.cat_serial );
    Error_Dialog( mesg, NONFATAL );
    Cancel_CAT();
    return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  if( fcntl( serial_fd, F_SETLK, &lockinfo ) == -1 )
  {
    perror( rc_data.cat_serial );
    if( fcntl(serial_fd, F_GETLK, &lockinfo) != -1 )
    {
      snprintf( mesg, s,
          _("Failed to lock %s\n"\
            "Device locked by pid %d\n"\
            "CAT not enabled"),
          rc_data.cat_serial, lockinfo.l_pid );
      Error_Dialog( mesg, NONFATAL );
    }
    Cancel_CAT();
    return( FALSE );
  }

  /* Save the current serial port options */
  error |= tcgetattr( serial_fd, &serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( serial_fd, &new_options );

  /* Set the i/o baud rates */
  switch( rc_data.tcvr_type )
  {
    case FT847:
      error |= cfsetspeed( &new_options, B57600 );
      break;

    case FT857:
      error |= cfsetspeed( &new_options, B38400 );
      break;

    case K2: case K3:
      error |= cfsetspeed( &new_options, B38400 );
      break;
  }

  /* Set options for 'raw' I/O mode */
  cfmakeraw( &new_options );

  /* Setup read() timeout to .2 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 2;

  /* Setup the new options for the port */
  error |= tcsetattr( serial_fd, TCSAFLUSH, &new_options );
  if( error )
  {
    perror( "xsatcom: tcsetattr() in Open_Tcvr_Serial() failed" );
    Error_Dialog( _("Failed to Enable CAT"), NONFATAL );
    Cancel_CAT();
    return( FALSE );
  }

  /* Enable CAT by testing comms with transceiver */
  SetFlag( TCVR_SERIAL_TEST );
  switch( rc_data.tcvr_type )
  {
    case FT847:
      Write_Tcvr_Serial( FT847_CAT_ON, sizeof(FT847_CAT_ON) );
      Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );
      Read_Tcvr_Serial( test, 5 );
      break;

    case FT857:
      Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );
      Read_Tcvr_Serial( test, 5 );
      break;

    case K2:
      /* Get K2 VFO A as a test of serial port */
      Write_Tcvr_Serial( K3_VFOA_G, sizeof(K3_VFOA_G)-1 );
      Read_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );

      /* Get original tcvr status */
      K3_Original_Status( GET_TCVR_STATUS );
      break;

    case K3:
      /* Set K3 to mode K31 as a test of serial port */
      Strlcpy( K3_K31M_S, "K31;", sizeof(K3_K31M_S) );
      Write_Tcvr_Serial( K3_K31M_S, sizeof(K3_K31M_S)-1 );
      if( !K3_Get_Data(K3_K31M_G, K3_K31M_S) ||
          (strcmp(K3_K31M_S, "K31;") != 0) )
      {
        fprintf( stderr, "xsatcom: failed to enable K3 CAT\n" );
        Cancel_CAT();
        return( FALSE );
      }

      /* Get original tcvr status */
      K3_Original_Status( GET_TCVR_STATUS );
      break;

  } /* switch( rc_data.tcvr_type ) */

  ClearFlag( TCVR_SERIAL_TEST );
  SetFlag( CAT_SETUP );
  return( TRUE );
} /* End of Open_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Close_Tcvr_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Tcvr_Serial( void )
{
  if( isFlagSet(CAT_SETUP) )
  {
    switch( rc_data.tcvr_type )
    {
      /* Disable CAT for FT847/FT857 */
      case FT847:
        Write_Tcvr_Serial( FT8x7_PTT_OFF,   sizeof(FT8x7_PTT_OFF) );
        Write_Tcvr_Serial( FT847_CTCSS_OFF, sizeof(FT847_CTCSS_OFF) );
        Write_Tcvr_Serial( FT847_SATMD_OFF, sizeof(FT847_SATMD_OFF) );
        Write_Tcvr_Serial( FT847_CAT_OFF,   sizeof(FT847_CAT_OFF) );
        transceiver_status.duplex = RIG_SIMPLEX;
        break;

      case FT857:
        Write_Tcvr_Serial( FT8x7_PTT_OFF,   sizeof(FT8x7_PTT_OFF) );
        Write_Tcvr_Serial( FT857_CTCSS_OFF, sizeof(FT857_CTCSS_OFF) );
        break;

        /* Return K2/K3 to original status */
      case K2: case K3:
        Write_Tcvr_Serial( K3_TXRX_S, sizeof(K3_TXRX_S)-1 );
        K3_Original_Status( SET_TCVR_STATUS );
        break;
    }

    ClearFlag( CAT_SETUP );
    ClearFlag( TCVR_SERIAL_TEST );

    if( serial_fd > 0 )
    {
      tcsetattr( serial_fd, TCSANOW, &serial_old_options );
      close( serial_fd );
      serial_fd = 0;
    }
  } /* if( isFlagSet(CAT_SETUP) ) */

} /* End of Close_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/* K3_SubRx_Status()
 *
 * Sets the K3 Sub Rx status (on or off)
 */
  static gboolean
K3_SubRx_Status( int action, transceiver_status_t *tcvr_status )
{
  if( isFlagClear(CAT_SETUP) ) return( FALSE );

  /* Set the command string */
  if( action == RIG_DUPLEX )
    Strlcpy( K3_SBRX_S, "SB1;", sizeof(K3_SBRX_S) );
  else if( action == RIG_SIMPLEX )
    Strlcpy( K3_SBRX_S, "SB0;", sizeof(K3_SBRX_S) );

  /* Send the Sub Rx status command and check it */
  Write_Tcvr_Serial( K3_SBRX_S, sizeof(K3_SBRX_S)-1 );
  sleep(1);
  if( K3_Get_Data(K3_SBRX_G, K3_SBRX_S) &&
      (strcmp(K3_SBRX_S, "SB1;") == 0) )
    tcvr_status->duplex = RIG_DUPLEX;
  else if( strcmp(K3_SBRX_S, "SB0;") == 0 )
    tcvr_status->duplex = RIG_SIMPLEX;

  return( TRUE );
} /* K3_SubRx_Status() */

/*-------------------------------------------------------------------------*/

/* K3_Set_Freq_Mode()
 *
 * Sets the mode and bandwidth of the K3
 */
  static gboolean
K3_Set_Freq_Mode( int action, transceiver_status_t *tcvr_status )
{
  char modes[] = { K3_MODE_CODES };
  int flbws[]  = { K3_FILTER_BW };

  if( isFlagClear(CAT_SETUP) ) return( FALSE );

  /* Write Tx mode, frequency and bandwitdh */
  if( action & SELECT_K3_MAIN_TX )
  {
    /* Set Tx frequency if it has changed */
    if( tcvr_status->tx_freq != tcvr_status->tx_freq_ref )
    {
      tcvr_status->tx_freq_ref = tcvr_status->tx_freq;

      /* Write Tx freq if not manually changed */
      if( !(tcvr_status->flags & TXFREQ_TUNE) )
      {
        snprintf( K3_VFOA_S, sizeof(K3_VFOA_S),
            "FA%011d;", tcvr_status->tx_freq );
        Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      }
      tcvr_status->flags &= ~TXFREQ_TUNE;
    }

    /* Write mode if changed */
    if( tcvr_status->tx_mode != tcvr_status->tx_mode_ref )
    {
      tcvr_status->tx_mode_ref = tcvr_status->tx_mode;
      unsigned char mode = modes[tcvr_status->tx_mode] & 0x0F;
      if( mode > 9 ) mode = 9;
      snprintf( K3_MODA_S, sizeof(K3_MODA_S), "MD%d;", mode );
      Write_Tcvr_Serial( K3_MODA_S, sizeof(K3_MODA_S)-1 );

      /* This is needed to cancel the change of frequency
       * by the rig itself when the mode is changed */
      Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );

      /* Write bandwidth */
      snprintf( K3_FBWA_S, sizeof(K3_FBWA_S),
          "BW%04d;", flbws[tcvr_status->tx_mode] );
      Write_Tcvr_Serial( K3_FBWA_S, sizeof(K3_FBWA_S)-1 );
      sleep(1);
    }
  }

  /* Write Main Rx, frequency mode and bandwitdh */
  if( action & SELECT_K3_MAIN_RX )
  {
    /* Set Rx frequency if it has changed */
    if( tcvr_status->rx_freq != tcvr_status->rx_freq_ref )
    {
      tcvr_status->rx_freq_ref = tcvr_status->rx_freq;

      /* Write Rx freq if not manually changed */
      if( !(tcvr_status->flags & RXFREQ_TUNE) )
      {
        snprintf( K3_VFOA_S, sizeof(K3_VFOA_S),
            "FA%011d;", tcvr_status->rx_freq );
        Write_Tcvr_Serial( K3_VFOA_S, sizeof(K3_VFOA_S)-1 );
      }
      tcvr_status->flags &= ~RXFREQ_TUNE;
    }

    /* Write mode if changed */
    if( tcvr_status->rx_mode != tcvr_status->rx_mode_ref )
    {
      tcvr_status->rx_mode_ref = tcvr_status->rx_mode;
      unsigned char mode = modes[tcvr_status->rx_mode] & 0x0F;
      if( mode > 9 ) mode = 9;
      snprintf( K3_MODA_S, sizeof(K3_MODA_S), "MD%d;", mode );
      Write_Tcvr_Serial( K3_MODA_S, sizeof(K3_MODA_S)-1 );

      /* This is needed to cancel the change of frequency
       * by the rig itself when the mode is changed */
      Write_Tcvr_Serial( K3_VFOA_S, strlen(K3_VFOA_S) );

      /* Write bandwidth */
      snprintf( K3_FBWA_S, sizeof(K3_FBWA_S),
          "BW%04d;", flbws[tcvr_status->rx_mode] );
      Write_Tcvr_Serial( K3_FBWA_S, sizeof(K3_FBWA_S)-1 );
      sleep(1);
    }
  }

  /* Write Sub Rx mode, frequency and bandwitdh */
  if( action & SELECT_K3_SUB_RX )
  {
    /* Set Rx frequency if it has changed */
    if( tcvr_status->rx_freq != tcvr_status->rx_freq_ref )
    {
      tcvr_status->rx_freq_ref = tcvr_status->rx_freq;

      /* Write Rx freq if not manually changed */
      if( !(tcvr_status->flags & RXFREQ_TUNE) )
      {
        /* Write Rx frequency */
        snprintf( K3_VFOB_S, sizeof(K3_VFOB_S),
            "FB%011d;", tcvr_status->rx_freq );
        Write_Tcvr_Serial( K3_VFOB_S, sizeof(K3_VFOB_S)-1 );
      }
      tcvr_status->flags &= ~RXFREQ_TUNE;
    }

    /* Write mode if changed */
    if( tcvr_status->rx_mode != tcvr_status->rx_mode_ref )
    {
      tcvr_status->rx_mode_ref = tcvr_status->rx_mode;
      unsigned char mode = modes[tcvr_status->rx_mode] & 0x0F;
      if( mode > 9 ) mode = 9;
      snprintf( K3_MODB_S, sizeof(K3_MODB_S), "MD$%d;", mode );
      Write_Tcvr_Serial( K3_MODB_S, sizeof(K3_MODB_S)-1 );

      /* This is needed to cancel the change of frequency
       * by the rig itself when the mode is changed */
      Write_Tcvr_Serial( K3_VFOB_S, sizeof(K3_VFOB_S)-1 );

      /* Write bandwidth */
      snprintf( K3_FBWB_S, sizeof(K3_FBWB_S),
          "BW$%04d;", flbws[tcvr_status->rx_mode] );
      Write_Tcvr_Serial( K3_FBWB_S, sizeof(K3_FBWB_S)-1 );
      sleep(2);
    }
  }

  return( TRUE );
} /* K3_Set_Freq_Mode() */

/*-------------------------------------------------------------------------*/

/* FT8x7_Set_CTCSS()
 *
 * Sets or disables the CTCSS tone frequency
 */
  static gboolean
FT8x7_Set_CTCSS( transceiver_status_t *tcvr_status )
{
  int idx;

  /* Set CTCSS tone freq if given */
  if( tcvr_status->tone_freq == tcvr_status->tone_freq_ref )
    return( TRUE );

  if( rc_data.tcvr_type == FT847 )
  {
    char ctcss_parm[5];
    memset( ctcss_parm, 0, sizeof(ctcss_parm) );

    /* Disable CTCSS */
    if( tcvr_status->tone_freq == 0 )
    {
      Write_Tcvr_Serial( FT847_CTCSS_OFF, sizeof(FT847_CTCSS_OFF) );
    }
    else
    {
      /* CTCSS tone codes */
      int tones[] = { CTCSS_TONES };

      /* Find tone code */
      for( idx = 0; idx < NUM_CTCSS_TONES; idx++ )
        if( tones[idx] == tcvr_status->tone_freq )
          break;
      if( idx == NUM_CTCSS_TONES ) idx--;

      /* Set CTCSS frequency */
      ctcss_parm[0] = (char)tones[idx + NUM_CTCSS_TONES];
      ctcss_parm[4] = 0x2B;

      /* Set CTCSS to Sat Tx */
      Write_Tcvr_Serial( FT847_CTCSS_ON, sizeof(FT847_CTCSS_ON) );

      Write_Tcvr_Serial( ctcss_parm, sizeof(ctcss_parm) );
    } /* if( tcvr_status->tone_freq == 0 ) */

  } /* if( rc_data.tcvr_type == FT847 ) */
  else
  {
    if( tcvr_status->tone_freq == 0 )
    {
      /* Disable CTCSS */
      FT857_CTCSS_ON[0] = (char)0x8A;
      Write_Tcvr_Serial( FT857_CTCSS_ON, sizeof(FT857_CTCSS_ON) );
    }
    else
    {
      char ctcss_str[9];

      /* Enable CTCSS and CTCSS Encoder ON */
      FT857_CTCSS_ON[0] = 0x2A;
      Write_Tcvr_Serial( FT857_CTCSS_ON, sizeof(FT857_CTCSS_ON) );

      FT857_CTCSS_ON[0] = 0x4A;
      Write_Tcvr_Serial( FT857_CTCSS_ON, sizeof(FT857_CTCSS_ON) );

      /* Set CTCSS frequency */
      snprintf( ctcss_str, sizeof(ctcss_str), "%04d%04d",
          tcvr_status->tone_freq, tcvr_status->tone_freq );

      for( idx = 0; idx < 4; idx++ )
      {
        FT857_CTCSS_S[idx]  = (char)((ctcss_str[idx*2]  - '0') << 4);
        FT857_CTCSS_S[idx] |= ctcss_str[idx*2+1] - '0';
      }

      Write_Tcvr_Serial( FT857_CTCSS_S, sizeof(FT857_CTCSS_S) );
    }
  } /* if( rc_data.tcvr_type == FT857 ) */

  tcvr_status->tone_freq_ref = tcvr_status->tone_freq;
  usleep(50000);

  return( TRUE );
} /* FT8x7_Set_CTCSS() */

/*-------------------------------------------------------------------------*/

/* FT8x7_Set_Freq_Mode()
 *
 * Sets the frequency and mode of the FT847 or FT857
 */
  static gboolean
FT8x7_Set_Freq_Mode( int action, transceiver_status_t *tcvr_status )
{
  char
    *rx_frq_cmnd = FT847_STRX_VFO_S,
    *tx_frq_cmnd = FT847_STTX_VFO_S,
    *rx_mod_cmnd = FT847_STRX_MODE_S,
    *tx_mod_cmnd = FT847_STTX_MODE_S;

  size_t cmnd_len = 5;
  int rx_freq = 0, tx_freq = 0;
  char frq_str[9];
  int idx;
  char ft847_modes[] = { FT847_MODE_CODES };
  char ft857_modes[] = { FT857_MODE_CODES };
  char *ft8x7_modes  = NULL;

  /* Select modes list */
  if( rc_data.tcvr_type == FT847 )
    ft8x7_modes = ft847_modes;
  else
    ft8x7_modes = ft857_modes;

  /* Set Rx freq and mode to satellite Rx */
  if( action & SELECT_FT8x7_SAT_RX )
  {
    rx_freq = tcvr_status->rx_freq;
    if( rc_data.tcvr_type == FT847 )
    {
      rx_frq_cmnd = FT847_STRX_VFO_S;
      rx_mod_cmnd = FT847_STRX_MODE_S;
    }
    else
    {
      rx_frq_cmnd = FT8x7_MAIN_VFO_S;
      rx_mod_cmnd = FT8x7_MAIN_MODE_S;
    }
  }

  /* Set Tx freq and mode to satellite Tx */
  if( action & SELECT_FT8x7_SAT_TX )
  {
    tx_freq = tcvr_status->tx_freq;
    if( rc_data.tcvr_type == FT847 )
    {
      tx_frq_cmnd = FT847_STTX_VFO_S;
      tx_mod_cmnd = FT847_STTX_MODE_S;
    }
    else
    {
      tx_frq_cmnd = FT8x7_MAIN_VFO_S;
      tx_mod_cmnd = FT8x7_MAIN_MODE_S;
    }
  }

  /* Set Rx freq and mode from main VFO */
  if( action & SELECT_FT8x7_MAIN_RX )
  {
    rx_freq = tcvr_status->rx_freq;
    rx_frq_cmnd = FT8x7_MAIN_VFO_S;
    rx_mod_cmnd = FT8x7_MAIN_MODE_S;
  }

  /* Set Tx freq and mode to main VFO */
  if( action & SELECT_FT8x7_MAIN_TX )
  {
    tx_freq = tcvr_status->tx_freq;
    tx_frq_cmnd = FT8x7_MAIN_VFO_S;
    tx_mod_cmnd = FT8x7_MAIN_MODE_S;
  }

  /* Set Tx frequency and mode */
  if( action & (SELECT_FT8x7_MAIN_TX | SELECT_FT8x7_SAT_TX) )
  {
    /* Set Tx mode if it has changed */
    if( tcvr_status->tx_mode != tcvr_status->tx_mode_ref )
    {
      tcvr_status->tx_mode_ref = tcvr_status->tx_mode;
      tx_mod_cmnd[0] = ft8x7_modes[tcvr_status->tx_mode];
      Write_Tcvr_Serial( tx_mod_cmnd, cmnd_len );
    }

    /* Set Tx frequency if it has changed */
    if( tcvr_status->tx_freq != tcvr_status->tx_freq_ref )
    {
      tcvr_status->tx_freq_ref = tcvr_status->tx_freq;

      /* Do not write Tx freq if manually changed */
      if( !(tcvr_status->flags & TXFREQ_TUNE) )
      {
        if( tx_freq > 470000000 ) tx_freq = 470000000;
        if( tx_freq < 0 ) tx_freq = 0;
        snprintf( frq_str, sizeof(frq_str), "%08d", tx_freq / 10 );
        for( idx = 0; idx < 4; idx++ )
        {
          tx_frq_cmnd[idx]  = (char)((frq_str[idx*2]  - '0') << 4);
          tx_frq_cmnd[idx] |= frq_str[idx*2+1] - '0';
        }
        Write_Tcvr_Serial( tx_frq_cmnd, cmnd_len );
      }
      tcvr_status->flags &= ~TXFREQ_TUNE;

      /* Toggle FT857 to VFO B if in duplex */
      if( (action & SELECT_FT8x7_SAT_TX) &&
          (rc_data.tcvr_type == FT857) )
      {
        usleep(50000);
        Write_Tcvr_Serial( FT857_TOGGLE_AB, sizeof(FT857_TOGGLE_AB) );
      }
    }

    /* Set CTCSS tone as/if needed */
    if( !FT8x7_Set_CTCSS(tcvr_status) )
      return( FALSE );
  } /* if( action & (SELECT_FT8x7_MAIN_TX | */

  /* Set Rx frequency and mode */
  if( action & (SELECT_FT8x7_MAIN_RX | SELECT_FT8x7_SAT_RX) )
  {
    /* Set Rx mode if it has changed */
    if( tcvr_status->rx_mode != tcvr_status->rx_mode_ref )
    {
      tcvr_status->rx_mode_ref = tcvr_status->rx_mode;
      rx_mod_cmnd[0] = ft8x7_modes[tcvr_status->rx_mode];
      usleep(50000);
      Write_Tcvr_Serial( rx_mod_cmnd, cmnd_len );
    }

    /* Set Rx frequency if it has changed */
    if( tcvr_status->rx_freq != tcvr_status->rx_freq_ref )
    {
      tcvr_status->rx_freq_ref = tcvr_status->rx_freq;

      /* Do not write Rx freq if manually changed */
      if( !(tcvr_status->flags & RXFREQ_TUNE) )
      {
        if( rx_freq > 470000000 ) rx_freq = 470000000;
        if( rx_freq < 0 ) rx_freq = 0;
        snprintf( frq_str, sizeof(frq_str), "%08d", rx_freq / 10 );
        for( idx = 0; idx < 4; idx++ )
        {
          rx_frq_cmnd[idx]  = (char)((frq_str[idx*2]  - '0') << 4);
          rx_frq_cmnd[idx] |= frq_str[idx*2+1] - '0';
        }
        Write_Tcvr_Serial( rx_frq_cmnd, cmnd_len );
      }
      tcvr_status->flags &= ~RXFREQ_TUNE;
    }
  }

  return( TRUE );
} /* FT8x7_Set_Freq_Mode() */

/*-------------------------------------------------------------------------*/

/* FT8x7_Get_Freq_Mode()
 *
 * Gets the frequency, mode and status of the FT847 or FT857
 */
  static gboolean
FT8x7_Get_Freq_Mode( int action, transceiver_status_t *tcvr_status )
{
  char
    rx_freq_mode[5] = {0,0,0,0,0}, /* Returned Rx freq and mode data */
    tx_freq_mode[5] = {0,0,0,0,0}; /* Returned Tx freq and mode data */

  char
    rx_status = 0, /* Returned Rx status data */
    tx_status = 0; /* Returned Tx status data */

  int idx;
  char ft847_modes[] = { FT847_MODE_CODES };
  char ft857_modes[] = { FT857_MODE_CODES };
  char *ft8x7_modes = NULL;

  /* Select modes list */
  ft8x7_modes = ft847_modes;
  if( rc_data.tcvr_type == FT857 )
    ft8x7_modes = ft857_modes;

  /* Read Rx freq and mode from satellite Rx */
  if( action & SELECT_FT8x7_SAT_RX )
  {
    /* Get Rx freq and mode */
    if( rc_data.tcvr_type == FT847 )
    {
      Write_Tcvr_Serial( FT847_STRX_VFO_G, sizeof(FT847_STRX_VFO_G) );
    }
    else
      Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );

    Read_Tcvr_Serial( rx_freq_mode, sizeof(rx_freq_mode) );

    /* Get Rx status */
    Write_Tcvr_Serial( FT8x7_RX_STATUS_G, sizeof(FT8x7_RX_STATUS_G) );
    Read_Tcvr_Serial( &rx_status, sizeof(rx_status) );
  }

  /* Get Rx freq and mode from main VFO */
  if( action & SELECT_FT8x7_MAIN_RX )
  {
    Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );
    Read_Tcvr_Serial( rx_freq_mode, sizeof(rx_freq_mode) );

    /* Get Rx status */
    Write_Tcvr_Serial( FT8x7_RX_STATUS_G, sizeof(FT8x7_RX_STATUS_G) );
    Read_Tcvr_Serial( &rx_status, sizeof(rx_status) );
  }

  /* Get Tx freq and mode from main VFO */
  if( action & SELECT_FT8x7_MAIN_TX )
  {
    Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );
    Read_Tcvr_Serial( tx_freq_mode, sizeof(tx_freq_mode) );

    /* Get Tx status */
    Write_Tcvr_Serial( FT8x7_TX_STATUS_G, sizeof(FT8x7_TX_STATUS_G) );
    Read_Tcvr_Serial( &tx_status, sizeof(tx_status) );
  }

  /* Read Tx freq and mode from satellite Tx */
  if( action & SELECT_FT8x7_SAT_TX )
  {
    if( rc_data.tcvr_type == FT847 )
    {
      Write_Tcvr_Serial( FT847_STTX_VFO_G, sizeof(FT847_STTX_VFO_G) );
      Read_Tcvr_Serial( tx_freq_mode, sizeof(tx_freq_mode) );

      /* Get Tx status */
      Write_Tcvr_Serial( FT8x7_TX_STATUS_G, sizeof(FT8x7_TX_STATUS_G) );
      Read_Tcvr_Serial( &tx_status, sizeof(tx_status) );
    }
    else
    {
      /* Toggle FT857 to VFO A if in duplex */
      Write_Tcvr_Serial( FT857_TOGGLE_AB,  sizeof(FT857_TOGGLE_AB) );
      usleep(100000);

      /* Get Tx frequency */
      Write_Tcvr_Serial( FT8x7_MAIN_VFO_G, sizeof(FT8x7_MAIN_VFO_G) );
      Read_Tcvr_Serial( tx_freq_mode, sizeof(tx_freq_mode) );

      /* Get Tx status */
      Write_Tcvr_Serial( FT8x7_TX_STATUS_G, sizeof(FT8x7_TX_STATUS_G) );
      Read_Tcvr_Serial( &tx_status, sizeof(tx_status) );
    }
  }

  /* Decode Rx data */
  if( action & (SELECT_FT8x7_SAT_RX | SELECT_FT8x7_MAIN_RX) )
  {
    /* Decode Rx frequency data in 10x1Hz */
    tcvr_status->rx_freq = 0;
    for( idx = 0; idx < 4; idx++ )
    {
      tcvr_status->rx_freq += (rx_freq_mode[idx] & 0xF0) >> 4;
      tcvr_status->rx_freq *= 10;
      tcvr_status->rx_freq += rx_freq_mode[idx] & 0x0F;
      tcvr_status->rx_freq *= 10;
    }

    /* Decode Rx operating mode (LSB/USB etc) */
    for( idx = 0; idx < NUM_MODE_CODES; idx++ )
      if( ft8x7_modes[idx] == rx_freq_mode[4] )
        break;
    tcvr_status->rx_mode = (unsigned char)idx;

    /* Decode Rx S-meter */
    if( rc_data.tcvr_type == FT847 )
      tcvr_status->rx_smeter = (rx_status & FT847_SMETER) / 2;
    else if( rc_data.tcvr_type == FT857 )
      tcvr_status->rx_smeter = rx_status & FT847_SMETER;

  } /* if( action & (SELECT_FT8x7_SAT_RX | */

  /* Decode Tx data */
  if( action & (SELECT_FT8x7_SAT_TX | SELECT_FT8x7_MAIN_TX) )
  {
    /* Decode Tx frequency data in 10x1Hz */
    tcvr_status->tx_freq = 0;
    for( idx = 0; idx < 4; idx++ )
    {
      tcvr_status->tx_freq += (tx_freq_mode[idx] & 0xF0) >> 4;
      tcvr_status->tx_freq *= 10;
      tcvr_status->tx_freq += tx_freq_mode[idx] & 0x0F;
      tcvr_status->tx_freq *= 10;
    }

    /* Decode Tx operating mode (LSB/USB etc) */
    for( idx = 0; idx < NUM_MODE_CODES; idx++ )
      if( ft8x7_modes[idx] == tx_freq_mode[4] )
        break;
    tcvr_status->tx_mode = (unsigned char)idx;

    /* Decode Tx power O/P */
    if( rc_data.tcvr_type == FT847 )
      tcvr_status->tx_power = (tx_status & FT847_PO_ALC) * 2;
    else if( rc_data.tcvr_type == FT857 )
      tcvr_status->tx_power = (tx_status & FT857_PO_ALC) * 4;

  } /* if( action & (SELECT_FT8x7_SAT_TX | */

  return( TRUE );
} /* FT8x7_Get_Freq_Mode() */

/*-------------------------------------------------------------------------*/

/* K3_Tcvr_Status()
 *
 * Gets or Sets K3 or K2 Transceiver status
 */
  static gboolean
K3_Tcvr_Status( int action, transceiver_status_t *tcvr_status )
{
  int idx;
  unsigned char modes[] = { K3_MODE_CODES };

  /* Do according to action flag */
  switch( action )
  {
    case INITIALIZE_TCVR: /* Set basic configuration (simplex or duplex) */
      /* Get sub-receiver status */
      if( !K3_Get_Data(K3_SBRX_G, K3_SBRX_S) )
        return( FALSE );

      if( strncmp(K3_SBRX_S, "SB1;", sizeof(K3_SBRX_S)) == 0 )
        tcvr_status->duplex = RIG_DUPLEX;
      else
        tcvr_status->duplex = RIG_SIMPLEX;

      /* If rig is in simplex, set to duplex
       * if uplink and downlink data given */
      if( ((tcvr_status->flags & DUPLEX_DATA) == DUPLEX_DATA) &&
          !tcvr_status->duplex )
      {
        if( !K3_SubRx_Status(RIG_DUPLEX, tcvr_status) )
          return( FALSE );
      }
      /* If rig is in duplex, set to simplex */
      else if( ((tcvr_status->flags & DUPLEX_DATA) != DUPLEX_DATA) &&
          tcvr_status->duplex )
      {
        if( !K3_SubRx_Status(RIG_SIMPLEX, tcvr_status) )
          return( FALSE );
      }
      /* Drop through to setting status below */

    __attribute__((fallthrough));
    case SET_TCVR_STATUS: /* Set transceiver status */
      /* Clear sub-actions */
      action &= ~CAT_SUBACTIONS;

      /* Write Tx freq and mode to Main VFO if uplink data */
      if( tcvr_status->flags & UPLINK_DATA )
        action |= SELECT_K3_MAIN_TX;

      /* Write Rx freq and mode to Sub Rx if in duplex mode */
      if( tcvr_status->duplex )
        action |= SELECT_K3_SUB_RX;

      /* Write Rx freq and mode to Main Rx if downlink & no uplink data */
      if( (tcvr_status->flags & DNLINK_DATA) && !tcvr_status->duplex )
        action |= SELECT_K3_MAIN_RX;

      /* Set transceiver mode according to sub-action flags */
      if( !K3_Set_Freq_Mode(action, tcvr_status) )
        return( FALSE );

      /* Set CTCSS tone frequency if given */
      if( tcvr_status->tone_freq != tcvr_status->tone_freq_ref )
      {
        /* FIXME How? */
      }
      break; /* case SET_TCVR_STATUS */

    case GET_TCVR_STATUS: /* Get transceiver status */
      /* Get Rx freq and mode from Sub-Rx if in duplex mode */
      if( tcvr_status->duplex )
      {
        /* Get Rx frequency */
        if( K3_Get_Data(K3_VFOB_G, K3_VFOB_S) )
          tcvr_status->rx_freq = atoi( K3_VFOB_S+2 );
        else
          tcvr_status->rx_freq = tcvr_status->rx_freq_ref;

        /* Get Rx mode */
        if( K3_Get_Data(K3_MODB_G, K3_MODB_S) )
        {
          /* If the mode ref is the narrow BW code of
           * the mode read above, then use the ref code */
          tcvr_status->rx_mode = (unsigned char)atoi( K3_MODB_S+3 );
          if( (tcvr_status->rx_mode | 0x10) == modes[tcvr_status->rx_mode_ref] )
            tcvr_status->rx_mode = modes[tcvr_status->rx_mode_ref];

          /* Find mode number */
          for( idx = 0; idx < NUM_MODE_CODES; idx++ )
            if( modes[idx] == tcvr_status->rx_mode )
              break;
          tcvr_status->rx_mode = (unsigned char)idx;
        }
        else tcvr_status->rx_mode = tcvr_status->rx_mode_ref;

        /* Get Rx S-meter */
        tcvr_status->rx_smeter = 0;
        if( K3_Get_Data(K3_SMTB_G, K3_SMTB_S) )
          tcvr_status->rx_smeter = atoi( K3_SMTB_S+3 );
        else
          tcvr_status->rx_smeter = 0;
      } /* if( tcvr_status->duplex ) */

      /* Get Rx freq and mode from Main Rx if no uplink data */
      if( (tcvr_status->flags & DNLINK_DATA) && !tcvr_status->duplex )
      {
        /* Get Rx frequency */
        if( K3_Get_Data(K3_VFOA_G, K3_VFOA_S) )
          tcvr_status->rx_freq = atoi( K3_VFOA_S+2 );
        else
          tcvr_status->rx_freq = tcvr_status->rx_freq_ref;

        /* Get Rx mode */
        if( K3_Get_Data(K3_MODA_G, K3_MODA_S) )
        {
          /* If the mode ref is the narrow BW code of
           * the mode read above, then use the ref code */
          tcvr_status->rx_mode = (unsigned char)atoi( K3_MODA_S+2 );
          if( (tcvr_status->rx_mode | 0x10) == modes[tcvr_status->rx_mode_ref] )
            tcvr_status->rx_mode = modes[tcvr_status->rx_mode_ref];

          /* Find mode number */
          for( idx = 0; idx < NUM_MODE_CODES; idx++ )
            if( modes[idx] == tcvr_status->rx_mode )
              break;
          tcvr_status->rx_mode = (unsigned char)idx;
        }
        else tcvr_status->rx_mode = tcvr_status->rx_mode_ref;

        /* Get Rx S-meter */
        if( K3_Get_Data(K3_SMTA_G, K3_SMTA_S) )
          tcvr_status->rx_smeter = atoi( K3_SMTA_S+2 );
        else
          tcvr_status->rx_smeter = 0;
      } /* if( tcvr_status->flags & DNLINK_DATA && ) */

      /* Get Tx freq and mode from Main VFO if uplink data */
      if( tcvr_status->flags & UPLINK_DATA )
      {
        /* Get Tx frequency */
        if( K3_Get_Data(K3_VFOA_G, K3_VFOA_S) )
          tcvr_status->tx_freq = atoi( K3_VFOA_S+2 );
        else
          tcvr_status->tx_freq = tcvr_status->tx_freq_ref;

        /* Get Transmitter mode */
        if( K3_Get_Data(K3_MODA_G, K3_MODA_S) )
        {
          /* If the mode ref is the narrow BW code of
           * the mode read above, then use the ref code */
          tcvr_status->tx_mode = (unsigned char)atoi( K3_MODA_S+2 );
          if( (tcvr_status->tx_mode | 0x10) == modes[tcvr_status->tx_mode_ref] )
            tcvr_status->tx_mode = modes[tcvr_status->tx_mode_ref];
          for( idx = 0; idx < NUM_MODE_CODES; idx++ )
            if( modes[idx] == tcvr_status->tx_mode )
              break;
          tcvr_status->tx_mode = (unsigned char)idx;
        }
        else tcvr_status->tx_mode = tcvr_status->tx_mode_ref;

        /* Get Tx power o/p */
        if( K3_Get_Data(K3_POWR_G, K3_POWR_S) )
          tcvr_status->tx_power = atoi( K3_POWR_S+2 );
        else
          tcvr_status->tx_power = 0;
      } /* if( tcvr_status->flags & UPLINK_DATA ) */

      break; /* case GET_TCVR_STATUS */
  } /* switch( action ) */

  return( TRUE );
} /* K3_Tcvr_Status() */

/*-------------------------------------------------------------------------*/

/*  FT8x7_Tcvr_Status()
 *
 *  Gets or Sets the Transceiver Status
 *  of the Yaesu FT-847 or FT-857
 */

  static gboolean
FT8x7_Tcvr_Status( int action, transceiver_status_t *tcvr_status )
{
  /* Do according to action flag */
  switch( action )
  {
    case INITIALIZE_TCVR: /* Set basic configuration (simplex or duplex) */
      /* Set transceiver to duplex if uplink & downlink are given */
      if( ((tcvr_status->flags & DUPLEX_DATA) == DUPLEX_DATA) &&
          !tcvr_status->duplex )
      {
        if( rc_data.tcvr_type == FT847 )
        {
          Write_Tcvr_Serial( FT847_SATMD_ON, sizeof(FT847_SATMD_ON) );
          tcvr_status->duplex = RIG_DUPLEX;
        }
        else if( rc_data.tcvr_type == FT857 )
        {
          Write_Tcvr_Serial( FT857_SPLIT_ON, sizeof(FT857_SPLIT_ON) );
          tcvr_status->duplex = RIG_DUPLEX;
        }
      }
      else if( ((tcvr_status->flags & DUPLEX_DATA) != DUPLEX_DATA) &&
          tcvr_status->duplex )
      {
        if( rc_data.tcvr_type == FT847 )
        {
          Write_Tcvr_Serial( FT847_SATMD_OFF, sizeof(FT847_SATMD_OFF) );
          tcvr_status->duplex = RIG_SIMPLEX;
        }
        else if( rc_data.tcvr_type == FT857 )
        {
          Write_Tcvr_Serial( FT857_SPLIT_OFF, sizeof(FT857_SPLIT_OFF) );
          tcvr_status->duplex = RIG_SIMPLEX;
        }
      }
      /* Drop through to setting status below */

    __attribute__((fallthrough));
    case SET_TCVR_STATUS: /* Set transceiver status */
      /* Clear sub-actions */
      action &= ~CAT_SUBACTIONS;

      /* Write freq and mode to satellite Rx and Tx if in duplex mode */
      if( tcvr_status->duplex )
        action |= SELECT_FT8x7_SAT_RX | SELECT_FT8x7_SAT_TX;

      /* Write Tx freq and mode to Main VFO if uplink data only */
      if( (tcvr_status->flags & UPLINK_DATA) && !tcvr_status->duplex )
        action |= SELECT_FT8x7_MAIN_TX;

      /* Write Rx freq and mode to Main Rx if downlink data only */
      if( (tcvr_status->flags & DNLINK_DATA) && !tcvr_status->duplex )
        action |=  SELECT_FT8x7_MAIN_RX;

      /* Set transceiver mode and freq according to sub-action flags */
      if( !FT8x7_Set_Freq_Mode(action, tcvr_status) )
        return( FALSE );

      break; /* case SET_TCVR_STATUS */

    case GET_TCVR_STATUS: /* Get transceiver status */
      /* Clear sub-actions */
      action &= ~CAT_SUBACTIONS;

      /* Get satellite Rx and Tx freq and mode if in duplex */
      if( tcvr_status->duplex )
        action |= SELECT_FT8x7_SAT_RX | SELECT_FT8x7_SAT_TX;

      /* Get Rx freq and mode from Main Rx if downlink data only */
      if( (tcvr_status->flags & DNLINK_DATA) && !tcvr_status->duplex )
        action |= SELECT_FT8x7_MAIN_RX;

      /* Get Tx freq and mode from Main VFO if uplink data only */
      if( (tcvr_status->flags & UPLINK_DATA) && !tcvr_status->duplex )
        action |= SELECT_FT8x7_MAIN_TX;

      /* Set transceiver mode and freq according to sub-action flags */
      if( !FT8x7_Get_Freq_Mode(action, tcvr_status) )
        return( FALSE );

      break; /* case GET_TCVR_STATUS */
  } /* switch( action ) */

  return( TRUE );
} /* End of FT8x7_Tcvr_Status() */

/*------------------------------------------------------------------------*/

/* Tcvr_Status()
 *
 * Gets or Sets Transceiver status
 */
  static gboolean
Tcvr_Status( int action, transceiver_status_t *tcvr_status )
{
  if( isFlagClear(CAT_SETUP) )
    return( FALSE );

  switch( rc_data.tcvr_type )
  {
    case FT847: case FT857:
      return( FT8x7_Tcvr_Status(action, tcvr_status) );

    case K2: case K3:
      return( K3_Tcvr_Status(action, tcvr_status) );

    default: return( FALSE );
  }

} /* Tcvr_Status( int action, transceiver_status_t *tcvr_status ) */

/*-------------------------------------------------------------------------*/

/*  Initialize_Tcvr()
 *
 *  Initializes transceiver tx/rx modes and frequencies
 */

  gboolean
Initialize_Tcvr( int mode_idx )
{
  char mode[17];
  GtkButton *button;

  if( isFlagClear(CAT_SETUP) ) return( FALSE );

  /* Set new mode to Mode button */
  Strlcpy( mode, satellite_mode[mode_idx].mode_name, sizeof(mode) );
  button = GTK_BUTTON( Builder_Get_Object(satellite_track_builder, "mode_chng") );
  gtk_button_set_label( button, mode );

  /* Flags indicating what transponder data is given */
  transceiver_status.flags  = satellite_mode[mode_idx].flags;

  /* If no transponder data given, do nothing */
  if( satellite_mode[mode_idx].flags == NO_TRANSPONDER )
    return( TRUE );

  /* Downlink data (freq, mode etc) is specified */
  if( satellite_mode[mode_idx].flags & DNLINK_DATA )
  {
    transceiver_status.dnlink_freq = satellite_mode[mode_idx].dnlink_freq;
    transceiver_status.rx_freq     = satellite_mode[mode_idx].rx_freq;
    transceiver_status.rx_freq_ref = 0;
    transceiver_status.rx_mode     = satellite_mode[mode_idx].rx_mode;
    transceiver_status.rx_mode_ref = NO_TCVR_MODE;

    /* There is a frequency offset between receiver and downlink
     * frequency due to the use of a transverter or converter
     * and any frequency errors in transceiver or transponder */
    transceiver_status.rx_offset =
      transceiver_status.dnlink_freq - transceiver_status.rx_freq;

  } /* if( satellite_mode[mode_idx].flags & DNLINK_DATA ) */

  /* Uplink data (freq, mode etc) is specified */
  if( satellite_mode[mode_idx].flags & UPLINK_DATA )
  {
    transceiver_status.uplink_freq = satellite_mode[mode_idx].uplink_freq;
    transceiver_status.tx_freq     = satellite_mode[mode_idx].tx_freq;
    transceiver_status.tx_freq_ref = 0;
    transceiver_status.tx_mode     = satellite_mode[mode_idx].tx_mode;
    transceiver_status.tone_freq   = satellite_mode[mode_idx].tone_freq;
    transceiver_status.tx_mode_ref = NO_TCVR_MODE;

    /* These is a frequency offset between tx and uplink
     * frequency due to the use of a transverter or converter
     * and any frequency errors in transceiver or transponder */
    transceiver_status.tx_offset =
      transceiver_status.uplink_freq - transceiver_status.tx_freq;
  } /* if( satellite_mode[mode_idx].flags & UPLINK_DATA ) */
  else
    transceiver_status.tone_freq = 0;
  transceiver_status.tone_freq_ref = -1;

  /* This is a freq offset between the uplink and downlink  */
  /* frequencies such that a signal from the tx tunes in    */
  /* accurately in the rx. It depends on the mode used (CW  */
  /* or SSB) and passband invertion. It is needed for doing */
  /* accurate tx/rx tracking during manual tuning of tcvr.  */
  if( (satellite_mode[mode_idx].flags & UPLINK_DATA) &&
      (satellite_mode[mode_idx].flags & DNLINK_DATA) )
  {
    transceiver_status.upl_dnl_offset =
      transceiver_status.uplink_freq +
      ( transceiver_status.flags & PASSBAND_NORM ?
        -transceiver_status.dnlink_freq :
        +transceiver_status.dnlink_freq );

  } /* if( (satellite_mode[mode_idx].flags & UPLINK_DATA) && */

  /* If beacon data given, tx/rx data is made same */
  if( satellite_mode[mode_idx].flags & BEACON_DATA )
  {
    transceiver_status.dnlink_freq = satellite_mode[mode_idx].bcn_freq;
    transceiver_status.rx_freq     = satellite_mode[mode_idx].rx_freq;
    transceiver_status.rx_freq_ref = 0;
    transceiver_status.rx_mode     = satellite_mode[mode_idx].bcn_mode;
    transceiver_status.rx_mode_ref = NO_TCVR_MODE;
    transceiver_status.flags      |= DNLINK_DATA;

    /* This is a frequency offset between the rx   */
    /* frequency and beacon freq due to the use of */
    /* converter and any freq errors in the beacon */
    transceiver_status.rx_offset =
      transceiver_status.dnlink_freq - transceiver_status.rx_freq;
  } /* if( satellite_mode[mode_idx].flags & BEACON_DATA ) */

  /* Write new transceiver status */
  if( !Tcvr_Status(INITIALIZE_TCVR, &transceiver_status) )
    return( FALSE );

  return( TRUE );
} /* End of Initialize_Tcvr() */

/*------------------------------------------------------------------*/

/*  CAT_Control()
 *
 *  Controls the Transceiver via the CAT port
 *  mainly for Doppler shift compensation
 */

  void
CAT_Control( satellite_status_t *sat_status )
{
  static int
    rx_doppler = 0,         /* Rx doppler shift correction */
    tx_doppler = 0,         /* Tx doppler shift correction */
    rx_doppler_prev = 0,    /* Previous Rx doppler shift correction */
    tx_doppler_prev = 0;    /* Previous Tx doppler shift correction */

  /* Rate of change of doppler shift */
  int doppler_delta;

  /* Return if no transponder data given */
  if( transceiver_status.flags == NO_TRANSPONDER )
    return;

  /* Return if no data from serial port */
  if( isFlagClear(CAT_SETUP) )
    return;

  /* Cancel Tx/Rx Track if only Rx data given */
  if( !(transceiver_status.flags & UPLINK_DATA) )
  {
    GtkToggleButton *tgbtn = GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "txrx_track") );
    gtk_toggle_button_set_active( tgbtn, FALSE );
    ClearFlag(UPLNK_DNLNK_TRACK);
  }

  /* Read current transceiver status */
  if( !Tcvr_Status(GET_TCVR_STATUS, &transceiver_status) )
    return;

  /* If user has changed Rx or Downlink frequency manually */
  if( (transceiver_status.rx_freq != transceiver_status.rx_freq_ref) ||
      isFlagSet(NEW_DNLINK_FREQ) )
  {
    /* Update downlink freq by the change in Rx freq */
    if( isFlagClear(NEW_DNLINK_FREQ) )
    {
      transceiver_status.dnlink_freq +=
        transceiver_status.rx_freq -
        transceiver_status.rx_freq_ref;
      transceiver_status.flags |= RXFREQ_TUNE;
    }
    ClearFlag( NEW_DNLINK_FREQ );

    /* Make uplink track downlink frequency if enabled */
    if( isFlagSet(UPLNK_DNLNK_TRACK) &&
        (transceiver_status.flags & UPLINK_DATA) )
    {
      transceiver_status.uplink_freq =
        ( transceiver_status.flags & PASSBAND_NORM ?
          (transceiver_status.upl_dnl_offset +
           transceiver_status.dnlink_freq) :
          (transceiver_status.upl_dnl_offset -
           transceiver_status.dnlink_freq) );
    }
  } /* if( (transceiver_status.rx_freq != transceiver_status.rx_freq_ref) || */

  /* If user has changed Tx or Uplink frequency manually */
  if( (transceiver_status.tx_freq != transceiver_status.tx_freq_ref) ||
      isFlagSet(NEW_UPLINK_FREQ) )
  {
    /* Update uplink freq by the change in Tx freq */
    if( isFlagClear(NEW_UPLINK_FREQ) )
    {
      transceiver_status.uplink_freq +=
        transceiver_status.tx_freq -
        transceiver_status.tx_freq_ref;
      transceiver_status.flags |= TXFREQ_TUNE;
    }
    ClearFlag( NEW_UPLINK_FREQ );

    /* Make downlink track uplink frequency if enabled */
    if( isFlagSet(UPLNK_DNLNK_TRACK) &&
        (transceiver_status.flags & DNLINK_DATA) )
    {
      transceiver_status.dnlink_freq =
        ( transceiver_status.flags & PASSBAND_NORM ?
          (transceiver_status.uplink_freq -
           transceiver_status.upl_dnl_offset) :
          (transceiver_status.upl_dnl_offset -
           transceiver_status.uplink_freq) );

    }
  } /* if( transceiver_status.tx_freq != transceiver_status.tx_freq_ref ) */

  /* Match modes if Rx mode manually changed */
  if( transceiver_status.rx_mode != transceiver_status.rx_mode_ref )
  {
    /* Track Tx/Rx freq and mode */
    if( isFlagSet(UPLNK_DNLNK_TRACK) )
    {
      /* Use reverse mode for inverted passbands */
      if( transceiver_status.flags & PASSBAND_NORM )
        transceiver_status.tx_mode = transceiver_status.rx_mode;
      else
      {
        switch( transceiver_status.rx_mode )
        {
          case LSB:
            transceiver_status.tx_mode = USB;
            break;

          case USB:
            transceiver_status.tx_mode = LSB;
            break;

          case CW:
            transceiver_status.tx_mode = CWR;
            break;

          case CWR:
            transceiver_status.tx_mode = CW;
            break;

          case DATA:
            transceiver_status.tx_mode = DATAR;
            break;

          case DATAR:
            transceiver_status.tx_mode = DATA;
            break;

          default:
            transceiver_status.tx_mode = transceiver_status.rx_mode;
        } /* switch( transceiver_status.rx_mode ) */
      } /* if( transceiver_status.flags & PASSBAND_NORM ) */
    } /* if( isFlagSet(UPLNK_DNLNK_TRACK) ) */
  } /* if( transceiver_status.rx_mode != transceiver_status.rx_mode_ref ) */

  /* Match modes if Tx mode manually changed */
  if( transceiver_status.tx_mode != transceiver_status.tx_mode_ref )
  {
    /* Track Tx/Rx freq and mode */
    if( isFlagSet(UPLNK_DNLNK_TRACK) )
    {
      /* Use reverse mode for inverted passbands */
      if( transceiver_status.flags & PASSBAND_NORM )
        transceiver_status.rx_mode = transceiver_status.tx_mode;
      else
      {
        switch( transceiver_status.tx_mode )
        {
          case LSB:
            transceiver_status.rx_mode = USB;
            break;

          case USB:
            transceiver_status.rx_mode = LSB;
            break;

          case CW:
            transceiver_status.rx_mode = CWR;
            break;

          case CWR:
            transceiver_status.rx_mode = CW;
            break;

          case DATA:
            transceiver_status.rx_mode = DATAR;
            break;

          case DATAR:
            transceiver_status.rx_mode = DATA;
            break;

          default:
            transceiver_status.rx_mode = transceiver_status.tx_mode;
        } /* switch( transceiver_status.tx_mode ) */
      } /* if( transceiver_status.flags & PASSBAND_NORM ) */
    } /* if( isFlagSet(UPLNK_DNLNK_TRACK) ) */
  } /* if( transceiver_status.tx_mode != transceiver_status.tx_mode_ref ) */

  /* Calculate doppler shift. sat_status->doppler is in */
  /* Hz/Hz and transceiver_status.dnlink_freq etc in Hz.   */
  rx_doppler = (int)
    ( (double)transceiver_status.dnlink_freq *
      sat_status->doppler );
  tx_doppler = (int)
    ( (double)transceiver_status.uplink_freq *
      sat_status->doppler );

  if( isFlagClear(DOPPLER_DISABLED) )
  {
    /* Apply doppler shift correction. Note that Doppler shift is  */
    /* added to downlink and subtracted from the uplink frequency. */
    /* This is because uplink_freq and dnlink_freq are the signal  */
    /* frequencies arriving and leaving from the satellite itself. */
    if( !(transceiver_status.flags & RXFREQ_TUNE) )
    {
      doppler_delta = rx_doppler - rx_doppler_prev;
      transceiver_status.rx_freq =
        transceiver_status.dnlink_freq -
        transceiver_status.rx_offset   +
        ( rx_doppler + doppler_delta / 2 );
      rx_doppler_prev = rx_doppler;
    }

    if( !(transceiver_status.flags & TXFREQ_TUNE) )
    {
      doppler_delta = tx_doppler - tx_doppler_prev;
      transceiver_status.tx_freq =
        transceiver_status.uplink_freq -
        transceiver_status.tx_offset   -
        ( tx_doppler + doppler_delta / 2 );
      tx_doppler_prev = tx_doppler;
    }

  } /* End of if( isFlagSet(DOPPLER_DISABLED) ) */
  else
  {
    if( !(transceiver_status.flags & RXFREQ_TUNE) )
    {
      transceiver_status.rx_freq =
        transceiver_status.dnlink_freq -
        transceiver_status.rx_offset;
    }

    if( !(transceiver_status.flags & TXFREQ_TUNE) )
    {
      transceiver_status.tx_freq =
        transceiver_status.uplink_freq -
        transceiver_status.tx_offset;
    }
  }

  /* For the FT847/FT857, frequencies are rounded
   * to 10 as the resolution in CAT mode is 10 Hz */
  if( (rc_data.tcvr_type == FT847) ||
      (rc_data.tcvr_type == FT857) )
  {
    transceiver_status.rx_freq /= 10;
    transceiver_status.rx_freq *= 10;
    transceiver_status.tx_freq /= 10;
    transceiver_status.tx_freq *= 10;
  }

  /* Write new transceiver status */
  if( !Tcvr_Status(SET_TCVR_STATUS, &transceiver_status) )
    return;

  /* Enter doppler shift for display */
  transceiver_status.rx_doppler = rx_doppler;
  transceiver_status.tx_doppler = tx_doppler;

  /* Enter path loss for display */
  transceiver_status.rx_ploss = (int)
    ( sat_status->ploss + 20.0 *
      log10((double)transceiver_status.dnlink_freq/1.0E6) );
  transceiver_status.tx_ploss = (int)
    ( sat_status->ploss + 20.0 *
      log10((double)transceiver_status.uplink_freq/1.0E6) );

  return;
} /* End of CAT_Control() */

/*------------------------------------------------------------------*/

