/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "shared.h"

/* File variables */
static GtkWidget *error_dialog  = NULL;
static GtkWidget  *loc_dialog   = NULL;
static GtkBuilder *loc_builder  = NULL;
static GtkWidget  *azel_dialog  = NULL;
static GtkBuilder *azel_builder = NULL;

/* Popup menu */
static GtkWidget  *popup_menu    = NULL;
static GtkBuilder *popup_builder = NULL;

/*------------------------------------------------------------------------*/

/*  Error_Dialog()
 *
 *  Opens an error dialog box
 */
  void
Error_Dialog( char *mesg, gboolean fatal )
{
  GtkWidget *label;
  GtkBuilder *builder;

  Clear_Mode( ALL_MODES );
  if( !error_dialog )
  {
    error_dialog = create_error_dialog( &builder );
    label = Builder_Get_Object( builder, "error_mesg" );
    gtk_label_set_text( GTK_LABEL(label), mesg );
    if( fatal )
    {
      GtkWidget *btn = Builder_Get_Object( builder, "error_ok" );
      gtk_widget_hide( btn );
    }
    gtk_widget_show( error_dialog );
    g_object_unref( builder );
  }

  GtkWidget *btn = Builder_Get_Object(
      satellite_track_builder, "start_togglebutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(btn), FALSE );
}


  void
on_satellite_track_destroy(
    GObject       *object,
    gpointer       user_data)
{
  Cleanup();
  gtk_main_quit();
}


  gboolean
on_satellite_track_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  Clear_Mode( SINGLE_SAT );
  return( FALSE );
}


  void
on_txrx_track_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  /* Enable/disable Rx/Tx tracking */
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    gtk_label_set_markup( GTK_LABEL(trk_status), TRACK_ON );
    SetFlag( UPLNK_DNLNK_TRACK );
  }
  else
  {
    gtk_label_set_markup( GTK_LABEL(trk_status), TRACK_OFF );
    ClearFlag( UPLNK_DNLNK_TRACK );
  }
}


  void
on_update_track_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  GtkToggleButton *tgbutton;

  if( isFlagSet(CAT_SETUP) )
  {
    transceiver_status.upl_dnl_offset =
      ( transceiver_status.flags & PASSBAND_NORM ?
      (transceiver_status.uplink_freq - transceiver_status.dnlink_freq) :
      (transceiver_status.uplink_freq + transceiver_status.dnlink_freq) );

    SetFlag( UPLNK_DNLNK_TRACK );

    tgbutton = GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "txrx_track") );
    gtk_toggle_button_set_active( tgbutton, TRUE );
  }
}


  void
on_doppler_corr_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    Set_Doppler_Label( DOPPLER_ON );
    ClearFlag( DOPPLER_DISABLED );
  }
  else
  {
    Set_Doppler_Label( DOPPLER_OFF );
    SetFlag( DOPPLER_DISABLED );
  }
}


  void
on_tcvr_cat_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  GtkLabel *label;
  char *ret_str;

  /* Enable/disable cat control */
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    ret_str = Read_Transponder_Data( satellite_status.name );
    if( strcmp(ret_str, "OK!") != 0 )
    {
      Error_Dialog( ret_str, FATAL );
      return;
    }

    if( !Open_Tcvr_Serial() ) return;
    mode_index = 0;
    if( !Initialize_Tcvr(DEFAULT_MODE) )
      return;

    if( isFlagSet(DOPPLER_DISABLED) )
      Set_Doppler_Label( DOPPLER_OFF );
    else
      Set_Doppler_Label( DOPPLER_ON );

    if( isFlagSet(UPLNK_DNLNK_TRACK) )
      gtk_label_set_markup( GTK_LABEL(trk_status), TRACK_ON );
    else
      gtk_label_set_markup( GTK_LABEL(trk_status), TRACK_OFF );
  }
  else
  {
    Close_Tcvr_Serial();
    label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "rx_dplr") );
    gtk_label_set_text( label, _("Doppler") );
    label = GTK_LABEL( Builder_Get_Object(satellite_track_builder, "tx_dplr") );
    gtk_label_set_text( label, _("Doppler") );
    gtk_label_set_text( GTK_LABEL(dop_status), _("Doppler") );
    gtk_label_set_text( GTK_LABEL(trk_status), _("Tx/Rx Track") );
    gtk_label_set_text( GTK_LABEL(cat_status), _("Enable CAT") );
    gtk_button_set_label( GTK_BUTTON(
          Builder_Get_Object(satellite_track_builder, "mode_chng")), _("Change Mode") );
  }
}


  void
on_rotor_control_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  /* Enable/disable rotor control */
  if( gtk_toggle_button_get_active(togglebutton) )
    Enable_Tracking();
  else if( isFlagSet(ROTORS_ENABLED) )
  {
    int azim_dir, elev_dir;

    /* Read current position of rotors */
    Rotor_Read_Direction( &azim_dir, &elev_dir );

    /* Make current position the new demand */
    if( isFlagClear(ROTOR_SERIAL_FAIL) )
      Manual_Position_Rotors( azim_dir, elev_dir );

    Disable_Tracking();
    ClearFlag( TRACKING_FLAGS );

    gtk_label_set_text( GTK_LABEL(
          Builder_Get_Object(satellite_track_builder, "rot_status")),
        _("Enable Rotors") );
  }
}


  void
on_multi_sat_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( MULTI_SAT );
}


  void
on_pass_pred_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  if( isModeSet(XSATCOM_RUN) )
  {
    Pass_Predict_Mode( NULL );
    Set_Mode( PASS_PRED );
  }
}


  void
on_multi_loc_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( MULTI_LOC );
}


  void
on_illum_pred_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  if( isModeSet(XSATCOM_RUN) )
  {
    Illumination_Predict_Mode( NULL );
    Set_Mode( ILLUM_PRED );
  }
}


  void
on_xplanet_plot_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    SetFlag( XPLANET_PLOT );
    if( xplanet == XPLANET_STOPPED )
    {
      Config_Xplanet( &observer_status );
    }
  }
  else
  {
    ClearFlag( XPLANET_PLOT );
    if( isFlagClear(MULTISAT_PLOT) )
    {
      if( system("pkill xplanet") == -1 )
        fprintf( stderr, "xsatcom: system() function call failed\n" );
      xplanet = XPLANET_STOPPED;
    }
  }
}


  void
on_multisat_plot_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    SetFlag( MULTISAT_PLOT );
    if( xplanet == XPLANET_STOPPED )
    {
      Config_Xplanet( &observer_status );
    }
  }
  else
  {
    ClearFlag( MULTISAT_PLOT );
    if( isFlagClear(XPLANET_PLOT) )
    {
      if( system("pkill xplanet") == -1 )
        fprintf( stderr, "xsatcom: system() function call failed\n" );
      xplanet = XPLANET_STOPPED;
    }
  }
}


  gboolean
on_multi_satellite_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  Set_Mode( END_MULTI_SAT );
  return( FALSE );
}


  gboolean
on_pass_predict_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  while( g_main_context_iteration(NULL, FALSE) );
  Clear_Mode( PASS_PRED );
  return( FALSE );
}


  gboolean
on_multi_location_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  Set_Mode( END_MULTI_LOC );
  return( FALSE );
}


  gboolean
on_illum_predict_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  while( g_main_context_iteration(NULL, FALSE) );
  Clear_Mode( ILLUM_PRED );
  return( FALSE );
}


  void
on_error_quit_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Cleanup();
  gtk_main_quit();
}


  void
on_error_ok_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  gtk_widget_destroy( error_dialog );
}

  void
tree_selection_changed_cb(
    GtkTreeSelection *selection,
    gpointer          user_data)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *name;

  if( gtk_tree_selection_get_selected(selection, &model, &iter) )
  {
    gtk_tree_model_get( model, &iter, 0, &name, -1 );
    Strlcpy( user_data, name, NAME_SIZE );
    g_free( name );
    ClearFlag( USERTIME_PREDICT );
    Set_Mode( ILLUM_PRED_REDO );
  }
}

  void
on_access_pred_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( PASS_PRED_ACCESS );
}


  void
on_visible_pred_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  SetFlag(   VISIBLE_PREDICT  );
  ClearFlag( USERTIME_PREDICT );
}


  void
on_fwd_pred_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( PASS_PRED_FWD );
}


  void
on_pass_redo_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( PASS_PRED_RESET );
}


  void
on_loc_up_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = BACK_OBSERVER;
}


  void
on_loc_dn_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = FORWD_OBSERVER;
}


  void
on_loc_frst_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = FIRST_OBSERVER;
}


  void
on_loc_last_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = LAST_OBSERVER;
}


  void
on_ill_fwd_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( ILLUM_PRED_FWD );
}


  void
on_ill_back_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( ILLUM_PRED_BACK );
}


  void
on_ill_redo_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Set_Mode( ILLUM_PRED_REDO );
}


  void
on_sat_up_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = BACK_SATELLITE;
}


  void
on_sat_dn_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = FORWD_SATELLITE;
}


  void
on_sat_frst_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = FIRST_SATELLITE;
}


  void
on_sat_last_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  change_sat_loc = LAST_SATELLITE;
}

  gboolean
on_calibration_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  /* Prevent calibration window closure */
  return( TRUE );
}


  void
on_offset_calibration_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  Clear_Mode( ALL_MODES );
  g_idle_add( Show_Offset_Calibration, NULL );
}


  void
on_fullscale_calibration_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  Clear_Mode( ALL_MODES );
  g_idle_add( Show_Fullscale_Calibration, NULL );
}


  void
on_track_moon_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  ClearFlag( TRACKING_FLAGS );
  SetFlag( TRACK_MOON );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "rotor_control")), TRUE );
}


  void
on_track_sun_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  ClearFlag( TRACKING_FLAGS );
  SetFlag( TRACK_SUN );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "rotor_control")), TRUE );
}


  void
on_track_polar_star_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  ClearFlag( TRACKING_FLAGS );
  SetFlag( TRACK_POLAR );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "rotor_control")), TRUE );
}


  void
on_park_rotors_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  ClearFlag( TRACKING_FLAGS );
  SetFlag( PARK_ROTORS );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "rotor_control")), TRUE );
}


  void
on_position_rotors_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  azel_dialog = create_azel_dialog( &azel_builder );
  gtk_widget_show( azel_dialog );
}


  void
on_quit_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  gtk_widget_destroy( satellite_track );
}


  void
on_new_location_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  loc_dialog = create_loc_dialog( &loc_builder );
  gtk_widget_show( loc_dialog );
}


  gboolean
on_satellite_track_button_press_event(
    GtkWidget       *widget,
    GdkEventButton  *event,
    gpointer         user_data)
{
  static int first_call = TRUE;

  if( event->button == 3 )
  {
    /* Initialize on first call */
    if( first_call )
    {
      popup_menu = create_popup_menu( &popup_builder );
      first_call = FALSE;
    }

    gtk_menu_popup_at_pointer( GTK_MENU(popup_menu), NULL );
  }

  return( TRUE );
}


  void
on_azel_apply_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  GtkEntry *entry;

  ClearFlag( TRACKING_FLAGS );
  SetFlag( TRACK_MANUAL );

  entry = GTK_ENTRY( Builder_Get_Object(azel_builder, "az_entry") );
  azimuth = Atof( gtk_entry_get_text(entry) );
  entry = GTK_ENTRY( Builder_Get_Object(azel_builder, "el_entry") );
  elevation = Atof( gtk_entry_get_text(entry) );

  /* Set limits to values */
  if( azimuth > 360.0 ) azimuth = 360.;
  if( azimuth < 0.0 ) azimuth = 0.0;
  if( elevation > 180.0 ) elevation = 180.0;
  if( elevation < 0.0 ) elevation = 0.0;

  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
        Builder_Get_Object(satellite_track_builder, "rotor_control")), TRUE );
}


  void
on_azel_ok_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  on_azel_apply_clicked( button, user_data );
  gtk_widget_destroy( azel_dialog );
}

  void
on_loc_apply_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  GtkEntry *entry;
  char temp[12], gloc[7]="xxx";
  size_t s = sizeof( temp );
  int idx;
  double lon, lat;


  /* Reallocate memory to observer data */
  mem_realloc( (void **)&observer_data,
      sizeof(observer_status_t) * (size_t)(num_obs + 1),
      _("for observer_data in on_loc_apply_clicked()") );
  if( observer_data == NULL ) return;

  /* Clear entry flags */
  ClearFlag( LOC_ENTERED | POSITION_ENTERED );

  /* Enter location name and remove trailing spaces */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "loc_name") );
  Strlcpy( observer_data[num_obs].loc_name,
      gtk_entry_get_text(entry),
      sizeof(observer_data[num_obs].loc_name) );

  /* Dump trailing spaces from location name */
  idx = (int)strlen( observer_data[num_obs].loc_name );
  while( (observer_data[num_obs].loc_name[--idx] == ' ') && (idx != 0) );
  observer_data[num_obs].loc_name[++idx] = '\0';

  /* Enter QTH locator */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "qth_loc") );
  Strlcpy( observer_data[num_obs].grid_loc,
      gtk_entry_get_text(entry),
      sizeof(observer_data[num_obs].grid_loc) );

  /* Set QTH locator flag if entered */
  if( strlen(observer_data[num_obs].grid_loc) != 0 )
  {
    SetFlag( LOC_ENTERED );

    /* Display formatted locator if valid */
    if( Locator_Valid(observer_data[num_obs].grid_loc) )
      gtk_entry_set_text( entry, observer_data[num_obs].grid_loc );
    else
      gtk_entry_set_text( entry, _("ERROR") );
  }

  /* Enter longitude deg */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "lon_deg") );
  Strlcpy( temp, gtk_entry_get_text(entry), s );
  if( strlen(temp) != 0 )
  {
    SetFlag(POSITION_ENTERED);
    observer_data[num_obs].obs_geodetic.lon = Atof(temp);
  }

  /* Enter longitude min */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "lon_min") );
  Strlcpy( temp, gtk_entry_get_text(entry), s );
  if( observer_data[num_obs].obs_geodetic.lon < 0. )
    observer_data[num_obs].obs_geodetic.lon -= Atof(temp)/60.;
  else
    observer_data[num_obs].obs_geodetic.lon += Atof(temp)/60.;

  /* Limit longitude to +-180 deg */
  if( observer_data[num_obs].obs_geodetic.lon >  180. ||
      observer_data[num_obs].obs_geodetic.lon < -180. )
  {
    gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object(loc_builder, "lon_deg") ),
        _("ERROR") );
    return;
  }

  /* Enter latitude deg */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "lat_deg") );
  Strlcpy( temp, gtk_entry_get_text(entry), s );
  if( strlen(temp) != 0 )
  {
    SetFlag(POSITION_ENTERED);
    observer_data[num_obs].obs_geodetic.lat = Atof(temp);
  }

  /* Enter latitude min */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "lat_min") );
  Strlcpy( temp, gtk_entry_get_text(entry), s );
  if( observer_data[num_obs].obs_geodetic.lat < 0. )
    observer_data[num_obs].obs_geodetic.lat -= Atof(temp)/60.;
  else
    observer_data[num_obs].obs_geodetic.lat += Atof(temp)/60.;

  /* Limit latitude to +-90 deg */
  if( observer_data[num_obs].obs_geodetic.lat >  90. ||
      observer_data[num_obs].obs_geodetic.lat < -90. )
  {
    gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object(loc_builder, "lat_deg")),
        _("ERROR") );
    return;
  }

  /* If Grid Locator and Position data are specified, */
  /* Counter-check Grid Locator and warn on mismatch  */
  if( isFlagSet(LOC_ENTERED) &&
      isFlagSet(POSITION_ENTERED) )
  {
    /* Check grid locator against position */
    lon = observer_data[num_obs].obs_geodetic.lon;
    if( lon > 180. )
      lon -= 360.;
    lat = observer_data[num_obs].obs_geodetic.lat;
    Strlcpy( gloc, observer_data[num_obs].grid_loc, sizeof(gloc) );

    Position_to_Grid_Locator( lon, lat, gloc);

    if( strcmp(observer_data[num_obs].grid_loc, gloc) != 0 )
    {
      gtk_entry_set_text( GTK_ENTRY(
            Builder_Get_Object(loc_builder, "loc_name")),
          _("Locator/Position Mismatch") );
      return;
    }
  }
  /* If only Grid Locator is specified, calculate Position */
  else if( isFlagSet(LOC_ENTERED) &&
      isFlagClear(POSITION_ENTERED) )
  {
    Grid_Locator_to_Position(observer_data[num_obs].grid_loc,
        &lon, &lat);

    observer_data[num_obs].obs_geodetic.lon = lon;
    observer_data[num_obs].obs_geodetic.lat = lat;
  }
  /* If only Position is specified, calculate Grid Locator */
  else if( isFlagSet(POSITION_ENTERED) &&
      isFlagClear(LOC_ENTERED) )
  {
    lon = observer_data[num_obs].obs_geodetic.lon;
    if( lon > 180. ) lon -= 360.;
    lat = observer_data[num_obs].obs_geodetic.lat;
    Position_to_Grid_Locator(lon, lat, observer_data[num_obs].grid_loc);
  }
  else
  {
    gtk_entry_set_text( GTK_ENTRY(
          Builder_Get_Object(loc_builder, "loc_name")),
        _("No Valid Entries") );
    return;
  }

  /* Copy height */
  entry = GTK_ENTRY( Builder_Get_Object(loc_builder, "loc_height") );
  observer_data[num_obs].obs_geodetic.hgt =
    Atof( gtk_entry_get_text(entry) ) / 1000.0;

  /* Convert to rads */
  observer_data[num_obs].obs_geodetic.lat =
    Radians( observer_data[num_obs].obs_geodetic.lat );
  observer_data[num_obs].obs_geodetic.lon =
    Radians( observer_data[num_obs].obs_geodetic.lon );

  num_obs++;
}


  void
on_loc_close_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  gtk_widget_destroy( loc_dialog );
}


  void
on_loc_save_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  Save_Location( &observer_data[num_obs - 1] );
}


  gboolean
on_mode_chng_button_press_event(
    GtkWidget       *widget,
    GdkEventButton  *event,
    gpointer         user_data)
{
  if( isFlagSet(CAT_SETUP) )
  {
    switch( event->button )
    {
      case 1:
        if( ++mode_index >= num_modes )
          mode_index = 0;
        break;

      case 2:
        break;

      case 3:
        if( --mode_index < 0 )
          mode_index += num_modes;
        break;
    }

    Initialize_Tcvr( mode_index );
  }

  return( TRUE );
}


  gboolean
on_error_dialog_delete_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  return( TRUE );
}


  void
on_freeze_multisat_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) )
    SetFlag( FREEZE_MULTISAT );
  else
    ClearFlag( FREEZE_MULTISAT );
}


  void
on_error_dialog_destroy(
    GObject       *object,
    gpointer       user_data)
{
  error_dialog = NULL;
}


  void
on_azel_dialog_destroy(
    GObject       *object,
    gpointer       user_data)
{
  azel_dialog = NULL;
}


  void
on_calibration_destroy(
    GObject       *object,
    gpointer       user_data)
{
  calibrate = NULL;
}


  void
on_illum_predict_destroy(
    GObject       *object,
    gpointer       user_data)
{
  illum_predict = NULL;
}


  void
on_loc_dialog_destroy(
    GObject       *object,
    gpointer       user_data)
{
  loc_dialog = NULL;
}


  void
on_multi_location_destroy(
    GObject       *object,
    gpointer       user_data)
{
  multi_location = NULL;
}


  void
on_multi_satellite_destroy(
    GObject       *object,
    gpointer       user_data)
{
  multi_satellite = NULL;
}


  void
on_pass_predict_destroy(
    GObject       *object,
    gpointer       user_data)
{
  pass_predict = NULL;
}


  void
on_popup_menu_destroy(
    GObject       *object,
    gpointer       user_data)
{
  popup_menu = NULL;
}


  void
on_start_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  static guint s;
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    Set_Mode( SINGLE_SAT | XSATCOM_RUN );
    Central_Mode( NULL );
    s = g_timeout_add_seconds( 1, Central_Mode, "" );
  }
  else
  {
    ClearFlag( XSATCOM_RUN );
    g_source_remove( s );
  }
}


  void
on_download_tle_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data)
{
  char *ret_str;

  if( system("xterm -e \"get_tle.sh; sleep 5\"") == -1 )
    fprintf( stderr, "xsatcom: system() function call failed\n" );

  /* Load new TLE database from xsatcom.tle */
  ret_str = Load_Database_Tle_Data();
  if( strcmp(ret_str, "OK!") != 0 )
    Error_Dialog( ret_str, FATAL );
  else
    change_sat_loc = DNLD_TLE_SETS;
}


  gboolean
on_illum_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( isModeSet(ILLUM_PRED) && isModeSet(XSATCOM_RUN) )
    Draw_Illum_Predict( cr );
  return( FALSE );
}


  void
on_illum_drawingarea_destroy(
    GObject       *object,
    gpointer       user_data)
{
  illum_drawingarea = NULL;
}


  gboolean
on_pass_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( isModeSet(PASS_PRED) && isModeSet(XSATCOM_RUN) )
    Pass_Predict_Mode( cr );
  return( FALSE );
}


  void
on_pass_drawingarea_destroy(
    GObject       *object,
    gpointer       user_data)
{
  pass_drawingarea = NULL;
}


  gboolean
on_dnlink_freq_scroll_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  if( event->scroll.direction == GDK_SCROLL_UP )
    transceiver_status.dnlink_freq += 10;
  else if( event->scroll.direction == GDK_SCROLL_DOWN )
    transceiver_status.dnlink_freq -= 10;
  SetFlag( NEW_DNLINK_FREQ );

  Show_CAT_Control( &transceiver_status );
  return( TRUE );
}


  gboolean
on_uplink_freq_scroll_event(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data)
{
  if( event->scroll.direction == GDK_SCROLL_UP )
    transceiver_status.uplink_freq += 10;
  else if( event->scroll.direction == GDK_SCROLL_DOWN )
    transceiver_status.uplink_freq -= 10;
  SetFlag( NEW_UPLINK_FREQ );

  Show_CAT_Control( &transceiver_status );
  return( TRUE );
}

