/*  utils.c
 *
 *  Utility functions for the xsatcom application
 */

/*
 *  xsatcom: A X/GTK+ application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Abort_On_Error()
 *
 * Returns an error message
 */

  char *
Abort_On_Error( int why )
{
  switch( why )
  {
	case -1 :
	  return(_("Error reading ~/.xsatcom/xsatcom.tle"\
			"\nPlease quit and correct"));

	case -2 :
	  return(_("Incorrect number of lines in"\
			"\n~/.xsatcom/xsatcom.tle file"\
			"\nPlease quit and correct"));

	case -3 :
	  return(_("Invalid TLE set in"\
			"\n~/.xsatcom/xsatcom.tle"\
			"\nPlease quit and correct"));

	case -4 :
	  return(_("Error reading ~/.xsatcom/xsatcom.obs"\
			"\nPlease quit and correct"));

	case -5 :
	  return(_("No valid observer location name found"\
			"\nin ~/.xsatcom/xsatcom.obs file"\
			"\nPlease quit and correct"));

	case -6 :
	  return(_("Invalid observer location data"\
			"\nin ~/.xsatcom/xsatcom.obs file"\
			"\nPlease quit and correct"));

	case -7 :
	  return(_("Conflicting Grid Locator/Position data"\
			"\nin ~/.xsatcom/xsatcom.obs file"\
			"\nPlease quit and correct"));

	case -11 :
	  return(_("Error reading ~/.xsatcom/xsatcom.sat"\
			"\nPlease quit and correct"));

	case -12 :
	  return(_("No satellite names found in"\
			"\n~/.xsatcom/xsatcom.sat"\
			"\nPlease quit and correct"));

	case -13 :
	  return(_("No matching satellite name"\
			"\nfound in ~/.xsatcom/xsatcom.tle"\
			"\nPlease quit and correct"));

	case -14 :
	  return(_("Satellite transponder data in"\
			"\n~/.xsatcom/xsatcom.sat incomplete"\
			"\nPlease quit and correct"));

	case -15 :
	  return(_("Transponder passband flag in"\
			"\n~/.xsatcom/xsatcom.sat invalid"\
			"\nPlease quit and correct"));

	case -17 :
	  return( _("Failed to open file"\
			"\n~/.xsatcom/xsatcom.tle"\
			"\nPlease quit and correct"));

	case -18 :
	  return( _("Failed to open file"\
			"\n~/.xsatcom/xsatcom.sat"\
			"\nPlease quit and correct"));

	case -19 :
	  return( _("Failed to open file"\
			"\n~/.xsatcom/xsatcom.obs"\
			"\nPlease quit and correct"));

	case -20 :
	  return( _("No Transponder Mode name found"\
			"\nin ~/.xsatcom/xsatcom.sat"\
			"\nPlease quit and correct") );

	case -22 :
	  return(_("Invalid satellite name data"\
			"\nin ~/.xsatcom/xsatcom.obs file"\
			"\nPlease quit and correct"));

	case -23 :
	  return(_("Error in sgp4() routine"\
			"\n(Details printed to stderr)"\
			"\nPlease quit and correct"));

  } /* End of switch( why ) */

  return( _("BUG!: Incorrect error code") );

} /* End of Abort_On_Error() */

/*------------------------------------------------------------------------*/

/*  Cleanup()
 *
 *  Cleans up before exiting
 */

  void
Cleanup( void )
{
  /* Make current position the new demand */
  if( isFlagSet(ROTORS_ENABLED) )
  {
	int azim_dir, elev_dir;

	Rotor_Read_Direction( &azim_dir, &elev_dir );
	Manual_Position_Rotors( azim_dir, elev_dir );
	Disable_Tracking();
  }

  /* Return transceiver to original status */
  if( isFlagSet(CAT_SETUP) )
	Close_Tcvr_Serial();

  /* Delete xplanet files */
  unlink( rc_data.xplanet_marker );
  unlink( rc_data.xplanet_gcarcs );
  unlink( rc_data.xplanet_satfile );
  unlink( rc_data.xplanet_config );
  if( system(_("pkill xplanet")) == -1 )
	fprintf( stderr, "xsatcom: system() function call failed\n" );

} /* Cleanup() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flow control flags
 *
 *  See xsatcom.h for definition of flow control flags
 */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet(int flag)
{
  return( (Flags & flag) == flag );
}

  int
isFlagClear(int flag)
{
  return( (~Flags & flag) == flag );
}

  void
SetFlag(int flag)
{
  Flags |= flag;
}

  void
ClearFlag(int flag)
{
  Flags &= ~flag;
}

  void
ToggleFlag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

/*  Functions for testing and setting satellite status flags
 *
 *  See xsatcom.h for definition of satellite status flags
 */

  int
isSatFlagSet(satellite_status_t *sat_status, int flag)
{
  return( (sat_status->flags & flag) == flag );
}

  int
isSatFlagClear(satellite_status_t *sat_status, int flag)
{
  return( (~sat_status->flags & flag) == flag );
}

  void
Set_SatFlag(satellite_status_t *sat_status, int flag)
{
  sat_status->flags |= flag;
}

  void
Clear_SatFlag(satellite_status_t *sat_status, int flag)
{
  sat_status->flags &= ~flag;
}

/*------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n", _("Usage: xwxapt [-hv]") );

  fprintf( stderr, "%s\n",
	  _("       -h: Print this usage information and exit"));

  fprintf( stderr, "%s\n",
	  _("       -v: Print version number and exit"));

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;
 
  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src

 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  int idd = 0; /* dest index */
  int ids = 0; /* src  index */

  /* Find terminating null of dest */
  while( (n > 0) && (ch != '\0') )
  {
	idd++;
	ch = dest[idd];
	n--; /* Count remaining char's in dest */
  }

  /* Copy n-1 chars to leave room for terminating null */
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
	dest[idd] = src[ids];
	ids++;
	ch = src[ids];
	idd++;
	n--;
  }

  /* Terminate dest string */
  dest[idd] = '\0';

} /* Strlcat() */

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, int max_chr, char *messg, gboolean eof_err )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];
  size_t s = sizeof(error_mesg);

  /* Prepare error message */
  snprintf( error_mesg, s,
	  _("Error reading %s\n"\
		"Premature EOF (End Of File)"), messg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  chr = fgetc(pfile);
  if( eof_err && (chr == EOF) )
  {
	fprintf( stderr, "xsatcom: %s\n", error_mesg );
	fclose( pfile );
	Error_Dialog( error_mesg, FATAL );
	return( EOF );
  }

  /* Ignore commented lines, white spaces and eol/cr */
  while(
	  (chr == '#') ||
	  (chr == ' ') ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	{
	  /* Get next character, return error if chr = EOF */
	  chr = fgetc(pfile);
	  if( eof_err && (chr == EOF) )
	  {
		fprintf( stderr, "xsatcom: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, FATAL );
		return( EOF );
	  }
	}

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	{
	  /* Get next character, return error if chr = EOF */
	  chr = fgetc(pfile);
	  if( eof_err && (chr == EOF) )
	  {
		fprintf( stderr, "xsatcom: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg, FATAL );
		return( EOF );
	  }
	}

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = max_chr or EOF or CR/LF */
  while( num_chr < max_chr )
  {
	if( chr == HT ) chr = WS;

	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) ) break;

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	chr = fgetc( pfile );
	if( chr == EOF )
	  break;

  } /* End of while( num_chr < max_chr ) */

  /* Abort if end of line not reached at max_chr */
  if( (num_chr == max_chr) && (chr != LF) && (chr != CR) )
  {
	/* Terminate buffer as a string */
	buff[num_chr] = '\0';
	snprintf( error_mesg, s,
		_("Error reading %s\n"\
		  "Line longer than 80 characters"), messg );
	fprintf( stderr, "xsatcom: %s\n%s\n", error_mesg, buff );
	fclose( pfile );
	Error_Dialog( error_mesg, FATAL );
	return( ERROR );
  }

  /* Terminate buffer as a string and remove trailing blanks */
  buff[num_chr] = '\0';
  num_chr--;
  while( (num_chr >= 0) && (buff[num_chr] == WS) )
  {
	buff[num_chr] = '\0';
	num_chr--;
  }

  if( chr == EOF )
	return( EOF_NORM );
  else
	return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------*/

/***  Memory allocation/freeing utils ***/
void mem_alloc( void **ptr, size_t req, gchar *str )
{
  gchar mesg[MESG_SIZE];

  free_ptr( ptr );
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	snprintf( mesg, sizeof(mesg),
		_("Memory allocation denied %s"), str );
	fprintf( stderr, "%s\n", mesg );
	Error_Dialog( mesg, FATAL );
  }

} /* End of mem_alloc() */

/*------------------------------------------------------------------------*/

void mem_realloc( void **ptr, size_t req, gchar *str )
{
  gchar mesg[MESG_SIZE];

  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
	snprintf( mesg, sizeof(mesg),
		_("Memory re-allocation denied %s"), str );
	fprintf( stderr, "%s\n", mesg );
	Error_Dialog( mesg, FATAL );
  }

} /* End of mem_realloc() */

/*------------------------------------------------------------------------*/

void free_ptr( void **ptr )
{
  if( *ptr != NULL )
	free( *ptr );
  *ptr = NULL;

} /* End of free_ptr() */

/*------------------------------------------------------------------------*/

