/*     ----------------------------------------------------------------
 *
 *                               sgp4unit.cpp
 *
 *    this file contains the sgp4 procedures. the code was originally
 *    released in the 1980 and 1986 papers. in 1997 and 1998, the updated and
 *    copmbined code (sgp4 and sdp4) was released by nasa on the internet.
 *                 seawifs.gsfc.nasa.gov/~seawifsp/src/bobdays/
 *
 *                            companion code for
 *               fundamentals of astrodynamics and applications
 *                                    2004
 *                              by david vallado
 *
 *       (w) 719-573-2600, email dvallado@agi.com
 *
 *    current :
 *              14 aug 06  david vallado
 *                           chg lyddane choice back to strn3, constants, fmod,
 *                           separate debug and writes, misc doc
 *    changes :
 *              26 jul 05  david vallado
 *                           fixes for paper
 *                           note that each fix is preceded by a
 *                           comment with "sgp4fix" and an explanation of
 *                           what was changed
 *              14 may 01  david vallado
 *                           2nd edition baseline
 *                     97  nasa
 *                           internet version
 *                     80  norad
 *                           original baseline
 *       ----------------------------------------------------------------      */

#include "sgp4unit.h"

extern rc_data_t rc_data;

/* ----------- local functions - only ever used internally by sgp4 ---------- */
static void dpper(
 double e3,     double ee2,    double peo,     double pgho,   double pho,
 double pinco,  double plo,    double se2,     double se3,    double sgh2,
 double sgh3,   double sgh4,   double sh2,     double sh3,    double si2,
 double si3,    double sl2,    double sl3,     double sl4,    double t,
 double xgh2,   double xgh3,   double xgh4,    double xh2,    double xh3,
 double xi2,    double xi3,    double xl2,     double xl3,    double xl4,
 double zmol,   double zmos,   char init,      double *ep,    double *inclp,
 double *nodep,  double *argpp, double *mp );

static void dscom(
 double epoch,  double ep,     double argpp,   double tc,     double inclp,
 double nodep,  double np,
 double *snodm, double *cnodm, double *sinim,  double *cosim, double *sinomm,
 double *cosomm,double *day,   double *e3,     double *ee2,   double *em,
 double *emsq,  double *gam,   double *peo,    double *pgho,  double *pho,
 double *pinco, double *plo,   double *rtemsq, double *se2,   double *se3,
 double *sgh2,  double *sgh3,  double *sgh4,   double *sh2,   double *sh3,
 double *si2,   double *si3,   double *sl2,    double *sl3,   double *sl4,
 double *s1,    double *s2,    double *s3,     double *s4,    double *s5,
 double *s6,    double *s7,    double *ss1,    double *ss2,   double *ss3,
 double *ss4,   double *ss5,   double *ss6,    double *ss7,   double *sz1,
 double *sz2,   double *sz3,   double *sz11,   double *sz12,  double *sz13,
 double *sz21,  double *sz22,  double *sz23,   double *sz31,  double *sz32,
 double *sz33,  double *xgh2,  double *xgh3,   double *xgh4,  double *xh2,
 double *xh3,   double *xi2,   double *xi3,    double *xl2,   double *xl3,
 double *xl4,   double *nm,    double *z1,     double *z2,    double *z3,
 double *z11,   double *z12,   double *z13,    double *z21,   double *z22,
 double *z23,   double *z31,   double *z32,    double *z33,   double *zmol,
 double *zmos );

static void dsinit(
 double cosim,  double emsq,   double argpo,   double s1,     double s2,
 double s3,     double s4,     double s5,      double sinim,  double ss1,
 double ss2,    double ss3,    double ss4,     double ss5,    double sz1,
 double sz3,    double sz11,   double sz13,    double sz21,   double sz23,
 double sz31,   double sz33,   double t,       double tc,     double gsto,
 double mo,     double mdot,   double no,      double nodeo,  double nodedot,
 double xpidot, double z1,     double z3,      double z11,    double z13,
 double z21,    double z23,    double z31,     double z33,    double ecco,
 double eccsq,  double *em,    double *argpm,  double *inclm, double *mm,
 double *nm,    double *nodem,
 int *irez,
 double *atime, double *d2201, double *d2211,  double *d3210, double *d3222,
 double *d4410, double *d4422, double *d5220,  double *d5232, double *d5421,
 double *d5433, double *dedt,  double *didt,   double *dmdt,  double *dndt,
 double *dnodt, double *domdt, double *del1,   double *del2,  double *del3,
 double *xfact, double *xlamo, double *xli,    double *xni );

static void dspace(
 int irez,
 double d2201,  double d2211,  double d3210,   double d3222,  double d4410,
 double d4422,  double d5220,  double d5232,   double d5421,  double d5433,
 double dedt,   double del1,   double del2,    double del3,   double didt,
 double dmdt,   double dnodt,  double domdt,   double argpo,  double argpdot,
 double t,      double tc,     double gsto,    double xfact,  double xlamo,
 double no,
 double *atime, double *em,    double *argpm,  double *inclm, double *xli,
 double *mm,    double *xni,   double *nodem,  double *dndt,  double *nm );

static void initl(
 double ecco,   double epoch,  double inclo,   double *no,
 char *method,
 double *ainv,  double *ao,    double *con41,  double *con42, double *cosio,
 double *cosio2,double *eccsq, double *omeosq, double *posq,
 double *rp,    double *rteosq,double *sinio , double *gsto );

/* -----------------------------------------------------------------------------
 *
 *                           procedure dpper
 *
 *  this procedure provides deep space long period periodic contributions
 *    to the mean elements.  by design, these periodics are zero at epoch.
 *    this used to be dscom which included initialization, but it's really a
 *    recurring function.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    e3          -
 *    ee2         -
 *    peo         -
 *    pgho        -
 *    pho         -
 *    pinco       -
 *    plo         -
 *    se2 , se3 , sgh2, sgh3, sgh4, sh2, sh3, si2, si3, sl2, sl3, sl4 -
 *    t           -
 *    xh2, xh3, xi2, xi3, xl2, xl3, xl4 -
 *    zmol        -
 *    zmos        -
 *    ep          - eccentricity                           0.0 - 1.0
 *    inclo       - inclination - needed for lyddane modification
 *    nodep       - right ascension of ascending node
 *    argpp       - argument of perigee
 *    mp          - mean anomaly
 *
 *  outputs       :
 *    ep          - eccentricity                           0.0 - 1.0
 *    inclp       - inclination
 *    nodep        - right ascension of ascending node
 *    argpp       - argument of perigee
 *    mp          - mean anomaly
 *
 *  locals        :
 *    alfdp       -
 *    betdp       -
 *    cosip  , sinip  , cosop  , sinop  ,
 *    dalf        -
 *    dbet        -
 *    dls         -
 *    f2, f3      -
 *    pe          -
 *    pgh         -
 *    ph          -
 *    pinc        -
 *    pl          -
 *    sel   , ses   , sghl  , sghs  , shl   , shs   , sil   , sinzf , sis   ,
 *    sll   , sls
 *    xls         -
 *    xnoh        -
 *    zf          -
 *    zm          -
 *
 *  coupling      :
 *    none.
 *
 *  references    :
 *    hoots, roehrich, norad spacetrack report #3 1980
 *    hoots, norad spacetrack report #6 1986
 *    hoots, schumacher and glover 2004
 *    vallado, crawford, hujsak, kelso  2006
 ----------------------------------------------------------------------------*/

static void dpper(
 double e3,     double ee2,    double peo,     double pgho,   double pho,
 double pinco,  double plo,    double se2,     double se3,    double sgh2,
 double sgh3,   double sgh4,   double sh2,     double sh3,    double si2,
 double si3,    double sl2,    double sl3,     double sl4,    double t,
 double xgh2,   double xgh3,   double xgh4,    double xh2,    double xh3,
 double xi2,    double xi3,    double xl2,     double xl3,    double xl4,
 double zmol,   double zmos,   char init,      double *ep,    double *inclp,
 double *nodep,  double *argpp, double *mp )
{
  /* --------------------- local variables ------------------------ */
  double f2,    f3,    pe,    pgh,   ph,   pinc, pl ,
		 sel,   ses,   sghl,  sghs,  shll, shs,  sil,
		 sinzf, sis,   sll,   sls,   zf,   zm;

  /* --------------- calculate time varying periodics ----------- */
  zm    = zmos + zns * t;
  // be sure that the initial call has time set to zero
  if (init == 'y') zm = zmos;
  zf    = zm + 2.0 * zes * sin(zm);
  sinzf = sin(zf);
  f2    =  0.5 * sinzf * sinzf - 0.25;
  f3    = -0.5 * sinzf * cos(zf);
  ses   = se2* f2 + se3 * f3;
  sis   = si2 * f2 + si3 * f3;
  sls   = sl2 * f2 + sl3 * f3 + sl4 * sinzf;
  sghs  = sgh2 * f2 + sgh3 * f3 + sgh4 * sinzf;
  shs   = sh2 * f2 + sh3 * f3;
  zm    = zmol + znl * t;
  if (init == 'y') zm = zmol;
  zf    = zm + 2.0 * zel * sin(zm);
  sinzf = sin(zf);
  f2    =  0.5 * sinzf * sinzf - 0.25;
  f3    = -0.5 * sinzf * cos(zf);
  sel   = ee2 * f2 + e3 * f3;
  sil   = xi2 * f2 + xi3 * f3;
  sll   = xl2 * f2 + xl3 * f3 + xl4 * sinzf;
  sghl  = xgh2 * f2 + xgh3 * f3 + xgh4 * sinzf;
  shll  = xh2 * f2 + xh3 * f3;
  pe    = ses + sel;
  pinc  = sis + sil;
  pl    = sls + sll;
  pgh   = sghs + sghl;
  ph    = shs + shll;

  if (init == 'n')
  {
	double sinip, cosip;

	pe     = pe - peo;
	pinc   = pinc - pinco;
	pl     = pl - plo;
	pgh    = pgh - pgho;
	ph     = ph - pho;
	*inclp = *inclp + pinc;
	*ep    = *ep + pe;
	sinip  = sin(*inclp);
	cosip  = cos(*inclp);

	/* ----------------- apply periodics directly ------------ */
	//  sgp4fix for lyddane choice
	//  strn3 used original inclination - this is technically feasible
	//  gsfc used perturbed inclination - also technically feasible
	//  probably best to readjust the 0.2 limit value and limit discontinuity
	//  use next line for original strn3 approach and original inclination
	//  if (inclo >= 0.2)
	//  use next line for gsfc version and perturbed inclination
	if (*inclp >= 0.2)
	{
	  ph      = ph / sinip;
	  pgh     = pgh - cosip * ph;
	  *argpp  = *argpp + pgh;
	  *nodep  = *nodep + ph;
	  *mp     = *mp + pl;
	}
	else
	{
	  double xls, xnoh, alfdp, betdp, cosop, dalf, dbet, dls, sinop;

	  /* ---- apply periodics with lyddane modification ---- */
	  sinop  = sin(*nodep);
	  cosop  = cos(*nodep);
	  alfdp  = sinip * sinop;
	  betdp  = sinip * cosop;
	  dalf   =  ph * cosop + pinc * cosip * sinop;
	  dbet   = -ph * sinop + pinc * cosip * cosop;
	  alfdp  = alfdp + dalf;
	  betdp  = betdp + dbet;
	  *nodep = fmod(*nodep, twopi);
	  xls    = *mp + *argpp + cosip * *nodep;
	  dls    = pl + pgh - pinc * *nodep * sinip;
	  xls    = xls + dls;
	  xnoh   = *nodep;
	  *nodep = atan2(alfdp, betdp);
	  if (fabs(xnoh - *nodep) > pi)
	  {
		if (*nodep < xnoh)
		  *nodep = *nodep + twopi;
		else
		  *nodep = *nodep - twopi;
	  }
	  *mp    = *mp + pl;
	  *argpp = xls - *mp - cosip * *nodep;
	}
  }   // if init == 'n'

}  // end dpper

/*-----------------------------------------------------------------------------
 *
 *                           procedure dscom
 *
 *  this procedure provides deep space common items used by both the secular
 *    and periodics subroutines.  input is provided as shown. this routine
 *    used to be called dpper, but the functions inside weren't well organized.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    epoch       -
 *    ep          - eccentricity
 *    argpp       - argument of perigee
 *    tc          -
 *    inclp       - inclination
 *    nodep       - right ascension of ascending node
 *    np          - mean motion
 *
 *  outputs       :
 *    sinim  , cosim  , sinomm , cosomm , snodm  , cnodm
 *    day         -
 *    e3          -
 *    ee2         -
 *    em          - eccentricity
 *    emsq        - eccentricity squared
 *    gam         -
 *    peo         -
 *    pgho        -
 *    pho         -
 *    pinco       -
 *    plo         -
 *    rtemsq      -
 *    se2, se3         -
 *    sgh2, sgh3, sgh4        -
 *    sh2, sh3, si2, si3, sl2, sl3, sl4         -
 *    s1, s2, s3, s4, s5, s6, s7          -
 *    ss1, ss2, ss3, ss4, ss5, ss6, ss7, sz1, sz2, sz3         -
 *    sz11, sz12, sz13, sz21, sz22, sz23, sz31, sz32, sz33        -
 *    xgh2, xgh3, xgh4, xh2, xh3, xi2, xi3, xl2, xl3, xl4         -
 *    nm          - mean motion
 *    z1, z2, z3, z11, z12, z13, z21, z22, z23, z31, z32, z33         -
 *    zmol        -
 *    zmos        -
 *
 *  locals        :
 *    a1, a2, a3, a4, a5, a6, a7, a8, a9, a10         -
 *    betasq      -
 *    cc          -
 *    ctem, stem        -
 *    x1, x2, x3, x4, x5, x6, x7, x8          -
 *    xnodce      -
 *    xnoi        -
 *    zcosg  , zsing  , zcosgl , zsingl , zcosh  , zsinh  , zcoshl , zsinhl ,
 *    zcosi  , zsini  , zcosil , zsinil ,
 *    zx          -
 *    zy          -
 *
 *  coupling      :
 *    none.
 *
 *  references    :
 *    hoots, roehrich, norad spacetrack report #3 1980
 *    hoots, norad spacetrack report #6 1986
 *    hoots, schumacher and glover 2004
 *    vallado, crawford, hujsak, kelso  2006
 ----------------------------------------------------------------------------*/

static void dscom(
 double epoch,  double ep,     double argpp,   double tc,     double inclp,
 double nodep,  double np,
 double *snodm, double *cnodm, double *sinim,  double *cosim, double *sinomm,
 double *cosomm,double *day,   double *e3,     double *ee2,   double *em,
 double *emsq,  double *gam,   double *peo,    double *pgho,  double *pho,
 double *pinco, double *plo,   double *rtemsq, double *se2,   double *se3,
 double *sgh2,  double *sgh3,  double *sgh4,   double *sh2,   double *sh3,
 double *si2,   double *si3,   double *sl2,    double *sl3,   double *sl4,
 double *s1,    double *s2,    double *s3,     double *s4,    double *s5,
 double *s6,    double *s7,    double *ss1,    double *ss2,   double *ss3,
 double *ss4,   double *ss5,   double *ss6,    double *ss7,   double *sz1,
 double *sz2,   double *sz3,   double *sz11,   double *sz12,  double *sz13,
 double *sz21,  double *sz22,  double *sz23,   double *sz31,  double *sz32,
 double *sz33,  double *xgh2,  double *xgh3,   double *xgh4,  double *xh2,
 double *xh3,   double *xi2,   double *xi3,    double *xl2,   double *xl3,
 double *xl4,   double *nm,    double *z1,     double *z2,    double *z3,
 double *z11,   double *z12,   double *z13,    double *z21,   double *z22,
 double *z23,   double *z31,   double *z32,    double *z33,   double *zmol,
 double *zmos )
{
  /* --------------------- local variables ------------------------ */
  int lsflg;
  double a1    , a2    , a3    , a4    , a5    , a6    , a7    ,
		 a8    , a9    , a10   , betasq, cc    , ctem  , stem  ,
		 x1    , x2    , x3    , x4    , x5    , x6    , x7    ,
		 x8    , xnodce, xnoi  , zcosg , zcosgl, zcosh , zcoshl,
		 zcosi , zcosil, zsing , zsingl, zsinh , zsinhl, zsini ,
		 zsinil, zx    , zy;

  *nm     = np;
  *em     = ep;
  *snodm  = sin(nodep);
  *cnodm  = cos(nodep);
  *sinomm = sin(argpp);
  *cosomm = cos(argpp);
  *sinim  = sin(inclp);
  *cosim  = cos(inclp);
  *emsq   = *em * *em;
  betasq  = 1.0 - *emsq;
  *rtemsq = sqrt(betasq);

  /* ----------------- initialize lunar solar terms --------------- */
  *peo   = 0.0;
  *pinco = 0.0;
  *plo   = 0.0;
  *pgho  = 0.0;
  *pho   = 0.0;
  *day   = epoch + 18261.5 + tc / 1440.0;
  xnodce = fmod(4.5236020 - 9.2422029e-4 * *day, twopi);
  stem   = sin(xnodce);
  ctem   = cos(xnodce);
  zcosil = 0.91375164 - 0.03568096 * ctem;
  zsinil = sqrt(1.0 - zcosil * zcosil);
  zsinhl = 0.089683511 * stem / zsinil;
  zcoshl = sqrt(1.0 - zsinhl * zsinhl);
  *gam   = 5.8351514 + 0.0019443680 * *day;
  zx     = 0.39785416 * stem / zsinil;
  zy     = zcoshl * ctem + 0.91744867 * zsinhl * stem;
  zx     = atan2(zx, zy);
  zx     = *gam + zx - xnodce;
  zcosgl = cos(zx);
  zsingl = sin(zx);

  /* ------------------------- do solar terms --------------------- */
  zcosg = zcosgs;
  zsing = zsings;
  zcosi = zcosis;
  zsini = zsinis;
  zcosh = *cnodm;
  zsinh = *snodm;
  cc    = c1ss;
  xnoi  = 1.0 / *nm;

  for (lsflg = 1; lsflg <= 2; lsflg++)
  {
	a1  =   zcosg * zcosh + zsing * zcosi * zsinh;
	a3  =  -zsing * zcosh + zcosg * zcosi * zsinh;
	a7  =  -zcosg * zsinh + zsing * zcosi * zcosh;
	a8  =   zsing * zsini;
	a9  =   zsing * zsinh + zcosg * zcosi * zcosh;
	a10 =   zcosg * zsini;
	a2  =   *cosim * a7 + *sinim * a8;
	a4  =   *cosim * a9 + *sinim * a10;
	a5  =  -*sinim * a7 + *cosim * a8;
	a6  =  -*sinim * a9 + *cosim * a10;

	x1  =  a1 * *cosomm + a2 * *sinomm;
	x2  =  a3 * *cosomm + a4 * *sinomm;
	x3  = -a1 * *sinomm + a2 * *cosomm;
	x4  = -a3 * *sinomm + a4 * *cosomm;
	x5  =  a5 * *sinomm;
	x6  =  a6 * *sinomm;
	x7  =  a5 * *cosomm;
	x8  =  a6 * *cosomm;

	*z31 = 12.0 * x1 * x1 - 3.0 * x3 * x3;
	*z32 = 24.0 * x1 * x2 - 6.0 * x3 * x4;
	*z33 = 12.0 * x2 * x2 - 3.0 * x4 * x4;
	*z1  =  3.0 *  (a1 * a1 + a2 * a2) + *z31 * *emsq;
	*z2  =  6.0 *  (a1 * a3 + a2 * a4) + *z32 * *emsq;
	*z3  =  3.0 *  (a3 * a3 + a4 * a4) + *z33 * *emsq;
	*z11 = -6.0 * a1 * a5 + *emsq *  (-24.0 * x1 * x7-6.0 * x3 * x5);
	*z12 = -6.0 *  (a1 * a6 + a3 * a5) + *emsq *
	  (-24.0 * (x2 * x7 + x1 * x8) - 6.0 * (x3 * x6 + x4 * x5));
	*z13 = -6.0 * a3 * a6 + *emsq * (-24.0 * x2 * x8 - 6.0 * x4 * x6);
	*z21 =  6.0 * a2 * a5 + *emsq * (24.0 * x1 * x5 - 6.0 * x3 * x7);
	*z22 =  6.0 *  (a4 * a5 + a2 * a6) + *emsq *
	  (24.0 * (x2 * x5 + x1 * x6) - 6.0 * (x4 * x7 + x3 * x8));
	*z23 =  6.0 * a4 * a6 + *emsq * (24.0 * x2 * x6 - 6.0 * x4 * x8);
	*z1  = *z1 + *z1 + betasq * *z31;
	*z2  = *z2 + *z2 + betasq * *z32;
	*z3  = *z3 + *z3 + betasq * *z33;
	*s3  = cc * xnoi;
	*s2  = -0.5 * *s3 / *rtemsq;
	*s4  = *s3 * *rtemsq;
	*s1  = -15.0 * *em * *s4;
	*s5  = x1 * x3 + x2 * x4;
	*s6  = x2 * x3 + x1 * x4;
	*s7  = x2 * x4 - x1 * x3;

	/* ----------------------- do lunar terms ------------------- */
	if (lsflg == 1)
	{
	  *ss1   = *s1;
	  *ss2   = *s2;
	  *ss3   = *s3;
	  *ss4   = *s4;
	  *ss5   = *s5;
	  *ss6   = *s6;
	  *ss7   = *s7;
	  *sz1   = *z1;
	  *sz2   = *z2;
	  *sz3   = *z3;
	  *sz11  = *z11;
	  *sz12  = *z12;
	  *sz13  = *z13;
	  *sz21  = *z21;
	  *sz22  = *z22;
	  *sz23  = *z23;
	  *sz31  = *z31;
	  *sz32  = *z32;
	  *sz33  = *z33;
	  zcosg = zcosgl;
	  zsing = zsingl;
	  zcosi = zcosil;
	  zsini = zsinil;
	  zcosh = zcoshl * *cnodm + zsinhl * *snodm;
	  zsinh = *snodm * zcoshl - *cnodm * zsinhl;
	  cc    = c1l;
	}
  }

  *zmol = fmod(4.7199672 + 0.22997150  * *day - *gam, twopi);
  *zmos = fmod(6.2565837 + 0.017201977 * *day, twopi);

  /* ------------------------ do solar terms ---------------------- */
  *se2  =   2.0 * *ss1 * *ss6;
  *se3  =   2.0 * *ss1 * *ss7;
  *si2  =   2.0 * *ss2 * *sz12;
  *si3  =   2.0 * *ss2 * (*sz13 - *sz11);
  *sl2  =  -2.0 * *ss3 * *sz2;
  *sl3  =  -2.0 * *ss3 * (*sz3 - *sz1);
  *sl4  =  -2.0 * *ss3 * (-21.0 - 9.0 * *emsq) * zes;
  *sgh2 =   2.0 * *ss4 * *sz32;
  *sgh3 =   2.0 * *ss4 * (sz33 - sz31);
  *sgh4 = -18.0 * *ss4 * zes;
  *sh2  =  -2.0 * *ss2 * *sz22;
  *sh3  =  -2.0 * *ss2 * (*sz23 - *sz21);

  /* ------------------------ do lunar terms ---------------------- */
  *ee2  =   2.0 * *s1 * *s6;
  *e3   =   2.0 * *s1 * *s7;
  *xi2  =   2.0 * *s2 * *z12;
  *xi3  =   2.0 * *s2 * (*z13 - *z11);
  *xl2  =  -2.0 * *s3 * *z2;
  *xl3  =  -2.0 * *s3 * (z3 - z1);
  *xl4  =  -2.0 * *s3 * (-21.0 - 9.0 * *emsq) * zel;
  *xgh2 =   2.0 * *s4 * *z32;
  *xgh3 =   2.0 * *s4 * (*z33 - *z31);
  *xgh4 = -18.0 * *s4 * zel;
  *xh2  =  -2.0 * *s2 * *z22;
  *xh3  =  -2.0 * *s2 * (*z23 - *z21);

}  // end dscom

/*-----------------------------------------------------------------------------
 *
 *                           procedure dsinit
 *
 *  this procedure provides deep space contributions to mean motion dot due
 *    to geopotential resonance with half day and one day orbits.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    cosim, sinim-
 *    emsq        - eccentricity squared
 *    argpo       - argument of perigee
 *    s1, s2, s3, s4, s5      -
 *    ss1, ss2, ss3, ss4, ss5 -
 *    sz1, sz3, sz11, sz13, sz21, sz23, sz31, sz33 -
 *    t           - time
 *    tc          -
 *    gsto        - greenwich sidereal time                   rad
 *    mo          - mean anomaly
 *    mdot        - mean anomaly dot (rate)
 *    no          - mean motion
 *    nodeo       - right ascension of ascending node
 *    nodedot     - right ascension of ascending node dot (rate)
 *    xpidot      -
 *    z1, z3, z11, z13, z21, z23, z31, z33 -
 *    eccm        - eccentricity
 *    argpm       - argument of perigee
 *    inclm       - inclination
 *    mm          - mean anomaly
 *    xn          - mean motion
 *    nodem       - right ascension of ascending node
 *
 *  outputs       :
 *    em          - eccentricity
 *    argpm       - argument of perigee
 *    inclm       - inclination
 *    mm          - mean anomaly
 *    nm          - mean motion
 *    nodem       - right ascension of ascending node
 *    irez        - flag for resonance           0-none, 1-one day, 2-half day
 *    atime       -
 *    d2201, d2211, d3210, d3222, d4410, d4422, d5220, d5232, d5421, d5433    -
 *    dedt        -
 *    didt        -
 *    dmdt        -
 *    dndt        -
 *    dnodt       -
 *    domdt       -
 *    del1, del2, del3        -
 *    ses  , sghl , sghs , sgs  , shl  , shs  , sis  , sls
 *    theta       -
 *    xfact       -
 *    xlamo       -
 *    xli         -
 *    xni
 *
 *  locals        :
 *    ainv2       -
 *    aonv        -
 *    cosisq      -
 *    eoc         -
 *    f220, f221, f311, f321, f322, f330, f441, f442, f522, f523, f542, f543  -
 *    g200, g201, g211, g300, g310, g322, g410, g422, g520, g521, g532, g533  -
 *    sini2       -
 *    temp        -
 *    temp1       -
 *    theta       -
 *    xno2        -
 *
 *  coupling      :
*    getgravconst
*
*  references    :
*    hoots, roehrich, norad spacetrack report #3 1980
*    hoots, norad spacetrack report #6 1986
*    hoots, schumacher and glover 2004
*    vallado, crawford, hujsak, kelso  2006
----------------------------------------------------------------------------*/

static void dsinit(
	double cosim,  double emsq,   double argpo,   double s1,     double s2,
	double s3,     double s4,     double s5,      double sinim,  double ss1,
	double ss2,    double ss3,    double ss4,     double ss5,    double sz1,
	double sz3,    double sz11,   double sz13,    double sz21,   double sz23,
	double sz31,   double sz33,   double t,       double tc,     double gsto,
	double mo,     double mdot,   double no,      double nodeo,  double nodedot,
	double xpidot, double z1,     double z3,      double z11,    double z13,
	double z21,    double z23,    double z31,     double z33,    double ecco,
	double eccsq,  double *em,    double *argpm,  double *inclm, double *mm,
	double *nm,    double *nodem,
	int *irez,
	double *atime, double *d2201, double *d2211,  double *d3210, double *d3222,
	double *d4410, double *d4422, double *d5220,  double *d5232, double *d5421,
	double *d5433, double *dedt,  double *didt,   double *dmdt,  double *dndt,
	double *dnodt, double *domdt, double *del1,   double *del2,  double *del3,
	double *xfact, double *xlamo, double *xli,    double *xni )
{
  /* --------------------- local variables ------------------------ */
  double aonv, ses, sgs, sghl, sghs, shs, shll, sis, sls, theta, emsqo;

  /* -------------------- deep space initialization ------------ */
  *irez = 0;
  if ((*nm < 0.0052359877) && (*nm > 0.0034906585))
	*irez = 1;
  if ((*nm >= 8.26e-3) && (*nm <= 9.24e-3) && (*em >= 0.5))
	*irez = 2;

  /* ------------------------ do solar terms ------------------- */
  ses  =  ss1 * zns * ss5;
  sis  =  ss2 * zns * (sz11 + sz13);
  sls  = -zns * ss3 * (sz1  + sz3  - 14.0 - 6.0 * emsq);
  sghs =  ss4 * zns * (sz31 + sz33 - 6.0);
  shs  = -zns * ss2 * (sz21 + sz23);

  // sgp4fix for 180 deg incl
  if ((*inclm < 5.2359877e-2) || (*inclm > pi - 5.2359877e-2))
	shs = 0.0;
  if (sinim != 0.0)
	shs = shs / sinim;
  sgs   = sghs - cosim * shs;

  /* ------------------------- do lunar terms ------------------ */
  *dedt = ses + s1 * znl * s5;
  *didt = sis + s2 * znl * (z11 + z13);
  *dmdt = sls - znl * s3 * (z1 + z3 - 14.0 - 6.0 * emsq);
  sghl  = s4 * znl * (z31 + z33 - 6.0);
  shll  = -znl * s2 * (z21 + z23);
  // sgp4fix for 180 deg incl
  if ((*inclm < 5.2359877e-2) || (*inclm > pi - 5.2359877e-2))
	shll = 0.0;
  *domdt = sgs + sghl;
  *dnodt = shs;
  if (sinim != 0.0)
  {
	*domdt = *domdt - cosim / sinim * shll;
	*dnodt = *dnodt + shll / sinim;
  }

  /* ----------- calculate deep space resonance effects -------- */
  *dndt  = 0.0;
  theta  = fmod(gsto + tc * rptim, twopi);
  *em    = *em + *dedt * t;
  *inclm = *inclm + *didt * t;
  *argpm = *argpm + *domdt * t;
  *nodem = *nodem + *dnodt * t;
  *mm    = *mm + *dmdt * t;

  //   sgp4fix for negative inclinations
  //   the following if statement should be commented out
  //if (inclm < 0.0)
  //  {
  //    inclm  = -inclm;
  //    argpm  = argpm - pi;
  //    nodem = nodem + pi;
  //  }

  /* -------------- initialize the resonance terms ------------- */
  if (*irez != 0)
  {
	aonv = pow(*nm / rc_data.xke, tothrd);
	/* ---------- geopotential resonance for 12 hour orbits ------ */
	if (*irez == 2)
	{
	  double cosisq, eoc , f321, f322, f441 , f442, f522 , f523,
			 f542  , f543, g201, g211, g310 , g322, g410 , g422,
			 g520  , g521, g532, g533, sini2, temp, temp1, xno2,
			 emo, ainv2, f220, f221;

	  cosisq = cosim * cosim;
	  emo    = *em;
	  *em    = ecco;
	  emsqo  = emsq;
	  emsq   = eccsq;
	  eoc    = *em * emsq;
	  g201   = -0.306 - (*em - 0.64) * 0.440;

	  if (*em <= 0.65)
	  {
		g211 =    3.616  -  13.2470 * *em +  16.2900 * emsq;
		g310 =  -19.302  + 117.3900 * *em - 228.4190 * emsq +  156.5910 * eoc;
		g322 =  -18.9068 + 109.7927 * *em - 214.6334 * emsq +  146.5816 * eoc;
		g410 =  -41.122  + 242.6940 * *em - 471.0940 * emsq +  313.9530 * eoc;
		g422 = -146.407  + 841.8800 * *em - 1629.014 * emsq + 1083.4350 * eoc;
		g520 = -532.114  + 3017.977 * *em - 5740.032 * emsq + 3708.2760 * eoc;
	  }
	  else
	  {
		g211 =   -72.099 +   331.819 * *em -   508.738 * emsq +   266.724 * eoc;
		g310 =  -346.844 +  1582.851 * *em -  2415.925 * emsq +  1246.113 * eoc;
		g322 =  -342.585 +  1554.908 * *em -  2366.899 * emsq +  1215.972 * eoc;
		g410 = -1052.797 +  4758.686 * *em -  7193.992 * emsq +  3651.957 * eoc;
		g422 = -3581.690 + 16178.110 * *em - 24462.770 * emsq + 12422.520 * eoc;
		if (*em > 0.715)
		  g520 =-5149.66 + 29936.92 * *em - 54087.36 * emsq + 31324.56 * eoc;
		else
		  g520 = 1464.74 -  4664.75 * *em +  3763.64 * emsq;
	  }
	  if (*em < 0.7)
	  {
		g533 = -919.22770 + 4988.6100 * *em - 9064.7700 * emsq + 5542.21  * eoc;
		g521 = -822.71072 + 4568.6173 * *em - 8491.4146 * emsq + 5337.524 * eoc;
		g532 = -853.66600 + 4690.2500 * *em - 8624.7700 * emsq + 5341.4  * eoc;
	  }
	  else
	  {
		g533 =-37995.780 + 161616.52 * *em - 229838.20 * emsq + 109377.94 * eoc;
		g521 =-51752.104 + 218913.95 * *em - 309468.16 * emsq + 146349.42 * eoc;
		g532 =-40023.880 + 170470.89 * *em - 242699.48 * emsq + 115605.82 * eoc;
	  }

	  sini2=  sinim * sinim;
	  f220 =  0.75 * (1.0 + 2.0 * cosim + cosisq);
	  f221 =  1.5 * sini2;
	  f321 =  1.875 * sinim  *  (1.0 - 2.0 * cosim - 3.0 * cosisq);
	  f322 = -1.875 * sinim  *  (1.0 + 2.0 * cosim - 3.0 * cosisq);
	  f441 = 35.0 * sini2 * f220;
	  f442 = 39.3750 * sini2 * sini2;
	  f522 =  9.84375 * sinim * (sini2 * (1.0 - 2.0 * cosim- 5.0 * cosisq) +
		  0.33333333 * (-2.0 + 4.0 * cosim + 6.0 * cosisq) );
	  f523 = sinim * (4.92187512 * sini2 * (-2.0 - 4.0 * cosim +
			10.0 * cosisq) + 6.56250012 * (1.0+2.0 * cosim - 3.0 * cosisq));
	  f542 = 29.53125 * sinim * (2.0 - 8.0 * cosim+cosisq *
		  (-12.0 + 8.0 * cosim + 10.0 * cosisq));
	  f543 = 29.53125 * sinim * (-2.0 - 8.0 * cosim+cosisq *
		  (12.0 + 8.0 * cosim - 10.0 * cosisq));
	  xno2   =  *nm * *nm;
	  ainv2  =  aonv * aonv;
	  temp1  =  3.0 * xno2 * ainv2;
	  temp   =  temp1 * root22;
	  *d2201 =  temp * f220 * g201;
	  *d2211 =  temp * f221 * g211;
	  temp1  =  temp1 * aonv;
	  temp   =  temp1 * root32;
	  *d3210 =  temp * f321 * g310;
	  *d3222 =  temp * f322 * g322;
	  temp1  =  temp1 * aonv;
	  temp   =  2.0 * temp1 * root44;
	  *d4410 =  temp * f441 * g410;
	  *d4422 =  temp * f442 * g422;
	  temp1  =  temp1 * aonv;
	  temp   =  temp1 * root52;
	  *d5220 =  temp * f522 * g520;
	  *d5232 =  temp * f523 * g532;
	  temp   =  2.0 * temp1 * root54;
	  *d5421 =  temp * f542 * g521;
	  *d5433 =  temp * f543 * g533;
	  *xlamo =  fmod(mo + nodeo + nodeo-theta - theta, twopi);
	  *xfact =  mdot + *dmdt + 2.0 * (nodedot + *dnodt - rptim) - no;
	  *em    = emo;
	  emsq   = emsqo;
	}

	/* ---------------- synchronous resonance terms -------------- */
	if (*irez == 1)
	{
	  double f220, f311, f330, g200, g300, g310;

	  g200   = 1.0 + emsq * (-2.5 + 0.8125 * emsq);
	  g310   = 1.0 + 2.0 * emsq;
	  g300   = 1.0 + emsq * (-6.0 + 6.60937 * emsq);
	  f220   = 0.75 * (1.0 + cosim) * (1.0 + cosim);
	  f311   = 0.9375 * sinim * sinim * (1.0 + 3.0 * cosim) - 0.75 * (1.0 + cosim);
	  f330   = 1.0 + cosim;
	  f330   = 1.875 * f330 * f330 * f330;
	  *del1  = 3.0 * *nm * *nm * aonv * aonv;
	  *del2  = 2.0 * *del1 * f220 * g200 * q22;
	  *del3  = 3.0 * *del1 * f330 * g300 * q33 * aonv;
	  *del1  = *del1 * f311 * g310 * q31 * aonv;
	  *xlamo = fmod(mo + nodeo + argpo - theta, twopi);
	  *xfact = mdot + xpidot - rptim + *dmdt + *domdt + *dnodt - no;
	}

	/* ------------ for sgp4, initialize the integrator ---------- */
	*xli   = *xlamo;
	*xni   = no;
	*atime = 0.0;
	*nm    = no + *dndt;
  }

}  // end dsinit

/*-----------------------------------------------------------------------------
 *
 *                           procedure dspace
 *
 *  this procedure provides deep space contributions to mean elements for
 *    perturbing third body.  these effects have been averaged over one
 *    revolution of the sun and moon.  for earth resonance effects, the
 *    effects have been averaged over no revolutions of the satellite.
 *    (mean motion)
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    d2201, d2211, d3210, d3222, d4410, d4422, d5220, d5232, d5421, d5433 -
 *    dedt        -
 *    del1, del2, del3  -
 *    didt        -
 *    dmdt        -
 *    dnodt       -
 *    domdt       -
 *    irez        - flag for resonance           0-none, 1-one day, 2-half day
 *    argpo       - argument of perigee
 *    argpdot     - argument of perigee dot (rate)
 *    t           - time
 *    tc          -
 *    gsto        - gst
 *    xfact       -
 *    xlamo       -
 *    no          - mean motion
 *    atime       -
 *    em          - eccentricity
 *    ft          -
 *    argpm       - argument of perigee
 *    inclm       - inclination
 *    xli         -
 *    mm          - mean anomaly
 *    xni         - mean motion
 *    nodem       - right ascension of ascending node
 *
 *  outputs       :
 *    atime       -
 *    em          - eccentricity
 *    argpm       - argument of perigee
 *    inclm       - inclination
 *    xli         -
 *    mm          - mean anomaly
 *    xni         -
 *    nodem       - right ascension of ascending node
 *    dndt        -
 *    nm          - mean motion
 *
 *  locals        :
 *    delt        -
 *    ft          -
 *    theta       -
 *    x2li        -
 *    x2omi       -
 *    xl          -
 *    xldot       -
 *    xnddt       -
 *    xndt        -
 *    xomi        -
 *
 *  coupling      :
 *    none        -
 *
 *  references    :
 *    hoots, roehrich, norad spacetrack report #3 1980
 *    hoots, norad spacetrack report #6 1986
 *    hoots, schumacher and glover 2004
 *    vallado, crawford, hujsak, kelso  2006
----------------------------------------------------------------------------*/

static void dspace(
	int irez,
	double d2201,  double d2211,  double d3210,   double d3222,  double d4410,
	double d4422,  double d5220,  double d5232,   double d5421,  double d5433,
	double dedt,   double del1,   double del2,    double del3,   double didt,
	double dmdt,   double dnodt,  double domdt,   double argpo,  double argpdot,
	double t,      double tc,     double gsto,    double xfact,  double xlamo,
	double no,
	double *atime, double *em,    double *argpm,  double *inclm, double *xli,
	double *mm,    double *xni,   double *nodem,  double *dndt,  double *nm )
{
  double  ft, theta, fasx2, fasx4, fasx6, step2, stepn, stepp;

  fasx2 = 0.13130908;
  fasx4 = 2.8843198;
  fasx6 = 0.37448087;
  stepp = 720.0;
  stepn = -720.0;
  step2 = 259200.0;

  /* ----------- calculate deep space resonance effects ----------- */
  *dndt  = 0.0;
  theta  = fmod(gsto + tc * rptim, twopi);
  *em    = *em + dedt * t;

  *inclm  = *inclm + didt * t;
  *argpm  = *argpm + domdt * t;
  *nodem  = *nodem + dnodt * t;
  *mm     = *mm + dmdt * t;

  //   sgp4fix for negative inclinations
  //   the following if statement should be commented out
  //  if (inclm < 0.0)
  // {
  //    inclm = -inclm;
  //    argpm = argpm - pi;
  //    nodem = nodem + pi;
  //  }

  /* - update resonances : numerical (euler-maclaurin) integration - */
  /* ------------------------- epoch restart ----------------------  */
  //   sgp4fix for propagator problems
  //   the following integration works for negative time steps and periods
  //   the specific changes are unknown because the original code was so convoluted

  ft     = 0.0;
  *atime = 0.0;
  if (irez != 0)
  {
	int iretn, iret;
	double delt, x2li, x2omi, xl, xldot = 0.0, xnddt = 0.0, xndt = 0.0, xomi;

	if ((*atime == 0.0) || ((t >= 0.0) && (*atime < 0.0)) ||
		((t < 0.0) && (*atime >= 0.0)))
	{
	  if (t >= 0.0)
		delt = stepp;
	  else
		delt = stepn;
	  *atime = 0.0;
	  *xni   = no;
	  *xli   = xlamo;
	}
	iretn = 381; // added for do loop
	iret  =   0; // added for loop
	while (iretn == 381)
	{
	  if ((fabs(t) < fabs(*atime)) || (iret == 351))
	  {
		if (t >= 0.0)
		  delt = stepn;
		else
		  delt = stepp;
		iret  = 351;
		iretn = 381;
	  }
	  else
	  {
		if (t > 0.0)  // error if prev if has atime:=0.0 and t:=0.0 (ge)
		  delt = stepp;
		else
		  delt = stepn;
		if (fabs(t - *atime) >= stepp)
		{
		  iret  = 0;
		  iretn = 381;
		}
		else
		{
		  ft    = t - *atime;
		  iretn = 0;
		}
	  }

	  /* ------------------- dot terms calculated ------------- */
	  /* ----------- near - synchronous resonance terms ------- */
	  if (irez != 2)
	  {
		xndt  = del1 * sin(*xli - fasx2) + del2 * sin(2.0 * (*xli - fasx4)) +
		  del3 * sin(3.0 * (*xli - fasx6));
		xldot = *xni + xfact;
		xnddt = del1 * cos(*xli - fasx2) +
		  2.0 * del2 * cos(2.0 * (*xli - fasx4)) +
		  3.0 * del3 * cos(3.0 * (*xli - fasx6));
		xnddt = xnddt * xldot;
	  }
	  else
	  {
		/* --------- near - half-day resonance terms -------- */
		xomi  = argpo + argpdot * *atime;
		x2omi = xomi + xomi;
		x2li  = *xli + *xli;
		xndt  = d2201 * sin(x2omi + *xli - g22) + d2211 * sin(*xli - g22) +
		  d3210 * sin(xomi + *xli - g32)  + d3222 * sin(-xomi + *xli - g32)+
		  d4410 * sin(x2omi + x2li - g44)+ d4422 * sin(x2li - g44) +
		  d5220 * sin(xomi + *xli - g52)  + d5232 * sin(-xomi + *xli - g52)+
		  d5421 * sin(xomi + x2li - g54) + d5433 * sin(-xomi + x2li - g54);
		xldot = *xni + xfact;
		xnddt = d2201 * cos(x2omi + *xli - g22) + d2211 * cos(*xli - g22) +
		  d3210 * cos(xomi + *xli - g32) + d3222 * cos(-xomi + *xli - g32) +
		  d5220 * cos(xomi + *xli - g52) + d5232 * cos(-xomi + *xli - g52) +
		  2.0 * (d4410 * cos(x2omi + x2li - g44) +
			  d4422 * cos(x2li - g44) + d5421 * cos(xomi + x2li - g54) +
			  d5433 * cos(-xomi + x2li - g54));
		xnddt = xnddt * xldot;
	  }

	  /* ----------------------- integrator ------------------- */
	  if (iretn == 381)
	  {
		*xli   = *xli + xldot * delt + xndt * step2;
		*xni   = *xni + xndt * delt + xnddt * step2;
		*atime = *atime + delt;
	  }
	}  // while iretn = 381

	*nm = *xni + xndt * ft + xnddt * ft * ft * 0.5;
	xl  = *xli + xldot * ft + xndt * ft * ft * 0.5;
	if (irez != 1)
	{
	  *mm   = xl - 2.0 * *nodem + 2.0 * theta;
	  *dndt = *nm - no;
	}
	else
	{
	  *mm   = xl - *nodem - *argpm + theta;
	  *dndt = *nm - no;
	}
	*nm = no + *dndt;
  }

}  // end dsspace

/*-----------------------------------------------------------------------------
 *
 *                           procedure initl
 *
 *  this procedure initializes the spg4 propagator. all the initialization is
 *    consolidated here instead of having multiple loops inside other routines.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    ecco        - eccentricity                           0.0 - 1.0
 *    epoch       - epoch time in days from jan 0, 1950. 0 hr
 *    inclo       - inclination of satellite
 *    no          - mean motion of satellite
 *    satn        - satellite number
 *
 *  outputs       :
 *    ainv        - 1.0 / a
 *    ao          - semi major axis
 *    con41       -
 *    con42       - 1.0 - 5.0 cos(i)
 *    cosio       - cosine of inclination
 *    cosio2      - cosio squared
 *    eccsq       - eccentricity squared
 *    method      - flag for deep space                    'd', 'n'
 *    omeosq      - 1.0 - ecco * ecco
 *    posq        - semi-parameter squared
 *    rp          - radius of perigee
 *    rteosq      - square root of (1.0 - ecco*ecco)
 *    sinio       - sine of inclination
 *    gsto        - gst at time of observation               rad
 *    no          - mean motion of satellite
 *
 *  locals        :
 *    ak          -
 *    d1          -
 *    del         -
 *    adel        -
 *    po          -
 *
 *  coupling      :
 *    getgravconst
 *    gstime      - find greenwich sidereal time from the julian date
 *
 *  references    :
 *    hoots, roehrich, norad spacetrack report #3 1980
 *    hoots, norad spacetrack report #6 1986
 *    hoots, schumacher and glover 2004
 *    vallado, crawford, hujsak, kelso  2006
 ----------------------------------------------------------------------------*/

static void initl(
	double ecco,   double epoch,   double inclo,   double *no,
	char *method,
	double *ainv,  double *ao,     double *con41,  double *con42,
	double *cosio, double *cosio2, double *eccsq,  double *omeosq,
	double *posq,  double *rp,     double *rteosq, double *sinio ,
	double *gsto )
{
  /* --------------------- local variables ------------------------ */
  double ak, d1, del, adel, po;

  /* ------------- calculate auxillary epoch quantities ---------- */
  *eccsq  = ecco * ecco;
  *omeosq = 1.0 - *eccsq;
  *rteosq = sqrt(*omeosq);
  *cosio  = cos(inclo);
  *cosio2 = *cosio * *cosio;

  /* ------------------ un-kozai the mean motion ----------------- */
  ak    = pow(rc_data.xke / *no, tothrd);
  d1    = 0.75 * rc_data.j2 * (3.0 * *cosio2 - 1.0) / (*rteosq * *omeosq);
  del   = d1 / (ak * ak);
  adel  = ak * (1.0 - del * del - del *
	  (1.0 / 3.0 + 134.0 * del * del / 81.0));
  del   = d1/(adel * adel);
  *no   = *no / (1.0 + del);

  *ao     = pow(rc_data.xke / *no, tothrd);
  *sinio  = sin(inclo);
  po      = *ao * *omeosq;
  *con42  = 1.0 - 5.0 * *cosio2;
  *con41  = -*con42- *cosio2- *cosio2;
  *ainv   = 1.0 / *ao;
  *posq   = po * po;
  *rp     = *ao * (1.0 - ecco);
  *method = 'n';

  *gsto  = gstime(epoch + 2433281.5);

}  // end initl

/*-----------------------------------------------------------------------------
 *
 *                             procedure sgp4init
 *
 *  this procedure initializes variables for sgp4.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    satn        - satellite number
 *    bstar       - sgp4 type drag coefficient              kg/m2er
 *    ecco        - eccentricity
 *    epoch       - epoch time in days from jan 0, 1950. 0 hr
 *    argpo       - argument of perigee (output if ds)
 *    inclo       - inclination
 *    mo          - mean anomaly (output if ds)
 *    no          - mean motion
 *    nodeo       - right ascension of ascending node
 *
 *  outputs       :
 *    elem_set      - common values for subsequent calls
 *    return code - non-zero on error.
 *                   1 - mean elements, ecc >= 1.0 or ecc < -0.001 or a < 0.95 er
 *                   2 - mean motion less than 0.0
 *                   3 - pert elements, ecc < 0.0  or  ecc > 1.0
 *                   4 - semi-latus rectum < 0.0
 *                   5 - epoch elements are sub-orbital
 *                   6 - satellite has decayed
 *
 *  locals        :
 *    cnodm  , snodm  , cosim  , sinim  , cosomm , sinomm
 *    cc1sq  , cc2    , cc3
 *    coef   , coef1
 *    cosio4      -
 *    day         -
 *    dndt        -
 *    em          - eccentricity
 *    emsq        - eccentricity squared
 *    eeta        -
 *    etasq       -
 *    gam         -
 *    argpm       - argument of perigee
 *    nodem       -
 *    inclm       - inclination
 *    mm          - mean anomaly
 *    nm          - mean motion
 *    perige      - perigee
 *    pinvsq      -
 *    psisq       -
 *    qzms24      -
 *    rtemsq      -
 *    s1, s2, s3, s4, s5, s6, s7          -
 *    sfour       -
 *    ss1, ss2, ss3, ss4, ss5, ss6, ss7         -
 *    sz1, sz2, sz3
 *    sz11, sz12, sz13, sz21, sz22, sz23, sz31, sz32, sz33        -
 *    tc          -
 *    temp        -
 *    temp1, temp2, temp3       -
 *    tsi         -
 *    xpidot      -
 *    xhdot1      -
 *    z1, z2, z3          -
 *    z11, z12, z13, z21, z22, z23, z31, z32, z33         -
 *
 *  coupling      :
 *    getgravconst-
 *    initl       -
 *    dscom       -
 *    dpper       -
 *    dsinit      -
*    sgp4        -
*
*  references    :
*    hoots, roehrich, norad spacetrack report #3 1980
*    hoots, norad spacetrack report #6 1986
*    hoots, schumacher and glover 2004
*    vallado, crawford, hujsak, kelso  2006
----------------------------------------------------------------------------*/

int sgp4init( const int satn, const double epoch, elem_set_t *elem_set )
{
  /* --------------------- local variables ------------------------ */
  double
	ao  , ainv  , con42 , cosio , sinio, cosio2, eccsq ,
  omeosq, posq  , rp    , rteosq, cnodm, snodm , cosim ,
  sinim , cosomm, sinomm, day   , dndt , em    , emsq  ,
  gam   , argpm , nodem , inclm , mm   , nm    , rtemsq,
  s1    , s2    , s3    , s4    , s5   , s6    , s7    ,
  ss1   , ss2   , ss3   , ss4   , ss5  , ss6   , ss7   ,
  sz1   , sz2   , sz3   , sz11  , sz12 , sz13  , sz21  ,
  sz22  , sz23  , sz31  , sz32  , sz33 , z1    , z2    ,
  z3    , z11   , z12   , z13   , z21  , z22   , z23   ,
  z31   , z32   , z33   , qzms2t, ss;

  kinetics_t kin;

  /* ------------------------ initialization --------------------- */
  // sgp4fix divisor for divide by zero check on inclination
  const double temp4    =   1.0 + cos(pi-1.0e-9);

  /* ----------- set all near earth variables to zero ------------ */
  elem_set->isimp   = 0;   elem_set->method = 'n'; elem_set->aycof    = 0.0;
  elem_set->con41   = 0.0; elem_set->cc1    = 0.0; elem_set->cc4      = 0.0;
  elem_set->cc5     = 0.0; elem_set->d2     = 0.0; elem_set->d3       = 0.0;
  elem_set->d4      = 0.0; elem_set->delmo  = 0.0; elem_set->eta      = 0.0;
  elem_set->argpdot = 0.0; elem_set->omgcof = 0.0; elem_set->sinmao   = 0.0;
  elem_set->t       = 0.0; elem_set->t2cof  = 0.0; elem_set->t3cof    = 0.0;
  elem_set->t4cof   = 0.0; elem_set->t5cof  = 0.0; elem_set->x1mth2   = 0.0;
  elem_set->x7thm1  = 0.0; elem_set->mdot   = 0.0; elem_set->nodedot  = 0.0;
  elem_set->xlcof   = 0.0; elem_set->xmcof  = 0.0; elem_set->nodecf   = 0.0;

  /* ----------- set all deep space variables to zero ------------ */
  elem_set->irez  = 0;   elem_set->d2201 = 0.0; elem_set->d2211 = 0.0;
  elem_set->d3210 = 0.0; elem_set->d3222 = 0.0; elem_set->d4410 = 0.0;
  elem_set->d4422 = 0.0; elem_set->d5220 = 0.0; elem_set->d5232 = 0.0;
  elem_set->d5421 = 0.0; elem_set->d5433 = 0.0; elem_set->dedt  = 0.0;
  elem_set->del1  = 0.0; elem_set->del2  = 0.0; elem_set->del3  = 0.0;
  elem_set->didt  = 0.0; elem_set->dmdt  = 0.0; elem_set->dnodt = 0.0;
  elem_set->domdt = 0.0; elem_set->e3    = 0.0; elem_set->ee2   = 0.0;
  elem_set->peo   = 0.0; elem_set->pgho  = 0.0; elem_set->pho   = 0.0;
  elem_set->pinco = 0.0; elem_set->plo   = 0.0; elem_set->se2   = 0.0;
  elem_set->se3   = 0.0; elem_set->sgh2  = 0.0; elem_set->sgh3  = 0.0;
  elem_set->sgh4  = 0.0; elem_set->sh2   = 0.0; elem_set->sh3   = 0.0;
  elem_set->si2   = 0.0; elem_set->si3   = 0.0; elem_set->sl2   = 0.0;
  elem_set->sl3   = 0.0; elem_set->sl4   = 0.0; elem_set->gsto  = 0.0;
  elem_set->xfact = 0.0; elem_set->xgh2  = 0.0; elem_set->xgh3  = 0.0;
  elem_set->xgh4  = 0.0; elem_set->xh2   = 0.0; elem_set->xh3   = 0.0;
  elem_set->xi2   = 0.0; elem_set->xi3   = 0.0; elem_set->xl2   = 0.0;
  elem_set->xl3   = 0.0; elem_set->xl4   = 0.0; elem_set->xlamo = 0.0;
  elem_set->zmol  = 0.0; elem_set->zmos  = 0.0; elem_set->atime = 0.0;
  elem_set->xli   = 0.0; elem_set->xni   = 0.0;

  /* ------- init local variables (my addition) -------- */
  ss1  = ss2  = ss3  = ss4  = ss5  = ss6  = ss7  = 0.0;
  sz1  = sz2  = sz3  = sz11 = sz12 = sz13 = sz21 = 0.0;
  sz22 = sz23 = sz31 = sz32 = sz33 = 0.0;

  ss     = 78.0 / rc_data.radiusearthkm + 1.0;
  qzms2t = pow(((120.0 - 78.0) / rc_data.radiusearthkm), 4);
  elem_set->init = 'y';
  elem_set->t	 = 0.0;

  initl(
	  elem_set->ecco, epoch, elem_set->inclo, &elem_set->no,
	  &elem_set->method, &ainv, &ao, &elem_set->con41, &con42,
	  &cosio, &cosio2, &eccsq, &omeosq, &posq, &rp, &rteosq,
	  &sinio, &elem_set->gsto );
  elem_set->error = 0;

  if (rp < 1.0)
  {
	printf("# *** Satellite #%d epoch elts sub-orbital ***\n", satn);
	elem_set->error = 5;
  }

  if ((omeosq >= 0.0 ) || ( elem_set->no >= 0.0))
  {
	double
	  cc2  , cc3   , coef  , coef1 , cosio4, eeta ,
	  etasq, perige, pinvsq, psisq , qzms24, sfour,
	  temp1, temp2 , temp3 , tsi   , xpidot, xhdot1;

	elem_set->isimp = 0;
	if (rp < (220.0 / rc_data.radiusearthkm + 1.0))
	  elem_set->isimp = 1;
	sfour  = ss;
	qzms24 = qzms2t;
	perige = (rp - 1.0) * rc_data.radiusearthkm;

	/* - for perigees below 156 km, s and qoms2t are altered - */
	if (perige < 156.0)
	{
	  sfour = perige - 78.0;
	  if (perige < 98.0)
		sfour = 20.0;
	  qzms24 = pow(((120.0 - sfour) / rc_data.radiusearthkm), 4.0);
	  sfour  = sfour / rc_data.radiusearthkm + 1.0;
	}
	pinvsq = 1.0 / posq;

	tsi  = 1.0 / (ao - sfour);
	elem_set->eta  = ao * elem_set->ecco * tsi;
	etasq = elem_set->eta * elem_set->eta;
	eeta  = elem_set->ecco * elem_set->eta;
	psisq = fabs(1.0 - etasq);
	coef  = qzms24 * pow(tsi, 4.0);
	coef1 = coef / pow(psisq, 3.5);
	cc2   = coef1 * elem_set->no * (ao * (1.0 + 1.5 * etasq + eeta *
		  (4.0 + etasq)) + 0.375 * rc_data.j2 * tsi / psisq * elem_set->con41 *
		(8.0 + 3.0 * etasq * (8.0 + etasq)));
	elem_set->cc1   = elem_set->bstar * cc2;
	cc3   = 0.0;
	if (elem_set->ecco > 1.0e-4)
	  cc3 = -2.0 * coef * tsi * rc_data.j3oj2 * elem_set->no * sinio / elem_set->ecco;
	elem_set->x1mth2 = 1.0 - cosio2;
	elem_set->cc4    = 2.0* elem_set->no * coef1 * ao * omeosq *
	  (elem_set->eta * (2.0 + 0.5 * etasq) + elem_set->ecco *
	   (0.5 + 2.0 * etasq) - rc_data.j2 * tsi / (ao * psisq) *
	   (-3.0 * elem_set->con41 * (1.0 - 2.0 * eeta + etasq *
							   (1.5 - 0.5 * eeta)) + 0.75 * elem_set->x1mth2 *
		(2.0 * etasq - eeta * (1.0 + etasq)) * cos(2.0 * elem_set->argpo)));
	elem_set->cc5 = 2.0 * coef1 * ao * omeosq * (1.0 + 2.75 *
		(etasq + eeta) + eeta * etasq);
	cosio4 = cosio2 * cosio2;
	temp1  = 1.5 * rc_data.j2 * pinvsq * elem_set->no;
	temp2  = 0.5 * temp1 * rc_data.j2 * pinvsq;
	temp3  = -0.46875 * rc_data.j4 * pinvsq * pinvsq * elem_set->no;
	elem_set->mdot     = elem_set->no + 0.5 * temp1 * rteosq * elem_set->con41 + 0.0625 *
	  temp2 * rteosq * (13.0 - 78.0 * cosio2 + 137.0 * cosio4);
	elem_set->argpdot  = -0.5 * temp1 * con42 + 0.0625 * temp2 *
	  (7.0 - 114.0 * cosio2 + 395.0 * cosio4) +
	  temp3 * (3.0 - 36.0 * cosio2 + 49.0 * cosio4);
	xhdot1            = -temp1 * cosio;
	elem_set->nodedot = xhdot1 + (0.5 * temp2 * (4.0 - 19.0 * cosio2) +
		2.0 * temp3 * (3.0 - 7.0 * cosio2)) * cosio;
	xpidot            =  elem_set->argpdot+ elem_set->nodedot;
	elem_set->omgcof   = elem_set->bstar * cc3 * cos(elem_set->argpo);
	elem_set->xmcof    = 0.0;
	if (elem_set->ecco > 1.0e-4)
	  elem_set->xmcof = -tothrd * coef * elem_set->bstar / eeta;
	elem_set->nodecf = 3.5 * omeosq * xhdot1 * elem_set->cc1;
	elem_set->t2cof   = 1.5 * elem_set->cc1;
	// sgp4fix for divide by zero with xinco = 180 deg
	if (fabs(cosio+1.0) > 1.5e-12)
	  elem_set->xlcof = -0.25 * rc_data.j3oj2 * sinio * (3.0 + 5.0 * cosio) /
		(1.0 + cosio);
	else
	  elem_set->xlcof = -0.25 * rc_data.j3oj2 * sinio * (3.0 + 5.0 * cosio) /
		temp4;
	elem_set->aycof   = -0.5 * rc_data.j3oj2 * sinio;
	elem_set->delmo   = pow((1.0 + elem_set->eta * cos(elem_set->mo)), 3);
	elem_set->sinmao  = sin(elem_set->mo);
	elem_set->x7thm1  = 7.0 * cosio2 - 1.0;

	/* --------------- deep space initialization ------------- */
	if ((twopi / elem_set->no) >= 225.0)
	{
	  double tc;

	  elem_set->method = 'd';
	  elem_set->isimp  = 1;
	  tc    =  0.0;
	  inclm = elem_set->inclo;

	  dscom(
		 epoch, elem_set->ecco, elem_set->argpo, tc, elem_set->inclo, elem_set->nodeo,
		 elem_set->no, &snodm, &cnodm,  &sinim, &cosim, &sinomm, &cosomm,
		 &day, &elem_set->e3, &elem_set->ee2, &em, &emsq, &gam,
		 &elem_set->peo,  &elem_set->pgho, &elem_set->pho, &elem_set->pinco,
		 &elem_set->plo,  &rtemsq, &elem_set->se2, &elem_set->se3,
		 &elem_set->sgh2, &elem_set->sgh3,   &elem_set->sgh4,
		 &elem_set->sh2,  &elem_set->sh3,    &elem_set->si2, &elem_set->si3,
		 &elem_set->sl2,  &elem_set->sl3,    &elem_set->sl4, &s1, &s2, &s3, &s4, &s5,
		 &s6, &s7, &ss1, &ss2, &ss3, &ss4, &ss5, &ss6, &ss7, &sz1, &sz2, &sz3,
		 &sz11, &sz12, &sz13, &sz21, &sz22, &sz23, &sz31, &sz32, &sz33,
		 &elem_set->xgh2, &elem_set->xgh3, &elem_set->xgh4, &elem_set->xh2,
		 &elem_set->xh3, &elem_set->xi2, &elem_set->xi3, &elem_set->xl2,
		 &elem_set->xl3, &elem_set->xl4, &nm, &z1, &z2, &z3, &z11,
		 &z12, &z13, &z21, &z22, &z23, &z31, &z32, &z33,
		 &elem_set->zmol, &elem_set->zmos );

	  dpper(
		 elem_set->e3, elem_set->ee2, elem_set->peo, elem_set->pgho,
		 elem_set->pho, elem_set->pinco, elem_set->plo, elem_set->se2,
		 elem_set->se3, elem_set->sgh2, elem_set->sgh3, elem_set->sgh4,
		 elem_set->sh2, elem_set->sh3, elem_set->si2,  elem_set->si3,
		 elem_set->sl2, elem_set->sl3, elem_set->sl4,  elem_set->t,
		 elem_set->xgh2,elem_set->xgh3,elem_set->xgh4, elem_set->xh2,
		 elem_set->xh3, elem_set->xi2, elem_set->xi3,  elem_set->xl2,
		 elem_set->xl3, elem_set->xl4, elem_set->zmol, elem_set->zmos,
		 elem_set->init, &elem_set->ecco, &elem_set->inclo,
		 &elem_set->nodeo, &elem_set->argpo, &elem_set->mo );

	  argpm  = 0.0;
	  nodem  = 0.0;
	  mm     = 0.0;
	  dsinit(
		 cosim, emsq, elem_set->argpo, s1, s2, s3, s4, s5, sinim, ss1, ss2, ss3, ss4,
		 ss5, sz1, sz3, sz11, sz13, sz21, sz23, sz31, sz33, elem_set->t, tc,
		 elem_set->gsto, elem_set->mo, elem_set->mdot, elem_set->no, elem_set->nodeo,
		 elem_set->nodedot, xpidot, z1, z3, z11, z13, z21, z23, z31, z33,
		 elem_set->ecco, eccsq, &em, &argpm, &inclm, &mm, &nm, &nodem,
		 &elem_set->irez, &elem_set->atime,
		 &elem_set->d2201, &elem_set->d2211, &elem_set->d3210, &elem_set->d3222,
		 &elem_set->d4410, &elem_set->d4422, &elem_set->d5220, &elem_set->d5232,
		 &elem_set->d5421, &elem_set->d5433, &elem_set->dedt,  &elem_set->didt,
		 &elem_set->dmdt,  &dndt,         &elem_set->dnodt, &elem_set->domdt ,
		 &elem_set->del1, &elem_set->del2, &elem_set->del3, &elem_set->xfact,
		 &elem_set->xlamo, &elem_set->xli,   &elem_set->xni );
	}

	/* ----------- set variables if not deep space ----------- */
	if (elem_set->isimp != 1)
	{
	  double cc1sq, temp;

	  cc1sq        = elem_set->cc1 * elem_set->cc1;
	  elem_set->d2 = 4.0 * ao * tsi * cc1sq;
	  temp         = elem_set->d2 * tsi * elem_set->cc1 / 3.0;
	  elem_set->d3 = (17.0 * ao + sfour) * temp;
	  elem_set->d4 = 0.5 * temp * ao * tsi * (221.0 * ao + 31.0 * sfour) *
		elem_set->cc1;
	  elem_set->t3cof = elem_set->d2 + 2.0 * cc1sq;
	  elem_set->t4cof = 0.25 * (3.0 * elem_set->d3 + elem_set->cc1 *
		  (12.0 * elem_set->d2 + 10.0 * cc1sq));
	  elem_set->t5cof = 0.2 * (3.0 * elem_set->d4 +
		  12.0 * elem_set->cc1 * elem_set->d3 +
		  6.0 * elem_set->d2 * elem_set->d2 +
		  15.0 * cc1sq * (2.0 * elem_set->d2 + cc1sq));
	}
  } // if omeosq = 0 ...

  /* finally propogate to zero epoch to initialise all others. */
  if(elem_set->error == 0)
	sgp4( elem_set, 0.0, &kin );

  elem_set->init = 'n';

  return elem_set->error;
}  // end sgp4init

/*-----------------------------------------------------------------------------
 *
 *                             procedure sgp4
 *
 *  this procedure is the sgp4 prediction model from space command. this is an
 *  updated and combined version of sgp4 and sdp4, which were originally
 *  published separately in spacetrack report #3. this version follows the nasa
 *  release on the internet. there are a few fixes that are added to correct
 *  known errors in the existing implementations.
 *
 *  author        : david vallado                  719-573-2600   28 jun 2005
 *
 *  inputs        :
 *    elem_set	 - initialised structure from sgp4init() call.
 *    tsince	 - time eince epoch (minutes)
 *
 *  outputs       :
 *    r           - position vector                     km
 *    v           - velocity                            km/sec
 *  return code - non-zero on error.
 *                   1 - mean elements, ecc >= 1.0 or ecc < -0.001 or a < 0.95 er
 *                   2 - mean motion less than 0.0
 *                   3 - pert elements, ecc < 0.0  or  ecc > 1.0
 *                   4 - semi-latus rectum < 0.0
 *                   5 - epoch elements are sub-orbital
 *                   6 - satellite has decayed
 *
 *  locals        :
 *    am          -
 *    axnl, aynl        -
 *    betal       -
 *    cosim   , sinim   , cosomm  , sinomm  , cnod    , snod    , cos2u   ,
 *    sin2u   , coseo1  , sineo1  , cosi    , sini    , cosip   , sinip   ,
 *    cosisq  , cossu   , sinsu   , cosu    , sinu
 *    delm        -
 *    delomg      -
 *    dndt        -
 *    eccm        -
 *    emsq        -
 *    ecose       -
 *    el2         -
 *    eo1         -
 *    eccp        -
 *    esine       -
 *    argpm       -
 *    argpp       -
 *    omgadf      -
 *    pl          -
 *    r           -
 *    rtemsq      -
 *    rdotl       -
 *    rl          -
 *    rvdot       -
 *    rvdotl      -
 *    su          -
 *    t2  , t3   , t4    , tc
 *    tem5, temp , temp1 , temp2  , tempa  , tempe  , templ
 *    u   , ux   , uy    , uz     , vx     , vy     , vz
 *    inclm       - inclination
 *    mm          - mean anomaly
 *    nm          - mean motion
 *    nodem       - right asc of ascending node
 *    xinc        -
 *    xincp       -
 *    xl          -
 *    xlm         -
 *    mp          -
 *    xmdf        -
 *    xmx         -
 *    xmy         -
 *    nodedf      -
*    xnode       -
*    nodep       -
*    np          -
*
*  coupling      :
*    getgravconst-
*    dpper
*    dpspace
*
*  references    :
*    hoots, roehrich, norad spacetrack report #3 1980
*    hoots, norad spacetrack report #6 1986
*    hoots, schumacher and glover 2004
*    vallado, crawford, hujsak, kelso  2006
----------------------------------------------------------------------------*/

int sgp4( elem_set_t *elem_set, double tsince,  kinetics_t *kinetics )
{
  double
	am    , axnl ,  aynl, cosim , coseo1 = 0.0, cosip, delm  ,
	delomg, em   , ecose, el2   , eo1  , ep    ,
	esine , argpm, argpp, argpdf, pl    , mrt  , sinim ,
	sineo1 = 0.0, sinip, t2   , t3    , t4    , tem5 , temp  ,
	tempa , tempe, templ, u     , inclm , mm   , nm    ,
	nodem , xincp, xl   , xlm   , mp    , xmdf , nodedf,
	nodep , tc   , dndt; //, emsq;

  int ktr;

  /* ------------------ set mathematical constants --------------- */
  // sgp4fix divisor for divide by zero check on inclination
  const double temp4 = 1.0 + cos(pi-1.0e-9);
  mrt = 0.0;

  /* --------------------- clear sgp4 error flag ----------------- */
  elem_set->t     = tsince;
  elem_set->error = 0;

  /* ------- update for secular gravity and atmospheric drag ----- */
  xmdf    = elem_set->mo + elem_set->mdot * elem_set->t;
  argpdf  = elem_set->argpo + elem_set->argpdot * elem_set->t;
  nodedf  = elem_set->nodeo + elem_set->nodedot * elem_set->t;
  argpm   = argpdf;
  mm      = xmdf;
  t2      = elem_set->t * elem_set->t;
  nodem   = nodedf + elem_set->nodecf * t2;
  tempa   = 1.0 - elem_set->cc1 * elem_set->t;
  tempe   = elem_set->bstar * elem_set->cc4 * elem_set->t;
  templ   = elem_set->t2cof * t2;

  if (elem_set->isimp != 1)
  {
	delomg = elem_set->omgcof * elem_set->t;
	delm   = elem_set->xmcof *
	  (pow((1.0 + elem_set->eta * cos(xmdf)), 3) - elem_set->delmo);
	temp   = delomg + delm;
	mm     = xmdf + temp;
	argpm  = argpdf - temp;
	t3     = t2 * elem_set->t;
	t4     = t3 * elem_set->t;
	tempa  = tempa - elem_set->d2 * t2 - elem_set->d3 * t3 -
	  elem_set->d4 * t4;
	tempe  = tempe + elem_set->bstar * elem_set->cc5 * (sin(mm) -
		elem_set->sinmao);
	templ  = templ + elem_set->t3cof * t3 + t4 * (elem_set->t4cof +
		elem_set->t * elem_set->t5cof);
  }

  nm    = elem_set->no;
  em    = elem_set->ecco;
  inclm = elem_set->inclo;
  if (elem_set->method == 'd')
  {
	tc = elem_set->t;
	dspace(
	   elem_set->irez,
	   elem_set->d2201, elem_set->d2211,   elem_set->d3210,
	   elem_set->d3222, elem_set->d4410,   elem_set->d4422,
	   elem_set->d5220, elem_set->d5232,   elem_set->d5421,
	   elem_set->d5433, elem_set->dedt,    elem_set->del1,
	   elem_set->del2,  elem_set->del3,    elem_set->didt,
	   elem_set->dmdt,  elem_set->dnodt,   elem_set->domdt,
	   elem_set->argpo, elem_set->argpdot, elem_set->t, tc,
	   elem_set->gsto,  elem_set->xfact,   elem_set->xlamo,
	   elem_set->no,    &elem_set->atime,
	   &em,             &argpm,            &inclm, &elem_set->xli,
	   &mm,             &elem_set->xni,	   &nodem,
	   &dndt,           &nm );
  } // if method = d

  if (nm <= 0.0)
  {
	printf("# Error: nm %f\n", nm);
	elem_set->error = 2;
  }
  am = pow((rc_data.xke / nm), tothrd) * tempa * tempa;
  nm = rc_data.xke / pow(am, 1.5);
  em = em - tempe;

  // fix tolerance for error recognition
  if ((em >= 1.0) || (em < -0.001) || (am < 0.95))
  {
	printf("# Error: em %f\n", em);
	elem_set->error = 1;
  }
  if (em < 0.0)	em = 1.0e-6;
  mm     = mm + elem_set->no * templ;
  xlm    = mm + argpm + nodem;
  //emsq   = em * em;
  //temp   = 1.0 - emsq;

  nodem  = fmod(nodem, twopi);
  argpm  = fmod(argpm, twopi);
  xlm    = fmod(xlm, twopi);
  mm     = fmod(xlm - argpm - nodem, twopi);

  /* ----------------- compute extra mean quantities ------------- */
  sinim = sin(inclm);
  cosim = cos(inclm);

  /* -------------------- add lunar-solar periodics -------------- */
  ep     = em;
  xincp  = inclm;
  argpp  = argpm;
  nodep  = nodem;
  mp     = mm;
  sinip  = sinim;
  cosip  = cosim;
  if (elem_set->method == 'd')
  {
	dpper(
	   elem_set->e3,   elem_set->ee2,  elem_set->peo,
	   elem_set->pgho, elem_set->pho,  elem_set->pinco,
	   elem_set->plo,  elem_set->se2,  elem_set->se3,
	   elem_set->sgh2, elem_set->sgh3, elem_set->sgh4,
	   elem_set->sh2,  elem_set->sh3,  elem_set->si2,
	   elem_set->si3,  elem_set->sl2,  elem_set->sl3,
	   elem_set->sl4,  elem_set->t,    elem_set->xgh2,
	   elem_set->xgh3, elem_set->xgh4, elem_set->xh2,
	   elem_set->xh3,  elem_set->xi2,  elem_set->xi3,
	   elem_set->xl2,  elem_set->xl3,  elem_set->xl4,
	   elem_set->zmol, elem_set->zmos,
	   'n', &ep, &xincp, &nodep, &argpp, &mp );
	if (xincp < 0.0)
	{
	  xincp  = -xincp;
	  nodep = nodep + pi;
	  argpp  = argpp - pi;
	}
	if ((ep < 0.0 ) || ( ep > 1.0))
	{
	  printf("# Error: ep %f\n", ep);
	  elem_set->error = 3;
	}
  } // if method = d

  /* -------------------- long period periodics ------------------ */
  if (elem_set->method == 'd')
  {
	sinip =  sin(xincp);
	cosip =  cos(xincp);
	elem_set->aycof = -0.5*rc_data.j3oj2*sinip;
	// sgp4fix for divide by zero for xincp = 180 deg
	if (fabs(cosip+1.0) > 1.5e-12)
	  elem_set->xlcof = -0.25 * rc_data.j3oj2 * sinip * (3.0 + 5.0 * cosip) /
		(1.0 + cosip);
	else
	  elem_set->xlcof = -0.25 * rc_data.j3oj2 * sinip * (3.0 + 5.0 * cosip) /
		temp4;
  }
  axnl = ep * cos(argpp);
  temp = 1.0 / (am * (1.0 - ep * ep));
  aynl = ep* sin(argpp) + temp * elem_set->aycof;
  xl   = mp + argpp + nodep + temp * elem_set->xlcof * axnl;

  /* --------------------- solve kepler's equation --------------- */
  u    = fmod(xl - nodep, twopi);
  eo1  = u;
  tem5 = 9999.9;
  ktr  = 1;
  //   sgp4fix for kepler iteration
  //   the following iteration needs better limits on corrections
  while( (fabs(tem5) >= 1.0e-12) && (ktr <= 10) )
  {
	sineo1 = sin(eo1);
	coseo1 = cos(eo1);
	tem5   = 1.0 - coseo1 * axnl - sineo1 * aynl;
	tem5   = (u - aynl * coseo1 + axnl * sineo1 - eo1) / tem5;
	if(fabs(tem5) >= 0.95)
	  tem5 = tem5 > 0.0 ? 0.95 : -0.95;
	eo1    = eo1 + tem5;
	ktr    = ktr + 1;
  }

  /* ------------- short period preliminary quantities ----------- */
  ecose = axnl*coseo1 + aynl*sineo1;
  esine = axnl*sineo1 - aynl*coseo1;
  el2   = axnl*axnl + aynl*aynl;
  pl    = am*(1.0-el2);
  if (pl < 0.0)
  {
	printf("# Error: pl %f\n", pl);
	elem_set->error = 4;
  }
  else
  {
	double
	  betal, cnod , cos2u, cosi , cossu , cosu ,
	  mvt  , rdotl, rl   , rvdot, rvdotl, sin2u,
	  sini , sinsu, sinu , snod , su    , temp1,
	  temp2, ux   , uy   , uz   , vx    , vy   ,
	  vz   , xinc , xmx  , xmy  , xnode;

	rl     = am * (1.0 - ecose);
	rdotl  = sqrt(am) * esine/rl;
	rvdotl = sqrt(pl) / rl;
	betal  = sqrt(1.0 - el2);
	temp   = esine / (1.0 + betal);
	sinu   = am / rl * (sineo1 - aynl - axnl * temp);
	cosu   = am / rl * (coseo1 - axnl + aynl * temp);
	su     = atan2(sinu, cosu);
	sin2u  = (cosu + cosu) * sinu;
	cos2u  = 1.0 - 2.0 * sinu * sinu;
	temp   = 1.0 / pl;
	temp1  = 0.5 * rc_data.j2 * temp;
	temp2  = temp1 * temp;

	/* -------------- update for short period periodics ------------ */
	if (elem_set->method == 'd')
	{
	  double cosisq    = cosip * cosip;
	  elem_set->con41  = 3.0*cosisq - 1.0;
	  elem_set->x1mth2 = 1.0 - cosisq;
	  elem_set->x7thm1 = 7.0*cosisq - 1.0;
	}
	mrt   = rl * (1.0 - 1.5 * temp2 * betal * elem_set->con41) +
	  0.5 * temp1 * elem_set->x1mth2 * cos2u;
	su    = su - 0.25 * temp2 * elem_set->x7thm1 * sin2u;
	xnode = nodep + 1.5 * temp2 * cosip * sin2u;
	xinc  = xincp + 1.5 * temp2 * cosip * sinip * cos2u;
	mvt   = rdotl - nm * temp1 * elem_set->x1mth2 * sin2u / rc_data.xke;
	rvdot = rvdotl + nm * temp1 * (elem_set->x1mth2 * cos2u +
		1.5 * elem_set->con41) / rc_data.xke;

	/* --------------------- orientation vectors ------------------- */
	sinsu =  sin(su);
	cossu =  cos(su);
	snod  =  sin(xnode);
	cnod  =  cos(xnode);
	sini  =  sin(xinc);
	cosi  =  cos(xinc);
	xmx   = -snod * cosi;
	xmy   =  cnod * cosi;
	ux    =  xmx * sinsu + cnod * cossu;
	uy    =  xmy * sinsu + snod * cossu;
	uz    =  sini * sinsu;
	vx    =  xmx * cossu - cnod * sinsu;
	vy    =  xmy * cossu - snod * sinsu;
	vz    =  sini * cossu;

	/* Position and velocity */
	kinetics->pos.x = mrt * ux;
	kinetics->pos.y = mrt * uy;
	kinetics->pos.z = mrt * uz;
	kinetics->vel.x = (mvt * ux + rvdot * vx)*rc_data.xke;
	kinetics->vel.y = (mvt * uy + rvdot * vy)*rc_data.xke;
	kinetics->vel.z = (mvt * uz + rvdot * vz)*rc_data.xke;

	/* Phase in rads */
	kinetics->phase = xl - nodep - argpdf + twopi;
	if(kinetics->phase < 0)
	  kinetics->phase += twopi;
	kinetics->phase = FMod2p(kinetics->phase);

  }  // if pl > 0

  // sgp4fix for decaying satellites
  if (mrt < 1.0)
  {
	printf("# Decay condition: %11.6f \n", mrt);
	elem_set->error = 6;
  }

  if( elem_set->error )
  {
	Error_Dialog( Abort_On_Error(-23), FATAL );
	return elem_set->error;
  }

  return elem_set->error;

}  // end sgp4

/* -----------------------------------------------------------------------------
 *
 *                           function gstime
 *
 *  this function finds the greenwich sidereal time.
 *
 *  author        : david vallado                  719-573-2600    1 mar 2001
 *
 *  inputs          description                    range / units
 *    jdut1       - julian date in ut1             days from 4713 bc
 *
 *  outputs       :
 *    gstime      - greenwich sidereal time        0 to 2pi rad
 *
 *  locals        :
 *    temp        - temporary variable for doubles   rad
 *    tut1        - julian centuries from the
 *                  jan 1, 2000 12 h epoch (ut1)
 *
 *  coupling      :
 *    none
 *
 *  references    :
 *    vallado       2004, 191, eq 3-45
 * --------------------------------------------------------------------------- */

double gstime( double jdut1 )
{
  const double deg2rad = pi / 180.0;
  double       temp, tut1;

  tut1 = (jdut1 - 2451545.0) / 36525.0;
  temp = -6.2e-6* tut1 * tut1 * tut1 + 0.093104 * tut1 * tut1 +
	(876600.0*3600 + 8640184.812866) * tut1 + 67310.54841;  // sec
  temp = fmod(temp * deg2rad / 240.0, twopi); //360/86400 = 1/240, to deg, to rad

  // ------------------------ check quadrants ---------------------
  if (temp < 0.0) temp += twopi;

  return temp;
}  // end gstime

/* -----------------------------------------------------------------------------
 *
 *                           function getgravconst
 *
 *  this function gets constants for the propagator. note that mu is identified to
 *    facilitiate comparisons with newer models.
 *
 *  author        : david vallado                  719-573-2600   21 jul 2006
 *
 *  inputs        :
 *    whichconst  - which set of constants to use  72, 84
 *
 *  outputs       :
 *    tumin       - minutes in one time unit
 *    radiusearthkm - radius of the earth in km
 *    xke         - reciprocal of tumin
 *    j2, j3, j4  - un-normalized zonal harmonic values
 *    j3oj2       - j3 divided by j2
 *
 *  locals        :
 *    mu          - earth gravitational parameter
 *
 *  coupling      :
 *    none
 *
 *  references    :
 *    norad spacetrack report #3
 *    vallado, crawford, hujsak, kelso  2006
 --------------------------------------------------------------------------- */

void getgravconst( void )
{
  double mu;

  switch (rc_data.wgf)
  {
	// -- wgs-72 low precision str#3 constants --
	case 0:
	  rc_data.radiusearthkm = 6378.135;     // km
	  rc_data.xke    = 0.0743669161;
	  rc_data.tumin  = 1.0 / rc_data.xke;
	  rc_data.j2     =   0.001082616;
	  rc_data.j3     =  -0.00000253881;
	  rc_data.j4     =  -0.00000165597;
	  rc_data.j3oj2  =  rc_data.j3 / rc_data.j2;
	  break;

	  // ------------ wgs-72 constants ------------
	case 1:
	  mu = 398600.8;   // in km3 / s2
	  rc_data.radiusearthkm = 6378.135;  // km
	  rc_data.xke    = 60.0 / sqrt(rc_data.radiusearthkm *
		  rc_data.radiusearthkm * rc_data.radiusearthkm/mu);
	  rc_data.tumin  = 1.0 / rc_data.xke;
	  rc_data.j2     =   0.001082616;
	  rc_data.j3     =  -0.00000253881;
	  rc_data.j4     =  -0.00000165597;
	  rc_data.j3oj2  =  rc_data.j3 / rc_data.j2;
	  break;

	  // ------------ wgs-84 constants ------------
	case 2:
	  mu  = 398600.5;  // in km3 / s2
	  rc_data.radiusearthkm = 6378.137;  // km
	  rc_data.xke    = 60.0 / sqrt(rc_data.radiusearthkm *
		  rc_data.radiusearthkm * rc_data.radiusearthkm/mu);
	  rc_data.tumin  = 1.0 / rc_data.xke;
	  rc_data.j2     =   0.00108262998905;
	  rc_data.j3     =  -0.00000253215306;
	  rc_data.j4     =  -0.00000161098761;
	  rc_data.j3oj2  =  rc_data.j3 / rc_data.j2;
  }

}   // end getgravconst

