/* plots.c
 *
 * Code for plotting graphs of structure
 * data (VSWR, impedance, gain etc)
 */

/*
 *  xsatcom: A X/GTK+ application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "plots.h"

/*-----------------------------------------------------------------------*/

/* Draw_Plotting_Frame()
 *
 * Draws a graph plotting frame, including
 * horizontal and vertical divisions
 */
void
Draw_Plotting_Frame(
	GtkWidget *drawingarea,
	GdkRectangle *rect,
	gchar *title,
	int nhor, int nvert )
{
  PangoLayout *layout;
  int idx, xpw, xps, yph, yps;
  int layout_width, layout_height; /* Layout size */
  int scale_width, scale_height;   /* Scale size  */
  char scale[4];

  /* Width and height of drawing area */
  gint width, height;

  /* Get size of drawing area */
  width  = drawingarea->allocation.width;
  height = drawingarea->allocation.height;

  /* cairo context */
  cairo_t *cr = gdk_cairo_create( drawingarea->window );
  cairo_set_source_rgb( cr, BLACK );
  cairo_rectangle( cr,
	  0.0, 0.0,
	  (gdouble)width,
	  (gdouble)height );
  cairo_fill( cr );

  /* Get size of left scale */
  layout = gtk_widget_create_pango_layout( drawingarea, "000" );
  pango_layout_get_pixel_size( layout, &scale_width, &scale_height );

  /* Draw titles */
  cairo_set_source_rgb( cr, GREY );
  pango_layout_set_text( layout, _("% Illum/Day"), -1 );
  cairo_move_to( cr, (gdouble)scale_width, 0.0 );
  pango_cairo_show_layout( cr, layout );

  pango_layout_set_text( layout, title, -1 );
  pango_layout_get_pixel_size( layout, &layout_width, &layout_height);
  xpw = scale_width + (width-layout_width)/2;
  cairo_move_to( cr, (gdouble)xpw, 0.0 );
  pango_cairo_show_layout( cr, layout );

  /* Move to plot box and divisions */
  xpw = width  - scale_width;
  yph = height - scale_height - 5;

  /* Plot box size */
  rect->height = height - scale_height - layout_height - 5;
  rect->width  = width - 2 * scale_width;
  rect->x = scale_width;
  rect->y = layout_height + rect->height/nhor;

  /* Draw vertical divisions */
  nvert--;
  for( idx = 0; idx <= nvert; idx++ )
  {
	xps = scale_width + (idx * rect->width)/nvert;
	snprintf( scale, sizeof(scale), "%3d", idx+1 );
	pango_layout_set_text( layout, scale, -1 );
	cairo_move_to( cr,
		(gdouble)(xps - scale_width/2),
		(gdouble)(height - scale_height) );
	pango_cairo_show_layout( cr, layout );
	cairo_move_to( cr, (gdouble)xps, (gdouble)layout_height );
	cairo_line_to( cr, (gdouble)xps, (gdouble)yph );
	cairo_stroke( cr );
  }

  /* Draw horizontal divisions and scale */
  for( idx = 1; idx <= nhor; idx++ )
  {
	yps = layout_height + (idx * rect->height) / nhor;
	snprintf( scale, sizeof(scale), "%3d", (nhor-idx)*10 );
	pango_layout_set_text( layout, scale, -1 );
	cairo_move_to( cr, 0, yps - scale_height );
	pango_cairo_show_layout( cr, layout );
	cairo_move_to( cr, (gdouble)scale_width, (gdouble)yps );
	cairo_line_to( cr, (gdouble)xpw, (gdouble)yps );
	cairo_stroke( cr );
  }

  /* Draw last (top) line */
  yps = layout_height;
	cairo_move_to( cr, (gdouble)scale_width, (gdouble)yps );
	cairo_line_to( cr, (gdouble)xpw, (gdouble)yps );
	cairo_stroke( cr );

  g_object_unref( layout );
  cairo_destroy( cr );

} /* Draw_Plotting_Frame() */

/*-----------------------------------------------------------------------*/

/* Draw_Graph()
 *
 * Plots a graph of a vs b
 */
void
Draw_Graph(
	GtkWidget *drawingarea,
	GdkRectangle *rect,
	double *a, double *b,
	double amax, double amin,
	double bmax, double bmin,
	int nval )
{
  double ra, rb;
  int idx;
  GdkPoint *points = NULL;

  /* Allocate plotting buffer */
  mem_alloc( (void **)&points,
	  sizeof( GdkPoint ) * (size_t)nval,
	  _("for points in Draw_Graph()") );
  if( points == NULL ) return;

  /* cairo context */
  cairo_t *cr = gdk_cairo_create( drawingarea->window );
  cairo_set_source_rgb( cr, YELLOW );

  /* Range of values to plot */
  ra = amax - amin;
  rb = bmax - bmin;

  /* Calculate points to plot */
  for( idx = 0; idx < nval; idx++ )
  {
	points[idx].x = rect->x +
	  (int)( (double)rect->width  * (b[idx]-bmin) / rb + 0.5 );
	points[idx].y = rect->y +
	  (int)( (double)rect->height * (amax-a[idx]) / ra + 0.5 );

	/* Plot a small rectangle at point */
	cairo_rectangle( cr,
		(gdouble)(points[idx].x-3),
		(gdouble)(points[idx].y-3),
		6.0, 6.0 );
	cairo_fill( cr );
  }

  Cairo_Draw_Lines( cr, points, nval );
  cairo_destroy( cr );
  free_ptr( (void **)&points );

} /* Draw_Graph() */

/*-----------------------------------------------------------------------*/

/* Cairo_Draw_Lines()
 *
 * Draws lines between points
 */
  void
Cairo_Draw_Lines( cairo_t *cr, GdkPoint *points, int npoints )
{
  int idx;

  cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
  for( idx = 1; idx < npoints; idx++ )
	cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );
  cairo_stroke( cr );
} /* Cairo_Draw_Line() */

/*-----------------------------------------------------------------------*/

