/*  gridloc.c
 *
 *  Functions to calculate a Maidenhead grid locator
 *  given longitude/latitude and vice versa. Also to
 *  calculate great circle bearing and distance to a
 *  second location given a grid locator or lon/lat
 */

/*
 *  xsatcom: A X/GTK+ application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "gridloc.h"
#include "shared.h"

/*--------------------------------------------------------------------*/

/*  Position_to_Gridloc()
 *
 *  Function to calculate a grid locator give lon/lat
 */

  void
Position_to_Grid_Locator( double lon, double lat, char *grid_loc )
{
  double temp;

  /* Adjust longitude and latitude to references for the   */
  /* Maidenhead grid locator system and roundup to 1/2 sec */
  lon += 180.0 + 1.3888888888E-4;
  lat +=  90.0 + 1.3888888888E-4;

  /* Calculate first letter of field */
  temp = lon / 20.0;
  grid_loc[0] = (char)((int)temp + 'A');

  /* Calculate first number of square */
  lon -= floor(temp) * 20.0;
  temp = lon / 2.0;
  grid_loc[2] = (char)((int)temp + '0');

  /* Calculate first letter of sub-square */
  lon -= floor(temp) * 2.0;
  temp = lon * 12.0;
  grid_loc[4] = (char)((int)temp + 'A');

  /* Calculate second letter of field */
  temp = lat / 10.0;
  grid_loc[1] = (char)((int)temp + 'A');

  /* Calculate second number of square */
  lat -= floor(temp) * 10.0;
  temp = lat / 1.0;
  grid_loc[3] = (char)((int)temp + '0');

  /* Calculate second letter of sub-square */
  lat -= floor(temp) * 1.0;
  temp = lat * 24.0;
  grid_loc[5] = (char)((int)temp + 'A');

  grid_loc[6] = '\0';

} /* End of Position_to_Grid_Locator() */

/*--------------------------------------------------------------------*/

/*  Grid_Locator_to_Position()
 *
 *  Function to calculate longitude/latitude
 *  given a position's grid locator
 */

  void
Grid_Locator_to_Position( char *grid_loc, double *lon, double *lat )
{
  *lon  = (grid_loc[0] - 'A') * 20.0;
  *lon += (grid_loc[2] - '0') * 2.0;
  *lon += (grid_loc[4] - 'A') / 12.0;
  *lon -= 180.0;

  *lat  = (grid_loc[1] - 'A') * 10.0;
  *lat += (grid_loc[3] - '0') * 1.0;
  *lat += (grid_loc[5] - 'A') / 24.0;
  *lat -= 90.0;

} /* End of Grid_Locator_to_Position() */

/*--------------------------------------------------------------------*/

/*  Degrees_to_DMS()
 *
 * Function to convert an angle in decimal degrees to D:M:S
 */

  void
Degrees_to_DMS( double deg, int *ideg, int *imin, int *isec )
{
  double temp;

  deg += 1.0/7200.0; /* Round-up to 0.5 sec */
  *ideg = (int)deg;
  temp = (  deg - (double)*ideg ) * 60.0;
  *imin = (int)temp;
  temp = ( temp - (double)*imin ) * 60.0;
  *isec = (int)(temp);

} /* End of Degrees_to_DMS() */

/*--------------------------------------------------------------------*/

/*  DMS_to_Degrees()
 *
 *  Function to convert an angle in D:M:S to decimal degrees
 */

  void
DMS_to_Degrees( int ideg, int imin, int isec, double *deg )
{
  *deg = (double)ideg + (double)imin/60.0 + (double)isec/3600.0;

} /* End of DMS_to_Degrees() */

/*--------------------------------------------------------------------*/

/*  Bearing_Distance()
 *
 *  Function to calculate bearing and
 *  distance of location B from A
 */

void
Bearing_Distance(
	double lon_a, double lat_a, /* Lon/Lat of point A */
	double lon_b, double lat_b, /* Lon/Lat of point B */
	double *bearing, double *distance )/* From A to B */
{
  double
	gc_arc, cos_gc_arc,       /* Great circle arc   A to B */
	cos_bearing, sin_bearing, /* cos/sin of bearing A to B */
	lon_diff;       /* Difference in longitude of B from A */

  /* Convert to radians */
  lat_a = Radians( lat_a );
  lon_a = Radians( lon_a );
  lat_b = Radians( lat_b );
  lon_b = Radians( lon_b );

  /* Longitude differnce of B from A */
  lon_diff = lon_b - lon_a;

  /* Calculate great circle distance A to B */
  cos_gc_arc = cos(lon_diff)*cos(lat_a)*cos(lat_b) + sin(lat_a)*sin(lat_b);
  gc_arc = acos( cos_gc_arc );

  /* Distance in km */
  *distance = rc_data.radiusearthkm * gc_arc;

  /* Calculate bearing A to B */
  cos_bearing  = sin(lat_b) - sin(lat_a) * cos_gc_arc;
  sin_bearing  = sin(lon_diff) * cos(lat_a) * cos(lat_b);
  *bearing = atan2(sin_bearing, cos_bearing);

  /* Correct negative (anticlockwise) bearings */
  if( *bearing < 0.0 )
	*bearing = twopi + *bearing;

  /* Convert to degrees */
  *bearing = Degrees( *bearing );

} /* End of Bearing_Distance() */

/*--------------------------------------------------------------------*/

/*  Locator_Valid()
 *
 *  Validates a grid locator
 */

  gboolean
Locator_Valid( char *locator )
{
  int len, /* Length of user-entered data  */
	  idx;

  /* Ignore blank fields */
  len = (int)strlen( locator );
  if( len == 0 )
	return( TRUE );

  /* Capitalize letters */
  for( idx = 0; idx < len; idx++ )
  {
	if( (locator[idx] > 0x60) && (locator[idx] < 0x7b) )
	  locator[idx] -= 0x20;

	/* Validate grid locator */
	switch( idx )
	{
	  case 0: case 1: /* First letters */
		if( (locator[idx] < 'A') || (locator[idx] > 'S') )
		  return( /* FALSE */TRUE );
		break;

	  case 2: case 3: /* Middle numbers */
		if( (locator[idx] < '0') || (locator[idx] > '9') )
		  return( FALSE );
		break;

	  case 4: case 5: /* Last letters */
		if( (locator[idx] < 'A') || (locator[idx] > 'X') )
		  return( FALSE );

	} /* switch( len ) */

  } /* for( idx = 0; idx < len; idx++ ) */

  return( len == 6 );

} /* End of Locator_Valid()*/

/*--------------------------------------------------------------------*/
