/*  common.h
 *
 *  Common header file for xsatcom: An interface application
 *  for Yaesu's GS-232 Az-El antenna rotor controller
 */

/*
 *  xsatcom: A X/GTK+ application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef COMMON_H
#define COMMON_H	1

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <termios.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <sys/time.h>
#include <sys/timeb.h>

/* For the multi-sat tree view list store */
enum
{
  NAME_COLUMN,
  ACCESS_COLUMN,
  AZIM_COLUMN,
  ELEV_COLUMN,
  RANGE_COLUMN,
  PHASE_COLUMN,
  AOS_COLUMN,
  LOS_COLUMN,
  NLIST_COLUMNS
};

/* For the multi-loc tree view tree store */
enum
{
  LABEL1_COLUMN,
  DATA1_COLUMN,
  LABEL2_COLUMN,
  DATA2_COLUMN,
  LABEL3_COLUMN,
  DATA3_COLUMN,
  LABEL4_COLUMN,
  DATA4_COLUMN,
  NTREE_COLUMNS
};

/*-------------------------------------------------------------------*/

/* Definitions for selecting satellites and locations */
enum
{
  CLEAR_CHANGE,
  FORWD_SATELLITE,
  BACK_SATELLITE,
  FIRST_SATELLITE,
  LAST_SATELLITE,
  FORWD_OBSERVER,
  BACK_OBSERVER,
  FIRST_OBSERVER,
  LAST_OBSERVER,
  DNLD_TLE_SETS,
  TREE_SELECTION
};

/*-------------------------------------------------------------------*/

/* Special characters */
#define LF      0x0A /* Line Feed */
#define CR      0x0D /* Carriage Return */
#define HT      0x09 /* Horizontal Tab */
#define WS      0x20 /* White Space */

/*-------------------------------------------------------------------*/

/* Definitions for xsatcom operating modes */
#define SINGLE_SAT          0x00000001
#define MULTI_SAT           0x00000002
#define MULTI_LOC           0x00000004
#define PASS_PRED           0x00000008
#define ILLUM_PRED          0x00000010
#define END_SINGLE_SAT      0x00000020
#define END_MULTI_SAT       0x00000040
#define END_MULTI_LOC       0x00000080
#define END_PASS_PRED       0x00000100
#define END_ILLUM_PRED      0x00000200
#define PASS_PRED_FWD       0x00000400
#define PASS_PRED_RESET     0x00000800
#define PASS_PRED_ACCESS    0x00001000
#define ILLUM_PRED_FWD      0x00002000
#define ILLUM_PRED_BACK     0x00004000
#define ILLUM_PRED_REDO     0x00008000
#define ILLUM_PRED_DONE     0x00010000
#define ALL_MODES           -1

/* GC colors */
#define	BLACK	0.0, 0.0, 0.0
#define	WHITE	255, 255, 255
#define	YELLOW	1.0, 1.0, 0.0
#define	GREEN	  0, 255, 0
#define	CYAN	  0, 255, 255
#define	BLUE	  0, 0, 255
#define	GREY	0.8, 0.8, 0.8

/*-------------------------------------------------------------------*/

/* Definitions for Transceiver CAT control */
/* Operating mode names (usb/lsb/cw etc) and op-codes */
#define TCVR_MODE_NAMES \
  "LSB", "USB", "CW", "CW(R)", "CW(N)", "CW(NR)", "AM", "FM",\
  "AM(N)", "FM(N)", "DATA", "DATA(R)", "DIG", "PKT","???"
#define NUM_MODE_CODES		14  /* Number of mode codes above */
#define NO_TCVR_MODE		0xFF  /* No operating mode for tcvr given */

/* Definitions for satellite database flags */
#define NO_TRANSPONDER	0x0000  /* No transponder data (name only) */
#define UPLINK_DATA		0x0001  /* Transponder Uplink freq specified */
#define DNLINK_DATA		0x0002  /* Transponder Dnlink freq specified */
#define DUPLEX_DATA		0x0003  /* Transponder Dnlink & Uplink freq specified */
#define PASSBAND_NORM	0x0004  /* Transponder passband is non-inverted */
#define BEACON_DATA		0x0008  /* Beacon frequency specified */
#define TXFREQ_DATA		0x0010  /* Transmit frequency specified */
#define RXFREQ_DATA		0x0020  /* Receive frequency specified */
#define RXFREQ_TUNE		0x0100  /* Receive frequency manually tuned */
#define TXFREQ_TUNE		0x0200  /* Receive frequency manually tuned */

#define DEFAULT_MODE	0  /* Default transponder mode # loaded */

/*-------------------------------------------------------------------*/

/* Definitions for setting and testing flow control flags */

/* Flags used in Az/El Rotor control */
#define ROTOR_SERIAL_FAIL 0x00000001
#define CONTROL_INIT      0x00000002
#define DIRECTION_ERROR   0x00000004
#define ROTORS_RUNNING    0x00000008
#define ROTORS_ENABLED    0x00000010
#define TRACKING_ENABLED  0x00000020
#define REVERSE_TRACK     0x00000040
#define TRACK_MOON        0x00000080
#define TRACK_SUN         0x00000100
#define TRACK_POLAR       0x00000200
#define TRACK_MANUAL      0x00000400
#define PARK_ROTORS       0x00000800
#define TRACKING_FLAGS    0x00000F80

/* Flags used for transceiver CAT control */
#define TCVR_SERIAL_TEST  0x00001000
#define CAT_SETUP	      0x00002000
#define UPLNK_DNLNK_TRACK 0x00004000
#define DOPPLER_DISABLED  0x00008000
#define NEW_DNLINK_FREQ   0x00010000
#define NEW_UPLINK_FREQ   0x00020000

/* Flags used in various data entry forms */
#define LOC_ENTERED       0x00040000
#define POSITION_ENTERED  0x00080000

/* Flags used in controlling satellite predictions */
#define VISIBLE_PREDICT   0x00100000
#define USERTIME_PREDICT  0x00200000
#define FREEZE_MULTISAT	  0x00400000

/* Flags for controlling xplanet satellite plots */
#define XPLANET_PLOT      0x00800000
#define MULTISAT_PLOT     0x0100000
#define ALL_FLAGS        ~0

#define XPLANET_STOPPED		0
#define XPLANET_START		1
#define XPLANET_RUNNING		2

/* Return values */
#define SUCCESS		0
#define ERROR		1
#define EOF_NORM	2

/* Fatal error flags */
#define FATAL		TRUE
#define NONFATAL	FALSE

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE	128

/*-------------------------------------------------------------------*/

/* Definitions for satellite status flags */
#define SAT_RANK_FLAGS    0x0000FFFF
#define SAT_VISIBLE       0x00010000
#define SAT_ECLIPSED      0x00020000
#define SAT_INACCESSIBLE  0x00040000
#define SAT_GEOSTATIONARY 0x00080000
#define SAT_DECAYED       0x00100000
#define NO_AOS_LOS        0x00200000
#define NORTH_CROSSING    0x00400000
#define REQ_KEYBD_ENTRY   0x00800000
#define SKIP_AOS_LOS      0x01000000
#define PASS_VISIBLE      0x02000000
#define DEEP_SPACE        0x04000000

/* Label status indicators */
#define DOPPLER_ON		"<span foreground=\"darkgreen\">Doppler</span>"
#define DOPPLER_OFF		"<span foreground=\"darkred\">Doppler</span>"
#define TRACK_ON		"<span foreground=\"darkgreen\">Tx/Rx Track</span>"
#define TRACK_OFF		"<span foreground=\"darkred\">Tx/Rx Track</span>"

/*-------------------------------------------------------------------*/

/** Table of constant values taken from NORAD Spacetrack #3 **/
#define de2ra    1.74532925E-2   /* Degrees to Radians */
#define pi       3.14159265359   /* Pi */
#define pio2     1.57079633      /* Pi/2 */
#define x3pio2   4.71238898      /* 3*Pi/2 */
#define twopi    6.2831853       /* 2*Pi  */
#define tothrd   0.6666666667    /* 2/3 */
#define xmnpda   1440.0          /* Minutes per day */
#define secday   86400.0         /* Seconds per day */
#define f        3.352779E-3
#define omega_E  1.0027379
#define c1ss     2.9864797E-6
#define zns      1.19459E-5
#define zes      1.675E-2
#define znl      1.5835218E-4
#define zel      5.490E-2
#define c1l      4.7968065E-7
#define zcosis   0.91744867
#define zsinis   0.39785416
#define zsings  -0.98088458
#define zcosgs   0.1945905
#define q22      1.7891679E-6
#define q31      2.1460748E-6
#define q33      2.2123015E-7
#define g22      5.7686396
#define g32      0.95240898
#define g44      1.8014998
#define g52      1.0508330
#define g54      4.4108898
#define root22   1.7891679E-6
#define root32   3.7393792E-7
#define root44   7.3636953E-9
#define root52   1.1428639E-7
#define root54   2.1765803E-9
#define rptim    4.37526908801129966E-3 /* this equates to 7.29211514668855e-5 rad/sec */
#define mfactor  7.292115E-5
#define sr          696000 /* Solar radius - kilometers (IAU 76)      */
#define AU       149597870 /* Astronomical unit - kilometers (IAU 76) */
#define Cl	 	 299792458 /* Speed of light m/sec */

/*-------------------------------------------------------------------*/

/** Type definitions **/

/* Geodetic position structure */
typedef struct
{
  double lat, lon, hgt, theta;

} geodetic_t;

/* General three-dimensional vector structure */
typedef struct
{
  double x, y, z, w;

} vector_t;

/* General position/velocity vector structure */
typedef struct
{
  vector_t
  pos, vel;

  double phase; /* Phase of satellite in rad */
} kinetics_t;

/* Two-line-element satellite orbital data
 * and variables used by new "str6" sgp4() */
#define NAME_SIZE	25
typedef struct
{
  char init, method;
  int satnum, error;

  /* Near Earth */
  int    isimp;
  double aycof  , con41  , cc1     , cc4    , cc5    , d2      , d3    , d4    ,
		 delmo  , eta    , argpdot , omgcof , sinmao , t       , t2cof , t3cof ,
		 t4cof  , t5cof  , x1mth2  , x7thm1 , mdot   , nodedot , xlcof , xmcof ,
		 nodecf;

  /* Deep Space */
  int    irez;
  double d2201  , d2211  , d3210  , d3222  , d4410  , d4422  , d5220 , d5232 ,
		 d5421  , d5433  , dedt   , del1   , del2   , del3   , didt  , dmdt  ,
		 dnodt  , domdt  , e3     , ee2    , peo    , pgho   , pho   , pinco ,
		 plo    , se2    , se3    , sgh2   , sgh3   , sgh4   , sh2   , sh3   ,
		 si2    , si3    , sl2    , sl3    , sl4    , gsto   , xfact , xgh2  ,
		 xgh3   , xgh4   , xh2    , xh3    , xi2    , xi3    , xl2   , xl3   ,
		 xl4    , xlamo  , zmol   , zmos   , atime  , xli    , xni;

  double
	jdsatepoch,  /* Reference epoch */
	ndot,   /* First Time Derivative of the Mean Motion */
	nddot,  /* Second Time Derivative of Mean Motion    */
	bstar,  /* BSTAR drag term   */
	inclo,  /* Orbit Inclination */
	nodeo,  /* Right Ascension of the Ascending Node */
	ecco,   /* Eccentricity */
	argpo,  /* Argument of Perigee */
	mo,     /* Mean Anomaly */
	no,     /* Mean Motion  */
	revnum; /* Revolution Number */

  int elset;  /* Element Set */

  char
	name[NAME_SIZE], /* Satellite name string    */
	idesg[9];		 /* International Designator */

} elem_set_t;

/* Common arguments between deep-space functions */
typedef struct
{
  /* Used by dpinit part of Deep() */
  double
	eosq, sinio, cosio, betao, aodp, theta2, sing, cosg,
	betao2, xmdot, omgdot, xnodot, xnodp;

  /* Used by dpsec and dpper parts of Deep() */
  double
	xll, omgadf, xnode, em, xinc, xn, t;

  /* Used by thetg and Deep() */
  double ds50;

} deep_arg_t;

/* Data from satellite pass predicitons */
typedef struct
{
  int flags;

  double
	julian_utc, /* Julian UTC time     */
	elev,     	/* Satellite elevation */
	azim,     	/* Satellite azimuth   */
	range,   	/* Satellite range     */
	lon,      	/* Satellite longitude */
	lat,      	/* Satellite latitude  */
	phs256,   	/* Phase in units/256  */
	edepth;   	/* Sat. eclipse depth  */

} pass_data_t;

/* Satellite status data structure */
typedef struct
{
  char
	name[NAME_SIZE], /* Satellite Name */
	resol;			 /* Tracking resolution in deg */

  double
	azim,    /* Azimuth - deg    */
	elev,    /* Elevation - deg  */
	range,   /* Slant Range km   */
	ploss,   /* Path loss db@1Mhz*/
	rrate,   /* Range rate km/s  */
	doppler, /* Doppler - Hz/MHz */
	ssplon,  /* Longitude - deg  */
	ssplat,  /* Latitude - deg   */
	alt,     /* Altitude - km    */
	phs256,  /* Phase units/256  */
	vel,     /* Velocity - km/s  */
	ftprnt,  /* Footprint - km   */
	edepth,  /* Eclipse depth    */
	aos,     /* AOS time - UTC   */
	los,     /* LOS time - UTC   */
	revnum;  /* Current orbit no */

  int flags; /* Various flags    */

} satellite_status_t;

/* Observer status data structure */
typedef struct
{
  char
	loc_name[NAME_SIZE], /* Observer location name */
	grid_loc[7];		 /* Maidenhead grid locator */

  /* Observer geodetic position */
  geodetic_t obs_geodetic;

  struct tm
	utc,      /* UTC time */
	localtm;  /* Local time */

  vector_t
	solar_set, /* Solar Az/Elev/Range */
	lunar_set; /* Lunar Az/Elev/Range */

} observer_status_t;

/* Satellite transponder and tcvr modes */
typedef struct
{
  char mode_name[NAME_SIZE]; /* Name of tranponder mode */

  int
	uplink_freq,  /* Uplink frequency      */
	tx_freq,      /* Transmitter frequency */
	dnlink_freq,  /* Downlink frequency    */
	rx_freq,      /* Receiver frequency    */
	bcn_freq,     /* Beacon frequency      */
	tone_freq;    /* CTCSS Tone freq * 10  */

  unsigned char
	tx_mode,      /* Transmitter operating mode   */
	rx_mode,      /* Receiver operating mode      */
	bcn_mode,     /* Beacon operating mode        */
	flags;        /* Flags for various conditions */

} satellite_mode_t;

/* Transceiver status structure */
typedef struct
{
  int
	uplink_freq,    /* Required uplink frequency */
	tx_freq,        /* Transmitter frequency */
	tx_freq_ref,    /* Tx reference for uplink frequency calculations */
	tx_offset,      /* Tx freq offset from uplink */
	tx_doppler,     /* Uplink doppler shift */
	tx_ploss,       /* Receiver path loss */
	dnlink_freq,    /* Required downlink frequency */
	rx_freq,        /* Receiver frequency */
	rx_freq_ref,    /* Rx reference for dnlink frequency calculations */
	rx_offset,      /* Rx freq offset from dnlink  */
	rx_doppler,     /* Dnlink doppler shift */
	upl_dnl_offset,	/* Downlink-Uplink freq offset */
	rx_ploss,		/* Receiver path loss */
	rx_smeter,		/* Rx S-meter reading */
	tx_power,		/* Tx O/P power */
	bw_main,		/* Main Rx bandwidth */
	bw_subr,		/*  Sub Rx bandwidth */
	tone_freq,      /* CTCSS tone frequency */
	tone_freq_ref;	/* CTCSS tone frequency reference */

  unsigned char
	tx_mode,     /* Transmitter operating mode */
	tx_mode_ref, /* Tx reference for uplink mode change detection */
	tx_status,   /* Transmitter status */
	rx_mode,     /* Receiver operating mode */
	rx_mode_ref, /* Rx reference for dnlink mode change detection */
	duplex;		 /* Transceiver in duplex mode */

  int flags;	 /* Indicates what transponder data is given */

} tcvr_status_t;

/* Runtime config data */
typedef struct
{
  char
	rotor_serial[16],
	cat_serial[16],
	xplanet_config[81],
	xplanet_marker[81],
	xplanet_gcarcs[81],
	xplanet_satfile[81],
	xplanet_command[128],
	wgf;

  int
	low_resol,
	med_resol,
	high_resol,
	azim_parking,
	elev_parking,
	azim_offset,
	angle_step,
	time_step,
	tcvr_type;

  double horizon;

  /* WGF data for str6/sgp4 */
  double tumin, radiusearthkm, xke, j2, j3, j4, j3oj2;

} rc_data_t;

/* Transceiver type */
enum
{
  NONE = 0,
  FT847,
  FT857,
  K2,
  K3
};

/*-------------------------------------------------------------------*/

/* Function Prototypes produced by cproto */
/* callbacks.c */
void Error_Dialog(char *message, gboolean fatal);
void on_sat_track_destroy(GtkObject *object, gpointer user_data);
gboolean on_sat_track_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_txrx_track_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_update_track_clicked(GtkButton *button, gpointer user_data);
void on_doppler_corr_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_tcvr_cat_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_rotor_control_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_multi_sat_clicked(GtkButton *button, gpointer user_data);
void on_pass_pred_clicked(GtkButton *button, gpointer user_data);
void on_multi_loc_clicked(GtkButton *button, gpointer user_data);
void on_illum_pred_clicked(GtkButton *button, gpointer user_data);
void on_xplanet_plot_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_multisat_plot_toggled(GtkToggleButton *togglebutton, gpointer user_data);
gboolean on_multi_sat_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_pass_pred_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_multi_loc_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_illum_pred_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_error_quit_clicked(GtkButton *button, gpointer user_data);
void tree_selection_changed_cb(GtkTreeSelection *selection, gpointer user_data);
void on_access_pred_clicked(GtkButton *button, gpointer user_data);
void on_visible_pred_clicked(GtkButton *button, gpointer user_data);
void on_fwd_pred_clicked(GtkButton *button, gpointer user_data);
void on_pass_redo_clicked(GtkButton *button, gpointer user_data);
void on_loc_up_clicked(GtkButton *button, gpointer user_data);
void on_loc_dn_clicked(GtkButton *button, gpointer user_data);
void on_loc_frst_clicked(GtkButton *button, gpointer user_data);
void on_loc_last_clicked(GtkButton *button, gpointer user_data);
void on_ill_fwd_clicked(GtkButton *button, gpointer user_data);
void on_ill_back_clicked(GtkButton *button, gpointer user_data);
void on_ill_redo_clicked(GtkButton *button, gpointer user_data);
void on_sat_up_clicked(GtkButton *button, gpointer user_data);
void on_sat_dn_clicked(GtkButton *button, gpointer user_data);
void on_sat_frst_clicked(GtkButton *button, gpointer user_data);
void on_sat_last_clicked(GtkButton *button, gpointer user_data);
gboolean on_calibration_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_offset_calibration_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_fullscale_calibration_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_track_moon_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_track_sun_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_track_polar_star_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_park_rotors_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_position_rotors_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_quit_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_new_location_activate(GtkMenuItem *menuitem, gpointer user_data);
gboolean on_sat_track_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_azel_apply_clicked(GtkButton *button, gpointer user_data);
void on_azel_ok_clicked(GtkButton *button, gpointer user_data);
void on_loc_apply_clicked(GtkButton *button, gpointer user_data);
void on_loc_close_clicked(GtkButton *button, gpointer user_data);
void on_loc_save_clicked(GtkButton *button, gpointer user_data);
gboolean on_mode_chng_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_freeze_multisat_clicked(GtkButton *button, gpointer user_data);
void on_error_dialog_destroy(GtkObject *object, gpointer user_data);
void on_start_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_download_tle_activate(GtkMenuItem *menuitem, gpointer user_data);
gboolean on_illum_drawingarea_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
/* cat.c */
gboolean Open_Tcvr_Serial(void);
void Close_Tcvr_Serial(void);
void Cancel_CAT(void);
gboolean Read_Tcvr_Serial(char *data, size_t len);
gboolean Write_Tcvr_Serial(char *data, size_t len);
gboolean Tcvr_Status(int action, tcvr_status_t *tcvr_status);
gboolean K3_Get_Data(char *get, char *data);
gboolean K3_SubRx_Status(int action, tcvr_status_t *tcvr_status);
gboolean K3_Set_Freq_Mode(int action, tcvr_status_t *tcvr_status);
gboolean K3_Original_Status(int action);
gboolean K3_Tcvr_Status(int action, tcvr_status_t *tcvr_status);
gboolean FT8x7_Set_Freq_Mode(int action, tcvr_status_t *tcvr_status);
gboolean FT8x7_Get_Freq_Mode(int action, tcvr_status_t *tcvr_status);
gboolean FT8x7_Set_CTCSS(tcvr_status_t *tcvr_status);
gboolean FT8x7_Tcvr_Status(int action, tcvr_status_t *tcvr_status);
gboolean Initialize_Tcvr(int mode_idx);
void CAT_Control(satellite_status_t *sat_status);
/* date_time.c */
double Julian_Date_of_Epoch(double epoch);
double Epoch_Time(double jd);
int DOY(int yr, int mo, int dy);
double Fraction_of_Day(int hr, int mi, int se);
void Calendar_Date(double jd, struct tm *cdate);
void Time_of_Day(double jd, struct tm *cdate);
double Julian_Date(struct tm *cdate);
void Date_Time(double julian_date, struct tm *cdate);
int Check_Date(struct tm *cdate);
struct tm Time_to_UTC(struct tm *cdate);
struct tm Time_from_UTC(struct tm *cdate);
double JD_to_UTC(double jt);
double JD_from_UTC(double jt);
double Delta_ET(double year);
double Julian_Date_of_Year(double year);
double ThetaG(double epoch, deep_arg_t *deep_arg);
double ThetaG_JD(double jd);
void Calendar_Time_Now(struct tm *utc, struct tm *localtm);
double Current_Daynum(void);
double Theta(double julian_utc, double lon);
/* earthtrack.c */
double Arc_Cos(double x, double y);
void Visibility_to_Color(satellite_status_t *sat_status, char *color);
void Range_Lat_Lon(satellite_status_t *sat_status, int azim, double *range_lat, double *range_lon);
int Display_Satellite(satellite_status_t *sat_status);
int Config_Xplanet(observer_status_t *obs_status);
/* gridloc.c */
void Position_to_Grid_Locator(double lon, double lat, char *grid_loc);
void Grid_Locator_to_Position(char *grid_loc, double *lon, double *lat);
void Degrees_to_DMS(double deg, int *ideg, int *imin, int *isec);
void DMS_to_Degrees(int ideg, int imin, int isec, double *deg);
void Bearing_Distance(double lon_a, double lat_a, double lon_b, double lat_b, double *bearing, double *distance);
gboolean Locator_Valid(char *locator);
/* gui.c */
void Show_Observer_Status(GtkWidget *window, observer_status_t *obs_status);
void Show_Satellite_Data(GtkWidget *window, satellite_status_t *sat_status);
void Show_Multiobserver_Status(GtkTreeView *multiloc_treeview, GtkTreeStore *store, satellite_status_t *mxsat_status);
void Show_Satellite_Status(satellite_status_t *sat_status);
void Show_Multisatellite_Status(GtkTreeView *multisat_treeview, satellite_status_t *multisat_status);
void Show_Pass_Predictions(GtkWidget *pass_predict, satellite_status_t *sat_status, pass_data_t *pass_data, int num);
void Show_Illumination_Predictions(double *jdate, int *illum);
void Show_Rotor_Control(double azim, double elev, int resol);
void Show_CAT_Control(tcvr_status_t *tcvr_status);
gboolean Show_Offset_Calibration(gpointer data);
gboolean Show_Fullscale_Calibration(gpointer data);
void Set_GC_Attributes(GdkGC *gc, int red, int green, int blue, gint line_width, GtkWidget *widget);
void Set_Doppler_Label(gchar *markup);
/* input.c */
char *Load_Database_Tle_Data(void);
void Convert_Satellite_Data(char *tle_set, elem_set_t *elem_set);
void Load_Tle_Set(char *sat_name, elem_set_t *sat_elem_set);
char *Read_Transponder_Data(char *sat_name);
unsigned char Read_Operating_Mode(char *line);
void Find_New_Satellite(int which, elem_set_t *sat_elem_set, satellite_status_t *sat_status, observer_status_t *obs_status);
void New_Status(gboolean new_sat, elem_set_t *sat_elem_set, satellite_status_t *sat_status, observer_status_t *obs_status);
void Set_Accessibility_Flags(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
int Aos_Possible(elem_set_t *sat_elem_set, observer_status_t *obs_status);
int Satellite_Decayed(elem_set_t *sat_elem_set, double julian_utc);
int Satellite_Geostationary(elem_set_t *sat_elem_set);
int Checksum_Good(char *tle_set);
gboolean Good_Elements(char *tle_set);
char *Load_Observer_Data(void);
void Load_Observer_Set(char *loc_name, observer_status_t *obs_status);
void Save_Location(observer_status_t *obs_status);
int Load_Line(char *buff, FILE *pfile, int max_chr, char *messg, gboolean eof_err);
/* interface.c */
GtkWidget *create_sat_track(void);
GtkWidget *create_multi_sat(void);
GtkWidget *create_pass_pred(void);
GtkWidget *create_multi_loc(void);
GtkWidget *create_illum_pred(void);
GtkWidget *create_error_dialog(void);
GtkWidget *create_calibration(void);
GtkWidget *create_main_menu(void);
GtkWidget *create_azel_dialog(void);
GtkWidget *create_loc_dialog(void);
/* main.c */
int main(int argc, char *argv[]);
gboolean Load_Config(gpointer data);
/* math_func.c */
int Sign(double arg);
double Sqr(double arg);
double Cube(double arg);
double Radians(double arg);
double Degrees(double arg);
double ArcSin(double arg);
double ArcCos(double arg);
void Magnitude(vector_t *v);
void Vec_Add(vector_t *v1, vector_t *v2, vector_t *v3);
void Vec_Sub(vector_t *v1, vector_t *v2, vector_t *v3);
void Scalar_Multiply(double k, vector_t *v1, vector_t *v2);
void Scale_Vector(double k, vector_t *v);
double Dot(vector_t *v1, vector_t *v2);
double Angle(vector_t *v1, vector_t *v2);
void Cross(vector_t *v1, vector_t *v2, vector_t *v3);
void Normalize(vector_t *v);
double AcTan(double sinx, double cosx);
double FMod2p(double x);
double Modulus(double arg1, double arg2);
double Frac(double arg);
int Round(double arg);
double Int(double arg);
void Convert_Sat_State(kinetics_t *sat_kinetics);
double Footprint(vector_t *pos);
double Atof(const char *nptr);
/* mode.c */
gboolean Central_Mode(gpointer data);
void Single_Satellite_Mode(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Multi_Satellite_Mode(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Multi_Observer_Mode(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Pass_Prediction_Mode(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Illumination_Prediction_Mode(elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Calculate_Satellite_Status(double julian_utc, elem_set_t *sat_elem_set, kinetics_t *solar_kinetics, kinetics_t *obs_kinetics, observer_status_t *obs_status, satellite_status_t *sat_status);
void Manual_Position_Rotors(int azim, int elev);
void Auto_Track(double julian_utc, satellite_status_t *sat_status);
int isModeSet(int flag);
int isModeClear(int flag);
void Set_Mode(int flag);
void Clear_Mode(int flag);
void Toggle_Mode(int flag);
/* observer.c */
void Calculate_User_PosVel(double time, observer_status_t *obs_status, kinetics_t *obs_kinetics);
void Calculate_LatLonAlt(double time, vector_t *pos, geodetic_t *geodetic);
void Calculate_Observer(kinetics_t *sat_kinetics, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
void Calculate_RADec(double time, kinetics_t *sat_kinetics, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
void RaDec_to_AzEl(double ra, double dec, double theta, geodetic_t *obs_geodetic, vector_t *lunar_set);
void RaDec_to_Topocentric(double ra, double dec, vector_t *topocentric);
double Find_Horizon_Crossing(double julian_utc, elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Find_Next_AOS_LOS(double julian_utc, elem_set_t *sat_elem_set, observer_status_t *obs_status, satellite_status_t *sat_status);
void Calculate_Orbit(double julian_utc, elem_set_t *sat_elem_set, geodetic_t *sat_geodetic, kinetics_t *sat_kinetics, satellite_status_t *sat_status, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
/* plots.c */
void Draw_Plotting_Frame(GtkWidget *drawingarea, GdkRectangle *rect, gchar *title, int nhor, int nvert);
void Draw_Graph(GtkWidget *drawingarea, GdkRectangle *rect, double *a, double *b, double amax, double amin, double bmax, double bmin, int nval);
void Cairo_Draw_Lines(cairo_t *cr, GdkPoint *points, int npoints);
/* rotor.c */
gboolean Open_Rotor_Serial(void);
int Write_Rotor_Serial(char *ch);
char Read_Rotor_Serial(void);
void Close_Rotor_Serial(void);
void Rotor_Send_String(char *strn);
void Rotor_Read_String(char *strn, int len);
void Rotor_Send_Direction(int azim, int elev);
void Rotor_Read_Direction(int *azim, int *elev);
void Enable_Tracking(void);
void Disable_Tracking(void);
/* sgp4unit.c */
int sgp4init(const int satn, const double epoch, elem_set_t *elem_set);
int sgp4(elem_set_t *elem_set, double tsince, kinetics_t *kinetics);
double gstime(double jdut1);
void getgravconst(void);
/* shared.c */
/* solar_lunar.c */
void Calculate_Solar_Position(double time, kinetics_t *solar_kinetics);
int Satellite_Eclipsed(kinetics_t *sat_kinetics, kinetics_t *sol_kinetics, double *depth);
void Lunar_Position(double daynum, geodetic_t *obs_geodetic, vector_t *lunar_set);
void Calculate_Moon_Position(double julian_utc, observer_status_t *obs_status, kinetics_t *obs_kinetics);
/* support.c */
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name);
void add_pixmap_directory(const gchar *directory);
GtkWidget *create_pixmap(GtkWidget *widget, const gchar *filename);
GdkPixbuf *create_pixbuf(const gchar *filename);
void glade_set_atk_action_description(AtkAction *action, const gchar *action_name, const gchar *description);
/* utils.c */
char *Abort_On_Error(int why);
void Cleanup(void);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
int isSatFlagSet(satellite_status_t *sat_status, int flag);
int isSatFlagClear(satellite_status_t *sat_status, int flag);
void Set_SatFlag(satellite_status_t *sat_status, int flag);
void Clear_SatFlag(satellite_status_t *sat_status, int flag);
void Usage(void);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);
void mem_alloc(void **ptr, size_t req, gchar *str);
void mem_realloc(void **ptr, size_t req, gchar *str);
void free_ptr(void **ptr);

#endif /* End of xsatcom.h */

