/*  satcom.h
 *
 *  Header file for satcom: An interface application
 *  for Yaesu's GS-232 Az-El antenna rotor controller
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef SATCOM_H
#define SATCOM_H 1

#include <math.h>
#include <signal.h>
#include <form.h>
#include <ncurses.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <time.h>
#include <sys/time.h>
#include <sys/timeb.h>

/************ CUSTOMIZE THESE DEFINITIONS FOR YOUR NEEDS *************/

/* The following four entries *must* be customized */

/* Default Home position (East long. and North lat.) for the */
/* grid locator calculator. Format "E/ddd:mm:ss N/dd:mm:ss"  */
#define HOME_POS "E/033:01:36 N/34:42:31"

/* Serial device used by Yaesu GS-232 rotor controller */
#define ROTOR_SERIAL_DEV "/dev/ttyS1"

/* Serial device used by Yaesu FT-847 transceiver */
#define TCVR_SERIAL_DEV "/dev/ttyS0"

/* Default satellite name at startup */
#define SAT_NAME "AO-40"

/* Azimuth and Elevation tracking   */
/* resolutions in deg !(Minimum=1)! */
#define LOW_RESOL   5
#define MED_RESOL   2
#define HIGH_RESOL  1

/* Default Azimuth/Elevation  */
/* parking position of rotors */
#define AZIM_PARKING 180
#define ELEV_PARKING   0

/* Time delay in *tenths* of seconds */
/* between orbital calculations      */
#define UPDATE_DELAY 10

/* Azimuth offset correction needed */
/* when using the G-5500 Azim/Elev   */
/* rotors with the GS232 controller */
#define AZIM_OFFSET 180

/* Horizon threshold for AOS/LOS searches (in deg) */
#define HORIZON 0.03
/* Azim/Elev angular step in horizon search (in deg) */
#define ANGL_STEP 2
/* Time step of AOS/LOS search (in min) */
#define TIME_STEP 2

/* Definitions of some color pairs and color numbers */
#define WHITE		COLOR_PAIR( COLOR_WHITE )
#define RED			COLOR_PAIR( COLOR_RED )
#define GREEN		COLOR_PAIR( COLOR_GREEN )
#define BLUE		COLOR_PAIR( COLOR_BLUE )
#define YELLOW		COLOR_PAIR( COLOR_YELLOW ) | A_BOLD
#define CYAN		COLOR_PAIR( COLOR_CYAN )
#define MAGENTA		COLOR_PAIR( COLOR_MAGENTA )
#define KEYSTROKE	COLOR_PAIR( COLOR_KEYSTROKE ) | A_BOLD
#define COMMAND  	COLOR_PAIR( COLOR_COMMAND )   | A_BOLD
#define ERROR   	COLOR_PAIR( COLOR_ERROR )  | A_BOLD
#define APPROX  	COLOR_PAIR( COLOR_APPROX ) | A_BOLD
#define FINISH  	COLOR_PAIR( COLOR_FINISH ) | A_BOLD
#define FUNCTION	COLOR_PAIR( COLOR_COMMAND )

#define COLOR_KEYSTROKE  8  /* Yellow fore, blue background   */
#define COLOR_COMMAND    9  /*  White fore, blue background   */
#define COLOR_ERROR     10  /*  White fore, red background    */
#define COLOR_APPROX    11  /*  White fore, yellow background */
#define COLOR_FINISH    12  /*  White fore, green background  */

/* xplanet directories and files */
#define XPLANET_DIR			"/usr/share/xplanet"
#define XPLANET_ARCFILE		"arcs/satcom"
#define XPLANET_MARKERFILE	"markers/satcom"
#define XPLANET_CONFIGFILE	"config/satcom"

/**************** END OF CUSTOMIZABLE SECTION ************************/

/* Definitions of some keycodes */
#define KEY_LFD      0x0A /* Keycode of LINEFEED character     */
#define KEY_RET      0x0D /* Keycode of RETURN keyboard key    */
#define KEY_ESC      0x1B /* Keycode of ESCAPE keyboard key    */
#define KEY_SPACE    0x20 /* Keycode of SPACE keyboard key     */

/*-------------------------------------------------------------------*/

/* Definitions for selecting satellites */
#define DEFAULT  1
#define NEXT     2
#define PREVIOUS 3
#define FIRST    4
#define LAST     5

/*-------------------------------------------------------------------*/

/* Definitions for Transceiver CAT control */

/* Operating mode names (usb/lsb/cw etc) and op-codes */
#define TCVR_MODE_NAMES  "LSB", "USB", "CW-R", "CW(N)", "CWN-R",\
  "CW", "AM(N)", "FM(N)", "AM", "FM", "??? "
#define TCVR_MODE_CODES  0x00,0x01,0x03,0x82,0x83,0x02,0x84,0x88,0x04,0x08
#define NUM_MODE_CODES     10  /* Number of mode codes above */
#define NO_TCVR_MODE     0xFF  /* No operating mode for tcvr given */

#define PO_ALC           0x1F /* Mask for Tx PO/ALC status  */
#define S_METER          0x1F /* Mask for Rx S-Meter status */

/* Command codes ('P1' op-code) */
#define CAT_ON             0x00
#define CAT_OFF            0x80
#define SATELLITE_ON       0x4E
#define SATELLITE_OFF      0x8E
#define RECEIVER_STATUS    0xE7
#define TRANSMIT_STATUS    0xF7
#define READ_SAT_RX_VFO    0x13
#define READ_SAT_TX_VFO    0x23
#define WRITE_SAT_RX_VFO   0x11
#define WRITE_SAT_TX_VFO   0x21
#define WRITE_SAT_RX_MODE  0x17
#define WRITE_SAT_TX_MODE  0x27

/* Operating mode codes ('D1' op-code) */
#define LSB      0x00
#define USB      0x01
#define CW       0x02
#define CWR      0x03
#define AM       0x04
#define FM       0x08
#define CWN      0x82
#define CWNR     0x83
#define AMN      0x84
#define FMN      0x88

/* Definitions for satellite database flags */
#define NO_DATA        0x00  /* No transponder data (name only)   */
#define PBAND_INV      0x01  /* Transponder passband is inverted  */
#define TRANSPND_DATA  0x02  /* Transponder u/l, d/l freq. given  */
#define BEACON_DATA    0x04  /* Beacon frequency specified        */
#define DEFAULT_MODE   0     /* Default transponder mode # loaded */

/*-------------------------------------------------------------------*/

/* Definitions for grid locator calculator */
#define HOME_ROW   0 /* Define which row of data fields are read */
#define REMOTE_ROW 9 /* by read_position() and read_locator()    */

/*-------------------------------------------------------------------*/

/* Definitions for operating mode flags */
#define MAIN_SCREEN_MODE       1
#define SINGLE_SAT_MODE        2
#define MULTI_SAT_MODE         3
#define MULTI_OBS_MODE         4
#define MANUAL_COMM_MODE       5
#define OFFSET_CALIB_MODE      6
#define FULLSCALE_CALIB_MODE   7
#define GRID_LOC_MODE          8
#define PASS_PREDICTION_MODE   9
#define ILLUM_PREDICTION_MODE 10

/*-------------------------------------------------------------------*/

/* Definitions for setting and testing flow control flags */

/* Flags used in SGP4/SDP4 calculations */
#define SGP_SDP_FLAGS          0x000001FF
#define SGP4_INITIALIZED_FLAG  0x00000001
#define SDP4_INITIALIZED_FLAG  0x00000002
#define SIMPLE_CALC_FLAG       0x00000004
#define LUNAR_TERMS_DONE_FLAG  0x00000008
#define NEW_EPHEMERIS_FLAG     0x00000010
#define DO_LOOP_FLAG           0x00000020
#define RESONANCE_FLAG         0x00000040
#define SYNCHRONOUS_FLAG       0x00000080
#define EPOCH_RESTART_FLAG     0x00000100

/* Flags used in Az/El Rotor control */
#define ROTOR_READ_FAIL_FLAG   0x00000200
#define CONTROL_INIT_FLAG      0x00000400
#define DIRECTION_ERROR_FLAG   0x00000800
#define ROTORS_RUNNING_FLAG    0x00001000
#define ROTORS_ENABLED_FLAG    0x00002000
#define TRACKING_ENABLED_FLAG  0x00004000
#define REVERSE_TRACK_FLAG     0x00008000
#define TRACK_MOON_FLAG        0x00010000

/* Flags used for transceiver CAT control */
#define TCVR_READ_FAIL_FLAG    0x00020000
#define CAT_ENABLED_FLAG       0X00040000
#define UPLNK_DNLNK_TRACK_FLAG 0X00080000
#define DOPPLER_DISABLED_FLAG  0X00100000

/* Flags used in various data entry forms */
#define HOME_LOC_ENTERED_FLAG  0X00200000
#define REM_LOC_ENTERED_FLAG   0X00400000
#define POSITION_ENTERED_FLAG  0X00800000

/* Flags used in controlling satellite predictions */
#define VISIBLE_PREDICT_FLAG   0X01000000
#define USERTIME_PREDICT_FLAG  0X02000000

/* Toggle Earthtrack flag */
#define EARTHTRACK_FLAG        0x04000000
#define XPLANET_RUN			   0x08000000

#define ALL_FLAGS              -1

/*-------------------------------------------------------------------*/

/* Definitions for satellite status flags */
#define SAT_RANK_FLAGS         0x000003FF
#define SAT_VISIBLE_FLAG       0x00000400
#define SAT_ECLIPSED_FLAG      0x00000800
#define SAT_INACCESSIBLE_FLAG  0x00001000
#define SAT_GEOSTATIONARY_FLAG 0x00002000
#define SAT_DECAYED_FLAG       0x00004000
#define NO_AOS_LOS_FLAG        0x00008000
#define NORTH_CROSSING_FLAG    0x00010000
#define REQ_KEYBD_ENTRY_FLAG   0x00020000
#define SKIP_AOS_LOS_FLAG      0x00040000
#define PASS_VISIBLE_FLAG      0x00080000
#define DEEP_SPACE_FLAG        0x00100000

/* Definitions for ephemeris type */
#define DEEP_SPACE   0
#define NEAR_EARTH   1

/* Definitions for control of rotor status display */
#define DISPLAY    1
#define NO_DISPLAY 0

/*-------------------------------------------------------------------*/

/** Type definitions **/

/* Geodetic position structure */
typedef struct
{
  double
	lat, lon, hgt, theta;
} geodetic_t;

/* General three-dimensional vector structure */
typedef struct
{
  double
	x, y, z, w;
} vector_t;

/* General position/velocity vector structure */
typedef struct
{
  vector_t
	pos, vel;

  double phase; /* Phase of satellite in rad */
} kinetics_t;

/* Lunar perturbation data */
typedef struct
{
  /* Multipliers for D, M, M' and F */
  int
	Dm, Ms, Mm, Fm;

  /* Coefficient of the sine or cosine of argument */
  double a;
} lunar_perturbation_t;

/* Number of lunar perterbation terms used in Calculate_Moon_Position() */
#define PERTB_NTERMS    60

/* Two-line-element satellite orbital data */
typedef struct
{
  double
	epoch,  /* Reference epoch */
	xndt2o, /* First Time Derivative of the Mean Motion */
	xndd6o, /* Second Time Derivative of Mean Motion    */
	bstar,  /* BSTAR drag term   */
	xincl,  /* Orbit Inclination */
	xnodeo, /* Right Ascension of the Ascending Node */
	eo,     /* Eccentricity */
	omegao, /* Argument of Perigee */
	xmo,    /* Mean Anomaly */
	xno;    /* Mean Motion  */

  int
	catnr,  /* Catalogue Number  */
	elset,  /* Element Set       */
	revnum; /* Revolution Number */

  char
	ephem,    /* Ephemeris type (SGP4 or SDP4) */
	name[25], /* Satellite name string    */
	idesg[9]; /* International Designator */

} tle_t;

/* Satellite status data structure */
typedef struct
{
  char
	name[25],  /* Satellite Name */
	resol;     /* Tracking resolution in deg */

  double
	azim,    /* Azimuth - deg    */
	elev,    /* Elevation - deg  */
	range,   /* Slant Range km   */
	ploss,   /* Path loss db@1Mhz*/
	rrate,   /* Range rate km/s  */
	doppler, /* Doppler - Hz/MHz */
	ssplon,  /* Longitude - deg  */
	ssplat,  /* Latitude - deg   */
	alt,     /* Altitude - km    */
	phs256,  /* Phase units/256  */
	vel,     /* Velocity - km/s  */
	ftprnt,  /* Footprint - km   */
	edepth,  /* Eclipse depth    */
	aos,     /* AOS time - UTC   */
	los;     /* LOS time - UTC   */

  int
	revnum,   /* Current orbit no */
	flags;    /* Various flags    */

} satellite_status_t;

/* Observer status data structure */
typedef struct
{
  char
	loc_name[12], /* Observer location name */
	grid_loc[7];  /* Maidenhead grid locator */

  geodetic_t obs_geodetic; /* Observer geodetic position */

  struct tm
	utc,      /* UTC time */
	localtm;  /* Local time */

  vector_t
	solar_set, /* Solar Az/Elev/Range */
	lunar_set; /* Lunar Az/Elev/Range */

} observer_status_t;

/* Transceiver status structure */
typedef struct
{
  char mode_name[10]; /* Name of tranponder mode */

  int
	uplink_freq,     /* Required uplink frequency  */
	tx_freq,         /* Transmitter frequency */
	tx_offset,       /* Tx freq offset from uplink */
	tx_ref_freq,     /* Tx reference for uplink frequency calculations */
	tx_doppler,      /* Uplink doppler shift  */
	tx_ploss,        /* Receiver path loss    */
	dnlink_freq,     /* Required downlink frequency */
	rx_freq,         /* Receiver frequency    */
	rx_offset,       /* Rx freq offset from dnlink  */
	rx_ref_freq,     /* Rx reference for dnlink frequency calculations */
	rx_doppler,      /* Dnlink doppler shift  */
	rx_ploss,        /* Receiver path loss    */
	upl_dnl_offset;  /* Downlink-Uplink freq offset */

  unsigned char
	tx_mode,      /* Transmitter operating mode */
	tx_mode_ref,  /* Tx reference for uplink mode change detection */
	tx_status,    /* Transmitter status */
	rx_mode,      /* Receiver operating mode */
	rx_mode_ref,  /* Rx reference for dnlink mode change detection */
	rx_status,    /* Receiver status */
	flags;        /* Indicates what transponder data is given */

} tcvr_status_t;

/* Satellite transponder and tcvr modes */
typedef struct
{
  char mode_name[8]; /* Name of tranponder mode */

  int
	uplink_freq,  /* Uplink frequency      */
	tx_freq,      /* Transmitter frequency */
	dnlink_freq,  /* Downlink frequency    */
	rx_freq,      /* Receiver frequency    */
	bcn_freq;     /* Beacon frequency      */

  unsigned char
	tx_mode,      /* Transmitter operating mode   */
	rx_mode,      /* Receiver operating mode      */
	bcn_mode,     /* Beacon operating mode      */
	flags;        /* Flags for various conditions */

} satellite_mode_t;

/* Data from satellite pass predicitons */
typedef struct
{
  int flags[15];

  double
	julian_utc[15], /* Julian UTC time     */
	elev[15],     /* Satellite elevation */
	azim[15],     /* Satellite azimuth   */
	range[15],    /* Satellite range     */
	lon[15],      /* Satellite longitude */
	lat[15],      /* Satellite latitude  */
	phs256[15],   /* Phase in units/256  */
	edepth[15];   /* Sat. eclipse depth  */

} pass_data_t;

/* Common arguments between deep-space functions */
typedef struct
{
  /* Used by dpinit part of Deep() */
  double
	eosq, sinio, cosio, betao, aodp, theta2, sing, cosg,
	betao2, xmdot, omgdot, xnodot, xnodp;

  /* Used by dpsec and dpper parts of Deep() */
  double
	xll, omgadf, xnode, em, xinc, xn, t;

  /* Used by thetg and Deep() */
  double ds50;

} deep_arg_t;

/*-------------------------------------------------------------------*/

/** Table of constant values taken from NORAD Spacetrack #3 **/
#define de2ra    1.74532925E-2   /* Degrees to Radians */
#define pi       3.14159265      /* Pi */
#define pio2     1.57079633      /* Pi/2 */
#define x3pio2   4.71238898      /* 3*Pi/2 */
#define twopi    6.2831853       /* 2*Pi  */
#define e6a      1.0E-6
#define tothrd   6.6666667E-1    /* 2/3 */
#define xj2      1.082616E-3     /* J2 Harmonic */
#define xj3     -2.53881E-6      /* J3 Harmonic */
#define xj4     -1.65597E-6      /* J4 Harmonic */
#define xke      7.43669161E-2
#define xkmper   6.378135E3      /* Earth radius km */
#define eperim   4.0074973E4     /* Earth perimeter in km */
#define xmnpda   1.44E3          /* Minutes per day */
#define ae       1.0
#define ck2      5.413079E-4
#define ck4      6.209887E-7
#define f        3.352779E-3
#define ge       3.986008E5
#define s        1.012229
#define qoms2t   1.880279E-09
#define secday   8.6400E4        /* Seconds per day */
#define omega_E  1.0027379
#define omega_ER 6.3003879
#define zns      1.19459E-5
#define c1ss     2.9864797E-6
#define zes      1.675E-2
#define znl      1.5835218E-4
#define c1l      4.7968065E-7
#define zel      5.490E-2
#define zcosis   9.1744867E-1
#define zsinis   3.9785416E-1
#define zsings  -9.8088458E-1
#define zcosgs   1.945905E-1
#define zcoshs   1
#define zsinhs   0
#define q22      1.7891679E-6
#define q31      2.1460748E-6
#define q33      2.2123015E-7
#define g22      5.7686396
#define g32      9.5240898E-1
#define g44      1.8014998
#define g52      1.0508330
#define g54      4.4108898
#define root22   1.7891679E-6
#define root32   3.7393792E-7
#define root44   7.3636953E-9
#define root52   1.1428639E-7
#define root54   2.1765803E-9
#define thdt     4.3752691E-3
#define rho      1.5696615E-1
#define mfactor  7.292115E-5
#define sr       6.96000E5      /* Solar radius - kilometers (IAU 76)      */
#define AU       1.49597870E8   /* Astronomical unit - kilometers (IAU 76) */
#define Cl		 2.99792458E8	/* Speed of light m/sec */

/* Entry points of Deep() */
#define dpinit   1 /* Deep-space initialization code */
#define dpsec    2 /* Deep-space secular code        */
#define dpper    3 /* Deep-space periodic code       */

/*-------------------------------------------------------------------*/

/* Function Prototypes produced by cproto */
/* comms.c */
void Rotor_Send_String(char *strn);
void Rotor_Read_String(char *strn);
void Rotor_Send_Direction(int azim, int elev);
void Rotor_Read_Direction(int *azim, int *elev);
void Read_Tcvr_Status(tcvr_status_t *tcvr_status);
void Write_Tcvr_Status(tcvr_status_t *tcvr_status);
/* date_time.c */
double Julian_Date_of_Epoch(double epoch);
double Epoch_Time(double jd);
int DOY(int yr, int mo, int dy);
double Fraction_of_Day(int hr, int mi, int se);
void Calendar_Date(double jd, struct tm *cdate);
void Time_of_Day(double jd, struct tm *cdate);
double Julian_Date(struct tm *cdate);
void Date_Time(double julian_date, struct tm *cdate);
int Check_Date(struct tm *cdate);
struct tm Time_to_UTC(struct tm *cdate);
struct tm Time_from_UTC(struct tm *cdate);
double JD_to_UTC(double jt);
double JD_from_UTC(double jt);
double Delta_ET(double year);
double Julian_Date_of_Year(double year);
double ThetaG(double epoch, deep_arg_t *deep_arg);
double ThetaG_JD(double jd);
void Calendar_Time_Now(struct tm *utc, struct tm *localtm);
double Current_Daynum(void);
double Theta(double julian_utc, double lon);
/* earthtrack.c */
double Arc_Cos(double x, double y);
void Visibility_to_Color(satellite_status_t *sat_status, char *color);
void Range_Lat_Lon(satellite_status_t *sat_status, int azim, double *range_lat, double *range_lon);
void Range_Circle(satellite_status_t *sat_status);
int Display_Satellite(satellite_status_t *sat_status, observer_status_t *obs_status);
int Config_Xplanet(void);
/* gridloc.c */
void Position_to_Grid_Locator(double lon, double lat, char *grid_loc);
void Grid_Locator_to_Position(char *grid_loc, double *lon, double *lat);
void Degrees_to_DMS(double deg, int *ideg, int *imin, int *isec);
void DMS_to_Degrees(int ideg, int imin, int isec, double *deg);
void Bearing_Distance(double lon_a, double lat_a, double lon_b, double lat_b, double *bearing, double *distance);
/* input.c */
void Load_Database_Tle_Data(void);
void Load_Tle_Set(char *sat_name, tle_t *sat_tle);
int Read_Transponder_Data(char *sat_name, satellite_mode_t *sat_mod);
unsigned char Read_Operating_Mode(char *line);
void Request_New_Satellite(tle_t *sat_tle, satellite_status_t *sat_status, observer_status_t *obs_status);
void Find_New_Satellite(int which, tle_t *sat_tle, satellite_status_t *sat_status, observer_status_t *obs_status);
void New_Satellite(tle_t *sat_tle, satellite_status_t *sat_status, observer_status_t *obs_status);
void Set_Accessibility_Flags(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
int Aos_Possible(tle_t *sat_tle, observer_status_t *obs_status);
int Satellite_Decayed(tle_t *sat_tle, double julian_utc);
int Satellite_Geostationary(tle_t *sat_tle);
void Convert_Satellite_Data(char *tle_set, tle_t *tle);
void Select_Ephemeris(tle_t *tle);
int Checksum_Good(char *tle_set);
int Good_Elements(char *tle_set);
void Load_Observer_Data(void);
void Load_Observer_Set(char *loc_name, observer_status_t *obs_status);
void New_Observer(tle_t *sat_tle, satellite_status_t *sat_status, observer_status_t *obs_status);
void Save_Location(observer_status_t obs_status);
int Load_Line(char *buff, int max_chr, FILE *pfile, int err_num);
/* math.c */
int Sign(double arg);
double Sqr(double arg);
double Cube(double arg);
double Radians(double arg);
double Degrees(double arg);
double ArcSin(double arg);
double ArcCos(double arg);
void Magnitude(vector_t *v);
void Vec_Add(vector_t *v1, vector_t *v2, vector_t *v3);
void Vec_Sub(vector_t *v1, vector_t *v2, vector_t *v3);
void Scalar_Multiply(double k, vector_t *v1, vector_t *v2);
void Scale_Vector(double k, vector_t *v);
double Dot(vector_t *v1, vector_t *v2);
double Angle(vector_t *v1, vector_t *v2);
void Cross(vector_t *v1, vector_t *v2, vector_t *v3);
void Normalize(vector_t *v);
double AcTan(double sinx, double cosx);
double FMod2p(double x);
double Modulus(double arg1, double arg2);
double Frac(double arg);
int Round(double arg);
double Int(double arg);
void Convert_Sat_State(kinetics_t *sat_kinetics);
double Footprint(vector_t *pos);
/* mode.c */
void Main_Screen_Mode(void);
void Single_Satellite_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Multi_Satellite_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Multi_Observer_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Pass_Prediction_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Illumination_Prediction_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Manual_Command_Mode(tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Calculate_Satellite_Status(double julian_utc, tle_t *sat_tle, kinetics_t *solar_kinetics, kinetics_t *obs_kinetics, observer_status_t *obs_status, satellite_status_t *sat_status);
void Enable_Tracking(void);
void Disable_Tracking(void);
void Manual_Position_Rotors(int azim, int elev);
void Auto_Track(double julian_utc, satellite_status_t *sat_status);
void Enable_CAT(void);
void Disable_CAT(void);
void Initialize_Tcvr(int mode_idx);
void CAT_Control(satellite_status_t *sat_status);
/* observer.c */
void Calculate_User_PosVel(double time, observer_status_t *obs_status, kinetics_t *obs_kinetics);
void Calculate_LatLonAlt(double time, vector_t *pos, geodetic_t *geodetic);
void Calculate_Observer(kinetics_t *sat_kinetics, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
void Calculate_RADec(double time, kinetics_t *sat_kinetics, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
void RaDec_to_AzEl(double ra, double dec, double theta, geodetic_t *obs_geodetic, vector_t *lunar_set);
void RaDec_to_Topocentric(double ra, double dec, vector_t *topocentric);
double Find_Horizon_Crossing(double julian_utc, tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Find_Next_AOS_LOS(double julian_utc, tle_t *sat_tle, observer_status_t *obs_status, satellite_status_t *sat_status);
void Calculate_Orbit(double julian_utc, tle_t *sat_tle, geodetic_t *sat_geodetic, kinetics_t *sat_kinetics, satellite_status_t *sat_status, observer_status_t *obs_status, kinetics_t *obs_kinetics, vector_t *obs_set);
/* satcom.c */
int main(void);
void Abort_On_Error(int why);
int isFlagSet(int flag);
int isFlagClear(int flag);
void Set_Flag(int flag);
void Clear_Flag(int flag);
void Toggle_Flag(int flag);
int isSatFlagSet(satellite_status_t *sat_status, int flag);
int isSatFlagClear(satellite_status_t *sat_status, int flag);
void Set_SatFlag(satellite_status_t *sat_status, int flag);
void Clear_SatFlag(satellite_status_t *sat_status, int flag);
/* screen.c */
void Screen_Root(void);
void Screen_Main(void);
void Screen_Observer_Status(observer_status_t *obs_status);
void Screen_Multiobserver_Status(int num_loc, observer_status_t *multiobs_status, satellite_status_t *mxsat_status);
void Screen_Satellite_Status(satellite_status_t *sat_status);
void Screen_Multisatellite_Status(int *stats, observer_status_t *obs_status, satellite_status_t *sat_status, satellite_status_t *multisat_status);
void Screen_Pass_Predictions(int num, pass_data_t *pass_data, observer_status_t *obs_status, satellite_status_t *sat_status);
void Screen_Illumination_Predictions(double *jdate, int *illum, observer_status_t *obs_status, satellite_status_t *sat_status);
void Screen_Progress_Bar(int prog);
void Screen_Rotor_Control(double azim, double elev, int resol, int displ);
void Screen_CAT_Control(tcvr_status_t *tcvr_status);
void Screen_Commands(int mode);
void Screen_Offset_Calibration(void);
void Screen_Fullscale_Calibration(void);
void Post_Manual_Form(int *azim_dir, int *elev_dir);
void Post_Name_Form(char *name, WINDOW *form_win);
void Post_Location_Form(int num, observer_status_t *multiobs_status);
void Post_UserTime_Form(double *user_utc);
void Post_Grid_Locator_Form(void);
void Read_Position(FIELD *data_field[], double *lon, double *lat, int row);
void Read_Locator(FIELD *data_field[], double *lon, double *lat, int row);
void Draw_Box(int rows, int cols, int top, int left);
void Draw_Hline(WINDOW *win, int top, int left, int right);
void Draw_Vline(WINDOW *win, int top, int left, int bottom);
/* serial.c */
void Open_Tcvr_Serial(void);
int Write_Tcvr_Serial(unsigned char *ch);
int Read_Tcvr_Serial(unsigned char *ch, int block_size);
void Close_Tcvr_Serial(void);
void Open_Rotor_Serial(void);
int Write_Rotor_Serial(char *ch);
int Read_Rotor_Serial(void);
void Close_Rotor_Serial(void);
/* sgp4sdp4.c */
void SGP4(double tsince, tle_t *tle, kinetics_t *sat_kinetics);
void SDP4(double tsince, tle_t *tle, kinetics_t *sat_kinetics);
void Deep(int ientry, tle_t *tle, deep_arg_t *deep_arg);
/* solar_lunar.c */
void Calculate_Solar_Position(double time, kinetics_t *solar_kinetics);
int Satellite_Eclipsed(kinetics_t *sat_kinetics, kinetics_t *sol_kinetics, double *depth);
void Lunar_Position(double daynum, geodetic_t *obs_geodetic, vector_t *lunar_set);
void Calculate_Moon_Position(double julian_utc,
	observer_status_t *obs_status, kinetics_t *obs_kinetics);

/*-------------------------------------------------------------------*/

#endif /* End of satcom.h */


