/*  satcom.c
 *
 *  Main body of satcom application
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "satcom.h"

/* All-satellite TLE data */
tle_t *tle_data = NULL;

/* Number of entries in tle_data */
int num_tles;

/* TLE set currently in use */
int idx_tle;

/* All-observer location data */
observer_status_t *obs_data = NULL;

/* Number of entries in obs_data */
int num_obs;

/* Observer set currently in use */
int idx_obs;

/* Signal handler */
static void sig_handler( int signal );

/* File descriptors for xplanet files */
FILE
  *marker_fd   = NULL,
  *greatarc_fd = NULL,
  *config_fd   = NULL;

/* Strings for xplanet's file names */
char
  xplanet_markerfile[50],
  xplanet_greatarcfile[50],
  xplanet_configfile[50],
  xplanet_command[200];


  int
main( void )
{
  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Initialize ncurses */
  initscr();
  keypad( stdscr, TRUE );
  nonl();
  cbreak();
  noecho();
  curs_set( FALSE );
  halfdelay( UPDATE_DELAY );

  /* Assuming terminal has color! */
  if( has_colors() )
  {
	start_color();

	init_pair( COLOR_WHITE,   COLOR_WHITE,  COLOR_BLACK  );
	init_pair( COLOR_RED,	  COLOR_RED,    COLOR_BLACK  );
	init_pair( COLOR_GREEN,	  COLOR_GREEN,  COLOR_BLACK  );
	init_pair( COLOR_BLUE,	  COLOR_BLUE,   COLOR_BLACK  );
	init_pair( COLOR_YELLOW,  COLOR_YELLOW, COLOR_BLACK  );
	init_pair( COLOR_CYAN,	  COLOR_CYAN,   COLOR_BLACK  );
	init_pair( COLOR_MAGENTA,   COLOR_MAGENTA,COLOR_BLACK  );
	init_pair( COLOR_KEYSTROKE, COLOR_YELLOW, COLOR_BLUE   );
	init_pair( COLOR_COMMAND,   COLOR_WHITE,  COLOR_BLUE   );
	init_pair( COLOR_ERROR,     COLOR_WHITE,  COLOR_RED    );
	init_pair( COLOR_APPROX,    COLOR_WHITE,  COLOR_YELLOW );
	init_pair( COLOR_FINISH,    COLOR_WHITE,  COLOR_GREEN  );
  }

  /* Prepare xplanet's file names and command */
  snprintf( xplanet_markerfile,   50, "%s/%s", XPLANET_DIR, XPLANET_MARKERFILE );
  snprintf( xplanet_greatarcfile, 50, "%s/%s", XPLANET_DIR, XPLANET_ARCFILE );
  snprintf( xplanet_configfile,   50, "%s/%s", XPLANET_DIR, XPLANET_CONFIGFILE );
  snprintf( xplanet_command, 200,
	  "xplanet -config %s -wait 5 -window"
	  " -geometry 1024x640 -projection rectangular &",
	  xplanet_configfile );

  /* Set initial display to Main screen */
  Main_Screen_Mode();

  free( tle_data );
  free( obs_data );

  /* Wrap up and quit on 'Q' */
  sig_handler( 0 );

  return(0);

} /* End of main() */

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Cleanup */
  clear();
  refresh();
  endwin();

  /* Delete xplanet files */
  unlink( xplanet_markerfile );
  unlink( xplanet_greatarcfile );
  unlink( xplanet_configfile );
  system( "pkill xplanet" );

  /* Return transceiver to original status */
  if( isFlagSet(CAT_ENABLED_FLAG) )
	Disable_CAT();

  fprintf( stderr, "\n" );
  switch( signal )
  {
	case 0 :
	  exit( 0 );

	case SIGINT :
	  fprintf( stderr, "%s\n",  "satcom: Exiting via User Interrupt" );
	  break;

	case SIGSEGV :
	  fprintf( stderr, "%s\n",  "satcom: Segmentation Fault" );
	  break;

	case SIGFPE :
	  fprintf( stderr, "%s\n",  "satcom: Floating Point Exception" );
	  break;

	case SIGABRT :
	  fprintf( stderr, "%s\n",  "satcom: Abort Signal received" );
	  break;

	case SIGTERM :
	  fprintf( stderr, "%s\n",  "satcom: Termination Request received" );
  }

  exit( signal );

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

/*  Abort_On_Error()
 *
 *  Closes open files and ends ncurses,
 *  then prints an error message and exits
 */

  void
Abort_On_Error( int why )
{
  /* Cleanup */
  clear();
  refresh();
  endwin();

  /* Delete xplanet files */
  unlink( xplanet_markerfile );
  unlink( xplanet_greatarcfile );
  unlink( xplanet_configfile );
  system( "pkill xplanet" );

  fprintf( stderr, "\n" );
  switch( why )
  {
	case -1 :
	  fprintf( stderr, "%s\n",
		  "satcom: Error reading ~/.satcom/satcom.tle file - exiting");
	  break;

	case -2 :
	  fprintf( stderr, "%s\n",
		  "satcom: Incorrect number of lines in ~/.satcom/satcom.tle file - exiting");
	  break;

	case -3 :
	  fprintf( stderr, "%s\n",
		  "satcom: Invalid TLE set in ~/.satcom/satcom.tle  - exiting");
	  break;

	case -4 :
	  fprintf( stderr, "%s\n",
		  "satcom: Error reading ~/.satcom/satcom.obs file - exiting");
	  break;

	case -5 :
	  fprintf( stderr, "%s\n", "satcom: No valid observer location name found "
		  "in ~/.satcom/satcom.obs file - exiting");
	  break;

	case -6 :
	  fprintf( stderr, "%s\n", "satcom: Invalid observer location data "
		  "in ~/.satcom/satcom.obs file - exiting");
	  break;

	case -7 :
	  fprintf( stderr, "%s\n", "satcom: Conflicting Grid Locator/Position data in "
		  "~/.satcom/satcom.obs file - exiting");
	  break;

	case -8 :
	  fprintf( stderr, "%s\n",
		  "satcom: Call to malloc() for TLE database failed - exiting");
	  break;

	case -9 :
	  fprintf( stderr, "%s\n",
		  "satcom: Call to malloc() for observer database failed - exiting");
	  break;

	case -10 :
	  fprintf( stderr, "%s\n",
		  "satcom: Call to malloc() for multisat status buffer failed - exiting");
	  break;

	case -11 :
	  fprintf( stderr, "%s\n",
		  "satcom: Error reading ~/.satcom/satcom.sat file - exiting");
	  break;

	case -12 :
	  fprintf( stderr, "%s\n",
		  "satcom: No satellite names found in ~/.satcom/satcom.sat - exiting");
	  break;

	case -13 :
	  fprintf( stderr, "%s\n", "satcom: No matching satellite name found in "
		  "~/.satcom/satcom.tle - exiting");
	  break;

	case -14 :
	  Disable_CAT();
	  fprintf( stderr, "%s\n", "satcom: Satellite transponder data in "
		  "~/.satcom/satcom.sat incomplete - exiting");
	  break;

	case -15 :
	  Disable_CAT();
	  fprintf( stderr, "%s\n", "satcom: Transponder passband flag in "
		  "~/.satcom/satcom.sat not recognized - exiting");
	  break;

	case -16 :
	  fprintf( stderr, "%s\n",  "satcom: Call to malloc() for transponder "
		  "modes buffer failed - exiting");

  } /* End of switch( why ) */

  exit( why );

} /* End of Abort_On_Error() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flow control flags
 *
 *  See satcom.h for definition of flow control flags
 */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet(int flag)
{
  return( (Flags & flag) == flag );
}

  int
isFlagClear(int flag)
{
  return( (~Flags & flag) == flag );
}

  void
Set_Flag(int flag)
{
  Flags |= flag;
}

  void
Clear_Flag(int flag)
{
  Flags &= ~flag;
}

  void
Toggle_Flag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

/*  Functions for testing and setting satellite status flags
 *
 *  See satcom.h for definition of satellite status flags
 */

  int
isSatFlagSet(satellite_status_t *sat_status, int flag)
{
  return( (sat_status->flags & flag) == flag );
}

  int
isSatFlagClear(satellite_status_t *sat_status, int flag)
{
  return( (~sat_status->flags & flag) == flag );
}

  void
Set_SatFlag(satellite_status_t *sat_status, int flag)
{
  sat_status->flags |= flag;
}

  void
Clear_SatFlag(satellite_status_t *sat_status, int flag)
{
  sat_status->flags &= ~flag;
}

/*------------------------------------------------------------------*/

