/*  solar_lunar.c
 *
 *  Functions to calculate the position of sun and moon
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "satcom.h"

/*  Calculate_Solar_Position()
 *
 *  Calculates solar position vector
 */

  void
Calculate_Solar_Position( double time, kinetics_t *solar_kinetics )
{
  double mjd, year, T, M, L, e, C, O, Lsa, nu, R, eps;

  mjd = time - 2415020.0;
  year = 1900 + mjd/365.25;
  T = (mjd + Delta_ET(year)/secday)/36525.0;
  M = Radians(Modulus(358.47583 + Modulus(35999.04975*T,360.0)
		- (0.000150 + 0.0000033*T)*Sqr(T),360.0));
  L = Radians(Modulus(279.69668 + Modulus(36000.76892*T,360.0)
		+ 0.0003025*Sqr(T),360.0));
  e = 0.01675104 - (0.0000418 + 0.000000126*T)*T;
  C = Radians((1.919460 - (0.004789 + 0.000014*T)*T)*sin(M)
	  + (0.020094 - 0.000100*T)*sin(2*M) + 0.000293*sin(3*M));
  O = Radians(Modulus(259.18 - 1934.142*T,360.0));
  Lsa = Modulus(L + C - Radians(0.00569 - 0.00479*sin(O)),twopi);
  nu = Modulus(M + C,twopi);
  R = 1.0000002*(1 - Sqr(e))/(1 + e*cos(nu));
  eps = Radians(23.452294 - (0.0130125 + (0.00000164 -
		  0.000000503*T)*T)*T + 0.00256*cos(O));
  R = AU*R;

  solar_kinetics->pos.x = R*cos(Lsa);
  solar_kinetics->pos.y = R*sin(Lsa)*cos(eps);
  solar_kinetics->pos.z = R*sin(Lsa)*sin(eps);
  solar_kinetics->pos.w = R;

} /* End of Calculate_Solar_Position() */

/*------------------------------------------------------------------*/

/*  Satellite_Eclipsed()
 *
 *  Calculates stellite's eclipse status and depth
 */

int
Satellite_Eclipsed(
	kinetics_t *sat_kinetics,
	kinetics_t *sol_kinetics,
	double *depth)
{
  double sd_sun, sd_earth, delta;
  vector_t Rho, earth;

  /* Determine partial eclipse */
  sd_earth = ArcSin(xkmper/sat_kinetics->pos.w);
  Vec_Sub( &sol_kinetics->pos, &sat_kinetics->pos, &Rho);
  sd_sun = ArcSin(sr/Rho.w);
  Scalar_Multiply( -1, &sat_kinetics->pos, &earth);
  delta = Angle( &sol_kinetics->pos, &earth);
  *depth = sd_earth - sd_sun - delta;

  if( sd_earth < sd_sun )
	return( 0 );
  else
	if( *depth >= 0 )
	  return( 1 );
	else
	  return( 0 );

} /* End of Function Sat _Eclipsed() */

/*------------------------------------------------------------------*/

/*  Lunar_Position()
 *
 *  Calculates azimuth and elevation of moon from observer.
 *  Adapted from code in the 'predict' tracker but is not in use.
 */

void
Lunar_Position(
	double daynum,
	geodetic_t *obs_geodetic,
	vector_t *lunar_set)
{
  double
	t1, t2, t3, ew, en, e, n, si, co, ra, gh,
	lh, gm, de, se, gs, dc, el, az, mm, yr, ro;

  gm=24.0*(daynum-floor(daynum));
  ew=FMod2p(1.134193+daynum*0.229971506);
  mm=FMod2p(1.319238+daynum*0.228027135);
  t1=FMod2p(6.217512+daynum*0.01720196977);
  t2=2.0*FMod2p(2.550677+daynum*0.212768711);
  t3=FMod2p(4.7652214+daynum*0.230895723);
  ew=ew+0.01148*sin(t2)+0.10976*sin(mm);
  ew=ew-0.022235*sin(mm-t2)-0.003246*sin(t1);
  ew=ew+0.003735*sin(2.0*mm)-0.0019897*sin(2.0*t3);
  ew=ew-0.0010297*sin(2.0*mm-t2)-0.0009948*sin(mm+t1-t2);
  en=t3+0.0115070*sin(t2)+0.10873924*sin(mm);
  en=en-0.0222006*sin(mm-t2);
  en=0.0897797*sin(en)-0.002548*sin(t3-t2);

  si=zcosis*sin(en)+zsinis*cos(en)*sin(ew);
  co=sqrt(1.0-(si*si));
  dc=atan2(si,co);
  si=sin(ew)*zcosis-tan(en)*zsinis;
  co=cos(ew);
  ra=atan2(si,co);

  if (ra<0.0)
	ra+=twopi;

  t1=(double)(long)(daynum-39410.0);
  t2=floor((t1+32044.75)/36524.25);
  t2+=t1-t2/4.0+1486.0;
  yr=2084.0+floor((t2-122.1)/365.25);

  t1=yr-1.0;
  de=floor(365.25*(t1-1980.0))-floor(t1/100.0)+floor(t1/400.0)+381.0;
  t1=(de+29218.5)/36525.0;
  t1=6.6460656+t1*(2400.051262+t1*2.581e-5);
  se=t1-24.0*(yr-1900.0);
  t1=(se+0.0657098*(daynum-de)+gm*1.002738)/24.0;
  gs=24.0*(t1-floor(t1));

  t1=gs/24.0-ra/twopi;
  gh=twopi*(t1-floor(t1));

  n=obs_geodetic->lat;   /* North latitude of tracking station */
  e=obs_geodetic->lon ;  /* East longitude of tracking station */

  lh=gh+e;

  si=cos(lh)*cos(dc)*cos(n)+sin(dc)*sin(n);
  co=sqrt(1.0-(si*si));
  el=atan2(si,co);

  si=-sin(lh)*cos(dc)*cos(n);
  co=sin(dc)-sin(n)*sin(el);
  az=atan2(si,co);

  if (az<0.0)
	az+=twopi;

  ro=0.996986/(1.0+0.0549*cos(mm+0.10976*sin(mm)));
  el=el-0.0166*cos(el)/ro;

  lunar_set->x=az;
  lunar_set->y=el;

} /* End of Lunar_Position() */

/*------------------------------------------------------------------*/

/*  Moon_Position()
 *
 *  Calculates azimuth, elevation and range to moon from observer.
 *  Adapted from code in the 'trk' tracker and is in use in satcom.
 */

void
Moon_Position(
	double julian_utc,
	observer_status_t *obs_status,
	kinetics_t *obs_kinetics )
{
  /* Lunar perturbation constants */
  const lunar_perterb_t ptml[60]=
  {
	{0,	  0,	1,	 0,	6288774.},
	{2,	  0,   	1,	 0,	1274027.},
	{2,	  0,	0,	 0,	 658314.},
	{0,	  0,	2,	 0,	 213618.},
	{0,	  1,	0,	 0,	-185116.},
	{0,	  0,	0,	 2,	-114332.},
	{2,	  0,   -2,	 0,	  58793.},
	{2,  -1,   -1,	 0,	  57066.},
	{2,	  0,	1,	 0,	  53322.},
	{2,  -1,	0,	 0,	  45758.},
	{0,	  1,   -1,	 0,	 -40923.},
	{1,	  0,	0,	 0,	 -34720.},
	{0,	  1,	1,	 0,	 -30383.},
	{2,	  0,	0,	-2,	  15327.},
	{0,	  0,	1,	 2,	 -12528.},
	{0,	  0,	1, 	-2,	  10980.},
	{4,	  0,   -1,	 0,	  10675.},
	{0,	  0,	3,	 0,	  10034.},
	{4,	  0,   -2,	 0,	   8548.},
	{2,	  1,   -1,	 0,	  -7888.},
	{2,	  1,	0,	 0,	  -6766.},
	{1,	  0,   -1,	 0,	  -5163.},
	{1,	  1,	0,	 0,	   4987.},
	{2,  -1,	1,	 0,	   4036.},
	{2,	  0,	2,	 0,	   3994.},
	{4,	  0,	0,	 0,	   3861.},
	{2,	  0,   -3,	 0,	   3665.},
	{0,	  1,   -2,	 0,	  -2689.},
	{2,	  0,   -1,	 2,	  -2602.},
	{2,  -1,   -2,	 0,	   2390.},
	{1,	  0,	1,	 0,	  -2348.},
	{2,  -2,	0,	 0,	   2236.},
	{0,	  1,	2,	 0,	  -2120.},
	{0,	  2,	0,	 0,	  -2069.},
	{2,  -2,   -1,	 0,    2048.},
	{2,	  0,	1, 	-2,	  -1773.},
	{2,	  0,	0,	 2,	  -1595.},
	{4,  -1,   -1,	 0,	   1215.},
	{0,	  0,	2,	 2,	  -1110.},
	{3,	  0,   -1,	 0,	   -892.},
	{2,	  1,	1,	 0,	   -810.},
	{4,  -1,   -2,	 0,     759.},
	{0,	  2,   -1,	 0,    -713.},
	{2,	  2,   -1,	 0,	   -700.},
	{2,	  1,   -2,	 0,	    691.},
	{2,  -1,	0, 	-2,	    596.},
	{4,	  0,	1,	 0, 	549.},
	{0,	  0,	4,	 0,	    537.},
	{4,  -1,	0,	 0,	    520.},
	{1,	  0,   -2,	 0,	   -487.},
	{2,	  1,	0, 	-2,	   -399.},
	{0,	  0,	2, 	-2,	   -381.},
	{1,	  1,	1,	 0,	    351.},
	{3,	  0,   -2,	 0,	   -340.},
	{4,	  0,   -3,	 0,	    330.},
	{2,  -1,	2,	 0,	    327.},
	{0,	  2,	1,	 0,	   -323.},
	{1,	  1,   -1,	 0,	    299.},
	{2,	  0,	3,	 0,	    294.},
	{2,	  0,   -1, 	-2,	      0.}
  };

  const lunar_perterb_t ptmr[60]=
  {
	{0,	  0,	1,	 0,-20905355.},
	{2,	  0,   -1,	 0,	-3699111.},
	{2,	  0,	0,	 0, -2955968.},
	{0,	  0,	2,	 0,  -569925.},
	{0,	  1,	0,	 0,    48888.},
	{0,	  0,	0,	 2,	   -3149.},
	{2,	  0,   -2,	 0,	  246158.},
	{2,  -1,   -1,	 0,	 -152138.},
	{2,	  0,	1,	 0,	 -170733.},
	{2,  -1,	0,	 0,	 -204586.},
	{0,	  1,   -1,	 0,	 -129620.},
	{1,	  0,	0,	 0,	  108743.},
	{0,	  1,	1,	 0,	  104755.},
	{2,	  0,	0, 	-2,	   10321.},
	{0,	  0,	1,	 2,	       0.},
	{0,	  0,	1,	-2,	   79661.},
	{4,	  0,   -1,	 0,	  -34782.},
	{0,	  0,	3,	 0,	  -23210.},
	{4,	  0,   -2,	 0,	  -21636.},
	{2,	  1,   -1,	 0,	   24208.},
	{2,	  1,	0,	 0,	   30824.},
	{1,	  0,   -1,	 0,	   -8379.},
	{1,	  1,	0,	 0,	  -16675.},
	{2,  -1,	1,	 0,	  -12831.},
	{2,	  0,	2,	 0,	  -10445.},
	{4,	  0,	0,	 0,	  -11650.},
	{2,	  0,   -3,	 0,	   14403.},
	{0,	  1,   -2,	 0,	   -7003.},
	{2,	  0,   -1,	 2,	       0.},
	{2,  -1,   -2,	 0,	   10056.},
	{1,	  0,	1,	 0,	    6322.},
	{2,  -2,	0,	 0,	   -9884.},
	{0,	  1,	2,	 0,	    5751.},
	{0,	  2,	0,	 0,	       0.},
	{2,  -2,   -1,	 0,	   -4950.},
	{2,	  0,	1, 	-2,	    4130.},
	{2,	  0,	0,	 2,	       0.},
	{4,  -1,   -1,	 0,	   -3958.},
	{0,	  0,	2,	 2,	       0.},
	{3,	  0,   -1,	 0,	    3258.},
	{2,	  1,	1,	 0,	    2616.},
	{4,  -1,   -2,	 0,	   -1897.},
	{0,	  2,   -1,	 0,	   -2117.},
	{2,	  2,   -1,	 0,	    2354.},
	{2,	  1,   -2,	 0,	       0.},
	{2,  -1,	0, 	-2,	       0.},
	{4,	  0,	1,	 0,	   -1423.},
	{0,	  0,	4,	 0,	   -1117.},
	{4,  -1,	0,	 0,	   -1571.},
	{1,	  0,   -2,	 0,	   -1739.},
	{2,	  1,	0, 	-2,	       0.},
	{0,	  0,	2, 	-2,	   -4421.},
	{1,	  1,	1,	 0,	       0.},
	{3,	  0,   -2,	 0,	       0.},
	{4,	  0,   -3,	 0,	       0.},
	{2,  -1,	2,	 0,	       0.},
	{0,	  2,	1,	 0,	    1165.},
	{1,	  1,   -1,	 0,	       0.},
	{2,	  0,	3,	 0,	       0.},
	{2,	  0,   -1, 	-2,	    8752.}
  };

  const lunar_perterb_t ptmb[60]=
  {
	{0,   0,	0,	 1,	5128122.},
	{0,	  0,	1,	 1,	 280602.},
	{0,	  0,	1, 	-1,	 277693.},
	{2,	  0,	0, 	-1,	 173237.},
	{2,	  0,   -1,	 1,	  55413.},
	{2,	  0,   -1, 	-1,	  46271.},
	{2,   0,	0,	 1,	  32573.},
	{0,   0,	2,	 1,	  17198.},
	{2,	  0,	1, 	-1,	   9266.},
	{0,	  0,	2, 	-1,	   8822.},
	{2,  -1,	0, 	-1,	   8216.},
	{2,   0,   -2, 	-1,	   4324.},
	{2,   0,	1, 	 1,	   4200.},
	{2,	  1,	0, 	-1,	  -3359.},
	{2,  -1,   -1,	 1,	   2463.},
	{2,  -1,	0,	 1,	   2211.},
	{2,  -1,   -1, 	-1,	   2065.},
	{0,	  1,   -1, 	-1,	  -1870.},
	{4,	  0,   -1, 	-1,	   1828.},
	{0,	  1,	0,	 1,	  -1794.},
	{0,	  0,	0,	 3,	  -1749.},
	{0,	  1,   -1,	 1,	  -1565.},
	{1,	  0,	0,	 1,	  -1491.},
	{0,	  1,	1,	 1,	  -1475.},
	{0,	  1,	1, 	-1,	  -1410.},
	{0,	  1,	0, 	-1,	  -1344.},
	{1,	  0,	0, 	-1,	  -1335.},
	{0,	  0,	3, 	 1,	   1107.},
	{4,	  0,	0, 	-1,	   1021.},
	{4,	  0,   -1,   1,	    833.},
	{0,	  0,	1, 	-3,	    777.},
	{4,	  0,   -2,	 1,	    671.},
	{2,	  0,	0, 	-3,	    607.},
	{2,	  0,	2, 	-1,	    596.},
	{2,  -1,	1, 	-1,	    491.},
	{2,   0,   -2,   1,	   -451.},
	{0,	  0,	3, 	-1,	    439.},
	{2,	  0,	2,	 1,	    422.},
	{2,	  0,   -3, 	-1,	    421.},
	{2,	  1,   -1,	 1,	   -366.},
	{2,	  1,	0,	 1,	   -351.},
	{4,	  0,	0,	 1,	    331.},
	{2,  -1,	1,	 1,	    315.},
	{2,  -2,	0, 	-1,	    302.},
	{0,	  0,	1,   3,	   -283.},
	{2,	  1,	1, 	-1,	   -229.},
	{1,	  1,	0, 	-1,	    223.},
	{1,	  1,	0,	 1,	    223.},
	{0,	  1,   -2, 	-1,	   -220.},
	{2,	  1,   -1, 	-1,	   -220.},
	{1,	  0,	1,	 1,	   -185.},
	{2,  -1,   -2, 	-1,	    181.},
	{0,	  1,	2,	 1,	   -177.},
	{4,	  0,   -2, 	-1,	    176.},
	{4,  -1,   -1, 	-1,	    166.},
	{1,	  0,	1, 	-1,	   -164.},
	{4,	  0,	1, 	-1,	    132.},
	{1,	  0,   -1, 	-1,	   -119.},
	{4,  -1,	0, 	-1,	    115.},
	{2,  -2,	0,	 1,	    107.}
  };

  double
	t, L1, D, M, M1, F, A1, A2, A3, sl, srr,
	sb, lambda, beta, eps, Rm, ra, dec, theta;

  int i;

  vector_t lunar_pos;

  t=(julian_utc-2451545.0)/36525.;
  L1=3.810341023+(8399.709113+(-2.3157E-5+(3.23904E-8 -1.56771E-10*t)*t)*t)*t;
  D=5.198466741+(7771.377147+(-2.8449E-5+(3.19735E-8 -1.56804E-11*t)*t)*t)*t;
  M=6.240060127+(628.3019552+(-2.6808E-6+7.1267E-10*t)*t)*t;
  M1=2.355555899+(8328.691426+(1.5703E-4+(2.5041E-7 -1.1863E-9*t)*t)*t)*t;
  F=1.627905233+(8433.466158+(-5.9392E-5+(-4.9499E-9 +2.02167E-11*t)*t)*t)*t;

  A1=2.09003+2.3012*t;
  A2=0.9266+8364.73985*t;
  A3=5.4707+8399.68473*t;

  /* Si trascura l'effetto della riduzione dell'   */
  /* eccentricita` dell'orbita terrestre, per ora  */

  sl=0.;  srr=0.;  sb=0.;
  for(i=0; i<NTERM; i++)
  {
	sl+=ptml[i].k*sin(ptml[i].D*D+ptml[i].M*M+ptml[i].M1*M1+ptml[i].F*F);
	srr+=ptmr[i].k*cos(ptmr[i].D*D+ptmr[i].M*M+ptmr[i].M1*M1+ptmr[i].F*F);
	sb+=ptmb[i].k*sin(ptmb[i].D*D+ptmb[i].M*M+ptmb[i].M1*M1+ptmb[i].F*F);
  }

  sl+=3958.*sin(A1)+1962*sin(L1-F)+318*sin(A2);
  sb+=-2235*sin(L1)+382*sin(A3)+175*sin(A1-F)+175*
	sin(A1+F)+127*sin(L1-M1)-115*sin(L1+M1);

  /* La trasformazione da lambda,beta a ra,dec va messa in coord.c */

  lambda=Radians( sl/1.E6 ) + L1;
  beta=Radians( sb/1.E6 );
  eps=0.409092804-(2.269655E-4+(2.86E-9+8.7897E-9*t)*t)*t;

  ra=atan2(sin(lambda)*cos(eps)-tan(beta)*sin(eps), cos(lambda));
  dec=asin(sin(beta)*cos(eps)+cos(beta)*sin(eps)*sin(lambda));

  Rm = 385000.56 + srr/1E3;

  RaDec_to_Topocentric( ra, dec, &lunar_pos );
  Scale_Vector( Rm, &lunar_pos );

  theta = Theta( julian_utc, Degrees(obs_status->obs_geodetic.lon) );
  RaDec_to_AzEl( ra, dec, theta,
	  &obs_status->obs_geodetic, &obs_status->lunar_set );

  obs_status->lunar_set.z = sqrt(Sqr(lunar_pos.x - obs_kinetics->pos.x)+
	  Sqr(lunar_pos.y - obs_kinetics->pos.y)+
	  Sqr(lunar_pos.z - obs_kinetics->pos.z));

  obs_status->lunar_set.x = Degrees(obs_status->lunar_set.x);
  obs_status->lunar_set.y = Degrees(obs_status->lunar_set.y);

} /* End of Moon_Position() */

/*------------------------------------------------------------------------*/

