/*  serial.c
 *
 *  Functions to handle the serial port
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


/* Serial port File descriptor */
static int rotor_serial_fd = 0;
static int  tcvr_serial_fd = 0;

/* Original serial port options */
struct termios rotor_serial_old_options;
struct termios  tcvr_serial_old_options;

#include "satcom.h"

/*  Open_Tcvr_Serial()
 *
 *  Opens Tcvr's Serial Port device, returns the file
 *  descriptor on success or -1 on error
 */

  void
Open_Tcvr_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */

  /* Open Serial device, return on error */
  tcvr_serial_fd = open( TCVR_SERIAL_DEV, O_RDWR | O_NOCTTY );
  if( tcvr_serial_fd < 0 )
	return;

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  if( fcntl( tcvr_serial_fd, F_SETLK, &lockinfo ) < 0 )
  {
	/* Cleanup */
	clear();
	refresh();
	endwin();
	fcntl( tcvr_serial_fd, F_GETLK, &lockinfo );
	printf( "satcom: "TCVR_SERIAL_DEV
		": Device already locked by pid %d%s\n", lockinfo.l_pid,
		" - aborting" );
	exit( -1 );
  }

  /* Save the current serial port options */
  tcgetattr( tcvr_serial_fd, &tcvr_serial_old_options );

  /* Read the current serial port options */
  tcgetattr( tcvr_serial_fd, &new_options );

  /* Set the i/o baud rates to 57600 */
  cfsetispeed( &new_options, B57600 );
  cfsetospeed( &new_options, B57600 );

  /* Set options for 'raw' I/O mode */
  new_options.c_iflag &=
	~( IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON );
  new_options.c_oflag &= ~( OPOST | OLCUC | ONLCR | OCRNL | ONOCR | ONLRET );
  new_options.c_lflag &= ~( ECHO | ECHONL | ICANON | ISIG | IEXTEN );
  new_options.c_cflag &= ~( CSIZE | PARENB );
  new_options.c_cflag |= ( CS8 | CLOCAL | CREAD | CSTOPB );

  /* Setup read() timeout to .2 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 2;

  /* Setup the new options for the port */
  tcsetattr( tcvr_serial_fd, TCSAFLUSH, &new_options );

} /* End of Open_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Tcvr_Serial()
 *
 *  Writing Data to the Tcvr's Serial Port
 */

  int
Write_Tcvr_Serial( unsigned char *ch )
{
  tcflush( tcvr_serial_fd, TCIOFLUSH );
  if( write( tcvr_serial_fd, ch, 5 ) < 5 ) /* Error condition */
	return( -1 );

  /* Give time to CAT to do its thing */
  usleep(2000);

  return( 0 );

} /* End of Write_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Tcvr_Serial()
 *
 *  Reading Data from the Tcvr's Serial Port
 */

  int
Read_Tcvr_Serial( unsigned char *ch, int block_size )
{
  /* Error condition */
  if( read( tcvr_serial_fd, ch, block_size ) < block_size )
  {
	tcflush( tcvr_serial_fd, TCIOFLUSH );
	Set_Flag( TCVR_READ_FAIL_FLAG );
	return( -1 );
  }
  else
  {
	Clear_Flag( TCVR_READ_FAIL_FLAG );
	return( 0 );
  }

} /* End of Read_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Close_Tcvr_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Tcvr_Serial( void )
{
  if( tcvr_serial_fd > 0 )
  {
	tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
	close( tcvr_serial_fd );
	tcvr_serial_fd = 0;
  }

} /* End of Close_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Open_Rotor_Serial()
 *
 *  Opens Rotor's Serial Port device, returns the file
 *  descriptor on success or -1 on error
 */

  void
Open_Rotor_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */

  /* Open Serial device, return on error */
  rotor_serial_fd = open( ROTOR_SERIAL_DEV, O_RDWR | O_NOCTTY );
  if( rotor_serial_fd < 0 )
	return;

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  if( fcntl( rotor_serial_fd, F_SETLK, &lockinfo ) < 0 )
  {
	/* Cleanup */
	clear();
	refresh();
	endwin();
	fcntl( rotor_serial_fd, F_GETLK, &lockinfo );
	printf( "satcom: "ROTOR_SERIAL_DEV
		": Device already locked by pid %d%s\n", lockinfo.l_pid,
		" - aborting" );
	exit( -1 );
  }

  /* Save the current serial port options */
  tcgetattr( rotor_serial_fd, &rotor_serial_old_options );

  /* Read the current serial port options */
  tcgetattr( rotor_serial_fd, &new_options );

  /* Set the i/o baud rates to 9600 */
  cfsetispeed( &new_options, B9600 );
  cfsetospeed( &new_options, B9600 );

  /* Set options for 'raw' I/O mode */
  new_options.c_iflag &=
	~( IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON );
  new_options.c_oflag &= ~OPOST;
  new_options.c_lflag &= ~( ECHO | ECHONL | ICANON | ISIG | IEXTEN );
  new_options.c_cflag &= ~( CSIZE | PARENB | CSTOPB );
  new_options.c_cflag |= ( CS8 | CLOCAL | CREAD );

  /* Setup read() timeout to .5 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 5;

  /* Setup the new options for the port */
  tcsetattr( rotor_serial_fd, TCSAFLUSH, &new_options );

} /* End of Open_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Rotor_Serial()
 *
 *  Writing Data to the Rotor's Serial Port
 */

  int
Write_Rotor_Serial( char *ch )
{
  if( write( rotor_serial_fd, ch, 1 ) < 1 ) /* Error condition */
	return( -1 );

  return( 0 );

} /* End of Write_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Rotor_Serial()
 *
 *  Reading Data from the Rotor's Serial Port
 */

  int
Read_Rotor_Serial(void)
{
  char ch;

  if( read( rotor_serial_fd, &ch, 1 ) < 1 ) /* Error condition */
  {
	Set_Flag( ROTOR_READ_FAIL_FLAG );
	return( -1 );
  }
  else
  {
	Clear_Flag( ROTOR_READ_FAIL_FLAG );
	return( ch );
  }

} /* End of Read_Rotor_Serial() */

/*-------------------------------------------------------------------------*/

/*  Close_Rotor_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Rotor_Serial( void )
{
  if( rotor_serial_fd > 0 )
  {
	tcsetattr( rotor_serial_fd, TCSANOW, &rotor_serial_old_options );
	close( rotor_serial_fd );
	rotor_serial_fd = 0;
  }

} /* End of Close_Rotor_Serial() */

/*-------------------------------------------------------------------------*/
