/*  screen.c
 *
 *  New Curses display routines for satcom
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "satcom.h"

/* All-satellite TLE data */
extern tle_t *tle_data;

/* Number of entries in tle_data */
extern int num_tles;

/* TLE set currently in use */
extern int idx_tle;

/* All-observer location data */
extern observer_status_t *obs_data;

/* Number of entries in obs_data */
extern int num_obs;

/* Observer set currently in use */
extern int idx_obs;


/*  Screen_Root()
 *
 *  Function to create the root window
 */

  void
Screen_Root( void )
{
  /* Box and add titles on stdscr */
  attrset( CYAN );
  box( stdscr, ACS_VLINE, ACS_HLINE );
  mvaddstr( 0, 19, " SATCOM - SATELLITE COMMUNICATIONS SYSTEM " );

  refresh(); /* Done with root window */

} /* End of Screen_Root() */

/*------------------------------------------------------------------------*/

/*  Screen_Main()
 *
 *  Screens the startup main screen of satcom
 */

  void
Screen_Main( void )
{
  WINDOW *main_win;
  main_win = newwin( 15, 78, 6, 1 );

  /* Box main screen window and add labels */
  wattrset( main_win, CYAN );
  box( main_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( main_win, 0, 25, " AVAILABLE 'MODES' IN SATCOM " );

  wattrset(  main_win, YELLOW );
  mvwaddstr( main_win,  1, 1, "<S>" );
  mvwaddstr( main_win,  2, 1, "<M>" );
  mvwaddstr( main_win,  3, 1, "<L>" );
  mvwaddstr( main_win,  4, 1, "<P>" );
  mvwaddstr( main_win,  5, 1, "<I>" );
  mvwaddstr( main_win,  6, 1, "<C>" );
  mvwaddstr( main_win,  7, 1, "<O>" );
  mvwaddstr( main_win,  8, 1, "<F>" );
  mvwaddstr( main_win,  9, 1, "<G>" );

  wattrset(  main_win, WHITE );
  mvwaddstr( main_win,  1, 4,
	  "Single Satellite/Observer mode. Control for G-5500 rotor & CAT for FT-847" );
  mvwaddstr( main_win,  2, 4,
	  "Multi-Satellite/Single-Observer mode. Tracks up to 30 satellites at once " );
  mvwaddstr( main_win,  3, 4,
	  "Multi-Observer/Single-Satellite mode. Tracks one satellite at 4 locations" );
  mvwaddstr( main_win,  4, 4,
	  "Pass Predictions mode. Text and Graphical predictions of satellite passes" );
  mvwaddstr( main_win,  5, 4,
	  "Solar Illumination Predictions mode. Calculates satellite % illumination " );
  mvwaddstr( main_win,  6, 4,
	  "Manual Commands mode. Manual Azim/Elev, parking rotors, tracking Moon/Sun" );
  mvwaddstr( main_win,  7, 4,
	  "Offset Calibration mode. Offset calibration of Yaesu G-5500 Az/El rotors " );
  mvwaddstr( main_win,  8, 4,
	  "Fullscale Calibration mode. Fullscale calibration of Yaesu G-5500 rotors " );
  mvwaddstr( main_win,  9, 4,
	  "Maidenhead Grid Locator mode. Calculates Grid Locator, bearing/distance  " );

  wattrset(  main_win, CYAN );
  mvwaddstr( main_win, 11, 4,
	  "         SATCOM-1.2       Satellite Communications Software             " );
  mvwaddstr( main_win, 12, 4,
	  " Released under the GPL, version 2 or higher, by Neoklis Kyriazis 5B4AZ  " );

  /* Done with main_win */
  wnoutrefresh( main_win );
  delwin( main_win );
  doupdate();

} /* End of Screen_Main() */

/*------------------------------------------------------------------------*/

/*  Screen_Observer_Status()
 *
 *  Function to display the observer's location,
 *  UTC time and solar/lunar azimuth/elevation
 */

  void
Screen_Observer_Status( observer_status_t *obs_status )
{
  /* Formatted time string */
  char ftime[12];

  WINDOW *observer_status_win;
  observer_status_win = newwin( 5, 78, 1, 1 );

  /* Box observer status window and add labels */
  wattrset( observer_status_win, CYAN );
  box( observer_status_win, ACS_VLINE, ACS_HLINE );
  Draw_Vline( observer_status_win, 0, 10, 4 );
  Draw_Vline( observer_status_win, 0, 22, 4 );
  Draw_Vline( observer_status_win, 0, 30, 4 );
  Draw_Vline( observer_status_win, 0, 41, 4 );
  Draw_Vline( observer_status_win, 0, 50, 4 );
  Draw_Vline( observer_status_win, 0, 59, 4 );
  Draw_Vline( observer_status_win, 0, 68, 4 );
  mvwaddstr(  observer_status_win, 0, 26, " OBSERVER LOCATION AND DATE " );

  wattrset(  observer_status_win, MAGENTA );
  mvwaddstr( observer_status_win, 1,  2, "Location" );
  mvwaddstr( observer_status_win, 1, 24, "Long E" );
  mvwaddstr( observer_status_win, 1, 44, "Latd N" );
  mvwaddstr( observer_status_win, 1, 60, "Height m" );
  mvwaddstr( observer_status_win, 2,  1, "Grid Lctr" );
  mvwaddstr( observer_status_win, 2, 23, "UT Date" );
  mvwaddstr( observer_status_win, 2, 42, "UTC Time" );
  mvwaddstr( observer_status_win, 2, 60, "Lcl Time" );
  mvwprintw( observer_status_win, 3,  1,"%7s", "Sun Az/El" );
  mvwprintw( observer_status_win, 3, 23,"%7s", "Moon Rm" );
  mvwprintw( observer_status_win, 3, 42,"%8s", "Moon Azm" );
  mvwprintw( observer_status_win, 3, 60,"%8s", "Moon Elv" );

  wattrset(  observer_status_win, GREEN );

  mvwprintw( observer_status_win, 1, 33,"%8.4f",
	  Degrees(obs_status->obs_geodetic.lon) );
  mvwprintw( observer_status_win, 1, 51,"%8.4f",
	  Degrees(obs_status->obs_geodetic.lat) );
  mvwprintw( observer_status_win, 1, 69,"%8d",
	  (int)(obs_status->obs_geodetic.hgt*1000. + .5) );
  mvwprintw( observer_status_win, 1, 11,"%11s", obs_status->loc_name );

  mvwprintw( observer_status_win, 2, 11,"QTH: %6s", obs_status->grid_loc );
  strftime( ftime, 12, "%d%b %Y", &obs_status->utc );
  mvwprintw( observer_status_win, 2, 31,"%s", ftime );
  strftime( ftime, 12, "%H:%M:%S", &obs_status->utc );
  mvwprintw( observer_status_win, 2, 51,"%s", ftime );
  strftime( ftime, 12, "%H:%M:%S", &obs_status->localtm );
  mvwprintw( observer_status_win, 2, 69,"%s", ftime );

  mvwprintw( observer_status_win, 3, 11,"%5.1f/",
	  Degrees(obs_status->solar_set.x) );

  if( obs_status->solar_set.y < 0. )
	wattrset(observer_status_win, RED );
  mvwprintw( observer_status_win, 3, 17,"%+5.1f",
	  Degrees(obs_status->solar_set.y) );

  wattrset(  observer_status_win, GREEN );
  mvwprintw( observer_status_win, 3, 31,"%10.2f", obs_status->lunar_set.z );
  mvwprintw( observer_status_win, 3, 52,"%7.3f",
	  obs_status->lunar_set.x );

  if( obs_status->lunar_set.y < 0. )
	wattrset(observer_status_win, RED );
  mvwprintw( observer_status_win, 3, 70,"%7.3f",
	  obs_status->lunar_set.y );

  /* Done with observer_status_win */
  wnoutrefresh( observer_status_win );
  delwin( observer_status_win );
  doupdate();

} /* End of Screen_Observer_Status() */

/*------------------------------------------------------------------------*/

/*  Screen_Multiobserver_Status()
 *
 *  Screens multi-location observer status
 */

void
Screen_Multiobserver_Status(
	int num_loc,
	observer_status_t  *multiobs_status,
	satellite_status_t *mxsat_status )
{
  int
	row, /* Row counter for printing */
	idx; /* Index for loops etc */

  /* Formatted time and date strings */
  char
	fdat[12],
	futc[9],
	ftime[13];

  double
	julian_utc,   /* Julian UTC */
	home_aos,
	rem_aos,
	overlap_from, /* AOS/LOS overlap from time */
	overlap_to;   /* AOS/LOS overlap to time   */

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  WINDOW *multiobs_status_win;
  multiobs_status_win = newwin( 20, 78, 1, 1 );

  strftime( fdat, 12, "%d/%b/%Y", &multiobs_status[0].utc );
  strftime( futc,  9, "%H:%M:%S", &multiobs_status[0].utc );

  /* Display multi-observer status */
  for( idx = 0; idx <= num_loc; idx++ )
  {
	row = idx * 5;
	wattrset(  multiobs_status_win, MAGENTA );
	mvwaddstr( multiobs_status_win, row,  1, "Location:" );
	mvwaddstr( multiobs_status_win, row, 22, "Grid Loc:" );
	mvwaddstr( multiobs_status_win, row, 39, "Lon:" );
	mvwaddstr( multiobs_status_win, row, 54, "Lat:" );
	mvwaddstr( multiobs_status_win, row, 69, "Hgt:" );

	mvwaddstr( multiobs_status_win, row+1,  1, "UTC Date:" );
	mvwaddstr( multiobs_status_win, row+1, 33, "Sun Az/El:" );
	mvwaddstr( multiobs_status_win, row+1, 56, "Moon Az/El:" );

	mvwaddstr( multiobs_status_win, row+2,  0, "Satellite:" );
	mvwaddstr( multiobs_status_win, row+2, 22, "Azim:" );
	mvwaddstr( multiobs_status_win, row+2, 33, "Elev:" );
	mvwaddstr( multiobs_status_win, row+2, 44, "Range:" );
	mvwaddstr( multiobs_status_win, row+2, 56, "Doppler Hz/Mhz:" );

	mvwaddstr( multiobs_status_win, row+3,  0, "Next AOS:" );
	mvwaddstr( multiobs_status_win, row+3, 22, "Next LOS:" );
	if( idx == 0 )
	{
	  mvwaddstr( multiobs_status_win, row+3, 44, "Lon:" );
	  mvwaddstr( multiobs_status_win, row+3, 55, "Lat:" );
	  mvwaddstr( multiobs_status_win, row+3, 66, "Alt:" );
	}
	else
	{
	  mvwaddstr( multiobs_status_win, row+3, 44, "Ovrlap:" );
	  mvwaddch(  multiobs_status_win, row+3, 64, '-' );
	}

	wattrset( multiobs_status_win, GREEN );
	mvwprintw( multiobs_status_win, row, 10, "%11s",
		multiobs_status[idx].loc_name );

	if( strcmp(multiobs_status[idx].grid_loc, "ERROR!") == 0 )
	  wattrset( multiobs_status_win, RED );
	mvwprintw( multiobs_status_win, row, 31, "%6s",
		multiobs_status[idx].grid_loc );

	wattrset( multiobs_status_win, GREEN );
	mvwprintw( multiobs_status_win, row, 43, "%9.5f",
		Degrees(multiobs_status[idx].obs_geodetic.lon) );
	mvwprintw( multiobs_status_win, row, 58, "%9.5f",
		Degrees(multiobs_status[idx].obs_geodetic.lat) );
	mvwprintw( multiobs_status_win, row, 73, "%5d",
		(int)(multiobs_status[idx].obs_geodetic.hgt*1000. + .5) );

	mvwprintw( multiobs_status_win, row+1, 10,"%s", fdat );
	mvwprintw( multiobs_status_win, row+1, 22,"%s", futc );
	mvwprintw( multiobs_status_win, row+1, 43,"%5.1f/%5.1f",
		Degrees(multiobs_status[idx].solar_set.x),
		Degrees(multiobs_status[idx].solar_set.y) );
	mvwprintw( multiobs_status_win, row+1, 67,"%5.1f/%5.1f",
		multiobs_status[idx].lunar_set.x,
		multiobs_status[idx].lunar_set.y );

	/* Display satellite status */
	mvwprintw( multiobs_status_win, row+2, 10,"%11s",
		mxsat_status[idx].name );
	mvwprintw( multiobs_status_win, row+2, 27,"%5.1f",
		mxsat_status[idx].azim );
	if( mxsat_status[idx].elev < 0. )
	  wattrset( multiobs_status_win, RED );
	mvwprintw( multiobs_status_win, row+2, 38,"%5.1f",
		mxsat_status[idx].elev );
	mvwprintw( multiobs_status_win, row+2, 71,"%7.3f",
		mxsat_status[idx].doppler );
	wattrset( multiobs_status_win, GREEN );
	mvwprintw( multiobs_status_win, row+2, 50,"%5d",
		(int)mxsat_status[idx].range );

	/* Display AOS/LOS according to relevant flags */
	if( isSatFlagSet(&mxsat_status[idx], SAT_GEOSTATIONARY_FLAG) )
	{
	  wattrset(  multiobs_status_win, YELLOW );
	  mvwprintw( multiobs_status_win, row+3,  9,"%s", "  Geostatic " );
	  mvwprintw( multiobs_status_win, row+3, 31,"%s", "  Geostatic " );
	}
	else
	  if( isSatFlagSet(&mxsat_status[idx], NO_AOS_LOS_FLAG) )
	  {
		wattrset(  multiobs_status_win, RED );
		mvwprintw( multiobs_status_win, row+3,  9,"%s", " No AOS/LOS" );
		mvwprintw( multiobs_status_win, row+3, 31,"%s", " No AOS/LOS" );
	  }
	  else
	  {
		wattrset(  multiobs_status_win, GREEN );
		Date_Time( mxsat_status[idx].aos, &date_time );
		strftime( ftime, 12, "%d/%b", &date_time );
		mvwprintw( multiobs_status_win, row+3,  9,"%s", ftime );
		strftime( ftime, 12, "%H:%M", &date_time );
		mvwprintw( multiobs_status_win, row+3, 16,"%s", ftime );
		Date_Time( mxsat_status[idx].los, &date_time );
		strftime( ftime, 12, "%d/%b", &date_time );
		mvwprintw( multiobs_status_win, row+3, 31,"%s", ftime );
		strftime( ftime, 12, "%H:%M", &date_time );
		mvwprintw( multiobs_status_win, row+3, 38,"%s", ftime );
	  }

	wattrset(  multiobs_status_win, GREEN );

	/* For first ('Home') observer, display satellite lat,lon,alt */
	/* For additional observers, display visibility overlap times */

	if( idx == 0 )
	{
	  mvwprintw( multiobs_status_win, row+3, 48, "%6.2f",
		  mxsat_status[0].ssplon );
	  mvwprintw( multiobs_status_win, row+3, 59, "%6.2f",
		  mxsat_status[0].ssplat );
	  mvwprintw( multiobs_status_win, row+3, 70, "%8.1f",
		  mxsat_status[0].alt );
	}
	else
	{
	  if( isSatFlagClear(&mxsat_status[idx], NO_AOS_LOS_FLAG) )
	  {
		julian_utc = Julian_Date( &multiobs_status[0].utc );
		/* If aos>los, then sat is visible and aos is set to utc */
		if( mxsat_status[0].aos >  mxsat_status[0].los )
		  home_aos = julian_utc;
		else
		  home_aos = mxsat_status[0].aos;

		if( mxsat_status[idx].aos >  mxsat_status[idx].los )
		  rem_aos = julian_utc;
		else
		  rem_aos = mxsat_status[idx].aos;

		/* Furthest aos and nearest los are    */
		/* the start/end of visibility overlap */
		if( home_aos >  rem_aos )
		  overlap_from = home_aos;
		else
		  overlap_from = rem_aos;

		if( mxsat_status[0].los < mxsat_status[idx].los )
		  overlap_to = mxsat_status[0].los;
		else
		  overlap_to = mxsat_status[idx].los;

		/* If start of overlap time nearer than end */
		/* then there is overlap, so display times  */
		if( overlap_from < overlap_to )
		{
		  wattrset( multiobs_status_win, YELLOW );
		  Date_Time( overlap_from, &date_time );
		  strftime( ftime, 13, "%d/%b %H:%M", &date_time );
		  mvwaddstr( multiobs_status_win, row+3, 51, ftime );
		  Date_Time( overlap_to, &date_time );
		  strftime( ftime, 13, "%d/%b %H:%M", &date_time );
		  mvwaddstr( multiobs_status_win, row+3, 66, ftime );
		}
	  }
	  else
	  {
		wattrset( multiobs_status_win, RED );
		mvwaddstr( multiobs_status_win, row+3, 51,
			"  - - - - - - - - - - - -  " );
	  }

	} /* End of if( idx == 0 ) */

	wattrset( multiobs_status_win, CYAN );
	mvwhline( multiobs_status_win, row+4, 0, ACS_HLINE, 78 );

  } /* End of for( idx = 0; idx <= num_loc; idx++ ) */

  /* Done with observer_status_win */
  wnoutrefresh( multiobs_status_win );
  delwin( multiobs_status_win );
  doupdate();

} /* End of Screen_Multiobserver_Status() */

/*------------------------------------------------------------------------*/

/*  Screen_Satellite_Status()
 *
 *  Function to display status data of a satellite
 */

  void
Screen_Satellite_Status( satellite_status_t  *sat_status )
{
  /* Formatted time string */
  char ftime[12];

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Satellite status window */
  WINDOW *satellite_status_win;
  satellite_status_win = newwin( 7, 78, 6, 1 );

  /* The following lines build satellite_status_win */
  /* and post the selected satellite's status data  */
  wattrset(  satellite_status_win, GREEN );
  mvwprintw( satellite_status_win, 1, 16, "%11s",  sat_status->name );
  mvwprintw( satellite_status_win, 1, 46, "%6.2f", sat_status->azim );
  mvwprintw( satellite_status_win, 1, 71, "%6.2f", sat_status->ssplon );
  mvwprintw( satellite_status_win, 2, 71, "%6.2f", sat_status->ssplat );
  mvwprintw( satellite_status_win, 3, 45, "%7d", (int)sat_status->range );
  mvwprintw( satellite_status_win, 3, 70, "%7d", (int)sat_status->alt );
  mvwprintw( satellite_status_win, 4, 70, "%7d", (int)sat_status->ftprnt);
  mvwprintw( satellite_status_win, 5, 70, "%7d",      sat_status->revnum );

  if( isSatFlagSet(sat_status, DEEP_SPACE_FLAG) )
	mvwaddstr( satellite_status_win, 2, 16, "NORAD: SDP4" );
  else
	mvwaddstr( satellite_status_win, 2, 16, "NORAD: SGP4" );

  mvwprintw( satellite_status_win, 4, 46, "%6.2f", sat_status->phs256 );

  if( sat_status->elev < 0. )
  {
	wattrset(  satellite_status_win, RED );
	mvwprintw( satellite_status_win, 2, 46, "%6.2f", sat_status->elev );
	mvwprintw( satellite_status_win, 5, 45, "%7.2f", sat_status->doppler );
  }
  else
  {
	wattrset(satellite_status_win, YELLOW );
	mvwprintw( satellite_status_win, 2, 46, "%6.2f", sat_status->elev );
	wattrset(satellite_status_win, GREEN );
	mvwprintw( satellite_status_win, 5, 45, "%7.2f", sat_status->doppler );
  }

  if( isSatFlagSet(sat_status, SAT_ECLIPSED_FLAG) )
  {
	wattrset(  satellite_status_win, RED );
	mvwaddstr( satellite_status_win, 5, 16, "   Eclipsed" );
  }
  else
  {
	wattrset(  satellite_status_win, GREEN );
	mvwaddstr( satellite_status_win, 5, 16, "In Sunlight" );
  }

  /* Check and implement AOS/LOS flags */
  if( isSatFlagSet(sat_status, NO_AOS_LOS_FLAG) )
  {
	wattrset( satellite_status_win, RED );
	if( isSatFlagSet(sat_status, SAT_INACCESSIBLE_FLAG) )
	{
	  mvwprintw( satellite_status_win, 3, 16,"%s", " No AOS/LOS" );
	  mvwprintw( satellite_status_win, 4, 16,"%s", " No AOS/LOS" );
	}
	else
	  if( isSatFlagSet(sat_status, SAT_DECAYED_FLAG) )
	  {
		mvwaddstr( satellite_status_win, 3, 16, "!DECAYED ??" );
		mvwaddstr( satellite_status_win, 4, 16, "!DECAYED ??" );
	  }
	  else
		if( isSatFlagSet(sat_status, SAT_GEOSTATIONARY_FLAG) )
		{
		  wattrset( satellite_status_win, YELLOW );
		  mvwaddstr( satellite_status_win, 3, 16, " Geostatic " );
		  mvwaddstr( satellite_status_win, 4, 16, " Geostatic " );
		}
  }
  else
  {
	wattrset( satellite_status_win, GREEN );
	Date_Time( sat_status->aos, &date_time );
	strftime( ftime, 12, "%d%b %H:%M", &date_time );
	mvwprintw( satellite_status_win, 3, 16,"%s", ftime );
	Date_Time( sat_status->los, &date_time );
	strftime( ftime, 12, "%d%b %H:%M", &date_time );
	mvwprintw( satellite_status_win, 4, 16,"%s", ftime );

  } /* End of if( isSatFlagSet(sat_status, NO_AOS_LOS_FLAG) ) */

  /* Draw some lines to box the results and add labels */
  wattrset(  satellite_status_win, MAGENTA );
  mvwaddstr( satellite_status_win, 1,  1, "Satellite Name" );
  mvwaddstr( satellite_status_win, 2,  1, "Tracking Model" );
  mvwaddstr( satellite_status_win, 3,  1, "Next AOS - UTC" );
  mvwaddstr( satellite_status_win, 4,  1, "Next LOS - UTC" );
  mvwaddstr( satellite_status_win, 5,  1, "Eclipse Status" );
  mvwaddstr( satellite_status_win, 1, 29, "    Azimuth deg" );
  mvwaddstr( satellite_status_win, 2, 29, "  Elevation deg" );
  mvwaddstr( satellite_status_win, 3, 29, " Slant Range km" );
  mvwaddstr( satellite_status_win, 4, 29, "Phase units/256" );
  mvwaddstr( satellite_status_win, 5, 29, " Doppler Hz/MHz" );
  mvwaddstr( satellite_status_win, 1, 54, "Longitude E deg" );
  mvwaddstr( satellite_status_win, 2, 54, " Latitude N deg" );
  mvwaddstr( satellite_status_win, 3, 54, "    Altitude km" );
  mvwaddstr( satellite_status_win, 4, 54, "   Footprint km" );
  mvwaddstr( satellite_status_win, 5, 54, " Revolution Num" );

  wattrset(  satellite_status_win, CYAN );

  box(       satellite_status_win, ACS_VLINE, ACS_HLINE);
  Draw_Vline(satellite_status_win, 0, 15, 6 );
  Draw_Vline(satellite_status_win, 0, 27, 6 );
  mvwaddstr( satellite_status_win, 0,  3, " SATELLITE " );
  mvwaddstr( satellite_status_win, 0, 32, " DIRECTION ");
  mvwaddstr( satellite_status_win, 0, 57, " POSITION " );

  Draw_Vline(satellite_status_win, 0, 28, 6 );
  Draw_Vline(satellite_status_win, 0, 44, 6 );
  Draw_Vline(satellite_status_win, 0, 52, 6 );
  Draw_Vline(satellite_status_win, 0, 53, 6 );
  Draw_Vline(satellite_status_win, 0, 69, 6 );

  /* If new satellite requested, open name entry form */
  if( isSatFlagSet(sat_status, REQ_KEYBD_ENTRY_FLAG) )
  {
	Clear_SatFlag( sat_status, REQ_KEYBD_ENTRY_FLAG );
	Post_Name_Form( sat_status->name, satellite_status_win );
  }

  /* Done with satellite_status_win */
  wnoutrefresh( satellite_status_win );
  delwin( satellite_status_win );

  doupdate();

} /* End of Screen_Satellite_Status() */

/*------------------------------------------------------------------------*/

/*  Screen_Multisatellite_Status()
 *
 *  Displays muti-satellite orbital data and selects
 *  an individual satellite for single-tracking
 */

void
Screen_Multisatellite_Status(
	int *stats,
	observer_status_t *obs_status,
	satellite_status_t *sat_status,
	satellite_status_t *multisat_status )
{
  int
	sel_hlght,/* Satellite selected to be highlighted */
	frm_dspl, /* Satellite rank to start display from */
	to_dspl,  /* Satellite rank to end display at     */
	sat_rank, /* Satellite's rank in the display   */
	row,      /* Display row count    */
	col,      /* Display column count */
	idx;      /* Index for loops etc  */

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Formatted time string */
  char ftime[21];

  WINDOW *multitrack_win;

  multitrack_win = newwin( 20, 78, 1, 1 );

  /* Add labels and lines */
  wattrset(  multitrack_win, CYAN );
  mvwhline(  multitrack_win,  2,  0, ACS_HLINE, 78 );
  mvwhline(  multitrack_win, 19,  0, ACS_HLINE, 78 );
  Draw_Vline(multitrack_win,  2, 38, 19);
  Draw_Vline(multitrack_win,  2, 39, 19);
  mvwaddstr( multitrack_win,  2,  3, " MULTI-SATELLITE TRACKING STATUS " );
  mvwaddstr( multitrack_win,  2, 42, " SORTED BY VISIBILITY AND AOS/LOS " );

  wattrset(  multitrack_win, MAGENTA );
  mvwaddstr( multitrack_win, 0,  0, " Location:" );
  mvwaddstr( multitrack_win, 0, 22, "Grid Lc:" );
  mvwaddstr( multitrack_win, 0, 37, "Lc Time:" );
  mvwaddstr( multitrack_win, 0, 54, "Sun:" );
  mvwaddstr( multitrack_win, 0, 66, "Moon:" );

  mvwaddstr( multitrack_win, 1,  0, "Satellite:" );
  mvwaddstr( multitrack_win, 1, 22, "Azm:" );
  mvwaddstr( multitrack_win, 1, 32, "Elv:" );
  mvwaddstr( multitrack_win, 1, 42, "Range:" );
  mvwaddstr( multitrack_win, 1, 54, "UTC:" );

  /* Display data */
  wattrset(  multitrack_win, GREEN );
  mvwprintw( multitrack_win, 0, 10, "%11s", obs_status->loc_name );
  mvwprintw( multitrack_win, 0, 30, "%6s", obs_status->grid_loc );
  strftime( ftime, 12, "%H:%M:%S", &obs_status->localtm );
  mvwprintw( multitrack_win, 0, 45, "%8s", ftime );

  mvwprintw( multitrack_win, 0, 58, "%3d/",
	  (int)(Degrees(obs_status->solar_set.x)+0.5) );

  if( obs_status->solar_set.y < 0. )
	wattrset(  multitrack_win, RED );
  mvwprintw( multitrack_win, 0, 62, "%+3d",
	  (int)(Degrees(obs_status->solar_set.y)+0.5) );

  wattrset(  multitrack_win, GREEN );
  mvwprintw( multitrack_win, 0, 71, "%3d/",
	  (int)obs_status->lunar_set.x );

  if( obs_status->lunar_set.y < 0. )
	wattrset(  multitrack_win, RED );
  mvwprintw( multitrack_win, 0, 75, "%+3d",
	  (int)(obs_status->lunar_set.y+0.5) );

  wattrset(  multitrack_win, GREEN );
  mvwprintw( multitrack_win, 1, 10, "%11s",  sat_status->name );
  mvwprintw( multitrack_win, 1, 26, "%4.1f", sat_status->azim );

  if( sat_status->elev < 0. )
	wattrset(  multitrack_win, RED );
  mvwprintw( multitrack_win, 1, 36, "%+4.1f", sat_status->elev );

  wattrset(  multitrack_win, GREEN );
  mvwprintw( multitrack_win, 1, 48, "%5d",
	  (int)(sat_status->range+0.5) );

  strftime( ftime, 21, "%d/%b/%Y %H:%M:%S", &obs_status->utc );
  mvwprintw( multitrack_win, 1, 58, "%20s", ftime );

  wattrset(  multitrack_win, MAGENTA );
  mvwaddstr( multitrack_win, 3,  0, "  Satellite Azm Elv Range Next AOS|LOS" );
  mvwaddstr( multitrack_win, 3, 40, "  Satellite Azm Elv Range Next AOS|LOS" );

  /* Derive display range and highlight-select from 'stats' */
  sel_hlght = (*stats >> 20) & 0X3FF;
  frm_dspl  = (*stats >> 10) & 0X3FF;
  to_dspl   =  *stats & 0X3FF;

  /* Display multi-satellite status */
  for( idx = 0; idx < num_tles; idx++ )
  {
	sat_rank = (multisat_status[idx].flags & SAT_RANK_FLAGS);

	if( (sat_rank >= frm_dspl) && (sat_rank <= to_dspl) )
	{
	  row = sat_rank + 4 - frm_dspl;

	  if( sel_hlght == (row-4) )
	  {
		*stats = idx << 20;
		wattrset( multitrack_win, (multisat_status[idx].elev > 0. ?
			  KEYSTROKE : COMMAND) );
	  }
	  else
		wattrset( multitrack_win,
			( multisat_status[idx].elev > 0. ? YELLOW : GREEN ) );

	  if( row >= 19 )
	  {
		row -= 15;
		col = 40;
	  }
	  else
		col = 0;

	  mvwprintw( multitrack_win, row, col, "%11s ",
		  multisat_status[idx].name );
	  mvwprintw( multitrack_win, row, col+12, "%3d ",
		  (int)(multisat_status[idx].azim+0.5) );
	  mvwprintw( multitrack_win, row, col+16, "%+3d ",
		  (int)(multisat_status[idx].elev+0.5) );
	  mvwprintw( multitrack_win, row, col+20, "%5d ",
		  (int)(multisat_status[idx].range+0.5) );

	  /* Don't display AOS/LOS if sat is inaccessible or geostationary */
	  if( isSatFlagSet(&multisat_status[idx], SAT_GEOSTATIONARY_FLAG) )
	  {
		wattrset(  multitrack_win, GREEN );
		mvwaddstr( multitrack_win, row, col+26, "  Geostatic " );
	  }
	  else
		if( isSatFlagSet(&multisat_status[idx], NO_AOS_LOS_FLAG) )
		{
		  wattrset(  multitrack_win, RED );
		  mvwaddstr( multitrack_win, row, col+26, " No AOS/LOS " );
		}
		else
		{
		  if( multisat_status[idx].elev > 0. )
			Date_Time( multisat_status[idx].los, &date_time );
		  else
			Date_Time( multisat_status[idx].aos, &date_time );

		  strftime( ftime, 21, "%d/%b %H:%M", &date_time );
		  mvwprintw( multitrack_win, row, col+26, "%12s", ftime );

		} /*if(isSatFlagSet(&multisat_status[idx],SAT_GEOSTATIONARY_FLAG))*/

	} /* End of if( (multisat_status[idx].flags & 0xFF) < 30 ) */

  } /* End of for( idx = 0; idx < num_tles; idx++ ) */

  wnoutrefresh( multitrack_win );
  doupdate();
  delwin( multitrack_win );

} /* End of Screen_Multisatellite_Status() */

/*------------------------------------------------------------------------*/

/*  Screen_Pass_Predictions()
 *
 *  Displays satellite pass predictions
 */

void
Screen_Pass_Predictions(
	int num,
	pass_data_t *pass_data,
	observer_status_t *obs_status,
	satellite_status_t *sat_status )
{
  int
	col, /* Screen column */
	idx; /* Index for loops etc */

  /* Max and min values of azim/elev needed in plots */
  double
	max_elev,
	min_elev,
	max_azim,
	min_azim;

  /* Range of azim/elev data (max-min) */
  double
	rng_elev,
	rng_azim;


  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Formatted time string */
  char ftime[21];

  WINDOW *pass_data_win;

  pass_data_win = newwin( 20, 78, 1, 1 );

  /* Add labels and lines */
  wattrset(  pass_data_win, CYAN );
  mvwhline(  pass_data_win,  2,  0, ACS_HLINE, 78 );
  mvwhline(  pass_data_win, 19,  0, ACS_HLINE, 78 );

  /* Don't display AOS/LOS if sat is inaccessible or geostationary */
  if( isSatFlagSet(sat_status, SAT_GEOSTATIONARY_FLAG) )
  {
	wattrset(  pass_data_win, YELLOW );
	mvwaddstr( pass_data_win,  2, 5,
		" THIS SATELLITE IS GEOSTATIONARY. AOS/LOS PREDICTIONS DO NOT APPLY " );
  }
  else
	if( isSatFlagSet(sat_status, NO_AOS_LOS_FLAG) )
	{
	  wattrset(  pass_data_win, RED );
	  mvwaddstr( pass_data_win,  2, 4,
		  " THIS SATELLITE APPEARS TO BE INACCESSIBLE FROM THE CURRENT LOCATION " );
	}
	else
	{
	  wattrset(  pass_data_win, CYAN | A_BOLD );

	  if( isFlagSet(VISIBLE_PREDICT_FLAG) )
		mvwaddstr( pass_data_win,  2,  19, " VISIBLE PASS PREDICTIONS " );
	  else
		mvwaddstr( pass_data_win,  2,  16, " ACCESSIBLE PASS PREDICTIONS " );

	  if( isFlagSet(USERTIME_PREDICT_FLAG) )
		mvwaddstr( pass_data_win,  2,  44, " IN USER-DEFINED TIME " );
	  else
		mvwaddstr( pass_data_win,  2,  44, " IN REAL TIME " );

	} /* if( isSatFlagSet(&multisat_status[idx], NO_AOS_LOS_FLAG) ) */

  wattrset(  pass_data_win, MAGENTA );
  mvwaddstr( pass_data_win, 0,  0, " Location:" );
  mvwaddstr( pass_data_win, 0, 22, "Grid Lc:" );
  mvwaddstr( pass_data_win, 0, 37, "Lc Time:" );
  mvwaddstr( pass_data_win, 0, 54, "Sun:" );
  mvwaddstr( pass_data_win, 0, 66, "Moon:" );

  mvwaddstr( pass_data_win, 1,  0, "Satellite:" );
  mvwaddstr( pass_data_win, 1, 22, "Azm:" );
  mvwaddstr( pass_data_win, 1, 32, "Elv:" );
  mvwaddstr( pass_data_win, 1, 42, "Range:" );
  mvwaddstr( pass_data_win, 1, 54, "UTC:" );

  /* Display observer data */
  wattrset(  pass_data_win, GREEN );
  mvwprintw( pass_data_win, 0, 10, "%11s", obs_status->loc_name );
  mvwprintw( pass_data_win, 0, 30,  "%6s", obs_status->grid_loc );
  strftime( ftime, 12, "%H:%M:%S", &obs_status->localtm );
  mvwprintw( pass_data_win, 0, 45, "%8s", ftime );

  mvwprintw( pass_data_win, 0, 58, "%3d/",
	  (int)(Degrees(obs_status->solar_set.x)+0.5) );

  if( obs_status->solar_set.y < 0. )
	wattrset(  pass_data_win, RED );
  mvwprintw( pass_data_win, 0, 62, "%+3d",
	  (int)(Degrees(obs_status->solar_set.y)+0.5) );

  wattrset(  pass_data_win, GREEN );
  mvwprintw( pass_data_win, 0, 71, "%3d/",
	  (int)obs_status->lunar_set.x );

  if( obs_status->lunar_set.y < 0. )
	wattrset(  pass_data_win, RED );
  mvwprintw( pass_data_win, 0, 75, "%+3d",
	  (int)(obs_status->lunar_set.y+0.5) );

  wattrset(  pass_data_win, GREEN );
  mvwprintw( pass_data_win, 1, 10, "%11s",  sat_status->name );
  mvwprintw( pass_data_win, 1, 26, "%4.1f", sat_status->azim );

  if( sat_status->elev < 0. )
	wattrset(  pass_data_win, RED );
  mvwprintw( pass_data_win, 1, 36, "%+4.1f", sat_status->elev );

  wattrset(  pass_data_win, GREEN );
  mvwprintw( pass_data_win, 1, 48, "%5d",
	  (int)(sat_status->range+0.5) );

  strftime( ftime, 21, "%d/%b/%Y %H:%M:%S", &obs_status->utc );
  mvwprintw( pass_data_win, 1, 58, "%20s", ftime );

  /* If there is AOS/LOS data available */
  if( num != 0 )
  {
	wattrset(  pass_data_win, MAGENTA );
	mvwaddstr( pass_data_win, 3,  0,
		" Date  UTC         Elevation Plot            Azimuth Plot    Phase  Long  Latd" );

	/* Find max and min values of azim/elev */
	max_elev = min_elev = pass_data->elev[0];
	max_azim = min_azim = pass_data->azim[0];

	for( idx = 1; idx < num; idx++ )
	{
	  if( pass_data->elev[idx] > max_elev )
		max_elev = pass_data->elev[idx];
	  if( pass_data->elev[idx] < min_elev )
		min_elev = pass_data->elev[idx];

	  if( pass_data->azim[idx] > max_azim )
		max_azim = pass_data->azim[idx];
	  if( pass_data->azim[idx] < min_azim )
		min_azim = pass_data->azim[idx];
	}

	/* Calculate azim/elev range of values */
	rng_elev = max_elev - min_elev;
	rng_azim = max_azim - min_azim;

	for( idx = 0; idx < num; idx++ )
	{
	  /* Display date/time */
	  wattrset( pass_data_win, GREEN );
	  Date_Time( pass_data->julian_utc[idx], &date_time );
	  strftime( ftime, 12, "%d%b %H:%M", &date_time );
	  mvwprintw( pass_data_win, idx+4, 0, "%11s", ftime );

	  /* Plot elevation values */
	  wattrset( pass_data_win, YELLOW );
	  mvwprintw( pass_data_win, idx+4, 11, "%5.1f", pass_data->elev[idx] );
	  col = 17 + (int)( (pass_data->elev[idx]-min_elev)*17./rng_elev+0.5 );

	  /* Plot elevation according to visibility */
	  if( isFlagSet(VISIBLE_PREDICT_FLAG) )
	  {
		if( (pass_data->flags[idx] & SAT_VISIBLE_FLAG) )
		{
		  wattrset( pass_data_win, CYAN|A_BOLD );
		  mvwaddch( pass_data_win, idx+4, col, '*' );
		}
		else
		{
		  wattrset( pass_data_win, BLUE|A_BOLD );
		  mvwaddch( pass_data_win, idx+4, col, '.' );
		}
	  } /* End of if( isFlagSet(VISUAL_PREDICT_FLAG) ) */
	  else
	  {
		wattrset( pass_data_win, YELLOW );
		if( (pass_data->flags[idx] & SAT_ECLIPSED_FLAG) )
		  mvwaddch( pass_data_win, idx+4, col, '.' );
		else
		  mvwaddch( pass_data_win, idx+4, col, '*' );
	  }


	  /* Plot azimuth values */
	  wattrset( pass_data_win, BLUE|A_BOLD );
	  mvwprintw( pass_data_win, idx+4, 36, "%5.1f", pass_data->azim[idx] );
	  col = 42 + (int)( (pass_data->azim[idx]-min_azim)*17./rng_azim+0.5 );
	  mvwaddch( pass_data_win, idx+4, col, '+' );

	  wattrset( pass_data_win, GREEN );
	  mvwprintw( pass_data_win, idx+4, 61, "%5.1f", pass_data->phs256[idx] );
	  mvwprintw( pass_data_win, idx+4, 67, "%5.1f", pass_data->lon[idx] );
	  mvwprintw( pass_data_win, idx+4, 73, "%5.1f", pass_data->lat[idx] );

	} /* End of for( idx = 0; idx < num; idx++ ) */

  } /* End of if( num != 0) */

  wnoutrefresh( pass_data_win );
  doupdate();
  delwin( pass_data_win );

} /* End of Screen_Pass_Predictions() */


/*------------------------------------------------------------------------*/

/*  Screen_Illumination_Predictions()
 *
 *  Displays satellite pass predictions
 */

void
Screen_Illumination_Predictions(
	double *jdate,
	int *illum,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  int idx; /* Index for loops etc */

  /* Conversion between julian and struct tm time */
  struct tm date_time;

  /* Formatted time string */
  char ftime[21];

  WINDOW *illum_data_win;

  illum_data_win = newwin( 20, 78, 1, 1 );

  /* Add labels and lines */
  wattrset(  illum_data_win, CYAN );
  mvwhline(  illum_data_win,  2,  0, ACS_HLINE, 78 );
  mvwhline(  illum_data_win, 19,  0, ACS_HLINE, 78 );
  mvwvline(  illum_data_win,  3, 39, ACS_VLINE, 16 );
  mvwaddch(  illum_data_win, 19, 39, ACS_BTEE );

  mvwaddstr( illum_data_win,  2,  19, " SATELLITE ILLUMINATION PREDICTIONS " );

  wattrset(  illum_data_win, MAGENTA );
  mvwaddstr( illum_data_win, 0,  0, " Location:" );
  mvwaddstr( illum_data_win, 0, 22, "Grid Lc:" );
  mvwaddstr( illum_data_win, 0, 37, "Lc Time:" );
  mvwaddstr( illum_data_win, 0, 54, "Sun:" );
  mvwaddstr( illum_data_win, 0, 66, "Moon:" );

  mvwaddstr( illum_data_win, 1,  0, "Satellite:" );
  mvwaddstr( illum_data_win, 1, 22, "Azm:" );
  mvwaddstr( illum_data_win, 1, 32, "Elv:" );
  mvwaddstr( illum_data_win, 1, 42, "Range:" );
  mvwaddstr( illum_data_win, 1, 54, "UTC:" );

  /* Display observer data */
  wattrset(  illum_data_win, GREEN );
  mvwprintw( illum_data_win, 0, 10, "%11s", obs_status->loc_name );
  mvwprintw( illum_data_win, 0, 30,  "%6s", obs_status->grid_loc );
  strftime( ftime, 12, "%H:%M:%S", &obs_status->localtm );
  mvwprintw( illum_data_win, 0, 45, "%8s", ftime );

  mvwprintw( illum_data_win, 0, 58, "%3d/",
	  (int)(Degrees(obs_status->solar_set.x)+0.5) );

  if( obs_status->solar_set.y < 0. )
	wattrset(  illum_data_win, RED );
  mvwprintw( illum_data_win, 0, 62, "%+3d",
	  (int)(Degrees(obs_status->solar_set.y)+0.5) );

  wattrset(  illum_data_win, GREEN );
  mvwprintw( illum_data_win, 0, 71, "%3d/",
	  (int)obs_status->lunar_set.x );

  if( obs_status->lunar_set.y < 0. )
	wattrset(  illum_data_win, RED );
  mvwprintw( illum_data_win, 0, 75, "%+3d",
	  (int)(obs_status->lunar_set.y+0.5) );

  wattrset(  illum_data_win, GREEN );
  mvwprintw( illum_data_win, 1, 10, "%11s",  sat_status->name );
  mvwprintw( illum_data_win, 1, 26, "%4.1f", sat_status->azim );

  if( sat_status->elev < 0. )
	wattrset(  illum_data_win, RED );
  mvwprintw( illum_data_win, 1, 36, "%+4.1f", sat_status->elev );

  wattrset(  illum_data_win, GREEN );
  mvwprintw( illum_data_win, 1, 48, "%5d",
	  (int)(sat_status->range+0.5) );

  strftime( ftime, 21, "%d/%b/%Y %H:%M:%S", &obs_status->utc );
  mvwprintw( illum_data_win, 1, 58, "%20s", ftime );

  /* Display illumination data */
  wattrset(  illum_data_win, MAGENTA );
  mvwaddstr( illum_data_win, 3,  0,
	  "      Date         Min/day    % Illum" );
  mvwaddstr( illum_data_win, 3,  41,
	  "      Date         Min/day    % Illum" );

  wattrset( illum_data_win, GREEN );
  for( idx = 0; idx < 15; idx++ )
  {
	Date_Time( jdate[idx], &date_time );
	strftime( ftime, 16, "%a %d/%b/%Y", &date_time );
	mvwprintw( illum_data_win, idx+4, 0, "%16s", ftime );
	mvwprintw( illum_data_win, idx+4, 21, "%5d", illum[idx] );
	mvwprintw( illum_data_win, idx+4, 31, "%6.2f",
		(double)illum[idx]/14.4 );

	Date_Time( jdate[idx+15], &date_time );
	strftime( ftime, 16, "%a %d/%b/%Y", &date_time );
	mvwprintw( illum_data_win, idx+4, 41, "%16s", ftime );
	mvwprintw( illum_data_win, idx+4, 62, "%5d", illum[idx+15] );
	mvwprintw( illum_data_win, idx+4, 72, "%6.2f",
		(double)illum[idx+15]/14.4 );

  } /* End of for( idx = 0; idx < 14; idx++ ) */

  wnoutrefresh( illum_data_win );
  doupdate();
  delwin( illum_data_win );

} /* End of Screen_Illumination_Predictions() */

/*------------------------------------------------------------------------*/

/*  Screen_Progress_Bar()
 *
 *  Screens a progress bar used with illumination predictions
 */

  void
Screen_Progress_Bar( int prog )
{
  int idx; /* Index for loops */

  WINDOW *progress_win;

  progress_win = newwin( 1, 78, 21, 1 );

  wattrset( progress_win, COMMAND );
  mvwaddstr(progress_win, 0, 0,
	  "Calculating solar illumination prediction ... [                              ]");

  wattrset( progress_win, KEYSTROKE );
  for( idx = 0; idx < prog; idx++ )
	mvwaddch(progress_win, 0, idx+47, '=' );
  mvwaddch(progress_win, 0, idx+47, '>' );

  wnoutrefresh( progress_win );
  doupdate();
  delwin( progress_win );

} /* End of Screen_Progress_Bar() */

/*------------------------------------------------------------------------*/

/*  Screen_Rotor_Control()
 *
 *  Function to display Az/El rotor status and send
 *  directional demands to the GS-232 controller
 */

  void
Screen_Rotor_Control( double azim, double elev, int resol, int displ )
{
  int
	azim_dir,   /* Azimuth direction   */
	elev_dir,   /* Elevation direction */
	azim_error, /* Azimuth error flag  */
	elev_error; /* Elevation error flag*/

  static int
	azim_dem,  /* Azimuth demand   */
	elev_dem;  /* Elevation demand */

  /* Rotor status window */
  WINDOW *rotor_control_win;


  /* Read current rotor directions. If serial port    */
  /* read fails, then default azim/elev position to 0 */
  Rotor_Read_Direction( &azim_dir, &elev_dir );
  if( isFlagSet(ROTOR_READ_FAIL_FLAG) )
	azim_dir = elev_dir = 0;

  /* Initialize Azimuth and Elevation demand when satcom is started */
  if( isFlagClear(CONTROL_INIT_FLAG) &&
	  isFlagClear(ROTOR_READ_FAIL_FLAG) )
  {
	Set_Flag( CONTROL_INIT_FLAG );
	azim_dem = azim_dir;
	elev_dem = elev_dir;
  }

  /* Calculate new Az/El demand if rotors are not running */
  if( isFlagClear(ROTORS_RUNNING_FLAG) )
  {
	/* Calculate Az/El demand, rounded to tracking resolution */
	azim_dem = (int)( azim + resol/2.0 );
	elev_dem = (int)( elev + resol/2.0 );
	azim_dem = ( azim_dem / resol ) * resol;
	elev_dem = ( elev_dem / resol ) * resol;

	/* Set limits to azimuth and elevation demand just in case */
	azim_dem = ((azim_dem < 0) ? 0 : ((azim_dem > 360) ? 360 : azim_dem));
	elev_dem = ((elev_dem < 0) ? 0 : ((elev_dem > 180) ? 180 : elev_dem));

	/* If reverse tracking is enabled, the elevation is modified to */
	/* 180-elev and azimuth to azim-180. This helps with overhead   */
	/* and North-crossing passes to avoid wide changes of azimuth   */
	if( isFlagSet(REVERSE_TRACK_FLAG) )
	{
	  elev_dem = 180 - elev_dem;
	  azim_dem -= 180;
	  azim_dem = ( (azim_dem < 0) ? azim_dem+360 : azim_dem );
	}

  } /* End if( isFlagClear(ROTORS_RUNNING_FLAG) ) */

  /* Set direction error flags if demand and   */
  /* actual position differ by more than 2 deg */
  azim_error = abs( azim_dem - azim_dir );
  azim_error =
	((azim_error >  2) && (azim_error < 358)) ||
	((azim_dem ==   0) && (azim_dir > 358) && (azim_dir < 362)) ||
	((azim_dem == 360) && (azim_dir >  -2) && (azim_dir <   2));

  elev_error = abs( elev_dem - elev_dir );
  elev_error = ( elev_error > 2 );

  /* If azim or elev error, set the relevant flag */
  if( (azim_error || elev_error) )
	Set_Flag( DIRECTION_ERROR_FLAG );
  else
  {
	if( isFlagClear(TRACKING_ENABLED_FLAG) )
	{
	  Close_Rotor_Serial();
	  Clear_Flag( ROTORS_ENABLED_FLAG );
	}

	Clear_Flag( DIRECTION_ERROR_FLAG |
		ROTORS_RUNNING_FLAG );
  }

  /* If direction error and rotors not already running, */
  /* send a new position demand to the rotor controller */
  if( isFlagSet(  DIRECTION_ERROR_FLAG) &&
	  isFlagClear(ROTORS_RUNNING_FLAG) )
  {
	Set_Flag( ROTORS_RUNNING_FLAG );
	Rotor_Send_Direction( azim_dem, elev_dem );
  }

  if( displ ) /* Display rotor status if enabled */
  {
	/* Draw some lines to box the results and add labels */
	rotor_control_win = newwin( 8, 16, 13, 1 );
	wattrset( rotor_control_win, CYAN );
	box(      rotor_control_win, ACS_VLINE, ACS_HLINE );

	Draw_Vline(rotor_control_win, 1, 7, 7 );
	Draw_Vline(rotor_control_win, 1, 8, 7 );
	Draw_Hline(rotor_control_win, 1, 0, 15);
	Draw_Hline(rotor_control_win, 4, 0, 15);
	mvwaddstr( rotor_control_win, 4, 3, "ELEVATION" );
	mvwaddstr( rotor_control_win, 1, 4, "AZIMUTH" );

	wattrset(  rotor_control_win, MAGENTA );
	mvwaddstr( rotor_control_win, 2, 1, "Demand" );
	mvwaddstr( rotor_control_win, 2, 9, "Actual" );
	mvwaddstr( rotor_control_win, 5, 1, "Demand" );
	mvwaddstr( rotor_control_win, 5, 9, "Actual" );

	if( isFlagSet(TRACKING_ENABLED_FLAG) )
	{
	  wattrset(rotor_control_win, YELLOW );
	  mvwaddstr( rotor_control_win,  0, 3, " TRACKING " );
	}
	else
	  if( isFlagSet(ROTORS_RUNNING_FLAG) )
	  {
		wattrset(rotor_control_win, YELLOW );
		mvwaddstr( rotor_control_win,  0, 3, " RUNNING " );
	  }

	/* Display Reverse tracking flag if enabled */
	if( isFlagSet(REVERSE_TRACK_FLAG) )
	{
	  wattrset(  rotor_control_win, YELLOW );
	  mvwaddch(  rotor_control_win, 3, 9, 'R' );
	  mvwaddch(  rotor_control_win, 6, 9, 'R' );
	}

	/* Display Azim/Elev demand */
	wattrset(  rotor_control_win, GREEN );
	mvwprintw( rotor_control_win, 3, 2, "%4d", azim_dem );
	mvwaddch(  rotor_control_win, 3, 6, ACS_DEGREE );
	mvwprintw( rotor_control_win, 6, 2, "%4d", elev_dem );
	mvwaddch(  rotor_control_win, 6, 6, ACS_DEGREE );

	/* Display 'Stop' if no direction error */
	if( !azim_error )
	{
	  /* Display "Stop" for Azim demand */
	  wattrset(  rotor_control_win, YELLOW );
	  mvwprintw( rotor_control_win, 3, 1, "%s", " Stop " );
	}

	if( !elev_error )
	{
	  /* Display "Stop" for Elev demand */
	  wattrset(  rotor_control_win, YELLOW );
	  mvwprintw( rotor_control_win, 6, 1, "%s", " Stop " );
	}

	/* If serial port read fail flag is set, show an alarm */
	if( isFlagSet(ROTOR_READ_FAIL_FLAG) )
	{
	  wattrset(  rotor_control_win, ERROR );
	  mvwprintw( rotor_control_win, 3, 9, "%s", "!Data?" );
	  mvwprintw( rotor_control_win, 6, 9, "%s", "!Data?" );
	}
	else
	{
	  /* Display directions in red if not equal to demand +- 2 */
	  wattrset(  rotor_control_win, azim_error ? RED : GREEN );
	  mvwprintw( rotor_control_win, 3, 10, "%4d", azim_dir );
	  mvwaddch(  rotor_control_win, 3, 14, ACS_DEGREE );
	  wattrset(  rotor_control_win, elev_error ? RED : GREEN );
	  mvwprintw( rotor_control_win, 6, 10, "%4d", elev_dir );
	  mvwaddch(  rotor_control_win, 6, 14, ACS_DEGREE );
	}

	/* Done with rotor_control_win */
	if( isFlagClear(ROTORS_ENABLED_FLAG) )
	  wclear( rotor_control_win );
	wnoutrefresh( rotor_control_win );
	delwin( rotor_control_win );
	doupdate();
  } /* End of if( displ ) */

} /* End of Screen_Rotor_Control() */

/*------------------------------------------------------------------------*/

/*  Screen_CAT_Control()
 *
 *  Function to display the Transceiver (FT-847) CAT window
 */

  void
Screen_CAT_Control( tcvr_status_t *tcvr_status )
{
  int
	idx,       /* Index for loops etc */
	graph_idx, /* Index to length of bar graph displays    */
	mhz,       /* MHz, KHz, Hzx10 parts of tx/rx frequency */
	khz,
	hz10;

  /* Mode strings (LSB, USB etc) */
  char *mode_names[NUM_MODE_CODES+1] = { TCVR_MODE_NAMES };

  /* Code numbers of tcvr modes */
  unsigned char mode_codes[NUM_MODE_CODES] = { TCVR_MODE_CODES };

  char graph[22]; /* PO/ALC or S-Meter bar graph  */

  /* Frequency converted to a string "MHz,Khz,Hz" */
  char freq_string[14];

  /* Transceiver status window */
  WINDOW *transceiver_control_win;

  /* Draw some lines to box the results and add labels */
  transceiver_control_win = newwin( 8, 61, 13, 18 );
  wattrset( transceiver_control_win, CYAN );
  box(      transceiver_control_win, ACS_VLINE, ACS_HLINE );

  wattrset(  transceiver_control_win, MAGENTA );
  mvwaddstr( transceiver_control_win, 2,  2, "Mode  Frequency   Doppler TRACK" );
  mvwaddstr( transceiver_control_win, 2, 35, "Mode  Frequency   Doppler" );
  mvwaddstr( transceiver_control_win, 5,  1, "Dnlnk:              PLoss:" );
  mvwaddstr( transceiver_control_win, 5, 31, "Uplnk:              PLoss:" );
  mvwaddstr( transceiver_control_win, 6,  1, "S-Meter:" );
  mvwaddstr( transceiver_control_win, 6, 31, "PO/ALC:" );

  wattrset( transceiver_control_win, CYAN );
  Draw_Hline(transceiver_control_win, 1,  0, 60 );
  Draw_Hline(transceiver_control_win, 4,  0, 60 );
  Draw_Vline(transceiver_control_win, 4, 30,  7 );
  Draw_Vline(transceiver_control_win, 1,  6,  4 );
  Draw_Vline(transceiver_control_win, 1, 18,  4 );
  Draw_Vline(transceiver_control_win, 1, 27,  4 );
  Draw_Vline(transceiver_control_win, 1, 33,  4 );
  Draw_Vline(transceiver_control_win, 1, 39,  4 );
  Draw_Vline(transceiver_control_win, 1, 51,  4 );

  wattrset(  transceiver_control_win, YELLOW );
  mvwaddstr( transceiver_control_win, 0, 14, " CAT CONTROL FOR THE YAESU FT-847 ");

  wattrset(  transceiver_control_win, CYAN );
  mvwaddstr( transceiver_control_win, 1,  3, " RECEIVER STATUS " );
  mvwaddstr( transceiver_control_win, 1, 38, " TRANSMITTER STATUS " );

  if( isFlagClear(TCVR_READ_FAIL_FLAG) )
  {
	/* Display transceiver mode */
	wattrset(  transceiver_control_win, GREEN );

	/* Rx mode of operation */
	for( idx = 0; idx < NUM_MODE_CODES; idx++ )
	  if( tcvr_status->rx_mode == mode_codes[idx] )
		break;
	mvwprintw( transceiver_control_win, 3, 1, "%5s", mode_names[idx] );

	/* Rx frequency */
	mhz  =  tcvr_status->rx_freq / 100000;
	khz  = (tcvr_status->rx_freq - mhz*100000) / 100;
	hz10 = (tcvr_status->rx_freq - mhz*100000 - khz*100);

	snprintf( freq_string, 14, "%3d,%03d,%02d0", mhz, khz, hz10 );
	freq_string[11] = 0;
	mvwaddstr( transceiver_control_win, 3, 7, freq_string );

	/* Downlink Frequency */
	mhz  =  tcvr_status->dnlink_freq / 100000;
	khz  = (tcvr_status->dnlink_freq - mhz*100000) / 100;
	hz10 = (tcvr_status->dnlink_freq - mhz*100000 - khz*100);

	snprintf( freq_string, 14, "%5d,%03d,%02d0", mhz, khz, hz10 );
	freq_string[13] = 0;
	mvwaddstr( transceiver_control_win, 5, 7, freq_string );

	/* Rx path loss */
	mvwprintw( transceiver_control_win, 5, 27, "%3d", tcvr_status->rx_ploss);

	/* S-Meter bar graph */
	wattrset(  transceiver_control_win, GREEN );
	graph_idx = 1 + (tcvr_status->rx_status & S_METER);
	graph_idx += graph_idx >> 2;
	graph_idx >>= 1;
	for( idx = 0; idx < graph_idx; idx++ )
	  graph[idx] = '=';
	graph[idx++] = '>';
	graph[idx]   = '\0';
	mvwaddstr( transceiver_control_win, 6, 8, graph );

	wattrset(  transceiver_control_win, GREEN );
	/* Tx mode of operation */
	for( idx = 0; idx < NUM_MODE_CODES; idx++ )
	  if( tcvr_status->tx_mode == mode_codes[idx] )
		break;
	mvwprintw( transceiver_control_win, 3, 34, "%5s", mode_names[idx] );

	/* Tx frequency */
	mhz =  tcvr_status->tx_freq / 100000;
	khz = (tcvr_status->tx_freq - mhz*100000) / 100;
	hz10  = (tcvr_status->tx_freq - mhz*100000 - khz*100);
	snprintf( freq_string, 14, "%03d,%03d,%02d0", mhz, khz, hz10 );
	freq_string[11] = 0;
	mvwaddstr( transceiver_control_win, 3, 40, freq_string );

	/* Uplink Frequency */
	mhz  =  tcvr_status->uplink_freq / 100000;
	khz  = (tcvr_status->uplink_freq - mhz*100000) / 100;
	hz10 = (tcvr_status->uplink_freq - mhz*100000 - khz*100);

	snprintf( freq_string, 14, "%5d,%03d,%02d0", mhz, khz, hz10 );
	freq_string[13] = 0;
	mvwaddstr( transceiver_control_win, 5, 37, freq_string );

	/* Tx path loss */
	mvwprintw( transceiver_control_win, 5, 57, "%3d", tcvr_status->tx_ploss);

	/* PO/ALC bar graph */
	wattrset(  transceiver_control_win, GREEN );
	graph_idx = 1 + (tcvr_status->tx_status & PO_ALC);
	graph_idx += graph_idx >> 2;
	graph_idx >>= 1;
	for( idx = 0; idx < graph_idx; idx++ )
	  graph[idx] = '=';
	graph[idx++] = '>';
	graph[idx]   = '\0';
	mvwaddstr( transceiver_control_win, 6, 39, graph );

	if( isFlagClear(DOPPLER_DISABLED_FLAG) )
	{
	  /* Tx doppler shift */
	  mvwprintw( transceiver_control_win, 3, 52, "%7d",
		  tcvr_status->tx_doppler);

	  /* Rx doppler shift */
	  mvwprintw( transceiver_control_win, 3, 19, "%7d",
		  tcvr_status->rx_doppler);
	}
	else
	{
	  /* No Tx doppler shift */
	  mvwaddstr( transceiver_control_win, 3, 54, "-----" );

	  /* No Rx doppler shift */
	  mvwaddstr( transceiver_control_win, 3, 21, "-----" );
	}

	/* Display Tx-Rx tracking status */
	wattrset(  transceiver_control_win, GREEN );
	mvwaddstr( transceiver_control_win, 3, 28, " --- " );

	if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) )
	{
	  wattrset(  transceiver_control_win, YELLOW );
	  mvwaddstr( transceiver_control_win, 3, 28, "<--->" );
	}

	/* Display transponder mode */
	if( tcvr_status->flags == NO_DATA )
	  wattrset(  transceiver_control_win, RED );
	else
	  wattrset(  transceiver_control_win, BLUE|A_BOLD );
	mvwaddstr( transceiver_control_win, 1, 22, "[MODE:" );
	mvwaddstr( transceiver_control_win, 1, 28, tcvr_status->mode_name );
	mvwaddch(  transceiver_control_win, 1,
		28+strlen(tcvr_status->mode_name), ']' );

  } /* End of if( isFlagClear(TCVR_READ_FAIL_FLAG) ) */
  else
  {
	wattrset(  transceiver_control_win, RED );
	mvwaddstr( transceiver_control_win, 1, 22, " SERIAL FAIL! " );
  }

  /* Done with transceiver_control_win */
  wnoutrefresh( transceiver_control_win );
  delwin( transceiver_control_win );
  doupdate();

} /* End of Screen_CAT_Control() */

/*------------------------------------------------------------------------*/

/*  Screen_Commands()
 *
 *  Displays the keystroke commands at bottom of root window
 */

  void
Screen_Commands( int mode )
{
  WINDOW *commands_win;
  commands_win = newwin( 3, 78, 21, 1 );

  wattrset( commands_win, COMMAND );

  switch( mode )
  {
	case MAIN_SCREEN_MODE : /* Splash-page mode */
	  mvwaddstr(commands_win, 0, 3,
		  "Single-Satellite-Mode    Multi-Satellite-Mode      Multiple-Locations-Mode ");
	  mvwaddstr(commands_win, 1, 3,
		  "Pass-Predictions-Mode    Manual-Commands-Mode      Illumination-Predictions");
	  mvwaddstr(commands_win, 2, 3,
		  "Offset-Calibration       Fullscale-Calibration     QTH-Grid-Locator    Quit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 0,  0, "<S>");
	  mvwaddstr(commands_win, 0, 25, "<M>");
	  mvwaddstr(commands_win, 0, 51, "<L>");

	  mvwaddstr(commands_win, 1,  0, "<P>");
	  mvwaddstr(commands_win, 1, 25, "<C>");
	  mvwaddstr(commands_win, 1, 51, "<I>");

	  mvwaddstr(commands_win, 2,  0, "<O>");
	  mvwaddstr(commands_win, 2, 25, "<F>");
	  mvwaddstr(commands_win, 2, 51, "<G>");
	  mvwaddstr(commands_win, 2, 71, "<Q>");
	  break;

	case SINGLE_SAT_MODE : /* Single satellite/observer mode */
	  mvwaddstr(commands_win, 0, 3,
		  "Toggle-CAT    Toggle-Track     Update-Track     Next-Mode    Toggle-Doppler");

	  mvwaddstr(commands_win, 1, 3,
		  "Toggle-AE-Track    Earth-Track    Multi-Satell    Pass-Predict    Multi-Loc");

	  mvwaddstr(commands_win, 2, 3,
		  "Select-Satellite    Previous    Next    First    Last      Default     Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 0,  0, "<C>");
	  mvwaddstr(commands_win, 0, 14, "<T>");
	  mvwaddstr(commands_win, 0, 31, "<U>");
	  mvwaddstr(commands_win, 0, 48, "<W>");
	  mvwaddstr(commands_win, 0, 61, "<D>");

	  mvwaddstr(commands_win, 1,  0, "<R>");
	  mvwaddstr(commands_win, 1, 19, "<E>");
	  mvwaddstr(commands_win, 1, 34, "<M>");
	  mvwaddstr(commands_win, 1, 50, "<P>");
	  mvwaddstr(commands_win, 1, 66, "<L>");

	  mvwaddstr(commands_win, 2,  0, "<N>");
	  mvwaddstr(commands_win, 2, 20, "< >");
	  mvwaddch( commands_win, 2, 21, ACS_LARROW);
	  mvwaddstr(commands_win, 2, 32, "< >");
	  mvwaddch( commands_win, 2, 33, ACS_RARROW);
	  mvwaddstr(commands_win, 2, 40, "< >");
	  mvwaddch( commands_win, 2, 41, ACS_UARROW);
	  mvwaddstr(commands_win, 2, 49, "< >");
	  mvwaddch( commands_win, 2, 50, ACS_DARROW);
	  mvwaddstr(commands_win, 2, 57, "<Ins>");
	  mvwaddstr(commands_win, 2, 71, "<Q>");
	  break;

	case MULTI_SAT_MODE : /* Multi-satellite/single-observer mode */
	  mvwaddstr(commands_win, 0, 7,
		  "Previous-Group       First-Group    Previous-Location     Home-Location");
	  mvwaddstr(commands_win, 1, 7,
		  "Next-Group           Last-Group     Next-Location      Default-Satelite");
	  mvwaddstr(commands_win, 2, 3,
		  "Previous    Next    Left-Page    Right-Page      Select-Sat & Exit     Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 0,  0, "<Pg-Up>");
	  mvwaddstr(commands_win, 0, 22, "<Home>");
	  mvwaddstr(commands_win, 0, 40, "<P>");
	  mvwaddstr(commands_win, 0, 62, "<H>");

	  mvwaddstr(commands_win, 1,  0, "<Pg-Dn>");
	  mvwaddstr(commands_win, 1, 23, "<End>");
	  mvwaddstr(commands_win, 1, 40, "<N>");
	  mvwaddstr(commands_win, 1, 57, "<Ins>");

	  mvwaddstr(commands_win, 2,  0, "< >");
	  mvwaddch( commands_win, 2,  1, ACS_UARROW);
	  mvwaddstr(commands_win, 2, 12, "< >");
	  mvwaddch( commands_win, 2, 13, ACS_DARROW);
	  mvwaddstr(commands_win, 2, 20, "< >");
	  mvwaddch( commands_win, 2, 21, ACS_LARROW);
	  mvwaddstr(commands_win, 2, 33, "< >");
	  mvwaddch( commands_win, 2, 34, ACS_RARROW);
	  mvwaddstr(commands_win, 2, 47, "<RET>");
	  mvwaddstr(commands_win, 2, 71, "<Q>");
	  break;

	case MULTI_OBS_MODE : /* Multi-observer/single-satellite mode */
	  mvwaddstr(commands_win, 0, 7,
		  "Add-New-Location       Previous-Location    Home-Location    User-Entry");
	  mvwaddstr(commands_win, 1, 7,
		  "Remove-Location        Next-Location        Verify-Enter     Save-Enter");
	  mvwaddstr(commands_win, 2, 3,
		  "Multi-Sat-Mode      Previous    Next    First    Last      Default     Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 0,  0, "<Pg-Dn>");
	  mvwaddstr(commands_win, 0, 24, "<Home>");
	  mvwaddstr(commands_win, 0, 48, "<H>");
	  mvwaddstr(commands_win, 0, 65, "<U>");

	  mvwaddstr(commands_win, 1,  0, "<Pg-Up>");
	  mvwaddstr(commands_win, 1, 25, "<End>");
	  mvwaddstr(commands_win, 1, 46, "<RET>");
	  mvwaddstr(commands_win, 1, 65, "<S>");

	  mvwaddstr(commands_win, 2, 0, "<M>");
	  mvwaddstr(commands_win, 2,20, "< >");
	  mvwaddch( commands_win, 2,21, ACS_LARROW);
	  mvwaddstr(commands_win, 2,32, "< >");
	  mvwaddch( commands_win, 2,33, ACS_RARROW);
	  mvwaddstr(commands_win, 2,40, "< >");
	  mvwaddch( commands_win, 2,41, ACS_UARROW);
	  mvwaddstr(commands_win, 2,49, "< >");
	  mvwaddch( commands_win, 2,50, ACS_DARROW);
	  mvwaddstr(commands_win, 2,57, "<Ins>");
	  mvwaddstr(commands_win, 2,71, "<Q>");
	  break;

	case PASS_PREDICTION_MODE : /* Predict satellite passes */
	  mvwaddstr(commands_win, 0, 3,
		  "Accessible-Pass    User-Input    Move-Forward         Next-Location    Home");
	  mvwaddstr(commands_win, 1, 3,
		  "Visible-Pass       Real-Time     Reset-Search         Previous-Location    ");
	  mvwaddstr(commands_win, 2, 3,
		  "Multi-Sat-Mode     Previous      Next    First    Last      Default    Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 0,  0, "<A>");
	  mvwaddstr(commands_win, 0, 19, "<U>");
	  mvwaddstr(commands_win, 0, 33, "<F>");
	  mvwaddstr(commands_win, 0, 50, "<Pg-Dn>");
	  mvwaddstr(commands_win, 0, 71, "<H>");

	  mvwaddstr(commands_win, 1,  0, "<V>");
	  mvwaddstr(commands_win, 1, 19, "<R>");
	  mvwaddstr(commands_win, 1, 33, "<S>");
	  mvwaddstr(commands_win, 1, 50, "<Pg-Up>");

	  mvwaddstr(commands_win, 2, 0, "<M>");
	  mvwaddstr(commands_win, 2,19, "< >");
	  mvwaddch( commands_win, 2,20, ACS_LARROW);
	  mvwaddstr(commands_win, 2,33, "< >");
	  mvwaddch( commands_win, 2,34, ACS_RARROW);
	  mvwaddstr(commands_win, 2,41, "< >");
	  mvwaddch( commands_win, 2,42, ACS_UARROW);
	  mvwaddstr(commands_win, 2,50, "< >");
	  mvwaddch( commands_win, 2,51, ACS_DARROW);
	  mvwaddstr(commands_win, 2,58, "<Ins>");
	  mvwaddstr(commands_win, 2,71, "<Q>");
	  break;

	case ILLUM_PREDICTION_MODE : /* Predict satellite illumination */
	  mvwaddstr(commands_win, 1, 3,
		  "Move-Forward       Move-Backward         Reset-Predictions       User-Input");
	  mvwaddstr(commands_win, 2, 3,
		  "Multi-Sat-Mode     Previous      Next    First    Last      Default    Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 1,  0, "<F>");
	  mvwaddstr(commands_win, 1, 19, "<B>");
	  mvwaddstr(commands_win, 1, 41, "<R>");
	  mvwaddstr(commands_win, 1, 65, "<U>");

	  mvwaddstr(commands_win, 2, 0, "<M>");
	  mvwaddstr(commands_win, 2,19, "< >");
	  mvwaddch( commands_win, 2,20, ACS_LARROW);
	  mvwaddstr(commands_win, 2,33, "< >");
	  mvwaddch( commands_win, 2,34, ACS_RARROW);
	  mvwaddstr(commands_win, 2,41, "< >");
	  mvwaddch( commands_win, 2,42, ACS_UARROW);
	  mvwaddstr(commands_win, 2,50, "< >");
	  mvwaddch( commands_win, 2,51, ACS_DARROW);
	  mvwaddstr(commands_win, 2,58, "<Ins>");
	  mvwaddstr(commands_win, 2,71, "<Q>");
	  break;

	case MANUAL_COMM_MODE : /* Manual Commands mode */
	  mvwaddstr(commands_win, 1, 3,
		  "Left    Right    Manual-Azimuth/Elev    Track-Moon    Track-Polar-Star     ");

	  mvwaddstr(commands_win, 2, 3,
		  "Up      Down     Stop    Park-Rotors    Track-Sun                      Exit");

	  wattrset( commands_win, KEYSTROKE );
	  mvwaddstr(commands_win, 1,  0, "< >");
	  mvwaddch( commands_win, 1,  1, ACS_LARROW);
	  mvwaddstr(commands_win, 1,  8, "< >");
	  mvwaddch( commands_win, 1,  9, ACS_RARROW);
	  mvwaddstr(commands_win, 1, 17, "<M>");
	  mvwaddstr(commands_win, 1, 40, "<L>");
	  mvwaddstr(commands_win, 1, 54, "<P>");

	  mvwaddstr(commands_win, 2,  0, "< >");
	  mvwaddch( commands_win, 2,  1, ACS_UARROW);
	  mvwaddstr(commands_win, 2,  8, "< >");
	  mvwaddch( commands_win, 2,  9, ACS_DARROW);
	  mvwaddstr(commands_win, 2, 40, "<S>");
	  mvwaddstr(commands_win, 2, 17, "< >");
	  mvwaddstr(commands_win, 2, 25, "<H>");
	  mvwaddstr(commands_win, 2, 71, "<Q>");

  } /* End of switch( mode ) */

  /* Done with commands_win */
  wnoutrefresh( commands_win );
  delwin( commands_win );
  doupdate();

} /* End of Screen_Commands() */

/*------------------------------------------------------------------------*/

/*  Screen_Offset_Calibration()
 *
 *  Function to display the Offset calibration window
 */

  void
Screen_Offset_Calibration(void)
{
  int
	idx,    /* Index for loops etc */
	ileft,  /* Left value of offset equation  */
	iright, /* Right value of offset equation */
	ierror; /* Offset calibration error */

  char
	strn[28], /* String for reading data from Controller    */
	buff[6];  /* Buffer for cloning data from input strings */

  /* Offset calibration window */
  WINDOW * offset_calib_win;
  offset_calib_win = newwin( 13, 25, 2, 4 );

  bzero( strn, 28 );
  bzero( buff, 6 );

  /* Draw some lines to box the results and add labels */
  wattrset(  offset_calib_win, CYAN );
  box(       offset_calib_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( offset_calib_win,  0, 2, " OFFSET CALIBRATION " );

  wattrset(  offset_calib_win, YELLOW );
  mvwaddstr( offset_calib_win, 10, 1, "Please Reset or switch" );
  mvwaddstr( offset_calib_win, 11, 1, "power OFF/ON when done" );

  wattrset(  offset_calib_win, MAGENTA );
  mvwaddstr( offset_calib_win, 2, 4, " Azimuth Offset " );
  mvwaddstr( offset_calib_win, 6, 2, " Elevation Offset " );

  /* Create bar graphs for displaying calibration status */
  wattrset( offset_calib_win, ERROR );
  for( idx = 2; idx < 9; idx++ )
  {
	mvwaddch( offset_calib_win, 3, idx,   ACS_VLINE );
	mvwaddch( offset_calib_win, 3, idx+14,ACS_VLINE );
	mvwaddch( offset_calib_win, 7, idx,   ACS_VLINE );
	mvwaddch( offset_calib_win, 7, idx+14,ACS_VLINE );
  }

  wattrset( offset_calib_win, APPROX );
  for( idx = 9; idx < 12; idx++ )
  {
	mvwaddch( offset_calib_win, 3, idx,   ACS_VLINE );
	mvwaddch( offset_calib_win, 3, idx+4, ACS_VLINE );
	mvwaddch( offset_calib_win, 7, idx,   ACS_VLINE );
	mvwaddch( offset_calib_win, 7, idx+4, ACS_VLINE );
  }

  wattrset( offset_calib_win, FINISH );
  mvwaddch( offset_calib_win, 3, 12, ACS_VLINE );
  mvwaddch( offset_calib_win, 7, 12, ACS_VLINE );

  /* Draw lines to box bargraph displays */
  wattrset(   offset_calib_win, CYAN );
  Draw_Hline( offset_calib_win, 1, 0, 24 );
  Draw_Hline( offset_calib_win, 5, 0, 24 );
  Draw_Hline( offset_calib_win, 9, 0, 24 );

  wattrset( offset_calib_win, A_BOLD );

  /* Request offset calibration data from Controller */
  Open_Rotor_Serial();
  Rotor_Send_String( "O2" );
  Clear_Flag( ROTOR_READ_FAIL_FLAG );

  Screen_Root();

  /* Read in data from controller. After sending an O2 command, the */
  /* GS232 controller sends offset data in the following format:    */
  /* AZ+llll=+rrrr EL+llll=+rrrr where llll and rrrr are 4-digit    */
  /* integer numbers that must be equal for zero offset calibration.*/
  while( isFlagClear(ROTOR_READ_FAIL_FLAG) )
  {
	/* Input Az-El offset calibration data */
	Rotor_Read_String( strn );

	/* Clone left-side number of Az data */
	strncpy( buff, &strn[2], 5 );
	buff[5] = '\0';
	ileft = atoi( buff );

	/* Clone right-side number of Az data */
	strncpy( buff, &strn[8], 5 );
	buff[5] = '\0';
	iright = atoi( buff );

	/* Calculate and limit calibration error to +- 10 */
	ierror = iright - ileft;
	if( ierror >  10 )
	  ierror =  10;
	if( ierror < -10 )
	  ierror = -10;
	/* Display a vertical arrow to indicate error on bargraph */
	mvwaddstr( offset_calib_win, 4, 1, "                       " );
	mvwaddch( offset_calib_win, 4, 12 + ierror, ACS_UARROW );

	/* Clone left-side number of El data */
	strncpy( buff, &strn[16], 5 );
	buff[5] = '\0';
	ileft = atoi( buff );

	/* Clone right-side number of El data */
	strncpy( buff, &strn[22], 5 );
	buff[5] = '\0';
	iright = atoi( buff );

	/* Calculate and limit calibration error to +- 10 */
	ierror = iright - ileft;
	if( ierror >  10 )
	  ierror =  10;
	if( ierror < -10 )
	  ierror = -10;

	/* Display a vertical arrow to indicate error on bargraph */
	mvwaddstr( offset_calib_win, 8, 1, "                       " );
	mvwaddch(  offset_calib_win, 8, 12 + ierror, ACS_UARROW );

	/* Refresh window */
	wnoutrefresh( offset_calib_win );
	doupdate();

  } /* End of while() */

  /* Display an error message if no signal from Controller */
  wattrset(  offset_calib_win, ERROR );
  mvwaddstr( offset_calib_win, 10, 1, "No data from Controller" );
  mvwaddstr( offset_calib_win, 11, 1, "This window is closing " );

  /* Refresh window */
  wnoutrefresh( offset_calib_win );
  doupdate();

  sleep(5);

  Clear_Flag( ROTOR_READ_FAIL_FLAG );

  /* Done with offset_calib_win */
  delwin( offset_calib_win );

  Close_Rotor_Serial();

} /* End of Screen_Offset_Calibration() */

/*------------------------------------------------------------------------*/

/*  Screen_Fullscale_Calibration()
 *
 *  Function to display the Full-scale calibration window
 */

  void
Screen_Fullscale_Calibration(void)
{
  int
	idx,   /* Index for loops etc */
	azim,  /* Azimuth value   */
	elev,  /* Elevation value */
	error; /* Fullscale calibration error */

  char
	strn[15], /* String for reading data from Controller */
	buff[6];  /* Buffer for cloning data from input strings */

  /* Full-scale calibration window */
  WINDOW * fullscale_calib_win;
  fullscale_calib_win = newwin( 13, 25, 2, 4 );

  bzero( strn, 15 );
  bzero( buff, 6 );

  /* Draw some lines to box the results and add labels */
  wattrset(  fullscale_calib_win, CYAN );
  box(       fullscale_calib_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( fullscale_calib_win,  0, 1, " FULLSCALE CALIBRATION " );

  wattrset(  fullscale_calib_win, YELLOW );
  mvwaddstr( fullscale_calib_win, 10, 1, "Please Reset or switch" );
  mvwaddstr( fullscale_calib_win, 11, 1, "power OFF/ON when done" );

  wattrset(  fullscale_calib_win, MAGENTA );
  mvwaddstr( fullscale_calib_win, 2, 4, " Azimuth Full-scale " );
  mvwaddstr( fullscale_calib_win, 6, 2, " Elevation Full-scale " );

  /* Create bar graphs for displaying calibration status */
  wattrset( fullscale_calib_win, ERROR );
  for( idx = 2; idx < 9; idx++ )
  {
	mvwaddch( fullscale_calib_win, 3, idx,   ACS_VLINE );
	mvwaddch( fullscale_calib_win, 3, idx+14,ACS_VLINE );
	mvwaddch( fullscale_calib_win, 7, idx,   ACS_VLINE );
	mvwaddch( fullscale_calib_win, 7, idx+14,ACS_VLINE );
  }

  wattrset( fullscale_calib_win, APPROX );
  for( idx = 9; idx < 12; idx++ )
  {
	mvwaddch( fullscale_calib_win, 3, idx,   ACS_VLINE );
	mvwaddch( fullscale_calib_win, 3, idx+4, ACS_VLINE );
	mvwaddch( fullscale_calib_win, 7, idx,   ACS_VLINE );
	mvwaddch( fullscale_calib_win, 7, idx+4, ACS_VLINE );
  }

  wattrset( fullscale_calib_win, FINISH );
  mvwaddch( fullscale_calib_win, 3, 12, ACS_VLINE );
  mvwaddch( fullscale_calib_win, 7, 12, ACS_VLINE );

  /* Draw lines to box bargraph displays */
  wattrset(   fullscale_calib_win, CYAN );
  Draw_Hline( fullscale_calib_win, 1, 0, 24 );
  Draw_Hline( fullscale_calib_win, 5, 0, 24 );
  Draw_Hline( fullscale_calib_win, 9, 0, 24 );

  wattrset( fullscale_calib_win, A_BOLD );

  /* Request fullscale calibration data from Controller */
  Open_Rotor_Serial();
  Rotor_Send_String( "F2" );
  Clear_Flag( ROTOR_READ_FAIL_FLAG );

  Screen_Root();

  /* Read in data from controller. After sending an F2 command, the */
  /* GS232 controller sends fullscale data in the following format: */
  /* +aaaa+eeee where aaaa is Azimuth and eeee is Elevation. For    */
  /* Full-scale calibration, these should be ??? */
  while( isFlagClear(ROTOR_READ_FAIL_FLAG) )
  {
	/* Input Az-El fullscale calibration data */
	Rotor_Read_String( strn );

	/* Clone Azimuth value */
	strncpy( buff, strn, 5 );
	buff[5] = '\0';
	azim = atoi( buff ) - AZIM_OFFSET;
	if( azim < 180 )
	  azim += 360;

	/* Calculate and limit calibration error to +- 10 */
	error = azim - 360;
	if( error >  10 )
	  error =  10;
	if( error < -10 )
	  error = -10;

	/* Display a vertical arrow to indicate error on bargraph */
	mvwaddstr( fullscale_calib_win, 4, 1, "                       " );
	mvwaddch( fullscale_calib_win, 4, 12 + error, ACS_UARROW );

	/* Clone Elevation value */
	strncpy( buff, &strn[5], 5 );
	buff[5] = '\0';
	elev = atoi( buff );

	/* Calculate and limit calibration error to +- 10 */
	error = elev - 180;
	if( error >  10 )
	  error =  10;
	if( error < -10 )
	  error = -10;

	/* Display a vertical arrow to indicate error on bargraph */
	mvwaddstr( fullscale_calib_win, 8, 1, "                       " );
	mvwaddch(  fullscale_calib_win, 8, 12 + error, ACS_UARROW );

	/* Refresh window */
	wnoutrefresh( fullscale_calib_win );
	doupdate();

  } /* End of while() */

  /* Display an error message if no signal from Controller */
  wattrset(  fullscale_calib_win, ERROR );
  mvwaddstr( fullscale_calib_win, 10, 1, "No data from Controller" );
  mvwaddstr( fullscale_calib_win, 11, 1, "This window is closing " );

  /* Refresh window */
  wnoutrefresh( fullscale_calib_win );
  doupdate();

  sleep(5);

  Clear_Flag( ROTOR_READ_FAIL_FLAG );

  /* Done with fullscale_calib_win */
  delwin( fullscale_calib_win );

  Close_Rotor_Serial();

} /* End of Screen_Fullscale_Calibration() */

/*------------------------------------------------------------------------*/

/*  Post_Manual_Form()
 *
 *  Function to display the Manual rotor position entry
 *  form and send position request to GS232 controller
 */

  void
Post_Manual_Form( int *azim_dir, int *elev_dir )
{
  int
	idx,  /* Index for loops etc */
	cols, /* Columns in form     */
	rows, /* Rows in form        */
	chr,  /* Input from getch()  */
	azim, /* Azimuth value       */
	elev; /* Elevation value     */

  int
	form_entry, /* Data entered in form flag */
	form_done;  /* Form entries done flag    */

  /* Form for entering Manual Position parameters */
  FORM   *manual_form;
  FIELD  *data_field[ 5 ];
  /* Form containing windows */
  WINDOW
	*form_win,
	*form_subwin;

  /* Setup Manual Position fields and form */
  for( idx = 0; idx < 2; idx++ )
  {
	data_field[ idx ] = new_field( 1, 3, idx, 17, 0, 0 );
	field_opts_off( data_field[ idx ], O_AUTOSKIP );
  }

  data_field[ 2 ] = new_field( 1, 9, 3, 11, 0, 0 );
  field_opts_off(   data_field[ 2 ], O_EDIT );
  set_field_buffer( data_field[ 2 ], 0, "< Done! >" );

  data_field[ 3 ] = new_field( 1, 9, 3, 0, 0, 0 );
  field_opts_off(   data_field[ 3 ], O_EDIT );
  set_field_buffer( data_field[ 3 ], 0, "< Abort >" );

  data_field[ 4 ] = ( FIELD * ) NULL;

  set_field_type( data_field[ 0 ], TYPE_INTEGER, 0, 0, 360 );
  set_field_type( data_field[ 1 ], TYPE_INTEGER, 0, 0, 180 );

  /* Declare manual entries form */
  manual_form = new_form( data_field );

  /* Obtain size of from */
  scale_form( manual_form, &rows, &cols );

  /* Create form window and sub-window */
  form_win    = newwin( rows+2, cols+2, 7, 1 );
  form_subwin = newwin( rows,   cols,   8, 2 );

  /* Set form window and sub-window */
  set_form_win( manual_form, form_win );
  set_form_sub( manual_form, form_subwin );

  /* Box and label form window */
  wattrset(  form_win, CYAN );
  box(       form_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( form_win, 0,  2, " MANUAL ENTRY " );
  mvwaddch(  form_win, 3,  0, ACS_LTEE );
  mvwaddch(  form_win, 3, cols+1, ACS_RTEE );
  mvwaddch(  form_win, 0, 17, ACS_TTEE );

  post_form( manual_form );

  /* Enter various labels in the Manual settings form box */
  wattrset(  form_subwin, MAGENTA );
  mvwaddstr( form_subwin, 0,  0, "   Azimuth - deg" );
  mvwaddstr( form_subwin, 1,  0, " Elevation - deg" );
  wattrset(  form_subwin, CYAN );
  mvwhline(  form_subwin, 2, 0, ACS_HLINE, cols );
  mvwvline(  form_subwin, 0, 16, ACS_VLINE, 2 );
  mvwaddch(  form_subwin, 2, 16, ACS_BTEE );

  /* Prepare for the form's data input loop */
  form_driver( manual_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  form_done = form_entry = FALSE;

  wnoutrefresh( form_win );
  wnoutrefresh( form_subwin );
  doupdate();

  /* Process keystrokes from user */
  while( !form_done )
  {
	if( (field_index(current_field(manual_form)) == 2) ||
		(field_index(current_field(manual_form)) == 3) )
	  curs_set( FALSE );
	else
	  curs_set( TRUE );

	chr = getch();
	switch( chr )
	{
	  case KEY_RET : /* RETURN on 'Done!' field */
		if( field_index(current_field(manual_form)) == 2 )
		{
		  form_done = form_entry = TRUE;
		  break;
		}

		/* RETURN on 'Abort' field */
		if ( field_index(current_field(manual_form)) == 3 )
		{
		  form_done = TRUE;
		  break;
		}
		set_field_back( current_field( manual_form ), A_NORMAL );
		form_driver( manual_form, REQ_NEXT_FIELD );
		form_driver( manual_form, REQ_BEG_FIELD );
		set_field_back( current_field(manual_form), KEYSTROKE );
		break;

	  case KEY_DOWN : /* Down a field */
		set_field_back(current_field(manual_form), A_NORMAL);
		form_driver( manual_form, REQ_NEXT_FIELD );
		set_field_back( current_field(manual_form), KEYSTROKE );
		break;

	  case KEY_UP : /* Up a field */
		set_field_back(current_field(manual_form), A_NORMAL);
		form_driver( manual_form, REQ_PREV_FIELD );
		set_field_back( current_field(manual_form), KEYSTROKE );
		break;

	  case KEY_BACKSPACE : /* Back and delete a character */
		form_driver( manual_form, REQ_PREV_CHAR );
		form_driver( manual_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC : /* Delete character */
		form_driver( manual_form, REQ_DEL_CHAR );
		break;

	  case KEY_LEFT : /* Back one character */
		form_driver( manual_form, REQ_PREV_CHAR );
		break;

	  case KEY_RIGHT : /* Forward one character */
		form_driver( manual_form, REQ_NEXT_CHAR );
		break;

	  case KEY_ESC : /* Cancel operation */
		form_done = TRUE;
		curs_set( FALSE );
		break;

	  default : /* Enter character */
		form_driver( manual_form, chr );

	} /* End of switch( chr )  */

	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !form_done ) */

  if( form_entry )
  {
	/* Read position data from manual form */
	azim = atoi( field_buffer(data_field[0], 0) );
	elev = atoi( field_buffer(data_field[1], 0) );

	/* Return user entry to calling function */
	*azim_dir = azim;
	*elev_dir = elev;

	Screen_Root();
	Manual_Position_Rotors( azim, elev );

  } /* End of if( form_entry ) */

  /* Unpost form and free fields */
  unpost_form( manual_form );
  free_form(   manual_form );
  for ( idx = 0; idx < 5; idx++ ) free_field( data_field[ idx ] );

} /* End of Post_Manual_Form() */

/*------------------------------------------------------------------------*/

/*  Post_Name_Form()
 *
 *  Function to display a satellite name entry form
 */

  void
Post_Name_Form( char *name, WINDOW *form_win )
{
  int
	idx,  /* Index for loops etc */
	chr;  /* Input from getch()  */

  int
	form_entry, /* Data entered in form flag */
	form_done;  /* Form entries done flag    */

  FORM   *name_form;
  FIELD  *data_field[2];
  WINDOW *form_subwin;

  /* Setup Manual Position fields and form */
  data_field[ 0 ] = new_field( 1, 11, 0, 0, 0, 0 );
  data_field[ 1 ] = (FIELD *) NULL;

  field_opts_off( data_field[ 0 ], O_AUTOSKIP );

  /* Declare manual entries form */
  name_form = new_form( data_field );

  /* Set form window and sub-window */
  form_subwin = newwin( 1, 11, 7, 17);
  set_form_win( name_form, form_win );
  set_form_sub( name_form, form_subwin );

  post_form( name_form );

  /* Prepare for the form's data input loop */
  form_driver( name_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  form_done = form_entry = FALSE;

  curs_set( TRUE );

  /* Display name entry form */
  wnoutrefresh( form_win );
  wnoutrefresh( form_subwin );
  doupdate();

  /* Process keystrokes from user */
  while( !form_done )
  {
	chr = getch();
	switch( chr )
	{
	  /* RETURN on 'Done!' field */
	  case KEY_RET : /* Return, finished input */
		form_driver( name_form, REQ_NEXT_FIELD );
		form_entry = form_done = TRUE;
		break;

	  case KEY_BACKSPACE : /* Back and delete a character */
		form_driver( name_form, REQ_PREV_CHAR );
		form_driver( name_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC : /* Delete character */
		form_driver( name_form, REQ_DEL_CHAR );
		break;

	  case KEY_LEFT : /* Back one character */
		form_driver( name_form, REQ_PREV_CHAR );
		break;

	  case KEY_RIGHT : /* Forward one character */
		form_driver( name_form, REQ_NEXT_CHAR );
		break;

	  case KEY_ESC : /* Cancel operation */
		form_done = TRUE;
		break;

	  default : /* Enter character */
		form_driver( name_form, chr );

	} /* End of switch( chr )  */

	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !form_done ) */

  /* Enter satellite name and remove trailing spaces */
  if( form_entry )
  {
	strcpy( name, field_buffer(data_field[0], 0) );

	/* Dump trailing spaces from name */
	idx = 11;
	while( (name[--idx] == ' ') && ( idx != 0) );
	name[++idx] = '\0';
  }

  /* Unpost form and free fields */
  unpost_form( name_form );
  free_form(   name_form );
  for ( idx = 0; idx < 2; idx++ )
	free_field( data_field[ idx ] );

  /* Kill cursor */
  curs_set( FALSE );

} /* End of Post_Name_Form() */

/*------------------------------------------------------------------------*/

/*  Post_Location_Form()
 *
 *  Function to post a form for on-the-fly observer location entry
 */

  void
Post_Location_Form( int num, observer_status_t *multiobs_status )
{
  int
	form_entry, /* Data entered in form flag */
	save_file,  /* Save data to satcom.obs   */
	form_done,  /* Form entries done flag    */
	idx,        /* Index for loops etc */
	chr;        /* Input from getch()  */

  char gloc[7];  /* Grid locator buffer */

  /* Temp lon/lat for grid locator */
  double
	lon,
	lat;

  FORM   *location_form;
  FIELD  *data_field[6];
  WINDOW *form_subwin;

  curs_set( TRUE );
  cbreak();
  nodelay( stdscr, FALSE );

  /* Setup Manual Position fields and form */
  data_field[ 0 ] = new_field( 1, 11, 0,  0, 0, 0 );
  data_field[ 1 ] = new_field( 1,  6, 0, 21, 0, 0 );
  data_field[ 2 ] = new_field( 1, 10, 0, 33, 0, 0 );
  data_field[ 3 ] = new_field( 1, 10, 0, 48, 0, 0 );
  data_field[ 4 ] = new_field( 1,  5, 0, 63, 0, 0 );

  data_field[ 5 ] = (FIELD *) NULL;

  for( idx = 0; idx < 5; idx++ )
	field_opts_off( data_field[ idx ], O_AUTOSKIP );

  set_field_type( data_field[ 1 ], TYPE_REGEXP,
	  "[A-S][A-S][0-9][0-9][A-X][A-X]");
  set_field_type( data_field[ 2 ], TYPE_NUMERIC, 5, -180., 360. );
  set_field_type( data_field[ 3 ], TYPE_NUMERIC, 5, -90.,  90. );
  set_field_type( data_field[ 4 ], TYPE_NUMERIC, 1, -500., 15000. );

  /* Declare manual entries form */
  location_form = new_form( data_field );

  /* Set form window and sub-window */
  form_subwin = newwin( 1, 68, 5*num+1, 11);
  set_form_sub( location_form, form_subwin );

  post_form( location_form );

  wattrset(  form_subwin, MAGENTA );
  mvwaddstr( form_subwin, 0, 12, "Grid Loc:" );
  mvwaddstr( form_subwin, 0, 29, "Lon:" );
  mvwaddstr( form_subwin, 0, 44, "Lat:" );
  mvwaddstr( form_subwin, 0, 59, "Hgt:" );

  /* Prepare for the form's data input loop */
  form_driver( location_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );

  /* Display name entry form */
  wnoutrefresh( form_subwin );
  doupdate();

  Clear_Flag( POSITION_ENTERED_FLAG );
  Clear_Flag( HOME_LOC_ENTERED_FLAG );

  form_entry = save_file = form_done = FALSE;

  /* Process keystrokes from user */
  while( !form_done )
  {
	chr = getch();
	switch( chr )
	{
	  /* RETURN on 'Done!' field */
	  case KEY_RET : /* Return, finished input */
		form_driver( location_form, REQ_NEXT_FIELD );
		form_entry = form_done = TRUE;
		break;

	  case KEY_BACKSPACE : /* Back and delete a character */
		form_driver( location_form, REQ_PREV_CHAR );
		form_driver( location_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC : /* Delete character */
		form_driver( location_form, REQ_DEL_CHAR );
		break;

	  case KEY_LEFT : /* Back one character */
		set_field_back(current_field(location_form), A_NORMAL);
		form_driver( location_form, REQ_LEFT_FIELD );
		set_field_back( current_field(location_form), KEYSTROKE );
		break;

	  case KEY_RIGHT : /* Forward one character */
		set_field_back(current_field(location_form), A_NORMAL);
		form_driver( location_form, REQ_RIGHT_FIELD );
		set_field_back( current_field(location_form), KEYSTROKE );
		break;

	  case KEY_ESC : /* Cancel operation */
		form_done = TRUE;
		break;

	  case 's' : case 'S' : /* Save new observer to file */
		/* Ignore 's' keystroke command in location and grid fields */
		if( (field_index(current_field(location_form)) == 0) ||
			(field_index(current_field(location_form)) == 1) )
		{
		  form_driver( location_form, chr );
		  break;
		}

		form_driver( location_form, REQ_NEXT_FIELD );
		save_file = form_entry = form_done = TRUE;
		break;

	  default : /* Enter character */
		/* Record fields where data was enterd */
		if( field_index(current_field(location_form)) == 1 )
		  Set_Flag(HOME_LOC_ENTERED_FLAG);

		if( (field_index(current_field(location_form)) == 2) ||
			(field_index(current_field(location_form)) == 3) )
		  Set_Flag(POSITION_ENTERED_FLAG);

		form_driver( location_form, chr );

	} /* End of switch( chr )  */

	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !form_done ) */

  if( form_entry )
  {
	/* Enter location name and remove trailing spaces */
	strcpy( multiobs_status[num].loc_name,
		field_buffer(data_field[0], 0) );

	/* Dump trailing spaces from location name */
	idx = 11;
	while( (multiobs_status[num].loc_name[--idx] == ' ') &&	(idx != 0) );
	multiobs_status[num].loc_name[++idx] = '\0';

	/* If Grid Locator and Position data are specified, */
	/* Counter-check Grid Locator and warn on mismatch  */
	if( isFlagSet(HOME_LOC_ENTERED_FLAG) &&
		isFlagSet(POSITION_ENTERED_FLAG) )
	{
	  /* Copy Grid Locator */
	  strcpy( multiobs_status[num].grid_loc,
		  field_buffer(data_field[1], 0) );

	  /* Copy longitude/latitude */
	  multiobs_status[num].obs_geodetic.lon =
		Radians( atof(field_buffer(data_field[2], 0)) );
	  multiobs_status[num].obs_geodetic.lat =
		Radians( atof( field_buffer(data_field[3], 0)) );

	  /* Check grid locator against position */
	  lon = Degrees( multiobs_status[num].obs_geodetic.lon );
	  if( lon > 180. )
		lon -= 360.;
	  lat = Degrees(multiobs_status[num].obs_geodetic.lat);

	  Position_to_Grid_Locator( lon, lat, gloc);

	  if( strncmp( multiobs_status[num].grid_loc, gloc, strlen(gloc)) != 0 )
		strcpy( multiobs_status[num].grid_loc, "ERROR!" );
	}
	else /* If only Grid Locator is specified, calculate Position */
	  if( isFlagSet(HOME_LOC_ENTERED_FLAG) &&
		  isFlagClear(POSITION_ENTERED_FLAG) )
	  {
		/* Copy Grid Locator */
		strcpy( multiobs_status[num].grid_loc,
			field_buffer(data_field[1], 0) );

		Grid_Locator_to_Position(multiobs_status[num].grid_loc, &lon, &lat);
		if( lon < 0. )
		  lon += 360.;

		multiobs_status[num].obs_geodetic.lon = Radians( lon );
		multiobs_status[num].obs_geodetic.lat = Radians( lat );
	  }
	  else /* If only Position is specified, calculate Grid Locator */
		if( isFlagSet(POSITION_ENTERED_FLAG) &&
			isFlagClear(HOME_LOC_ENTERED_FLAG) )
		{
		  /* Copy longitude/latitude */
		  multiobs_status[num].obs_geodetic.lon =
			Radians( atof(field_buffer(data_field[2], 0)) );
		  multiobs_status[num].obs_geodetic.lat =
			Radians( atof( field_buffer(data_field[3], 0)) );

		  lon = Degrees( multiobs_status[num].obs_geodetic.lon );
		  if( lon > 180. )
			lon -= 360.;
		  lat = Degrees(multiobs_status[num].obs_geodetic.lat);

		  Position_to_Grid_Locator(lon, lat, multiobs_status[num].grid_loc);
		}

	/* Copy height */
	multiobs_status[num].obs_geodetic.hgt =
	  atof( field_buffer(data_field[4], 0) )/1.E3;

	/* Save file to satcom.obs if requested */
	if( (save_file) &&
		(strcmp(multiobs_status[num].grid_loc, "ERROR!") != 0) )
	{
	  Save_Location( multiobs_status[num] );

	  /* Report save to file */
	  attrset( KEYSTROKE );
	  curs_set( FALSE );
	  Draw_Box( 4, 27, 16, 24 );
	  mvaddstr( 17, 25, " Location data saved to: " );
	  mvaddstr( 18, 25, " ~/.satcom/satcom.obs    " );
	  refresh();
	  doupdate();
	  sleep(3);
	  Screen_Root();
	  Screen_Commands( MULTI_OBS_MODE );

	  free( obs_data );
	  Load_Observer_Data();

	} /* End of if( save_file ) */

  } /* End of if( form_entry ) */

  /* Unpost form and free fields */
  unpost_form( location_form );
  free_form(   location_form );
  for ( idx = 0; idx < 6; idx++ )
	free_field( data_field[ idx ] );

  /* Kill cursor */
  curs_set( FALSE );
  cbreak();
  halfdelay( UPDATE_DELAY );

} /* End of Post_Location_Form() */

/*------------------------------------------------------------------------*/

/*  Post_UserTime_Form()
 *
 *  Function to post a form for user-defined
 *  pass prediction staring time and date
 */

  void
Post_UserTime_Form( double *user_utc )
{
  int
	idx,  /* Index for loops etc */
	chr;  /* Input from getch()  */

  int
	form_entry, /* Data entered in form flag */
	form_done;  /* Form entries done flag    */

  struct tm cdate;

  FORM   *usertime_form;
  FIELD  *data_field[7];
  WINDOW *form_subwin;

  /* Setup Manual Position fields and form */
  data_field[ 0 ] = new_field( 1, 2, 0,  1, 0, 0 );
  data_field[ 1 ] = new_field( 1, 2, 0,  4, 0, 0 );
  data_field[ 2 ] = new_field( 1, 4, 0,  7, 0, 0 );
  data_field[ 3 ] = new_field( 1, 2, 0, 12, 0, 0 );
  data_field[ 4 ] = new_field( 1, 2, 0, 15, 0, 0 );
  data_field[ 5 ] = new_field( 1, 2, 0, 18, 0, 0 );

  data_field[ 6 ] = (FIELD *) NULL;

  for( idx = 0; idx < 6; idx++ )
	field_opts_off( data_field[ idx ], O_AUTOSKIP );

  set_field_type( data_field[ 0 ], TYPE_INTEGER, 2, 1, 31 );
  set_field_type( data_field[ 1 ], TYPE_INTEGER, 2, 1, 12 );
  set_field_type( data_field[ 2 ], TYPE_INTEGER, 4, 1900, 2056 );
  set_field_type( data_field[ 3 ], TYPE_INTEGER, 2, 0, 23 );
  set_field_type( data_field[ 4 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 5 ], TYPE_INTEGER, 2, 0, 59 );

  set_field_buffer( data_field[ 0 ], 0, "dd" );

  /* Declare manual entries form */
  usertime_form = new_form( data_field );

  /* Set form window and sub-window */
  form_subwin = newwin( 1, 20, 2, 59);
  set_form_sub( usertime_form, form_subwin );

  post_form( usertime_form );

  wattrset(  form_subwin, WHITE );
  mvwaddstr( form_subwin, 0, 0, " dd/mm/yyyy hh:mm:ss" );

  /* Prepare for the form's data input loop */
  form_driver( usertime_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );

  curs_set( TRUE );

  /* Display name entry form */
  wnoutrefresh( form_subwin );
  doupdate();

  form_done = form_entry = FALSE;

  /* Process keystrokes from user */
  while( !form_done )
  {
	chr = getch();
	switch( chr )
	{
	  /* RETURN on 'Done!' field */
	  case KEY_RET : /* Return, finished input */
		form_driver( usertime_form, REQ_NEXT_FIELD );

		/* Read form data */
		cdate.tm_mday = atoi( field_buffer(data_field[0], 0) );
		cdate.tm_mon  = atoi( field_buffer(data_field[1], 0) ) - 1;
		cdate.tm_year = atoi( field_buffer(data_field[2], 0) ) - 1900;
		cdate.tm_hour = atoi( field_buffer(data_field[3], 0) );
		cdate.tm_min  = atoi( field_buffer(data_field[4], 0) );
		cdate.tm_sec  = atoi( field_buffer(data_field[5], 0) );

		/* Check date before exiting form */
		if( Check_Date(&cdate) )
		  form_entry = form_done = TRUE;
		else
		  for( idx = 0; idx < 6; idx++ )
			set_field_back( data_field[ idx ], ERROR );

		break;

	  case KEY_BACKSPACE : /* Back and delete a character */
		form_driver( usertime_form, REQ_PREV_CHAR );
		form_driver( usertime_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC : /* Delete character */
		form_driver( usertime_form, REQ_DEL_CHAR );
		break;

	  case KEY_LEFT : /* Back one character */
		set_field_back(current_field(usertime_form), A_NORMAL);
		form_driver( usertime_form, REQ_LEFT_FIELD );
		set_field_back( current_field(usertime_form), KEYSTROKE );
		break;

	  case KEY_RIGHT : /* Forward one character */
		set_field_back(current_field(usertime_form), A_NORMAL);
		form_driver( usertime_form, REQ_RIGHT_FIELD );
		set_field_back( current_field(usertime_form), KEYSTROKE );
		break;

	  case KEY_ESC : /* Cancel operation */
		form_done = TRUE;
		break;

	  default : /* Enter character */
		form_driver( usertime_form, chr );

	} /* End of switch( chr )  */

	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !form_done ) */

  /* Read time as struct_tm format */
  if( form_entry )
  {
	*user_utc = Julian_Date( &cdate );
  }
  else
  {
	/* Default to current time */
	Calendar_Time_Now( &cdate, &cdate );
	*user_utc = Julian_Date( &cdate );
	Clear_Flag( USERTIME_PREDICT_FLAG );
  }

  /* Unpost form and free fields */
  unpost_form( usertime_form );
  free_form(   usertime_form );
  for ( idx = 0; idx < 7; idx++ )
	free_field( data_field[ idx ] );

  /* Kill cursor */
  curs_set( FALSE );

} /* End of Post_UserTime_Form() */

/*------------------------------------------------------------------------*/

/*  Post_Grid_Locator_Form()
 *
 *  Function to display the Home and Remote position and
 *  grid locator form, and bearing/distance calculator
 */

  void
Post_Grid_Locator_Form( void )
{
  int
	idx,  /* Index for loops etc */
	cols, /* Columns in form     */
	rows, /* Rows in form        */
	chr,  /* Input from getch()  */
	ideg, /* Used for calculating bearings */
	form_done;  /* Form entries done flag  */

  double
	/* 'Home' position */
	lon_a, lat_a,
	/* Remote position */
	lon_b, lat_b,
	/* Bearing and distance to remote */
	/* (Short path and long path)     */
	bearing_spath, bearing_lpath,
	distance_spath, distance_lpath;

  /* Buffer for conversions */
  char buff[23];


  /* Form for entering Manual Position parameters */
  FORM  *gridloc_form;
  FIELD *data_field[ 19 ];

  /* Form containing windows */
  WINDOW *form_win, *form_subwin;

  /* Setup Manual Position fields and form */
  data_field[ 0 ]  = new_field( 1, 1, 2, 15, 0, 0 );
  data_field[ 1 ]  = new_field( 1, 3, 2, 17, 0, 0 );
  data_field[ 2 ]  = new_field( 1, 2, 2, 21, 0, 0 );
  data_field[ 3 ]  = new_field( 1, 2, 2, 24, 0, 0 );
  data_field[ 4 ]  = new_field( 1, 1, 2, 41, 0, 0 );
  data_field[ 5 ]  = new_field( 1, 2, 2, 43, 0, 0 );
  data_field[ 6 ]  = new_field( 1, 2, 2, 46, 0, 0 );
  data_field[ 7 ]  = new_field( 1, 2, 2, 49, 0, 0 );
  data_field[ 8 ]  = new_field( 1, 6, 2, 61, 0, 0 );
  data_field[ 9 ]  = new_field( 1, 1, 6, 15, 0, 0 );
  data_field[ 10 ] = new_field( 1, 3, 6, 17, 0, 0 );
  data_field[ 11 ] = new_field( 1, 2, 6, 21, 0, 0 );
  data_field[ 12 ] = new_field( 1, 2, 6, 24, 0, 0 );
  data_field[ 13 ] = new_field( 1, 1, 6, 41, 0, 0 );
  data_field[ 14 ] = new_field( 1, 2, 6, 43, 0, 0 );
  data_field[ 15 ] = new_field( 1, 2, 6, 46, 0, 0 );
  data_field[ 16 ] = new_field( 1, 2, 6, 49, 0, 0 );
  data_field[ 17 ] = new_field( 1, 6, 6, 61, 0, 0 );

  set_field_type( data_field[  0 ], TYPE_REGEXP, "[E|W]" );
  set_field_type( data_field[  1 ], TYPE_INTEGER, 3, 0, 180 );
  set_field_type( data_field[  2 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  3 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  4 ], TYPE_REGEXP, "[N|S]" );
  set_field_type( data_field[  5 ], TYPE_INTEGER, 2, 0, 90 );
  set_field_type( data_field[  6 ], TYPE_INTEGER, 2, 0, 59);
  set_field_type( data_field[  7 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  8 ], TYPE_REGEXP,
	  "[A-S][A-S][0-9][0-9][A-X][A-X]");
  set_field_type( data_field[  9 ], TYPE_REGEXP, "[E|W]" );
  set_field_type( data_field[ 10 ], TYPE_INTEGER, 3, 0, 180 );
  set_field_type( data_field[ 11 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 12 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 13 ], TYPE_REGEXP, "[N|S]" );
  set_field_type( data_field[ 14 ], TYPE_INTEGER, 2, 0, 90 );
  set_field_type( data_field[ 15 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 16 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 17 ], TYPE_REGEXP,
	  "[A-S][A-S][0-9][0-9][A-X][A-X]");

  set_field_buffer( data_field[ 0], 0, "E" );
  set_field_buffer( data_field[ 4], 0, "N" );
  set_field_buffer( data_field[ 9], 0, "E" );
  set_field_buffer( data_field[13], 0, "N" );

  for( idx = 0; idx < 17; idx++ )
	field_opts_off( data_field[ idx ], O_AUTOSKIP );

  data_field[ 18 ] = ( FIELD * ) NULL;

  /* Declare manual entries form */
  gridloc_form = new_form( data_field );

  /* Obtain size of from */
  scale_form( gridloc_form, &rows, &cols );

  /* Create form window and sub-window */
  form_win    = newwin( rows+6, cols+2, 7, 5 );
  form_subwin = newwin( rows+4, cols  , 8, 6 );

  /* Set form window and sub-window */
  set_form_win( gridloc_form, form_win );
  set_form_sub( gridloc_form, form_subwin );

  /* Box and label form window */
  wattrset(  form_win, CYAN );
  box(       form_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( form_win, 0, 5,
	  " MAIDENHEAD GRID LOCATOR AND BEARING/DISTANCE CALCULATOR " );
  mvwaddch( form_win, 2, 0, ACS_LTEE );
  mvwaddch( form_win, 2, cols+1, ACS_RTEE );
  mvwaddch( form_win, 4, 0, ACS_LTEE );
  mvwaddch( form_win, 4, cols+1, ACS_RTEE );
  mvwaddch( form_win, 6, 0, ACS_LTEE );
  mvwaddch( form_win, 6, cols+1, ACS_RTEE );
  mvwaddch( form_win, 6, 0, ACS_LTEE );
  mvwaddch( form_win, 6, cols+1, ACS_RTEE );
  mvwaddch( form_win, 8, 0, ACS_LTEE );
  mvwaddch( form_win, 8, cols+1, ACS_RTEE );
  mvwaddch( form_win,10, 0, ACS_LTEE );
  mvwaddch( form_win,10, cols+1, ACS_RTEE );

  post_form( gridloc_form );

  wattrset(  form_subwin, GREEN );
  mvwaddstr( form_subwin, 0,  1,
	  "Enter the \'Home\' position\'s Longitude/Latitude or Grid Locator" );
  mvwaddstr( form_subwin, 4,  1,
	  "Enter the Remote position\'s Longitude/Latitude or Grid Locator" );

  /* Enter various labels in the gridloc form box */
  wattrset(  form_subwin, MAGENTA  );
  mvwaddstr( form_subwin, 2,  1, "Longitude DMS" );
  mvwaddstr( form_subwin, 2, 28, "Latitude DMS" );
  mvwaddstr( form_subwin, 2, 53, "Locator" );
  mvwaddstr( form_subwin, 6,  1, "Longitude DMS" );
  mvwaddstr( form_subwin, 6, 28, "Latitude DMS" );
  mvwaddstr( form_subwin, 6, 53, "Locator" );
  mvwaddstr( form_subwin, 8,  1, "Shortpath Bearing=    Dist=" );
  mvwaddstr( form_subwin, 8, 36, "Longpath Bearing=    Dist=" );

  wattrset( form_subwin, COMMAND );
  mvwaddstr( form_subwin, 10, 0,
	  "    Locator    Position    Bearing/Distance    Defaults       Quit " );

  wattrset(  form_subwin, KEYSTROKE );
  mvwaddstr( form_subwin, 10, 0, " <G>" );
  mvwaddstr( form_subwin, 10, 12, "<P>" );
  mvwaddstr( form_subwin, 10, 24, "<B>" );
  mvwaddstr( form_subwin, 10, 44, "<D>" );
  mvwaddstr( form_subwin, 10, 59, "<Q>" );

  wattrset( form_subwin, CYAN );
  mvwhline( form_subwin, 1, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 3, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 5, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 7, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 9, 0, ACS_HLINE, cols );

  Draw_Vline( form_subwin, 1, 14, 3 );
  Draw_Vline( form_subwin, 1, 16, 3 );
  Draw_Vline( form_subwin, 1, 20, 3 );
  Draw_Vline( form_subwin, 1, 23, 3 );
  Draw_Vline( form_subwin, 1, 26, 3 );
  Draw_Vline( form_subwin, 1, 40, 3 );
  Draw_Vline( form_subwin, 1, 42, 3 );
  Draw_Vline( form_subwin, 1, 45, 3 );
  Draw_Vline( form_subwin, 1, 48, 3 );
  Draw_Vline( form_subwin, 1, 51, 3 );
  Draw_Vline( form_subwin, 1, 60, 3 );
  Draw_Vline( form_subwin, 5, 14, 7 );
  Draw_Vline( form_subwin, 5, 16, 7 );
  Draw_Vline( form_subwin, 5, 20, 7 );
  Draw_Vline( form_subwin, 5, 23, 7 );
  Draw_Vline( form_subwin, 5, 26, 7 );
  Draw_Vline( form_subwin, 5, 40, 7 );
  Draw_Vline( form_subwin, 5, 42, 7 );
  Draw_Vline( form_subwin, 5, 45, 7 );
  Draw_Vline( form_subwin, 5, 48, 7 );
  Draw_Vline( form_subwin, 5, 51, 7 );
  Draw_Vline( form_subwin, 5, 60, 7 );
  Draw_Vline( form_subwin, 7, 33, 9 );

  /* Enter default 'home' location */
  /* Disable cursor and field highlighting */
  set_field_back( data_field[8], A_NORMAL );
  set_field_back(current_field(gridloc_form), A_NORMAL);
  curs_set( FALSE );

  strcpy( buff, HOME_POS );
  /* Set terminators in between deg, min, sec */
  /* of home long and lat. Crude but it works */
  buff[1]  = '\0';
  buff[5]  = '\0';
  buff[8]  = '\0';
  buff[11] = '\0';
  buff[13] = '\0';
  buff[16] = '\0';
  buff[19] = '\0';
  buff[22] = '\0';

  /* Display default home position */
  set_field_buffer( data_field[ 0 ], 0, &buff[0] );
  set_field_buffer( data_field[ 1 ], 0, &buff[2] );
  set_field_buffer( data_field[ 2 ], 0, &buff[6] );
  set_field_buffer( data_field[ 3 ], 0, &buff[9] );
  set_field_buffer( data_field[ 4 ], 0, &buff[12] );
  set_field_buffer( data_field[ 5 ], 0, &buff[14] );
  set_field_buffer( data_field[ 6 ], 0, &buff[17] );
  set_field_buffer( data_field[ 7 ], 0, &buff[20] );

  /* Read Home position and convert to locator */
  Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );
  Set_Flag( HOME_LOC_ENTERED_FLAG );

  /* Prepare for the form's data input loop */
  form_driver( gridloc_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );

  /* Display form */
  wnoutrefresh( form_win );
  wnoutrefresh( form_subwin );
  doupdate();

  /* Make keyboard blocking */
  cbreak();
  nodelay( stdscr, FALSE );

  form_done = FALSE;

  /* Process keystrokes from user */
  while( !form_done )
  {
	chr = getch();
	switch( chr )
	{
	  case KEY_RET :
		{
		  switch( field_index(current_field(gridloc_form)) )
		  {
			case 0 : /* Toggle East/West flags */
			  Clear_Flag( HOME_LOC_ENTERED_FLAG );
			  if( strcmp(field_buffer(data_field[0], 0), "E") )
				set_field_buffer( data_field[0], 0, "E" );
			  else
				set_field_buffer( data_field[0], 0, "W" );
			  break;

			case 4 : /* Toggle North/South flags */
			  Clear_Flag( HOME_LOC_ENTERED_FLAG );
			  if( strcmp(field_buffer(data_field[4], 0), "N") )
				set_field_buffer( data_field[4], 0, "N" );
			  else
				set_field_buffer( data_field[4], 0, "S" );
			  break;

			case 9 : /* Toggle East/West flags */
			  Clear_Flag( REM_LOC_ENTERED_FLAG );
			  if( strcmp(field_buffer(data_field[9], 0), "E") )
				set_field_buffer( data_field[9], 0, "E" );
			  else
				set_field_buffer( data_field[9], 0, "W" );
			  break;

			case 13 : /* Toggle North/South flags */
			  Clear_Flag( REM_LOC_ENTERED_FLAG );
			  if( strcmp(field_buffer(data_field[13], 0), "N") )
				set_field_buffer( data_field[13], 0, "N" );
			  else
				set_field_buffer( data_field[13], 0, "S" );
			  break;

			default: /* Move to next field */
			  curs_set( TRUE );
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  form_driver( gridloc_form, REQ_NEXT_FIELD );
			  form_driver( gridloc_form, REQ_BEG_FIELD );
			  set_field_back( current_field(gridloc_form), KEYSTROKE );
		  }
		}
		break;

	  case KEY_LEFT  : /* Back one field */
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_LEFT_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_RIGHT : /* Forward one field */
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_RIGHT_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_DOWN :	/* Down one field */
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_DOWN_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_UP : /* Up one field */
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_UP_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_BACKSPACE : /* Back and delete character */
		curs_set( TRUE );
		form_driver( gridloc_form, REQ_PREV_CHAR );
		form_driver( gridloc_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC : /* Delete character */
		curs_set( TRUE );
		form_driver( gridloc_form, REQ_DEL_CHAR );
		break;

	  case 'g' :case 'G' :/* Calculate Home locator from position */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back( current_field(gridloc_form), A_NORMAL );
		  curs_set( FALSE );

		  /* Read Home position and convert to locator */
		  Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );
		  Set_Flag( HOME_LOC_ENTERED_FLAG );
		}
		break;

	  case 'p' : case 'P' : /* Calculate Home position from locator */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Do not process an empty locator field */
		  if( isFlagSet(HOME_LOC_ENTERED_FLAG) )
		  {
			/* Disable cursor and field highlighting */
			set_field_back(current_field(gridloc_form), A_NORMAL);
			curs_set( FALSE );
			Read_Locator( data_field, &lon_a, &lat_a, HOME_ROW );
		  }
		  else
		  {
			set_field_back( data_field[8], YELLOW);
			set_field_buffer( data_field[8], 0, "??????" );
		  }
		}
		break;

	  case 'b' : case 'B' : /* Remote locator and bearing/distance */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back(current_field(gridloc_form), A_NORMAL);
		  curs_set( FALSE );

		  if( isFlagSet(HOME_LOC_ENTERED_FLAG) )
			/* Calculte Home position from grid locator */
			Read_Locator( data_field, &lon_a, &lat_a, HOME_ROW );
		  else
			/* Calculte Home position and grid locator */
			Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );

		  if( isFlagSet(REM_LOC_ENTERED_FLAG) )
			/* Calculte Remote position from grid locator */
			Read_Locator( data_field, &lon_b, &lat_b, REMOTE_ROW );
		  else
			/* Calculte Remote position and grid locator */
			Read_Position( data_field, &lon_b, &lat_b, REMOTE_ROW );

		  /* Calculate bearing and distance to Remote location */
		  Bearing_Distance( lon_a, lat_a, lon_b, lat_b,
			  &bearing_spath, &distance_spath );
		  bearing_lpath = bearing_spath + 180.0;
		  distance_lpath = eperim - distance_spath;

		  /* Display bearing and distance to remote point */
		  wattrset( form_subwin, WHITE );
		  ideg = (int)(bearing_spath+0.5);
		  if( ideg >= 360 )
			ideg -= 360;
		  mvwprintw( form_subwin, 8, 19, "%03d", ideg );
		  mvwprintw( form_subwin, 8, 28, "%5d", (int)(distance_spath+0.5) );
		  ideg = (int)(bearing_lpath+0.5);
		  if( ideg >= 360 )
			ideg -= 360;
		  mvwprintw( form_subwin, 8, 53, "%03d", ideg );
		  mvwprintw( form_subwin, 8, 62, "%5d", (int)(distance_lpath+0.5) );
		}
		break;

	  case 'd' : case 'D' : /* Enter default Home position */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back(current_field(gridloc_form), A_NORMAL);
		  curs_set( FALSE );

		  strcpy( buff, HOME_POS );
		  /* Set terminators in between deg, min, sec */
		  /* of home long and lat. Crude but it works */
		  buff[1]  = '\0';
		  buff[5]  = '\0';
		  buff[8]  = '\0';
		  buff[11] = '\0';
		  buff[13] = '\0';
		  buff[16] = '\0';
		  buff[19] = '\0';
		  buff[22] = '\0';

		  /* Display default home position */
		  set_field_buffer( data_field[ 0 ], 0, &buff[0] );
		  set_field_buffer( data_field[ 1 ], 0, &buff[2] );
		  set_field_buffer( data_field[ 2 ], 0, &buff[6] );
		  set_field_buffer( data_field[ 3 ], 0, &buff[9] );
		  set_field_buffer( data_field[ 4 ], 0, &buff[12] );
		  set_field_buffer( data_field[ 5 ], 0, &buff[14] );
		  set_field_buffer( data_field[ 6 ], 0, &buff[17] );
		  set_field_buffer( data_field[ 7 ], 0, &buff[20] );

		  /* Read Home position and convert to locator */
		  Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );
		  Set_Flag( HOME_LOC_ENTERED_FLAG );
		}
		break;

	  case 'q' : case 'Q' : case KEY_ESC : /* Exit */
		form_done = TRUE;
		break;

	  default : /* Enter character and set 'data entered' flags */
		form_driver( gridloc_form, chr );

		idx = field_index(current_field(gridloc_form));

		/* Set Home Locator Entered flag */
		/* if keystroke in Locator field */
		if( idx == 8 )
		  Set_Flag( HOME_LOC_ENTERED_FLAG );

		/* Clear Home Locator Entered flag  */
		/* if keystroke in a position field */
		if( idx >= 0 && idx <= 7 )
		  Clear_Flag( HOME_LOC_ENTERED_FLAG );

		/* Set Remote Locator Entered flag */
		/* if keystroke in Locator field   */
		if( idx == 17 )
		  Set_Flag( REM_LOC_ENTERED_FLAG );

		/* Clear Remote Locator Entered flag */
		/* if keystroke in a position field  */
		if( idx >= 9 && idx <= 16 )
		  Clear_Flag( REM_LOC_ENTERED_FLAG );

	} /* End of switch( chr )  */

	/* Refresh gridloc form */
	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !form_done ) */

  /* Unpost form and free fields */
  unpost_form( gridloc_form );
  free_form( gridloc_form );
  for ( idx=0; idx < 19; idx++ ) free_field( data_field[ idx ] );

  /* Delete form windows */
  delwin( form_subwin );
  delwin( form_win );

  /* Return to half-delayed mode */
  halfdelay( UPDATE_DELAY );

} /* End of Post_Grid_Locator_Form() */

/*------------------------------------------------------------------------*/

/*  Read_Position()
 *
 *  Reads position fields (Lon/Lat) and calculates Lon/Lat
 */

  void
Read_Position( FIELD *data_field[], double *lon, double *lat, int row )
{
  int ideg, imin, isec;

  char locator[7];

  /* Calculte Home position and grid locator */
  ideg = atoi( field_buffer(data_field[1+row], 0) );
  imin = atoi( field_buffer(data_field[2+row], 0) );
  isec = atoi( field_buffer(data_field[3+row], 0) );
  DMS_to_Degrees( ideg, imin, isec, lon );
  if( strcmp(field_buffer(data_field[0+row],0), "W")==0 )
	*lon = - *lon;

  ideg = atoi( field_buffer(data_field[5+row], 0) );
  imin = atoi( field_buffer(data_field[6+row], 0) );
  isec = atoi( field_buffer(data_field[7+row], 0) );
  DMS_to_Degrees( ideg, imin, isec, lat );
  if( strcmp(field_buffer(data_field[4+row],0), "S")==0 )
	*lat = - *lat;

  /* Calculate grid locator */
  Position_to_Grid_Locator( *lon, *lat, locator );
  set_field_buffer( data_field[8+row], 0, locator );

} /* End of Read_Position() */

/*------------------------------------------------------------------------*/

/*  Read_Locator()
 *
 *  Reads a Locator field and calculates Lon/Lat
 */

  void
Read_Locator( FIELD *data_field[], double *lon, double *lat, int row )
{
  int ideg, imin, isec;

  double lng, ltd;

  char buff[4];

  /* Read grid locator and calculate position */
  Grid_Locator_to_Position( field_buffer(data_field[8+row], 0), lon, lat );

  /* Negative longitude shown as West */
  if( *lon < 0.0 )
  {
	lng = - *lon;
	set_field_buffer( data_field[0+row], 0, "W" );
  }
  else
  {
	lng = *lon;
	set_field_buffer( data_field[0+row], 0, "E" );
  }

  Degrees_to_DMS( lng, &ideg, &imin, &isec );
  snprintf( buff, 4, "%03d", ideg );
  set_field_buffer( data_field[1+row], 0, buff );
  snprintf( buff, 3, "%02d", imin );
  set_field_buffer( data_field[2+row], 0, buff );
  snprintf( buff, 3, "%02d", isec );
  set_field_buffer( data_field[3+row], 0, buff );

  /* Negative latitude shown as South */
  if( *lat < 0.0 )
  {
	ltd = - *lat;
	set_field_buffer( data_field[4+row], 0, "S" );
  }
  else
  {
	ltd = *lat;
	set_field_buffer( data_field[4+row], 0, "N" );
  }

  Degrees_to_DMS( ltd, &ideg, &imin, &isec );
  snprintf( buff, 3, "%02d", ideg );
  set_field_buffer( data_field[5+row], 0, buff );
  snprintf( buff, 3, "%02d", imin );
  set_field_buffer( data_field[6+row], 0, buff );
  snprintf( buff, 3, "%02d", isec );
  set_field_buffer( data_field[7+row], 0, buff );

} /* End of Read_Locator() */

/*------------------------------------------------------------------------*/

/*  Draw_Box()
 *
 *  Function to draw a box of given size and position in the root window
 */

  void
Draw_Box( int rows, int cols, int top, int left )
{
  mvvline( top + 1, left, ACS_VLINE, rows - 2 );
  mvvline( top + 1, left + cols - 1, ACS_VLINE, rows - 2 );
  mvhline( top, left + 1, ACS_HLINE, cols - 2 );
  mvhline( top + rows - 1, left + 1, ACS_HLINE, cols - 2 );
  mvaddch( top, left, ACS_ULCORNER );
  mvaddch( top, left + cols - 1, ACS_URCORNER );
  mvaddch( top + rows - 1, left, ACS_LLCORNER );
  mvaddch( top + rows - 1, left + cols - 1, ACS_LRCORNER );

} /* End of Draw_Box() */

/*------------------------------------------------------------------------*/

/*  Draw_Hline()
 *
 *  Draws a horizontal line in a window including Tees at ends
 */

  void
Draw_Hline( WINDOW *win, int top, int left, int right )
{
  mvwhline( win, top, left+1, ACS_HLINE, (right - left - 1) );
  mvwaddch( win, top, left,  ACS_LTEE );
  mvwaddch( win, top, right, ACS_RTEE );

} /* End of Draw_Hline() */

/*------------------------------------------------------------------------*/

/*  Draw_Vline()
 *
 *  Draws a vertical line in a window including Tees at ends
 */

  void
Draw_Vline( WINDOW *win, int top, int left, int bottom )
{
  mvwvline( win, top+1, left, ACS_VLINE, (bottom - top -1) );
  mvwaddch( win, top, left, ACS_TTEE );
  mvwaddch( win, bottom, left, ACS_BTEE );

} /* End of Draw_Vline() */

/*------------------------------------------------------------------------*/
