/*  mode.c
 *
 *  Functions used for running satcom in different 'modes'
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "satcom.h"

/* Buffer for reading satellite transponder */
/* modes from the satellite database file   */
satellite_mode_t *sat_mode = NULL;

/* All-satellite TLE data */
extern tle_t *tle_data;

/* Number of entries in tle_data */
extern int num_tles;

/* TLE set currently in use */
extern int idx_tle;

/* Transceiver status structure */
tcvr_status_t tcvr_status;

/* Number of transponder modes found in database */
int num_modes;

/* All-observer location data */
extern observer_status_t *obs_data;

/* Number of entries in obs_data */
extern int num_obs;

/* Observer set currently in use */
extern int idx_obs;

/* Strings for xplanet's file names */
extern char
  xplanet_markerfile[50],
  xplanet_greatarcfile[50],
  xplanet_configfile[50],
  xplanet_command[200];


/*  Main_Screen_Mode()
 *
 *  Displays the main detailed menu startup screen of satcom
 */

  void
Main_Screen_Mode( void )
{
  /* Used for inputting characters with getch() */
  int chr=0;

  /* Observer's status data */
  observer_status_t obs_status;

  /* Satellite status data */
  satellite_status_t sat_status;

  /* Two-line Orbital Elements */
  /* of the selected satellite */
  tle_t sat_tle ;

  double julian_utc; /* Julian UTC */

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */


  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  /* Load the observer location data base from satcom.obs */
  Load_Observer_Data();

  /* Load 'Home' oserver form database */
  Load_Observer_Set( "Home", &obs_status);

  /* Load TLE database from satcom.tle */
  Load_Database_Tle_Data();

  /* Input the default TLE set from TLE file  */
  Load_Tle_Set( SAT_NAME, &sat_tle );

  /* Pre-process satellite data */
  New_Satellite( &sat_tle, &sat_status, &obs_status );

  /* Display main screen */
  Screen_Root();
  Screen_Main();
  Screen_Commands( MAIN_SCREEN_MODE );

  /*
   * This is the main loop of the program. It reads keystrokes
   * and directs program flow accordingly. Quits on 'Q' keystroke
   */

  do
  {
	switch( chr )
	{
	  case 's' : case 'S' : /* Single satellite/observer mode */
		Single_Satellite_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'm' : case 'M' : /* Multi-satellite/single-observer mode */
		Multi_Satellite_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'l' : case 'L' : /* Multi-observer/single-satellite mode */
		Multi_Observer_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'p' : case 'P' : /* Pass prediction mode */
		Pass_Prediction_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'i' : case 'I' : /* Illumination prediction mode */
		Illumination_Prediction_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'c' : case 'C' : /* Manual Commands mode */
		Manual_Command_Mode( &sat_tle, &obs_status, &sat_status );
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'o' : case 'O' : /* Display Offset Calibration window */
		Screen_Offset_Calibration();
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'f' : case 'F' : /* Display Fullscale Calibration window */
		Screen_Fullscale_Calibration();
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );
		break;

	  case 'g' : case 'G' : /* Maidenhead Grid Locator calculator */
		Screen_Root();
		Screen_Observer_Status( &obs_status );
		Post_Grid_Locator_Form();
		Screen_Main();
		Screen_Commands( MAIN_SCREEN_MODE );

	} /* End of switch( chr ) */

	/* Get UTC calendar and convert to Julian */
	Calendar_Time_Now( &obs_status.utc, &obs_status.localtm );
	julian_utc = Julian_Date( &obs_status.utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( julian_utc, &solar_kinetics );

	/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	Calculate_Satellite_Status( julian_utc, &sat_tle,
		&solar_kinetics, &obs_kinetics,
		&obs_status, &sat_status );

	/* Calculate lunar azimuth and elevation */
	Moon_Position( julian_utc, &obs_status, &obs_kinetics );

	/* Display observer status */
	Screen_Observer_Status( &obs_status );

  } /* End of main do loop */
  while( ((chr = getch()) != 'q') && (chr != 'Q') ); /* Quit on 'Q' */

} /* End of Main_Screen_Mode() */

/*------------------------------------------------------------------*/

/*  Single_Satellite_Mode()
 *
 *  Does all the calculations needed to fill the
 *  satellite and observer status structures and
 *  to display single observer/satellite situation
 */

void
Single_Satellite_Mode(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  int
	azim_dir,  /* Current azimuth direction   */
	elev_dir,  /* Current elevation direction */
	mode_idx,  /* Current transponder mode    */
	chr=0;     /* Keyboard input character    */

  double
	julian_utc, /* Julian UTC */
	azim = 0.,  /* Azimuth position demand    */
	elev = 0.;  /* Elevation position demand  */

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */

  Screen_Root();
  Screen_Commands( SINGLE_SAT_MODE );

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  mode_idx = 0;

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case 'r' : case 'R' : /* Toggle antenna tracking request */
		if( isFlagClear(TRACKING_ENABLED_FLAG) )
		  Enable_Tracking();
		else
		{
		  /* Read current position of rotors */
		  Rotor_Read_Direction( &azim_dir, &elev_dir );
		  /* Make current position the new demand */
		  if( isFlagClear(ROTOR_READ_FAIL_FLAG) )
			Manual_Position_Rotors( azim_dir, elev_dir );
		  Disable_Tracking();
		  Screen_Root();
		  Screen_Commands( SINGLE_SAT_MODE );
		}
		break;

	  case 'e' : case 'E' : /* Toggle earthtrack (xplanet display) */
		if( isFlagClear(EARTHTRACK_FLAG) )
		{
		  Config_Xplanet();
		  Set_Flag( EARTHTRACK_FLAG );
		}
		else
		{
		  Clear_Flag( EARTHTRACK_FLAG );
		  system( "pkill xplanet" );
		  Clear_Flag( XPLANET_RUN );
		  /* Delete xplanet's files */
		  unlink( xplanet_greatarcfile );
		  unlink( xplanet_markerfile );
		  unlink( xplanet_configfile );
		}
		break;

	  case 'c' : case 'C' : /* Toggle CAT control */
		if( isFlagClear(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Enable_CAT();
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		else
		{
		  Screen_Root();
		  Screen_Commands( SINGLE_SAT_MODE );
		  Disable_CAT();
		}
		break;

	  case 'u' : case 'U' : /* Update uplink/dnlink offset */
		tcvr_status.upl_dnl_offset  = tcvr_status.uplink_freq;
		tcvr_status.upl_dnl_offset += ( tcvr_status.flags & PBAND_INV ?
			+tcvr_status.dnlink_freq :
			-tcvr_status.dnlink_freq );
		Set_Flag( UPLNK_DNLNK_TRACK_FLAG );
		break;

	  case 'w' : case 'W' : /* Move to next transponder mode */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  if( ++mode_idx >= num_modes )
			mode_idx = 0;
		  Initialize_Tcvr( mode_idx );
		}
		break;

	  case 't' : case 'T' : /* Toggle uplink/downlink tracking flag */
		Toggle_Flag( UPLNK_DNLNK_TRACK_FLAG );
		break;

	  case 'd' : case 'D' : /* Toggle Doppler Shift correction flag */
		Toggle_Flag( DOPPLER_DISABLED_FLAG );
		break;

	  case 'n' : case 'N' : /* Request the new satellite name form */
		Request_New_Satellite( sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case KEY_RIGHT : /* Request next satellite in TLE file */
		Find_New_Satellite( NEXT, sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case KEY_LEFT : /* Request previous satellite in TLE file */
		Find_New_Satellite( PREVIOUS, sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case KEY_UP : /* Request first satellite in TLE file */
		Find_New_Satellite( FIRST, sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case KEY_DOWN : /* Request last satellite in TLE file */
		Find_New_Satellite( LAST, sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case KEY_IC : /* Request default satellite */
		Find_New_Satellite( DEFAULT, sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  mode_idx = 0;
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		break;

	  case 'l' : case 'L' : /* Multi observer mode */
		Multi_Observer_Mode( sat_tle, obs_status, sat_status );
		Screen_Root();
		Screen_Commands( SINGLE_SAT_MODE );
		break;

	  case 'p' : case 'P' : /* Pass prediction mode */
		Pass_Prediction_Mode( sat_tle, obs_status, sat_status );
		Screen_Root();
		Screen_Commands( SINGLE_SAT_MODE );
		break;

	  case 'm' : case 'M' : /* Multi-satellite/single-observer mode */
		Multi_Satellite_Mode( sat_tle, obs_status, sat_status );
		Screen_Root();
		Screen_Commands( SINGLE_SAT_MODE );
		sat_status->elev = 0.; /* Enable AOS/LOS recalculation */

	} /* End of switch( chr ) */

	/* Get UTC calendar and convert to Julian */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
	julian_utc = Julian_Date( &obs_status->utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( julian_utc, &solar_kinetics );

	/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	Calculate_Satellite_Status( julian_utc, sat_tle,
		&solar_kinetics, &obs_kinetics,
		obs_status, sat_status );

	/* Calculate lunar azimuth and elevation */
	Moon_Position( julian_utc, obs_status, &obs_kinetics );

	/* Display satellite status */
	Screen_Satellite_Status( sat_status );

	/* Display observer status */
	Screen_Observer_Status( obs_status );

	/* Display satellite status with xplanet */
	if( isFlagSet(EARTHTRACK_FLAG) )
	  Display_Satellite( sat_status, obs_status );

	if( isFlagSet(ROTORS_ENABLED_FLAG) )
	{
	  /* If tracking enabled, refresh Az/El demand */
	  if( isFlagSet(TRACKING_ENABLED_FLAG) )
	  {
		azim = sat_status->azim;
		elev = sat_status->elev;

		if( isSatFlagSet( sat_status, NORTH_CROSSING_FLAG) )
		  Set_Flag( REVERSE_TRACK_FLAG );
		else
		  Clear_Flag( REVERSE_TRACK_FLAG );
	  }
	  else /* Use current directions */
	  {
		azim = azim_dir;
		elev = elev_dir;
	  }

	  /* Control and display rotor position and status */
	  Screen_Rotor_Control( azim, elev, sat_status->resol, DISPLAY );

	} /* End of if( isFlagSet(ROTORS_ENABLED_FLAG) ) */

	/* Activate transceiver CAT control if enabled */
	if( isFlagSet(CAT_ENABLED_FLAG) )
	{
	  /* Control FT-847 parameters */
	  CAT_Control( sat_status );

	  /* Refresh CAT control window */
	  Screen_CAT_Control( &tcvr_status );
	}

  }/* End of do */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );

  /* Make current position the new demand */
  if( isFlagSet(ROTORS_ENABLED_FLAG) )
  {
	Rotor_Read_Direction( &azim_dir, &elev_dir );
	Manual_Position_Rotors( azim_dir, elev_dir );
  }

  /* Return transceiver to original status */
  if( isFlagSet(CAT_ENABLED_FLAG) )
	Disable_CAT();

}/* End of Single_Satellite_Mode() */

/*------------------------------------------------------------------*/

/*  Multi_Satellite_Mode()
 *
 *  Does all the calculations needed to fill the
 *  multi-satellite and observer status structures and
 *  to display single-observer/multi-satellite situation
 */

void
Multi_Satellite_Mode(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  double julian_utc; /* Julian UTC */

  int
	stats,    /* Data needed for the multisat display */
	num_dspl, /* Number of satellites to display      */
	frm_dspl, /* Satellite rank to start display from */
	to_dspl,  /* Satellite rank to end display at     */
	vis_sats, /* Number of visible satellites */
	icp,      /* Index for comparison loops   */
	idx,      /* Index for loops etc          */
	chr = 0;  /* Keyboard input character     */

  /* Satellite selected for highlighting  */
  static int sel_hlght = 0;

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */

  /* Multisatellite tracking status */
  satellite_status_t *multisat_status;

  /* Allocate memory for multisat_status */
  multisat_status = malloc( sizeof(satellite_status_t) * num_tles );
  if( multisat_status == NULL )
	Abort_On_Error( -10 );

  /* Number of satellites to be displayed */
  num_dspl = ( num_tles < 30 ? num_tles : 30 );

  /* Pre-process satellite data */
  for( idx = 0; idx < num_tles; idx++ )
	New_Satellite( &tle_data[idx], &multisat_status[idx], obs_status );

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  Screen_Root();
  Screen_Commands( MULTI_SAT_MODE );

  frm_dspl = 0;
  to_dspl  = num_tles - 1;
  if( to_dspl > 29 )
	to_dspl = 29;

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case KEY_NPAGE : /* Request next 30 satellites in rank */
		sel_hlght = 0;

		to_dspl  += 30;
		if( to_dspl >= num_tles )
		{
		  to_dspl  = num_tles - 1;
		  frm_dspl = to_dspl - 29;
		  if( frm_dspl < 0 )
			frm_dspl = 0;
		}
		else
		  frm_dspl += 30;
		break;

	  case KEY_PPAGE : /* Request previous 30 satellites in rank */
		sel_hlght = 0;

		frm_dspl  -= 30;
		if( frm_dspl < 0 )
		{
		  frm_dspl = 0;
		  to_dspl  = frm_dspl + 29;
		  if( to_dspl >= num_tles )
			to_dspl = num_tles - 1;
		}
		else
		  to_dspl -= 30;
		break;

	  case KEY_HOME : /* Request top 30 satellites in rank */
		sel_hlght = 0;
		frm_dspl  = 0;
		to_dspl   = num_tles - 1;
		if( to_dspl > 29 )
		  to_dspl = 29;
		break;

	  case KEY_END : /* Request last 30 satellites in rank */
		sel_hlght = 0;
		to_dspl   = num_tles - 1;
		frm_dspl  = to_dspl - 29;
		if( frm_dspl < 0 )
		  frm_dspl = 0;
		break;

	  case KEY_IC : /* Request default satellite */
		Load_Tle_Set( SAT_NAME, sat_tle );
		/* Pre-process satellite data */
		New_Satellite( sat_tle, sat_status, obs_status );
		break;

	  case KEY_LEFT :	/* Scroll left in multisat display */
		sel_hlght -= 15;
		sel_hlght = ( sel_hlght < 0 ? sel_hlght+15 : sel_hlght );
		break;

	  case KEY_RIGHT : /* Scroll right in multisat display */
		sel_hlght += 15;
		sel_hlght = ( sel_hlght >= 30 ? sel_hlght-15 : sel_hlght );
		sel_hlght = ( sel_hlght >= num_dspl-1 ? num_dspl-1 : sel_hlght );
		break;

	  case KEY_UP : /* Scroll up in multisat display */
		sel_hlght = ( --sel_hlght < 0 ? sel_hlght+num_dspl : sel_hlght );
		break;

	  case KEY_DOWN : /* Scroll down in multisat display */
		sel_hlght = (++sel_hlght >= num_dspl ? sel_hlght-num_dspl : sel_hlght);
		break;

	  case KEY_RET : /* Select highlighted satellite  */
		Load_Tle_Set( tle_data[stats>>20].name, sat_tle );

		/* Pre-process satellite data */
		New_Satellite( sat_tle, sat_status, obs_status );

		/* Re-initialize transceiver */
		if( isFlagSet(CAT_ENABLED_FLAG) )
		{
		  Read_Transponder_Data( sat_status->name, sat_mode );
		  Initialize_Tcvr( DEFAULT_MODE );
		}
		free( multisat_status );
		return;

	  case 'p' : case 'P' : /* Load previous observer data from satcom.obs */
		if( --idx_obs < 0 )
		  idx_obs = num_obs - 1;
		*obs_status = obs_data[idx_obs];

		/* Pre-process satellite data */
		for( idx = 0; idx < num_tles; idx++ )
		  New_Satellite( &tle_data[idx],
			  &multisat_status[idx],
			  obs_status );

		/* Process new observer */
		New_Observer( sat_tle,
			sat_status,
			obs_status );
		break;

	  case 'n' : case 'N' : /* Load next observer data set from satcom.obs */
		if( ++idx_obs == num_obs )
		  idx_obs = 0;
		*obs_status = obs_data[idx_obs];

		/* Pre-process satellite data */
		for( idx = 0; idx < num_tles; idx++ )
		  New_Satellite( &tle_data[idx],
			  &multisat_status[idx],
			  obs_status );

		/* Process new observer */
		New_Observer( sat_tle,
			sat_status,
			obs_status );
		break;

	  case 'h' : case 'H' : /* Load 'Home observer data from satcom.obs */
		Load_Observer_Set( "Home", obs_status );

		/* Pre-process satellite data */
		for( idx = 0; idx < num_tles; idx++ )
		  New_Satellite( &tle_data[idx],
			  &multisat_status[idx],
			  obs_status );

		/* Process new observer */
		New_Observer( sat_tle,
			sat_status,
			obs_status );

	} /* End of switch( chr ) */

	/* Get UTC calendar and convert to Julian */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
	julian_utc = Julian_Date( &obs_status->utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position(julian_utc, &solar_kinetics);

	vis_sats = 0;

	/* Calculate multisatellite status */
	for( idx = 0; idx < num_tles; idx++ )
	{
	  /* Clear SGP4/SDP4 control flags */
	  Clear_Flag( SGP_SDP_FLAGS );

	  Calculate_Satellite_Status( julian_utc, &tle_data[idx],
		  &solar_kinetics, &obs_kinetics,
		  obs_status, &multisat_status[idx] );

	  /* Count visible (elev>0) satellites */
	  if( multisat_status[idx].elev > 0. )
		vis_sats++;

	  /* Clear satellite rank register */
	  Clear_SatFlag( &multisat_status[idx], SAT_RANK_FLAGS );

	} /* End of for( idx = 0; idx <  num_tles; idx++ ) */


	/* Rank satellites: Top rank, visible satellites with furthest LOS */
	/* time. Lowest rank, invisible satellites with furthest AOS time. */
	/* Rank is entered in satellite status flags lowest byte.          */
	for( idx = 0; idx < num_tles; idx++ )
	{
	  /* If sat below horizon, init. rank to number of visible sats */
	  if( multisat_status[idx].elev <= 0. )
		multisat_status[idx].flags += vis_sats;

	  for( icp = 0; icp < num_tles; icp++ )
	  {
		/* If both sats visible, rank according to   */
		/* > LOS and according to accessibility flag */
		if( (multisat_status[icp].elev > 0.) &&
			(multisat_status[idx].elev > 0.) )
		{
		  if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagSet(&multisat_status[icp],NO_AOS_LOS_FLAG)) &&
			  (idx > icp) )
			multisat_status[idx].flags++;

		  if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagClear(&multisat_status[icp],NO_AOS_LOS_FLAG)) )
			multisat_status[idx].flags++;

		  if( (isSatFlagClear(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagClear(&multisat_status[icp],NO_AOS_LOS_FLAG)) )
		  {
			if(multisat_status[idx].los < multisat_status[icp].los)
			  multisat_status[idx].flags++;
			else
			  if( (multisat_status[idx].los == multisat_status[icp].los) &&
				  (idx > icp) )
				multisat_status[idx].flags++;
		  }

		} /* End of if( (multisat_status[icp].elev > 0.) && ... */


		/* If both sats invisible, rank according to */
		/* < AOS and according to accessibility flag */
		if( (multisat_status[icp].elev <= 0.) &&
			(multisat_status[idx].elev <= 0.) )
		{
		  if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagSet(&multisat_status[icp],NO_AOS_LOS_FLAG)) &&
			  (idx > icp) )
			multisat_status[idx].flags++;

		  if( (isSatFlagSet(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagClear(&multisat_status[icp],NO_AOS_LOS_FLAG)) )
			multisat_status[idx].flags++;

		  if( (isSatFlagClear(&multisat_status[idx],NO_AOS_LOS_FLAG)) &&
			  (isSatFlagClear(&multisat_status[icp],NO_AOS_LOS_FLAG)) )
		  {
			if(multisat_status[idx].aos > multisat_status[icp].aos)
			  multisat_status[idx].flags++;
			else
			  if( (multisat_status[idx].aos == multisat_status[icp].aos) &&
				  (idx > icp) )
				multisat_status[idx].flags++;
		  }

		} /* End of if( (multisat_status[icp].elev <= 0.) && ... */

	  } /* End of for( icp = 0; icp < num_tles; icp++ ) */

	} /* End of for( idx = 0; idx < num_tles; idx++ ) */

	/* Calculate single satellite status */
	Clear_Flag( SGP_SDP_FLAGS ); /* Clear SGP4/SDP4 control flags */
	Calculate_Satellite_Status( julian_utc, sat_tle,
		&solar_kinetics, &obs_kinetics,
		obs_status, sat_status );

	/* Calculate lunar azimuth and elevation */
	Moon_Position( julian_utc, obs_status, &obs_kinetics );

	/* Display multi-satellite status. Highlight selection */
	/* and number of satellites entered in 'stats' integer */
	stats = (sel_hlght << 20) | (frm_dspl << 10) | to_dspl;
	Screen_Multisatellite_Status( &stats, obs_status,
		sat_status, multisat_status );

	/* Control rotor position if enabled */
	if( isFlagSet(ROTORS_ENABLED_FLAG | TRACKING_ENABLED_FLAG) )
	{
	  if( isSatFlagSet( sat_status, NORTH_CROSSING_FLAG) )
		Set_Flag( REVERSE_TRACK_FLAG );
	  else
		Clear_Flag( REVERSE_TRACK_FLAG );

	  Screen_Rotor_Control( sat_status->azim, sat_status->elev,
		  sat_status->resol, NO_DISPLAY );
	}

	/* Activate transceiver CAT control if enabled */
	if( isFlagSet(CAT_ENABLED_FLAG) )
	  /* Control FT-847 parameters */
	  CAT_Control( sat_status );

  } /* End of main do loop */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );

  free( multisat_status );

} /* End of Multi_Satellite_Mode() */

/*------------------------------------------------------------------*/

/*  Multi_Observer_Mode()
 *
 *  Single-satellite/Multi-observer tracking mode
 */

void
Multi_Observer_Mode(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  int
	idx,       /* Index for loops etc */
	chr=0,     /* Used by getch for keyboard entry */
	num_loc;   /* Number of locations after 'Home' */

  double julian_utc; /* Julia UTC */

  /* Multi-observer status array */
  observer_status_t multiobs_status[4];

  /* Multi-observer satellite status array */
  satellite_status_t mxsat_status[4];

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  Screen_Commands( MULTI_OBS_MODE );

  /* Initialize the 'Home' elements */
  multiobs_status[0] = *obs_status;
  mxsat_status[0]    = *sat_status;

  num_loc = idx_obs = 0;

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case KEY_NPAGE : /* Add a new observer location set */
		if( ++num_loc > 3 )
		  num_loc = 3;
		else
		{
		  /* Load next observer data set from satcom.obs */
		  if( ++idx_obs == num_obs )
			idx_obs = 0;
		  multiobs_status[num_loc] = obs_data[idx_obs];

		  /* Process new observer */
		  New_Observer( sat_tle,
			  &mxsat_status[num_loc],
			  &multiobs_status[num_loc] );
		}
		break;

	  case KEY_PPAGE : /* Remove last observer location set */
		num_loc = ( --num_loc < 0 ? 0 : num_loc );

		/* Move to previous observer data set in satcom.obs */
		idx_obs = ( --idx_obs < 0 ? num_obs - 1 : idx_obs );
		break;

	  case KEY_HOME : /* Load previous observer data set from satcom.obs */
		/* Move to previous observer data set in satcom.obs */
		idx_obs = ( --idx_obs < 0 ? num_obs - 1 : idx_obs );

		multiobs_status[num_loc] = obs_data[idx_obs];

		/* Process new observer */
		New_Observer( sat_tle,
			&mxsat_status[num_loc],
			&multiobs_status[num_loc] );
		break;

	  case KEY_END : /* Load next observer data set from satcom.obs */
		idx_obs = ( ++idx_obs == num_obs ?  0 : idx_obs );
		multiobs_status[num_loc] = obs_data[idx_obs];

		/* Process new observer */
		New_Observer( sat_tle,
			&mxsat_status[num_loc],
			&multiobs_status[num_loc] );
		break;

	  case 'h' : case 'H' : /* Load 'Home observer data from satcom.obs */
		Load_Observer_Set( "Home", &multiobs_status[num_loc] );

		/* Process new observer */
		New_Observer( sat_tle,
			&mxsat_status[num_loc],
			&multiobs_status[num_loc] );
		break;

	  case KEY_RIGHT : /* Request next satellite in TLE file */
		Find_New_Satellite( NEXT, sat_tle, sat_status, obs_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );
		break;

	  case KEY_LEFT : /* Request previous satellite in TLE file */
		Find_New_Satellite( PREVIOUS, sat_tle, sat_status, obs_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );
		break;

	  case KEY_UP : /* Request first satellite in TLE file */
		Find_New_Satellite( FIRST, sat_tle, sat_status, obs_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );
		break;

	  case KEY_DOWN : /* Request last satellite in TLE file */
		Find_New_Satellite( LAST, sat_tle, sat_status, obs_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );
		break;

	  case KEY_IC : /* Request default satellite */
		Find_New_Satellite( DEFAULT, sat_tle, sat_status, obs_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );
		break;

	  case 'm' : case 'M' : /* Go to multisatellite tracking mode */
		Multi_Satellite_Mode( sat_tle, obs_status, sat_status );

		/* Process observers for new satellite */
		for( idx = 0; idx <= num_loc; idx++ )
		  New_Observer( sat_tle,
			  &mxsat_status[idx],
			  &multiobs_status[idx] );

		Screen_Root();
		Screen_Commands( MULTI_OBS_MODE );
		break;

	  case 'u' : case 'U' : /* Request keyboard entry form */
		Post_Location_Form( num_loc, multiobs_status );
		/* Process new observer */
		New_Observer( sat_tle,
			&mxsat_status[num_loc],
			&multiobs_status[num_loc] );

	} /* End of switch( chr ) */

	/* Get UTC calendar and convert to Julian */
	for( idx = 0; idx <= num_loc; idx++ )
	  Calendar_Time_Now( &multiobs_status[idx].utc,
		  &multiobs_status[idx].localtm );

	julian_utc = Julian_Date( &multiobs_status[0].utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( julian_utc, &solar_kinetics );

	for( idx = 0; idx <= num_loc; idx++ )
	{
	  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	  Calculate_Satellite_Status( julian_utc, sat_tle,
		  &solar_kinetics,
		  &obs_kinetics,
		  &multiobs_status[idx],
		  &mxsat_status[idx] );

	  /* Calculate lunar azimuth and elevation */
	  Moon_Position(julian_utc, &multiobs_status[idx], &obs_kinetics);
	}

	Screen_Multiobserver_Status( num_loc, multiobs_status, mxsat_status );

	/* Control rotor position */
	if( isFlagSet(ROTORS_ENABLED_FLAG | TRACKING_ENABLED_FLAG) )
	{
	  if( isSatFlagSet( sat_status, NORTH_CROSSING_FLAG) )
		Set_Flag( REVERSE_TRACK_FLAG );
	  else
		Clear_Flag( REVERSE_TRACK_FLAG );

	  Screen_Rotor_Control( mxsat_status[0].azim, mxsat_status[0].elev,
		  sat_status->resol, NO_DISPLAY );
	}

	/* Activate transceiver CAT control if enabled */
	if( isFlagSet(CAT_ENABLED_FLAG) )
	  /* Control FT-847 parameters */
	  CAT_Control( sat_status );

  }  /* End of main do loop. (Quit on Q or ESC) */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );

} /* End of Multi_Observer_Mode() */

/*------------------------------------------------------------------*/

/*  Pass_Prediction_Mode()
 *
 *  Predictions of satellite passes
 */

void
Pass_Prediction_Mode(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  int
	chr=0, /* Used by getch for keyboard entry */
	idx,   /* Index for loops etc */
	num;   /* Number of entries to pass_data */

  double
	aos_utc,    /* AOS time for start of prediction */
	los_utc,    /* LOS time for start of prediction */
	old_los,    /* Stored LOS time, needed for flow control */
	time_now,   /* Real time julian UTC            */
	user_utc,   /* User-input start of predictions */
	from_utc,   /* Start time of pass predictions  */
	to_utc,     /* Ending time of pass predictions */
	step_utc,   /* Time step for pass predictions  */
	old_utc,    /* UTC used in previous prediction */
	julian_utc; /* Julian UTC in calculations */

  /* Observer and solar position/velocity */
  kinetics_t
	solar_kinetics,
	obs_kinetics;

  /* Temporary observer and satellite status used in predictions */
  satellite_status_t temp_sat_status;
  observer_status_t  temp_obs_status;
  satellite_status_t old_sat_status;

  /* Pass prediction data */
  pass_data_t pass_data;

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  Screen_Commands( PASS_PREDICTION_MODE );

  /* Initialize for pass predictions */
  temp_sat_status = *sat_status;
  temp_obs_status = *obs_status;
  Clear_Flag( VISIBLE_PREDICT_FLAG  );
  Clear_Flag( USERTIME_PREDICT_FLAG );

  /* Initialise to avoid error messages from gcc */
  time_now = old_los = aos_utc = los_utc = step_utc = 0.;
  julian_utc = old_utc = user_utc = from_utc = to_utc = 0.;

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case 'a' : case 'A' : /* Accessible pass prediction */
		Clear_Flag( VISIBLE_PREDICT_FLAG  );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		old_los = 0.;
		break;

	  case 'v' : case 'V' : /* Visible pass prediction */
		Set_Flag(   VISIBLE_PREDICT_FLAG  );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case 'u' : case 'U' : /* User-specified start time */
		Set_Flag( USERTIME_PREDICT_FLAG );
		Post_UserTime_Form( &user_utc );
		from_utc = user_utc;
		break;

	  case 'r' : case 'R' : /* Real-time pass predictions */
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case 'f' : case 'F' : /* Forward to next pass prediction */
		if( isFlagClear(USERTIME_PREDICT_FLAG) )
		{
		  Set_Flag( USERTIME_PREDICT_FLAG );
		  /* Save start of user-time search */
		  user_utc = time_now;
		}

		/* Move ahead of pass prediction end time by ~5 min */
		from_utc = to_utc + 0.035;
		break;

	  case 's' : case 'S' : /* Back to user time for prediction */
		from_utc = user_utc;
		break;

	  case KEY_RIGHT : /* Request next satellite in TLE file */
		Find_New_Satellite( NEXT, sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_LEFT : /* Request previous satellite in TLE file */
		Find_New_Satellite( PREVIOUS, sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_UP :	/* Request first satellite in TLE file */
		Find_New_Satellite( FIRST, sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_DOWN : /* Request last satellite in TLE file */
		Find_New_Satellite( LAST, sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_IC : /* Request default satellite */
		Find_New_Satellite( DEFAULT, sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_NPAGE : /* Load next observer data from satcom.obs */
		if( ++idx_obs == num_obs )
		  idx_obs = 0;
		*obs_status = obs_data[idx_obs];
		/* Process new observer */
		New_Observer( sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case KEY_PPAGE : /* Load previous observer data from satcom.obs */
		if( --idx_obs < 0 )
		  idx_obs = num_obs - 1;
		*obs_status = obs_data[idx_obs];

		/* Process new observer */
		New_Observer( sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case 'h' : case 'H' : /* Load 'Home' observer data from satcom.obs */
		Load_Observer_Set( "Home", obs_status );
		/* Process new observer */
		New_Observer( sat_tle, sat_status, obs_status );
		Clear_Flag( USERTIME_PREDICT_FLAG );
		break;

	  case 'm' : case 'M' : /* Multi-satellite/single-observer mode */
		Multi_Satellite_Mode( sat_tle, obs_status, sat_status );
		Screen_Root();
		Screen_Commands( PASS_PREDICTION_MODE );
		Clear_Flag( USERTIME_PREDICT_FLAG );

	} /* End of switch( chr ) */


	/* Calculate time now as julian UTC */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
	time_now = Julian_Date( &obs_status->utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( time_now, &solar_kinetics );

	/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	Calculate_Satellite_Status( time_now, sat_tle,
		&solar_kinetics,
		&obs_kinetics,
		obs_status,
		sat_status );

	/* Calculate lunar azimuth and elevation */
	Moon_Position( time_now, obs_status, &obs_kinetics );

	/* If satellite is accessible and we have */
	/* AOS/LOS data then do pass predictions  */
	if( isSatFlagClear(sat_status, NO_AOS_LOS_FLAG) )
	{
	  /* Number of entries in prediction chart */
	  num = 15;

	  /* Initialize for real-time pass predictions */
	  if( isFlagClear(USERTIME_PREDICT_FLAG) )
	  {
		temp_sat_status = *sat_status;
		temp_obs_status = *obs_status;

		/* If aos > los, satellite is above horizon.  */
		/* Predictions are done starting in real time */
		if( temp_sat_status.aos > temp_sat_status.los )
		  aos_utc = time_now;
		else
		  aos_utc = temp_sat_status.aos;
		los_utc = temp_sat_status.los;

	  } /* End of if( isFlagClear(USERTIME_PREDICT_FLAG) ) */
	  else
		if( from_utc != old_utc )
		{
		  /* Enable AOS/LOS calculation */
		  temp_sat_status.elev = 0.;

		  /* Calculate topocentric position of sun */
		  Calculate_Solar_Position( from_utc, &solar_kinetics );

		  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
		  Calculate_Satellite_Status( from_utc, sat_tle,
			  &solar_kinetics,
			  &obs_kinetics,
			  &temp_obs_status,
			  &temp_sat_status );

		  /* If aos > los, satellite is above horizon.  */
		  /* Predictions are done starting in user time */
		  if( temp_sat_status.aos > temp_sat_status.los )
			aos_utc = from_utc;
		  else
			aos_utc = temp_sat_status.aos;
		  los_utc = temp_sat_status.los;

		} /* End of if( from_utc != old_utc ) */


	  /* Look for (visually) visible passes.  */
	  /* Loop till a visible pass is detected */
	  if( isFlagSet(VISIBLE_PREDICT_FLAG) &&
		  isSatFlagClear(&temp_sat_status, PASS_VISIBLE_FLAG) )
	  {
		if( old_los != temp_sat_status.los )
		{
		  old_los = temp_sat_status.los;
		  while( isSatFlagClear(&temp_sat_status, PASS_VISIBLE_FLAG) )
		  {
			/* Move 5 min ahead of current LOS time */
			julian_utc = temp_sat_status.los + 0.0035;

			/* Enable AOS/LOS calculation */
			temp_sat_status.elev = 0.;

			/* Calculate topocentric position of sun */
			Calculate_Solar_Position( julian_utc, &solar_kinetics );

			/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
			Calculate_Satellite_Status( julian_utc, sat_tle,
				&solar_kinetics,
				&obs_kinetics,
				&temp_obs_status,
				&temp_sat_status );

		  } /* End of while( isSatFlagClear(&temp_sat_status, ... ) */

		  /* If aos > los, satellite is above horizon.  */
		  /* Predictions are done starting in real time */
		  if( temp_sat_status.aos > temp_sat_status.los )
			aos_utc = time_now;
		  else
			aos_utc = temp_sat_status.aos;
		  los_utc = temp_sat_status.los;

		  /* Calculate time-step for pass plot */
		  from_utc = aos_utc;
		  to_utc = los_utc;
		  step_utc = (los_utc - aos_utc)/(double)(num-1);
		}

	  } /* End of isFlagSet(VISIBLE_PREDICT_FLAG) ) */
	  else
	  {
		/* Calculate time-step for pass plot */
		from_utc = aos_utc;
		to_utc = los_utc;
		step_utc = (los_utc - aos_utc)/(double)(num-1);
	  }

	  /* Refresh predictions if from_utc changes */
	  if( from_utc != old_utc )
	  {
		/* Save temporary sat_status */
		old_sat_status = temp_sat_status;

		/* Setup start time and save original value */
		julian_utc = old_utc = from_utc;

		/* Suppress AOS/LOS computations while plotting pass */
		Set_SatFlag( &temp_sat_status, NO_AOS_LOS_FLAG );

		/* Calculate pass prediction data */
		for( idx = 0; idx < num; idx++ )
		{
		  /* Calculate topocentric position of sun */
		  Calculate_Solar_Position( julian_utc, &solar_kinetics );

		  /* Calculate sat az/el/range/lat/lon/alt/visibility etc */
		  Calculate_Satellite_Status( julian_utc, sat_tle,
			  &solar_kinetics,
			  &obs_kinetics,
			  &temp_obs_status,
			  &temp_sat_status );

		  /* Enter pass prediciton data */
		  pass_data.flags[idx]  = temp_sat_status.flags;
		  pass_data.elev[idx]   = temp_sat_status.elev;
		  pass_data.azim[idx]   = temp_sat_status.azim;
		  pass_data.range[idx]  = temp_sat_status.range;
		  pass_data.lon[idx]    = temp_sat_status.ssplon;
		  pass_data.lat[idx]    = temp_sat_status.ssplat;
		  pass_data.phs256[idx] = temp_sat_status.phs256;
		  pass_data.edepth[idx] = temp_sat_status.edepth;
		  pass_data.julian_utc[idx] = julian_utc;

		  julian_utc += step_utc;

		} /* End of for( idx = 0; idx < num; idx++ ) */

		/* Restore AOS/LOS search computations */
		Clear_SatFlag( &temp_sat_status, NO_AOS_LOS_FLAG );

		/* Restore temporary sat_status */
		temp_sat_status = old_sat_status;

	  } /* End of if( from_utc != old_utc ) */

	} /* End of if(isSatFlagClear(sat_status, NO_AOS_LOS_FLAG)) */
	else
	  num = 0; /* No entries in prediction chart */

	Screen_Pass_Predictions( num, &pass_data, obs_status, sat_status );

	/* Control rotor position */
	if( isFlagSet(ROTORS_ENABLED_FLAG | TRACKING_ENABLED_FLAG) )
	{
	  if( isSatFlagSet( sat_status, NORTH_CROSSING_FLAG) )
		Set_Flag( REVERSE_TRACK_FLAG );
	  else
		Clear_Flag( REVERSE_TRACK_FLAG );

	  Screen_Rotor_Control( sat_status->azim, sat_status->elev,
		  sat_status->resol, NO_DISPLAY );
	}

	/* Activate transceiver CAT control if enabled */
	if( isFlagSet(CAT_ENABLED_FLAG) )
	  /* Control FT-847 parameters */
	  CAT_Control( sat_status );

  }  /* End of main do loop. (Quit on Q or ESC) */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );


} /* End of Pass_Prediction_Mode() */

/*------------------------------------------------------------------*/

/*  Illumination_Prediction_Mode()
 *
 *  Predicitons of satellite illumination
 */

void
Illumination_Prediction_Mode(
	tle_t *sat_tle,
	observer_status_t *obs_status,
	satellite_status_t *sat_status )
{
  double
	jdate[30], /* Dates of illumination predictions */
	julian_utc,/* Julian time UTC */
	from_utc,  /* Time to start predicitons from */
	user_utc,  /* Time to start predicitons from, specified by user */
	age,       /* Days since epoch    */
	tsince;    /* Minutes since epoch */

  int
	illum[30], /* Illumunation prediction values */
	chr=0,     /* Character input by getch */
	idx,       /* Index for loops etc */
	ilm,       /* Index used in illumination predicitons */
	done;      /* Predicitons completed flag */

  /* Solar and satellite position/velocity */
  kinetics_t
	solar_kinetics,
	sat_kinetics;

  Screen_Commands( ILLUM_PREDICTION_MODE );

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  done = FALSE;

  Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
  from_utc = user_utc = Julian_Date( &obs_status->utc );

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case 'u' : case 'U' : /* User-specified start time */
		Post_UserTime_Form( &user_utc );
		from_utc = user_utc;
		done = FALSE;
		break;

	  case 'f' : case 'F' : /* Forward to next pass prediction */
		from_utc += 30.; /* + 30 days */
		done = FALSE;
		break;

	  case 'b' : case 'B' : /* Forward to next pass prediction */
		from_utc -= 30.; /* - 30 days */
		done = FALSE;
		break;

	  case 'r' : case 'R' : /* Back to user time for prediction */
		from_utc = user_utc;
		done = FALSE;
		break;

	  case 'm' : case 'M' : /* Multi-satellite/single-observer mode */
		Multi_Satellite_Mode( sat_tle, obs_status, sat_status );
		Screen_Root();
		Screen_Commands( ILLUM_PREDICTION_MODE );
		done = FALSE;
		break;

	  case KEY_RIGHT : /* Request next satellite in TLE file */
		Find_New_Satellite( NEXT, sat_tle, sat_status, obs_status );
		done = FALSE;
		break;

	  case KEY_LEFT : /* Request previous satellite in TLE file */
		Find_New_Satellite( PREVIOUS, sat_tle, sat_status, obs_status );
		done = FALSE;
		break;

	  case KEY_UP :	/* Request first satellite in TLE file */
		Find_New_Satellite( FIRST, sat_tle, sat_status, obs_status );
		done = FALSE;
		break;

	  case KEY_DOWN : /* Request last satellite in TLE file */
		Find_New_Satellite( LAST, sat_tle, sat_status, obs_status );
		done = FALSE;
		break;

	  case KEY_IC : /* Request default satellite */
		Find_New_Satellite( DEFAULT, sat_tle, sat_status, obs_status );
		done = FALSE;
		break;

	} /* End of switch( chr ) */

	/* Calculate time now as julian UTC */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );

	julian_utc = from_utc;

	for( idx = 0; (idx < 30) && (done == FALSE); idx++ )
	{
	  illum[idx] = 0;
	  jdate[idx] = julian_utc;

	  for( ilm = 0; ilm < 1440; ilm++ )
	  {
		/* Calculate topocentric position of sun */
		Calculate_Solar_Position( julian_utc, &solar_kinetics );

		/* Calculate time since epoch in minutes */
		age = julian_utc - sat_tle->epoch;
		tsince = age * xmnpda;

		/* Call NORAD routines according to deep-space flag */
		if( sat_tle->ephem == DEEP_SPACE )
		  SDP4( tsince, sat_tle, &sat_kinetics );
		else
		  SGP4( tsince, sat_tle, &sat_kinetics );

		/* Scale position and velocity vectors to km and km/sec */
		Convert_Sat_State( &sat_kinetics );

		/* Calculate scalar magnitude of velocity */
		Magnitude( &sat_kinetics.vel );

		/* Calculate satellite's illumination */
		if( !Satellite_Eclipsed(&sat_kinetics, &solar_kinetics,
			  &sat_status->edepth) )
		  illum[idx]++;

		/* Step time by 1 minute of day */
		julian_utc += 6.944444E-4;

	  }/* End of for( ilm = 0; ilm < 1440; ilm++ ) */

	  Screen_Progress_Bar( idx );

	} /* End of for( idx = 0; idx < 30; idx++ ) */

	done = TRUE;

	Screen_Illumination_Predictions( jdate, illum,
		obs_status,
		sat_status );

  }  /* End of main do loop. (Quit on Q or ESC) */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );

} /* End of Illumination_Prediction_Mode() */

/*------------------------------------------------------------------*/

/*  Manual_Command_Mode()
 *
 *  Manual control of rotors and wireless set
 */

void
Manual_Command_Mode(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  int
	azim_man, /* Current manual position demand for azimuth rotor   */
	elev_man, /* Current manual position demand for elevation rotor */
	chr=0;    /* Used by getch for keyboard entries */

  /* Azimuth and elevation demand for rotor control */
  double
	azim=0.,
	elev=0.;

  double julian_utc; /* Julian UTC */

  kinetics_t
	obs_kinetics,   /*  Observer position/velocity vectors */
	solar_kinetics; /*     Solar position/velocity vectors */

  /* Solar velocity vector not used at the moment */
  solar_kinetics.vel.x = 0.;
  solar_kinetics.vel.y = 0.;
  solar_kinetics.vel.z = 0.;
  solar_kinetics.vel.w = 0.;

  /* Assume rotors are in parking position */
  azim_man = 180;
  elev_man = 0;

  /* Azimuth and elevation demand for rotor control */

  Screen_Root();
  Screen_Commands( MANUAL_COMM_MODE );

  /* Keystroke command loop  */
  /* Exit mode on 'Q' or ESC */
  do
  {
	switch( chr )
	{
	  case 'm' : case 'M' : /* Post manual azim/elev entry form */
		Post_Manual_Form( &azim_man, &elev_man );
		Screen_Root();
		Screen_Commands( MANUAL_COMM_MODE );
		break;

	  case 'h' : case 'H' : /* Move rotors to parking position */
		azim_man = AZIM_PARKING;
		elev_man = ELEV_PARKING;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case 'p' : case 'P' : /* Move rotors to Polar Star */
		azim_man = 0;
		elev_man = Degrees(obs_status->obs_geodetic.lat) + 0.5;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case KEY_LEFT :	/* Manual rotate left command */
		azim_man = 0;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case KEY_RIGHT : /* Manual rotate right command */
		azim_man = 360;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case KEY_UP : /* Manual rotate up command */
		elev_man = 180;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case KEY_DOWN : /* Manual rotate down command */
		elev_man = 0;
		Manual_Position_Rotors( azim_man, elev_man );
		break;

	  case KEY_SPACE : /* Manual All stop command */
		if( isFlagSet(ROTORS_ENABLED_FLAG) )
		{
		  /* Read current position of rotors */
		  Rotor_Read_Direction( &azim_man, &elev_man );
		  /* Make current position the new demand */
		  Manual_Position_Rotors( azim_man, elev_man );
		}
		break;

	  case 'l' : case 'L' : /* Track the Moon */
		Set_Flag( TRACK_MOON_FLAG );

		if( isFlagClear(TRACKING_ENABLED_FLAG) )
		  Enable_Tracking();
		else
		{
		  /* Read current position of rotors */
		  Rotor_Read_Direction( &azim_man, &elev_man );
		  /* Make current position the new demand */
		  Manual_Position_Rotors( azim_man, elev_man );
		  Disable_Tracking();
		}
		break;

	  case 's' : case 'S' : /* Track the Sun */
		Clear_Flag( TRACK_MOON_FLAG );

		if( isFlagClear(TRACKING_ENABLED_FLAG) )
		  Enable_Tracking();
		else
		{
		  /* Read current position of rotors */
		  Rotor_Read_Direction( &azim_man, &elev_man );
		  /* Make current position the new demand */
		  Manual_Position_Rotors( azim_man, elev_man );
		  Disable_Tracking();
		}

	} /* End of switch( chr ) */

	/* Get UTC calendar and convert to Julian */
	Calendar_Time_Now( &obs_status->utc, &obs_status->localtm );
	julian_utc = Julian_Date( &obs_status->utc );

	/* Calculate topocentric position of sun */
	Calculate_Solar_Position( julian_utc, &solar_kinetics );

	/* Calculate sat az/el/range/lat/lon/alt/visibility etc */
	Calculate_Satellite_Status(
		julian_utc, sat_tle,
		&solar_kinetics, &obs_kinetics,
		obs_status, sat_status );

	/* Calculate lunar azimuth and elevation */
	Moon_Position( julian_utc, obs_status, &obs_kinetics );

	/* Display observer status */
	Screen_Observer_Status( obs_status );

	if( isFlagSet(ROTORS_ENABLED_FLAG) )
	{
	  /* If tracking enabled, refresh Az/El demand */
	  if( isFlagSet(TRACKING_ENABLED_FLAG) )
	  {
		if( isFlagSet(TRACK_MOON_FLAG) )
		{
		  azim = obs_status->lunar_set.x;
		  elev = obs_status->lunar_set.y;
		}
		else
		{
		  azim = Degrees( obs_status->solar_set.x );
		  elev = Degrees( obs_status->solar_set.y );
		}

	  } /* End of if( isFlagSet(TRACKING_ENABLED_FLAG) ) */
	  else
	  {
		azim = azim_man;
		elev = elev_man;
	  }

	  /* Control and display rotor position and status */
	  Screen_Rotor_Control( azim, elev, HIGH_RESOL, DISPLAY );
	}

  } /* End of main do loop */
  while( ((chr = getch()) != 'q') && (chr != 'Q') && (chr != KEY_ESC) );

  /* Make current position the new demand */
  if( isFlagSet(ROTORS_ENABLED_FLAG) )
  {
	Rotor_Read_Direction( &azim_man, &elev_man );
	Manual_Position_Rotors( azim_man, elev_man );
  }

} /* End of Manual_Command_Mode() */

/*------------------------------------------------------------------*/

/*  Calculate_Satellite_Status()
 *
 *  Does all the calculations needed to
 *  fill the  satellite status structure
 */

void
Calculate_Satellite_Status(
	double julian_utc,
	tle_t *sat_tle,
	kinetics_t *solar_kinetics,
	kinetics_t *obs_kinetics,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )

{
  double
	horizon, /* Horizon threshold */
	age;     /* Elpsed time after sat epoch */

  /* Satellite's predicted geodetic position */
  geodetic_t sat_geodetic;

  /* Satellite Azim, Elev, Range, Range rate */
  vector_t obs_set;

  /* Satellite position/velocity vectors */
  kinetics_t sat_kinetics;

  horizon = Radians(HORIZON);

  /* Calculate all satellite orbital data */
  Calculate_Orbit( julian_utc, sat_tle,
	  &sat_geodetic, &sat_kinetics,
	  sat_status, obs_status,
	  obs_kinetics, &obs_set );

  /* Calculate solar position and satellite eclipse depth, */
  /* set or clear the satellite eclipsed flag accordingly  */
  Calculate_Observer( solar_kinetics, obs_status,
	  obs_kinetics, &obs_status->solar_set );

  /* Calculate satellite's illumination */
  if( Satellite_Eclipsed(&sat_kinetics, solar_kinetics, &sat_status->edepth) )
	Set_SatFlag( sat_status, SAT_ECLIPSED_FLAG );
  else
	Clear_SatFlag( sat_status, SAT_ECLIPSED_FLAG );

  /* If sun is 12 deg below horizon, set satellite visible flag */
  if( (Degrees(obs_status->solar_set.y) < -12.0) &&
	  (obs_set.y > -horizon) 			 &&
	  isSatFlagClear(sat_status, SAT_ECLIPSED_FLAG) )
	Set_SatFlag( sat_status, SAT_VISIBLE_FLAG );
  else
	Clear_SatFlag( sat_status, SAT_VISIBLE_FLAG );

  /* Find next AOS/LOS in real time after horizon crossing */
  if( sat_status->elev * obs_set.y <= 0. )
	Find_Next_AOS_LOS(julian_utc, sat_tle, obs_status, sat_status);

  age = julian_utc - sat_tle->epoch;

  /* Calculate and fill in satellite status data */
  sat_status->azim    = Degrees(obs_set.x);
  sat_status->elev    = Degrees(obs_set.y);
  sat_status->range   = obs_set.z;
  sat_status->ploss   = 32.4 + 20.0*log10(obs_set.z);
  sat_status->rrate   = obs_set.w;
  sat_status->doppler = -sat_status->rrate/Cl*1.0E9;
  sat_status->ssplat  = Degrees(sat_geodetic.lat);
  sat_status->ssplon  = Degrees(sat_geodetic.lon);
  sat_status->alt     = sat_geodetic.hgt;
  sat_status->ftprnt  = Footprint( &sat_kinetics.pos );
  sat_status->vel     = sat_kinetics.vel.w;
  sat_status->revnum  = (int)floor(
	  (sat_tle->xno*xmnpda/twopi +
	   age*sat_tle->bstar*ae)*age +
	  sat_tle->xmo/twopi ) +
	sat_tle->revnum;

} /* End of  Calculate_Satellite_Status() */

/*------------------------------------------------------------------*/

/*  Enable_Tracking()
 *
 *  Enables tracking of the selected satellite
 */

  void
Enable_Tracking( void )
{
  /* Open rotor's serial port if closed */
  if( isFlagClear(ROTORS_ENABLED_FLAG) )
  {
	Set_Flag(ROTORS_ENABLED_FLAG);
	Open_Rotor_Serial();
  }

  Clear_Flag( ROTORS_RUNNING_FLAG   );
  Set_Flag(   TRACKING_ENABLED_FLAG );

  Rotor_Send_String( "S" ); /* All stop */

} /* Enable_Tracking() */

/*------------------------------------------------------------------*/

/*  Disable_Tracking()
 *
 *  Enables tracking of the selected satellite
 */

  void
Disable_Tracking( void )
{
  Close_Rotor_Serial();

  Clear_Flag( ROTORS_ENABLED_FLAG   |
	  TRACKING_ENABLED_FLAG |
	  REVERSE_TRACK_FLAG    |
	  DIRECTION_ERROR_FLAG );

} /* Disable_Tracking() */

/*------------------------------------------------------------------*/

/*  Manual_Position_Rotors()
 *
 *  Move az/el rotors to position usually related to a
 *  manual operation like up/down left/right or stop
 */

  void
Manual_Position_Rotors( int azim, int elev )
{
  /* Open rotor's serial port if closed */
  if( isFlagClear(ROTORS_ENABLED_FLAG) )
  {
	Set_Flag(ROTORS_ENABLED_FLAG);
	Open_Rotor_Serial();
  }

  Rotor_Send_String( "S" ); /* All stop */

  Clear_Flag( ROTORS_RUNNING_FLAG   |
	  TRACKING_ENABLED_FLAG |
	  REVERSE_TRACK_FLAG );

  Screen_Rotor_Control( azim, elev, HIGH_RESOL, DISPLAY );

} /* End of Manual_Position_Rotors() */

/*------------------------------------------------------------------*/

/*  Auto_Track()
 *
 *  Enables or disables satellite tracking automatically
 */

  void
Auto_Track( double julian_utc, satellite_status_t *sat_status )
{
  /* Enable tracking about 5 min before AOS */
  if( julian_utc >= (sat_status->aos - 0.0035) &&
	  isFlagClear(TRACKING_ENABLED_FLAG) )
	Enable_Tracking();

  /* Disable tracking about 5 min after LOS */
  if( julian_utc >= (sat_status->los + 0.0035) &&
	  isFlagSet(TRACKING_ENABLED_FLAG) )
	Disable_Tracking();

} /* End of Auto_Track() */

/*------------------------------------------------------------------*/

/*  Enable_CAT()
 *
 *  Enables CAT control of Yaesu FT-847 transceiver
 */

  void
Enable_CAT( void )
{
  /* A 5-char string for sending and receiving */
  /* 5-byte strings to and from the transceiver */
  unsigned char cmnd_parm[5];

  Set_Flag( CAT_ENABLED_FLAG );

  bzero( cmnd_parm, 5 );

  /* Set up transceiver for CAT control in satellite mode */
  Open_Tcvr_Serial();
  cmnd_parm[4] = CAT_ON;
  Write_Tcvr_Serial( cmnd_parm );
  cmnd_parm[4] = SATELLITE_ON;
  Write_Tcvr_Serial( cmnd_parm );

  /* Initialize transceiver status */
  Write_Tcvr_Status( &tcvr_status );

} /* End of Enable_CAT() */

/*------------------------------------------------------------------*/

/*  Disable_CAT()
 *
 *  Disables CAT control of Yaesu FT-847 transceiver
 */

  void
Disable_CAT( void )
{
  /* A 5-char string for sending and receiving */
  /* 5-byte strings to and from the transceiver */
  unsigned char cmnd_parm[5];

  Clear_Flag( CAT_ENABLED_FLAG );

  bzero( cmnd_parm, 5 );

  /* Restore transceiver to manual */
  cmnd_parm[4] = SATELLITE_OFF;
  Write_Tcvr_Serial( cmnd_parm );
  cmnd_parm[4] = CAT_OFF;
  Write_Tcvr_Serial( cmnd_parm );

  Close_Tcvr_Serial();

} /* End of Disable_CAT() */

/*------------------------------------------------------------------*/

/*  Initialize_Tcvr()
 *
 *  Initializes transceiver tx/rx modes and frequencies
 */

  void
Initialize_Tcvr( int mode_idx )
{
  strcpy(tcvr_status.mode_name, sat_mode[mode_idx].mode_name);

  /* If transponder data is given, initialize tcvr status */
  if( sat_mode[mode_idx].flags & TRANSPND_DATA )
  {
	tcvr_status.uplink_freq = sat_mode[mode_idx].uplink_freq;
	tcvr_status.tx_freq     = sat_mode[mode_idx].tx_freq;
	tcvr_status.tx_ref_freq = sat_mode[mode_idx].tx_freq;
	tcvr_status.tx_mode     = sat_mode[mode_idx].tx_mode;
	tcvr_status.tx_mode_ref = NO_TCVR_MODE;
	tcvr_status.dnlink_freq = sat_mode[mode_idx].dnlink_freq;
	tcvr_status.rx_freq     = sat_mode[mode_idx].rx_freq;
	tcvr_status.rx_ref_freq = sat_mode[mode_idx].rx_freq;
	tcvr_status.rx_mode     = sat_mode[mode_idx].rx_mode;
	tcvr_status.rx_mode_ref = NO_TCVR_MODE;

	/* Flags indicating what transponder data is given */
	tcvr_status.flags = sat_mode[mode_idx].flags;

	/* These are frequency offsets between tx/rx frequencies */
	/* and uplink/dnlink due to the use of a transverter or  */
	/* converter and any freq errors in tcvr or transponder  */
	tcvr_status.rx_offset = tcvr_status.dnlink_freq - tcvr_status.rx_freq;
	tcvr_status.tx_offset = tcvr_status.uplink_freq - tcvr_status.tx_freq;

	/* This is a freq offset between the uplink and downlink  */
	/* frequencies such that a signal from the tx tunes in    */
	/* accurately in the rx. It depends on the mode used (CW  */
	/* or SSB) and passband invertion. It is needed for doing */
	/* accurate tx/rx tracking during manual tuning of tcvr.  */
	tcvr_status.upl_dnl_offset  = tcvr_status.uplink_freq;
	tcvr_status.upl_dnl_offset += ( tcvr_status.flags & PBAND_INV ?
		+tcvr_status.dnlink_freq :
		-tcvr_status.dnlink_freq );
  }
  else
	/* If beacon data given, tx/rx data is made same */
	if( sat_mode[mode_idx].flags & BEACON_DATA )
	{
	  /* Read current transceiver status  */
	  /* to read the current tx frequency */
	  Read_Tcvr_Status( &tcvr_status );

	  tcvr_status.dnlink_freq = sat_mode[mode_idx].bcn_freq;
	  tcvr_status.rx_freq     = sat_mode[mode_idx].rx_freq;
	  tcvr_status.rx_ref_freq = sat_mode[mode_idx].rx_freq;
	  tcvr_status.rx_mode     = sat_mode[mode_idx].bcn_mode;
	  tcvr_status.rx_mode_ref = NO_TCVR_MODE;
	  tcvr_status.uplink_freq = tcvr_status.tx_freq;

	  /* This is a frequency offset between the rx   */
	  /* frequency and beacon freq due to the use of */
	  /* converter and any freq errors in the beacon */
	  tcvr_status.rx_offset = tcvr_status.dnlink_freq - tcvr_status.rx_freq;

	  /* Flags indicating what transponder data is given */
	  tcvr_status.flags = sat_mode[mode_idx].flags;
	}
	else
	  if( sat_mode[mode_idx].flags == NO_DATA )
	  {
		tcvr_status.flags = NO_DATA;
		return;
	  }

  /* Write new transceiver status */
  Write_Tcvr_Status( &tcvr_status );

} /* End of Initialize_Tcvr() */

/*------------------------------------------------------------------*/

/*  CAT_Control()
 *
 *  Controls the YAESU FT-847 transceiver via the
 *  CAT port mainly for Doppler shift compensation
 */

  void
CAT_Control( satellite_status_t *sat_status )
{
  int
	rx_doppler = 0,   /* Rx doppler shift correction */
	tx_doppler = 0;   /* Tx doppler shift correction */


  /* Read current transceiver status */
  Read_Tcvr_Status( &tcvr_status );

  /* Return if no data from serial port */
  if( isFlagSet(TCVR_READ_FAIL_FLAG) )
	return;

  /* Return if no transponder data given */
  if( tcvr_status.flags == NO_DATA )
	return;

  /* If user has changed frequency manually, make new */
  /* frequencies the reference for uplink/downlink    */
  /* frequency and doppler shift calculations.        */
  if( tcvr_status.rx_freq != tcvr_status.rx_ref_freq )
  {
	tcvr_status.dnlink_freq += tcvr_status.rx_freq -
	  tcvr_status.rx_ref_freq;

	/* Make uplink track downlink frequency if enabled */
	if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) &&
		(tcvr_status.flags & TRANSPND_DATA) )
	{
	  tcvr_status.uplink_freq  = tcvr_status.upl_dnl_offset;

	  tcvr_status.uplink_freq += ( tcvr_status.flags & PBAND_INV ?
		  -tcvr_status.dnlink_freq :
		  +tcvr_status.dnlink_freq );
	}
  } /* if( tcvr_status.rx_freq != tcvr_status.rx_ref_freq ) */
  else
	if( tcvr_status.tx_freq != tcvr_status.tx_ref_freq )
	{
	  tcvr_status.uplink_freq += tcvr_status.tx_freq -
		tcvr_status.tx_ref_freq;

	  /* Make downlink track uplink frequency if enabled */
	  if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) &&
		  (tcvr_status.flags & TRANSPND_DATA) )
	  {
		tcvr_status.dnlink_freq = tcvr_status.upl_dnl_offset -
		  tcvr_status.uplink_freq;

		tcvr_status.dnlink_freq = ( tcvr_status.flags & PBAND_INV ?
			+tcvr_status.dnlink_freq :
			-tcvr_status.dnlink_freq );

	  } /* if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) &&... */

	} /* if( tcvr_status.tx_freq != tcvr_status.tx_ref_freq ) */

  /* Match transmitter/receiver modes if manually changed */
  if( tcvr_status.rx_mode != tcvr_status.rx_mode_ref )
  {
	tcvr_status.rx_mode_ref = tcvr_status.rx_mode;
	if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) )
	{
	  if( tcvr_status.flags & PBAND_INV )
	  {
		if( tcvr_status.rx_mode == USB )
		  tcvr_status.tx_mode = LSB;
		else
		  if( tcvr_status.rx_mode == LSB )
			tcvr_status.tx_mode = USB;
		  else
			tcvr_status.tx_mode = tcvr_status.rx_mode;
	  }
	  else
		tcvr_status.tx_mode = tcvr_status.rx_mode;

	} /* if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) ) */

  } /* if( tcvr_status.rx_mode != tcvr_status.rx_mode_ref ) */
  else
	if( tcvr_status.tx_mode != tcvr_status.tx_mode_ref )
	{
	  tcvr_status.tx_mode_ref = tcvr_status.tx_mode;
	  if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) )
	  {
		if( tcvr_status.flags & PBAND_INV )
		{
		  if( tcvr_status.tx_mode == USB )
			tcvr_status.rx_mode = LSB;
		  else
			if( tcvr_status.tx_mode == LSB )
			  tcvr_status.rx_mode = USB;
			else
			  tcvr_status.rx_mode = tcvr_status.tx_mode;
		}
		else
		  tcvr_status.rx_mode = tcvr_status.tx_mode;

	  } /* if( isFlagSet(UPLNK_DNLNK_TRACK_FLAG) ) */

	} /* if( tcvr_status.tx_mode != tcvr_status.tx_mode_ref ) */

  /* Calculate doppler shift. sat_status->doppler is in */
  /* Hz/MHz and tcvr_status.dnlink_freq etc in 10xHz.   */
  rx_doppler = (tcvr_status.dnlink_freq*1.E-5) * sat_status->doppler;
  tx_doppler = (tcvr_status.uplink_freq*1.E-5) * sat_status->doppler;

  if( isFlagClear(DOPPLER_DISABLED_FLAG) )
  {
	/* Apply doppler shift correction. Note that Doppler shift is  */
	/* added to downlink and subtracted from the uplink frequency. */
	/* This is because uplink_freq and dnlink_freq are the signal  */
	/* frequencies arriving and leaving from the satellite itself. */
	tcvr_status.rx_freq = tcvr_status.dnlink_freq -
	  tcvr_status.rx_offset +
	  rx_doppler/10.;

	tcvr_status.tx_freq = tcvr_status.uplink_freq -
	  tcvr_status.tx_offset -
	  tx_doppler/10.;

  } /* End of if( isFlagSet(DOPPLER_DISABLED_FLAG) ) */
  else
  {
	tcvr_status.rx_freq = tcvr_status.dnlink_freq -
	  tcvr_status.rx_offset;
	tcvr_status.tx_freq = tcvr_status.uplink_freq -
	  tcvr_status.tx_offset;
  }

  /* Write new transceiver status */
  Write_Tcvr_Status( &tcvr_status );

  /* Reset references */
  tcvr_status.rx_ref_freq = tcvr_status.rx_freq;
  tcvr_status.tx_ref_freq = tcvr_status.tx_freq;

  /* Enter doppler shift for display */
  tcvr_status.rx_doppler = rx_doppler;
  tcvr_status.tx_doppler = tx_doppler;

  /* Enter path loss for display */
  tcvr_status.rx_ploss = sat_status->ploss + 20. *
	log10(tcvr_status.dnlink_freq/1.E5);
  tcvr_status.tx_ploss = sat_status->ploss + 20. *
	log10(tcvr_status.uplink_freq/1.E5);

} /* End of CAT_Control() */

/*------------------------------------------------------------------*/
