/*  input.c
 *
 *  Functions to do file i/o operations and carry out
 *  checking and preprocessing on data, mostly TLE sets,
 *  observer location and satellite databases
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "satcom.h"

/* Buffer for reading satellite transponder */
/* modes from the satellite database file   */
extern satellite_mode_t *sat_mode;

/* All-satellite TLE data */
extern tle_t *tle_data;

/* Number of entries in tle_data */
extern int num_tles;

/* TLE set currently in use */
extern int idx_tle;

/* All-observer location data */
extern observer_status_t *obs_data;

/* Number of entries in obs_data */
extern int num_obs;

/* Observer set currently in use */
extern int idx_obs;

/* Number of transponder modes found in database */
extern int num_modes;


/*  Load_Database_Tle_Data()
 *
 *  Loads the TLE sets of all satellites in the satcom.sat
 *  database file into buffer, checking for validity.
 */
  void
Load_Database_Tle_Data( void )
{
  int
	chr,       /* Character read from file    */
	tle_eof,   /* satcom.tle file EOF flag    */
	sat_eof,   /* satcom.sat file EOF flag    */
	dup_flg,   /* Duplicate entry flag        */
	tle_lines, /* Number of lines in TLE file */
	num_ld,    /* Number of TLE sets loaded   */
	num_sats,  /* Number of satellites in use */
	icp,       /* Index for comparison loops  */
	idxa,      /* Index for loops etc         */
	idxb;      /* Index for loops etc         */

  char
	home_dir[25],  /* Home directory name string */
	tle_fpath[50], /* File path to satcom.tle    */
	sat_fpath[50], /* File path to satcom.sat    */
	tle_set[139],  /* Two lines of a TLE set     */
	line[81],      /* Line buff for Load_Line()  */
	name[25];      /* Satellite name buffer      */

  FILE
	*tle_file,  /* Keplerian TLE database */
	*sat_file;  /* Satellites database    */

  /* Setup 'home' directory path */
  strncpy( home_dir, getenv("HOME"), 24 );
  home_dir[24] = '\0';

  /* Setup file path to satcom.tle (~/.satcom/satcom.tle) */
  snprintf( tle_fpath, 50, "%s/.satcom/satcom.tle", home_dir );

  /* Open TLE database file */
  tle_file = fopen( tle_fpath, "r" );
  if( tle_file == NULL )
  {
	perror( tle_fpath );
	exit( -1 );
  }

  /* Count number of lines in satcom.tle */
  tle_lines = 0;
  while( (chr = fgetc(tle_file)) != EOF )
	if( chr == KEY_LFD )
	  tle_lines++;

  /* Check that number of lines is %3 and not 0 */
  if( (tle_lines % 3) || (tle_lines == 0) )
	Abort_On_Error( -2 );

  /* Setup file path to satcom.sat (~/.satcom/satcom.sat) */
  snprintf( sat_fpath, 50, "%s/.satcom/satcom.sat", home_dir );

  /* Open satellite database file */
  sat_file = fopen( sat_fpath, "r" );
  if( sat_file == NULL )
  {
	perror( sat_fpath );
	exit( -1 );
  }

  /* Load TLE data from satcom.tle for sat names in satcom.sat */

  /* Count number of satellites in satcom.sat */
  num_sats = 0;
  do
  {
	sat_eof = Load_Line(line, 80, sat_file, -11);
	icp = 1;
	if( line[0] == '[' )
	{
	  while( (line[icp++] != ']') && (icp < 81) );
	  if( icp < 81 )
		num_sats++;
	}

  } /* End of while( sat_eof != EOF ) */
  while( sat_eof != EOF );

  /* Abort if no satellite names found */
  if( num_sats == 0 )
	Abort_On_Error( -12 );

  /* Allocate memory to TLE database buffer */
  tle_data = realloc( tle_data, sizeof(tle_t) * num_sats );
  if( tle_data == NULL )
	Abort_On_Error( -8 );

  /* Look for a satellite name in satcom.sat */
  /* and load its TLE set from satcom.tle    */
  rewind( sat_file );
  num_ld = 0;

  for( idxa = 0; idxa < num_sats; idxa++ )
  {
	do
	{
	  /* Read a line, break at EOF */
	  sat_eof = Load_Line(line, 80, sat_file, -11);

	  icp = 1;
	  /* Look for a satellite name in [...] */
	  if( line[0] == '[' )
	  {
		while( (line[icp++] != ']') && (icp < 81) );

		/* Read satellite name */
		if( icp < 81 )
		{
		  line[icp-1] = '\0'; /* Kill ']' */

		  /* Truncate name to 24 chars */
		  strncpy( name, &line[1], 24 );
		  name[24] = '\0';

		} /* End of if( icp < 81 ) */

	  } /* End of if( line[0] == '[' ) */

	} /* End of do. Continue if ']' not found in line */
	while( ((icp == 1) || (icp == 81)) && (sat_eof != EOF) );

	/* Find a TLE entry for the satellite name  */
	/* that was loaded from the satcom.sat file */
	tle_eof = 0;
	rewind(tle_file);

	for( idxb = 0; ( (idxb < tle_lines) && (!tle_eof) ); idxb++ )
	{
	  Load_Line( line, 80, tle_file, -1 );
	  if( strncmp(line, name, strlen(name)) == 0 )
		break;
	} /* End of for( idxb = 0; idx < num_tles; idx++ ) */

	/* Abort if no matching name found by end of file */
	if( idxb == tle_lines)
	{
	  /* Cleanup */
	  clear();
	  refresh();
	  endwin();
	  printf( "satcom: %s: Satellite not found in "
		  "~/.satcom/satcom.tle file - aborting\n",
		  name );
	  exit( -13 );
	}

	/* Read in first line of TLE set */
	if( Load_Line( tle_set, 69, tle_file, -1 ) == EOF )
	  Abort_On_Error( -1 );

	/* Read in second line of TLE set */
	tle_eof = Load_Line( &tle_set[69], 69, tle_file, -1 );

	/* Check for duplicate TLE set before entering to buffer */
	dup_flg = 0;
	for( icp = 0; icp < num_ld; icp++ )
	  if( strncmp(tle_data[icp].name, name, strlen(name)) == 0 )
		dup_flg = 1;

	/* Enter TLE set to buffer if not duplicate */
	if( !dup_flg )
	{
	  /* Enter name in TLE structure */
	  strncpy( tle_data[num_ld].name, name, 25 );

	  /* Check TLE set and Abort -3 if not valid */
	  if( !Good_Elements(tle_set) )
		Abort_On_Error( -3 );

	  /* Convert the TLE set to orbital elements */
	  Convert_Satellite_Data( tle_set, &tle_data[num_ld] );

	  /* Convert satellite's epoch time to Julian  */
	  tle_data[num_ld].epoch =
		Julian_Date_of_Epoch(tle_data[num_ld].epoch);

	  /** Select ephemeris type! **/
	  /* Will set or clear the DEEP_SPACE_FLAG */
	  /* depending on the TLE parameters of the satellite. */
	  /* It will also pre-process tle members for the      */
	  /* ephemeris functions SGP4 or SDP4 so this function */
	  /* MUST be called each time a new tle set is used    */
	  Select_Ephemeris( &tle_data[num_ld] );

	  num_ld++;

	} /* End of if( !dup_flg ) */

  } /* End of for( idxa = 0; idxa < num_sats; idxa++ ) */

  num_tles = num_ld;

  fclose( tle_file );
  fclose( sat_file );

  return;

} /* End of Load_Database_Tle_Data() */

/*------------------------------------------------------------------*/

/*  Load_Tle_Set()
 *
 *  Loads the TLE set of a given satellite
 */

  void
Load_Tle_Set( char *sat_name, tle_t *sat_tle )
{
  /* index for loops etc */
  int idx;

  /* Clear satellite name string */
  strcpy( sat_tle->name, "" );

  /* Look for a matching satellite name in database */
  for( idx = 0; idx < num_tles; idx++ )
	if( strncmp(tle_data[idx].name, sat_name, strlen(sat_name)) == 0 )
	{
	  /* Load TLE set on match */
	  *sat_tle = tle_data[idx];
	  idx_tle = idx;
	  break;
	}

  /* If no matching name, load 1st TLE set as default */
  if( strcmp(sat_tle->name, "") == 0 )
  {
	*sat_tle = tle_data[0];
	idx_tle = 0;
  }

} /* End of Load_Tle_Set() */

/*------------------------------------------------------------------*/

/*  Read_Transponder_Data()
 *
 *  Reads satellite's transponder modes including uplink
 *  and downlink frequencies, tcvr's corresponding
 *  transmit and receive frequencies and beacons
 */

  int
Read_Transponder_Data( char *sat_name, satellite_mode_t *sat_mod )
{
  char
	home_dir[25],  /* Home directory name string */
	sat_fpath[50], /* File path to satcom.sat    */
	line[81];      /* Line buffer for Load_Line  */

  int
	idx,  /* Index for loops */
	eof;  /* EOF flag memory */

  FILE *sat_file;  /* Satellites database */

  /* Allocate memory to transponder mode data buffer */
  if( sat_mode == NULL )
	sat_mode = realloc( sat_mode, sizeof(satellite_mode_t) );
  if( sat_mode == NULL )
	Abort_On_Error( -16 );

  /* Setup 'home' directory path */
  strncpy( home_dir, getenv("HOME"), 24 );
  home_dir[24] = '\0';

  /* Setup file path to satcom.tle (~/.satcom/satcom.tle) */
  snprintf( sat_fpath, 50, "%s/.satcom/satcom.sat", home_dir );

  /* Open satellite database file */
  sat_file = fopen( sat_fpath, "r" );
  if( sat_file == NULL )
  {
	perror( sat_fpath );
	exit( -1 );
  }

  /* Limit sat name comparison to */
  /* first 10 char. to avoid '~'  */
  idx = strlen( sat_name );
  if( idx > 10 )
	idx = 10;

  /* Find the satellite's data block */
  do
  {
	eof = Load_Line(line, 80, sat_file, -11);

	/* Break if sat_name found in database */
	if( strncmp( &line[1], sat_name, idx ) == 0 )
	  break;
  }
  while( eof != EOF );

  num_modes = 0;
  /* Read next line in satcom.sat */
  eof = Load_Line( line, 80, sat_file, -11 );

  /* Read satellite's transponder data if available */
  for(;;)
  {
	/* Clear satellite mode flags */
	sat_mode[num_modes].flags = NO_DATA;

	/* Look for transponder mode and frequency data.   */
	/* Frequencies are specified in float MHz and are  */
	/* rounded to Hzx10 units as used by CAT control.  */

	/* Look for a 'Mode' entry, return if none */
	if( strncmp(line, "Mode", 4) == 0 )
	{
	  /* Remove leading spaces from mode name */
	  for( idx = 4; idx < strlen(line); idx++)
		if( line[idx] != ' ' )
		  break;
	}
	else
	{
	  strcpy( sat_mode[num_modes].mode_name, "No Mode" );
	  return( -1 );
	}

	/* No mode name! */
	if( (strlen(&line[idx]) == 0) || (eof == EOF) )
	{
	  strcpy( sat_mode[num_modes].mode_name, "No Mode" );
	  return( -1 );
	}

	/* Enter mode name in buffer */
	if( idx > 73 )
	  idx = 73; /* Avoid line buf overrun */
	strncpy( sat_mode[num_modes].mode_name, &line[idx], 7 );
	sat_mode[num_modes].mode_name[7] = '\0';

	/* Read next line in satcom.sat */
	eof = Load_Line( line, 80, sat_file, -11 );

	/* Look for 'Uplink' entry, read freq and mode */
	if( strncmp(line, "Uplink", 6 ) == 0 )
	{
	  sat_mode[num_modes].uplink_freq = atof( &line[6] )*1.E5 + 5.E-6;
	  sat_mode[num_modes].tx_mode     = Read_Operating_Mode( line );

	  /* Abort if no 'Downlink' entry */
	  /* after 'Uplink' due to EOF    */
	  if( eof == EOF )
		Abort_On_Error( -14 );

	  /* Read next line. May be 'Transmit', */
	  /* if not it should be 'Downlink'.    */
	  eof = Load_Line( line, 80, sat_file, -11 );

	  /* If it is 'Transmit', read tx freq and mode */
	  if( strncmp(line, "Transmit", 8) == 0 )
	  {
		sat_mode[num_modes].tx_freq = atof( &line[8] )*1.E5 + 5.E-6;

		/* Abort if no 'Downlink' entry */
		/* after 'Transmit' due to EOF  */
		if( eof == EOF )
		  Abort_On_Error( -14 );

		/* Read next line, should be 'Downlink' entry */
		eof = Load_Line( line, 80, sat_file, -11 );

	  } /* if(strncmp(line, "Transmit") == 0) */
	  else
		/* If no 'Transmit' entry, assume tx freq same as uplink */
		sat_mode[num_modes].tx_freq = sat_mode[num_modes].uplink_freq;

	  /* Look for a "Downlink" entry, read freq */
	  /* and mode. If its not 'Downlink', abort */
	  if( strncmp(line, "Downlink", 8) == 0 )
	  {
		sat_mode[num_modes].dnlink_freq = atof( &line[8] )*1.E5 + 5.E-6;
		sat_mode[num_modes].rx_mode     = Read_Operating_Mode( line );

		/* Set 'transponder data given' flag */
		sat_mode[num_modes].flags |= TRANSPND_DATA;

		/* If no 'Receive' entry due to EOF */
		/* assume rx freq same as downlink  */
		if( eof == EOF )
		{
		  sat_mode[num_modes].rx_freq = sat_mode[num_modes].dnlink_freq;
		  return( eof );
		}

		/* Read next line, may be 'Receive' or  */
		/* 'Passband' or 'Beacon' freq. entry   */
		eof = Load_Line( line, 80, sat_file, -11 );

	  } /* if(strncmp(line, "Downlink") == 0) */
	  else
		/* Abort if no 'Downlink' after 'Uplink' */
		Abort_On_Error( -14 );

	  /* Look for 'Receive' entry, read freq */
	  if( strncmp(line, "Receive", 7) == 0 )
	  {
		sat_mode[num_modes].rx_freq = atof( &line[7] )*1.E5 + 5.E-6;

		/* Return if EOF */
		if( eof == EOF )
		  return( eof );

		/* Read next line, may be 'Passband' or 'Beacon' */
		eof = Load_Line( line, 80, sat_file, -11 );

	  } /* if(strncmp(line, "Receive") == 0) */
	  else
		/* If no 'Receive' entry, assume rx freq same as downlink */
		sat_mode[num_modes].rx_freq = sat_mode[num_modes].dnlink_freq;

	  /* Look for a 'Passband' entry */
	  if( strncmp(line, "Passband", 8) == 0 )
	  {
		/* Look for passband type entry */
		if( strstr(line, "invert") != NULL )
		  sat_mode[num_modes].flags |= PBAND_INV;
		else
		  if( strstr(line, "normal") != NULL )
			sat_mode[num_modes].flags &= ~PBAND_INV;
		  else
			Abort_On_Error( -15 );

		/* Return if EOF */
		if( eof == EOF )
		  return( eof );

		/* Read next line, may be 'Uplink' or 'Beacon' */
		eof = Load_Line( line, 80, sat_file, -11 );

	  } /* if( strncmp(line, "Passband") == 0 ) */
	  else
		/* Assume passband inverted if not specified */
		sat_mode[num_modes].flags |= PBAND_INV;

	} /* if(strncmp(line, "Uplink") == 0) */
	else
	  /* Read beacon freq, else assume no */
	  /* transponder data for satellite.  */
	  if( strncmp(line, "Beacon", 6) == 0 )
	  {
		sat_mode[num_modes].bcn_freq = atof( &line[6] )*1.E5 + 5.E-6;
		sat_mode[num_modes].bcn_mode = Read_Operating_Mode( line );
		sat_mode[num_modes].flags   |= BEACON_DATA;

		/* Return if EOF, assume rx freq same as beacon */
		if( eof == EOF )
		{
		  sat_mode[num_modes].rx_freq = sat_mode[num_modes].bcn_freq;
		  return( eof );
		}

		/* Read next line, may be 'Receive' */
		eof = Load_Line( line, 80, sat_file, -11 );

		/* Look for 'Receive' entry, read freq */
		if( strncmp(line, "Receive", 7) == 0 )
		{
		  sat_mode[num_modes].rx_freq = atof( &line[7] )*1.E5 + 5.E-6;

		  /* Return if EOF */
		  if( eof == EOF )
			return( eof );

		  /* Read next line, may be 'Uplink' or 'Beacon' */
		  eof = Load_Line( line, 80, sat_file, -11 );

		} /* if(strncmp(line, "Receive") == 0) */
		else
		  /* If no 'Receive' entry, assume rx freq same as beacon */
		  sat_mode[num_modes].rx_freq = sat_mode[num_modes].bcn_freq;

	  } /* if( strncmp( line, "Beacon") == 0 ) */

	/* Break if no more transponder data */
	if( sat_mode[num_modes].flags == NO_DATA )
	  break;

	/* Reallocate modes buffer for more data */
	num_modes++;
	sat_mode = realloc( sat_mode, (num_modes+1) * sizeof(satellite_mode_t) );
	if( sat_mode == NULL )
	  Abort_On_Error( -16 );

  } /* End of for(;;) */

  fclose( sat_file );

  return( 0 );

} /* End of Read_Transponder_Data() */

/*------------------------------------------------------------------*/

/*  Read_Operating_Mode()
 *
 *  Reads the uplink/downlink operating mode
 *  specified in the satellite database file
 */

  unsigned char
Read_Operating_Mode( char *line )
{
  /* Mode strings (LSB, USB etc) */
  char *mode_names[NUM_MODE_CODES+1] = { TCVR_MODE_NAMES };

  /* Code numbers of tcvr modes */
  unsigned char mode_codes[NUM_MODE_CODES] = { TCVR_MODE_CODES };

  int idx;  /* Index for loops */

  /* Search the line buffer for a mode */
  /* string and return the mode code   */
  for( idx = 0; idx < 10; idx++ )
	if( strstr(line, mode_names[idx]) != NULL )
	  return( mode_codes[idx] );

  return( NO_TCVR_MODE );

} /* End of Read_Operating_Mode() */

/*------------------------------------------------------------------*/

/*  Request_New_Satellite()
 *
 *  Request the new satellite name entry form. This appears in
 *  the Satellite Name field of the IDENTITY box in main screen
 */

  void
Request_New_Satellite( tle_t *sat_tle,
	satellite_status_t *sat_status,
	observer_status_t  *obs_status )
{
  /* Request new satellite name form */
  Set_SatFlag( sat_status, REQ_KEYBD_ENTRY_FLAG );
  Screen_Satellite_Status( sat_status );

  /* Input a TLE set from file */
  Load_Tle_Set( sat_status->name, sat_tle );

  /* Proprocess new satellite data */
  New_Satellite( sat_tle, sat_status, obs_status );

} /* End of Request_New_Satellite() */

/*------------------------------------------------------------------*/

/*  Find_New_Satellite()
 *
 *  Scrolls up/down in the TLE buffer to find a new satellite.
 *  The new satellite status is then displayed in main screen
 */

  void
Find_New_Satellite( int which, tle_t *sat_tle,
	satellite_status_t *sat_status,
	observer_status_t  *obs_status )
{
  switch( which ) /* How to search */
  {
	case DEFAULT : /* Find default satellite ('Home' keystroke) */
	  /* Input a TLE set from database */
	  Load_Tle_Set( SAT_NAME, sat_tle );
	  break;

	case NEXT : /* Find next satellite ('Right' keystroke) */
	  /* Input next TLE set from database */
	  idx_tle = ( ++idx_tle == num_tles ? 0 : idx_tle );
	  *sat_tle = tle_data[idx_tle];
	  break;

	case PREVIOUS : /* Find previous satellite ('Left' keystroke) */
	  /* Input previous TLE set from database */
	  idx_tle = ( --idx_tle < 0 ? num_tles-1 : idx_tle );
	  *sat_tle = tle_data[idx_tle];
	  break;

	case FIRST : /* Find first satellite ('Up' keystroke) */
	  /* Input a TLE set from database */
	  *sat_tle = tle_data[0];
	  idx_tle = 0;
	  break;

	case LAST : /* Find last satellite ('Down' keystroke) */
	  /* Input a TLE set from file */
	  *sat_tle = tle_data[num_tles-1];
	  idx_tle = num_tles-1;

  } /* End of switch( which ) */

  /* Proprocess new satellite data */
  New_Satellite( sat_tle, sat_status, obs_status );

} /* End of Find_New_Satellite() */

/*------------------------------------------------------------------*/

/*  New_Satellite()
 *
 *  Does the pre-processing needed after a change of satellite
 */

  void
New_Satellite(
	tle_t *sat_tle,
	satellite_status_t *sat_status,
	observer_status_t  *obs_status )
{
  /* Copy satellite name into status structure */
  strncpy(sat_status->name, sat_tle->name, 11);
  sat_status->name[11]  = '\0';

  /* If name is longer than 11 char, make ~ the last char. */
  if( strlen(sat_tle->name) > 11 )
	sat_status->name[10] = '~';

  sat_status->name[11] = '\0';

  /* Clear elevation status, needed in main() */
  /* to determine horizon crossing events     */
  sat_status->elev = 0.;

  /** !Clear all SGP4/SDP4 flags! **/
  /* Before calling a different ephemeris or changing the  */
  /* TLE set, flow control flags MUST be cleared in main() */
  Clear_Flag(SGP_SDP_FLAGS);

  /* Clear all satellite flags */
  Clear_SatFlag( sat_status, ALL_FLAGS );

  /* Set the rotors' tracking resolution according to orbit */
  if( sat_tle->ephem == NEAR_EARTH )
  {
	Clear_SatFlag( sat_status, DEEP_SPACE_FLAG );
	sat_status->resol = LOW_RESOL;
  }
  else
  {
	Set_SatFlag( sat_status, DEEP_SPACE_FLAG );
	if( Satellite_Geostationary(sat_tle) )
	  sat_status->resol = HIGH_RESOL;
	else
	  sat_status->resol = MED_RESOL;
  }

  /* Check whether satellite can ever have an AOS/LOS */
  Set_Accessibility_Flags( sat_tle, obs_status, sat_status );

} /* End of New_Satellite() */

/*------------------------------------------------------------------*/

/*  Set_Acessibility_Flags()
 *
 *  Checks whether satellite can be accessed and sets flags
 */

void
Set_Accessibility_Flags(
	tle_t *sat_tle,
	observer_status_t  *obs_status,
	satellite_status_t *sat_status )
{
  struct tm
	utc,      /* Calendar date and time (UTC)   */
  localtm;  /* Calendar date and time (local) */

  double julian_utc;

  /* Get UTC calendar and convert to Julian */
  Calendar_Time_Now( &utc, &localtm );
  julian_utc = Julian_Date( &utc );

  if( Aos_Possible(sat_tle, obs_status) )
	Clear_SatFlag( sat_status, SAT_INACCESSIBLE_FLAG );
  else
	Set_SatFlag( sat_status, SAT_INACCESSIBLE_FLAG | NO_AOS_LOS_FLAG );

  if( Satellite_Geostationary(sat_tle) )
	Set_SatFlag( sat_status, SAT_GEOSTATIONARY_FLAG | NO_AOS_LOS_FLAG );
  else
	Clear_SatFlag( sat_status, SAT_GEOSTATIONARY_FLAG );

  if( Satellite_Decayed(sat_tle, julian_utc) )
	Set_SatFlag( sat_status, SAT_DECAYED_FLAG | NO_AOS_LOS_FLAG );
  else
	Clear_SatFlag( sat_status, SAT_DECAYED_FLAG );

} /* End of Set_Acessibility_Flags() */

/*------------------------------------------------------------------*/

/*  Aos_Possible()
 *
 *  This function returns a 1 if the satellite of interest
 *  can ever rise above the horizon of the ground station
 */

  int
Aos_Possible( tle_t *sat_tle, observer_status_t  *obs_status )
{
  double incl, sma, apogee;

  incl = sat_tle->xincl;
  if (incl >= pio2)
	incl = pi-incl;

  sma = 331.25*exp( log(1./sat_tle->xno*twopi) * tothrd );
  apogee = sma*(1. + sat_tle->eo) - xkmper;

  if( (acos(xkmper/(apogee+xkmper))+incl) > fabs(obs_status->obs_geodetic.lat) )
	return 1;
  else
	return 0;

} /* End of Aos_Possible() */

/*------------------------------------------------------------------*/

/*  Satellite_Decayed()
 *
 *  This function returns a 1 if it appears that the
 *  satellite of interest has decayed at time 'julian_utc'
 */

  int
Satellite_Decayed( tle_t *sat_tle, double julian_utc )
{
  double
	drag,
	test;

  drag = ( sat_tle->xndt2o == 0. ? 1.E-8 : sat_tle->xndt2o ) * xmnpda;

  test = sat_tle->epoch + ( (0.072722052-sat_tle->xno)/(10.0*fabs(drag)) );

  if( test < julian_utc )
	return 1;
  else
	return 0;

} /* End of Satellite_Decayed() */

/*------------------------------------------------------------------*/

/*  Satellite_Geostationary()
 *
 *  This function returns a 1 if the satellite of
 *  interest appears to be in a geostationary orbit
 */

  int
Satellite_Geostationary( tle_t *sat_tle )
{
  if( fabs(sat_tle->xno - 0.0043751) < 1.3E-6 )
	return 1;
  else
	return 0;

} /* End of Satellite_Geostationary() */

/*------------------------------------------------------------------*/

/*  Convert_Satellite_Data()
 *
 *  Converts the strings in a raw two-line element set
 *  to their intended numerical values. No processing
 *  of these values is done, e.g. from deg to rads etc.
 *  This is done in the select_ephemeris() function.
 */

  void
Convert_Satellite_Data( char *tle_set, tle_t *tle )
{
  char buff[15];

  /** Decode Card 1 **/
  /* Satellite's catalogue number */
  strncpy( buff, &tle_set[2], 5 );
  buff[5] = '\0';
  tle->catnr = atoi(buff);

  /* International Designator for satellite */
  strncpy( tle->idesg, &tle_set[9], 8 );
  tle->idesg[8] = '\0';
  if( tle->idesg[7] == ' ' )
	tle->idesg[7] = '\0';
  if( tle->idesg[6] == ' ' )
	tle->idesg[6] = '\0';

  /* Satellite's epoch */
  strncpy( buff, &tle_set[18], 14 );
  buff[14] = '\0';
  tle->epoch = atof(buff);

  /* Satellite's First Time Derivative */
  strncpy( buff, &tle_set[33], 10 );
  buff[10] = '\0';
  tle->xndt2o = atof(buff);

  /* Satellite's Second Time Derivative */
  strncpy( buff, &tle_set[44], 1 );
  buff[1] = '.';
  strncpy( &buff[2], &tle_set[45], 5 );
  buff[7] = 'E';
  strncpy( &buff[8], &tle_set[50], 2 );
  buff[10] = '\0';
  tle->xndd6o = atof(buff);

  /* Satellite's bstar drag term */
  strncpy( buff, &tle_set[53], 1 );
  buff[1] = '.';
  strncpy( &buff[2], &tle_set[54], 5 );
  buff[7] = 'E';
  strncpy( &buff[8], &tle_set[59], 2 );
  buff[10] = '\0';
  tle->bstar = atof(buff);

  /* Element Number */
  strncpy( buff, &tle_set[64], 4 );
  buff[4] = '\0';
  tle->elset = atoi(buff);

  /** Decode Card 2 **/
  /* Satellite's Orbital Inclination (degrees) */
  strncpy( buff, &tle_set[77], 8 );
  buff[8] = '\0';
  tle->xincl = atof(buff);

  /* Satellite's RAAN (degrees) */
  strncpy( buff, &tle_set[86], 8 );
  buff[8] = '\0';
  tle->xnodeo = atof(buff);

  /* Satellite's Orbital Eccentricity */
  buff[0] = '.';
  strncpy( &buff[1], &tle_set[95], 7 );
  buff[8] = '\0';
  tle->eo = atof(buff);

  /* Satellite's Argument of Perigee (degrees) */
  strncpy( buff, &tle_set[103], 8 );
  buff[8] = '\0';
  tle->omegao = atof(buff);

  /* Satellite's Mean Anomaly of Orbit (degrees) */
  strncpy( buff, &tle_set[112], 8 );
  buff[8] = '\0';
  tle->xmo = atof(buff);

  /* Satellite's Mean Motion (rev/day) */
  strncpy( buff, &tle_set[121], 10 );
  buff[10] = '\0';
  tle->xno = atof(buff);

  /* Satellite's Revolution number at epoch */
  strncpy( buff, &tle_set[132], 5 );
  buff[5] = '\0';
  tle->revnum = atof(buff);

} /* Procedure Convert_Satellite_Data */

/*------------------------------------------------------------------*/

/*  Select_Ephemeris()
 *
 *  Selects the apropriate ephemeris type to be used
 *  for predictions according to the data in the TLE
 *  It also processes values in the tle set so that
 *  they are apropriate for the sgp4/sdp4 routines
 */

  void
Select_Ephemeris(tle_t *tle)
{
  double ao,xnodp,dd1,dd2,delo,temp,a1,del1,r1;

  /* Preprocess tle set */
  tle->xnodeo = Radians( tle->xnodeo );
  tle->omegao = Radians( tle->omegao );
  tle->xmo    = Radians( tle->xmo );
  tle->xincl  = Radians( tle->xincl );

  temp = twopi/xmnpda/xmnpda;

  tle->xno    *= temp*xmnpda;
  tle->xndt2o *= temp;
  tle->xndd6o *= temp/xmnpda;
  tle->bstar  /= ae;

  /* Period > 225 minutes is deep space */
  dd1 = (xke/tle->xno);
  dd2 = tothrd;
  a1 = pow(dd1, dd2);
  r1 = cos(tle->xincl);
  dd1 = (1.0-tle->eo*tle->eo);
  temp = ck2*1.5f*(r1*r1*3.0-1.0)/pow(dd1, 1.5);
  del1 = temp/(a1*a1);
  ao = a1*(1.0-del1*(tothrd*.5+del1*
		(del1*1.654320987654321+1.0)));
  delo = temp/(ao*ao);
  xnodp = tle->xno/(delo+1.0);

  /* Select a deep-space/near-earth ephemeris */
  if (twopi/xnodp/xmnpda >= .15625)
	tle->ephem = DEEP_SPACE;
  else
	tle->ephem = NEAR_EARTH;

  return;

} /* End of Select_Ephemeris() */

/*------------------------------------------------------------------*/

/*  Checksum_Good()
 *
 *  Calculates the checksum mod 10 of a line from a TLE set and
 *  returns 1 if it compares with checksum in column 68, else 0.
 *  tle_set is a character string holding the two lines read
 *  from a text file containing NASA format Keplerian elements.
 */

  int
Checksum_Good( char *tle_set )
{
  int i, check_digit, value, checksum = 0;

  for(i = 0; i < 68; i++)
  {
	if( (tle_set[i] >= '0') && (tle_set[i] <= '9') )
	  value = tle_set[i] - '0';
	else if( tle_set[i] == '-' )
	  value = 1;
	else
	  value = 0;

	checksum += value;

  } /* End for(i = 0; i < 68; i++) */

  checksum %= 10;
  check_digit = tle_set[68] - '0';

  return( checksum == check_digit );

} /* Function Checksums_Good */

/*------------------------------------------------------------------*/

/*  Good_Elements()
 *
 *  Carries out various checks on a TLE set to verify its validity.
 *  tle_set is a character string holding the two lines read
 *  from a text file containing NASA format Keplerian elements.
 */

  int
Good_Elements( char *tle_set )
{
  /* Verify checksum of both lines of a TLE set */
  if( !Checksum_Good(&tle_set[0]) || !Checksum_Good(&tle_set[69]) )
	return (0);

  /* Check the line number of each line */
  if( (tle_set[0] != '1') || (tle_set[69] != '2') )
	return (0);

  /* Verify that Satellite Number is same in both lines */
  if( strncmp(&tle_set[2], &tle_set[71], 5) != 0 )
	return (0);

  /* Check that various elements are in the right place */
  if(
	  (tle_set[ 23] != '.') ||
	  (tle_set[ 34] != '.') ||
	  (tle_set[ 80] != '.') ||
	  (tle_set[ 89] != '.') ||
	  (tle_set[106] != '.') ||
	  (tle_set[115] != '.') ||
	  (tle_set[123] != '.') ||
	  (strncmp(&tle_set[61], " 0 ", 3) != 0)
	)
	return (0);

  return(1);

} /* Function Good_Elements */

/*------------------------------------------------------------------*/

/*  Load_Observer_Data()
 *
 *  Loads a block of observer location data from satcom.obs file.
 */

  void
Load_Observer_Data( void )
{
  int
	eof, /* EOF flag memory */
	isr, /* Index used in searches */
	idx; /* Index for loops etc    */

  char
	home_dir[25],  /* Home directory name     */
	obs_fpath[50], /* File path to satcom.obs */
	name[25],      /* Location name buffer    */
	line[81];      /* Line buff for Load_Line */

  /* To convert longitude to +- 180 deg. */
  double lon;

  FILE *obs_file;

  /* Setup 'home' directory path */
  strncpy( home_dir, getenv("HOME"), 24 );
  home_dir[24] = '\0';

  /* Setup file path to satcom.tle (~/.satcom/satcom.tle) */
  snprintf( obs_fpath, 50, "%s/.satcom/satcom.obs", home_dir );

  /* Open observer file */
  obs_file = fopen( obs_fpath, "r" );
  if( obs_file == NULL )
  {
	perror( obs_fpath );
	exit( -1 );
  }

  /* Count the number of location entries */
  num_obs = eof = 0;

  /* Abort if EOF on first read from file */
  eof = Load_Line(line, 80, obs_file, -4);
  if( eof == EOF )
	Abort_On_Error( -4 );

  while( eof != EOF )
  {
	isr = 1;
	if( line[0] == '[' )
	{
	  while( (line[isr++] != ']') && (isr < 81) );
	  if( isr < 81 )
		num_obs++;
	}

	eof = Load_Line(line, 80, obs_file, -4);
  } /* End of while( eof != EOF ) */

  /* Abort if no location data found */
  if( num_obs == 0 )
	Abort_On_Error( -5 );

  rewind( obs_file );

  /* Allocate memory to observer database */
  obs_data = realloc( obs_data, sizeof(observer_status_t) * num_obs );
  if( obs_data == NULL )
	Abort_On_Error( -9 );

  /* Read a line from satcom.obs */
  Load_Line(line, 80, obs_file, -4);

  /* Read observer data into buffer */
  for( idx = 0; idx < num_obs; idx++ )
  {
	/* Look for a data block name */
	do
	{
	  isr = 1;
	  /* Look for a data block name in [...] */
	  if( line[0] == '[' )
	  {
		while( (line[isr++] != ']') && (isr < 81) );

		/* Enter location name */
		if( isr < 81 )
		{
		  line[isr-1] = '\0'; /* Kill ']' */
		  /* If name > 11 char, make ~ the last char. */
		  if( strlen(&line[1]) > 11 )
		  {
			line[11] = '~';
			line[12] = '\0';
		  }
		  strncpy(obs_data[idx].loc_name, &line[1], 11);
		  obs_data[idx].loc_name[11] = '\0';

		} /* End of if( isr < 81 ) */

	  } /* End of if( line[0] == '[' ) */

	  /* Read a line, return at EOF */
	  if( Load_Line(line, 80, obs_file, -4) == EOF )
		break;

	} /* End of do */
	/* Break if ']' found in line */
	while( (isr == 1) || (isr == 81) );

	Clear_Flag( HOME_LOC_ENTERED_FLAG );
	Clear_Flag( POSITION_ENTERED_FLAG );

	/* Set default values for missing items */
	obs_data[idx].obs_geodetic.lon = 0.;
	obs_data[idx].obs_geodetic.lat = 0.;
	obs_data[idx].obs_geodetic.hgt = 0.;
	obs_data[idx].grid_loc[0] = '\0';

	/* Read items from location data block till next block name */
	while( (line[0] != '[') )
	{
	  if( strncmp(line, "lon", 3) == 0 ) /* Read lon. */
	  {
		Set_Flag( POSITION_ENTERED_FLAG );
		obs_data[idx].obs_geodetic.lon = Radians( atof(&line[3]) );
	  }
	  else
		if( strncmp(line, "lat", 3) == 0 ) /* Read lat. */
		{
		  Set_Flag( POSITION_ENTERED_FLAG );
		  obs_data[idx].obs_geodetic.lat = Radians( atof(&line[3]) );
		}
		else
		  if( strncmp(line, "height", 6) == 0 ) /* Read height */
			obs_data[idx].obs_geodetic.hgt = atof(&line[6])/1.E3;
		  else
			if( strncmp(line, "QTHloc", 6) == 0 ) /* Read QTH loc. */
			{
			  Set_Flag( HOME_LOC_ENTERED_FLAG );
			  isr = 5;

			  /* Reject spaces after 'QTHloc' identifier */
			  while( (line[++isr] == KEY_SPACE) && (isr < 80) );
			  strncpy(obs_data[idx].grid_loc, &line[isr], 6);
			  obs_data[idx].grid_loc[6] = '\0';
			}

	  /* Read a line, return at EOF */
	  if( Load_Line(line, 80, obs_file, -4) == EOF )
		break;

	} /* End of while( line[0] != '[' ) */

	/* If Grid Locator and Position data are specified, */
	/* counter-check Grid Locator and exit on mismatch  */
	if( isFlagSet(HOME_LOC_ENTERED_FLAG) &&
		isFlagSet(POSITION_ENTERED_FLAG) )
	{
	  lon = Degrees( obs_data[idx].obs_geodetic.lon );
	  if( lon > 180 )
		lon -=360;

	  Position_to_Grid_Locator( lon,
		  Degrees(obs_data[idx].obs_geodetic.lat),
		  name );
	  /* Abort if Grid Locator does not agree with position */
	  if( strncmp( obs_data[idx].grid_loc, name, strlen(name)) != 0 )
		Abort_On_Error( -7 );
	}
	else /* If only Grid Locator is specified, calculate Position */
	  if( isFlagSet(HOME_LOC_ENTERED_FLAG) &&
		  isFlagClear(POSITION_ENTERED_FLAG) )
	  {
		Grid_Locator_to_Position( obs_data[idx].grid_loc,
			&obs_data[idx].obs_geodetic.lon,
			&obs_data[idx].obs_geodetic.lat );
		obs_data[idx].obs_geodetic.lon =
		  Radians(obs_data[idx].obs_geodetic.lon);
		obs_data[idx].obs_geodetic.lat =
		  Radians(obs_data[idx].obs_geodetic.lat);
	  }

	  else /* If only Position is specified, calculate Grid Locator */
		if( isFlagSet(POSITION_ENTERED_FLAG) &&
			isFlagClear(HOME_LOC_ENTERED_FLAG) )
		  Position_to_Grid_Locator( Degrees(obs_data[idx].obs_geodetic.lon),
			  Degrees(obs_data[idx].obs_geodetic.lat),
			  obs_data[idx].grid_loc );

	/* Do some sanity checking on data, abort on error */
	if( (fabs(obs_data[idx].obs_geodetic.lon) > twopi) ||
		(fabs(obs_data[idx].obs_geodetic.lat) >  pio2) ||
		(obs_data[idx].obs_geodetic.hgt > 15.) ||
		(obs_data[idx].obs_geodetic.hgt < -0.5) )
	  Abort_On_Error( -6 );

  } /* End of for( idx = 0; idx < num_obs; idx++ ) */

  fclose( obs_file );

} /* End of Load_Observer_Data() */

/*------------------------------------------------------------------*/

/* Load_Observer_Set()
 *
 *  Loads an observer's location data
 */

  void
Load_Observer_Set( char *loc_name, observer_status_t *obs_status )
{
  /* index for loops etc */
  int idx;

  /* Clear satellite name string */
  strcpy( obs_status->loc_name, "" );

  /* Look for a matching observer name in database */
  for( idx = 0; idx < num_obs; idx++ )
	if( strncmp(obs_data[idx].loc_name, loc_name, strlen(loc_name)) == 0 )
	{
	  /* Load TLE set on match */
	  *obs_status = obs_data[idx];
	  idx_obs = idx;
	  break;
	}

  /* If no matching name, load 1st observer set as default */
  if( strcmp(obs_status->loc_name, "") == 0 )
  {
	*obs_status = obs_data[0];
	idx_obs = 0;
  }

} /* End of Load_Observer_Set() */

/*------------------------------------------------------------------*/

/*  New_Observer()
 *
 *  Does the pre-processing needed after
 *  a change of satellite or observer
 */

void
New_Observer(
	tle_t *sat_tle,
	satellite_status_t *sat_status,
	observer_status_t  *obs_status )
{
  /* Copy satellite name into status structure */
  strncpy(sat_status->name, sat_tle->name, 11);
  if( strlen(sat_tle->name) > 11 )
	sat_status->name[10] = '~';

  sat_status->name[11] = '\0';

  /* Clear elevation status, needed in main() */
  /* to determine horizon crossing events     */
  sat_status->elev = 0.;

  /* Clear all satellite flags */
  Clear_SatFlag( sat_status, ALL_FLAGS );

  /* Set the rotors' tracking resolution according to orbit */
  if( sat_tle->ephem == NEAR_EARTH )
	sat_status->resol = LOW_RESOL;
  else
	if( Satellite_Geostationary(sat_tle) )
	  sat_status->resol = HIGH_RESOL;
	else
	  sat_status->resol = MED_RESOL;

  /* Check whether satellite can ever have an AOS/LOS */
  Set_Accessibility_Flags( sat_tle, obs_status, sat_status );

}/* End of New_Observer() */

/*------------------------------------------------------------------*/

/*  Save_Location()
 *
 *  Saves user-entered location data in ~/.satcom/satcom.obs
 */

  void
Save_Location( observer_status_t obs_status )
{
  char
	obs_fpath[50], /* Path to satcom.obs file */
	home_dir[25],  /* Path to home directory  */
	conv[12];      /* Conversions buffer      */

  FILE *obs_file;

  /* Setup 'home' directory path */
  strncpy( home_dir, getenv("HOME"), 24 );
  home_dir[24] = '\0';

  /* Setup file path to satcom.tle (~/.satcom/satcom.tle) */
  snprintf( obs_fpath, 50, "%s/.satcom/satcom.obs", home_dir );

  /* Open observer file */
  obs_file = fopen( obs_fpath, "a" );
  if( obs_file == NULL )
  {
	perror( obs_fpath );
	exit( -1 );
  }

  fputs( "\n\n[", obs_file );
  fputs( obs_status.loc_name, obs_file );
  fputs( "]", obs_file );

  snprintf( conv, 12, "%10.5f", Degrees(obs_status.obs_geodetic.lon) );
  fputs( "\nlon  ", obs_file );
  fputs( conv, obs_file );

  snprintf( conv, 12, "%10.5f", Degrees(obs_status.obs_geodetic.lat) );
  fputs( "\nlat  ", obs_file );
  fputs( conv, obs_file );

  snprintf( conv, 8, "%d", (int)(obs_status.obs_geodetic.hgt*1000.0) );
  fputs( "\nheight  ", obs_file );
  fputs( conv, obs_file );

  fputs( "\nQTHloc  ", obs_file );
  fputs( obs_status.grid_loc, obs_file );

  fputs( "\nDescr ", obs_file );
  fputs( obs_status.loc_name, obs_file );

  fclose( obs_file );

} /* End of Save_Location() */

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure with err_num
 *  Lines beginning with a '#' are ignored as comments
 */

  int
Load_Line( char *buff, int max_chr, FILE *pfile, int err_num )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc */

  num_chr = 0;

  /* Ignore commented lines, white spaces and eol/cr */
  if( (chr = fgetc(pfile)) == EOF )
	return( EOF );
  while(
	  (chr == '#')       ||
	  (chr == KEY_SPACE) ||
	  (chr == KEY_LFD)   ||
	  (chr == KEY_RET) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != KEY_RET) && (chr != KEY_LFD) )
	  if( (chr = fgetc(pfile)) == EOF )
		return( EOF );

	/* Dump any CR/LF remaining (compatibility with stupid DOS format) */
	while( (chr == KEY_RET) || (chr == KEY_LFD) )
	  if( (chr = fgetc(pfile)) == EOF )
		return( EOF );
  } /* End of while( (chr == '#') || ... */

  while( num_chr < max_chr )
  {
	/* If LF/CR reached before filling buffer, return */
	if( (chr == KEY_LFD) || (chr == KEY_RET) )
	  break;

	buff[num_chr++] = chr;

	if( (chr = fgetc(pfile)) == EOF )
	  return( EOF );

	/* Abort if end of line not reached at max_chr. */
	/* max_char is max number of characters to read */
	if( (num_chr == max_chr) &&
		((chr != KEY_LFD) && (chr != KEY_RET)) )
	  Abort_On_Error( err_num );

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( 0 );

} /* End of Load_Line() */

/*------------------------------------------------------------------*/

