/*  earthtrack.c
 *
 *  Functions to calculate the range circle of a satellite
 *  and to display its position and footprint using xplanet.
 *  The code in this file was based on the earthtrack client
 *  from the PREDICT package by John Magliacane.
 */

/*
 *  satcom: A ncurses application to track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "satcom.h"

extern FILE
  *marker_fd,   /* Marker file for xplanet with sat/obs location */
  *greatarc_fd; /* Great circle arc co-ordinates for xplanet     */

/* Name strings for above files */
extern char
  xplanet_markerfile[50],
  xplanet_greatarcfile[50],
  xplanet_configfile[50],
  xplanet_command[100];



/*  Arc_Cos()
 *
 *  This function implements the arccosine function,
 *  returning a value between 0 and two pi.
 */

  double
Arc_Cos( double x, double y )
{

  double
	result = 0.0,
    fraction;

  fraction = x / y;

  if( (x > 0.0) && (y > 0.0) )
	result = acos( fraction );

  if( (x < 0.0) && (y > 0.0) )
	result = acos(fraction);

  if( (x < 0.0) && (y < 0.0) )
	result = pi + acos(fraction);

  if( (x > 0.0) && (y < 0.0) )
	result = pi + acos( fraction );

  return result;

} /* End of Arc_Cos() */

/*------------------------------------------------------------------------*/

/*  Visibility_to_Color()
 *
 *  This function generates a character string based on the
 *  visibility information returned by satcom.  It is used
 *  to set the colors of the markers and range circles
 *  plotted by xearth and xplanet.
 */

  void
Visibility_to_Color( satellite_status_t *sat_status, char *color )
{
  if( isSatFlagClear(sat_status, SAT_ECLIPSED_FLAG) )
	strcpy( color, "color=white # In sunlight" );
  else
	strcpy( color, "color=blue # In darkness" );

  if( sat_status->elev > 0.0 )
	strcpy( color, "color=yellow # Accessible" );

} /* End of Visibility_to_Color() */

/*------------------------------------------------------------------------*/

/*  Range_Lat_Lon()
 *
 *  Calculates lat/long co-ordinates for a given
 *  azimuth angle in the satellite's range circle
 */

  void
Range_Lat_Lon( satellite_status_t *sat_status, int azim,
	double *range_lat, double *range_lon )
{
  double
	ssp_lat,  /* Sub-satellite point latitude     */
	ssp_lon,  /* Sub-satellite point longitude    */
	beta,     /* Footprint factor in calculations */
	azimuth,  /* Azimuth angle in range circle    */
	denom,
	numer;

  /* Convert to radians */
  ssp_lat = Radians( sat_status->ssplat );
  ssp_lon = Radians( sat_status->ssplon );
  beta = (0.5 * sat_status->ftprnt) / xkmper;

  azimuth = Radians( (double)azim );
  *range_lat = asin( sin(ssp_lat) * cos(beta) +
	  cos(azimuth)  * sin(beta) * cos(ssp_lat) );
  numer = cos(beta) - ( sin(ssp_lat) * sin(*range_lat) );
  denom = cos(ssp_lat) * cos(*range_lat);

  if( azim == 0 && (beta > pio2 - ssp_lat) )
	*range_lon = ssp_lon + pi;
  else
	if( azim == 180 && (beta > pio2 + ssp_lat) )
	  *range_lon = ssp_lon + pi;
	else
	  if( fabs(numer / denom) > 1.0 )
		*range_lon = ssp_lon;
	  else
	  {
		if( (180 - azim) >= 0)
		  *range_lon = ssp_lon - Arc_Cos( numer, denom );
		else
		  *range_lon = ssp_lon + Arc_Cos( numer, denom );
	  }

  while( *range_lon < 0.0 )
	*range_lon += twopi;

  while( *range_lon > twopi )
	*range_lon -= twopi;

  *range_lat = Degrees( *range_lat );
  *range_lon = Degrees( *range_lon );

} /* End of Range_Lat_Lon() */

/*------------------------------------------------------------------------*/

/*  Range_Circle()
 *
 *  This function generates a sequence of latitude and
 *  longitude positions used to plot range circles of
 *  satellites based on the spacecraft's sub-satellite
 *  latitude, longitude, and footprint.  The visibility
 *  information is used to set the range circle to an
 *  appropriate color.  Output is written to ~/.greatarcfile,
 *  and read and processed by xplanet.
 */

  void
Range_Circle( satellite_status_t *sat_status )
{
  /* Latitude and longitude of range   */
  /* circle points for xplanet to plot */
  double
	range_lat,
	range_lon;

  /* Azimuth angle in range circle */
  int azim;

  /* Color to be used by xplanet for plots */
  char color[33];

  azim = 0;
  while( azim < 360 )
  {
	/* Write range circle data to greatarcfile */
	Range_Lat_Lon( sat_status, azim++, &range_lat, &range_lon );
	fprintf( greatarc_fd, "%8.3f %8.3f ", range_lat, range_lon );

	/* Write range circle data to greatarcfile */
	Range_Lat_Lon( sat_status, azim++, &range_lat, &range_lon );
	fprintf( greatarc_fd, "%8.3f %8.3f ", range_lat, range_lon );

	/* Write color for range circle */
	Visibility_to_Color( sat_status, color );
	fprintf( greatarc_fd, "%s\n", color );

  } /* while( azim < 360 ) */

} /* End of Range_Circle() */

/*------------------------------------------------------------------------*/

/*  Display_Satellite
 *
 *  Displays satellite tracks using xearth or xplanet
 */

  int
Display_Satellite( satellite_status_t *sat_status,
	observer_status_t *obs_status )
{
  int
	flags,  /* Satellite flags (visibility/accessibility) */
	error;  /* Error flag for system calls */

  char color[33]; /* Color string for xplanet */

  /* The following statics are used to detect changes */
  /* in satellite status and to trigger 'xplanet'     */
  static char current_sat[25]="";

  static int
	current_ssplat = 0,
	current_ssplon = 0,
	current_flags  = 0;

  /* Compare the static values with satellite */
  /* status and return if there is no change  */
  flags = (sat_status->elev > 0.0) |
	(sat_status->flags & SAT_ECLIPSED_FLAG);

  if( (strcmp(current_sat, sat_status->name) != 0) ||
	  (current_ssplat != (int)sat_status->ssplat)  ||
	  (current_ssplon != (int)sat_status->ssplon)  ||
	  (current_flags  != flags) ||
	  isFlagClear(XPLANET_RUN) )
  {
	strcpy( current_sat, sat_status->name );
	current_ssplat = (int)sat_status->ssplat;
	current_ssplon = (int)sat_status->ssplon;
	current_flags  = flags;
  }
  else
	return(0);

  error = 0;

  /* Open xplanet file descriptors */
  marker_fd   = fopen( xplanet_markerfile,   "w" );
  greatarc_fd = fopen( xplanet_greatarcfile, "w" );

  /* Calculate range circle co-ordinates */
  Range_Circle( sat_status );

  /* Determine color for xplanet to use */
  Visibility_to_Color( sat_status, color );

  /* Enter satellite position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s %03d/%02d\" %s\n",
	  sat_status->ssplat, sat_status->ssplon,
	  sat_status->name, (int)sat_status->azim,
	  (int)sat_status->elev, color );

  /* Enter observer position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s\"\n",
	  Degrees(obs_status->obs_geodetic.lat),
	  Degrees(obs_status->obs_geodetic.lon),
	  obs_status->loc_name );

  fclose( greatarc_fd );
  fclose( marker_fd );

  /* Invoke xplanet */
  error = 0;
  if( isFlagClear(XPLANET_RUN) )
  {
	Set_Flag( XPLANET_RUN );
	error = system( xplanet_command );
  }

  /* On error delete xplanet's files */
  if( error )
  {
	unlink( xplanet_greatarcfile );
	unlink( xplanet_markerfile );
	unlink( xplanet_configfile );
  }

  return( error );

} /* End of Display_Satellite() */

/*------------------------------------------------------------------------*/

/* Config_Xplanet()
 *
 * Writes an xplanet config file for xsatcom
 */

  int
Config_Xplanet( void )
{
  /* xplanet config file desc. */
  FILE *config_fd;

  /* Open xplanet config file descriptor */
  config_fd = fopen( xplanet_configfile, "w" );
  if( (config_fd == NULL) )
  {
	perror( "xplanet config file" );
	return(-1);
  }

  /* Write the xplanet config file */
  fprintf( config_fd,
	  "# xplanet config file produced by satom\n"
	  "shade=30\n"
	  "text_color={255,0,0}\n"
	  "twilight=6\n" );

  /* Enable plot of current sat */
  fprintf( config_fd,
	  "arc_file=%s\n"
	  "marker_file=%s\n",
	  xplanet_greatarcfile, xplanet_markerfile );

  fclose( config_fd );

  return( 0 );

} /* Config_Xplanet() */

/*------------------------------------------------------------------------*/

