/*  comms.c
 *
 *  Functions to send and receive data from the
 *  Yaesu GS-232 Az-El antenna rotor Controller
 *  and the Yaesu FT-847 transceiver (CAT control)
 */

/*
 *  Satcom: A Ncurses Application To Track satellites using the
 *  NORAD SGP4/SDP4 orbit calculation routines. The moon and sun
 *  are also tracked.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "satcom.h"

/*  Rotor_Send_String()
 *
 *  Function to send a command or data string to
 *  the Yaesu GS-232 Az-El antenna rotor Controller
 */

  void
Rotor_Send_String( char *strn )
{
  int idx; /* Index for loops etc */
  char ret = KEY_RET; /* RETURN keycode */

  for( idx = 0; idx < strlen( strn ); idx++ )
	Write_Rotor_Serial( &strn[ idx ] );

  /* Send a RETURN after string */
  Write_Rotor_Serial( &ret );

  /* Wait for CR response from controller */
  usleep(10000);

} /* End of Send_String */

/*------------------------------------------------------------------------*/

/*  Rotor_Read_String()
 *
 *  Function to read a data string from the
 *  Yaesu GS-232 Az-El antenna rotor Controller
 */

  void
Rotor_Read_String( char *strn )
{
  char chr;
  int idx = 0;

  /* Read characters until CR is received. Ignore LF */
  while( ((chr = Read_Rotor_Serial()) != -1) && (chr != KEY_RET) )
	if( chr != KEY_LFD )
	  strn[ idx++ ] = chr;

  strn[ idx ]= '\0';

} /* End of Read_String */

/*------------------------------------------------------------------------*/

/*  Rotor_Send_Direction()
 *
 *  Send azimuth and elevation request to GS232 controller
 */

  void
Rotor_Send_Direction( int azim, int elev )
{
  /* String to enter Azim/Elev data  */
  /* for sending to GS232 controller */
  char strn[9];

  /* Correct azimuth value with the offset constant,   */
  /* which is due to some difference between the G5500 */
  /* Az/El rotor position transducer output volts and  */
  /* the volts expected by the GS232 firmware. (It now */
  /* seems that the GS232 treats 0 volts as Az 180 deg)*/
  azim += AZIM_OFFSET;

  /* Enter position data in 'strn' in the */
  /* 'W' command's format for sending out */
  strn[0] = 'W';
  snprintf( &strn[1], 4, "%03d", azim );
  strn[4] = ' ';
  snprintf( &strn[5], 4, "%03d", elev );
  Rotor_Send_String( strn );

} /* End of Rotor_Send_Direction() */


/*------------------------------------------------------------------------*/

/*  Rotor_Read_Direction()
 *
 *  Read azimuth and elevation data from GS232 controller
 */

  void
Rotor_Read_Direction( int *azim, int *elev )
{
  /* String to enter Azim/Elev data */
  /* read from GS232 controller     */
  char strn[11];

  /* Temporary buffer for cloning Az/El from strn */
  char buff[6];

  /* Request and read rotor position string */
  Rotor_Send_String( "C2" );
  Rotor_Read_String( strn );

  /* Return parking values for Az/El if serial read fails */
  if( isFlagSet(ROTOR_READ_FAIL_FLAG) )
  {
	*azim = AZIM_PARKING;
	*elev = ELEV_PARKING;
  }
  else
  {
	/* Clone Azimuth value */
	strncpy( buff, strn, 5 );
	buff[5] = '\0';
	*azim = Modulus( atoi(buff) + AZIM_OFFSET, 360 );

	/* Clone Elevation value */
	strncpy( buff, &strn[5], 5 );
	buff[5] = '\0';
	*elev = atoi( buff );

  }  /* End of if( isFlagSet(ROTOR_READ_FAIL_FLAG) ) */

} /* End of Rotor_Read_Direction() */

/*------------------------------------------------------------------------*/

/*  Read_Tcvr_Status()
 *
 *  Reads the transceiver's status (Tx/Rx freq, mode,
 *  po/s-meter etc) from the Yaesu FT-847 transceiver.
 */

  void
Read_Tcvr_Status( tcvr_status_t *tcvr_status )
{
  /* A 5-char string for sending and receiving */
  /* 5-byte strings to and from the transceiver */
  unsigned char cmnd_parm[5];

  int idx; /* Index for loops etc */

  bzero( cmnd_parm, 5 );

  /* Read Rx status */
  cmnd_parm[4] = RECEIVER_STATUS;
  Write_Tcvr_Serial( cmnd_parm );
  Read_Tcvr_Serial( cmnd_parm, 1 );
  if( isFlagSet(TCVR_READ_FAIL_FLAG) )
	return;
  tcvr_status->rx_status = cmnd_parm[0];

  /* Read Rx mode and frequency */
  cmnd_parm[4] = READ_SAT_RX_VFO;
  Write_Tcvr_Serial( cmnd_parm );
  Read_Tcvr_Serial( cmnd_parm, 5 );
  if( isFlagSet(TCVR_READ_FAIL_FLAG) )
	return;

  /* Decode frequency data in 10xHz */
  tcvr_status->rx_freq = 0;
  for( idx = 0; idx < 4; idx++ )
  {
	tcvr_status->rx_freq += (cmnd_parm[idx] & 0xF0) >> 4;
	tcvr_status->rx_freq *= 10;
	tcvr_status->rx_freq += cmnd_parm[idx] & 0x0F;
	tcvr_status->rx_freq *= 10;
  }
  tcvr_status->rx_freq /= 10;

  /* Enter operating mode (LSB/USB etc) */
  tcvr_status->rx_mode = cmnd_parm[4];

  /* Read Tx status */
  cmnd_parm[4] = TRANSMIT_STATUS;
  Write_Tcvr_Serial( cmnd_parm );
  Read_Tcvr_Serial( cmnd_parm, 1 );
  if( isFlagSet(TCVR_READ_FAIL_FLAG) )
	return;
  tcvr_status->tx_status = cmnd_parm[0];

  /* Read Tx mode and frequency */
  cmnd_parm[4] = READ_SAT_TX_VFO;
  Write_Tcvr_Serial( cmnd_parm );
  Read_Tcvr_Serial( cmnd_parm, 5 );
  if( isFlagSet(TCVR_READ_FAIL_FLAG) )
	return;

  /* Decode frequency data in 10xHz */
  tcvr_status->tx_freq = 0;
  for( idx = 0; idx < 4; idx++ )
  {
	tcvr_status->tx_freq += (cmnd_parm[idx] & 0xF0) >> 4;
	tcvr_status->tx_freq *= 10;
	tcvr_status->tx_freq += cmnd_parm[idx] & 0x0F;
	tcvr_status->tx_freq *= 10;
  }
  tcvr_status->tx_freq /= 10;

  /* Enter operating mode (LSB/USB etc) */
  tcvr_status->tx_mode = cmnd_parm[4];

} /* End of Read_Tcvr_Status() */

/*------------------------------------------------------------------------*/

/*  Write_Tcvr_Status()
 *
 *  Writes the transceiver status (Tx/Rx freq,
 *  mode) to the Yaesu FT-847 transceiver.
 */

  void
Write_Tcvr_Status( tcvr_status_t *tcvr_status )
{
  /* A 5-char string for sending and receiving */
  /* 5-byte strings to and from the transceiver */
  unsigned char cmnd_parm[5];

  /* Buffer used for converting freq. to string */
  char freq_buff[9];

  int idx; /* Index for loops etc */

  bzero( cmnd_parm, 5 );

  /* Write Rx operating mode if it has changed */
  if( tcvr_status->rx_mode != tcvr_status->rx_mode_ref )
  {
	tcvr_status->rx_mode_ref = tcvr_status->rx_mode;
	cmnd_parm[0] = tcvr_status->rx_mode;
	cmnd_parm[4] = WRITE_SAT_RX_MODE;
	Write_Tcvr_Serial( cmnd_parm );
  }

  /* Write Tx operating mode if it has changed */
  if( tcvr_status->tx_mode != tcvr_status->tx_mode_ref )
  {
	tcvr_status->tx_mode_ref = tcvr_status->tx_mode;
	cmnd_parm[0] = tcvr_status->tx_mode;
	cmnd_parm[4] = WRITE_SAT_TX_MODE;
	Write_Tcvr_Serial( cmnd_parm );
  }

  /* Set Rx frequency */
  snprintf( freq_buff, 9, "%08d", (tcvr_status->rx_freq) );
  for( idx = 0; idx < 4; idx++ )
  {
	cmnd_parm[idx]  = (freq_buff[2*idx]   - '0') << 4;
	cmnd_parm[idx] |= (freq_buff[2*idx+1] - '0');
  }
  cmnd_parm[4] = WRITE_SAT_RX_VFO;
  Write_Tcvr_Serial( cmnd_parm );

  /* Set Tx frequency */
  snprintf( freq_buff, 9, "%08d", (tcvr_status->tx_freq) );
  for( idx = 0; idx < 4; idx++ )
  {
	cmnd_parm[idx]  = (freq_buff[2*idx]   - '0') << 4;
	cmnd_parm[idx] |= (freq_buff[2*idx+1] - '0');
  }
  cmnd_parm[4] = WRITE_SAT_TX_VFO;
  Write_Tcvr_Serial( cmnd_parm );

} /* End of Write_Tcvr_Status() */

/*------------------------------------------------------------------------*/

