/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  serial.c
 *
 *  Functions to handle the serial port
 */

#include "rwgps.h"

/*  'open_serial()'
 *
 *  Opens Serial Port device, returns the file
 *  descriptor on success or -1 on error
 */

static int serial_fd; /* Serial port File descriptor  */
static int err_num;	  /* Records system 'errno' */
static struct termios old_options; /* Original serial port options */

  void
open_serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */

  /* Open Serial device, abort on error */
  serial_fd = open( SERIAL_DEV, O_RDWR | O_NOCTTY );
  if( serial_fd < 0 )
  {
	perror( "rwgps: open serial device" );
	exit( -1 );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  if( fcntl( serial_fd, F_SETLK, &lockinfo ) < 0 )
  {
	fcntl( serial_fd, F_GETLK, &lockinfo );
	printf( "rwgps: /dev/"SERIAL_DEV": Device already locked by pid %d%s\n",
		lockinfo.l_pid,
		" - aborting" );
	exit( -2 );
  }

  /* Save the current serial port options */
  tcgetattr( serial_fd, &old_options );

  /* Read the current serial port options */
  tcgetattr( serial_fd, &new_options );

  /* Set options for 'raw' I/O mode */
  cfmakeraw( &new_options );

  /* Set the i/o baud rates to 9600 */
  cfsetispeed( &new_options, B9600 );
  cfsetospeed( &new_options, B9600 );

  /* Setup read() Timeout to 12 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 120;

  /* Enable parity */
  new_options.c_cflag |= ( PARENB | PARODD );

  /* Setup the new options for the port */
  tcsetattr( serial_fd, TCSAFLUSH, &new_options );

} /* End of 'open_serial()' */

/*-------------------------------------------------------------------------*/

/*  'write_serial()'
 *
 *  Writing Data to the Serial Port
 */

  void
write_serial( unsigned char *ch )
{
  if( write( serial_fd, ch, 1 ) < 1 ) /* Error condition */
  {
	err_num = errno;
	set_mode( MD_WRITEFAIL | MD_PROGHALT );
	perror( "rwgps: write()" );
  }

} /* End of 'write_serial()' */

/*-------------------------------------------------------------------------*/

/*  'read_serial()'
 *
 *  Reading Data from the Serial Port
 */

  void
read_serial( unsigned char *ch )
{
  if( read( serial_fd, ch, 1 ) < 1 ) /* Error condition */
  {
	err_num = errno;
	set_mode( MD_READFAIL | MD_PROGHALT );
	perror( "rwgps: read()" );
  }

} /* End of 'read_serial()' */

/*-------------------------------------------------------------------------*/

/*  'close_serial()'
 *
 *  Restore old options and close the Serial port
 */

  void
close_serial( void )
{
  tcsetattr( serial_fd, TCSANOW, &old_options );
  close( serial_fd );

} /* End of 'close_serial()' */

/*-------------------------------------------------------------------------*/

/*  'error_num()'
 *
 *  Returns the errno produced by read() and write()
 */

  int
error_num( void )
{
  return( err_num );
}

/*-------------------------------------------------------------------------*/

/*  'clear_error()'
 *
 *  Clears error_num produced by read() and write()
 */

  void
clear_error( void )
{
  err_num = 0;
}

/*-------------------------------------------------------------------------*/

