/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  screen.c
 *
 *  New Curses display routines for rwgps
 */

#include "rwgps.h"

/* chtype string constants for the displays */
static chtype  ok[] = { ' ', NORMAL | 'O', NORMAL | 'K', '\0' };
static chtype nok[] = { ALARM | '!', ALARM | 'O', ALARM | 'K', '\0' };
static chtype yes[] = { NORMAL | 'Y', NORMAL | 'E', NORMAL | 'S', '\0' };
static chtype  no[] = { ALARM | 'N', ALARM | 'O', '\0' };

/*  'ncscreen_root()'
 *
 *  Function to creat the root window
 */

  void
ncscreen_root( void )
{
  /* Box and add titles on stdscr */
  attrset( LABEL );
  box( stdscr, ACS_VLINE, ACS_HLINE );
  mvaddstr( 0, 18, " ROCKWELL MICROTRACKER LP(tm) GPS RECEIVER " );

  /* Commands 'toolbar' at the bottom of stdscr */
  attrset(COMMAND );
  mvaddstr(23, 4,
	  "SelfTest    Warm Start    Main Screen    Set Clock    Power    Halt    Quit");

  attrset(KEYSTROKE);
  mvaddstr(23,  1, "<T>"); mvaddstr(23, 13, "<W>"); mvaddstr(23, 27, "<M>");
  mvaddstr(23, 42, "<C>"); mvaddstr(23, 55, "<P>"); mvaddstr(23, 64, "<H>");
  mvaddstr(23, 72, "<Q>");

  refresh(); /* Done with root window */

} /* End of 'ncscreen_root()' */

/*------------------------------------------------------------------------*/

/*  'ncscreen_message101()'
 *
 *  Function to display the Built-In-Test (BIT) results
 *  provided by the GPS Receiver in Message 101
 */

  void
ncscreen_message101( message101_t * message101 )
{
  int  idx;     /* Index for loops etc */
  int  channel; /* Counter for receiver channel number */

  WINDOW * receiver_health_win; /* Receiver Health Test Results  */
  WINDOW * channel_test_win;    /* Receiver Channel Test Results */

  receiver_health_win  = newwin( 11, 60,  1, 1 );
  channel_test_win     = newwin( 11, 60, 12, 1 );

  /*
   *  The following lines enter various labels and receiver
   *  health test data into the receiver health window
   */

  wattrset( receiver_health_win, ITEM );
  mvwaddstr( receiver_health_win, 1, 1,
	  "   Receiver System        EEPROM Programming Errors" );
  mvwaddstr( receiver_health_win, 2, 1,
	  "        ROM Health           EEPROM Checksum Errors" );
  mvwaddstr( receiver_health_win, 3, 1,
	  "     Low Order RAM            Low Order RAM Pattern" );
  mvwaddstr( receiver_health_win, 4, 1,
	  "    High Order RAM           High Order RAM Pattern" );
  mvwaddstr( receiver_health_win, 5, 1,
	  "    RAM Addressing         Preprocessor RAM Pattern" );
  mvwaddstr( receiver_health_win, 6, 1,
	  "        SPI Device       Serial Transmission Errors" );
  mvwaddstr( receiver_health_win, 7, 1,
	  " Interrupt Control          Message Checksum Errors" );
  mvwaddstr( receiver_health_win, 8, 1,
	  "   Low Power Timer      Auxiliary Port Trans Errors" );
  mvwaddstr( receiver_health_win, 9, 1,
	  "  Options Register          Software Version Number" );

  wattrset(    receiver_health_win, message101->eeprom_prog_err ? ALARM : NORMAL );
  mvwprintw(   receiver_health_win, 1, 53, "%6d", message101->eeprom_prog_err );
  mvwaddchstr( receiver_health_win, 1, 20,
	  ( message101->bit_summary & 0x01 ) ? nok : ok );

  wattrset(   receiver_health_win, message101->eeprom_cksum_err ? ALARM : NORMAL );
  mvwprintw(  receiver_health_win, 2, 53, "%6d", message101->eeprom_cksum_err );
  mvwaddchstr(receiver_health_win, 2, 20,
	  (message101->bit_summary & 0x0100 ) ? nok : ok);

  mvwprintw(   receiver_health_win, 3, 53, "0X%4X", message101->low_ram_test );
  mvwaddchstr( receiver_health_win, 3, 20,
	  ( message101->bit_summary & 0x0200 ) ? nok : ok );

  mvwprintw(   receiver_health_win, 4, 53, "0X%4X", message101->high_ram_test );
  mvwaddchstr( receiver_health_win, 4, 20,
	  ( message101->bit_summary & 0x0400 ) ? nok : ok );

  mvwprintw(   receiver_health_win, 5, 53, "0X%4X", message101->preproc_ram_test );
  mvwaddchstr( receiver_health_win, 5, 20,
	  ( message101->bit_summary & 0x1000 ) ? nok : ok );

  wattrset(    receiver_health_win, message101->serial_errors ? ALARM : NORMAL );
  mvwprintw(   receiver_health_win, 6, 53, "%6d", message101->serial_errors );
  mvwaddchstr( receiver_health_win, 6, 20,
	  ( message101->bit_summary & 0x0800 ) ? nok : ok );

  wattrset(    receiver_health_win, message101->checksum_errors ? ALARM : NORMAL );
  mvwprintw(   receiver_health_win, 7, 53, "%6d", message101->checksum_errors );
  mvwaddchstr( receiver_health_win, 7, 20,
	  ( message101->interface_test & 0x01 ) ? nok : ok );

  wattrset(    receiver_health_win, message101->aux_port_errors ? ALARM : NORMAL );
  mvwprintw(   receiver_health_win, 8, 53, "%6d", message101->aux_port_errors );
  mvwaddchstr( receiver_health_win, 8, 20,
	  ( message101->interface_test & 0x0100 ) ? nok : ok );

  mvwprintw(   receiver_health_win, 9, 53, "%6.2f", message101->software_version );
  mvwaddchstr( receiver_health_win, 9, 20,
	  ( message101->interface_test & 0x0200 ) ? nok : ok );

  /* Draw some lines to box the results and add labels */
  wattrset(  receiver_health_win, LABEL );
  box(       receiver_health_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( receiver_health_win, 0, 26, " RECEIVER TEST RESULTS " );
  mvwaddstr( receiver_health_win, 0,  2, " RECEIVER STATE " );
  draw_vline(receiver_health_win, 0, 19, 10);
  draw_vline(receiver_health_win, 0, 23, 10);
  draw_vline(receiver_health_win, 0, 52, 10);

  /*
   *  The following lines enter various labels and receiver
   *  channel health test data into the channel test window
   */

  /* Add various labels to the receiver channel test results window */
  wattrset(  channel_test_win, ITEM );
  mvwaddstr( channel_test_win, 1, 8, "        Receiver Channel Number" );
  mvwaddstr( channel_test_win, 2, 8, " Preprocessor Self-Test Results" );
  mvwaddstr( channel_test_win, 3, 8, "  Preprocessor Wrap-Around Test" );
  mvwaddstr( channel_test_win, 4, 8, "          Code VCO Test Results" );
  mvwaddstr( channel_test_win, 5, 8, "       Carrier VCO Test Results" );
  mvwaddstr( channel_test_win, 6, 8, "    Code Generator Test Results" );
  mvwaddstr( channel_test_win, 7, 8, "  Signal Injection Test Results" );

  /* Enter channel number and various test results */
  wattrset( channel_test_win, NORMAL );
  channel = 1;
  for ( idx = 0; idx <= 16; idx += 4 )
  {
	mvwprintw( channel_test_win, 1, 40 + idx, "%2d", channel );

	mvwaddchstr( channel_test_win, 2, 40 + idx,
		( message101->preproc_test & ((unsigned int)0x01 << (channel-1))) ? nok : ok );
	mvwaddchstr( channel_test_win, 3, 40 + idx,
		( message101->preproc_test & ((unsigned int)0x0100 << (channel-1))) ? nok : ok );
	mvwaddchstr( channel_test_win, 4, 40 + idx,
		( message101->vco_test & ((unsigned int)0x01 << (channel-1))) ? nok : ok );
	mvwaddchstr( channel_test_win, 5, 40 + idx,
		( message101->vco_test & ((unsigned int)0x0100 << (channel-1))) ? nok : ok );
	mvwaddchstr( channel_test_win, 6, 40 + idx,
		( message101->gen_inject_test & ((unsigned int)0x01 << (channel-1))) ? nok : ok );
	mvwaddchstr( channel_test_win, 7, 40 + idx,
		( message101->gen_inject_test & ((unsigned int)0x0100 << (channel-1))) ? nok : ok );

	channel++;
  }

  /* Enter the Software Configuration information */
  wattrset( channel_test_win, ITEM );
  mvwaddstr( channel_test_win, 9, 1,
	  " DIFF GPS:     SATL INFO:     POWER MANAGE:     NMEA:" );
  mvwaddchstr( channel_test_win, 9, 11,
	  ( message101->software_config & 0x01 ) ? yes : no );
  mvwaddchstr( channel_test_win, 9, 26,
	  ( message101->software_config & 0x02 ) ? yes : no );
  mvwaddchstr( channel_test_win, 9, 44,
	  ( message101->software_config & 0x04 ) ? yes : no );
  mvwaddchstr( channel_test_win, 9, 54,
	  ( message101->software_config & 0x08 ) ? yes : no );

  /* Draw some lines to box the results and add labels */
  wattrset(  channel_test_win, LABEL );
  box(       channel_test_win, ACS_VLINE, ACS_HLINE );
  draw_hline(channel_test_win, 8, 0, 59 );
  mvwaddstr( channel_test_win, 0,  4, " RECEIVER CHANNELS TEST RESULTS " );
  mvwaddstr( channel_test_win, 8,  9, " SOFTWARE CONFIGURATION " );
  for ( idx = 0; idx <= 16 ; idx += 4 )
	draw_vline( channel_test_win, 0, 39 + idx, 8 );

  set_mode( MD_PROGHALT ); /* Halt progarm after displaying BIT results  */
  ncscreen_root();			/* Display root window           */
  wnoutrefresh( receiver_health_win );	/* Done with receiver_health_win */
  wnoutrefresh( channel_test_win );	/* Done with channel_test_win    */

  /* Delete windows after displaying */
  delwin( receiver_health_win );
  delwin( channel_test_win );

  doupdate();

} /* End of 'ncscreen_message101()' */

/*------------------------------------------------------------------------*/

/*  'ncscreen_message102()'
 *
 *  Function to display GPS Constellation  data
 *  supplied by the Receiver in Message 102
 */

  void
ncscreen_message102( message102_t * message102 )
{
  unsigned int idx;        /* Index for loops etc */
  int elev, azim; /* Sattelite Elevation and Azimuth */

  WINDOW * satellite_data_win;  /* GPS Satellite data (Message 102) */
  satellite_data_win  = newwin( 12, 14, 7, 1 );

  /*
   *  The following lines build satellite_data_win and post GPS
   *  Constellation data provided by the receiver's Message 102
   */

  wattrset( satellite_data_win, NORMAL );
  for ( idx = 0; idx < message102->num_visible_sats; idx++ )
  {
	elev = (int) ( message102->satellite_ele[ idx ] * RAD_TO_DEG +0.5 );
	azim = (int) ( message102->satellite_azi[ idx ] * RAD_TO_DEG +0.5 );
	mvwprintw( satellite_data_win, (int)idx + 2, 1, " %2d %3d %4d",
		message102->satellite_prn[ idx ], elev, azim );
  }

  wattrset( satellite_data_win, WARNING );
  for ( ; idx < 9; idx++ )
	mvwprintw( satellite_data_win, (int)idx + 2, 1, " --  --  ---" );

  /* Draw some lines to box the results and add labels */
  wattrset(  satellite_data_win, LABEL );
  box(       satellite_data_win, ACS_VLINE, ACS_HLINE );
  draw_vline(satellite_data_win, 0, 4, 11);
  draw_vline(satellite_data_win, 0, 8, 11);
  mvwaddstr( satellite_data_win, 0, 1, " SATELLITES " );
  mvwaddstr( satellite_data_win, 1, 1, "PRN ELE AZIM" );

  /* Done with satellite_data_win */
  wnoutrefresh( satellite_data_win );
  delwin( satellite_data_win );

  doupdate();

} /* End of 'ncscreen_message102()' */

/*------------------------------------------------------------------------*/

/*  'ncscreen_message103()'
 *
 *  Function to display the main rwgps windows and
 *  enter data provided by the Receiver in Message 103
 */

  void
ncscreen_message103( message103_t * message103 )
{
  int idx;      /* Index for loops etc */
  int deg, min, sec, sec10; /* Degrees, minutes, seconds, sec/10 */
  int hr;       /* Time in hours			 */
  double temp;  /* Temporary double used in conversions	 */

  WINDOW * navigation_data_win; /* Receiver position/velocity data	  */
  WINDOW * error_data_win;      /* Various expected errors in GPS data    */
  WINDOW * channel_data_win;    /* Status/activity of Receiver's channels */
  WINDOW * utc_data_win;        /* GPS UTC date and time information      */
  WINDOW * receiver_status_win; /* Info on the status of the GPS Receiver */

  navigation_data_win = newwin(  6, 78,  1,  1 );
  error_data_win      = newwin( 12, 22,  7, 15 );
  channel_data_win    = newwin(  8, 21,  7, 37 );
  utc_data_win        = newwin(  8, 21,  7, 58 );
  receiver_status_win = newwin(  8, 42, 15, 37 );

  /*
   *  The following lines build receiver_status_win
   *  and post data on the Status of the GPS Receiver
   */
  {
	int sats_in_use = ( message103->status_word2 >> 4 ) & 0x07;
	int flag1, flag2, flag3; /* Flags for color-highlighting data */

	wattrset(  receiver_status_win, NORMAL );
	mvwprintw( receiver_status_win, 1, 39, "%2d", sats_in_use );
	mvwprintw( receiver_status_win, 2, 39, "%2d",
		( message103->status_word2 >> 8 ) & 0x0F );
	mvwprintw( receiver_status_win, 3, 39, "%2d",
		( message103->status_word2 >> 12 ) & 0x07 );

	wattrset(  receiver_status_win, ITEM );
	mvwaddstr( receiver_status_win, 1, 26, " Sats in Use" );
	mvwaddstr( receiver_status_win, 2, 26, "Fig of Merit" );
	mvwaddstr( receiver_status_win, 3, 26, "Utility Chnl" );
	mvwaddstr( receiver_status_win, 1,  1, " Navigation Mode:" );
	mvwaddstr( receiver_status_win, 2,  1, "Differential GPS:" );
	mvwaddstr( receiver_status_win, 2,  1, "Differential GPS:" );
	mvwaddstr( receiver_status_win, 3,  1, "Power Management:" );
	mvwaddstr( receiver_status_win, 4,  1, "Amended Alt Hold:" );
	mvwaddstr( receiver_status_win, 5,  1, " Alt in Solution:" );
	mvwaddstr( receiver_status_win, 6,  1, "Alt Hold Enabled:" );
	mvwaddstr( receiver_status_win, 6,  1, "Alt Hold Enabled:" );
	mvwaddstr( receiver_status_win, 5, 25, "StaticRAM:" );
	mvwaddstr( receiver_status_win, 6, 25, "Recvr RTC:" );

	flag1 =    message103->status_word2 & 0x80;
	flag2 = !( message103->status_word2 & 0x08 );
	flag3 = ( sats_in_use > 3 ) ? TRUE : FALSE;

	wattrset( receiver_status_win,
		flag1 ? ALARM : flag2 ? A_NORMAL : flag3 ? NORMAL : WARNING );

	wattrset( navigation_data_win,
		flag1 ? ALARM : flag2 ? A_NORMAL : flag3 ? NORMAL : WARNING );

	mvwaddstr( receiver_status_win, 1, 18,
		flag1 ? "COLDST" : flag2 ? "ACQUIS" : flag3 ? "NAV-3D" : "NAV-2D" );

	flag1 = message103->status_word2 & 0x8000;
	wattrset(  receiver_status_win, flag1 ? NORMAL : WARNING );
	mvwaddstr( receiver_status_win, 2, 18, flag1 ? "IN USE" : "UNUSED" );

	flag1 = message103->status_word1 & 0x01;
	wattrset(  receiver_status_win, flag1 ? NORMAL : WARNING );
	mvwaddstr( receiver_status_win, 3, 18, flag1 ? " FULL " : "CYCLED" );

	flag1 = message103->status_word2 & 0x01;
	wattrset(  receiver_status_win, flag1 ? NORMAL : WARNING );
	mvwaddstr( receiver_status_win, 4, 18, flag1 ? "ACCEPT" : "REJECT" );

	flag1 = message103->status_word2 & 0x02;
	wattrset(  receiver_status_win, flag1 ? NORMAL : WARNING );
	mvwaddstr( receiver_status_win, 5, 18, flag1 ? "IN USE" : "UNUSED" );

	flag1 = message103->status_word2 & 0x04;
	wattrset(  receiver_status_win, flag1 ? NORMAL : WARNING );
	mvwaddstr( receiver_status_win, 6, 18, flag1 ? "ENABLE" : "DISABL" );

	flag1 = message103->status_word1 & 0x04;
	wattrset(  receiver_status_win, flag1 ? ALARM : NORMAL );
	mvwaddstr( receiver_status_win, 5, 35, flag1 ? "!VALID" : " VALID");

	flag1 = message103->status_word1 & 0x08;
	wattrset(  receiver_status_win, flag1 ? ALARM : NORMAL );
	mvwaddstr( receiver_status_win, 6, 35, flag1 ? "!VALID" : " VALID");
  }

  /* Draw some lines to box the results and add labels */
  wattrset(  receiver_status_win, LABEL );
  box(       receiver_status_win, ACS_VLINE, ACS_HLINE );
  draw_vline(receiver_status_win, 0, 17, 7 );
  draw_vline(receiver_status_win, 0, 24, 7 );
  draw_vline(receiver_status_win, 4, 34, 7 );
  draw_vline(receiver_status_win, 0, 38, 4 );
  mvwaddstr( receiver_status_win, 4, 25, "STARTUP VALIDITY" );
  mvwaddstr( receiver_status_win, 0,  6, " GPS RECEIVER ENGINE STATUS " );

  /* Done with receiver_status_win */
  wnoutrefresh( receiver_status_win );
  delwin( receiver_status_win );

  /*
   *  The following lines build navigation_data_win and post
   *  Navigation and UTC data provided by the Receiver's Message 103
   */

  /* Convert Latitude to deg/min/sec (round up to nearest .1 arc-sec) */
  temp  = ( message103->latitude_north * RAD_TO_DEG ) + 1.3888888889E-05;
  deg   = ( int ) temp;
  temp  = ( temp - deg ) * 60;
  min   = ( int ) temp;
  temp  = ( temp - min ) * 60;
  sec   = ( int ) ( temp );
  temp  = ( temp - sec ) * 10;
  sec10 = ( int ) ( temp );

  /* Remove sign from min, sec and sec10 */
  min = abs( min ); sec = abs( sec ); sec10 = abs( sec10 );

  mvwprintw( navigation_data_win, 1, 18,
	  "%3d:%02d:%02d.%1d", deg, min, sec, sec10 );
  mvwprintw( navigation_data_win, 1, 44, "%8d", (int) message103->position_x );
  mvwprintw( navigation_data_win, 1, 70, "%7.2f", message103->velocity_north );

  /* Convert Longitude to deg/min/sec (round up to nearest .1 arc-sec) */
  temp = ( message103->longitude_east * RAD_TO_DEG ) + 1.3888888889E-05;
  deg   = ( int ) temp;
  temp  = ( temp - deg ) * 60;
  min   = ( int ) temp;
  temp  = ( temp - min ) * 60;
  sec   = ( int ) ( temp );
  temp  = ( temp - sec ) * 10;
  sec10 = ( int ) ( temp );

  /* Remove sign from min, sec and sec10 */
  min = abs( min ); sec = abs( sec ); sec10 = abs( sec10 );

  mvwprintw( navigation_data_win, 2, 17,
	  "%4d:%02d:%02d.%1d", deg, min, sec, sec10 );
  mvwprintw( navigation_data_win, 2, 44, "%8d", (int) message103->position_y );
  mvwprintw( navigation_data_win, 2, 70, "%7.2f", message103->velocity_east );

  mvwprintw( navigation_data_win, 3, 20, "%10.2f", message103->altitude_meters );
  mvwprintw( navigation_data_win, 3, 44, "%8d", (int) message103->position_z );
  mvwprintw( navigation_data_win, 3, 70, "%7.2f", message103->velocity_up );

  /* Convert time-of-day from sec to hr/min/sec ( round up to nearest sec) */
  temp = ( message103->time_of_day + 0.5 ) / 3600;
  hr   = (int) temp;
  temp = ( temp - hr ) * 60;
  min  = (int) temp;
  sec  = (int) ( ( temp - min ) * 60 );

  wattrset(  navigation_data_win, message103->utc_validity ? NORMAL : ALARM );
  mvwprintw( navigation_data_win, 4, 19, "%02d:%02d:%04d",
	  message103->utc_day, message103->utc_month, message103->utc_year );
  mvwprintw( navigation_data_win, 4, 44, "%02d:%02d:%02d", hr, min, sec );
  mvwaddstr( navigation_data_win, 4, 70,
	  message103->utc_validity ? " VALID " : "INVALID" );

  /* Set hardware clock if requested by 'C' keystroke */
  if( isModeSet( MD_SETCLOCK ) )
  {
	if( message103->utc_validity )
	{
	  char utc[ 20 ]; /* UTC date/time string */
	  snprintf( utc, 20, "%02d/%02d/%04d %02d:%02d:%02d",
		  message103->utc_month,
		  message103->utc_day,
		  message103->utc_year,
		  hr, min, sec );

	  /* Return to main screen if no error at setting the hw clock */
	  if( set_hwclock( utc ) != ERROR ) set_mode( MD_MAINSCREEN );
	}
	else
	  set_mode( MD_UTCINVALID | MD_PROGHALT);
  }

  /* Draw some lines to box the results and add labels */
  wattrset(  navigation_data_win, ITEM );
  mvwaddstr( navigation_data_win, 1,  2, "Latitude d:m:s" );
  mvwaddstr( navigation_data_win, 1, 31, "Position-X m" );
  mvwaddstr( navigation_data_win, 1, 55, "Velocity-N m/s" );
  mvwaddstr( navigation_data_win, 2,  1, "Longitude d:m:s" );
  mvwaddstr( navigation_data_win, 2, 31, "Position-Y m" );
  mvwaddstr( navigation_data_win, 2, 55, "Velocity-E m/s" );
  mvwaddstr( navigation_data_win, 3,  2, "Altitude-ASL m" );
  mvwaddstr( navigation_data_win, 3, 31, "Position-Z m" );
  mvwaddstr( navigation_data_win, 3, 55, "Velocity-U m/s" );
  mvwaddstr( navigation_data_win, 4,  1, "Date dd:mm:yyyy" );
  mvwaddstr( navigation_data_win, 4, 31, "UTC hh:mm:ss" );
  mvwaddstr( navigation_data_win, 4, 54, "UTC Data Status" );

  wattrset(  navigation_data_win, LABEL );
  box(       navigation_data_win, ACS_VLINE, ACS_HLINE );
  draw_vline(navigation_data_win, 0, 16, 5 );
  draw_vline(navigation_data_win, 0, 29, 5 );
  draw_vline(navigation_data_win, 0, 30, 5 );
  draw_vline(navigation_data_win, 0, 43, 5 );
  draw_vline(navigation_data_win, 0, 52, 5 );
  draw_vline(navigation_data_win, 0, 53, 5 );
  draw_vline(navigation_data_win, 0, 69, 5 );
  mvwaddstr( navigation_data_win, 0,  2, " NAVIGATION " );
  mvwaddstr( navigation_data_win, 0, 32, " POSITION " );
  mvwaddstr( navigation_data_win, 0, 56, " VELOCITY " );

  /* Done with navigation_data_win */
  wnoutrefresh( navigation_data_win );
  delwin( navigation_data_win );

  /*
   *  The following lines build error_data_win and post as
   *  defaults, the extreme values of the various data items
   */

  wattrset(  error_data_win, ITEM );
  mvwaddstr( error_data_win,  1, 1, "  Horizontal" );
  mvwaddstr( error_data_win,  2, 1, "    Vertical" );
  mvwaddstr( error_data_win,  3, 1, "    Position" );
  mvwaddstr( error_data_win,  4, 1, "    GPS Time" );
  mvwaddstr( error_data_win,  5, 1, "   Geometric" );
  mvwaddstr( error_data_win,  7, 1, "Hor Position" );
  mvwaddstr( error_data_win,  8, 1, "Ver Position" );
  mvwaddstr( error_data_win,  9, 1, "    GPS Time" );
  mvwaddstr( error_data_win, 10, 1, "Hor Velocity" );

  wattrset( error_data_win, NORMAL );
  for ( idx = 0; idx < 5; idx++ )
	mvwprintw( error_data_win, idx + 1, 14,
		"%7.2f", message103->prec_dilution[ idx ] / 100.0 );

  mvwprintw( error_data_win,  7, 14, "%7d", message103->error_horiz );
  mvwprintw( error_data_win,  8, 14, "%7d", message103->error_vert );
  mvwprintw( error_data_win,  9, 14, "%7d", (int) (message103->error_time / C) );
  mvwprintw( error_data_win, 10, 14, "%7.2f", message103->error_velocity/100.0 );

  /* Draw some lines to box the results and add labels */
  wattrset(  error_data_win, LABEL );
  box(       error_data_win, ACS_VLINE, ACS_HLINE );
  draw_hline(error_data_win,  6, 0, 21 );
  draw_vline(error_data_win,  0,13, 11 );
  mvwaddstr( error_data_win,  0,  1, " PRECISION DILUSION " );
  mvwaddstr( error_data_win,  6,  4, " EXPECTED ERROR " );

  /* Done with error_data_win */
  wnoutrefresh( error_data_win );
  delwin( error_data_win );

  /*
   *  The following lines build channel_data_win and post
   *  information on the activity of the Receiver's channels
   */

  wattrset( channel_data_win, NORMAL );
  for ( idx = 0; idx < 5; idx++ )
  {
	mvwprintw( channel_data_win, idx + 2, 1, " %1d %3d          %3d",
		( message103->channel_state[ idx ] >>12 ) & 0x07,
		message103->channel_state[ idx ] & 0x1F,
		( message103->channel_cno[ idx ] >> 8 ) & 0x3F );
  }

  {
	int flag1, flag2; /* Flags used for color-highlighting data */

	for ( idx = 0; idx < 5; idx++ )
	{
	  flag1 = message103->channel_state[ idx ] & 0x0800;
	  flag2 = message103->channel_state[ idx ] & 0x0200;

	  wattrset(  channel_data_win, flag1 ? NORMAL : flag2 ? WARNING : ALARM );
	  mvwaddstr( channel_data_win, idx + 2, 8,
		  flag1 ? "TRACKING" : flag2 ? " SEARCH " : "  IDLE  ");
	}
  }

  /* Draw some lines to box the results and add labels */
  wattrset(  channel_data_win, LABEL );
  box(       channel_data_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( channel_data_win, 1, 1, "CH PRN ACTIVITY S/N" );
  draw_vline(channel_data_win, 0, 3, 7 );
  draw_vline(channel_data_win, 0, 7, 7 );
  draw_vline(channel_data_win, 0,16, 7 );
  mvwaddstr( channel_data_win, 0, 1, " CHANNEL ACTIVITY " );

  /* Done with channel_data_win */
  wnoutrefresh( channel_data_win );
  delwin( channel_data_win );

  /*
   *  The following lines build utc_data_win and post as
   *  defaults, the extreme values of the various data items
   */

  wattrset(  utc_data_win, message103->utc_validity ? NORMAL : ALARM );
  mvwprintw( utc_data_win, 1, 10, "%02d:%02d:%04d",
	  message103->utc_day, message103->utc_month, message103->utc_year );
  mvwprintw( utc_data_win, 2, 10, "%02d:%02d:%02d", hr, min, sec );
  mvwprintw( utc_data_win, 3, 10, "%-d", message103->week_number );
  mvwprintw( utc_data_win, 4, 10, "%-d",(int) message103->time_of_week );
  mvwprintw( utc_data_win, 5, 10, "%-d", message103->set_time );
  mvwaddstr( utc_data_win, 6, 10,
	  message103->utc_validity ? "VALID  " : "INVALID" );

  wattrset(  utc_data_win, ITEM );
  mvwaddstr( utc_data_win, 1, 1, "UTC Date " );
  mvwaddstr( utc_data_win, 2, 1, "UTC Time " );
  mvwaddstr( utc_data_win, 3, 1, "Week Num" );
  mvwaddstr( utc_data_win, 4, 1, "T/ofWeek" );
  mvwaddstr( utc_data_win, 5, 1, "Set Time" );
  mvwaddstr( utc_data_win, 6, 1, "UTC Data " );

  /* Draw some lines to box the results and add labels */
  wattrset(  utc_data_win, LABEL );
  box(       utc_data_win, ACS_VLINE, ACS_HLINE );
  draw_vline(utc_data_win, 0, 9, 7 );
  mvwaddstr( utc_data_win, 0, 2, " GPS DATE/TIME " );

  /* Done with utc_data_win */
  wnoutrefresh( utc_data_win );
  delwin( utc_data_win );

  doupdate();

} /* End of 'ncscreen_message103()' */

/*------------------------------------------------------------------------*/

/*  'ncscreen_status()'
 *
 *  Display the status of GPS Receiver and rwgps program
 */

  void
ncscreen_status(void)
{
  /* Show outgoing message id in yellow */
  if( message_id() > 200 )
	attrset( WARNING );
  else
	attrset( NORMAL );
  mvprintw( 20, 14, "%3d", message_id() );

  /* Default status */
  attrset( NORMAL );
  mvaddstr( 20, 28, "No Error" );

  /* Display status according to mode falgs */
  switch( current_mode() & ~MD_PROGHALT )
  {
	case MD_MAINSCREEN  : mvaddstr( 20, 2, "MAIN SCRN" );
						  break;
	case MD_WARMSTART   : attrset( WARNING );
						  mvaddstr( 20, 2, "WARM STRT" );
						  break;
	case MD_BITRESULTS  : attrset( WARNING );
						  mvaddstr( 20, 2, "BIT RESLT" );
						  break;
	case MD_NULLMODE	: attrset( WARNING );
						  mvaddstr( 20, 2, "PROG HALT" );
						  break;
	case MD_POWERMANAGE : attrset( WARNING );
						  mvaddstr( 20, 2, "POWER MAN" );
						  break;
	case MD_SETCLOCK	: attrset( ALARM );
						  mvaddstr( 20, 2, "SET CLOCK" );
						  break;
	case MD_READFAIL	: attrset( WARNING );
						  mvaddstr( 20,  2, "PROG HALT" );
						  attrset( ALARM );
						  mvaddstr( 20, 28, " !read()" );
						  break;
	case MD_WRITEFAIL	: attrset( WARNING );
						  mvaddstr( 20,  2, "PROG HALT" );
						  attrset( ALARM );
						  mvaddstr( 20, 28, "!write()" );
						  break;
	case MD_POPENFAIL	: attrset( WARNING );
						  mvaddstr( 20,  2, "PROG HALT" );
						  attrset( ALARM );
						  mvaddstr( 20, 28, "!popen()" );
						  break;
	case MD_UTCINVALID  : attrset( WARNING );
						  mvaddstr( 20,  2, "PROG HALT" );
						  attrset( ALARM );
						  mvaddstr( 20, 28, "  !UTC  " );
  } /* End switch( currenet_mode() ) */

  switch( message_protocol() )
  {
	case ACK_RECEIVED  : attrset( WARNING );
						 mvaddstr( 20, 21, " ACK" );
						 break;
	case 0             : attrset( NORMAL );
						 mvaddstr( 20, 21, "NULL" );
						 break;
	case NACK_RECEIVED : attrset( ALARM );
						 mvaddstr( 20, 21, "NACK" );

  } /* End switch ( message_protocol() ) */

  /* If errno is not set, use mode flags to display errors */
  if( error_num() == SUCCESS )
  {
	if( isModeSet(MD_READFAIL | MD_WRITEFAIL) )
	{
	  attrset( ALARM );
	  mvaddstr( 21, 2, " Serial Device read/write Timeout " );
	}
	else if( isModeSet(MD_POPENFAIL) )
	{
	  attrset( ALARM );
	  mvaddstr( 21, 2, " Set HW clock to GPS Time failed  " );
	}
	else if( isModeSet(MD_UTCINVALID) )
	{
	  attrset( ALARM );
	  mvaddstr( 21, 2, " The UTC Validity flag is DOWN!   " );
	}
	else
	{
	  attrset( NORMAL );
	  mvaddstr( 21, 2, " No error returned from System    " );
	}
  }

  else
  {
	attrset( ALARM );
	mvaddstr( 21, 2, strerror( error_num() ) );
  } /* End if( error_num == SUCCESS) */

  clear_error(); /* Clear error_num after display */

  /* Draw some lines to box the results and add labels */
  attrset( LABEL );
  draw_box(  4, 36, 19, 1 );
  mvaddstr( 19,  2, " PR MODE " );
  mvaddstr( 19, 12, " MSGID " );
  mvaddstr( 19, 20, " PROTO " );
  mvaddstr( 19, 28, " ERRORS " );
  mvaddch(  20, 11, ACS_VLINE );
  mvaddch(  20, 19, ACS_VLINE );
  mvaddch(  20, 27, ACS_VLINE );
  mvaddch(  19, 11, ACS_TTEE );
  mvaddch(  19, 19, ACS_TTEE );
  mvaddch(  19, 27, ACS_TTEE );

  refresh();

} /* End of 'ncscreen_status()' */

/*------------------------------------------------------------------------*/

/*  'ncpost_warm_start_form()'
 *
 *  Function to display the Warm Start Parameters entry form
 *  and read data from the buffers into message201
 */

  void
ncpost_warm_start_form( void )
{
  message201_t message201;

  int idx;   	 /* Index for loops etc    */
  int ch;   	 /* Input from getch()     */
  bool done; 	 /* Form entries done flag */

  /*
   *  Form for entering Warm Start parameters
   *  (Receiver position and UTC date/time)
   */

  FORM   * warm_start_form;
  FIELD  * data_field[ 14 ];

  /* Setup Warm Start fields and form */
  for( idx = 0; idx < 11; idx++ )
  {
	data_field[ idx ] = new_field( 1, 5, idx + 2, 20, 0, 0 );
	field_opts_off( data_field[ idx ], O_AUTOSKIP );
  }

  data_field[ 11 ] = new_field( 1, 9, 14, 16, 0, 0 );
  field_opts_off(   data_field[ 11 ], O_EDIT );
  set_field_buffer( data_field[ 11 ], 0, "< Done! >" );

  data_field[ 12 ] = new_field( 1, 9, 14, 3, 0, 0 );
  field_opts_off(   data_field[ 12 ], O_EDIT );
  set_field_buffer( data_field[ 12 ], 0, "< Abort >" );

  /*  The following block of code enters
   *  default values in the Warm Start form
   */
  {
	time_t ct; 	   /* System calender time */
	struct tm * utc; /* UTC date and time	   */
	char tmp[ 5 ];   /* Temporary string     */

	/* Make time in GMT format */
	time( &ct );
	utc = gmtime( &ct );

	/* Enter default position values in warm start form */
	set_field_buffer( data_field[ 0 ], 0, LAT_DEG );
	set_field_buffer( data_field[ 1 ], 0, LAT_MIN );
	set_field_buffer( data_field[ 2 ], 0, LON_DEG );
	set_field_buffer( data_field[ 3 ], 0, LON_MIN );
	set_field_buffer( data_field[ 4 ], 0, ALT_MTR );

	/* Enter time and date from computer's clock */
	snprintf( tmp, 5, "%02d", utc->tm_hour );
	set_field_buffer( data_field[ 5 ], 0, tmp );
	snprintf( tmp, 5, "%02d", utc->tm_min );
	set_field_buffer( data_field[ 6 ], 0, tmp );
	snprintf( tmp, 5, "%02d", utc->tm_mday );
	set_field_buffer( data_field[ 7 ], 0, tmp );
	snprintf( tmp, 5, "%02d", utc->tm_mon + 1 );
	set_field_buffer( data_field[ 8 ], 0, tmp );
	snprintf( tmp, 5, "%4d", utc->tm_year +1900 );
	set_field_buffer( data_field[ 9 ], 0, tmp );
  } /* End of defaults for Warn Start form */

  data_field[ 13 ] = ( FIELD * ) NULL;

  set_field_type( data_field[  0 ], TYPE_INTEGER, 2,  -90,    90 );
  set_field_type( data_field[  1 ], TYPE_INTEGER, 2,    0,    59 );
  set_field_type( data_field[  2 ], TYPE_INTEGER, 3, -180,   180 );
  set_field_type( data_field[  3 ], TYPE_INTEGER, 2,    0,    59 );
  set_field_type( data_field[  4 ], TYPE_INTEGER, 1,    0, 49999 );
  set_field_type( data_field[  5 ], TYPE_INTEGER, 2,    0,    23 );
  set_field_type( data_field[  6 ], TYPE_INTEGER, 2,    0,    59 );
  set_field_type( data_field[  7 ], TYPE_INTEGER, 2,    1,    31 );
  set_field_type( data_field[  8 ], TYPE_INTEGER, 2,    1,    12 );
  set_field_type( data_field[  9 ], TYPE_INTEGER, 4, 2000,  2048 );
  set_field_type( data_field[ 10 ], TYPE_INTEGER, 1,    0,     1 );

  /* Default value for static nav enable */
  set_field_buffer( data_field[ 10 ], 0, "0" );

  warm_start_form = new_form( data_field );
  post_form( warm_start_form );

  /* Display root window, box the form and add label  */
  nodelay( stdscr, FALSE ); /* Make keyboard blocking */
  ncscreen_root();
  ncscreen_status();  /* Keep Status window on screen */

  attrset( LABEL );
  draw_box( 15, 26, 1, 1 );
  draw_vline( stdscr,  1, 18, 13 );
  draw_hline( stdscr, 13, 1, 26 );
  mvaddstr(  1,  2, " WARM START PARAMETERS? " );

  /* Enter various labels in the Warm Start form box */
  attrset( ITEM );
  mvaddstr(  2, 2, "  Latitude - deg" );
  mvaddstr(  3, 2, "  Latitude - min" );
  mvaddstr(  4, 2, " Longitude - deg" );
  mvaddstr(  5, 2, " Longitude - min" );
  mvaddstr(  6, 2, "  Altitude - mtr" );
  mvaddstr(  7, 2, "  Time now - hrs" );
  mvaddstr(  8, 2, "  Time now - min" );
  mvaddstr(  9, 2, "  Date now - day" );
  mvaddstr( 10, 2, "  Date now - mon" );
  mvaddstr( 11, 2, "  Date now - yrs" );
  mvaddstr( 12, 2, "Allow Static Nav" );

  /*
   *  Prepare for the form's data input loop
   */
  form_driver( warm_start_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  done = FALSE;

  /* Process keystrokes from user */
  while( !done )
  {
	if( (current_field( warm_start_form ) == data_field[ 11 ]) ||
		(current_field( warm_start_form ) == data_field[ 12 ]) )
	  curs_set( FALSE );
	else
	  curs_set( TRUE );

	ch = getch();
	switch( ch )
	{
	  case KEY_RET :
		/* RETURN on 'Done!' field */
		if( current_field( warm_start_form ) == data_field[ 11 ] )
		{
		  done = TRUE;
		  break;
		}
		/* RETURN on 'Abort' field */
		if( current_field( warm_start_form ) == data_field[ 12 ] )
		{
		  done = TRUE;
		  set_mode( MD_MAINSCREEN );
		  break;
		}

		set_field_back( current_field( warm_start_form ), A_NORMAL );
		form_driver( warm_start_form, REQ_NEXT_FIELD );
		form_driver( warm_start_form, REQ_BEG_FIELD );
		set_field_back( current_field(warm_start_form), KEYSTROKE );
		break;

	  case KEY_DOWN :	set_field_back(current_field(warm_start_form), A_NORMAL);
						form_driver( warm_start_form, REQ_NEXT_FIELD );
						set_field_back( current_field(warm_start_form), KEYSTROKE );
						break;

	  case KEY_UP : set_field_back(current_field(warm_start_form), A_NORMAL);
					form_driver( warm_start_form, REQ_PREV_FIELD );
					set_field_back( current_field(warm_start_form), KEYSTROKE );
					break;

	  case KEY_BACKSPACE : form_driver( warm_start_form, REQ_PREV_CHAR );
						   form_driver( warm_start_form, REQ_DEL_CHAR );
						   break;

	  case KEY_DC    : form_driver( warm_start_form, REQ_DEL_CHAR );
					   break;
	  case KEY_LEFT  : form_driver( warm_start_form, REQ_PREV_CHAR );
					   break;
	  case KEY_RIGHT : form_driver( warm_start_form, REQ_NEXT_CHAR );
					   break;
	  case KEY_ESC   : done = TRUE;
					   set_mode( MD_MAINSCREEN );
					   curs_set( FALSE );
					   break;
	  default        : form_driver( warm_start_form, ch );

	} /* End of switch( ch )    */

  }     /* End of while( !done )  */

  if( isModeClear( MD_MAINSCREEN ) ) /* If warm start not aborted */
  {
	/* Read initialization data from warm_start_form */

	/* Initialization control word */
	message201.init_control  = (unsigned int)
	  (0x7C | atoi( field_buffer( data_field[10], 0 ) ) << 7);
	message201.position_x    = 0.0;  /* Not used - disabled in init_control */
	message201.position_y    = 0.0;  /* Not used - disabled in init_control */
	message201.position_z    = 0.0;  /* Not used - disabled in init_control */

	message201.latitude_north  = ( atoi( field_buffer( data_field[0], 0 ) ) +
		atoi( field_buffer( data_field[1], 0 ) ) / 60.0 ) / RAD_TO_DEG;

	message201.longitude_east  = ( atoi( field_buffer( data_field[2], 0 ) ) +
		atoi( field_buffer( data_field[3], 0 ) ) / 60.0 ) / RAD_TO_DEG;

	message201.altitude_meters = atoi( field_buffer(data_field[4], 0) );
	message201.velocity_north  = 0.0; /* Not used - disabled in init_control */
	message201.velocity_east   = 0.0; /* Not used - disabled in init_control */
	message201.velocity_up     = 0.0; /* Not used - disabled in init_control */
	message201.utc_hours       = (unsigned int)atoi( field_buffer( data_field[5], 0 ) );
	message201.utc_minutes     = (unsigned int)atoi( field_buffer( data_field[6], 0 ) );
	message201.utc_seconds     = 0;
	message201.utc_day		   = (unsigned int)atoi( field_buffer( data_field[7], 0 ) );
	message201.utc_month	   = (unsigned int)atoi( field_buffer( data_field[8], 0 ) );
	message201.utc_year		   = (unsigned int)atoi( field_buffer( data_field[9], 0 ) );

	/* Send Warm Start parameters to Receiver */
	send_message201( &message201 );
  }

  unpost_form( warm_start_form );
  free_form( warm_start_form );
  for ( idx=0; idx < 14; idx++ ) free_field( data_field[ idx ] );

  nodelay( stdscr, TRUE ); /* Make keyboard non-blocking */
  ncscreen_root();         /* Display root window        */
  ncscreen_status();       /* Display status window      */

} /* End of 'ncpost_warm_start_form()' */

/*------------------------------------------------------------------------*/

/*  'ncpost_powermanage_form()'
 *
 *  Function to display the Power Management Parameters
 *  entry form and read data from the buffers into message201
 */

  void
ncpost_powermanage_form( void )
{
  message211_t message211;

  int idx;   /* Index for loops etc    */
  int ch;    /* Input from getch()     */
  bool done; /* Form entries done flag */

  /*
   *  Form for entering Power Management parameters
   */

  FORM   * powermanage_form;
  FIELD  * data_field[ 7 ];

  /* Setup Power Management fields and form */
  for( idx = 0; idx < 4; idx++ )
  {
	data_field[ idx ] = new_field( 1, 5, idx + 2, 25, 0, 0 );
	field_opts_off( data_field[ idx ], O_AUTOSKIP );
  }

  data_field[ 4 ] = new_field( 1, 9, 7, 20, 0, 0 );
  field_opts_off(   data_field[ 4 ], O_EDIT );
  set_field_buffer( data_field[ 4 ], 0, "< Done! >" );

  data_field[ 5 ] = new_field( 1, 9, 7, 4, 0, 0 );
  field_opts_off(   data_field[ 5 ], O_EDIT );
  set_field_buffer( data_field[ 5 ], 0, "< Abort >" );

  set_field_buffer( data_field[ 0 ], 0, "1" );
  set_field_buffer( data_field[ 1 ], 0, "1" );
  set_field_buffer( data_field[ 2 ], 0, "90" );
  set_field_buffer( data_field[ 3 ], 0, "-1" );

  data_field[ 6 ] = ( FIELD * ) NULL;

  set_field_type( data_field[ 0 ], TYPE_INTEGER, 0,  1, 2 );
  set_field_type( data_field[ 1 ], TYPE_INTEGER, 0,  1, 5 );
  set_field_type( data_field[ 2 ], TYPE_INTEGER, 0, -1, 0xFFFF );
  set_field_type( data_field[ 3 ], TYPE_INTEGER, 0, -1, 0xFFFF );

  powermanage_form = new_form( data_field );
  post_form( powermanage_form );

  /* Display root window, box the form and add label  */
  nodelay( stdscr, FALSE ); /* Make keyboard blocking */
  ncscreen_root();
  ncscreen_status(); /* Keep Status window on screen */

  attrset( LABEL );
  draw_box( 8, 31, 1, 1 );
  draw_hline( stdscr, 6,  1, 31 );
  draw_vline( stdscr, 1, 23,  6 );
  mvaddstr( 1, 2, " POWER MANAGEMENT PARAMETERS " );

  /* Enter various labels in the Warm Start form box */
  attrset( ITEM );
  mvaddstr(  2, 2, "  Command/Status Flag" );
  mvaddstr(  3, 2, " Solution Update Rate" );
  mvaddstr(  4, 2, "  Acquisition Timeout" );
  mvaddstr(  5, 2, "    Low Power Timeout" );

  /* Prepare for the form's data input loop */
  form_driver( powermanage_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  done = FALSE;

  /* Process keystrokes from user */
  while( !done )
  {
	if( (current_field( powermanage_form ) == data_field[ 4 ]) ||
		(current_field( powermanage_form ) == data_field[ 5 ]) )
	  curs_set( FALSE );
	else
	  curs_set( TRUE );

	ch = getch();
	switch( ch )
	{
	  /* RETURN on 'Done!' field */
	  case KEY_RET :
		if( current_field( powermanage_form ) == data_field[ 4 ] )
		{
		  done = TRUE;
		  break;
		}
		/* RETURN on 'Abort' field */
		if ( current_field( powermanage_form ) == data_field[ 5 ] )
		{
		  done = TRUE;
		  set_mode( MD_MAINSCREEN );
		  break;
		}
		set_field_back( current_field( powermanage_form ), A_NORMAL );
		form_driver( powermanage_form, REQ_NEXT_FIELD );
		form_driver( powermanage_form, REQ_BEG_FIELD );
		set_field_back( current_field(powermanage_form), KEYSTROKE );
		break;

	  case KEY_DOWN :
		set_field_back(current_field(powermanage_form), A_NORMAL);
		form_driver( powermanage_form, REQ_NEXT_FIELD );
		set_field_back( current_field(powermanage_form), KEYSTROKE );
		break;

	  case KEY_UP :
		set_field_back(current_field(powermanage_form), A_NORMAL);
		form_driver( powermanage_form, REQ_PREV_FIELD );
		set_field_back( current_field(powermanage_form), KEYSTROKE );
		break;

	  case KEY_BACKSPACE :
		form_driver( powermanage_form, REQ_PREV_CHAR );
		form_driver( powermanage_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC    :
		form_driver( powermanage_form, REQ_DEL_CHAR );
		break;
	  case KEY_LEFT  :
		form_driver( powermanage_form, REQ_PREV_CHAR );
		break;
	  case KEY_RIGHT :
		form_driver( powermanage_form, REQ_NEXT_CHAR );
		break;
	  case KEY_ESC   :
		done = TRUE;
		set_mode( MD_MAINSCREEN );
		curs_set( FALSE );
		break;
	  default        :
		form_driver( powermanage_form, ch );

	} /* End of switch( ch )    */

  }     /* End of while( !done )  */

  if( isModeClear( MD_MAINSCREEN ) )
  {
	/* Read initialization data from powermanage_form */
	message211.command_flag = (unsigned int)atoi( field_buffer(data_field[0], 0) );
	message211.update_rate  = (unsigned int)atoi( field_buffer(data_field[1], 0) );
	message211.acq_timeout  = (unsigned int)atoi( field_buffer(data_field[2], 0) );
	message211.lp_timeout   = (unsigned int)atoi( field_buffer(data_field[3], 0) );

	/* Send Power Management parameters to Receiver */
	send_message211( &message211 );
  }

  unpost_form( powermanage_form );
  free_form( powermanage_form );
  for ( idx=0; idx < 7; idx++ ) free_field( data_field[ idx ] );

  nodelay( stdscr, TRUE ); /* Make keyboard non-blocking */
  ncscreen_root();         /* Display root window        */
  ncscreen_status();       /* Display status window      */

} /* End of 'ncpost_powermanage_form()' */

/*------------------------------------------------------------------------*/

/*  'ncpost_set_hwclock_form()'
 *
 *  Function to display a Confirmation message
 *  for the Set Hardware Clock request
 */

  void
ncpost_set_hwclock_form( void )
{
  int idx;   /* Index for loops etc    */
  int ch;    /* Input from getch()     */
  bool done; /* Form entries done flag */

  /*
   *  Form for entering Power Management parameters
   */

  FORM   * set_hwclock_form;
  FIELD  * data_field[ 3 ];

  /* Setup Power Management fields and form */
  data_field[ 0 ] = new_field( 1, 10, 8, 17, 0, 0 );
  field_opts_off(   data_field[ 0 ], O_EDIT );
  set_field_buffer( data_field[ 0 ], 0, "< Do it! >" );

  data_field[ 1 ] = new_field( 1, 9, 8, 4, 0, 0 );
  field_opts_off(   data_field[ 1 ], O_EDIT );
  set_field_buffer( data_field[ 1 ], 0, "< Abort >" );

  data_field[ 2 ] = ( FIELD * ) NULL;

  set_hwclock_form = new_form( data_field );
  post_form( set_hwclock_form );

  /* Display root window, box the form and add label  */
  nodelay( stdscr, FALSE ); /* Make keyboard blocking */
  ncscreen_root();
  ncscreen_status(); /* Keep Status window on screen */

  attrset( LABEL );
  draw_box( 9, 30, 1, 1 );
  draw_hline( stdscr, 7, 1, 30 );
  mvaddstr( 1, 5, " SET HARDWARE CLOCK " );

  /* Enter various labels in the Set Clock form box */
  attrset( ITEM );
  mvaddstr(  2, 3, "This will set the computer" );
  mvaddstr(  3, 3, "Hardware (CMOS) Clock to" );
  mvaddstr(  4, 3, "GPS (UTC) time and date!" );
  mvaddstr(  6, 3, "Please confirm this action" );

  /* Prepare for the form's data input loop */
  form_driver( set_hwclock_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  done = FALSE;

  /* Process keystrokes from user */
  while( !done )
  {
	ch = getch();
	switch( ch )
	{
	  /* RETURN on 'Done' field */
	  case KEY_RET :
		if (current_field(set_hwclock_form) == data_field[ 0 ])
		{
		  done = TRUE;
		  break;
		}
		/* RETURN on 'Abort' field */
		if ( current_field( set_hwclock_form ) == data_field[ 1 ] )
		{
		  done = TRUE;
		  set_mode( MD_MAINSCREEN );
		  break;
		}
		set_field_back( current_field( set_hwclock_form ), A_NORMAL );
		form_driver( set_hwclock_form, REQ_NEXT_FIELD );
		form_driver( set_hwclock_form, REQ_BEG_FIELD );
		set_field_back( current_field(set_hwclock_form), KEYSTROKE );
		break;

	  case KEY_DOWN :
		set_field_back(current_field(set_hwclock_form), A_NORMAL);
		form_driver( set_hwclock_form, REQ_NEXT_FIELD );
		set_field_back( current_field(set_hwclock_form), KEYSTROKE );
		break;

	  case KEY_UP   :
		set_field_back(current_field(set_hwclock_form), A_NORMAL);
		form_driver( set_hwclock_form, REQ_PREV_FIELD );
		set_field_back( current_field(set_hwclock_form), KEYSTROKE );
		break;

	  case KEY_ESC  :
		done = TRUE; set_mode( MD_MAINSCREEN );

	} /* End of switch( ch )    */

  }     /* End of while( !done )  */

  unpost_form( set_hwclock_form );
  free_form( set_hwclock_form );
  for ( idx=0; idx < 3  ; idx++ ) free_field( data_field[ idx ] );

  nodelay( stdscr, TRUE ); /* Make keyboard non-blocking */
  ncscreen_root();         /* Display root window        */
  ncscreen_status();       /* Display status window      */

} /* End of 'ncpost_set_hwclock_form()' */

/*------------------------------------------------------------------------*/

/*  'draw_box()'
 *
 *  Function to draw a box of given size and position in the root window
 */

  void
draw_box( int rows, int cols, int top, int left )
{
  mvvline( top + 1, left, ACS_VLINE, rows - 2 );
  mvvline( top + 1, left + cols - 1, ACS_VLINE, rows - 2 );
  mvhline( top, left + 1, ACS_HLINE, cols - 2 );
  mvhline( top + rows - 1, left + 1, ACS_HLINE, cols - 2 );
  mvaddch( top, left, ACS_ULCORNER );
  mvaddch( top, left + cols - 1, ACS_URCORNER );
  mvaddch( top + rows - 1, left, ACS_LLCORNER );
  mvaddch( top + rows - 1, left + cols - 1, ACS_LRCORNER );

} /* End of 'draw_box()' */

/*------------------------------------------------------------------------*/

/*  draw_hline()
 *
 *  Draws a horizontal line in a window including Tees at ends
 */

  void
draw_hline( WINDOW *win, int top, int left, int right )
{
  mvwhline( win, top, left+1, ACS_HLINE, (right - left - 1) );
  mvwaddch( win, top, left,  ACS_LTEE );
  mvwaddch( win, top, right, ACS_RTEE );
}

/*------------------------------------------------------------------------*/

/*  draw_vline()
 *
 *  Draws a horizontal line in a window including Tees at ends
 */

  void
draw_vline( WINDOW *win, int top, int left, int bottom )
{
  mvwvline( win, top+1, left, ACS_VLINE, (bottom - top -1) );
  mvwaddch( win, top, left, ACS_TTEE );
  mvwaddch( win, bottom, left, ACS_BTEE );
}

/*------------------------------------------------------------------------*/
