/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  rwgps.h
 *
 *  Header file for rwgps: An interface application
 *  for Rockwell's Microtracker LP(tm) GPS Receiver
 */

#ifndef RWGPS_H
#define RWGPS_H 1

#include <signal.h>
#include <form.h>
#include <ncurses.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <sys/time.h>
#include <time.h>

/*----------------------------------------------------------------------*/

/*** Serial device used by rwgps ***/
#define SERIAL_DEV "/dev/ttyUSB1"

/*----------------------------------------------------------------------*/

/*** Definitons of default receiver postion ***/
#define LAT_DEG "34"  /* Latitude - deg  */
#define LAT_MIN "44"  /* Latitude - min  */
#define LON_DEG "32"  /* Longitude - deg */
#define LON_MIN "54"  /* Longitude - min */
#define ALT_MTR "390" /* Altitude - mtr  */

/*----------------------------------------------------------------------*/

/* Definitions of some color pairs and color numbers */

#define ALARM		COLOR_PAIR( COLOR_RED )
#define NORMAL		COLOR_PAIR( COLOR_GREEN )
#define BLUE		COLOR_PAIR( COLOR_BLUE )
#define WARNING		COLOR_PAIR( COLOR_YELLOW ) | A_BOLD
#define LABEL		COLOR_PAIR( COLOR_CYAN )
#define ITEM		COLOR_PAIR( COLOR_MAGENTA )
#define KEYSTROKE	COLOR_PAIR( COLOR_KEYSTROKE ) | A_BOLD
#define COMMAND  	COLOR_PAIR( COLOR_COMMAND )   | A_BOLD

#define COLOR_KEYSTROKE  8  /* Yellow fore, blue background */
#define COLOR_COMMAND    9  /*  White fore, blue background */

/*----------------------------------------------------------------------*/

/* Definitions of some useful constants */

#define RAD_TO_DEG  57.29577951308232286465 /* Radians to degrees    */
#define C	     	0.299792458	    		/* Speed of light/10^9   */
#define BUFF_SIZE   58                      /* Size of Serial buffer */

/*----------------------------------------------------------------------*/

/* Definitions of some keycodes */
#define KEY_RET      0x0D	/* Keycode of RETURN keyboard key 	*/
#define KEY_ESC      0x1B	/* Keycode of ESCAPE keyboard key	*/

/*----------------------------------------------------------------------*/

/* Definitions for the Message Header ('magic' first word and flags)    */

#define HEADER_START	0x81FF
#define ACK_REQUEST		0x1000
#define ACK_RECEIVED	0x0800
#define NACK_RECEIVED	0x0400

/*----------------------------------------------------------------------*/

/*  Definitions for setting and testing Mode (flow control) flags.
 *  Those that involve sending a Message to the Receiver have the
 *  message number for their value
 */

#define MD_NULLMODE     0
#define MD_PROGHALT		0x00000100
#define MD_MAINSCREEN	0x00000200
#define MD_SETCLOCK		0x00000400
#define MD_READFAIL		0x00000800
#define MD_WRITEFAIL	0x00001000
#define MD_POPENFAIL    0x00002000
#define MD_UTCINVALID   0x00004000
#define MD_WARMSTART	0x000000C9
#define MD_BITRESULTS	0x000000CA
#define MD_POWERMANAGE	0x000000D3

/*----------------------------------------------------------------------*/

#define ERROR (unsigned int) 0x80000000
#define SUCCESS (unsigned int) 0

/*----------------------------------------------------------------------*/

/* Structure definitions for rwgps */

/*
 *  Structure to hold Built In Test (BIT) results
 *  provided by Microtracker's Message 101
 */

typedef struct
{
  unsigned int
	bit_summary,	  /* Built In Test Summary */
	low_ram_test,	  /* Low Order RAM Test Results  */
	high_ram_test,	  /* High Order RAM Test Results */
	preproc_test,	  /* Preprocessor Test Results   */
	preproc_ram_test, /* Preprocessor Ram Test Results */
	interface_test,   /* Multifunction Interface Test Results	*/
	vco_test,		  /* Code/Carrier VCO Test Results */
	gen_inject_test,  /* Code/Signal Injection Test Results */
	serial_errors,	  /* Serial Transmission Error Count */
	checksum_errors,  /* Serial Message Checksum Error Count */
	eeprom_prog_err,  /* EEPROM Programming Error Count */
	eeprom_cksum_err, /* EEPROM Checksum Error Count */
	aux_port_errors,  /* Auxiliary Port Transmission Errors */
	software_config;  /* Software Configuration */

  double software_version;		/* Software Version	*/
} message101_t;

/*
 *  Structure to hold Visible Satellite Information
 *  provided by Microtracker's Message 102
 */

typedef struct
{
  unsigned int
	set_time,			/* Set Time */
	week_number,		/* GPS Week Number */
	num_visible_sats,	/* Number of Visible Satellites	*/
	satellite_prn[9];	/* Visible Sattelite PRN Number	*/

  double
	satellite_ele[9],	/* Visible Sattelite Elevation	*/
	satellite_azi[9],	/* Visible Satellite Azimuth	*/
	time_of_week;		/* GPS Time of Week	*/
} message102_t;

/*
 *  Structure to hold data provided by Microtracker's Message 103
 *  (Receiver position, Velocity, Time Solutions and Status)
 */

typedef struct
{
  unsigned int
	week_number,	  /* GPS Week Number	*/
	utc_validity,	  /* UTC Validity flag */
	utc_day,		  /* UTC Day   */
	utc_month,		  /* UTC Month */
	utc_year,		  /* UTC Year	 */
	set_time,		  /* Set Time	 */
	prec_dilution[5], /* Dilution of Precision of Nav. Data */
	channel_state[5], /* Channel Measurement State */
	channel_cno[5],	  /* Channel c/no signal-to-noise ratio */
	error_horiz,	  /* Expected Horizontal Position Error */
	error_vert,		  /* Expected Vertical Position Error	*/
	error_time,		  /* Expected Time Position Error		*/
	error_velocity,	  /* Expected Velocity Position Error	*/
	status_word1,	  /* Receiver Status Word 1 */
	status_word2;	  /* Receiver Status Word 2 */

  double
	position_x,		 /* Position X */
	position_y,		 /* Position Y */
	position_z,		 /* Position Z */
	velocity_east,	 /* Velocity East  */
	velocity_north,	 /* Velocity North */
	velocity_up,	 /* Velocity Up */
	latitude_north,	 /* Latitude (+ve East)  */
	longitude_east,	 /* Latitude (+ve North)	*/
	altitude_meters, /* Altitude in meters */
	time_of_week,	 /* GPS Time of Week	*/
	time_of_day;	 /* UTC Time of Day	 */
} message103_t;

/*
 *  Structure to hold Warm Start Data for sending Message 201 to Receiver
 */

typedef struct
{
  unsigned int
	init_control, /* Initialization Control Word */
	utc_hours,	  /* Initial UTC Hours   */
	utc_minutes,  /* Initial UTC Minutes */
	utc_seconds,  /* Initial UTC Seconds */
	utc_day,	  /* Initial UTC Day	 */
	utc_month,	  /* Initial UTC Month */
	utc_year;	  /* Initial UTC Year  */

  double
	position_x,		 /* Initial Position X m */
	position_y,		 /* Initial Position Y m	*/
	position_z,		 /* Initial Position Z m	*/
	latitude_north,	 /* Initial Latitude (+ve North)	rad	*/
	longitude_east,	 /* Initial Longitude (+ve East)	rad	*/
	altitude_meters, /* Initial Altitude m */
	velocity_east,	 /* Initial Velocity East m/s  */
	velocity_north,	 /* Initial Velocity North m/s */
	velocity_up;	 /* Initial Velocity Up m/s */
} message201_t;

/*
 *  Structure of Power Management Data for sending Message 211 to Receiver
 */

typedef struct
{
  unsigned int
	command_flag, /* Command or Request for Status flags */
	update_rate,  /* Lowe Power - Solution Update Rate   */
	acq_timeout,  /* Sattelite Acquisition Timeout */
	lp_timeout;	  /* Low Power Timeout */
} message211_t;

/*----------------------------------------------------------------------*/

/* Function Prototypes produced by cproto */
/* comms.c */
unsigned int input_word(void);
void output_word(unsigned int word);
void output_int(unsigned int nint);
unsigned int read_word(void);
void ack_handler(void);
void send_header(unsigned int message_id, unsigned int word_count, unsigned int protocol);
unsigned int input_header(void);
unsigned int store_data(void);
void process_checksum(unsigned int *checksum);
unsigned int message_id(void);
unsigned int message_protocol(void);
/* convert.c */
double read_float(void);
double read_extended_float(void);
unsigned int read_integer(void);
unsigned int read_double_integer(void);
void convert_float(double fpnum, unsigned int *bit_field);
void convert_extended_float(double fpnum, unsigned int *lower_field, unsigned int *upper_field);
/* main.c */
int main(void);
void ncscreen_message(void);
int set_hwclock(char *utc);
int isModeSet(int mode);
int isModeClear(int mode);
void set_mode(int mode);
void clear_mode(int mode);
int current_mode(void);
/* messages.c */
void read_message101(message101_t *message101);
void read_message102(message102_t *message102);
void read_message103(message103_t *message103);
void send_message201(message201_t *message201);
void send_message202(void);
void send_message211(message211_t *message211);
/* screen.c */
void ncscreen_root(void);
void ncscreen_message101(message101_t *message101);
void ncscreen_message102(message102_t *message102);
void ncscreen_message103(message103_t *message103);
void ncscreen_status(void);
void ncpost_warm_start_form(void);
void ncpost_powermanage_form(void);
void ncpost_set_hwclock_form(void);
void draw_box(int rows, int cols, int top, int left);
void draw_hline(WINDOW *win, int top, int left, int right);
void draw_vline(WINDOW *win, int top, int left, int bottom);
/* serial.c */
void open_serial(void);
void write_serial(unsigned char *ch);
void read_serial(unsigned char *ch);
void close_serial(void);
int error_num(void);
void clear_error(void);
/* tests.c */
int main(void);

/*----------------------------------------------------------------------*/

#endif /* End of rwgps.h */

