/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  messages.c
 *
 *  Functions to read and decode data from the various
 *  Messages provided by the GPS Receiver or to send
 *  Messages with commands or data to the GPS Receiver
 */

#include "rwgps.h"

/*  'read_message101()'
 *
 *  Function to read Built-In-Test results from buffer
 */

  void
read_message101( message101_t * message101 )
{
  message101->bit_summary      = read_word();
  message101->low_ram_test     = read_word();
  message101->high_ram_test    = read_word();
  message101->preproc_test     = read_word();
  message101->preproc_ram_test = read_word();
  message101->interface_test   = read_word();
  message101->vco_test         = read_word();
  message101->gen_inject_test  = read_word();
  message101->serial_errors    = read_integer();
  message101->checksum_errors  = read_integer();
  message101->eeprom_prog_err  = read_integer();
  message101->eeprom_cksum_err = read_integer();
  message101->aux_port_errors  = read_integer();
  message101->software_config  = read_word();
  message101->software_version = read_float();

} /* End of 'read_message101()' */

/*-------------------------------------------------------------------------*/

/*  'read_message102()'
 *
 *  Function to read Visible Satellite Information from buffer
 */

  void
read_message102( message102_t * message102 )
{
  unsigned int idx; /* Index for 'for' loops */

  message102->set_time			= read_double_integer();
  message102->time_of_week		= read_extended_float();
  message102->week_number		= read_integer();
  message102->num_visible_sats	= read_integer();

  for (idx = 0; idx < message102->num_visible_sats; idx++ )
  {
	message102->satellite_prn[ idx ] = read_integer();
	message102->satellite_ele[ idx ] = read_float();
	message102->satellite_azi[ idx ] = read_float();
  }

} /* End of 'read_message102()' */

/*-------------------------------------------------------------------------*/

/*  'read_message103()'
 *
 *  Function to read Receiver Position, Velocity,
 *  Time and Status from the input buffer
 */

  void
read_message103( message103_t * message103 )
{
  int idx; /* Index for 'for' loops */

  message103->time_of_week		= read_extended_float();
  message103->week_number		= read_integer();
  message103->utc_validity		= read_word();
  message103->time_of_day		= read_extended_float();
  message103->utc_day			= read_integer();
  message103->utc_month			= read_integer();
  message103->utc_year			= read_integer();
  message103->set_time			= read_double_integer();
  message103->position_x		= read_extended_float();
  message103->position_y		= read_extended_float();
  message103->position_z		= read_extended_float();
  message103->velocity_east		= read_float();
  message103->velocity_north	= read_float();
  message103->velocity_up		= read_float();
  message103->latitude_north	= read_extended_float();
  message103->longitude_east	= read_extended_float();
  message103->altitude_meters	= read_float();

  for ( idx = 0; idx < 5; idx++ )
	message103->prec_dilution[ idx ] = read_integer();

  for ( idx = 0; idx < 5; idx++ )
  {
	message103->channel_state[ idx ] = read_word();
	message103->channel_cno[ idx ]   = read_word();
  }

  message103->error_horiz    = read_integer();
  message103->error_vert     = read_integer();
  message103->error_time     = read_integer();
  message103->error_velocity = read_integer();

  read_word(); /* Step over the reserved word no. 61 */

  message103->status_word1	= read_word();
  message103->status_word2	= read_word();

} /* End of 'read_message103()' */

/*-------------------------------------------------------------------------*/

/*  'send_message201()'
 *
 *  Function to read the Warm Start Parameters, send
 *  them to GPS Receiver and set the AcksPending Flag
 */

  void
send_message201( message201_t *message201 )
{
  unsigned int lower_field; /* Used for holding bit-fields of converted doubles */
  unsigned int upper_field; /* Used for holding bit-fields of converted doubles */
  unsigned int checksum;    /* Calculated checksum of the message               */

  send_header( 201, 30, ACK_REQUEST ) ;

  checksum = message201->init_control;
  output_word( message201->init_control );

  convert_extended_float( message201->position_x, &lower_field, &upper_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  checksum += ( upper_field & 0xFFFF );
  output_int( lower_field ); output_word( upper_field );

  convert_extended_float( message201->position_y, &lower_field, &upper_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  checksum += ( upper_field & 0xFFFF );
  output_int( lower_field ); output_word( upper_field );

  convert_extended_float( message201->position_z, &lower_field, &upper_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  checksum += ( upper_field & 0xFFFF );
  output_int( lower_field ); output_word( upper_field );

  convert_extended_float( message201->latitude_north, &lower_field, &upper_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  checksum += ( upper_field & 0xFFFF );
  output_int( lower_field ); output_word( upper_field );

  convert_extended_float( message201->longitude_east, &lower_field, &upper_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  checksum += ( upper_field & 0xFFFF );
  output_int( lower_field ); output_word( upper_field );

  convert_float( message201->altitude_meters, &lower_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  output_int( lower_field );

  convert_float( message201->velocity_east, &lower_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  output_int( lower_field );

  convert_float( message201->velocity_north, &lower_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  output_int( lower_field );

  convert_float( message201->velocity_up, &lower_field );
  checksum += ( ( lower_field & 0xFFFF ) + ( lower_field >> 16 ) );
  output_int ( lower_field );

  checksum +=  message201->utc_hours;
  output_word( message201->utc_hours );

  checksum +=  message201->utc_minutes;
  output_word( message201->utc_minutes );

  checksum +=  message201->utc_seconds;
  output_word( message201->utc_seconds );

  checksum +=  message201->utc_day;
  output_word( message201->utc_day );

  checksum +=  message201->utc_month;
  output_word( message201->utc_month );

  checksum +=  message201->utc_year;
  output_word( message201->utc_year );

  /* Process checksum to match GPS Receiver requirements */
  process_checksum( &checksum );
  output_word( checksum );

  /*  Handle ack/nack response from Receiver and return
   *  to Main screen if Program Halt mode was not set
   */
  ack_handler();
  if( isModeSet(MD_PROGHALT) ) return;

  set_mode( MD_MAINSCREEN );

} /* End of send_message201() */

/*-------------------------------------------------------------------------*/

/*  'send_message202()'
 *
 *  Function to request the GPS Receiver to
 *  perform a Built In Test (Send Message 202)
 */

  void
send_message202( void )
{
  int idx; /* Index for loops etc */

  /* Send header, two 0 dummy words and a 0 checksum to GPS receiver */
  send_header( 202, 2, ACK_REQUEST );
  for ( idx = 0; idx < 3; idx++ ) output_word( 0 );

  ack_handler(); /* Handle ack/nack response from Receiver */

} /* End of 'send_ message202( void )' */

/*-------------------------------------------------------------------------*/

/*  'send_message211()'
 *
 *  Function to send a Power Management Message
 *  (send Message 211) and set ACK_PENDING Flag
 */

  void
send_message211( message211_t * message211 )
{
  unsigned int checksum; /* Calculated checksum of the message */

  checksum = message211->command_flag + message211->update_rate +
	message211->acq_timeout + message211->lp_timeout;

  /* Process checksum to match GPS Receiver requirements */
  process_checksum( &checksum );

  /* Send Power Management Parameters to GPS Receiver */
  send_header( 211, 6, ACK_REQUEST );
  output_word( message211->command_flag );
  output_word( message211->update_rate );
  output_word( message211->acq_timeout );
  output_word( message211->lp_timeout );
  output_word( 0 );
  output_word( 0 );
  output_word( checksum );

  /*  Handle ack/nack response from Receiver and return
   *  to Main screen if Program Halt mode was not set
   */
  ack_handler();
  if( isModeSet(MD_PROGHALT) ) return;

  set_mode( MD_MAINSCREEN );

} /* End of 'send_message211()' */

/*-------------------------------------------------------------------------*/
