/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  convert.c
 *
 *  Functions that read numbers in GPS Receiver format from
 *  the input buffer and convert them to native formats, or
 *  convert numbers in native format to GPS Receiver format
 */

#include "rwgps.h"

/*  'read_float()'
 *
 *  Function to convert a Receiver-format 'float' number
 *  and return a native IEEE754-format double float.
 */

  double
read_float( void )
{
  unsigned int lower_field;	/* 32-bit field used to assemble a 64-bit double      */
  unsigned int upper_field;	/* 32-bit field used to assemble a 64-bit double      */
  unsigned int * iptr;	    /* Pointer to int used in manipulating bit fields	  */
  double fpnum;				/* Holds result of conversion: Receiver float->double */
  double * dptr;			/* Pointer to double used in manipulating bit fields  */
  unsigned int wd;			/* Temporary integer to store a Receiver 16-bit 'word'*/

  /*  Read lower and upper 16-bit 'words' of the Receiver-format 'float'
   *  and shuffle bit fields to convert to IEEE754 double format.
   *  If read_word() fails, a zero is returned as a default value
   */

  wd = read_word();
  if( wd == ERROR ) return( 0.0 );
  upper_field = ( ( wd & 0x00FF ) << 20 ) | ( ( wd & 0xFF00 ) >> 13 );
  lower_field = ( wd & 0xFF00 ) << 21;
  wd = read_word();
  if( wd == ERROR ) return( 0.0 );
  upper_field |= ( ( wd & 0x8000 ) << 16 ) | ( ( wd & 0X7FFF ) << 5 );

  /*  If Receiver's 'float' is not 0 then add 894 to exponent of
   *  IEEE754 double to compensate for different mantissa formats
   */
  if ( ( lower_field | upper_field ) != 0 )  upper_field += ( 0x37E << 20 );

  /*  Hopefully lower_field and upper_field now hold the lower and upper
   *  32-bit patterns of the IEEE754-format 'double' equivalent of the
   *  Receiver's 'float' format number. These now have to be entered
   *  into 'fpnum' for returning. THIS PROCESS IS ENDIAN-DEPENDENT!
   */
  dptr    = &fpnum;
  iptr    = ( unsigned int *) dptr;
  *iptr   = lower_field; /* Try exchanging these lines if convertion fails */
  iptr++; /* ^v */
  *iptr   = upper_field; /* Try exchanging these lines if convertion fails */
  fpnum   = *dptr;

  return( fpnum );
} /* End of 'read_float()' */

/*-------------------------------------------------------------------------*/

/*  'read_extended_float()'
 *
 *  Function to convert a Receiver-format 'extended float'
 *  number and return a native IEEE754-format double float.
 */

  double
read_extended_float( void )
{
  unsigned int lower_field;	/* 32-bit words used to assemble a 64-bit double	  */
  unsigned int upper_field;	/* 32-bit words used to assemble a 64-bit double	  */
  unsigned int * iptr;		/* Pointer to int used in manipulating bit fields	  */
  double fpnum;				/* Result of conversion:Receiver extend float->double */
  double *dptr;				/* Pointer to double used in manipulating bit fields  */
  unsigned int wd;			/* Temporary integer to store a Receiver 16-bit 'word'*/

  /*  Read lower, middle and upper 16-bit 'words' of the Receiver-format
   *  'extended float' and shuffle bit fields to convert to IEEE754 double
   *  format. If read_word() fails, a zero is returned as a default value
   */

  wd = read_word();
  if( wd == ERROR ) return( 0.0 );
  upper_field = ( wd & 0x00FF ) << 20;
  lower_field = ( wd & 0xFF00 ) << 5;
  wd = read_word();
  if( wd == ERROR ) return( 0.0 );
  upper_field |= ( wd & 0xFF00 ) >> 11;
  lower_field |= ( wd & 0xFFFF ) << 21;
  wd = read_word();
  if( wd == ERROR ) return( 0.0 );
  upper_field |= ( ( wd & 0x8000 ) << 16 ) | ( ( wd & 0X7FFF ) << 5 );

  /*  If Receiver's 'extended float' number is not 0, add 894 to exponent of
   *  IEEE754 'double' number to compensate for mantissa format differences
   */
  if ( ( lower_field | upper_field ) !=0 ) upper_field += ( 0x37E << 20 );

  /*  Hopefully lower_field and upper_field now hold the lower and upper
   *  32-bit patterns of the IEEE754-format 'double' equivalent of the
   *  Receiver's 'float' format number. These now have to be entered
   *  into 'fpnum' for returning. THIS PROCESS IS ENDIAN-DEPENDENT!
   */
  dptr    = &fpnum;
  iptr    = (unsigned int *) dptr;
  *iptr   = lower_field; /* Try exchanging these lines if convertion fails */
  iptr++;                                /* ^v */
  *iptr   = upper_field; /* Try exchanging these lines if convertion fails */
  fpnum   = *dptr;

  return( fpnum );

} /* End of 'read_extended_float()' */

/*-------------------------------------------------------------------------*/

/*  'read_integer( void )'
 *
 *  Function to convert a Receiver-format 16-bit 'integer' to native int
 */

  unsigned int
read_integer( void )
{
  unsigned int itmp; /* Holds the result of convertion: Receiver 'integer'->int	*/

  /* If read_word() fails, a zero is returned as a default value */
  itmp  = read_word();
  if( itmp == ERROR  ) return( 0 );
  itmp |= ( itmp & 0x8000 ) << 16;	/* Move 'sign' bit to bit 31    */
  itmp &= 0xFFFF7FFF;				/* Clear 'sign' bit from bit 15 */

  return( itmp );
} /* End of 'read_integer()' */

/*-------------------------------------------------------------------------*/

/*  'read_double_integer()'
 *
 *  Function to convert a Receiver-format 32-bit integer to native int
 */

  unsigned int
read_double_integer( void )
{
  unsigned int itmp; /* Holds the result of convertion: Receiver 'long'->integer */
  unsigned int wd;   /* Used for reading 16-bit 'words' from buffer */

  /* If read_word() fails, a zero is returned as a default value */
  wd = read_word();
  if( wd == ERROR ) return( 0 );
  itmp = wd;
  wd = read_word();
  if( wd == ERROR ) return( 0 );
  itmp |= wd << 16;

  return( itmp );
} /* End of 'read_double_integer()' */

/*------------------------------------------------------------------------*/

/*  'convert_float()'
 *
 *  Function to convert IEEE754-format 'double' fpnum
 *  to a 32-bit Receiver-format 'float' held in 'bit_field'.
 *  THIS PROCESS IS ENDIAN-DEPENDENT!
 */

  void
convert_float( double fpnum, unsigned int *bit_field )
{
  unsigned int *iptr;	/* Pointer to int used in manipulating bit field     */
  double *dptr; /* Pointer to double used in manipulating bit fields */

  *bit_field = 0;
  if ( fpnum != 0.0 )
  {
	dptr = &fpnum;
	iptr = (unsigned int *)dptr;
	unsigned int itmp = *iptr;
	/* Try exchanging these lines if convertion fails */
	*bit_field += ( ( itmp >> 21 ) & 0x700 ); /* ^v */
	iptr++;                                   /* ^v */
	itmp = *iptr;                             /* ^v */
	/* Try exchanging these lines if convertion fails */
	*bit_field += ( itmp >> 20 & 0x7FF ) - 0x37E + ( itmp & 0x80000000 ) +
	  ( itmp << 11 & 0x7FFFF800 );
  }
  else return;

} /* End of 'convert_float()' */

/*-------------------------------------------------------------------------*/

/*  'convert_extended_float()'
 *
 *  Function to convert IEEE754-format 'double' fpnum to a 48-bit
 *  Receiver-format 'extended float' held in lower_field and upper_field.
 *  THIS PROCESS IS ENDIAN-DEPENDENT!
 */

  void
convert_extended_float( double fpnum, unsigned int *lower_field, unsigned int *upper_field )
{
  unsigned int *iptr;     /* Pointer to int used in manipulating bit fields	*/
  double *dptr;  /* Pointer to double used in manipulating bit fields	*/

  *lower_field = *upper_field = 0;
  if ( fpnum != 0.0 )
  {
	dptr = &fpnum;
	iptr = ( unsigned int *) dptr;
	unsigned int itmp = *iptr;
	/* Try exchanging these lines if convertion fails */
	*lower_field += ( itmp >> 5 & 0x7FFFF00 ); /* ^v */
	iptr++;                                    /* ^v */
	itmp = *iptr;                              /* ^v */
	/* Try exchanging these lines if convertion fails */
	*lower_field += ( itmp >> 20 & 0x7FF ) - 0x37E + ( itmp << 27 & 0xF8000000 );
	*upper_field += ( itmp >> 16 & 0x8000 ) + ( itmp >> 5 & 0x7FFF );
  }
  else return;

} /* End of 'convert_extended_float()' */

/*------------------------------------------------------------------------*/
