/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  comms.c
 *
 *  Functions to read and write data to the GPS Receiver
 *  via the serial port in single bytes or 16-bit 'words',
 *  or to send and receive basic blocks such as message
 *  headers and ACK/NACK signals
 */

#include "rwgps.h"

/* Buffer for Receiver data, stored as 16-bit 'words' */
static unsigned int buffer[ BUFF_SIZE ];
static int buff_index;			/* Index into buffer above */
static unsigned int msg_id;		/* Message Identity number */
static unsigned int wd_count;	/* Message Data Word Count */
static unsigned int proto;		/* Message protocol number */

/*  'input_word()'
 *
 *  Function to input a Receiver-format
 *  16-bit 'word' from the Serial port
 */

/*-------------------------------------------------------------------------*/

  unsigned int
input_word( void )
{
  unsigned int word; /* Used to assemble the Receiver-format 'word' */
  unsigned char ch;  /* Byte read from the serial port     */

  read_serial( &ch );
  if( isModeSet(MD_READFAIL) )
	return( ERROR );
  word = (unsigned int)ch;

  read_serial( &ch );
  if( isModeSet(MD_READFAIL) )
	return( ERROR );
  word |= ((unsigned int)ch) << 8;

  return( word );
}

/*-------------------------------------------------------------------------*/

/*  'output_word()'
 *
 *  Function to send a 16-bit 'word' to the Receiver
 */

  void
output_word( unsigned int word )
{
  /* Character used to hold intermediate values */
  unsigned char ch;

  ch = (unsigned char)(word & 0xFF);
  write_serial( &ch );
  ch = (unsigned char)((word >> 8) & 0xFF);
  write_serial( &ch );

} /* End of 'output_word()' */

/*-------------------------------------------------------------------------*/

/*  'output_int()'
 *
 *  Function to send a 32-bit native 'int' to the Receiver
 */

  void
output_int( unsigned int nint )
{
  unsigned char ch; /* Character used to hold intermediate values */
  int idx; 	    /* index for loops etc */

  for ( idx = 0; idx < 4; idx++ )
  {
	ch = (unsigned char)(nint & 0xFF);
	write_serial( &ch );
	nint >>= 8;
  }

} /* End of 'output_int()' */

/*-------------------------------------------------------------------------*/

/*  'read_word()'
 *
 *  Function to read a Receiver-format 16-bit 'word' from the buffer
 */

  unsigned int
read_word( void )
{
  if( buff_index < BUFF_SIZE ) /* Just in case, should never fail */
	return ( buffer[ buff_index++ ] );
  else
	return ( ERROR );

} /* End of 'read_word()' */

/*-------------------------------------------------------------------------*/

/*  'ack_handler()'
 *
 *  Function to handle ACK/NACK messages from
 *  GPS Receiver sent in response to a
 *  transmitted Command or Data Message
 */

  void
ack_handler( void )
{
  /* Identity number of transmitted Message */
  unsigned int xmitted_id;

  /* For transmitted messages, Mode flags = message id */
  xmitted_id = current_mode() & 0xFF;

  /*  Wait for a valid Header from the Receiver
   *  with a message id matching the id of the
   *  Message that was sent to the Receiver
   */
  do
  {
	/* Abort if input_header() fails */
	if( input_header() == ERROR ) return;
  }
  /* Look for matching message id */
  while( message_id() != xmitted_id );

  /* Halt program if a NACK response
   * is received from Receiver */
  if( message_protocol() == NACK_RECEIVED )
	set_mode( MD_PROGHALT );

  /* A chance to see the Receiver's response */
  ncscreen_status();
  sleep(2);

} /* End of 'ack_handler()' */

/*-------------------------------------------------------------------------*/

/*  'send_header()'
 *
 *  Function to output a Message Header to GPS Receiver
 */

  void
send_header( unsigned int message_id, unsigned int word_count, unsigned int protocol )
{
  unsigned int checksum; /* Calculated checksum to be output to Receiver */

  /* Process checksum to Receiver format  */
  checksum = HEADER_START + message_id + word_count + protocol;
  process_checksum( &checksum );

  /* Output first 'magic' word of headers */
  output_word( HEADER_START );
  output_word( message_id );
  output_word( word_count );
  output_word( protocol );
  output_word( checksum );

} /* End of 'send_header()' */

/*-------------------------------------------------------------------------*/

/*  'input_header()'
 *
 *  Function to detect a valid Message Header from the Serial port stream
 *  and return Header ID, Data Word Count and Communication Protocol
 */

  unsigned int
input_header( void )
{
  unsigned char ch;			  /* Byte read from the serial port	*/
  unsigned int magic = 0;     /* Used in detecting 'magic' Header word (=0x81FF) */
  unsigned int calc_chksum;   /* Calculated checksum of header */
  unsigned int header_chksum; /* Header checksum supplied by Receiver */

  do  /* Look for a valid header */
  {
	do  /* Look for a header-start 'magic word' */
	{
	  /* Abort on Serial device read() failure */
	  read_serial( &ch );
	  if( isModeSet(MD_READFAIL) )
		return( ERROR );
	  magic >>= 8;          /* All this bit-field juggling is needed */
	  magic &= 0x00FF;      /* because the GPS Receiver transmitts   */
	  magic |= (unsigned int)(ch << 8); /* data in little-endian serial format */
	}
	/* Loop until a valid header start is detected (0x81FF) */
	while( magic != HEADER_START );

	/* Read in Message Id from Serial port */
	msg_id = input_word();
	if( msg_id == ERROR ) return( ERROR );

	/* Read in Message Word Count from Serial port */
	wd_count = input_word();
	if( wd_count == ERROR ) return( ERROR );

	/* Read in Message Protocol from Serial port */
	proto = input_word();
	if( proto == ERROR ) return( ERROR );

	/* Read in Header Checksum from Serial port */
	header_chksum = input_word();
	if( header_chksum == ERROR ) return( ERROR );

	/* Process checksum to Receiver format  */
	calc_chksum = magic + msg_id + wd_count + proto;
	process_checksum( &calc_chksum );
  }
  /* Loop until a valid header checksum is detected */
  while( calc_chksum != header_chksum );

  return( SUCCESS );
} /* End of 'input_header' */

/*-------------------------------------------------------------------------*/

/*  'store_data()'
 *
 *  Function to read a Message's data block and store in buffer
 */

  unsigned int
store_data( void )
{
  unsigned int idx;			/* Index for loops etc. */
  unsigned int wd;			/* Holds value of inputed 16-bit Receiver 'word' */
  unsigned int calc_chksum; /* Calculated checksum of Message data   */

  /* Just in case, should never happen */
  if( wd_count > BUFF_SIZE ) return ( ERROR );

  buff_index = calc_chksum = 0; /* Clear buffer index and checksum */

  /* Input Message data block and store in buffer */
  for( idx = 0; idx < wd_count; idx++ )
  {
	/* Abort on read() failure */
    wd = input_word();
	if( wd == ERROR ) return ( ERROR );
	buffer[ buff_index++ ] = wd;
	calc_chksum += wd; /* Add each 'word' to checksum */
  }

  buff_index = 0; /* Reset index to start of buffer */

  process_checksum( &calc_chksum ); /* Process calculated checksum */

  /*  If calculated checksum does not match checksum
   *  provided by Receiver, return error value
   */
  if( calc_chksum != input_word() ) return( ERROR );

  return( SUCCESS );
} /* End of 'store_data()' */

/*-------------------------------------------------------------------------*/

/*  'process_checksum()'
 *
 *  Function to process a calculated checksum
 *  to the format of the GPS Receiver
 */

  void
process_checksum( unsigned int *checksum )
{
  *checksum &= 0xFFFF; /* Reduce checksum to Mod 2^16 */

  /* Negate calculated checksum if not equal to 0xFFFF then Mod 2^16 */
  if ( *checksum != 0xFFFF )
	*checksum = ( - *checksum ) & 0xFFFF;

} /* End of 'process_checksum()' */

/*-------------------------------------------------------------------------*/

/*  'message_id()'
 *
 *  Returns the Message Identity number
 */

  unsigned int
message_id(void)
{
  return( msg_id );

} /* End of 'message_id()' */

/*-------------------------------------------------------------------------*/

/*  'message_protocol()'
 *
 *  Returns the Message Protocol number
 */

  unsigned int
message_protocol(void)
{
  return( proto );

} /* End of 'message_protocol()' */

/*-------------------------------------------------------------------------*/

