/*
 *  rwgps: A serial port driver/interface application for
 *  Rockwell's Microtracker LP (tm) GPS Receiver Module.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  main.c
 *
 *  Main body of rwgps application
 */

#include "rwgps.h"

static void sig_handler( int signal ); /* Signal handler */

  int
main( void )
{
  int ch;  /* Used for inputting characters with getch() */

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Open Serial port device (will exit if it fails) */
  open_serial();

  /* Initialize ncurses */
  initscr();
  keypad( stdscr, TRUE );
  nonl();
  cbreak();
  noecho();
  curs_set( FALSE );
  nodelay( stdscr, TRUE );

  /* Assuming terminal has color! */
  if( has_colors() )
  {
	start_color();

	init_pair( COLOR_WHITE,       COLOR_WHITE,   COLOR_BLACK );
	init_pair( COLOR_RED,	  COLOR_RED,     COLOR_BLACK );
	init_pair( COLOR_GREEN,	  COLOR_GREEN,   COLOR_BLACK );
	init_pair( COLOR_BLUE,	  COLOR_BLUE,    COLOR_BLACK );
	init_pair( COLOR_YELLOW,	  COLOR_YELLOW,  COLOR_BLACK );
	init_pair( COLOR_CYAN,	  COLOR_CYAN,    COLOR_BLACK );
	init_pair( COLOR_MAGENTA,	  COLOR_MAGENTA, COLOR_BLACK );
	init_pair( COLOR_KEYSTROKE,   COLOR_YELLOW,  COLOR_BLUE  );
	init_pair( COLOR_COMMAND,     COLOR_WHITE,   COLOR_BLUE  );
  }

  /* Set initial display to Main screen */
  set_mode( MD_MAINSCREEN );
  ncscreen_root();

  /*  This is the central program flow control loop in which input
   *  from the GPS Receiver is processed and displayed, user keystrokes
   *  are handled, commands are sent to the GPS Receiver and the response
   *  monitored through the acknowledgement messages of the Receiver
   */

  while( ((ch = getch()) != 'q') && (ch != 'Q') ) /* Quit on 'Q' */
  {
	switch( ch ) /* Set program 'mode' according to user keystroke */
	{
	  case 'm' : case 'M' : ncscreen_root();
							set_mode( MD_MAINSCREEN );
							break;
	  case 't' : case 'T' : set_mode( MD_BITRESULTS );
							send_message202();
							break;
	  case 'w' : case 'W' : set_mode( MD_WARMSTART );
							ncpost_warm_start_form();
							break;
	  case 'p' : case 'P' : set_mode( MD_POWERMANAGE );
							ncpost_powermanage_form();
							break;
	  case 'c' : case 'C' : set_mode( MD_SETCLOCK );
							ncpost_set_hwclock_form();
							break;
	  case 'h' : case 'H' : set_mode( MD_PROGHALT );

	} /* End of switch( ch ) */

	/*
	 *  Now direct program flow according to current 'mode'
	 */
	/* If HALT flag is set */
	if( isModeSet(MD_PROGHALT) )
	  sleep(1);  /* Loop at 1 sec intervals */
	else
	  if( isModeSet(MD_MAINSCREEN | MD_BITRESULTS | MD_SETCLOCK) )
	  {
		/*  Wait for a valid Message Header, then store raw data
		 *  from the Receiver into the buffer and if cheksum
		 *  validates, display the apropriate Message windows
		 */
		if( ( input_header() == 0 ) && ( store_data() == 0 ) )
		  ncscreen_message();
	  }

  } /* End of main 'while' loop */

  sig_handler( 0 ); /* Wrap up and quit on 'Q' */

  return( 0 );	    /* Keep compiler quite!    */

} /* End of 'main()' */

/*------------------------------------------------------------------------*/

/*  'ncscreen_message()'
 *
 *  Displays whichever Message has been stored in the Buffer
 */

  void
ncscreen_message( void )
{
  switch ( message_id() )
  {
	case 101 : /* Built-In-Test Results */
	  {
		message101_t message101;
		read_message101( &message101 );
		ncscreen_message101( &message101 );
		break;
	  }
	case 102 : /* GPS Visible Sattelite Data */
	  {
		message102_t message102;
		ncscreen_status();
		read_message102( &message102 );
		ncscreen_message102( &message102 );
		break;
	  }
	case 103 : /* Main Navigation, Time and Status Data */
	  {
		message103_t message103;
		ncscreen_status();
		read_message103( &message103 );
		ncscreen_message103( &message103 );
	  }

  } /* End of switch( message_id() ) */

} /* End of 'ncscreen_message()' */

/*------------------------------------------------------------------------*/

/*  'set_hwclock()'
 *
 *  Function to set the (CMOS) hardware clock to GPS (UTC)
 *  date and time from Receiver, using a pipe to hwclock(8)
 */

  int
set_hwclock( char * utc )
{
  FILE * pop;      /* popen() file pointer       */
  char cmnd[ 55 ]; /* Command string for popen() */

  /* Print UTC data in 'utc' into command string 'cmnd' */
  snprintf( cmnd, 55, "hwclock --utc --set --date=\"%s --utc\"", utc );
  pop = popen( cmnd, "w" ); /* open pipe to 'hwclock' */

  /* Return with error if popen() fails */
  if( pop == NULL )
  {
	set_mode( MD_POPENFAIL | MD_PROGHALT );
	return( -1 );
  }

  pclose( pop );

  return( 0 );
} /* End of 'set_hwclock' */

/*------------------------------------------------------------------------*/

/*  'isModeSet()', 'isModeClear()', 'set_mode()', 'clear_mode()'
 *
 *  Functions to test a given Mode flag and return TRUE if set or clear,
 *  and to set a given mode. Mode flags control program flow and
 *  determine which ncurses windows are displayed according to user
 *  input from the 'toolbar' at the bottom of the root window
 */

/* Used as a bit-field for various mode flags */
static int ModeFlags = 0;

/* Test if mode flag is set */
  int
isModeSet( int mode )
{
  return( ModeFlags & mode );
}

/*------------------------------------------------------------------------*/

/* Test if mode flag is clear */
  int
isModeClear( int mode )
{
  return( ~ModeFlags & mode );
}

/*------------------------------------------------------------------------*/

/* Set a given mode flag */
  void
set_mode( int mode )
{
  ModeFlags = mode;  /* Previously set flags are cleared */
  ncscreen_status(); /* Screen new status */
}

/*------------------------------------------------------------------------*/

/* Clear a given mode flag */
  void
clear_mode( int mode )
{
  ModeFlags &= ~mode;
  ncscreen_status(); /* Screen new status */
}

/*------------------------------------------------------------------------*/

/*  Returns the current mode flags */
  int
current_mode(void)
{
  return( ModeFlags );

} /* End of 'current_mode()' */

/*------------------------------------------------------------------------*/

/*  'sig_handler()'
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  clear();
  refresh();
  endwin();
  close_serial();

  fprintf( stderr, "\n" );
  switch( signal )
  {
	case SIGINT  : fprintf( stderr,  "rwgps: Exiting via User Interrupt\n" );
				   break;
	case SIGSEGV : fprintf( stderr,  "rwgps: Segmentation Fault\n" );
				   break;
	case SIGFPE  : fprintf( stderr,  "rwgps: Floating Point Exception\n" );
				   break;
	case SIGABRT : fprintf( stderr,  "rwgps: Abort Signal received\n" );
				   break;
	case SIGTERM : fprintf( stderr,  "rwgps: Termination Request received\n" );

  }

  exit( 0 );
} /* End of 'sig_handler()' */

/*------------------------------------------------------------------------*/

