/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "SoapySDR.h"
#include "filters.h"
#include "ifft.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../Glrpt/callback_func.h"
#include "../Glrpt/interface.h"
#include "../Glrpt/display.h"
#include "../Glrpt/utils.h"
#include "../Glrpt/display.h"
#include "../lrpt_decode/medet.h"
#include <SoapySDR/Device.h>
#include <SoapySDR/Formats.h>
#include <complex.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <semaphore.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Range of Gain slider */
#define GAIN_SCALE  100.0

/* Range factors for level gauges */
#define AGC_GAIN_RANGE1   1.0
#define AGC_GAIN_RANGE2   0.01

#define DATA_SCALE  10.0

static SoapySDRDevice *sdr = NULL;
static SoapySDRStream *rxStream = NULL;
static complex short  *stream_buff = NULL;
static size_t   stream_mtu;
static uint32_t sdr_decimate;
static double   data_scale;
uint8_t input_buf_cnt = 0;

/* DSP Filter Parameters */
#define FILTER_RIPPLE   5.0
#define FILTER_POLES    6

/*----------------------------------------------------------------------*/

/* SoapySDR_Stream()
 *
 * Runs in a thread of its own and loops around the
 * SoapySDRDevice_readStream() streaming function
 */
  static void *
SoapySDR_Stream( void *pid )
{
  /* Soapy streaming buffers */
  void *buffs[] = { stream_buff };
  int flags = 0;
  long long timeNs = 0;
  long timeout;

  double temp_i, temp_q;

  uint32_t
    sdr_decim_cnt,      /* Samples decimation counter */
    samp_buf_idx  = 0,  /* Output samples buffer index */
    strm_buf_idx  = rc_data.sdr_buf_length; /* Streaming buffer index */

  /* Data transfer timeout in uSec,
   * 10x longer to avoid dropped samples */

  timeout  = (long)stream_mtu * 10000000;
  timeout /= (long)rc_data.sdr_samplerate;

  /* Loop around SoapySDRDevice_readStream()
   * till reception stopped by the user */
  while( isFlagSet(STATUS_STREAMING) )
  {
    /* We need sdr_decimate summations to decimate samples */
    while( samp_buf_idx < rc_data.sdr_buf_length )
    {
      /* Summate sdr_decimate samples into one samples buffer */
      temp_i = 0.0;
      temp_q = 0.0;
      for( sdr_decim_cnt = 0; sdr_decim_cnt < sdr_decimate; sdr_decim_cnt++ )
      {
        /* Read new data from the sample stream when exhausted */
        if( strm_buf_idx >= rc_data.sdr_buf_length )
        {
          /* Read stream I/Q data from SDR device */
          SoapySDRDevice_readStream(
              sdr, rxStream, buffs, stream_mtu, &flags, &timeNs, timeout );
          strm_buf_idx = 0;
        }

        /* Summate samples to decimate */
        temp_i += creal( (complex double)stream_buff[strm_buf_idx] );
        temp_q += cimag( (complex double)stream_buff[strm_buf_idx] );
        strm_buf_idx++;
      } /* for( sdr_decim_cnt = 0; sdr_decim_cnt < ... */

      /* Top up Chebyshev LP filter buffers */
      input_buf_i[input_buf_cnt][samp_buf_idx] = temp_i / data_scale;
      input_buf_q[input_buf_cnt][samp_buf_idx] = temp_q / data_scale;

      samp_buf_idx++;
    } /* while( samp_buf_idx < rc_data.sdr_buf_length ) */
    samp_buf_idx = 0;

    // Writes IQ samples to file, for testing only
    /* {
      static FILE *fdi = NULL, *fdq = NULL;
      if( fdi == NULL ) fdi = fopen( "i.s", "w" );
      if( fdq == NULL ) fdq = fopen( "q.s", "w" );
      fwrite( input_buf_i[input_buf_cnt],
          sizeof(double), (size_t)rc_data.sdr_buf_length, fdi );
      fwrite( input_buf_q[input_buf_cnt],
          sizeof(double), (size_t)rc_data.sdr_buf_length, fdq );
    } */

    // Reads IQ samples from file, for testing only
    /* {
      static FILE *fdi = NULL, *fdq = NULL;
      if( fdi == NULL ) fdi = fopen( "i.s", "r" );
      if( fdq == NULL ) fdq = fopen( "q.s", "r" );
      fread( input_buf_i[input_buf_cnt],
          sizeof(double), (size_t)rc_data.sdr_buf_length, fdi );
      fread( input_buf_q[input_buf_cnt],
          sizeof(double), (size_t)rc_data.sdr_buf_length, fdq );
    } */

    // Writes the phase angle of samples, for testing only
    /* if( isFlagSet(STATUS_DECODING) )
    {
      static double prev = 0.0;
      double phase, delta, x, y;
      for( uint32_t idx = 0; idx < rc_data.sdr_buf_length; idx++ )
      {
        x = (double)(input_buf_i[input_buf_cnt][idx]);
        y = (double)(input_buf_q[input_buf_cnt][idx]);
        phase = atan2( fabs(x), fabs(y) ) * 57.3;
        if( (x > 0.0) && (y < 0.0) ) phase = 360.0 - phase;
        if( (x < 0.0) && (y > 0.0) ) phase = 180.0 - phase;
        if( (x < 0.0) && (y < 0.0) ) phase = 180.0 + phase;
        delta = phase - prev;
        prev  = phase;
        printf( "%6.1f  %6.1f\n", phase, delta );
      }
    } */

    /* Increment and reset input buffer index */
    input_buf_cnt++;
    if( input_buf_cnt >= NUM_INPUT_BUFFERS )
      input_buf_cnt = 0;

    int sval;
    sem_getvalue( &demod_semaphore, &sval );
    if( !sval ) sem_post( &demod_semaphore );

  } /* while( isFlagSet(STATUS_STREAMING) ) */

  /* Close device when streaming is stopped */
  SoapySDR_Close_Device();

  return( NULL );
} /* SoapySDR_Stream() */

/*----------------------------------------------------------------------*/

/* SoapySDR_Set_Center_Freq()
 *
 * Sets the Center Frequency of the Tuner
 */
  Bool
SoapySDR_Set_Center_Freq( uint32_t center_freq )
{
  /* Set the Center Frequency of the Tuner */
  int ret = SoapySDRDevice_setFrequency(
      sdr, SOAPY_SDR_RX, 0, (double)center_freq, NULL );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Center Frequency"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Error_Dialog();
    Display_Icon( status_icon, "gtk-no" );
    return( False );
  }

  /* Display center frequency in messages */
  gchar mesg[ MESG_SIZE ];
  snprintf( mesg, sizeof(mesg),
      _("Set SDR Frequency to %0.1fkHz"),
      (double)center_freq / 1000.0 );
  Show_Message( mesg, "green" );
  Display_Icon( status_icon, "gtk-yes" );

  return( True );
} /* SoapySDR_Set_Center_Freq() */

/*-----------------------------------------------------------------------*/

/* SoapySDR_Set_Tuner_Gain_Mode()
 *
 * Sets the Tuner Gain mode to Auto or Manual
 */
  void
SoapySDR_Set_Tuner_Gain_Mode( void )
{
  int ret;

  if( isFlagSet(TUNER_GAIN_AUTO) )
  {
    /* Check for support of auto gain control */
    if( !SoapySDRDevice_hasGainMode(sdr, SOAPY_SDR_RX, 0) )
    {
      Show_Message( _("Device has no Auto Gain Control"), "red" );
      Error_Dialog();
      Display_Icon( status_icon, "gtk-no" );
      return;
    }

    /* Set auto gain control */
    SoapySDR_Set_Tuner_Gain( 100.0 );
    ret = SoapySDRDevice_setGainMode( sdr, SOAPY_SDR_RX, 0, True );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to Set Auto Gain Control"), "red" );
      Show_Message( SoapySDRDevice_lastError(), "red" );
      Error_Dialog();
      Display_Icon( status_icon, "gtk-no" );
      return;
    }

    Show_Message( _("Set Auto Gain Control"), "green" );
  }
  else
  {
    /* Set manual gain control */
    ret = SoapySDRDevice_setGainMode( sdr, SOAPY_SDR_RX, 0, 0 );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to Set Manual Gain Control"), "red" );
      Show_Message( SoapySDRDevice_lastError(), "red" );
      Display_Icon( status_icon, "gtk-no" );
      Error_Dialog();
      return;
    }

    Show_Message( _("Set Manual Gain Control"), "green" );
    Display_Icon( status_icon, "gtk-yes" );
    SoapySDR_Set_Tuner_Gain( rc_data.tuner_gain );
  }

} /* SoapySDR_Set_Tuner_Gain_Mode() */

/*----------------------------------------------------------------------*/

/* SoapySDR_Set_Tuner_Gain()
 *
 * Set the Tuner Gain if in Manual mode
 */
  void
SoapySDR_Set_Tuner_Gain( double gain )
{
  gchar mesg[MESG_SIZE];

  /* Get range of available gains */
  SoapySDRRange range =
    SoapySDRDevice_getGainRange( sdr, SOAPY_SDR_RX, 0 );
  snprintf( mesg, sizeof(mesg),
      _("Tuner Gain Range: %0.1f-%0.1f dB"), range.maximum, range.minimum );
  Show_Message( mesg, "green" );

  /* Scale gain request to range of available gains */
  gain *= range.maximum - range.minimum;
  gain /= GAIN_SCALE;
  gain += range.minimum;

  /* Set device receiver gain */
  int ret = SoapySDRDevice_setGain( sdr, SOAPY_SDR_RX, 0, gain );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Tuner Gain"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Display_Icon( status_icon, "gtk-no" );
    return;
  }
  snprintf( mesg, sizeof(mesg),
      _("Set Tuner Gain to %0.1f dB"), gain );
  Show_Message( mesg, "green" );
  Display_Icon( status_icon, "gtk-yes" );

} /* SoapySDR_Set_Tuner_Gain() */

/*----------------------------------------------------------------------*/

/* SoapySDR_Init()
 *
 * Initialize SoapySDR by finding the specified SDR device,
 * instance it and setting up its working parameters */
  Bool
SoapySDR_Init( void )
{
  int ret = 0;
  size_t length, key, idx, mreq;
  gchar mesg[ MESG_SIZE ];
  SoapySDRKwargs *results;
  SoapySDRRange  *range;

  uint32_t temp;

  /* Abort if already init */
  if( isFlagSet(STATUS_SOAPYSDR_INIT) )
    return( True );

  /* Enumerate SDR devices, abort if no devices found */
  results = SoapySDRDevice_enumerate( NULL, &length );
  if( length == 0 )
  {
    Show_Message( _("No SoapySDR Device found"), "red" );
    Error_Dialog();
    Display_Icon( status_icon, "gtk-no" );
    return( False );
  }

  /* Use SDR device specified by index alone
   * if "auto" driver name specified in config */
  if( isFlagSet(AUTO_DETECT_SDR) )
  {
    Show_Message( _("Will use Auto-Detected SDR Device"), "orange" );
    idx = rc_data.device_index;
  }
  else
  {
    /* Look for device matching specified driver */
    snprintf( mesg, sizeof(mesg),
        _("Searching for SDR Device \"%s\""), rc_data.device_driver );
    Show_Message( mesg, "green" );

    for( idx = 0; idx < length; idx++)
    {
      for( key = 0; key < results[idx].size; key++ )
      {
        /* Look for "driver" key */
        ret = strcmp( results[idx].keys[key], "driver" );
        if( ret == 0 )
        {
          /* Match driver to one specified in config */
          ret = strcmp( results[idx].vals[key], rc_data.device_driver );
          if( (ret == 0) && (idx == rc_data.device_index) )
            break;
        }
      }

      if( (ret == 0) && (idx == rc_data.device_index) )
        break;
    } /* for( idx = 0; idx < length; idx++) */

    /* If no matching device found, abort */
    if( idx == length )
    {
      snprintf( mesg, sizeof(mesg),
          _("No Device: \"%s\"  Index: %u found"),
          rc_data.device_driver, rc_data.device_index );
      Show_Message( mesg, "red" );
      Display_Icon( status_icon, "gtk-no" );
      Error_Dialog();
      return( False );
    }
  } /* if( strcmp(rc_data.device_driver, "auto") == 0 ) */

  /* Find SDR driver name for auto detect case */
  if( isFlagSet(AUTO_DETECT_SDR) )
    for( key = 0; key < results[idx].size; key++ )
    {
      /* Look for "driver" key */
      ret = strcmp( results[idx].keys[key], "driver" );
      if( ret == 0 )
        Strlcpy( rc_data.device_driver,
            results[idx].vals[key], sizeof(rc_data.device_driver) );
    }

  /* Report SoapySDR driver to be used */
  snprintf( mesg, sizeof(mesg),
      _("Using Driver: \"%s\"  Device: %d"),
      rc_data.device_driver, (int)idx );
  Show_Message( mesg, "black" );

  /* Create device instance, abort on error */
  sdr = SoapySDRDevice_make( &results[idx] );
  if( sdr == NULL )
  {
    Show_Message( _("SoapySDRDevice_make() failed:"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Display_Icon( status_icon, "gtk-no" );
    Error_Dialog();
    return( False );
  }
  SoapySDRKwargsList_clear( results, length );

  /* Display Tuner Type */
  char *hrd = SoapySDRDevice_getHardwareKey( sdr );
  GtkEntry *entry = GTK_ENTRY(
      Builder_Get_Object(main_window_builder, "sdr_tuner_entry") );
  gtk_entry_set_text( entry, hrd );
  snprintf( mesg, sizeof(mesg), _("Instantiated SDR Device \"%s\""), hrd );
  Show_Message( mesg, "green" );
  Display_Icon( status_icon, "gtk-yes" );
  Mem_Free( (void **)&hrd );

  /* Set the Center Frequency of the Device */
  if( !SoapySDR_Set_Center_Freq( rc_data.sdr_center_freq ) )
    return( False );

  /* Set the Frequency Correction factor for the device */
  if( SoapySDRDevice_hasFrequencyCorrection(sdr, SOAPY_SDR_RX, 0) )
  {
    Show_Message( _("Device has Frequency Correction"), "black" );
    if( rc_data.freq_correction )
    {
      ret = SoapySDRDevice_setFrequencyCorrection(
          sdr, SOAPY_SDR_RX, 0, rc_data.freq_correction );
      if( ret != SUCCESS )
      {
        Show_Message( _("Failed to set Frequency Correction"), "red" );
        Error_Dialog();
        Display_Icon( status_icon, "gtk-no" );
        return( False );
      }

      snprintf( mesg, sizeof(mesg),
          _("Set Frequency Correction to %d ppm"),
          rc_data.freq_correction );
      Show_Message( mesg, "green" );
      Display_Icon( status_icon, "gtk-yes" );
    }
  }

  /* Get the range of available sample rates */
  range = SoapySDRDevice_getSampleRateRange( sdr, SOAPY_SDR_RX, 0, &length );

  /*** List sampling rates and select lowest available ***/
  /* Prime this to high value to look for a minimum */
  rc_data.sdr_samplerate = 100000000;

  /* This is the minimum preferred value for the demodulator
   * effective sample rate. It could have been 2 * symbol_rate
   * but this can result in unfavorable sampling rates */
  temp = rc_data.oversampling * rc_data.symbol_rate;
  snprintf( mesg, sizeof(mesg),
      _("QPSK Symbol Rate: %u Sy/s"), rc_data.symbol_rate );
  Show_Message( mesg, "green" );

  /* Select lowest sampling rate above minimum demod sampling rate */
  Show_Message( _("Available Sample Rate Ranges S/s"), "black" );
  for( idx = 0; idx < length; idx++ )
  {
    snprintf( mesg, sizeof(mesg),
        "Max %u  Min %u",
        (uint32_t)range[idx].maximum, (uint32_t)range[idx].minimum );
    Show_Message( mesg, "black" );

    /* Select the lowest sample rate above demod sample rate */
    uint32_t min = (uint32_t)range[idx].minimum;
    if( (rc_data.sdr_samplerate > min) && (temp <= min) )
      rc_data.sdr_samplerate = (uint32_t)range[idx].minimum;

  } /* for( idx = 0; idx < length; idx++ ) */
  Mem_Free( (void **)&range );

  /* Set SDR Sample Rate */
  ret = SoapySDRDevice_setSampleRate(
      sdr, SOAPY_SDR_RX, 0, (double)rc_data.sdr_samplerate );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set ADC Sample Rate"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Display_Icon( status_icon, "gtk-no" );
    Error_Dialog();
    return( False );
  }

  /* Get actual sample rate and display */
  rc_data.sdr_samplerate =
    (uint32_t)( SoapySDRDevice_getSampleRate(sdr, SOAPY_SDR_RX, 0) );
  snprintf( mesg, sizeof(mesg),
      _("Set Sampling Rate to %uS/s"), rc_data.sdr_samplerate );
  Show_Message( mesg, "green" );

  /* Find sample rate decimation factor which
   * is the nearest power of 2, up to 32 */
  sdr_decimate = (uint32_t)rc_data.sdr_samplerate / temp;
  int sav = 1;
  int min = 64; /* Prime to find a min */
  for( int i = 0; i <= 5; i++ )
  {
    int diff = abs( (int)sdr_decimate - (1 << i) );
    if( min > diff )
    {
      min = diff;
      sav = i;
    }
  }
  sdr_decimate = (uint32_t)( 1 << sav );
  data_scale   = (double)sdr_decimate * DATA_SCALE;

  /* We now need to calculate the sample rate decimation factor for
   * high sample rates and the new effective demodulator sample rate */
  rc_data.demod_samplerate =
    (double)rc_data.sdr_samplerate / (double)sdr_decimate;
  snprintf( mesg, sizeof(mesg),
      _("Sampling Rate Decimation: %u"), sdr_decimate );
  Show_Message( mesg, "green" );
  snprintf( mesg, sizeof(mesg),
      _("Demod Sampling Rate: %8.1f"), rc_data.demod_samplerate );
  Show_Message( mesg, "green" );

  /* Set Tuner Gain Mode to auto or manual as per config file */
  SoapySDR_Set_Tuner_Gain_Mode();

  /* Set up receiving stream */
  Show_Message( _("Setting up Receive Stream"), "black" );
  ret = SoapySDRDevice_setupStream(
      sdr, &rxStream, SOAPY_SDR_RX, SOAPY_SDR_CS16, NULL, 0, NULL );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set up Receive Stream"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Error_Dialog();
    Display_Icon( status_icon, "gtk-no" );
    return( False );
  }
  Show_Message( _("Receive Stream set up OK"), "green" );

  /* Find stream MTU and use as read buffer size */
  stream_mtu = SoapySDRDevice_getStreamMTU( sdr, rxStream );
  snprintf( mesg, sizeof(mesg),
      _("Receive Stream MTU: %d"), (int)stream_mtu );
  Show_Message( mesg, "green" );
  rc_data.sdr_buf_length = (uint32_t)stream_mtu;

  /* Allocate stream buffer */
  mreq = stream_mtu * sizeof( complex short );
  Mem_Alloc( (void **)&stream_buff, mreq );

  /* Allocate local data buffers */
  mreq = stream_mtu * sizeof( double );
  for( idx = 0; idx < NUM_INPUT_BUFFERS; idx++ )
  {
    input_buf_i[idx] = NULL;
    input_buf_q[idx] = NULL;
    Mem_Alloc( (void **)&input_buf_i[idx], mreq );
    Mem_Alloc( (void **)&input_buf_q[idx], mreq );
  }

  /* Init Chebyshev I/Q data Low Pass Filters */
  Init_Chebyshev_Filter(
      &filter_data_i,
      rc_data.sdr_buf_length,
      rc_data.sdr_filter_bw,
      rc_data.demod_samplerate,
      FILTER_RIPPLE,
      FILTER_POLES,
      FILTER_LOWPASS );

  Init_Chebyshev_Filter(
      &filter_data_q,
      rc_data.sdr_buf_length,
      rc_data.sdr_filter_bw,
      rc_data.demod_samplerate,
      FILTER_RIPPLE,
      FILTER_POLES,
      FILTER_LOWPASS );

  /* Initialize ifft. Waterfall with is an odd number
   * to provide a center line. IFFT requires a width
   * that is a power of 2 */
  if( !Initialize_IFFT((int16_t)wfall_width + 1) )
    return( False );

  /* Wait a little for things to settle and set init OK flag */
  sleep( 1 );
  SetFlag( STATUS_SOAPYSDR_INIT );
  Show_Message( _("SoapySDR Initialized OK"), "green" );
  Display_Icon( status_icon, "gtk-yes" );

  return( True );
} /* SoapySDR_Init() */

/*----------------------------------------------------------------------*/

/* SoapySDR_Activate_Stream()
 *
 * Activates IQ data transfer from SDR device
 */
  Bool
SoapySDR_Activate_Stream( void )
{
  /* Thread ID for the newly created thread */
  pthread_t pthread_id;

  /* Activate receive stream */
  int ret = SoapySDRDevice_activateStream( sdr, rxStream, 0, 0, 0 );
  if( ret != SUCCESS )
  {
    ClearFlag( STATUS_STREAMING );
    Show_Message( _("Failed to activate Receive Stream"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Error_Dialog();
    Display_Icon( status_icon, "gtk-no" );
    return( False );
  }

  /* Create a thread for async read from SDR device */
  SetFlag( STATUS_STREAMING );
  ret = pthread_create( &pthread_id, NULL, SoapySDR_Stream, NULL );
  if( ret != SUCCESS )
  {
    ClearFlag( STATUS_STREAMING );
    ClearFlag( STATUS_SOAPYSDR_INIT );
    Show_Message( _("Failed to create Streaming thread"), "red" );
    Show_Message( SoapySDRDevice_lastError(), "red" );
    Error_Dialog();
    Display_Icon( status_icon, "gtk-no" );
    return( False );
  }

  Show_Message( _("Receive Stream activated OK"), "green" );
  Display_Icon( status_icon, "gtk-yes" );

  return( True );
} /* SoapySDR_Activate_Stream() */

/*----------------------------------------------------------------------*/

/* SoapySDR_Close_Device()
 *
 * Closes thr SDR device, if open
 */
  void
SoapySDR_Close_Device( void )
{
  int ret;

  /* Deactivate and close the stream */
  ClearFlag( STATUS_SOAPYSDR_INIT );
  if( (rxStream != NULL) && (sdr != NULL) )
  {
    ret = SoapySDRDevice_deactivateStream( sdr, rxStream, 0, 0 );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to Deactivate Stream"), "red" );
      Show_Message( SoapySDRDevice_lastError(), "red" );
      Error_Dialog();
    }

    ret = SoapySDRDevice_closeStream( sdr, rxStream );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to Close Stream"), "red" );
      Show_Message( SoapySDRDevice_lastError(), "red" );
      Error_Dialog();
    }

    rxStream = NULL;
  } /* if( rxStream != NULL ) */

  /* Close the SDR device */
  if( sdr != NULL )
  {
    ret = SoapySDRDevice_unmake( sdr );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to Close SDR device"), "red" );
      Show_Message( SoapySDRDevice_lastError(), "red" );
      Error_Dialog();
    }
    sdr = NULL;
  }

  /* Free the samples buffer */
  Mem_Free( (void **)&stream_buff );
  for( uint8_t idx = 0; idx < NUM_INPUT_BUFFERS; idx++ )
  {
    Mem_Free( (void **)&input_buf_i[idx] );
    Mem_Free( (void **)&input_buf_q[idx] );
  }

  /* De-initialize Low Pass filter */
  Deinit_Chebyshev_Filter( &filter_data_i );
  Deinit_Chebyshev_Filter( &filter_data_q );

  /* De-initialize and free memory allocations */
  Medet_Deinit();
  Demod_Deinit();
  Deinit_Ifft();

  /* Cancel any alarms */
  alarm( 0 );
  ClearFlag( ALARM_ACTION_START );
  ClearFlag( ALARM_ACTION_STOP );
  ClearFlag( STATUS_FLAGS_ALL );

  ClearFlag( STATUS_STREAMING );
  Display_Icon( status_icon, "gtk-no" );

} /* SoapySDR_Close_Device() */

/*-----------------------------------------------------------------------*/

