/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef JPEG_ENCODER_H
#define JPEG_ENCODER_H  1

#include "../common/common.h"
#include <stdint.h>
#include <stdio.h>

/*---------------------------------------------------------------------------*/

#define JPGE_MAX(a,b) (((a)>(b))?(a):(b))
#define JPGE_MIN(a,b) (((a)<(b))?(a):(b))

#define RGBA    1  // Signal image source to be rgba type
#define RGB     0  // Signal image source to be rgb type

/*---------------------------------------------------------------------------*/

typedef double dct_t;
typedef int16_t dctq_t; // quantized

// JPEG chroma subsampling factors. Y_ONLY (grayscale images)
// and H2V2 (color images) are the most common.
enum subsampling_t
{
  Y_ONLY = 0,
  H1V1   = 1,
  H2V1   = 2,
  H2V2   = 3
};

/*---------------------------------------------------------------------------*/

// JPEG compression parameters structure.
typedef struct
{
  // Quality: 1-100, higher is better. Typical values are around 50-95.
  float m_quality;

  // m_subsampling:
  // 0 = Y (grayscale) only
  // 1 = YCbCr, no subsampling (H1V1, YCbCr 1x1x1, 3 blocks per MCU)
  // 2 = YCbCr, H2V1 subsampling (YCbCr 2x1x1, 4 blocks per MCU)
  // 3 = YCbCr, H2V2 subsampling (YCbCr 4x1x1, 6 blocks per MCU-- very common)
  enum subsampling_t m_subsampling;

  // Disables CbCr discrimination - only intended for testing.
  // If true, the Y quantization table is also used for the CbCr channels.
  Bool m_no_chroma_discrim_flag;

} compression_params_t;

/*---------------------------------------------------------------------------*/

struct component
{
  uint8_t m_h_samp, m_v_samp;
  int m_last_dc_val;
};

struct huffman_table
{
  uint32_t m_codes[256];
  uint8_t  m_code_sizes[256];
  uint8_t  m_bits[17];
  uint8_t  m_val[256];
  uint32_t m_count[256];
};

struct huffman_dcac
{
  int32_t m_quantization_table[64];
  struct huffman_table dc, ac;
};

struct image
{
  int m_x, m_y;

  float *m_pixels;
  dctq_t *m_dctqs; // quantized dcts
};

struct sym_freq
{
  uint32_t m_key;
  uint32_t m_sym_index;
};

/*---------------------------------------------------------------------------*/

/* JPEG encoder data structure */
struct jpeg_encoder
{
  FILE *m_pStream;
  compression_params_t m_params;
  uint8_t m_num_components;
  struct component m_comp[3];
  struct huffman_dcac m_huff[2];
  enum { JPGE_OUT_BUF_SIZE = 2048 } buff;
  uint8_t m_out_buf[JPGE_OUT_BUF_SIZE];
  uint8_t *m_pOut_buf;
  uint32_t m_out_buf_left;
  uint32_t m_bit_buffer;
  uint32_t m_bits_in;
  uint8_t m_all_stream_writes_succeeded;
  int m_mcu_w, m_mcu_h;
  int m_x, m_y;
  struct image m_image[3];
};

/*---------------------------------------------------------------------------*/

/* jpeg.c */
Bool jepg_encoder_compression_parameters(compression_params_t *comp_params, float m_quality, enum subsampling_t m_subsampling, Bool m_no_chroma_discrim_flag);
Bool jepg_encoder_compress_image_to_file(char *file_name, int width, int height, int num_channels, const uint8_t *pImage_data, compression_params_t *comp_params);

/*---------------------------------------------------------------------------*/

#endif // JPEG_ENCODER

