/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "rc_config.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../lrpt_decode/medet.h"
#include "../lrpt_demod/demod.h"
#include "../sdr/ifft.h"
#include <cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>

/* IFFT Signal Amplitude averaging window */
#define AMPL_AVE_WIN   4
#define AMPL_AVE_MUL   3

/* Parameters used in coloring level bars */
#define TRANSITION_BAND   0.2
#define RED_THRESHOLD     4.0
#define GREEN_THRESHOLD   1.5

/* Number of QPSK constellation points to plot.
 *** Must be <= SYM_CHUNKSIZE / 2 ***/
#define QPSK_CONST_POINTS   512

/*------------------------------------------------------------------------*/

/* IFFT_Bin_Value()
 *
 * Calculates IFFT bin values with auto level control
 */
  static int
IFFT_Bin_Value( int sum_i, int sum_q, Bool reset )
{
  /* Value of ifft output "bin" */
  static int bin_val = 0;

  /* Maximum value of ifft bins */
  static int bin_max = 1000, max = 0;

  /* Calculate sliding window average of max bin value */
  if( reset )
  {
    bin_max = max;
    if( !bin_max ) bin_max = 1;
    max = 0;
  }
  else
  {
    /* Calculate average signal power at each frequency (bin) */
    bin_val  = bin_val * AMPL_AVE_MUL;
    bin_val += sum_i * sum_i + sum_q * sum_q;
    bin_val /= AMPL_AVE_WIN;

    /* Record max bin value */
    if( max < bin_val )
      max = bin_val;

    /* Scale bin values to 255 depending on max value */
    int ret = (255 * bin_val) / bin_max;
    if( ret > 255 ) ret = 255;
    return( ret );
  }

  return( 0 );
} /* IFFT_Bin_Value() */

/*------------------------------------------------------------------------*/

/* Color codes the pixels of the
 * waterfall according to their value
 */
  static void
Colorize( guchar *pix, int pixel_val )
{
  // Color palette (Viridis colormap)
  static uint8_t
    R[256] = {
      68,  68,  69,  69,  69,  70,  70,  70,
      70,  71,  71,  71,  71,  71,  72,  72,
      72,  72,  72,  72,  72,  72,  72,  72,
      72,  72,  72,  72,  72,  71,  71,  71,
      71,  71,  70,  70,  70,  70,  69,  69,
      69,  68,  68,  68,  67,  67,  67,  66,
      66,  65,  65,  65,  64,  64,  63,  63,
      62,  62,  61,  61,  60,  60,  59,  59,
      58,  58,  57,  57,  56,  56,  55,  55,
      54,  54,  53,  53,  52,  52,  51,  51,
      51,  50,  50,  49,  49,  48,  48,  47,
      47,  47,  46,  46,  45,  45,  45,  44,
      44,  43,  43,  43,  42,  42,  41,  41,
      41,  40,  40,  40,  39,  39,  38,  38,
      38,  37,  37,  37,  36,  36,  35,  35,
      35,  34,  34,  34,  33,  33,  33,  32,
      32,  32,  32,  31,  31,  31,  31,  31,
      30,  30,  30,  30,  30,  30,  30,  30,
      30,  31,  31,  31,  31,  32,  32,  33,
      33,  34,  35,  35,  36,  37,  38,  39,
      40,  41,  42,  43,  44,  46,  47,  48,
      50,  51,  53,  54,  56,  57,  59,  61,
      62,  64,  66,  68,  70,  72,  73,  75,
      77,  79,  81,  83,  85,  88,  90,  92,
      94,  96,  98, 101, 103, 105, 108, 110,
      112, 115, 117, 119, 122, 124, 127, 129,
      132, 134, 137, 139, 142, 144, 147, 149,
      152, 155, 157, 160, 163, 165, 168, 171,
      173, 176, 179, 181, 184, 187, 189, 192,
      195, 197, 200, 203, 205, 208, 211, 213,
      216, 219, 221, 224, 226, 229, 232, 234,
      237, 239, 242, 244, 247, 249, 251, 254
    },
    G[256] = {
      1,   2,   3,   5,   6,   8,   9,  11,
      12,  14,  15,  17,  18,  20,  21,  22,
      24,  25,  26,  28,  29,  30,  32,  33,
      34,  36,  37,  38,  39,  41,  42,  43,
      44,  46,  47,  48,  49,  51,  52,  53,
      54,  56,  57,  58,  59,  60,  62,  63,
      64,  65,  66,  67,  69,  70,  71,  72,
      73,  74,  75,  77,  78,  79,  80,  81,
      82,  83,  84,  85,  86,  87,  88,  89,
      91,  92,  93,  94,  95,  96,  97,  98,
      99, 100, 101, 102, 103, 104, 105, 106,
      107, 108, 109, 110, 111, 112, 112, 113,
      114, 115, 116, 117, 118, 119, 120, 121,
      122, 123, 124, 125, 126, 127, 128, 129,
      130, 131, 131, 132, 133, 134, 135, 136,
      137, 138, 139, 140, 141, 142, 143, 144,
      145, 146, 147, 147, 148, 149, 150, 151,
      152, 153, 154, 155, 156, 157, 158, 159,
      160, 161, 162, 163, 163, 164, 165, 166,
      167, 168, 169, 170, 171, 172, 173, 174,
      175, 175, 176, 177, 178, 179, 180, 181,
      182, 183, 183, 184, 185, 186, 187, 188,
      189, 190, 190, 191, 192, 193, 194, 194,
      195, 196, 197, 198, 198, 199, 200, 201,
      201, 202, 203, 204, 204, 205, 206, 206,
      207, 208, 208, 209, 210, 210, 211, 212,
      212, 213, 213, 214, 215, 215, 216, 216,
      217, 217, 218, 218, 219, 219, 220, 220,
      221, 221, 221, 222, 222, 223, 223, 223,
      224, 224, 225, 225, 225, 226, 226, 226,
      227, 227, 227, 228, 228, 228, 229, 229,
      229, 230, 230, 230, 230, 231, 231, 231
    },
    B[256] = {
      84,  85,  87,  88,  90,  91,  93,  94,
      96,  97,  98, 100, 101, 102, 104, 105,
      106, 108, 109, 110, 111, 112, 113, 115,
      116, 117, 118, 119, 120, 121, 121, 122,
      123, 124, 125, 126, 126, 127, 128, 129,
      129, 130, 131, 131, 132, 132, 133, 133,
      134, 134, 135, 135, 136, 136, 136, 137,
      137, 137, 138, 138, 138, 138, 139, 139,
      139, 139, 140, 140, 140, 140, 140, 140,
      141, 141, 141, 141, 141, 141, 141, 141,
      141, 142, 142, 142, 142, 142, 142, 142,
      142, 142, 142, 142, 142, 142, 142, 142,
      142, 142, 142, 142, 142, 142, 142, 142,
      142, 142, 142, 142, 142, 142, 142, 142,
      142, 142, 142, 142, 142, 142, 142, 142,
      142, 141, 141, 141, 141, 141, 141, 141,
      140, 140, 140, 140, 140, 139, 139, 139,
      139, 138, 138, 138, 137, 137, 137, 136,
      136, 136, 135, 135, 134, 134, 134, 133,
      133, 132, 131, 131, 130, 130, 129, 129,
      128, 127, 127, 126, 125, 124, 124, 123,
      122, 121, 121, 120, 119, 118, 117, 116,
      115, 114, 113, 112, 111, 110, 109, 108,
      107, 106, 105, 104, 102, 101, 100,  99,
      98,  96,  95,  94,  92,  91,  90,  88,
      87,  85,  84,  82,  81,  79,  78,  76,
      75,  73,  72,  70,  68,  67,  65,  63,
      62,  60,  58,  57,  55,  53,  51,  50,
      48,  46,  45,  43,  41,  39,  38,  36,
      35,  33,  32,  30,  29,  28,  27,  26,
      25,  24,  24,  24,  24,  24,  25,  25,
      26,  27,  28,  30,  31,  33,  35,  36
    };

  // Enter the palette to the pixmap
  pix[0] = R[pixel_val];
  pix[1] = G[pixel_val];
  pix[2] = B[pixel_val];

} /* Colorize() */

/*------------------------------------------------------------------------*/

/* Display_Waterfall()
 *
 * Displays IFFT Spectrum as "waterfall"
 */
  gboolean
Display_Waterfall( gpointer data )
{
  int
    vert_lim,  /* Limit of vertical index for copying lines */
    idh, idv,  /* Index to hor. and vert. position in warterfall */
    pixel_val, /* Greyscale value of pixel derived from ifft o/p  */
    idf,       /* Index to ifft output array */
    i, len;

  /* Pointer to current pixel */
  static guchar *pix;

  /* Copy each line of waterfall to next one */
  vert_lim = wfall_height - 2;
  for( idv = vert_lim; idv > 0; idv-- )
  {
    pix = wfall_pixels + wfall_rowstride * idv + wfall_n_channels;

    for( idh = 0; idh < wfall_width; idh++ )
    {
      pix[0] = pix[ -wfall_rowstride];
      pix[1] = pix[1-wfall_rowstride];
      pix[2] = pix[2-wfall_rowstride];
      pix += wfall_n_channels;
    }
  }

  /* Go to top left +1 hor. +1 vert. of pixbuf */
  pix = wfall_pixels + wfall_rowstride + wfall_n_channels;

  /* IFFT produces an output of positive and negative
   * frequencies and it output is handled accordingly */
  IFFT( ifft_data );

  /* Calculate bin values after IFFT */
  len = ifft_data_length / 4;

  /* Do the "positive" frequencies */
  idf = ifft_data_length / 2;
  for( i = 0; i < len; i++ )
  {
    /* Calculate vector magnitude of
     * signal at each freq. ("bin") */
    pixel_val = IFFT_Bin_Value(
        ifft_data[idf], ifft_data[idf + 1], False );
    idf += 2;

    /* Color code signal strength */
    Colorize( pix, pixel_val );
    pix += wfall_n_channels;

  } /* for( i = 1; i < len; i++ ) */

  /* Do the "negative" frequencies */
  idf = 2;
  for( i = 1; i < len; i++ )
  {
    /* Calculate vector magnitude of
     * signal at each freq. ("bin") */
    pixel_val = IFFT_Bin_Value(
        ifft_data[idf], ifft_data[idf + 1], False );
    idf += 2;

    /* Color code signal strength */
    Colorize( pix, pixel_val );
    pix += wfall_n_channels;

  } /* for( i = 1; i < len; i++ ) */

  /* Reset function */
  IFFT_Bin_Value( ifft_data[0], ifft_data[0], True );

  /* At last draw waterfall */
  gtk_widget_queue_draw( ifft_drawingarea );

  return( FALSE );
} /* Display_Waterfall() */

/*------------------------------------------------------------------------*/

/*  Display_QPSK_Const()
 *
 *  Displays the QPSK constellation
 */
  gboolean
Display_QPSK_Const( gpointer data )
{
  /* Pointer to current pixel */
  static guchar *pix;

  /* Horizontal and Vertical index
   * to QPSK drawingarea pixels */
  gint idh, idv;

  int cnt, idx = 0;

  for( cnt = 0; cnt < QPSK_CONST_POINTS; cnt++ )
  {
    /* Position of QPSK constellation point */
    idh = qpsk_center_x + ((int8_t *)data)[idx++] / 2;
    idv = qpsk_center_y - ((int8_t *)data)[idx++] / 2;

    /* Plot the QPSK constellation point */
    pix = qpsk_pixels + qpsk_rowstride * idv + qpsk_n_channels * idh;
    pix[0] = 0xff;
    pix[1] = 0xff;
    pix[2] = 0xff;
  }

  /* Draw QPSK display */
  gtk_widget_queue_draw( qpsk_drawingarea );
  
  return( FALSE );
} /* Display_QPSK_Const */

/*------------------------------------------------------------------------*/

typedef struct _ICON
{
  GtkWidget *img;
  const gchar *name;
} icon_t;

/* _Display_Icon()
 *
 * Sets an icon to be displayed in a GTK_IMAGE
 */
  static gboolean
_Display_Icon( gpointer data )
{
  /* Set the icon in the image */
  gtk_image_set_from_icon_name(
      GTK_IMAGE(((icon_t *)data)->img),
      ((icon_t *)data)->name,
      GTK_ICON_SIZE_BUTTON );

  return( FALSE );
} /* _Display_Icon() */

/* Display_Icon()
 *
 * Sets an icon to be displayed in a GTK_IMAGE
 * using the idle callback above.
 */
  void
Display_Icon( GtkWidget *img, const gchar *name )
{
  static icon_t icon;
  icon.img  = img;
  icon.name = name;
  g_idle_add( _Display_Icon, (gpointer *)&icon );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );
}/* Display_Icon() */

/*------------------------------------------------------------------------*/

/* Display_Demod_Params()
 *
 * Displays Demodulator parameters (AGC gain PLL freq etc)
 */
  gboolean
Display_Demod_Params( gpointer data )
{
  double freq, gain;
  uint32_t level;
  gchar txt[16];

  Demod_t *demod = (Demod_t *)data;

  /* Display AGC Gain and Signal Level */
  Agc_Gain( &gain );
  snprintf( txt, sizeof(txt), "%6.3f", gain );
  gtk_entry_set_text( GTK_ENTRY(agc_gain_entry), txt );
  Signal_Level( &level );
  snprintf( txt, sizeof(txt), "%6u", level );
  gtk_entry_set_text( GTK_ENTRY(sig_level_entry), txt );

  /* Display Costas PLL Frequency */
  freq = demod->costas->nco_freq * demod->sym_rate / M_2PI;
  if( (rc_data.psk_mode == DOQPSK) ||
      (rc_data.psk_mode == IDOQPSK) )
    freq *= 2.0;
  snprintf( txt, sizeof(txt), "%+8d", (int)freq );
  gtk_entry_set_text( GTK_ENTRY(pll_freq_entry), txt );

  /* Display Costas PLL Lock Detect Level */
  snprintf( txt, sizeof(txt), "%6.3f", demod->costas->moving_average );
  gtk_entry_set_text( GTK_ENTRY(pll_ave_entry), txt );

  /* Draw the level gauges */
  gtk_widget_queue_draw( sig_level_drawingarea );
  gtk_widget_queue_draw( sig_qual_drawingarea );
  gtk_widget_queue_draw( agc_gain_drawingarea );
  gtk_widget_queue_draw( pll_ave_drawingarea );

  /* Print decoder status data */
  snprintf( txt, sizeof(txt), "%d", mtd_record.sig_q );
  gtk_entry_set_text( GTK_ENTRY(sig_quality_entry), txt );
  int percent = ( 100 * ok_cnt ) / total_cnt;
  snprintf( txt, sizeof(txt), "%d:%d%%", ok_cnt, percent );
  gtk_entry_set_text( GTK_ENTRY(packet_cnt_entry), txt );

  return( FALSE );
} /* Display_Demod_Params() */

/*------------------------------------------------------------------------*/

/* Draw_Level_Gauge()
 *
 * Draws a color-coded level gauge into a drawingarea
 */
  void
Draw_Level_Gauge( GtkWidget *widget, cairo_t *cr, double level )
{
  GtkAllocation allocation;
  double dWidth, dHeight;
  int iWidth, idx;
  double red, green;

  /* Get size of drawingarea widget */
  gtk_widget_get_allocation( widget, &allocation );

  /* The -1 puts the margin rectangle in the drawingarea */
  dWidth  = (double)( allocation.width  - 1 );
  dHeight = (double)( allocation.height - 1 );

  /* Width to which gauge to be drawn, depending
   * on level. The -3 is needed to leave a margin
   * of 1 pixel around the colored bar */
  iWidth = (int)( (double)(allocation.width - 3) * level );

  /* Set line width to 1 pixel */
  cairo_set_line_width( cr, 1.0 );

  /* Draw a black outline around drawingarea */
  cairo_rectangle( cr, 0.5, 0.5, dWidth, dHeight );
  cairo_set_source_rgb( cr, 0.4, 0.4, 0.4 );
  cairo_stroke( cr );

  /* Fill the drawingarea grey, margin = 1.0 line */
  dHeight--;
  dWidth--;
  cairo_rectangle( cr, 1.0, 1.0, dWidth, dHeight );
  cairo_set_source_rgb( cr, 0.87, 0.87, 0.87 );
  cairo_fill( cr );

  /* These calculations arrange for the value of red and green
   * to be 1.0 (full) over a range of values of "level" */
  level /= TRANSITION_BAND;
  red = RED_THRESHOLD - level;
  red = dClamp( red, 0.0, 1.0 );
  green = level - GREEN_THRESHOLD;
  green = dClamp( green, 0.0, 1.0 );

  /* Draw vertical lines to width depending on level supplied */
  for( idx = 1; idx < iWidth; idx++ )
  {
    double dIdx = (double)idx;
    cairo_move_to( cr, dIdx + 1.5, 2.0 );
    cairo_line_to( cr, dIdx + 1.5, dHeight );
    cairo_set_source_rgb( cr, red, green, 0.0 );
    cairo_stroke( cr );
  }

} /* Draw_Level_Gauge() */

/*------------------------------------------------------------------------*/

typedef struct _MESSAGE
{
  const char *mesg;
  const char *attr;
} message_t;

/* Idle callback to show a message in the Text View
 */
  static gboolean
_Show_Message( gpointer data )
{
  static GtkTextIter iter;
  static Bool first_call = True;
  message_t *mesg = (message_t *)data;

  /* Initialize */
  if( first_call )
  {
    first_call = False;
    gtk_text_buffer_get_iter_at_offset( text_buffer, &iter, 0 );
  }

  /* Print message */
  gtk_text_buffer_insert_with_tags_by_name(
      text_buffer, &iter, mesg->mesg, -1, mesg->attr, NULL );
  gtk_text_buffer_insert( text_buffer, &iter, "\n", -1 );

  /* Scroll Text View to bottom */
  gtk_text_view_scroll_mark_onscreen( text_view,
      gtk_text_buffer_create_mark( text_buffer, "end", &iter, FALSE) );

  return( FALSE );
} /* End of _Show_Message() */

/*  Show_Message()
 *
 *  Prints a message string in the Text View
 *  scroller using the idle callback above
 */
  void
Show_Message( const char *mesg, const char *attr )
{
  static message_t message;
  message.mesg = mesg;
  message.attr = attr;
  g_idle_add( _Show_Message, (gpointer)&message );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );

} /* Show_Message() */

/*------------------------------------------------------------------------*/

