/*  gridloc.h
 *
 *  Header file for gridloc: A ncurses application to calculate
 *  Maidenhead grid locators give latitude/longitude and vice versa.
 *  Also to calculate bearing and distance to a second location.
 */

#ifndef GRIDLOCK_H
#define GRIDLOCK_H 1

#include <math.h>
#include <signal.h>
#include <form.h>
#include <ncurses.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>


/************ CUSTOMIZE THESE DEFINITIONS FOR YOUR NEEDS *************/

/* Definitions of some color pairs and color numbers */
#define RED			COLOR_PAIR( COLOR_RED )
#define NORMAL		COLOR_PAIR( COLOR_WHITE )
#define GREEN		COLOR_PAIR( COLOR_GREEN )
#define YELLOW		COLOR_PAIR( COLOR_YELLOW ) | A_BOLD
#define CYAN		COLOR_PAIR( COLOR_CYAN )
#define MAGENTA		COLOR_PAIR( COLOR_MAGENTA )
#define KEYSTROKE	COLOR_PAIR( COLOR_KEYSTROKE ) | A_BOLD
#define COMMAND  	COLOR_PAIR( COLOR_COMMAND )   | A_BOLD
#define ERROR   	COLOR_PAIR( COLOR_ERROR )  | A_BOLD
#define APPROX  	COLOR_PAIR( COLOR_APPROX ) | A_BOLD
#define FINISH  	COLOR_PAIR( COLOR_FINISH ) | A_BOLD

#define COLOR_KEYSTROKE  8  /* Yellow fore, blue background   */
#define COLOR_COMMAND    9  /*  White fore, blue background   */
#define COLOR_ERROR     10  /*  White fore, red background    */
#define COLOR_APPROX    11  /*  White fore, yellow background */
#define COLOR_FINISH    12  /*  White fore, green background  */

/*********************************************************************/

/* Definitions of some keycodes */
#define LF           0x0A /* Line Leed */
#define CR           0x0D /* Keycode of RETURN keyboard key */
#define KEY_ESC      0x1B /* Keycode of ESCAPE keyboard key */
#define KEY_SPACE    0x20 /* Keycode of SPACE keyboard key  */
#define KEY_TAB      0x09 /* Keycode of TAB keyboard key    */

/*-------------------------------------------------------------------*/

/* Definitions of some flags */
#define HOME_LOC_ENTERED   0X0001
#define REMT_LOC_ENTERED   0X0002
#define XPLANET_ENABLED    0X0004

#define HOME_ROW   0 /* Define which row of data fields are read */
#define REMT_ROW   9 /* by read_position() and read_locator()    */

/*-------------------------------------------------------------------*/

/** Table of constant values **/
#define deg2rad  1.74532925E-2   /* Degrees to Radians */
#define pi       3.1415926535898 /* Pi   */
#define pio2     1.5707963267949 /* Pi/2 */
#define twopi    6.2831853071796 /* 2*Pi */
#define eradius  6.378135E3      /* Earth radius in km */
#define eperim   4.0074973E4     /* Earth perimeter in km */

/*-------------------------------------------------------------------*/

/* Config file name */
#define RCFILE_HOME   ".gridlocrc"

/*-------------------------------------------------------------------*/

/* Function Prototypes produced by cproto */
/* calculate.c */
void Position_to_Gridloc(double lon, double lat, char *grid_loc);
void Gridloc_to_Position(char *grid_loc, double *lon, double *lat);
void Degrees_to_DMS(double deg, int *ideg, int *imin, int *isec);
void DMS_to_Degrees(int ideg, int imin, int isec, double *deg);
void Bearing_Distance(double lon_a, double lat_a, double lon_b, double lat_b, double *bearing, double *distance);
/* gridloc.c */
int main(void);
void Load_Config(void);
int Load_Line(char *buff, FILE *pfile);
int Display_Location(char *grid_loc, double home_lon, double home_lat, double remt_lon, double remt_lat, double remt_bng, double remt_dst);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
/* screen.c */
void Post_Gridloc_Form(void);
void Read_Position(FIELD *data_field[], double *lon, double *lat, int row);
void Read_Locator(FIELD *data_field[], double *lon, double *lat, int row);
void Draw_Hline(WINDOW *win, int top, int left, int right);
void Draw_Vline(WINDOW *win, int top, int left, int bottom);

/*-------------------------------------------------------------------*/

#endif /* End of gridloc.h */

