/*
 *  gridloc: A ncurses application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


/*  screen.c
 *
 *  New Curses display routines for gridloc
 */

#include "gridloc.h"

extern FILE
  *marker_fd,   /* Marker file for xplanet with sat/obs location */
  *greatarc_fd; /* Great circle arc co-ordinates for xplanet     */

/* Name strings for above files */
extern char
  markerfile[50],
  greatarcfile[50];

/* 'Home' position string */
extern char home_posn[23];

/*  Post_Gridloc_Form()
 *
 *  Function to display the Home and Remote position and
 *  grid locator form, and bearing/distance calculator
 */

  void
Post_Gridloc_Form( void )
{
  int
	idx,  /* Index for loops etc */
	cols, /* Columns in form     */
	rows, /* Rows in form        */
	chr,  /* Input from getch()  */
	ideg; /* Used for calculating bearings */

  double
	/* 'Home' position */
	lon_a, lat_a,
	/* Remote position */
	lon_b, lat_b,
	/* Bearing and distance to remote */
	/* (Short path and long path)     */
	bearing_spath, bearing_lpath,
	distance_spath, distance_lpath;

  /* Buffer for conversions */
  char buff[23];

  bool done; /* Form entries done flag */

  /* Form for entering Manual Position parameters */
  FORM  *gridloc_form;
  FIELD *data_field[ 19 ];

  /* Form containing windows */
  WINDOW *form_win, *form_subwin;

  /* Setup Manual Position fields and form */
  data_field[ 0 ]  = new_field( 1, 1, 2, 15, 0, 0 );
  data_field[ 1 ]  = new_field( 1, 3, 2, 17, 0, 0 );
  data_field[ 2 ]  = new_field( 1, 2, 2, 21, 0, 0 );
  data_field[ 3 ]  = new_field( 1, 2, 2, 24, 0, 0 );
  data_field[ 4 ]  = new_field( 1, 1, 2, 41, 0, 0 );
  data_field[ 5 ]  = new_field( 1, 2, 2, 43, 0, 0 );
  data_field[ 6 ]  = new_field( 1, 2, 2, 46, 0, 0 );
  data_field[ 7 ]  = new_field( 1, 2, 2, 49, 0, 0 );
  data_field[ 8 ]  = new_field( 1, 6, 2, 61, 0, 0 );

  data_field[ 9 ]  = new_field( 1, 1, 6, 15, 0, 0 );
  data_field[ 10 ] = new_field( 1, 3, 6, 17, 0, 0 );
  data_field[ 11 ] = new_field( 1, 2, 6, 21, 0, 0 );
  data_field[ 12 ] = new_field( 1, 2, 6, 24, 0, 0 );
  data_field[ 13 ] = new_field( 1, 1, 6, 41, 0, 0 );
  data_field[ 14 ] = new_field( 1, 2, 6, 43, 0, 0 );
  data_field[ 15 ] = new_field( 1, 2, 6, 46, 0, 0 );
  data_field[ 16 ] = new_field( 1, 2, 6, 49, 0, 0 );
  data_field[ 17 ] = new_field( 1, 6, 6, 61, 0, 0 );

  set_field_type( data_field[  0 ], TYPE_REGEXP, "[E|W]" );
  set_field_type( data_field[  1 ], TYPE_INTEGER, 3, 0, 180 );
  set_field_type( data_field[  2 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  3 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  4 ], TYPE_REGEXP, "[N|S]" );
  set_field_type( data_field[  5 ], TYPE_INTEGER, 2, 0, 90 );
  set_field_type( data_field[  6 ], TYPE_INTEGER, 2, 0, 59);
  set_field_type( data_field[  7 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[  8 ], TYPE_REGEXP,"[A-S][A-S][0-9][0-9][A-X][A-X]");
  set_field_type( data_field[  9 ], TYPE_REGEXP, "[E|W]" );
  set_field_type( data_field[ 10 ], TYPE_INTEGER, 3, 0, 180 );
  set_field_type( data_field[ 11 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 12 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 13 ], TYPE_REGEXP, "[N|S]" );
  set_field_type( data_field[ 14 ], TYPE_INTEGER, 2, 0, 90 );
  set_field_type( data_field[ 15 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 16 ], TYPE_INTEGER, 2, 0, 59 );
  set_field_type( data_field[ 17 ], TYPE_REGEXP,"[A-S][A-S][0-9][0-9][A-X][A-X]");

  set_field_buffer( data_field[ 0], 0, "E" );
  set_field_buffer( data_field[ 4], 0, "N" );
  set_field_buffer( data_field[ 9], 0, "E" );
  set_field_buffer( data_field[13], 0, "N" );

  for( idx = 0; idx < 18; idx++ )
	field_opts_off( data_field[ idx ], O_AUTOSKIP );

  data_field[ 18 ] = ( FIELD * ) NULL;

  /* Declare manual entries form */
  gridloc_form = new_form( data_field );
  /* Obtain size of from */
  scale_form( gridloc_form, &rows, &cols );
  /* Create form window and sub-window */
  form_win    = newwin( rows+6, cols+2, 2, 4 );
  form_subwin = newwin( rows+4, cols  , 3, 5 );
  /* Set form window and sub-window */
  set_form_win( gridloc_form, form_win );
  set_form_sub( gridloc_form, form_subwin );

  /* Box and label form window */
  wattrset(  form_win, CYAN );
  box(       form_win, ACS_VLINE, ACS_HLINE );
  mvwaddstr( form_win, 0, 5,
	  " MAIDENHEAD GRID LOCATOR AND BEARING/DISTANCE CALCULATOR " );
  mvwaddch( form_win, 2, 0, ACS_LTEE );
  mvwaddch( form_win, 2, cols+1, ACS_RTEE );
  mvwaddch( form_win, 4, 0, ACS_LTEE );
  mvwaddch( form_win, 4, cols+1, ACS_RTEE );
  mvwaddch( form_win, 6, 0, ACS_LTEE );
  mvwaddch( form_win, 6, cols+1, ACS_RTEE );
  mvwaddch( form_win, 6, 0, ACS_LTEE );
  mvwaddch( form_win, 6, cols+1, ACS_RTEE );
  mvwaddch( form_win, 8, 0, ACS_LTEE );
  mvwaddch( form_win, 8, cols+1, ACS_RTEE );
  mvwaddch( form_win,10, 0, ACS_LTEE );
  mvwaddch( form_win,10, cols+1, ACS_RTEE );

  post_form( gridloc_form );

  wattrset(  form_subwin, GREEN );
  mvwaddstr( form_subwin, 0,  1,
	  "Enter the \'Home\' position\'s Longitude/Latitude or Grid Locator" );
  mvwaddstr( form_subwin, 4,  1,
	  "Enter the Remote position\'s Longitude/Latitude or Grid Locator" );

  /* Enter various labels in the gridloc form box */
  wattrset(  form_subwin, MAGENTA  );
  mvwaddstr( form_subwin, 2,  1, "Longitude DMS" );
  mvwaddstr( form_subwin, 2, 28, "Latitude DMS" );
  mvwaddstr( form_subwin, 2, 53, "Locator" );
  mvwaddstr( form_subwin, 6,  1, "Longitude DMS" );
  mvwaddstr( form_subwin, 6, 28, "Latitude DMS" );
  mvwaddstr( form_subwin, 6, 53, "Locator" );
  mvwaddstr( form_subwin, 8,  1, "Shortpath Bearing=    Dist=" );
  mvwaddstr( form_subwin, 8, 36, "Longpath Bearing=    Dist=" );

  wattrset( form_subwin, COMMAND );
  mvwaddstr( form_subwin, 10, 0,
	  "    Locator    Position    Bearing    Defaults    Xplanet    Quit " );

  wattrset(  form_subwin, KEYSTROKE );
  mvwaddstr( form_subwin, 10, 0, " <G>" );
  mvwaddstr( form_subwin, 10, 12, "<P>" );
  mvwaddstr( form_subwin, 10, 24, "<B>" );
  mvwaddstr( form_subwin, 10, 35, "<D>" );
  mvwaddstr( form_subwin, 10, 47, "<X>" );
  mvwaddstr( form_subwin, 10, 59, "<Q>" );

  wattrset( form_subwin, CYAN );
  mvwhline( form_subwin, 1, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 3, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 5, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 7, 0, ACS_HLINE, cols );
  mvwhline( form_subwin, 9, 0, ACS_HLINE, cols );

  Draw_Vline( form_subwin, 1, 14, 3 );
  Draw_Vline( form_subwin, 1, 16, 3 );
  Draw_Vline( form_subwin, 1, 20, 3 );
  Draw_Vline( form_subwin, 1, 23, 3 );
  Draw_Vline( form_subwin, 1, 26, 3 );
  Draw_Vline( form_subwin, 1, 40, 3 );
  Draw_Vline( form_subwin, 1, 42, 3 );
  Draw_Vline( form_subwin, 1, 45, 3 );
  Draw_Vline( form_subwin, 1, 48, 3 );
  Draw_Vline( form_subwin, 1, 51, 3 );
  Draw_Vline( form_subwin, 1, 60, 3 );
  Draw_Vline( form_subwin, 5, 14, 7 );
  Draw_Vline( form_subwin, 5, 16, 7 );
  Draw_Vline( form_subwin, 5, 20, 7 );
  Draw_Vline( form_subwin, 5, 23, 7 );
  Draw_Vline( form_subwin, 5, 26, 7 );
  Draw_Vline( form_subwin, 5, 40, 7 );
  Draw_Vline( form_subwin, 5, 42, 7 );
  Draw_Vline( form_subwin, 5, 45, 7 );
  Draw_Vline( form_subwin, 5, 48, 7 );
  Draw_Vline( form_subwin, 5, 51, 7 );
  Draw_Vline( form_subwin, 5, 60, 7 );
  Draw_Vline( form_subwin, 7, 33, 9 );

  /* Make keyboard blocking, enable cursor */
  nodelay( stdscr, FALSE );
  curs_set( TRUE );

  /* Prepare for the form's data input loop */
  form_driver( gridloc_form, REQ_FIRST_FIELD );
  set_field_back( data_field[ 0 ], KEYSTROKE );
  done = FALSE;

  /* Display form */
  refresh();
  wnoutrefresh( form_win );
  wnoutrefresh( form_subwin );
  doupdate();

  /* Process keystrokes from user */
  while( !done )
  {
	chr = getch();
	switch( chr )
	{
	  case CR : /* Toggle East/West or North/South flags */
		{
		  switch( field_index(current_field(gridloc_form)) )
		  {
			case 0 :
			  ClearFlag( HOME_LOC_ENTERED );
			  if( strcmp(field_buffer(data_field[0], 0), "E") )
				set_field_buffer( data_field[0], 0, "E" );
			  else
				set_field_buffer( data_field[0], 0, "W" );
			  break;

			case 4 :
			  ClearFlag( HOME_LOC_ENTERED );
			  if( strcmp(field_buffer(data_field[4], 0), "N") )
				set_field_buffer( data_field[4], 0, "N" );
			  else
				set_field_buffer( data_field[4], 0, "S" );
			  break;

			case 9 :
			  ClearFlag( REMT_LOC_ENTERED );
			  if( strcmp(field_buffer(data_field[9], 0), "E") )
				set_field_buffer( data_field[9], 0, "E" );
			  else
				set_field_buffer( data_field[9], 0, "W" );
			  break;

			case 13 :
			  ClearFlag( REMT_LOC_ENTERED );
			  if( strcmp(field_buffer(data_field[13], 0), "N") )
				set_field_buffer( data_field[13], 0, "N" );
			  else
				set_field_buffer( data_field[13], 0, "S" );
			  break;

			case 8:
			  /* Disable cursor and field highlighting */
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  curs_set( FALSE );
			  form_driver( gridloc_form, REQ_NEXT_FIELD );
			  form_driver( gridloc_form, REQ_BEG_FIELD );
			  set_field_back( current_field(gridloc_form), KEYSTROKE );
			  Read_Locator( data_field, &lon_a, &lat_a, HOME_ROW );
			  break;

			case 17:
			  /* Disable cursor and field highlighting */
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  curs_set( FALSE );
			  form_driver( gridloc_form, REQ_NEXT_FIELD );
			  form_driver( gridloc_form, REQ_BEG_FIELD );
			  set_field_back( current_field(gridloc_form), KEYSTROKE );
			  Read_Locator( data_field, &lon_a, &lat_a, REMT_ROW );
			  break;

			default:
			  curs_set( TRUE );
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  form_driver( gridloc_form, REQ_NEXT_FIELD );
			  form_driver( gridloc_form, REQ_BEG_FIELD );
			  set_field_back( current_field(gridloc_form), KEYSTROKE );

		  } /* switch( field_index(current_field(gridloc_form)) ) */

		} /* case CR : */

		break;

	  case KEY_LEFT  :
		curs_set( TRUE );
		if( (field_index(current_field(gridloc_form)) == 0) ||
			(field_index(current_field(gridloc_form)) == 9) )
		{
		  set_field_back(current_field(gridloc_form), A_NORMAL);
		  form_driver( gridloc_form, REQ_LEFT_FIELD );
		  form_driver( gridloc_form, REQ_BEG_FIELD );
		  set_field_back( current_field(gridloc_form), KEYSTROKE );
		}
		form_driver( gridloc_form, REQ_PREV_CHAR );
		break;

	  case KEY_RIGHT :
		curs_set( TRUE );
		form_driver( gridloc_form, REQ_NEXT_CHAR );
		break;

	  case KEY_TAB :
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_RIGHT_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_DOWN :
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_DOWN_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_UP :
		curs_set( TRUE );
		set_field_back(current_field(gridloc_form), A_NORMAL);
		form_driver( gridloc_form, REQ_UP_FIELD );
		set_field_back( current_field(gridloc_form), KEYSTROKE );
		break;

	  case KEY_BACKSPACE :
		curs_set( TRUE );
		form_driver( gridloc_form, REQ_PREV_CHAR );
		form_driver( gridloc_form, REQ_DEL_CHAR );
		break;

	  case KEY_DC :
		curs_set( TRUE );
		form_driver( gridloc_form, REQ_DEL_CHAR );
		break;

	  case 'g' :
		/* Calculate Home locator from position */
		/* Ignore commands in Grid Loc fields */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back( current_field(gridloc_form), A_NORMAL );
		  curs_set( FALSE );

		  /* Read Home position and convert to locator */
		  Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );
		  SetFlag( HOME_LOC_ENTERED );
		}
		break;

	  case 'p' :
		/* Calculate Home position from locator */
		/* Ignore commands in Grid Loc fields */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Do not process an empty locator field */
		  if( field_index(current_field(gridloc_form)) < 9 )
			if( isFlagSet(HOME_LOC_ENTERED) )
			{
			  /* Disable cursor and field highlighting */
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  curs_set( FALSE );
			  Read_Locator( data_field, &lon_a, &lat_a, HOME_ROW );
			}
			else
			{
			  set_field_back( data_field[8], YELLOW);
			  set_field_buffer( data_field[8], 0, "??????" );
			} /* if( isFlagSet(HOME_LOC_ENTERED) ) */
		  else
			if( isFlagSet(REMT_LOC_ENTERED) )
			{
			  /* Disable cursor and field highlighting */
			  set_field_back(current_field(gridloc_form), A_NORMAL);
			  curs_set( FALSE );
			  Read_Locator( data_field, &lon_a, &lat_a, REMT_ROW );
			}
			else
			{
			  set_field_back( data_field[17], YELLOW);
			  set_field_buffer( data_field[17], 0, "??????" );
			} /* if( isFlagSet(REMT_LOC_ENTERED) ) */

		} /* if( (field_index(current_field(gridloc_form)) ==  8) || */
		break;

	  case 'b' :
		/* Remote locator and bearing/distance */
		/* Ignore commands in Grid Loc fields */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back(current_field(gridloc_form), A_NORMAL);
		  curs_set( FALSE );

		  if( isFlagSet(HOME_LOC_ENTERED) )
			/* Calculte Home position from grid locator */
			Read_Locator( data_field, &lon_a, &lat_a, HOME_ROW );
		  else
			/* Calculte Home position and grid locator */
			Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );

		  if( isFlagSet(REMT_LOC_ENTERED) )
			/* Calculte Remote position from grid locator */
			Read_Locator( data_field, &lon_b, &lat_b, REMT_ROW );
		  else
			/* Calculte Remote position and grid locator */
			Read_Position( data_field, &lon_b, &lat_b, REMT_ROW );

		  /* Calculate bearing and distance to Remote location */
		  Bearing_Distance( lon_a, lat_a, lon_b, lat_b,
			  &bearing_spath, &distance_spath );
		  bearing_lpath  = bearing_spath + 180.0;
		  distance_lpath = eperim - distance_spath;

		  /* Display bearing and distance to remote point */
		  wattrset( form_subwin, NORMAL );
		  ideg = (int)(bearing_spath+0.5);
		  if( ideg >= 360 )
			ideg -= 360;
		  mvwprintw( form_subwin, 8, 19, "%03d", ideg );
		  mvwprintw( form_subwin, 8, 28, "%5d", (int)(distance_spath+0.5) );

		  /* Display home/remote location using xpalnet if enabled */
		  if( isFlagSet(XPLANET_ENABLED) )
			Display_Location( field_buffer(data_field[8+REMT_ROW],0),
				lon_a, lat_a, lon_b, lat_b,
				ideg, distance_spath+0.5 );

		  ideg = (int)(bearing_lpath+0.5);
		  if( ideg >= 360 )
			ideg -= 360;
		  mvwprintw( form_subwin, 8, 53, "%03d", ideg );
		  mvwprintw( form_subwin, 8, 62, "%5d", (int)(distance_lpath+0.5) );
		}
		break;

	  case 'd' :
		/* Enter default Home position */
		/* Ignore commands in Grid Loc fields */
		if( (field_index(current_field(gridloc_form)) ==  8) ||
			(field_index(current_field(gridloc_form)) == 17) )
		  form_driver( gridloc_form, chr );
		else
		{
		  /* Disable cursor and field highlighting */
		  set_field_back( data_field[8], A_NORMAL );
		  set_field_back(current_field(gridloc_form), A_NORMAL);
		  curs_set( FALSE );

		  strcpy( buff, home_posn );
		  /* Set terminators in between deg, min, sec */
		  /* of home long and lat. Crude but it works */
		  buff[1]  = '\0';
		  buff[5]  = '\0';
		  buff[8]  = '\0';
		  buff[11] = '\0';
		  buff[13] = '\0';
		  buff[16] = '\0';
		  buff[19] = '\0';
		  buff[22] = '\0';

		  /* Display default home position */
		  set_field_buffer( data_field[ 0 ], 0, &buff[0] );
		  set_field_buffer( data_field[ 1 ], 0, &buff[2] );
		  set_field_buffer( data_field[ 2 ], 0, &buff[6] );
		  set_field_buffer( data_field[ 3 ], 0, &buff[9] );
		  set_field_buffer( data_field[ 4 ], 0, &buff[12] );
		  set_field_buffer( data_field[ 5 ], 0, &buff[14] );
		  set_field_buffer( data_field[ 6 ], 0, &buff[17] );
		  set_field_buffer( data_field[ 7 ], 0, &buff[20] );

		  /* Read Home position and convert to locator */
		  Read_Position( data_field, &lon_a, &lat_a, HOME_ROW );
		  SetFlag( HOME_LOC_ENTERED );
		}
		break;

	  case 'x':
		if( isFlagClear(XPLANET_ENABLED) )
		  SetFlag( XPLANET_ENABLED );
		else
		  ClearFlag( XPLANET_ENABLED );
		break;

	  case 'q' : case KEY_ESC :
		done = TRUE;
		break;

	  default :
		/* Enter character and set 'data entered' flags */
		form_driver( gridloc_form, chr );

		idx = field_index(current_field(gridloc_form));

		/* Set Home Locator Entered flag */
		/* if keystroke in Locator field */
		if( idx == 8 )
		  SetFlag( HOME_LOC_ENTERED );

		/* Clear Home Locator Entered flag  */
		/* if keystroke in a position field */
		if( idx >= 0 && idx <= 7 )
		  ClearFlag( HOME_LOC_ENTERED );

		/* Set Remote Locator Entered flag */
		/* if keystroke in Locator field   */
		if( idx == 17 )
		  SetFlag( REMT_LOC_ENTERED );

		/* Clear Remote Locator Entered flag */
		/* if keystroke in a position field  */
		if( idx >= 9 && idx <= 16 )
		  ClearFlag( REMT_LOC_ENTERED );

	} /* End of switch( chr )  */

	/* Refresh gridloc form */
	wnoutrefresh( form_subwin );
	doupdate();

  } /* End of while( !done ) */

  /* Unpost form and free fields */
  unpost_form( gridloc_form );
  free_form( gridloc_form );
  for ( idx=0; idx < 19; idx++ ) free_field( data_field[ idx ] );

  /* Delete form windows */
  delwin( form_subwin );
  delwin( form_win );

} /* End of Post_Gridloc_Form() */

/*------------------------------------------------------------------------*/

/*  Read_Position()
 *
 *  Reads position fields (Lon/Lat) and calculates Lon/Lat
 */

  void
Read_Position( FIELD *data_field[], double *lon, double *lat, int row )
{
  int ideg, imin, isec;

  char locator[7];

  /* Calculte Home position and grid locator */
  ideg = atoi( field_buffer(data_field[1+row], 0) );
  imin = atoi( field_buffer(data_field[2+row], 0) );
  isec = atoi( field_buffer(data_field[3+row], 0) );
  DMS_to_Degrees( ideg, imin, isec, lon );
  if( strcmp(field_buffer(data_field[0+row],0), "W")==0 )
	*lon = - *lon;

  ideg = atoi( field_buffer(data_field[5+row], 0) );
  imin = atoi( field_buffer(data_field[6+row], 0) );
  isec = atoi( field_buffer(data_field[7+row], 0) );
  DMS_to_Degrees( ideg, imin, isec, lat );
  if( strcmp(field_buffer(data_field[4+row],0), "S")==0 )
	*lat = - *lat;

  /* Calculate grid locator */
  Position_to_Gridloc( *lon, *lat, locator );
  set_field_buffer( data_field[8+row], 0, locator );

} /* End of void Read_Position() */

/*------------------------------------------------------------------------*/

/*  Read_Locator()
 *
 *  Reads a Locator field and calculates Lon/Lat
 */

  void
Read_Locator( FIELD *data_field[], double *lon, double *lat, int row )
{
  /* Used for converting degrees to ddmmss */
  int ideg, imin, isec;

  /* Temporary for displaying */
  double lng, ltd;

  char buff[4];

  /* Read grid locator and calculate position */
  Gridloc_to_Position( field_buffer(data_field[8+row], 0), lon, lat );

  /* Negative longitude shown as West */
  if( *lon < 0.0 )
  {
	lng = - *lon;
	set_field_buffer( data_field[0+row], 0, "W" );
  }
  else
  {
	lng = *lon;
	set_field_buffer( data_field[0+row], 0, "E" );
  }

  Degrees_to_DMS( lng, &ideg, &imin, &isec );
  snprintf( buff, 4, "%03d", ideg );
  set_field_buffer( data_field[1+row], 0, buff );
  snprintf( buff, 3, "%02d", imin );
  set_field_buffer( data_field[2+row], 0, buff );
  snprintf( buff, 3, "%02d", isec );
  set_field_buffer( data_field[3+row], 0, buff );

  /* Negative latitude shown as South */
  if( *lat < 0.0 )
  {
	ltd = - *lat;
	set_field_buffer( data_field[4+row], 0, "S" );
  }
  else
  {
	ltd = *lat;
	set_field_buffer( data_field[4+row], 0, "N" );
  }

  Degrees_to_DMS( ltd, &ideg, &imin, &isec );
  snprintf( buff, 3, "%02d", ideg );
  set_field_buffer( data_field[5+row], 0, buff );
  snprintf( buff, 3, "%02d", imin );
  set_field_buffer( data_field[6+row], 0, buff );
  snprintf( buff, 3, "%02d", isec );
  set_field_buffer( data_field[7+row], 0, buff );

} /* End of Read_Locator() */

/*------------------------------------------------------------------------*/

/*  Draw_Hline()
 *
 *  Draws a horizontal line in a window including Tees at ends
 */

  void
Draw_Hline( WINDOW *win, int top, int left, int right )
{
  mvwhline( win, top, left+1, ACS_HLINE, (right - left - 1) );
  mvwaddch( win, top, left,  ACS_LTEE );
  mvwaddch( win, top, right, ACS_RTEE );
}

/*------------------------------------------------------------------------*/

/*  Draw_Vline()
 *
 *  Draws a horizontal line in a window including Tees at ends
 */

  void
Draw_Vline( WINDOW *win, int top, int left, int bottom )
{
  mvwvline( win, top+1, left, ACS_VLINE, (bottom - top -1) );
  mvwaddch( win, top, left, ACS_TTEE );
  mvwaddch( win, bottom, left, ACS_BTEE );
}

/*------------------------------------------------------------------------*/
