/*
 *  gridloc: A ncurses application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position.
 *
 *  Copyright (C) 2001  Neoklis Kyriazis
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


/*  main.c
 *
 *  Main body of gridloc application
 */

#include "gridloc.h"

FILE
  *marker_fd,   /* Marker file for xplanet with sat/obs location */
  *greatarc_fd; /* Great circle arc co-ordinates for xplanet     */

char
  home_posn[23],  /* 'Home' position string */
  home_name[25];  /* 'Home' position name   */

/* Name strings for above files */
char
  markerfile[64],
  greatarcfile[64];

/* Signal handler */
static void Signal_Handler( int signal );

  int
main( void )
{
  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Initialize new actions */
  sa_new.sa_handler = Signal_Handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Read rc file */
  Load_Config();

  /* Initialize ncurses */
  initscr();
  keypad( stdscr, TRUE );
  nonl();
  cbreak();
  noecho();

  /* Assuming terminal has color! */
  if( has_colors() )
  {
	start_color();

	init_pair( COLOR_WHITE,     COLOR_WHITE,  COLOR_BLACK  );
	init_pair( COLOR_RED,	COLOR_RED,    COLOR_BLACK  );
	init_pair( COLOR_GREEN,	COLOR_GREEN,  COLOR_BLACK  );
	init_pair( COLOR_BLUE,	COLOR_BLUE,   COLOR_BLACK  );
	init_pair( COLOR_YELLOW,	COLOR_YELLOW, COLOR_BLACK  );
	init_pair( COLOR_CYAN,	COLOR_CYAN,   COLOR_BLACK  );
	init_pair( COLOR_MAGENTA,	COLOR_MAGENTA,COLOR_BLACK  );
	init_pair( COLOR_KEYSTROKE, COLOR_YELLOW, COLOR_BLUE   );
	init_pair( COLOR_COMMAND,   COLOR_WHITE,  COLOR_BLUE   );
	init_pair( COLOR_ERROR,     COLOR_WHITE,  COLOR_RED    );
	init_pair( COLOR_APPROX,    COLOR_WHITE,  COLOR_YELLOW );
	init_pair( COLOR_FINISH,    COLOR_WHITE,  COLOR_GREEN  );
  }

  /* Go to the ncurses gui */
  Post_Gridloc_Form();

  Signal_Handler( 0 );

  return(0);

} /* End of 'main()' */

/*------------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the gridloc configuration file
 */

  void
Load_Config( void )
{
  char
	*rc_fpath, /* File path to gridlocrc */
	line[81];  /* Buffer for Load_Line   */

  /* Config file pointer */
  FILE *gridlocrc;


  /* Setup file path to gridlocrc */
  rc_fpath = malloc( strlen(getenv("HOME")) + strlen(RCFILE_HOME) + 2 );
  if( rc_fpath == NULL )
	fprintf( stderr, "Malloc for rc file name pointer failed - aborting\n" );
  sprintf( rc_fpath, "%s/%s", getenv("HOME"), RCFILE_HOME );

  /* Open ~./gridlocrc file */
  gridlocrc = fopen( rc_fpath, "r" );
  if( gridlocrc == NULL )
  {
	fprintf( stderr, "gridloc: Failed to open ~/.gridlocrc - aborting\n" );
	exit( -1 );
  }
  free(rc_fpath);

  /*** Read runtime configuration data ***/

  /* Read 'Home' position, abort on EOF */
  if( Load_Line(line, gridlocrc) == EOF )
  {
	fprintf( stderr, "Error reading config file: ~/.gridlocrc - aborting\n" );
	exit(-1);
  }
  strncpy( home_posn, line, 22 );
  home_posn[22] = '\0';

  /* Read 'Home' position name, abort if EOF */
  if( (Load_Line(line, gridlocrc) == EOF) &&
	  (strlen(line) == 0) )
  {
	fprintf( stderr, "Error reading config file: ~/.gridlocrc - aborting\n" );
	exit(-1);
  }
  strncpy( home_name, line, 24 );
  home_name[24] = '\0';

  fclose( gridlocrc );

} /* End of Load_Config */

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc */

  num_chr = 0;

  /* Clear buffer at start */
  buff[0] = '\0';

  /* Ignore commented lines, white spaces and eol/cr */
  if( (chr = fgetc(pfile)) == EOF )
	return( EOF );

  while(
	  (chr=='#') ||
	  (chr==' ') ||
	  (chr==CR ) ||
	  (chr==LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) &&
		(chr != LF) )
	  if( (chr = fgetc(pfile)) == EOF )
		return( EOF );

	/* Dump any CR/LF remaining */
	while( (chr == CR) ||
		(chr == LF) )
	  if( (chr = fgetc(pfile)) == EOF )
		return( EOF );

  } /* End of while( (chr == '#') || ... */

  while( num_chr < 80 )
  {
	/* If LF/CR reached before filling buffer, return */
	if( (chr == LF) ||
		(chr == CR) )
	  break;

	buff[num_chr++] = chr;

	/* Terminate line buff on EOF */
	if( (chr = fgetc(pfile)) == EOF )
	{
	  buff[num_chr] = '\0';
	  return( EOF );
	}

	/* Abort if end of line not reached at 80 char. */
	if( (num_chr == 80) &&
		(chr != LF)     &&
		(chr != CR) )
	  fprintf( stderr, "Error reading config file: ~/gridlocrc" );

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( 0 );

} /* End of Load_Line() */

/*------------------------------------------------------------------*/

/*  Display_Location
 *
 *  Displays Home and Remote locations
 *  and Great Circle arc using xplanet
 */

int
Display_Location(
	char *grid_loc,
	double home_lon, double home_lat,
	double remt_lon, double remt_lat,
	double remt_bng, double remt_dst )
{
  int error;  /* Error flag for system calls */

  char command[166]; /* xplanet command string */

  error = 0;

  /* Prepare xplanet's file names */
  snprintf( markerfile,   64, "%s/.markerfile",   getenv("HOME") );
  snprintf( greatarcfile, 64, "%s/.greatarcfile", getenv("HOME") );

  /* Open xplanet file descriptors */
  marker_fd   = fopen( markerfile,   "w" );
  greatarc_fd = fopen( greatarcfile, "w" );

  /* Enter "Home" position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s %03d/%02d\" %s\n",
	  home_lat, home_lon, home_name,
	  (int)home_lon, (int)home_lat, "color=red" );

  /* Enter "Remote" position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s %03d/%d\" %s\n",
	  remt_lat, remt_lon, grid_loc,
	  (int)remt_bng, (int)remt_dst, "color=yellow" );

  /* Enter Home/Remote positions in greatarc file */
  fprintf( greatarc_fd, "%8.3f %8.3f %8.3f %8.3f %s\n",
	  home_lat, home_lon, remt_lat, remt_lon, "color=yellow" );

  fclose( greatarc_fd );
  fclose( marker_fd );

  /* Prepare command string for xplanet */
  snprintf( command, 166,
	  "xplanet -markerfile %s -greatarcfile %s",
	  markerfile, greatarcfile );

  /* Invoke xplanet */
  error = system( command );

  /* On error delete xplanet's files */
  if( error )
  {
	unlink( greatarcfile );
	unlink( markerfile );
  }

  return( error );

} /* End of Display_Location() */

/*------------------------------------------------------------------------*/

/*  Sig_Handler()
 *
 *  Signal Action Handler function
 */

static void Signal_Handler( int signal )
{
  endwin();

  /* Delete xplanet files */
  unlink( markerfile );
  unlink( greatarcfile );

  fprintf( stderr, "\n" );
  switch( signal )
  {
	case 0       : exit(0);
	case SIGINT  : fprintf( stderr,  "gridloc: Exiting via User Interrupt" );
				   break;
	case SIGSEGV : fprintf( stderr,  "gridloc: Segmentation Fault" );
				   break;
	case SIGFPE  : fprintf( stderr,  "gridloc: Floating Point Exception" );
				   break;
	case SIGABRT : fprintf( stderr,  "gridloc: Abort Signal received" );
				   break;
	case SIGTERM : fprintf( stderr,  "gridloc: Termination Request received" );
  }

  exit( -1 );

} /* End of Sig_Handler() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flags */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet(int flag)
{
  return (Flags & flag);
}

  int
isFlagClear(int flag)
{
  return (~Flags & flag);
}

  void
SetFlag(int flag)
{
  Flags |= flag;
}

  void
ClearFlag(int flag)
{
  Flags &= ~flag;
}

  void
ToggleFlag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

