/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "callback_func.h"
#include "draw.h"
#include "draw_radiation.h"
#include "draw_structure.h"
#include "editors.h"
#include "geom_edit.h"
#include "gnuplot.h"
#include "interface.h"
#include "main.h"
#include "optimize.h"
#include "nec2_model.h"
#include "rc_config.h"
#include "shared.h"
#include "utils.h"
#include "xnec2c.h"

#include <ctype.h>
#include <gtk/gtk.h>

/* Gain colorcode strip size */
#define COLORCODE_WIDTH     96
#define COLORCODE_HEIGHT    24
#define COLORCODE_MAX       768.0  /* Max value, 8 x COLORCODE_WIDTH */

typedef struct _SAVE_DATA
{
  GtkWidget *drawingarea;
  int width, height;
  char filename[LINE_LEN];
} save_data_t;

/*-----------------------------------------------------------------------*/

/* Save_Pixbuf()
 *
 * Saves pixbufs as png files
 */
  gboolean
Save_Pixbuf( gpointer save_data )
{
  GdkPixbuf *pixbuf;
  GError *error = NULL;
  GdkWindow *window =
    gtk_widget_get_window( ((save_data_t *)save_data)->drawingarea );

  /* Get image from pixbuf */
  pixbuf = gdk_pixbuf_get_from_window( window, 0, 0,
      ((save_data_t *)save_data)->width,
      ((save_data_t *)save_data)->height );

  /* Save image as PNG file */
  gdk_pixbuf_save( pixbuf,
      ((save_data_t *)save_data)->filename, "png", &error, NULL );
  g_object_unref( pixbuf );

  return( FALSE );

} /* Save_Pixbuf() */

/*-----------------------------------------------------------------------*/

/* Set_Spin_Button()
 *
 * Sets the value of a spin button
 */
  static void
Set_Spin_Button( GtkSpinButton *spin, gdouble value )
{
  /* Save original value and set new */
  gdouble sav = gtk_spin_button_get_value( spin );
  gtk_spin_button_set_value( spin, value );

  /* Issue a value_changed signal if needed (given same values) */
  if( sav == value )
    g_signal_emit_by_name( G_OBJECT(spin), "value_changed", NULL );

} /* Set_Spin_Button() */

/*-----------------------------------------------------------------------*/

  void
Main_Window_Delete( void )
{
  kill_window = main_window;
  SetFlag( MAIN_QUIT );

  /* Prompt user to save NEC2 data */
  if( Nec2_Edit_Save() ) return;

  /* Save GUI state for restoring windows */
  Get_GUI_State();
  Save_Config();

  /* Quit without confirmation dialog */
  if( !rc_config.confirm_quit )
  {
    Gtk_Widget_Destroy( &main_window );
    return;
  }

  Delete_Event( _("Really quit xnec2c?") );
} // Main_Window_Delete()

/*------------------------------------------------------------------*/

  void
New_Activate( void )
{
  /* No save/open file while freq loop is running */
  if( !Nec2_Save_Warn(
        _("A new NEC2 input file may not be created\n"
          "while the Frequency Loop is running") ) )
    return;

  SetFlag( OPEN_NEW_NEC2 );

  /* Reset on opening new file */
  calc_data.FR_cards    = 0;
  calc_data.FR_index    = 0;
  calc_data.steps_total = 0;
  calc_data.last_step   = 0;
  if( isFlagClear(OPTIMIZER_OUTPUT) )
  {
    calc_data.fmhz_save = 0.0;
    ClearFlag( PLOT_FREQ_LINE );
  }

  /* Prompt user to save NEC2 data */
  if( Nec2_Edit_Save() ) return;

  /* Open editor window if needed */
  if( nec2_edit_window == NULL )
  {
    Close_File( &input_fp );
    Open_Nec2_Editor( NEC2_EDITOR_NEW );
  }
  else Nec2_Input_File_Treeview( NEC2_EDITOR_NEW );

  rc_config.input_file[0] = '\0';
  selected_treeview = cmnt_treeview;
} // New_Activate()

/*------------------------------------------------------------------*/

  void
Open_Input_Activate( void )
{
  /* No save/open file while freq loop is running */
  if( !Nec2_Save_Warn(
        _("A new NEC2 input file may not be opened\n"
          "while the Frequency Loop is running")) )
    return;

  SetFlag( OPEN_INPUT );

  /* Reset on opening new file */
  calc_data.FR_cards    = 0;
  calc_data.FR_index    = 0;
  calc_data.steps_total = 0;
  calc_data.last_step   = 0;
  if( isFlagClear(OPTIMIZER_OUTPUT) )
  {
    calc_data.fmhz_save = 0.0;
    ClearFlag( PLOT_FREQ_LINE );
  }

  /* Prompt user to save NEC2 data */
  if( Nec2_Edit_Save() )
  {
    SetFlag( NEC2_SAVE );
    return;
  }

  /* Open file chooser to select a NEC2 input file */
  file_chooser = Open_Filechooser(
      GTK_FILE_CHOOSER_ACTION_OPEN,
      "*.nec", NULL, NULL, rc_config.working_dir );
} // Open_Input_Activate()

/*------------------------------------------------------------------*/

  void
Main_Save_Activate( int *saveas_width, int *saveas_height )
{
  char saveas[FILENAME_LEN + 25];
  size_t s = sizeof( saveas );

  /* Count number of structure image files saved of geometry,
   * currents or charges, to avoid over-writing saved files */
  static int cgm = 0, ccr = 0, cch = 0;

  if( strlen(rc_config.input_file) == 0 ) return;

  /* Make the structure image save file name from input file
   * name. The count of each image type saved is incremented */
  if( isFlagSet(DRAW_CURRENTS) )
    snprintf( saveas, s, "%s-%s_%03d.%s",
        rc_config.input_file, "current", ++ccr, "png" );
  else if( isFlagSet(DRAW_CHARGES) )
    snprintf( saveas, s, "%s-%s_%03d.%s",
        rc_config.input_file, "charge", ++cch, "png" );
  else
    snprintf( saveas, s, "%s-%s_%03d.%s",
        rc_config.input_file, "geometry", ++cgm, "png" );

  saveas_drawingarea = structure_drawingarea;
  *saveas_width      = structure_width;
  *saveas_height     = structure_height;

  /* Open file chooser to save structure image */
  SetFlag( IMAGE_SAVE );
  file_chooser = Open_Filechooser( GTK_FILE_CHOOSER_ACTION_SAVE,
      "*.png", NULL, saveas, rc_config.working_dir );
} // Main_Save_Activate()

/*------------------------------------------------------------------*/

  void
Optimizer_Output_Toggled( GtkMenuItem *menuitem )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    // Enable frequency data output to Optimizer's file
    SetFlag( OPTIMIZER_OUTPUT );

    // Create a thread to play back demodulation buffer
    pthread_t thrd;
    int ret = pthread_create( &thrd, NULL, Optimizer_Output, NULL );
    if( ret != 0 )
    {
      fprintf( stderr, "xnec2c: failed to create Optimizer Output thread\n" );
      perror( "xnec2c: pthread_create()" );
      exit( -1 );
    }
  }
  else
    ClearFlag( OPTIMIZER_OUTPUT );
} // Optimizer_Output_Toggled()

/*------------------------------------------------------------------*/

// Handles the on_quit_activate() callback
  void
Quit_Activate( void )
{
  kill_window = main_window;
  SetFlag( MAIN_QUIT );

  /* Prompt user to save NEC2 data */
  if( Nec2_Edit_Save() ) return;

  /* Save GUI state for restoring windows */
  Get_GUI_State();
  Save_Config();

  /* Quit without confirmation dialog */
  if( !rc_config.confirm_quit )
  {
    Gtk_Widget_Destroy( &main_window );
    return;
  }

  Delete_Event( _("Really quit xnec2c?") );
} // Quit_Activate()

/*------------------------------------------------------------------*/

// Handles the on_main_rdpattern_activate() callback
    void
On_Main_Rdpattern_Activate( GtkMenuItem *menuitem )
{
  /* Open radiation pattern rendering window */
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    GtkAllocation alloc;
    GtkWidget *widget;

    rdpattern_window = create_rdpattern_window( &rdpattern_window_builder );
    Set_Window_Geometry( rdpattern_window,
        rc_config.rdpattern_x, rc_config.rdpattern_y,
        rc_config.rdpattern_width, rc_config.rdpattern_height );
    gtk_widget_show( rdpattern_window );

    rdpattern_drawingarea = Builder_Get_Object(
        rdpattern_window_builder, "rdpattern_drawingarea" );
    gtk_widget_get_allocation( rdpattern_drawingarea, &alloc );
    rdpattern_width  = alloc.width;
    rdpattern_height = alloc.height;

    New_Projection_Parameters(
        rdpattern_width,
        rdpattern_height,
        &rdpattern_proj_params );

    rotate_rdpattern  = GTK_SPIN_BUTTON( Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_rotate_spinbutton") );
    incline_rdpattern = GTK_SPIN_BUTTON(Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_incline_spinbutton") );
    rdpattern_frequency = GTK_SPIN_BUTTON(Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_freq_spinbutton") );
    rdpattern_zoom = GTK_SPIN_BUTTON(Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_zoom_spinbutton") );
    rdpattern_fstep_entry = GTK_ENTRY(Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_fstep_entry") ) ;

    /* Restore radiation pattern window widgets state */
    if( rc_config.rdpattern_gain_togglebutton )
    {
      widget = Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_gain_togglebutton" );
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      SetFlag(DRAW_GAIN);
    }

    if( rc_config.rdpattern_eh_togglebutton )
    {
      widget = Builder_Get_Object(
          rdpattern_window_builder, "rdpattern_eh_togglebutton" );
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      SetFlag(DRAW_EHFIELD);
    }

    widget = Builder_Get_Object(
        rdpattern_window_builder, "rdpattern_e_field" );
    if( rc_config.rdpattern_e_field )
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), TRUE );
      SetFlag( DRAW_EFIELD );
    }
    else
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), FALSE );
      ClearFlag( DRAW_EFIELD );
    }

    widget = Builder_Get_Object(
        rdpattern_window_builder, "rdpattern_h_field" );
    if( rc_config.rdpattern_h_field )
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), TRUE );
      SetFlag( DRAW_HFIELD );
    }
    else
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), FALSE );
      ClearFlag( DRAW_HFIELD );
    }

    widget = Builder_Get_Object(
        rdpattern_window_builder, "rdpattern_poynting_vector" );
    if( rc_config.rdpattern_poynting_vector )
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), TRUE );
      SetFlag( DRAW_POYNTING );
    }
    else
    {
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), FALSE );
      ClearFlag( DRAW_POYNTING );
    }

    if( isFlagClear(INPUT_OPENED) )
    {
      GtkWidget *box =
        Builder_Get_Object( rdpattern_window_builder, "rdpattern_box" );
      gtk_widget_hide( box );
    }

    Main_Rdpattern_Activate( TRUE );
  } /* if( gtk_check_menu_item_get_active(...) ) */
  else if( isFlagSet(DRAW_ENABLED) )
    Gtk_Widget_Destroy( &rdpattern_window );

} // On_Main_Rdpattern_Activate()

/*------------------------------------------------------------------*/

// Handles the on_main_plots_activate() callback
  void
On_Main_Plots_Activate( GtkMenuItem *menuitem )
{
  /* Open window for plotting frequency
   * related data (gain, vswr etc) */
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    if( Main_Plots_Activate() )
    {
      GtkWidget *widget;

      plots_window = create_plots_window( &plots_window_builder );
      Set_Window_Geometry( plots_window,
          rc_config.plots_x, rc_config.plots_y,
          rc_config.plots_width, rc_config.plots_height );
      gtk_widget_show( plots_window );
      plots_drawingarea = Builder_Get_Object(
          plots_window_builder, "plots_drawingarea" );
      Set_Window_Labels();
      calc_data.ngraph = 0;

      /* Set the Zo spinbutton value */
      GtkWidget *spin = Builder_Get_Object(
          plots_window_builder, "plots_zo_spinbutton" );
      gtk_spin_button_set_value( GTK_SPIN_BUTTON(spin), (gdouble)calc_data.zo );

      GtkAllocation alloc;
      gtk_widget_get_allocation( plots_drawingarea, &alloc );
      plots_width  = alloc.width;
      plots_height = alloc.height;

      /* Restore frequency plots window widgets state */
      if( rc_config.plots_gmax_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_gmax_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_gdir_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_gdir_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_gviewer_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_gviewer_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_vswr_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_vswr_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_zrlzim_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_zrlzim_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_zmgzph_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_zmgzph_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_smith_togglebutton )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_smith_togglebutton" );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRUE );
      }

      if( rc_config.plots_net_gain )
      {
        widget = Builder_Get_Object(
            plots_window_builder, "plots_net_gain" );
        gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget), TRUE );
      }

      if( isFlagClear(INPUT_OPENED) )
      {
        GtkWidget *box =
          Builder_Get_Object( plots_window_builder, "plots_box" );
        gtk_widget_hide( box );
      }
    } /* if( Main_Plots_Activate() */
    else gtk_check_menu_item_set_active(
        GTK_CHECK_MENU_ITEM(menuitem), FALSE );
  }
  else if( isFlagSet(PLOT_ENABLED) )
    Gtk_Widget_Destroy( &plots_window );

} // Main_Plots_Activate()

/*------------------------------------------------------------------*/

// Handles the on_common_projection_activate() callback
  void
Common_Projection_Activate( GtkMenuItem *menuitem )
{
  /* Enable syncing of projection params
   * for structure and rad pattern drawing */
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    if( isFlagSet(DRAW_ENABLED) )
    {
      rdpattern_proj_params.Wr = structure_proj_params.Wr;
      rdpattern_proj_params.Wi = structure_proj_params.Wi;
      New_Viewer_Angle(
          rdpattern_proj_params.Wr,
          rdpattern_proj_params.Wi,
          rotate_rdpattern,
          incline_rdpattern,
          &rdpattern_proj_params );
    }
    SetFlag( COMMON_PROJECTION );
  }
  else
    ClearFlag( COMMON_PROJECTION );
} // Common_Projection_Activate()

/*------------------------------------------------------------------*/

// Handles the on_main_freq_spinbutton_value_changed() callback
  void
Main_Freq_Spinbutton_Value_Changed( GtkSpinButton *spinbutton )
{
  /* No redraws if new input pending */
  if( isFlagSet(INPUT_PENDING) )
    return;

  /* Frequency spinbutton value changed by frequency loop */
  if( isFlagSet(FREQ_LOOP_RUNNING) )
  {
    /* Wait for GTK to complete its tasks */
    gtk_widget_queue_draw( structure_drawingarea );
    //while( g_main_context_iteration(NULL, FALSE) );
  }
  else if( isFlagClear(FREQ_LOOP_INIT) ) /* by user */
  {
    /* Get frequency from spin button */
    gdouble fmhz = (gdouble)gtk_spin_button_get_value(spinbutton);

    /* If new freq calculations are enabled by
     * checkbutton next to freq spinbutton or
     * freq line plotting enabled, redo currents */
    if( isFlagSet(PLOT_FREQ_LINE)  || (isFlagSet(MAIN_NEW_FREQ) &&
         (isFlagSet(DRAW_CURRENTS) || isFlagSet(DRAW_CHARGES))) )
    {
      /* Recalc currents in structure */
      calc_data.freq_mhz = (double)fmhz;
      g_idle_add( Redo_Currents, NULL );
    }

    /* Sync rad pattern frequency spinbutton */
    /* Show current frequency */
    if( isFlagSet(DRAW_ENABLED)     &&
        isFlagSet(COMMON_FREQUENCY) &&
        isFlagSet(MAIN_NEW_FREQ) )
      gtk_spin_button_set_value( rdpattern_frequency, fmhz );

    /* Wait for GTK to complete its tasks */
    gtk_widget_queue_draw( structure_drawingarea );
    //while( g_main_context_iteration(NULL, FALSE) );
  } /* else */

  gtk_spin_button_update( spinbutton );
} // Main_Freq_Spinbutton_Value_Changed()

/*------------------------------------------------------------------*/

// Handles the on_plots_window_delete_event() callback
  void
Plots_Window_Delete( void )
{
  /* Disable auto setting of freq plots toggle buttons */
  rc_config.plots_gmax_togglebutton    = 0;
  rc_config.plots_gdir_togglebutton    = 0;
  rc_config.plots_gviewer_togglebutton = 0;
  rc_config.plots_vswr_togglebutton    = 0;
  rc_config.plots_zrlzim_togglebutton  = 0;
  rc_config.plots_zmgzph_togglebutton  = 0;
  rc_config.plots_smith_togglebutton   = 0;

  /* Close freq plots window without confirmation dialog */
  if( !rc_config.confirm_quit )
  {
    Gtk_Widget_Destroy( &plots_window );
    return;
  }

  SetFlag( PLOT_QUIT );
  kill_window = plots_window;
  Delete_Event( _("Really close window?") );

} // Plots_Window_Delete()

/*------------------------------------------------------------------*/

// Handles the on_plots_save_activate() callback
  void
Plots_Save_Activate( int *saveas_width, int *saveas_height )
{
  char saveas[FILENAME_LEN + 24];
  size_t s = sizeof( saveas );
  static int cnt = 0;

  if( (strlen(rc_config.input_file) == 0) ||
      isFlagClear(PLOT_SELECT) )
    return;

  saveas_drawingarea = plots_drawingarea;
  *saveas_width      = plots_width;
  *saveas_height     = plots_height;

  /* Make file name from input file name,
   * to save frequency plots drawing */
  snprintf( saveas, s, "%s-%s_%03d%s",
      rc_config.input_file, "plots", ++cnt, ".png" );

  /* Open file chooser to save frequency plots */
  SetFlag( IMAGE_SAVE );
  file_chooser = Open_Filechooser( GTK_FILE_CHOOSER_ACTION_SAVE,
      "*.png", NULL, saveas, rc_config.working_dir );
} // Plots_Save_Activate()

/*------------------------------------------------------------------*/

// Handles the on_rdpattern_window_delete_event() callback
  void
Rdpattern_Window_Delete( void )
{
  /* Disable auto setting of Gain and EH toggle buttons */
  rc_config.rdpattern_gain_togglebutton = 0;
  rc_config.rdpattern_eh_togglebutton   = 0;

  /* Close rdpattern window without confirmation dialog */
  if( !rc_config.confirm_quit )
  {
    Gtk_Widget_Destroy( &rdpattern_window );
    return;
  }

  SetFlag( DRAW_QUIT );
  kill_window = rdpattern_window;
  Delete_Event( _("Really close window?") );
} // Rdpattern_Window_Delete()

/*------------------------------------------------------------------*/

// Handles the  callback
  void
Rdpattern_Save_Activate( int *saveas_width, int *saveas_height )
{
  char saveas[FILENAME_LEN + 24];
  size_t s = sizeof( saveas );
  static int cgn = 0, ceh = 0;

  if( strlen(rc_config.input_file) == 0 ) return;

  saveas_drawingarea = rdpattern_drawingarea;
  *saveas_width      = rdpattern_width;
  *saveas_height     = rdpattern_height;

  /* Make the rad pattern save
   * file name from input name */
  if( isFlagSet(DRAW_GAIN) )
    snprintf( saveas, s, "%s-%s_%03d%s",
        rc_config.input_file, "gain", ++cgn, ".png" );
  else if( isFlagSet(DRAW_EHFIELD) )
    snprintf( saveas, s, "%s-%s_%03d%s",
        rc_config.input_file, "fields", ++ceh, ".png" );
  else return;

  /* Open file chooser to save frequency plots */
  SetFlag( IMAGE_SAVE );
  file_chooser = Open_Filechooser( GTK_FILE_CHOOSER_ACTION_SAVE,
      "*.png", NULL, saveas, rc_config.working_dir );
} // Rdpattern_Save_Activate()

/*------------------------------------------------------------------*/

// Handles the on_rdpattern_freq_spinbutton_value_changed() callback
  void
Rdpattern_Freq_Spinbutton_Value_Changed( GtkSpinButton *spinbutton )
{
  /* No redraws if new input pending */
  if( isFlagSet(INPUT_PENDING) )
    return;

  /* Frequency spinbutton value changed by frequency loop */
  if( isFlagSet(FREQ_LOOP_RUNNING) && isFlagSet(DRAW_ENABLED) )
  {
    /* Wait for GTK to complete its tasks */
    if( isFlagClear(OPTIMIZER_OUTPUT) )
      gtk_widget_queue_draw( rdpattern_drawingarea );
    //while( g_main_context_iteration(NULL, FALSE) );
  }
  else
  {
    /* Get frequency from spin button */
    gdouble fmhz = (gdouble)gtk_spin_button_get_value(spinbutton);

    /* If new freq calculations are enabled
     * by checkbutton next to freq spinbutton */
    GtkWidget *toggle =
      Builder_Get_Object( rdpattern_window_builder, "rdpattern_freq_checkbutton" );
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle)) &&
        (isFlagSet(DRAW_GAIN) || isFlagSet(DRAW_EHFIELD)) )
    {
      /* Recalc currents in structure and rad pattern */
      calc_data.freq_mhz = (double)fmhz;
      g_idle_add( Redo_Radiation_Pattern, NULL );

      /* Sync main window frequency spinbutton */
      if( isFlagSet(COMMON_FREQUENCY) )
        gtk_spin_button_set_value( mainwin_frequency, fmhz );
    }
  } /* else */

  gtk_spin_button_update( spinbutton );

} // Rdpattern_Freq_Spinbutton_Value_Changed()

/*------------------------------------------------------------------*/

// Handles the on_quit_okbutton_clicked() callback
  void
Quit_Okbutton_Clicked( void )
{
  if( isFlagSet(FREQ_LOOP_RUNNING) )
  {
    if( isFlagSet(MAIN_QUIT) )
    {
      SetFlag( FREQ_LOOP_STOP );
      gtk_label_set_text( GTK_LABEL(
            Builder_Get_Object(quit_dialog_builder, "quit_label")),
          _("Really quit Xnec2c?") );
      ClearFlag( MAIN_QUIT );
      return;
    }

    /* Stop freq loop if only one of plots
     * or radiation pattern windows is open */
    if( (isFlagSet(DRAW_ENABLED)   && isFlagClear(PLOT_ENABLED)) ||
        (isFlagClear(DRAW_ENABLED) && isFlagSet(PLOT_ENABLED)) )
      SetFlag( FREQ_LOOP_STOP );

  } /* if( isFlagSet(FREQ_LOOP_RUNNING) ) */

  ClearFlag( MAIN_QUIT | DRAW_QUIT | PLOT_QUIT );
  Gtk_Widget_Destroy( &quit_dialog );
  Gtk_Widget_Destroy( &kill_window );
} // Quit_Okbutton_Clicked()

/*------------------------------------------------------------------*/

// Handles the on_nec2_save_dialog_response() callback
  void
Nec2_Save_Dialog_Response( gint response_id )
{
  Gtk_Widget_Destroy( &nec2_save_dialog );

  /* Discard edited data */
  if( response_id == GTK_RESPONSE_NO )
  {
    /* Cancel NEC2 data save */
    ClearFlag( NEC2_EDIT_SAVE );
    ClearFlag( NEC2_SAVE );

    /* Open file chooser if user requested an input file to be opened */
    if( isFlagSet(OPEN_INPUT) )
    {
      /* Open file chooser to select a NEC2 input file */
      file_chooser = Open_Filechooser(
          GTK_FILE_CHOOSER_ACTION_OPEN, "*.nec", NULL, NULL, rc_config.working_dir );
      Gtk_Widget_Destroy( &nec2_save_dialog );
      return;
    }

    /* Open a new NEC2 project */
    if( isFlagSet(OPEN_NEW_NEC2) )
    {
      /* Open editor window if needed */
      if( nec2_edit_window == NULL )
      {
        Close_File( &input_fp );
        Open_Nec2_Editor( NEC2_EDITOR_NEW );
      }
      else Nec2_Input_File_Treeview( NEC2_EDITOR_NEW );

      rc_config.input_file[0] = '\0';
      selected_treeview = cmnt_treeview;
      ClearFlag( OPEN_NEW_NEC2 );
    }
  } /* if( response_id == GTK_RESPONSE_NO ) */
  else if( response_id == GTK_RESPONSE_YES )
  {
    /* Open file chooser to specify file name to save
     * NEC2 editor data to, if no file is already open */
    SetFlag( NEC2_SAVE );
    if( strlen(rc_config.input_file) == 0 )
    {
      file_chooser = Open_Filechooser( GTK_FILE_CHOOSER_ACTION_SAVE,
          "*.nec", NULL, "untitled.nec", rc_config.working_dir );
      return;
    }
    else /* Save to already open input file */
      Save_Nec2_Input_File( nec2_edit_window, rc_config.input_file );

    /* Re-open NEC2 input file */
    gboolean new = FALSE;
    if( Nec2_Apply_Checkbutton() && isFlagClear(MAIN_QUIT) )
      Open_Input_File( (gpointer)(&new) );

    /* Open file chooser if user requested an input file to be opened */
    if( isFlagSet(OPEN_INPUT) )
    {
      file_chooser = Open_Filechooser(
          GTK_FILE_CHOOSER_ACTION_OPEN, "*.nec", NULL, NULL, rc_config.working_dir );
      return;
    }

    /* Open a new NEC2 project */
    if( isFlagSet(OPEN_NEW_NEC2) )
    {
      /* Open editor window if needed */
      if( nec2_edit_window == NULL )
      {
        Close_File( &input_fp );
        Open_Nec2_Editor( NEC2_EDITOR_NEW );
      }
      else Nec2_Input_File_Treeview( NEC2_EDITOR_NEW );

      rc_config.input_file[0] = '\0';
      selected_treeview = cmnt_treeview;
    }
  } /* if( response_id == GTK_RESPONSE_YES ) */

  /* Save GUI state data for restoring
   * windows if user is quitting xnec2c */
  if( isFlagSet(MAIN_QUIT) )
  {
    Get_GUI_State();
    Save_Config();
  }

  /* Kill window that initiated the save dialog.
   * If it was the main window, xnec2c will exit */
  Gtk_Widget_Destroy( &kill_window );

} // Nec2_Save_Dialog_Response()

/*------------------------------------------------------------------*/

// Handles the on_nec2_row_add_clicked() callback
  void
Nec2_Row_Add_Clicked( void )
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter, sibling;
  int ncols;

  if( selected_treeview == NULL )
    return;

  /* Find selected row and add new after */
  selection = gtk_tree_view_get_selection( selected_treeview );
  if( !gtk_tree_selection_get_selected(selection, &model, &sibling) )
  {
    /* Empty tree view case */
    model = gtk_tree_view_get_model( selected_treeview );
    gtk_list_store_insert( GTK_LIST_STORE(model), &iter, 0 );
  }
  else gtk_list_store_insert_after(
      GTK_LIST_STORE(model), &iter, &sibling);
  gtk_tree_selection_select_iter( selection, &iter );

  /* Prime columns of new row */
  ncols = gtk_tree_model_get_n_columns( model );
  if( ncols == 2 ) /* Comments treeview */
    gtk_list_store_set( GTK_LIST_STORE(model), &iter, 0, "CM", -1 );
  else
  {
    int idx;
    for( idx = 0; idx < ncols; idx++ )
      gtk_list_store_set( GTK_LIST_STORE(model), &iter, idx, "--", -1 );
  }

  SetFlag( NEC2_EDIT_SAVE );
} // Nec2_Row_Add_Clicked()

/*------------------------------------------------------------------*/

/* New_Viewer_Angle()
 *
 * Sets parameters for a new viewer angle
 */
  void
New_Viewer_Angle(
    double wr, double wi,
    GtkSpinButton *wr_spb,
    GtkSpinButton *wi_spb,
    projection_parameters_t *params )
{
  /* Recalculate projection paramenters */
  params->Wr = wr;
  params->Wi = wi;

  /* Set new value */
  Set_Spin_Button( wr_spb, (gdouble)params->Wr );
  Set_Spin_Button( wi_spb, (gdouble)params->Wi );

} /* New_Viewer_Angle() */

/*-----------------------------------------------------------------------*/

/* Motion_Event()
 *
 * Handles pointer motion event on drawingareas
 */
  void
Motion_Event(
    GdkEventMotion *event,
    projection_parameters_t *params )
{
  /* Save previous pointer position */
  static gdouble x_prev = 0.0, y_prev = 0.0;

  gdouble x = event->x;
  gdouble y = event->y;
  gdouble dx, dy;
  gchar value[6];
  size_t s = sizeof( value );

  SetFlag( BLOCK_MOTION_EV );

  /* Initialize saved x,y */
  if( params->reset )
  {
    x_prev = x;
    y_prev = y;
    params->reset = FALSE;
  }

  /* Recalculate projection parameters
   * according to pointer motion */
  dx = x - x_prev;
  dy = y - y_prev;
  x_prev = x;
  y_prev = y;

  /* Other buttons are used for moving axes on screen */
  if( event->state & GDK_BUTTON1_MASK )
  {
    /* Set the structure rotate/incline spinbuttons */
    if( isFlagSet(COMMON_PROJECTION) ||
        (params->type == STRUCTURE_DRAWINGAREA) )
    {
      structure_proj_params.Wr -= dx / (gdouble)MOTION_EVENTS_COUNT;
      structure_proj_params.Wi += dy / (gdouble)MOTION_EVENTS_COUNT;
      snprintf( value, s, "%d", (int)structure_proj_params.Wr );
      gtk_entry_set_text( GTK_ENTRY(rotate_structure), value );
      snprintf( value, s, "%d", (int)structure_proj_params.Wi );
      gtk_entry_set_text( GTK_ENTRY(incline_structure), value );
    }

    /* Set the rdpattern rotate/incline spinbuttons */
    if( (isFlagSet(DRAW_ENABLED) && isFlagSet(COMMON_PROJECTION)) ||
        (params->type == RDPATTERN_DRAWINGAREA) )
    {
      rdpattern_proj_params.Wr -= dx / (gdouble)MOTION_EVENTS_COUNT;
      rdpattern_proj_params.Wi += dy / (gdouble)MOTION_EVENTS_COUNT;
      snprintf( value, s, "%d", (int)rdpattern_proj_params.Wr );
      gtk_entry_set_text( GTK_ENTRY(rotate_rdpattern), value );
      snprintf( value, s, "%d", (int)rdpattern_proj_params.Wi );
      gtk_entry_set_text( GTK_ENTRY(incline_rdpattern), value );
    }

    /* Rotate/incline structure */
    if( params->type == STRUCTURE_DRAWINGAREA )
    {
      New_Structure_Projection_Angle();
      if( isFlagSet(DRAW_ENABLED) &&
          isFlagSet(COMMON_PROJECTION) )
        New_Radiation_Projection_Angle();
    }
    else if( params->type == RDPATTERN_DRAWINGAREA )
    {
      /* Rotate/incline rdpattern */
      New_Radiation_Projection_Angle();
      if( isFlagSet(COMMON_PROJECTION) )
        New_Structure_Projection_Angle();
    }
  }    /* if( event->state & GDK_BUTTON1_MASK ) */
  else
  {
    /* Move structure or rdpattern axes on screen */
    params->dx_center += dx;
    params->dy_center -= dy;

    if( params->type == STRUCTURE_DRAWINGAREA )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( structure_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    if( params->type == RDPATTERN_DRAWINGAREA )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
  }

  ClearFlag( BLOCK_MOTION_EV );

} /* Motion_Event() */

/*-----------------------------------------------------------------------*/

/* Plot_Select()
 *
 * Sets up plotting of requested freq data
 */
  void
Plot_Select( GtkToggleButton *togglebutton, unsigned long long int flag )
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    SetFlag( flag | PLOT_SELECT );
    calc_data.ngraph++;
  }
  else
  {
    ClearFlag( flag );
    calc_data.ngraph--;
  }

  /* Trigger a redraw of frequency plots drawingarea */
  if( isFlagSet(PLOT_ENABLED) && isFlagSet(FREQ_LOOP_DONE) )
  {
    /* Wait for GTK to complete its tasks */
    gtk_widget_queue_draw( plots_drawingarea );
    //while( g_main_context_iteration(NULL, FALSE) );
  }

} /* Plot_Select() */

/*-----------------------------------------------------------------------*/

/* Nec2_Edit_Save()
 *
 * Prompts user to save NEC2 data if edited
 */
  gboolean
Nec2_Edit_Save( void )
{
  if( isFlagSet(NEC2_EDIT_SAVE) )
  {
    if( nec2_save_dialog == NULL )
    {
      nec2_save_dialog = create_nec2_save_dialog( &nec2_save_dialog_builder );
      gtk_widget_show( nec2_save_dialog );
    }
    return( TRUE );
  }
  else return( FALSE );

} /* Nec2_Edit_Save() */

/*-----------------------------------------------------------------------*/

/* Delete_Event()
 *
 * Handles user request to delete a window
 */
  void
Delete_Event( gchar *mesg )
{
  quit_dialog = create_quit_dialog( &quit_dialog_builder );
  gtk_widget_show( quit_dialog );

  if( isFlagSet(FREQ_LOOP_RUNNING) )
  {
    if( isFlagSet(MAIN_QUIT) )
      gtk_label_set_text( GTK_LABEL(
            Builder_Get_Object(quit_dialog_builder, "quit_label")),
          _("The frequency loop is running\n"
            "Really end operation?") );
    else
      gtk_label_set_text( GTK_LABEL(
          Builder_Get_Object(quit_dialog_builder, "quit_label")),
        _("The frequency loop is running\n"
          "Really close this window?") );
  }
  else gtk_label_set_text( GTK_LABEL(
        Builder_Get_Object(quit_dialog_builder, "quit_label")), mesg );

} /* Delete_Event() */

/*-----------------------------------------------------------------------*/

/* Set_Pol_Menuitem()
 *
 * Sets the polarization type menuitem to current setting
 */
  void
Set_Pol_Menuitem( int window )
{
  gchar *main_pol_menu[NUM_POL] =
  {
    "main_total",
    "main_horizontal",
    "main_vertical",
    "main_right_hand",
    "main_left_hand",
  };

  gchar *plots_pol_menu[NUM_POL] =
  {
    "plots_total",
    "plots_horizontal",
    "plots_vertical",
    "plots_right_hand",
    "plots_left_hand",
  };

  gchar *rdpattern_pol_menu[NUM_POL] =
  {
    "rdpattern_total",
    "rdpattern_horizontal",
    "rdpattern_vertical",
    "rdpattern_right_hand",
    "rdpattern_left_hand",
  };

  switch( window )
  {
    case MAIN_WINDOW:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(main_window_builder,
              main_pol_menu[calc_data.pol_type])), TRUE );
      break;

    case FREQPLOTS_WINDOW:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(plots_window_builder,
              plots_pol_menu[calc_data.pol_type])), TRUE );
      break;

    case RDPATTERN_WINDOW:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(rdpattern_window_builder,
              rdpattern_pol_menu[calc_data.pol_type])), TRUE );
      break;
  } /* switch( window ) */

} /* Set_Pol_Menuitem() */

/*-----------------------------------------------------------------------*/

/* Open_Editor()
 *
 * Pops up an Editor window on user
 * right-click on a NEC2 Editor treeview
 */
  gboolean
Open_Editor( GtkTreeView *view )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *card;
  GtkWidget *button;

  /* Find the selected treeview row */
  selection = gtk_tree_view_get_selection( view );
  if( !gtk_tree_selection_get_selected(selection, &model, &iter) )
    return( FALSE );

  /* Get the "card" name from first column */
  gtk_tree_model_get( model, &iter, 0, &card, -1);
  size_t s = strlen( card ) + 1;

  /* Ignore cards from comments */
  if( (strcmp(card, "CM") == 0) ||
      (strcmp(card, "CE") == 0) )
    return( TRUE );

  /* Some "cards" have common editors */
  if( strcmp(card, "GC") == 0 )
    Strlcpy( card, "GW", s );
  else if( strcmp(card, "SC") == 0 )
    Strlcpy( card, "SP", s );
  else if( strcmp(card, "SM") == 0 )
    Strlcpy( card, "SP", s );
  else if( strcmp(card, "NH") == 0 )
    Strlcpy( card, "NE", s );
  else if( strcmp(card, "GE") == 0 )
  {
    Gend_Editor( EDITOR_EDIT );
    return( TRUE );
  } /* EN Not editable */
  else if( strcmp(card, "EN") == 0 )
    return( TRUE );

  /* Send a "clicked" signal to the appropriate editor button */
  card[0] = (gchar)tolower((int)card[0]);
  card[1] = (gchar)tolower((int)card[1]);
  button = Builder_Get_Object( nec2_editor_builder, card );
  g_free(card);
  if( button != NULL )
    g_signal_emit_by_name( button, "clicked" );
  else return( FALSE );

  return( TRUE );
} /* Open_Editor() */

/*-----------------------------------------------------------------------*/

/* Card_Clicked()
 *
 * Performs actions needed when a "card" editor is to be opened
 */
  void
Card_Clicked(
    GtkWidget **editor,
    GtkBuilder **editor_builder,
    GtkWidget *create_fun(GtkBuilder **),
    void editor_fun(int),
    int *editor_action )
{

  if( isFlagSet(EDITOR_QUIT) )
  {
    if( *editor ) editor_fun( EDITOR_APPLY );
    ClearFlag( EDITOR_QUIT );
    return;
  }

  if( *editor == NULL )
  {
    *editor = create_fun( editor_builder );
    gtk_widget_show( *editor );
  }
  else editor_fun( EDITOR_APPLY );

  editor_fun( *editor_action );
  *editor_action = EDITOR_NEW;

}/* Card_Clicked() */

/*-----------------------------------------------------------------------*/

/* Main_Rdpattern_Activate()
 *
 * Callback function for the Radiation Pattern draw button
 */
  void
Main_Rdpattern_Activate( gboolean from_menu )
{
  /* Set structure overlay in Rad Pattern window */
  if( isFlagClear(OVERLAY_STRUCT) )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rdpattern_window_builder,
            "rdpattern_overlay_structure")), FALSE );
  else
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rdpattern_window_builder,
            "rdpattern_overlay_structure")), TRUE );

  /* Sync common projection spinbuttons */
  if( isFlagSet(COMMON_PROJECTION) )
  {
    gchar value[6];
    size_t s = sizeof( value ) - 1;

    rdpattern_proj_params.Wr = structure_proj_params.Wr;
    rdpattern_proj_params.Wi = structure_proj_params.Wi;
    snprintf( value, s, "%d", (int)rdpattern_proj_params.Wr );
    value[s] = '\0';
    gtk_entry_set_text( GTK_ENTRY(rotate_rdpattern), value );
    snprintf( value, s, "%d", (int)rdpattern_proj_params.Wi );
    value[s] = '\0';
    gtk_entry_set_text( GTK_ENTRY(incline_rdpattern), value );
  }
  else  /* Initialize radiation pattern projection angles */
  {
    rdpattern_proj_params.Wr =
      gtk_spin_button_get_value(rotate_rdpattern);
    rdpattern_proj_params.Wi =
      gtk_spin_button_get_value(incline_rdpattern);
  }
  New_Radiation_Projection_Angle();

  /* Reset zoom value */
  rdpattern_proj_params.xy_zoom  = gtk_spin_button_get_value( rdpattern_zoom );
  rdpattern_proj_params.xy_zoom /= 100.0;
  rdpattern_proj_params.xy_scale =
    rdpattern_proj_params.xy_scale1 * rdpattern_proj_params.xy_zoom;

  /* Redo currents if not reaching this function
   * from the menu callback (e.g. not user action) */
  if( !crnt.valid && !from_menu ) Redo_Currents( NULL );

  /* Display frequency in freq spinbutton */
  if( from_menu && calc_data.FR_cards )
  {
    char value[9];
    size_t s = sizeof( value );
    snprintf( value, s, "%.3f", calc_data.freq_mhz );
    value[s - 1] = '\0';
    gtk_entry_set_text( GTK_ENTRY(rdpattern_frequency), value );
  }

  /* Enable Gain or E/H field drawing */
  SetFlag( DRAW_ENABLED );

} /* Main_Rdpattern_Activate() */

/*-----------------------------------------------------------------------*/

/* Main_Plots_Activate()
 *
 * Callback function for for the main Frequency Plots button
 */
  gboolean
Main_Plots_Activate( void )
{
  /* No plots for Incident Field and
   * Elementary Current Source Excitation */
  if( (fpat.ixtyp != 0) && (fpat.ixtyp != 5) )
  {
    Stop( _("Not available for Incident Field or\n"
          "Elementary Current Source Excitation.\n"
          "(Excitation Types 1 to 4)"), ERR_OK );
    return( FALSE );
  }

  /* Enable freq data graph plotting */
  SetFlag( PLOT_ENABLED );

  return( TRUE );
} /* Main_Plots_Activate() */

/*-----------------------------------------------------------------------*/

/* Rdpattern_Gain_Togglebutton_Toggled()
 *
 * Callback function for Rad Pattern window Gain button
 */
  void
Rdpattern_Gain_Togglebutton_Toggled( gboolean flag )
{
  /* If radiation pattern data do not
   * allow drawing of radiation pattern */
  if( isFlagClear(ENABLE_RDPAT) ) return;

  /* Enable or not gain (radiation) pattern plotting */
  if( flag )
  {
    SetFlag( DRAW_GAIN );
    ClearFlag( DRAW_EHFIELD );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
          Builder_Get_Object(rdpattern_window_builder, "rdpattern_eh_togglebutton")),
        FALSE );

    /* Redraw radiation pattern drawingarea */
    if( isFlagSet(DRAW_ENABLED) && isFlagClear(FREQ_LOOP_RUNNING) )
    {
      if( !crnt.valid ) Redo_Currents( NULL );
      SetFlag( DRAW_NEW_RDPAT );

      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    Set_Window_Labels();
  }
  else
  {
    ClearFlag( DRAW_GAIN );
    /* Clear radiation pattern drawingarea */
    if( isFlagClear(DRAW_EHFIELD) && isFlagSet(DRAW_ENABLED) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
    Free_Draw_Buffers();
  }

  return;
} /* Rdpattern_Gain_Togglebutton_Toggled() */

/*-----------------------------------------------------------------------*/

/* Rdpattern_EH_Togglebutton_Toggled()
 *
 * Callback function for Rad Pattern window E/H field button
 */
  void
Rdpattern_EH_Togglebutton_Toggled( gboolean flag )
{
  /* If no near EH data */
  if( !fpat.nfeh ) return;

  /* Enable or not E/H fields plotting */
  if( flag )
  {
    SetFlag( DRAW_EHFIELD );
    ClearFlag( DRAW_GAIN );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(Builder_Get_Object(
            rdpattern_window_builder, "rdpattern_gain_togglebutton")), FALSE );

    /* Delegate near field calcuations to child
     * processes if forked and near field data valid */
    if( FORKED )
    {
      Alloc_Nearfield_Buffers(fpat.nrx, fpat.nry, fpat.nrz);
      Pass_EH_Flags();
    }

    /* Redraw radiation pattern drawingarea */
    if( isFlagSet(DRAW_ENABLED) && isFlagClear(FREQ_LOOP_RUNNING) )
    {
      if( !near_field.valid || !crnt.valid ) Redo_Currents( NULL );
      Near_Field_Pattern();
      SetFlag( DRAW_NEW_EHFIELD );

      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    Set_Window_Labels();
  }
  else
  {
    ClearFlag( NEAREH_ANIMATE );
    ClearFlag( DRAW_EHFIELD );

    /* Clear radiation pattern drawingarea */
    if( isFlagClear(DRAW_GAIN) && isFlagSet(DRAW_ENABLED) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    /* Disable near field calcuations
     * by child processes if forked */
    Pass_EH_Flags();
  }

} /* Rdpattern_EH_Togglebutton_Toggled() */

/*-----------------------------------------------------------------------*/

/* Alloc_Crnt_Buffs()
 *
 * Allocates memory for current/charge draw buffers
 */
  static void
Alloc_Crnt_Buffs( void )
{
  /* Patch currents buffer */
  if( data.m > 0 )
  {
    size_t mreq = (size_t)data.m * sizeof( double );
    Mem_Realloc( (void **)&ct1m, mreq, "in callback_func.c" );
    Mem_Realloc( (void **)&ct2m, mreq, "in callback_func.c" );
  }

  /* Segment currents buffer */
  if( data.n > 0 )
  {
    size_t mreq = (size_t)data.n * sizeof( double );
    Mem_Realloc( (void **)&cmag, mreq, "in callback_func.c" );
  }

} /* Alloc_Crnt_Buffs() */

/*-----------------------------------------------------------------------*/

/* Free_Crnt_Buffs()
 *
 * Frees current/charge draw buffers
 */
  static void
Free_Crnt_Buffs( void )
{
  Mem_Free( (void **)&ct1m );
  Mem_Free( (void **)&ct2m );
  Mem_Free( (void **)&cmag );
} /* Free_Crnt_Buffs() */

/*-----------------------------------------------------------------------*/

/* Main_Currents_Togglebutton_Toggled()
 *
 * Callback function for Main Currents toggle button
 */
  void
Main_Currents_Togglebutton_Toggled( gboolean flag )
{
  /* Enable calculation and rendering of structure curents */
  if( flag )
  {
    SetFlag( DRAW_CURRENTS );
    ClearFlag( DRAW_CHARGES );
    crnt.newer = 1;
    Alloc_Crnt_Buffs();

    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON(Builder_Get_Object(
            main_window_builder, "main_charges_togglebutton")), FALSE );
    gtk_label_set_text(GTK_LABEL(Builder_Get_Object(
            main_window_builder, "struct_label")), _("View Currents") );

    if( !crnt.valid && isFlagClear(FREQ_LOOP_RUNNING) )
      Redo_Currents( NULL );
    else if( crnt.valid )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( structure_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    if( isFlagSet(OVERLAY_STRUCT) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
  }
  else
  {
    ClearFlag( DRAW_CURRENTS );
    if( isFlagClear(DRAW_CHARGES) )
    {
      /* Redraw structure on screen if frequency loop is not running */
      gtk_label_set_text( GTK_LABEL(
            Builder_Get_Object(main_window_builder, "struct_label")),
          _("View Geometry") );
      if( isFlagClear(FREQ_LOOP_RUNNING) )
      {
        /* Wait for GTK to complete its tasks */
        gtk_widget_queue_draw( structure_drawingarea );
        //while( g_main_context_iteration(NULL, FALSE) );
      }
      Free_Crnt_Buffs();
    }
    if( isFlagSet(OVERLAY_STRUCT) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
  }

} /* Main_Currents_Togglebutton_Toggled() */

/*-----------------------------------------------------------------------*/

/* Main_Charges_Togglebutton_Toggled()
 *
 * Callback function for Main Charges toggle button
 */
  void
Main_Charges_Togglebutton_Toggled( gboolean flag )
{
  if( flag )
  {
    SetFlag( DRAW_CHARGES );
    ClearFlag( DRAW_CURRENTS );
    crnt.newer = 1;
    Alloc_Crnt_Buffs();

    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON(
          Builder_Get_Object(main_window_builder, "main_currents_togglebutton")),
        FALSE );
    gtk_label_set_text(GTK_LABEL(
          Builder_Get_Object(main_window_builder, "struct_label")),
        _("View Charges") );

    if( !crnt.valid && isFlagClear(FREQ_LOOP_RUNNING) )
      Redo_Currents( NULL );
    else if( crnt.valid )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( structure_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }

    if( isFlagSet(OVERLAY_STRUCT) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
  }
  else
  {
    ClearFlag( DRAW_CHARGES );
    if( isFlagClear(DRAW_CURRENTS) )
    {
      /* Redraw structure on screen if frequency loop is not running */
      gtk_label_set_text(
          GTK_LABEL(Builder_Get_Object(
              main_window_builder, "struct_label")), _("View Geometry") );

      if( isFlagClear(FREQ_LOOP_RUNNING) )
      {
        /* Wait for GTK to complete its tasks */
        gtk_widget_queue_draw( structure_drawingarea );
        //while( g_main_context_iteration(NULL, FALSE) );
      }

      Free_Crnt_Buffs();
    }

    if( isFlagSet(OVERLAY_STRUCT) )
    {
      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( rdpattern_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );
    }
  }

} /* Main_Charges_Togglebutton_Toggled() */

/*-----------------------------------------------------------------------*/

/* Open_Filechooser()
 *
 * Opens the file chooser dialog to select open or save files
 */
  GtkWidget *
Open_Filechooser(
    GtkFileChooserAction action,
    char *pattern,
    const char *prefix,
    char *filename,
    char *foldername )
{
  /* Create file chooser and set action */
  GtkBuilder *builder;
  GtkWidget *chooser = create_filechooserdialog( &builder );
  gtk_file_chooser_set_action( GTK_FILE_CHOOSER(chooser), action );

  /* Create and set a filter for the file pattern */
  GtkFileFilter *filter = gtk_file_filter_new();
  gtk_file_filter_add_pattern( filter, pattern );

  /* Set the filter name */
  if( strcmp(pattern, "*.png") == 0 )
    gtk_file_filter_set_name( filter, _("PNG images (*.png)") );
  else if( strcmp(pattern, "*.gplot") == 0 )
    gtk_file_filter_set_name( filter, _("GNUplot files (*.gplot)") );
  else if( strcmp(pattern, "*.nec") == 0 )
    gtk_file_filter_set_name( filter, _("NEC2 files (*.nec)") );

  /* Add and set filter */
  gtk_file_chooser_add_filter( GTK_FILE_CHOOSER(chooser), filter );
  gtk_file_chooser_set_filter( GTK_FILE_CHOOSER(chooser), filter );

  /* Set current filename if given */
  if( filename != NULL )
  {
    char fname[144];
    if( prefix != NULL )
    {
      Strlcpy( fname, rc_config.working_dir, sizeof(fname) );
      Strlcat( fname, filename, sizeof(fname) );
    }
    else
      Strlcpy( fname, filename, sizeof(fname) );

    gtk_file_chooser_set_current_name(
        GTK_FILE_CHOOSER(chooser), fname );
  }

  /* Set folder name if given */
  if( foldername != NULL )
    gtk_file_chooser_set_current_folder(
        GTK_FILE_CHOOSER(chooser), foldername );
  gtk_widget_show( chooser);
  g_object_unref( builder );

  return( chooser );

} /* Open_Filechooser() */

/*-----------------------------------------------------------------------*/

/* Filechooser_Response()
 *
 * Handles the on_filechooserdialog_response callback
 */
  void
Filechooser_Response(
    GtkDialog *dialog,
    gint response_id,
    int saveas_width,
    int saveas_height )
{
  /* User selects a file name to save a pixbuf to file */
  if( response_id == GTK_RESPONSE_OK )
  {
    gchar *fname;
    gchar filename[LINE_LEN];
    gboolean new;
    char *str;

    /* Get the "save as" file name */
    fname = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(dialog) );
    Strlcpy( filename, fname, sizeof(filename) );
    Gtk_Widget_Destroy( (GtkWidget **)&dialog );

    if( isFlagSet(NEC2_SAVE) )
    {
      /* Cat a file extension if not already there */
      str = strstr( filename, ".nec" );
      if( (str == NULL) || (str[4] != '\0') )
        Strlcat( filename, ".nec", sizeof(filename) );

      /* Use new file name as input file */
      Strlcpy( rc_config.input_file, filename, sizeof(rc_config.input_file) );
      Save_Nec2_Input_File( nec2_edit_window, rc_config.input_file );

      /* Re-open NEC2 input file */
      new = FALSE;
      if( Nec2_Apply_Checkbutton() && isFlagClear(MAIN_QUIT) )
        Open_Input_File( (gpointer)(&new) );
    }
    else if( isFlagSet(OPEN_INPUT) )
    {
      ClearFlag( FREQ_LOOP_READY );

      /* Save any changes to an open file */
      Strlcpy( rc_config.input_file, fname, sizeof(rc_config.input_file) );

      /* Open new file */
      new = TRUE;
      Open_Input_File( (gpointer)(&new) );
      ClearFlag( OPEN_INPUT );
    }
    else if( isFlagSet(IMAGE_SAVE) )
    {
      /* cat a file extension if not already there */
      str = strstr( filename, ".png" );
      if( (str == NULL) || (str[4] != '\0') )
        Strlcat( filename, ".png", sizeof(filename) );

      /* Save screen shots after redraw and when GTK is finished tasks */
      static save_data_t save_data;

      /* Wait for GTK to complete its tasks */
      gtk_widget_queue_draw( saveas_drawingarea );
      //while( g_main_context_iteration(NULL, FALSE) );

      save_data.drawingarea = saveas_drawingarea;
      save_data.width  = saveas_width;
      save_data.height = saveas_height;
      Strlcpy( save_data.filename, filename, sizeof(save_data.filename) );
      g_idle_add( Save_Pixbuf, (gpointer)&save_data );
      ClearFlag( IMAGE_SAVE );
    }
    else if( isFlagSet(RDPAT_GNUPLOT_SAVE) )
    {
      /* cat a file extension if not already there */
      str = strstr( filename, ".gplot" );
      if( (str == NULL) || (str[6] != '\0') )
        Strlcat( filename, ".gplot", sizeof(filename) );
      Save_RadPattern_Gnuplot_Data( filename );
      ClearFlag( RDPAT_GNUPLOT_SAVE );
    }
    else if( isFlagSet(PLOTS_GNUPLOT_SAVE) )
    {
      /* cat a file extension if not already there */
      str = strstr( filename, ".gplot" );
      if( (str == NULL) || (str[6] != '\0') )
        Strlcat( filename, ".gplot", sizeof(filename) );
      Save_FreqPlots_Gnuplot_Data( filename );
      ClearFlag( PLOTS_GNUPLOT_SAVE );
    }
    else if( isFlagSet(STRUCT_GNUPLOT_SAVE) )
    {
      /* cat a file extension if not already there */
      str = strstr( filename, ".gplot" );
      if( (str == NULL) || (str[6] != '\0') )
        Strlcat( filename, ".gplot", sizeof(filename) );
      Save_Struct_Gnuplot_Data( filename );
      ClearFlag( STRUCT_GNUPLOT_SAVE );
    }
    g_free( fname );

    /* Open file chooser if user requested an input file to be opened */
    if( isFlagSet(OPEN_INPUT) )
    {
      file_chooser = Open_Filechooser(
          GTK_FILE_CHOOSER_ACTION_OPEN,
          "*.nec", NULL, NULL, rc_config.working_dir );
      return;
    }

    /* Open a new NEC2 project */
    if( isFlagSet(OPEN_NEW_NEC2) )
    {
      /* Open editor window if needed */
      if( nec2_edit_window == NULL )
      {
        Close_File( &input_fp );
        Open_Nec2_Editor( NEC2_EDITOR_NEW );
      }
      else Nec2_Input_File_Treeview( NEC2_EDITOR_NEW );

      rc_config.input_file[0] = '\0';
      selected_treeview = cmnt_treeview;
    }

    /* Save GUI state data for restoring
     * windows if user is quitting xnec2c */
    if( isFlagSet(MAIN_QUIT) )
    {
      Get_GUI_State();
      Save_Config();
    }

    /* Kill window that initiated edited data save */
    Gtk_Widget_Destroy( &kill_window );

  } /* if( response_id == GTK_RESPONSE_OK ) */
  else
    ClearFlag( ALL_CHOOSER_FLAGS );

} /* Filechooser_Response() */

/*-----------------------------------------------------------------------*/

/* Open_Nec2_Editor()
 *
 * Opens NEC2 editor window and fills
 * tree view according to action flag
 */
  void
Open_Nec2_Editor( int action )
{
  nec2_edit_window = create_nec2_editor( &nec2_editor_builder );
  //while( g_main_context_iteration(NULL, FALSE) );
  Set_Window_Geometry( nec2_edit_window,
      rc_config.nec2_edit_x, rc_config.nec2_edit_y,
      rc_config.nec2_edit_width, rc_config.nec2_edit_height );
  gtk_widget_show( nec2_edit_window );

  cmnt_treeview = GTK_TREE_VIEW(
      Builder_Get_Object(nec2_editor_builder, "nec2_cmnt_treeview") );
  geom_treeview = GTK_TREE_VIEW(
      Builder_Get_Object(nec2_editor_builder, "nec2_geom_treeview") );
  cmnd_treeview = GTK_TREE_VIEW(
      Builder_Get_Object(nec2_editor_builder, "nec2_cmnd_treeview") );

  Nec2_Input_File_Treeview( action );

  geom_adjustment =
    gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(
          Builder_Get_Object(nec2_editor_builder, "geom_scrolledwindow")) );
  cmnd_adjustment =
    gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(
          Builder_Get_Object(nec2_editor_builder, "cmnd_scrolledwindow")) );

} /* Open_Nec2_Editor() */

/*-----------------------------------------------------------------------*/

/* Nec2_Apply_Checkbuton()
 *
 * Checks whether the NEC2 editor's "Apply" check button is active
 */

  gboolean
Nec2_Apply_Checkbutton( void )
{
  return( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
          Builder_Get_Object(nec2_editor_builder, "nec2_apply_checkbutton" ))) );
}

/*-----------------------------------------------------------------------*/

/* Gtk_Quit()
 *
 * Quits gtk main
 */
  void
Gtk_Quit( void )
{
  int i, k;

  Close_File( &input_fp );
  SetFlag( MAIN_QUIT );

  /* Kill child processes */
  if( FORKED && !CHILD )
    while( num_child_procs )
    {
      num_child_procs--;
      kill( forked_proc_data[num_child_procs]->child_pid, SIGKILL );
    }

  /* Kill possibly nested loops */
  k = (int)gtk_main_level();
  for( i = 0; i < k; i++ )
    gtk_main_quit();

} /* Gtk_Quit() */

/*-----------------------------------------------------------------------*/

/* Pass_EH_Flags
 *
 * Passes near field related flags to child processes
 */
  void
Pass_EH_Flags( void )
{
  char flag;
  size_t cnt;
  int idx;

  /* Abort if not forked */
  if( !FORKED ) return;

  /* Tell child process to calculate near field data */
  cnt = strlen( fork_commands[EHFIELD] );
  for( idx = 0; idx < calc_data.num_jobs; idx++ )
    Write_Pipe( idx, fork_commands[EHFIELD], (ssize_t)cnt, TRUE );

  /* Tell child to set near field flags */
  flag = 0;
  if( isFlagSet(DRAW_EHFIELD) )    flag |= 0x01;
  if( isFlagSet(NEAREH_SNAPSHOT) ) flag |= 0x02;
  if( isFlagSet(DRAW_EFIELD) )     flag |= 0x04;
  if( isFlagSet(DRAW_HFIELD) )     flag |= 0x08;

  cnt = sizeof( flag );
  for( idx = 0; idx < calc_data.num_jobs; idx++ )
    Write_Pipe( idx, &flag, (ssize_t)cnt, TRUE );

} /* Pass_EH_Flags */

/*-----------------------------------------------------------------------*/

/* Draw_Colorcode( cairo_t *cr )
 *
 * Draws the color code bar fopr structure
 * currents and radiation pattern gain range
 */
  void
Draw_Colorcode( cairo_t *cr )
{
  double red = 0.0, grn = 0.0, blu = 0.0;
  int idx;

  /* No redraws if new input pending */
  if( isFlagSet(INPUT_PENDING) ) return;

  /* Draw color-code bar in main window */
  for( idx = 0; idx < COLORCODE_WIDTH; idx++ )
  {
    Value_to_Color( &red, &grn, &blu, (double)(8 * idx), COLORCODE_MAX );
    cairo_set_source_rgb( cr, red, grn, blu );
    Cairo_Draw_Line( cr, idx, 0, idx, COLORCODE_HEIGHT );
  }

} /* Draw_Colorcode() */

/*-----------------------------------------------------------------------*/

/* Gtk_Widget_Destroy()
 *
 * A safety wrapper around gtk_widget_destroy()
 * Checks that the widget is not null before killing it
 */
  void
Gtk_Widget_Destroy( GtkWidget **widget )
{
  if( *widget != NULL )
  {
    gtk_widget_destroy( *widget );
    *widget = NULL;
  }

} /* Gtk_Widget_Destroy() */

/*------------------------------------------------------------------*/

