/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*
 * Functions used by wsprsim
 */

#include "wsprsim_utils.h"
#include "fano.h"
#include "nhash.h"
#include "shared.h"
#include "wsprd.h"
#include "wsprd_utils.h"
#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>

//------------------------------------------------------------------------

/* Get_Locator_Character_Code()
 *
 * Packs (I think!) a locator's character
 * into a minimum number of bits
 */
  static char
Get_Locator_Character_Code( char ch )
{
  if( ch >= 48 && ch <= 57 )  // 0-9
  {
    return( ch - 48 );
  }
  if( ch == 32 )  // space
  {
    return( 36 );
  }
  if( (ch >= 65) && (ch <= 82) )  // A-Z
  {
    return( ch - 65 );
  }

  return( -1 );
} // Get_Locator_Character_Code()

//------------------------------------------------------------------------

/* Get_Callsign_Character_Code()
 *
 * Packs (I think!) a callsign's character
 * into a minimum number of bits
 */
  static char
Get_Callsign_Character_Code( char ch )
{
  if( (ch >= 48) && (ch <= 57) )  // 0-9
  {
    return( ch - 48 );
  }
  if( ch == 32 )  // space
  {
    return( 36 );
  }
  if( (ch >= 65) && (ch <= 90) )  // A-Z
  {
    return( ch - 55 );
  }

  return( -1 );
} // Get_Callsign_Character_Code()

//------------------------------------------------------------------------

/* Pack_Grid4_Power()
 *
 * Packs (I think) a 4-character locator
 * and the Tx power into a long unsigned int
 */
  static long unsigned int
Pack_Grid4_Power( char const *grid4, int power )
{
  long unsigned int m;

  m = (unsigned long)( (179 - 10 * grid4[0] - grid4[2]) * 180 + 10 * grid4[1] + grid4[3] );
  m = m * 128 + (unsigned long)power + 64;
  return( m );
} // Pack_Grid4_Power()

//------------------------------------------------------------------------

/* Pack_Call()
 *
 * Packs (I tihnk!) a callsign into a long unsigned int
 */
  static long unsigned int
Pack_Call( char const *callsign )
{
  unsigned int i;
  long unsigned int n;
  char call6[7];

  memset( call6, ' ', sizeof(call6) );

  // callsign is 6 characters in length. Exactly.
  size_t call_len = strlen( callsign );
  if( (call_len > 6) || (call_len < 2) )
  {
    return 0;
  }

  if( isdigit(callsign[2]) )
  {
    for( i = 0; i < call_len; i++ )
    {
      call6[i] = callsign[i];
    }
  }
  else if( isdigit(callsign[1]) )
  {
    for( i = 1; i < call_len; i++ )
    {
      call6[i] = callsign[i - 1];
    }
  }

  for( i = 0; i < 6; i++ )
  {
    call6[i] = Get_Callsign_Character_Code( call6[i] );
  }

  n = (unsigned long)call6[0];
  n = n * 36 + (unsigned long)call6[1];
  n = n * 10 + (unsigned long)call6[2];
  n = n * 27 + (unsigned long)call6[3] - 10;
  n = n * 27 + (unsigned long)call6[4] - 10;
  n = n * 27 + (unsigned long)call6[5] - 10;

  return( n );
} // Pack_Call()

//------------------------------------------------------------------------

/* Pack_Prefix()
 *
 * Packs (I think!) a callsign prefix, somewhere...
 */
  static void
Pack_Prefix( char *callsign, int32_t *n, int32_t *m, int32_t *nadd )
{
  size_t i;
  char call6[7];
  size_t i1 = strcspn( callsign, "/" );

  if( i1 < 4 ) return;

  if( callsign[i1 + 2] == 0 )
  {
    // single char suffix
    for( i = 0; i < i1; i++ )
    {
      call6[i] = callsign[i];
    }
    call6[i] = '\0';
    *n = (int32_t)Pack_Call( call6 );
    *nadd = 1;
    int nc = callsign[i1 + 1];
    if( nc >= 48 && nc <= 57 )
    {
      *m = nc - 48;
    }
    else if( (nc >= 65) && (nc <= 90) )
    {
      *m = nc - 65 + 10;
    }
    else
    {
      *m = 38;
    }
    *m = 60000 - 32768 + *m;
  }
  else if( callsign[i1 + 3] == 0 )
  {
    // two char suffix
    for( i = 0; i < i1; i++ )
    {
      call6[i] = callsign[i];
    }
    *n = (int32_t)Pack_Call( call6 );
    *nadd = 1;
    *m = 10 * ( callsign[i1 + 1] - 48 ) + ( callsign[i1 + 2] - 48 );
    *m = 60000 + 26 + *m;
  }
  else
  {
    char const *pfx = strtok( callsign, "/" );
    char const *call = strtok( NULL, " " );
    *n = (int32_t)Pack_Call( call );
    size_t plen = strlen( pfx );
    if( plen == 1 )
    {
      *m = 36;
      *m = 37 * ( *m ) + 36;
    }
    else if( plen == 2 )
    {
      *m = 36;
    }
    else
    {
      *m = 0;
    }
    for( i = 0; i < plen; i++ )
    {
      int nc = callsign[i];
      if( nc >= 48 && nc <= 57 )
      {
        nc = nc - 48;
      }
      else if( nc >= 65 && nc <= 90 )
      {
        nc = nc - 65 + 10;
      }
      else
      {
        nc = 36;
      }
      *m = 37 * ( *m ) + nc;
    }
    *nadd = 0;
    if( *m > 32768 )
    {
      *m = *m - 32768;
      *nadd = 1;
    }
  }
} // Pack_Prefix()

//------------------------------------------------------------------------

#define INTERLEAVE  162

/* Interleave()
 *
 * Interleaves symbols
 */
  static void
Interleave( unsigned char *sym )
{
  unsigned char tmp[INTERLEAVE];
  unsigned char p, i;

  p = 0;
  i = 0;
  while( p < INTERLEAVE )
  {
    unsigned char j =
      ( (i * 0x80200802ULL) & 0x0884422110ULL ) * 0x0101010101ULL >> 32;
    if( j < INTERLEAVE )
    {
      tmp[j] = sym[p];
      p++;
    }
    i++;
  }
  for( i = 0; i < INTERLEAVE; i++ )
  {
    sym[i] = tmp[i];
  }

} // Interleave()

//------------------------------------------------------------------------

/* Get_Wspr_Channel_Symbols()
 *
 * Not sure, appears to produce the raw WSPR message to transmit
 */
  int
Get_Wspr_Channel_Symbols(
    const char *rawmessage,
    char *hashtab,
    char *loctab,
    unsigned char *symbols )
{
  int m = 0, ntype;
  long unsigned int n = 0;
  int i;
  const unsigned char pr3vector[INTERLEAVE] =
  {
    1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0,
    0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1,
    0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1,
    1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1,
    0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0,
    0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1,
    0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1,
    0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0,
    0, 0
  };

  const int nu[10] = { 0, -1, 1, 0, -1, 2, 1, 0, -1, 1 };
  char *callsign;
  const char *grid;
  const char *powstr;
  char message[MESSAGE_SIZE];

  memset( message, '\0', sizeof(char) * MESSAGE_SIZE );
  Strlcpy( message, rawmessage, MESSAGE_SIZE );

  size_t i1 = strcspn( message, " " );
  size_t i2 = strcspn( message, "/" );
  size_t i3 = strcspn( message, "<" );
  size_t i4 = strcspn( message, ">" );
  size_t mlen = strlen( message );

  // Use the presence and/or absence of "<" and "/" to decide what
  // type of message. No sanity checks! Beware!

  if( (i1 > 3 && i1 < 7) && (i2 == mlen) && (i3 == mlen) )
  {
    // Type 1 message: K9AN EN50 33
    //                 xxnxxxx xxnn nn
    callsign = strtok( message, " " );
    grid     = strtok( NULL, " " );
    powstr   = strtok( NULL, " " );
    int power = atoi( powstr );
    n = Pack_Call( callsign );

    char grid4[5];
    for( i = 0; i < 4; i++ )
    {
      grid4[i] = Get_Locator_Character_Code( *(grid + i) );
    }
    m = (int)Pack_Grid4_Power( grid4, power );
  }
  else if( (i3 == 0) && (i4 < mlen) )
  {
    // Type 3:      <K1ABC> EN50WC 33
    //          <PJ4/K1ABC> FK52UD 37
    // send hash instead of callsign to make room for 6 char grid.
    // if 4-digit locator is specified, 2 spaces are added to the end.
    callsign = strtok( message, "<> " );
    grid     = strtok( NULL, " " );
    powstr   = strtok( NULL, " " );
    int power = atoi( powstr );
    if( power < 0 )  power = 0;
    if( power > 60 ) power = 60;
    power = power + nu[power % 10];
    ntype = -( power + 1 );
    int ihash = (int)Nhash( callsign, strlen(callsign), (uint32_t)146 );
    m = 128 * ihash + ntype + 64;

    char grid6[7];
    memset( grid6, '\0', sizeof(char) * 7 );
    int j = (int)strlen( grid );
    for( i = 0; i < j - 1; i++ )
    {
      grid6[i] = grid[i + 1];
    }
    grid6[5] = grid[0];
    n = Pack_Call( grid6 );
  }
  else if( i2 < mlen )   // just looks for a right slash
  {
    // Type 2: PJ4/K1ABC 37
    callsign = strtok( message, " " );
    if( (i2 == 0) || (i2 > strlen(callsign)) )
      return( 0 );  // guards against pathological case
    powstr = strtok( NULL, " " );
    int power = atoi( powstr );
    if( power < 0 )  power = 0;
    if( power > 60 ) power = 60;
    power = power + nu[power % 10];
    int n1 = 0, ng = 0, nadd = 0;
    Pack_Prefix( callsign, &n1, &ng, &nadd );
    ntype = power + 1 + nadd;
    m = 128 * ng + ntype + 64;
    n = (unsigned long)n1;
  }
  else
  {
    return( 0 );
  }

  // pack 50 bits + 31 (0) tail bits into 11 bytes
  unsigned char it, data[11];
  memset( data, 0, sizeof(char) * 11 );
  it = 0xFF & ( n >> 20 );
  data[0] = it;
  it = 0xFF & ( n >> 12 );
  data[1] = it;
  it = 0xFF & ( n >> 4 );
  data[2] = it;
  it = (unsigned char)( ((n & (0x0F)) << 4) + ((m >> 18) & (0x0F)) );
  data[3] = it;
  it = 0xFF & ( m >> 10 );
  data[4] = it;
  it = 0xFF & ( m >> 2 );
  data[5] = it;
  it = (unsigned char)( (m & 0x03) << 6 );
  data[6]  = it;
  data[7]  = 0;
  data[8]  = 0;
  data[9]  = 0;
  data[10] = 0;

  // make sure that the 11-byte data vector is unpackable
  // unpack it with the routine that the decoder will use and display
  // the result. let the operator decide whether it worked.

  char
    check_call_loc_pow[MESSAGE_SIZE],
    check_callsign[CALL_SIZE],
    call[CALL_SIZE],
    loc[LOC_SIZE],
    pwr[PWR_SIZE];

    signed char check_data[11];
    memcpy( check_data, data, sizeof(char) * 11 );

    Unpk_( check_data, hashtab, loctab, check_call_loc_pow, call, loc, pwr, check_callsign );
    //printf( "Will decode as: %s\n", check_call_loc_pow );

    unsigned int nbytes = 11;       // The message with tail is packed into almost 11 bytes.
    unsigned char channelbits[176]; // 162 rounded up
    memset( channelbits, 0, sizeof(char) * 176 );

    Encode( channelbits, data, nbytes );

    Interleave( channelbits );

    for( i = 0; i < INTERLEAVE; i++ )
    {
      symbols[i] = 2 * channelbits[i] + pr3vector[i];
    }

    return( 1 );
} // Get_Wspr_Channel_Symbols()

//------------------------------------------------------------------------

/* White_Gaussian_Noise()
 *
 * Generates white noise with Gaussian distribution
 */
  static double
White_Gaussian_Noise( double factor )
{
  static double V2, S;
  double X;
  static int phase = 0;

  if( phase == 0 )
  {
    static double V1;
    do
    {
      double U1 = (double)(rand()) / RAND_MAX;
      double U2 = (double)(rand()) / RAND_MAX;
      V1 = 2 * U1 - 1;
      V2 = 2 * U2 - 1;
      S = V1 * V1 + V2 * V2;
    }
    while( (S >= 1.0) || (S == 0.0) );

    X = V1 * sqrt( -2 * log(S) / S );
  }
  else
  {
    X = V2 * sqrt( -2 * log(S) / S );
  }

  phase = 1 - phase;
  return( (double)X * factor );

} // White_Gaussian_Noise()

//------------------------------------------------------------------------

/* Decoder_Self_Test()
 *
 * WSPR Decoder self test function, hopefully
 */
  int32_t
Decoder_Self_Test( void )
{
  static double iSamples[SAMPLES_BUF_SIZE] = { 0 };
  static double qSamples[SAMPLES_BUF_SIZE] = { 0 };
  static uint32_t samples_len = SAMPLES_BUF_SIZE;
  int16_t n_results = 0;

  unsigned char symbols[INTERLEAVE] = { 0 };
  const char message[] = "K1JT FN20QI 20";
  char hashtab[32768 * 13] = { 0 };
  char loctab[32768 * 5]   = { 0 };  // EVAL: code update from wsprd

  // Compute sympbols from the message
  Get_Wspr_Channel_Symbols( message, hashtab, loctab, symbols );

  double f0  = 50.0;
  double t0  = 2.0;  // Caution!! Possible buffer overflow with the index calculation
  double amp = 1000.0;
  double wgn = 0.02;
  double phi = 0.0;
  double df  = 375.0 / 256.0;
  double dt  = 1 / 375.0;

  // Add signal
  for( int i = 0; i < INTERLEAVE; i++ )
  {
    double f = f0 + ( (double)symbols[i] - 1.5 ) * df;
    double dphi = M_2PI * dt * f;
    for( int j = 0; j < 256; j++ )
    {
      int index = (int)( t0 / dt + 256 * i ) + j;
      iSamples[index] = amp * cos( phi ) + White_Gaussian_Noise( wgn );
      qSamples[index] = amp * sin( phi ) + White_Gaussian_Noise( wgn );
      phi += dphi;
    }
  }

  /* Search & decode the signal */
  Wspr_Decode( iSamples, qSamples, (int)samples_len, decoder_results, &n_results );

  printf( "        SNR      DT        Freq Dr    Call    Loc Pwr\n" );
  for( int32_t i = 0; i < n_results; i++ )
  {
    printf( "Spot(%i) %6.2f %6.2f %10.6f %2d %7s %6s %2s\n",
        i,
        decoder_results[i].snr,
        decoder_results[i].dt,
        decoder_results[i].freq,
        (int)decoder_results[i].drift,
        decoder_results[i].call,
        decoder_results[i].loc,
        decoder_results[i].pwr );
  }

  /* Simple consistency check */
  if( strcmp(decoder_results[0].call, "K1JT") &&
      strcmp(decoder_results[0].loc,  "FN20") &&
      strcmp(decoder_results[0].pwr,  "20"))
  {
    return( 0 );
  }
  else
  {
    return( 1 );
  }
} // Decoder_Self_Test()

//------------------------------------------------------------------------

