/*
   This file is part of program wsprd, a detector/demodulator/decoder
   for the Weak Signal Propagation Reporter (WSPR) mode.

   File name: wsprd_utils.c

   Copyright 2001-2015, Joe Taylor, K1JT

   Most of the code is based on work by Steven Franke, K9AN, which
   in turn was based on earlier work by K1JT.

   Copyright 2014-2015, Steven Franke, K9AN

   License: GNU GPL v3

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "wsprd_utils.h"
#include "nhash.h"
#include "wsprd.h"
#include "../common/common.h"
#include "../common/utils.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdint.h>

//------------------------------------------------------------------------

/* Unpack_50()
 *
 * Unpacks (I think!) 50 decoded symbols
 */
  void
Unpack_50( const signed char *dat, int32_t *n1, int32_t *n2 )
{
  int32_t i, i4;

  i   = dat[0];
  i4  = i & 255;
  *n1 = i4 << 20;

  i   = dat[1];
  i4  = i & 255;
  *n1 = *n1 + ( i4 << 12 );

  i   = dat[2];
  i4  = i & 255;
  *n1 = *n1 + ( i4 << 4 );

  i   = dat[3];
  i4  = i & 255;
  *n1 = *n1 + ( (i4 >> 4) & 15 );
  *n2 = ( i4 & 15 ) << 18;

  i   = dat[4];
  i4  = i & 255;
  *n2 = *n2 + ( i4 << 10 );

  i   = dat[5];
  i4  = i & 255;
  *n2 = *n2 + ( i4 << 2 );

  i   = dat[6];
  i4  = i & 255;
  *n2 = *n2 + ( (i4 >> 6) & 3 );

} // Unpack_50()

//------------------------------------------------------------------------

/* Unpack_Call()
 *
 * Unpacks the callsign of the transmitting station
 */
  int
Unpack_Call( int32_t ncall, char *call )
{
  const char c[] =
  {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B',
    'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
    'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', ' '
  };

  int32_t n;
  char tmp[7];

  n = ncall;
  Strlcpy( call, "......", CALL_SIZE );
  if( n < 262177560 )
  {
    int i;
    i = n % 27 + 10;
    tmp[5] = c[i];
    n /= 27;
    i = n % 27 + 10;
    tmp[4] = c[i];
    n /= 27;
    i = n % 27 + 10;
    tmp[3] = c[i];
    n /= 27;
    i = n % 10;
    tmp[2] = c[i];
    n /= 10;
    i = n % 36;
    tmp[1] = c[i];
    n /= 36;
    i = n;
    tmp[0] = c[i];
    tmp[6] = '\0';

    // Remove leading whitespace
    for( i = 0; i < 5; i++ )
    {
      if( tmp[i] != c[36] )
        break;
    }
    //snprintf( call, CALL_SIZE, "%-6s", &tmp[i] );
    snprintf( call, CALL_SIZE, "%s", &tmp[i] );

    // Remove trailing whitespace
    for( i = 0; i < 6; i++ )
    {
      if( call[i] == c[36] )
        call[i] = '\0';
    }
  }
  else
  {
    return( False );
  }

  return( True );
} // Unpack_Call()

//------------------------------------------------------------------------

#define GRID_SIZE   5

/* Unpack_Grid()
 *
 * Unpacks the grid locator of the transmitting station
 */
  int
Unpack_Grid( int32_t ngrid, char *grid )
{
  const char c[] =
  {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B',
    'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
    'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', ' '
  };

  ngrid = ngrid >> 7;
  if( ngrid < 32400 )
  {
    int dlat, dlong;
    dlat = ( ngrid % 180 ) - 90;
    dlong = ( ngrid / 180 ) * 2 - 180 + 2;
    if( dlong < -180 ) dlong += 360;
    // if( dlong > 180 )  dlong -= 360; dead code
    int nlong = (int)( 60.0 * (180.0 - (double)dlong) / 5.0 );
    int n1 = nlong / 240;
    int n2 = ( nlong - 240 * n1 ) / 24;
    grid[0] = c[10 + n1];
    grid[2] = c[n2];

    int nlat = (int)( 60.0 * (dlat + 90) / 2.5 );
    n1 = nlat / 240;
    n2 = ( nlat - 240 * n1 ) / 24;
    grid[1] = c[10 + n1];
    grid[3] = c[n2];
  }
  else
  {
    Strlcpy( grid, "XXXX", GRID_SIZE );
    return( False );
  }

  return( True );
} // Unpack_Grid()

//------------------------------------------------------------------------

#define PFX_SIZE    4

/* Unpack_Pfx()
 *
 * Unpacks any prefix in the callsign of the transmitting station
 */
  int
Unpack_Pfx( int32_t nprefix, char *call )
{
  char nc, pfx[PFX_SIZE] = {'\0'}, tmpcall[CALL_SIZE];

  Strlcpy( tmpcall, call, CALL_SIZE );
  if( nprefix < 60000 )
  {
    int32_t n, i;

    // add a prefix of 1 to 3 characters
    n = nprefix;
    for( i = 2; i >= 0; i-- )
    {
      nc = n % 37;
      if( (nc >= 0) & (nc <= 9) )
      {
        pfx[i] = nc + 48;
      }
      else if( (nc >= 10) & (nc <= 35) )
      {
        pfx[i] = nc + 55;
      }
      else
      {
        pfx[i] = ' ';
      }
      n = n / 37;
    }

    const char *p = strrchr( pfx, ' ' );
    Strlcpy( call, (p ? p + 1 : pfx), CALL_SIZE );
    Strlcat( call, "/", CALL_SIZE );
    Strlcat( call, tmpcall, CALL_SIZE );
  }
  else
  {
    // add a suffix of 1 or 2 characters
    nc = (char)( nprefix - 60000 );
    if( nc <= 9 )
    {
      pfx[0] = nc + 48;
      pfx[1] = '\0';
      Strlcpy( call, tmpcall, CALL_SIZE );
      Strlcat( call, "/", CALL_SIZE );
      Strlcat( call, pfx, CALL_SIZE );
    }
    else if( nc <= 35 )
    {
      pfx[0] = nc + 55;
      pfx[1] = '\0';
      Strlcpy( call, tmpcall, CALL_SIZE );
      Strlcat( call, "/", CALL_SIZE );
      Strlcat( call, pfx, CALL_SIZE );
    }
    else if( nc <= 125 )
    {
      pfx[0] = (nc - 26) / 10 + 48;
      pfx[1] = (nc - 26) % 10 + 48;
      pfx[2] = '\0';
      Strlcpy( call, tmpcall, CALL_SIZE );
      Strlcat( call, "/", CALL_SIZE );
      Strlcat( call, pfx, CALL_SIZE );
    }
    else
    {
      return( False );
    }
  }

  return( True );
} // Unpack_Pfx( )

//------------------------------------------------------------------------

#define DEINTERLEAVE    162

/* Deinterleave()
 *
 * Deinterleaves (I think!) the symbols stream
 */
  void
Deinterleave( unsigned char *sym )
{
  uint8_t tmp[DEINTERLEAVE];
  uint8_t p, i;

  p = 0;
  i = 0;
  while( p < DEINTERLEAVE )
  {
    uint8_t j =
      ( (i * 0x80200802ULL) & 0x0884422110ULL ) * 0x0101010101ULL >> 32;
    if( j < DEINTERLEAVE )
    {
      tmp[p] = sym[j];
      p++;
    }
    i++;
  }
  for( i = 0; i < DEINTERLEAVE; i++ )
  {
    sym[i] = tmp[i];
  }

} // Deinterleave()

//------------------------------------------------------------------------

#define GRID6_SIZE  7
#define CDBM_SIZE   4

// used by qsort
  int
Double_Comp( const void *elem1, const void *elem2 )
{
  if( *(const double *)elem1 < *(const double *)elem2 )
    return( -1 );
  else if( *(const double *)elem1 > *(const double *)elem2 )
    return( 1 );
  else return( 0 );
} // Double_Comp()

//------------------------------------------------------------------------

/* Unpk_()
 *
 * Unpacks (I think!) the WSPR transmitted message
 */
  BOOLEAN
Unpk_(
    const signed char *message,
    char *hashtab,
    char *loctab,
    char *call_loc_pow,
    char *call,
    char *loc,
    char *pwr,
    char *callsign )
{
  int n1, n2, ndbm, ihash;
  BOOLEAN noprint = False;
  char grid[GRID_SIZE], cdbm[CDBM_SIZE];

  Unpack_50( message, &n1, &n2 );
  if( !Unpack_Call(n1, callsign) ) return( True );
  if( !Unpack_Grid(n2, grid) )     return( True );
  int ntype = ( n2 & 127 ) - 64;
  callsign[CALL_SIZE - 1] = '\0';
  grid[GRID_SIZE - 1]     = '\0';

  /*
   * Based on the value of ntype, decide whether this is a Type 1, 2, or
   * 3 message.
   * Type 1: 6 digit call, grid, power - ntype is positive and is a member
   * of the set {0,3,7,10,13,17,20...60}
   * Type 2: extended callsign, power - ntype is positive but not
   * a member of the set of allowed powers
   * Type 3: hash, 6 digit grid, power - ntype is negative.
   */

  if( (ntype >= 0) && (ntype <= 62) )
  {
    int nu = ntype % 10;
    if( (nu == 0) || (nu == 3) || (nu == 7) )
    {
      ndbm = ntype;
      memset( call_loc_pow, '\0', sizeof(char) * CALL_LOC_POW_SIZE );
      snprintf( cdbm, CDBM_SIZE, "%02d", ndbm );
      Strlcat( call_loc_pow, callsign, CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, " ", CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, grid, CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, " ", CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, cdbm, CALL_LOC_POW_SIZE );

      Strlcpy( call, callsign, CALL_SIZE );
      Strlcpy( loc, grid, LOC_SIZE );
      Strlcpy( pwr, cdbm, PWR_SIZE );

      ihash = (int)( Nhash(callsign, strlen(callsign), (uint32_t)146) );
      strcpy( hashtab + ihash * 13, callsign );
      strcpy( loctab  + ihash *  5, grid );
    }
    else
    {
      int nadd = nu;
      if( nu > 3 ) nadd = nu - 3;
      if( nu > 7 ) nadd = nu - 7;
      int n3 = n2 / 128 + 32768 * ( nadd - 1 );
      if( !Unpack_Pfx(n3, callsign) ) return( True );
      ndbm = ntype - nadd;
      memset( call_loc_pow, '\0', sizeof(char) * CALL_LOC_POW_SIZE );
      snprintf( cdbm, CDBM_SIZE, "%2d", ndbm );
      Strlcat( call_loc_pow, callsign, CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, " ", CALL_LOC_POW_SIZE );
      Strlcat( call_loc_pow, cdbm, CALL_LOC_POW_SIZE );

      Strlcpy( call, callsign, CALL_SIZE );
      Strlcpy( loc, "XXXX", LOC_SIZE );
      Strlcpy( pwr, cdbm, PWR_SIZE );

      nu = ndbm % 10;
      if( (nu == 0) || (nu == 3) || (nu == 7) )  // make sure power is OK
      {
        ihash = (int)( Nhash(callsign, strlen(callsign), (uint32_t)146) );
        strcpy( hashtab + ihash * 13, callsign );
      }
      else noprint = True;
    }
  }
  else if( ntype < 0 )
  {
    char grid6[GRID6_SIZE];
    ndbm = -( ntype + 1 );
    memset( grid6, '\0', sizeof(char) * GRID6_SIZE );
    size_t len = 6;
    strncat( grid6, callsign + len - 1, 1 );
    strncat( grid6, callsign, len - 1 );
    int nu = ndbm % 10;
    if( ((nu != 0) && (nu != 3) && (nu != 7)) ||
        !isalpha(grid6[0]) || !isalpha(grid6[1]) ||
        !isdigit(grid6[2]) || !isdigit(grid6[3]) )
    {
      // not testing 4'th and 5'th chars because of this case: <PA0SKT/2> JO33 40
      // grid is only 4 chars even though this is a hashed callsign...
      //         isalpha(grid6[4]) && isalpha(grid6[5]) ) )
      noprint = True;
    }

    ihash = ( n2 - ntype - 64 ) / 128;
    if( strncmp(hashtab + ihash * 13, "\0", 1) != 0 )
    {
      snprintf( callsign, CALL_SIZE, "<%s>", hashtab + ihash * 13 );
    }
    else
    {
      snprintf( callsign, CALL_SIZE, "%s", "<....>" );
    }

    memset( call_loc_pow, '\0', sizeof(char) * CALL_LOC_POW_SIZE );
    snprintf( cdbm, CDBM_SIZE, "%2d", ndbm );
    Strlcat( call_loc_pow, callsign, CALL_LOC_POW_SIZE );
    Strlcat( call_loc_pow, " ", CALL_LOC_POW_SIZE );
    Strlcat( call_loc_pow, grid6, CALL_LOC_POW_SIZE );
    Strlcat( call_loc_pow, " ", CALL_LOC_POW_SIZE );
    Strlcat( call_loc_pow, cdbm, CALL_LOC_POW_SIZE );

    Strlcpy( call, callsign, CALL_SIZE );
    Strlcpy( loc, grid6, LOC_SIZE );
    Strlcpy( pwr, cdbm, PWR_SIZE );

    // I don't know what to do with these... They show up as "A000AA" grids.
    if( ntype == -64 ) noprint = True;
  }

  return( noprint );
} // Unpk_()

//------------------------------------------------------------------------

