/*
 * This file is part of program wsprd, a detector/demodulator/decoder
 * for the Weak Signal Propagation Reporter (WSPR) mode.
 *
 * File name: wsprd.c
 *
 * Copyright 2001-2015, Joe Taylor, K1JT
 *
 * Much of the present code is based on work by Steven Franke, K9AN,
 * which in turn was based on earlier work by K1JT.
 *
 * Copyright 2014-2015, Steven Franke, K9AN
 *
 * Minor modifications

 * Copyright 2016, Guenael Jouchet, VA2GKA
 *
 * License: GNU GPL v3
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "wsprd.h"
#include "fano.h"
#include "mettab.h"
#include "shared.h"
#include "wsprd_utils.h"
#include "wsprsim_utils.h"
#include "../common/cfft.h"
#include "../common/utils.h"
#include <math.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

//------------------------------------------------------------------------

#define NBITS               81
#define NSYM                162
#define NSPERSYM            256
#define DF                  1.46484375         // 375.0 / 256.0
#define DT                  2.666666666667E-3  // 1.0 / 375.0
#define DF05                0.732421875        // DF * 0.5
#define DF15                2.197265625        // DF * 1.5
#define TWOPIDT             1.675516081915E-2  // 2 * M_PI * DT

static uint8_t pr3vector[NSYM] =
{
  1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 0,
  0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1,
  0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0, 1,
  1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1,
  0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0,
  0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1,
  0, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0,
  0, 0
};

//------------------------------------------------------------------------

/* mode = 0: no frequency or drift search. find best time lag.
 *        1: no time lag or drift search. find best frequency.
 *        2: no frequency or time lag search. calculate soft-decision
 *           symbols using passed frequency and shift.
 */
  void
Sync_And_Demodulate(
    const double *id,
    const double *qd,
    long    np,
    unsigned char *symbols,
    double *freq,
    int     ifmin,
    int     ifmax,
    double  fstep,
    int    *shift,
    int     lagmin,
    int     lagmax,
    int     lagstep,
    const double *drift,
    int     symfac,
    double *sync,
    int     mode )
{
  double
    i0[NSYM], q0[NSYM],
    i1[NSYM], q1[NSYM],
    i2[NSYM], q2[NSYM],
    i3[NSYM], q3[NSYM],
    c0[NSPERSYM], s0[NSPERSYM],
    c1[NSPERSYM], s1[NSPERSYM],
    c2[NSPERSYM], s2[NSPERSYM],
    c3[NSPERSYM], s3[NSPERSYM];
    double fsymb[NSYM];

    double fbest = 0.0, fsum = 0.0;

    int best_shift = 0;
    static double fplast = -10000.0;
    double syncmax = -1.0E30;

    if( mode == 0 )
    {
      ifmin = 0;
      ifmax = 0;
      fstep = 0.0;
    }
    else if( mode == 1 )
    {
      lagmin = *shift;
      lagmax = *shift;
    }
    else if( mode == 2 )
    {
      lagmin = *shift;
      lagmax = *shift;
      ifmin  = 0;
      ifmax  = 0;
    }

    for( int ifreq = ifmin; ifreq <= ifmax; ifreq++ )
    {
      double f0 = *freq + ifreq * fstep;
      for( int lag = lagmin; lag <= lagmax; lag = lag + lagstep )
      {
        double ss   = 0.0;
        double totp = 0.0;
        for( int i = 0; i < NSYM; i++ )
        {
          double fp = f0 + (*drift / 2.0) * ((double)i - NBITS) / NBITS;
          // only calculate sin/cos if necessary
          if( (i == 0) || (fp > fplast) || (fp < fplast) )
          {
            double dphi0  = TWOPIDT * ( fp - DF15 );
            double cdphi0 = cos( dphi0 );
            double sdphi0 = sin( dphi0 );

            double dphi1  = TWOPIDT * ( fp - DF05 );
            double cdphi1 = cos( dphi1 );
            double sdphi1 = sin( dphi1 );

            double dphi2  = TWOPIDT * ( fp + DF05 );
            double cdphi2 = cos( dphi2 );
            double sdphi2 = sin( dphi2 );

            double dphi3  = TWOPIDT * ( fp + DF15 );
            double cdphi3 = cos( dphi3 );
            double sdphi3 = sin( dphi3 );

            c0[0] = 1; s0[0] = 0;
            c1[0] = 1; s1[0] = 0;
            c2[0] = 1; s2[0] = 0;
            c3[0] = 1; s3[0] = 0;

            for( int j = 1; j < NSPERSYM; j++ )
            {
              c0[j] = c0[j - 1] * cdphi0 - s0[j - 1] * sdphi0;
              s0[j] = c0[j - 1] * sdphi0 + s0[j - 1] * cdphi0;
              c1[j] = c1[j - 1] * cdphi1 - s1[j - 1] * sdphi1;
              s1[j] = c1[j - 1] * sdphi1 + s1[j - 1] * cdphi1;
              c2[j] = c2[j - 1] * cdphi2 - s2[j - 1] * sdphi2;
              s2[j] = c2[j - 1] * sdphi2 + s2[j - 1] * cdphi2;
              c3[j] = c3[j - 1] * cdphi3 - s3[j - 1] * sdphi3;
              s3[j] = c3[j - 1] * sdphi3 + s3[j - 1] * cdphi3;
            }
            fplast = fp;
          }

          i0[i] = 0.0; q0[i] = 0.0;
          i1[i] = 0.0; q1[i] = 0.0;
          i2[i] = 0.0; q2[i] = 0.0;
          i3[i] = 0.0; q3[i] = 0.0;

          for( int j = 0; j < NSPERSYM; j++ )
          {
            int k = lag + i * NSPERSYM + j;
            if( (k > 0) && (k < np) )
            {
              i0[i] = i0[i] + id[k] * c0[j] + qd[k] * s0[j];
              q0[i] = q0[i] - id[k] * s0[j] + qd[k] * c0[j];
              i1[i] = i1[i] + id[k] * c1[j] + qd[k] * s1[j];
              q1[i] = q1[i] - id[k] * s1[j] + qd[k] * c1[j];
              i2[i] = i2[i] + id[k] * c2[j] + qd[k] * s2[j];
              q2[i] = q2[i] - id[k] * s2[j] + qd[k] * c2[j];
              i3[i] = i3[i] + id[k] * c3[j] + qd[k] * s3[j];
              q3[i] = q3[i] - id[k] * s3[j] + qd[k] * c3[j];
            }
          }

          double p0 = sqrt( i0[i] * i0[i] + q0[i] * q0[i] );
          double p1 = sqrt( i1[i] * i1[i] + q1[i] * q1[i] );
          double p2 = sqrt( i2[i] * i2[i] + q2[i] * q2[i] );
          double p3 = sqrt( i3[i] * i3[i] + q3[i] * q3[i] );

          totp = totp + p0 + p1 + p2 + p3;
          double cmet = ( p1 + p3 ) - ( p0 + p2 );
          ss = ( pr3vector[i] == 1 ) ? ss + cmet : ss - cmet;
          if( mode == 2 )  // Compute soft symbols
          {
            if( pr3vector[i] == 1 )
            {
              fsymb[i] = p3 - p1;
            }
            else
            {
              fsymb[i] = p2 - p0;
            }
          }
        }
        ss = ss / totp;
        if( ss > syncmax )  // Save best parameters
        {
          syncmax    = ss;
          best_shift = lag;
          fbest      = f0;
        }
      }  // lag loop
    }  // freq loop

    if( mode <= 1 )  // Send best params back to caller
    {
      *sync  = syncmax;
      *shift = best_shift;
      *freq  = fbest;
      return;
    }

    if( mode == 2 )
    {
      double f2sum = 0.0;
      *sync = syncmax;
      for( int i = 0; i < NSYM; i++ )  // Normalize the soft symbols
      {
        fsum  += fsymb[i] / NSYM;
        f2sum += fsymb[i] * fsymb[i] / NSYM;
      }

      double fac = sqrt( f2sum - fsum * fsum );
      for( int i = 0; i < NSYM; i++ )
      {
        fsymb[i] = symfac * fsymb[i] / fac;
        if( fsymb[i] > 127 )
          fsymb[i] = 127.0;
        else if( fsymb[i] < -128 )
          fsymb[i] = -128.0;
        symbols[i] = (unsigned char)( fsymb[i] + 128.0 );
      }
      return;
    }

    return;
} // Sync_And_Demodulate()

//------------------------------------------------------------------------

// symbol-by-symbol signal subtraction
  void
Subtract_Signal(
    double *id,
    double *qd,
    long  np,
    double f0,
    int   shift,
    double drift,
    const unsigned char *channel_symbols )
{
  double c0[NSPERSYM], s0[NSPERSYM];

  for( int i = 0; i < NSYM; i++ )
  {
    double fp    = f0 + ( (double)drift / 2.0 ) * ( (double)i - NBITS ) / NBITS;
    double dphi  = TWOPIDT * ( fp + ((double)channel_symbols[i] - 1.5) * DF );
    double cdphi = cos( dphi );
    double sdphi = sin( dphi );

    c0[0] = 1;
    s0[0] = 0;

    for( int j = 1; j < NSPERSYM; j++ )
    {
      c0[j] = c0[j - 1] * cdphi - s0[j - 1] * sdphi;
      s0[j] = c0[j - 1] * sdphi + s0[j - 1] * cdphi;
    }

    double i0 = 0.0;
    double q0 = 0.0;

    for( int j = 0; j < NSPERSYM; j++ )
    {
      int k = shift + i * NSPERSYM + j;
      if( (k > 0) & (k < np) )
      {
        i0 = i0 + id[k] * c0[j] + qd[k] * s0[j];
        q0 = q0 - id[k] * s0[j] + qd[k] * c0[j];
      }
    }

    // subtract the signal here.
    i0 = i0 / NSPERSYM;  // will be wrong for partial symbols at the edges...
    q0 = q0 / NSPERSYM;

    for( int j = 0; j < NSPERSYM; j++ )
    {
      int k = shift + i * NSPERSYM + j;
      if( (k > 0) & (k < np) )
      {
        id[k] = id[k] - (i0 * c0[j] - q0 * s0[j]);
        qd[k] = qd[k] - (q0 * c0[j] + i0 * s0[j]);
      }
    }
  }

  return;
} // Subtract_Signal()

//------------------------------------------------------------------------

#define NFILT  360  // NFILT must be even number.

// Subtract the coherent component of a signal
  void
Subtract_Signal2(
    double *id,
    double *qd,
    long    np,
    double  f0,
    int     shift,
    double  drift,
    const unsigned char *channel_symbols )
{
  double phi = 0.0;

  double refi[SIGNAL_SAMPLES] = {0}, refq[SIGNAL_SAMPLES] = {0},
         ci[SIGNAL_SAMPLES]   = {0}, cq[SIGNAL_SAMPLES]   = {0},
         cfi[SIGNAL_SAMPLES]  = {0}, cfq[SIGNAL_SAMPLES]  = {0};

  /******************************************************************************
    Measured signal:                    s(t)=a(t)*exp( j*theta(t) )
    Reference is:                       r(t) = exp( j*phi(t) )
    Complex amplitude is estimated as:  c(t)=LPF[s(t)*conjugate(r(t))]
    so c(t) has phase angle theta-phi
    Multiply r(t) by c(t) and subtract from s(t), i.e. s'(t)=s(t)-c(t)r(t)
   *******************************************************************************/

  /* create reference wspr signal vector, centered on f0. */
  for( int i = 0; i < NSYM; i++ )
  {
    double cs = (double)channel_symbols[i];

    double dphi =
      TWOPIDT * ( f0 + (drift / 2.0) * ((double)i - NSYM / 2.0) / (NSYM / 2.0) + (cs - 1.5) * DF );

    for( int j = 0; j < NSPERSYM; j++ )
    {
      int ii   = NSPERSYM * i + j;
      refi[ii] = cos( phi );  // cannot precompute sin/cos because dphi is changing
      refq[ii] = sin( phi );
      phi      = phi + dphi;
    }
  }

  double w[NFILT], norm = 0, partialsum[NFILT];

  /* lowpass filter and remove startup transient */
  for( int i = 0; i < NFILT; i++ )
  {
    partialsum[i] = 0.0;
  }
  for( int i = 0; i < NFILT; i++ )
  {
    w[i] = sin( M_PI * (double)i / (NFILT - 1) );
    norm = norm + w[i];
  }
  for( int i = 0; i < NFILT; i++ )
  {
    w[i] = w[i] / norm;
  }
  for( int i = 1; i < NFILT; i++ )
  {
    partialsum[i] = partialsum[i - 1] + w[i];
  }

  /* s(t) * conjugate(r(t))
   * beginning of first symbol in reference signal is at i=0
   * beginning of first symbol in received data is at shift value.
   * filter transient lasts NFILT samples leave NFILT zeros as
   * a pad at the beginning of the uNFILTered reference signal */
  for( int i = 0; i < NSYM * NSPERSYM; i++ )
  {
    int k = shift + i;
    if( (k > 0) && (k < np) )
    {
      ci[i + NFILT] = id[k] * refi[i] + qd[k] * refq[i];
      cq[i + NFILT] = qd[k] * refi[i] - id[k] * refq[i];
    }
  }

  // LPF
  for( int i = NFILT / 2; i < SIGNAL_SAMPLES - NFILT / 2; i++ )
  {
    cfi[i] = 0.0;
    cfq[i] = 0.0;
    for( int j = 0; j < NFILT; j++ )
    {
      cfi[i] = cfi[i] + w[j] * ci[i - NFILT / 2 + j];
      cfq[i] = cfq[i] + w[j] * cq[i - NFILT / 2 + j];
    }
  }

  /* subtract c(t)*r(t) here
   * (ci+j*cq)(refi+j*refq)=(ci*refi-cq*refq)+j(ci*refq+cq*refi)
   * beginning of first symbol in reference signal is at i=NFILT
   * beginning of first symbol in received data is at shift value. */
  for( int i = 0; i < NSYM * NSPERSYM; i++ )
  {
    if( i < NFILT / 2 )  // Take care of the end effect (LPF step response) here
    {
      norm = partialsum[NFILT / 2 + i];
    }
    else if( i > (NSYM * NSPERSYM - 1 - NFILT / 2) )
    {
      norm = partialsum[NFILT / 2 + NSYM * NSPERSYM - 1 - i];
    }
    else
    {
      norm = 1.0;
    }

    int k = shift + i;
    if( (k > 0) && (k < np) )
    {
      int j = i + NFILT;
      id[k] = id[k] - ( cfi[j] * refi[i] - cfq[j] * refq[i] ) / norm;
      qd[k] = qd[k] - ( cfi[j] * refq[i] + cfq[j] * refi[i] ) / norm;
    }
  }

  return;
} // Subtract_Signal2()

//------------------------------------------------------------------------

#define ALL_RESULTS     100
#define CANDIDATES      200
#define SMSPEC_SIZE     411
#define HASH_SIZE       32768

/* FFT buffer (512 bins) and signal averages */
static double *fftin_i = NULL, *fftin_q = NULL;
static double **ps = NULL;

/* Wspr_Decode()
 *
 * Decodes WPSR signals by analysing processed
 * (decimated) IQ samples from the Receiver
 */
  int
Wspr_Decode(
    double *idat,
    double *qdat,
    int    samples,
    decoder_results_t *decodes,
    int16_t *n_results )
{
  /* Parameters used for performance-tuning */
  double minsync1  = 0.10;                    // First sync limit
  double minsync2  = 0.12;                    // Second sync limit
  int    iifac     = 3;                       // Step size in final DT peakup
  int    symfac    = 50;                      // Soft-symbol normalizing factor
  int    maxdrift  = 4;                       // Maximum (+/-) drift
  double minrms    = 52.0 * (symfac / 64.0);  // Final test for plausible decoding
  int    delta     = 60;                      // Fano threshold step
  int    maxcycles = 10000;                   // Fano timeout limit
  double fmin      = -110.0;
  double fmax      =  110.0;

  /* Search live parameters */
  double fstep;
  int    lagmin;
  int    lagmax;
  int    lagstep;
  int    ifmin;
  int    ifmax;

  /* Decoder flags */
  int worth_a_try;
  int16_t uniques = 0;

  /* CPU usage stats */
  uint32_t metric, cycles = 0, maxnp;

  /* Candidates properties */
  struct cand candidates[CANDIDATES];

  /* Decoded candidate */
  uint8_t symbols[NBITS * 2] = {0};
  uint8_t decdata[(NBITS + 7) / 8] = {0};
  int8_t  message[12] = {0};

  /* Results */
  char   callsign[CALL_SIZE] = {'\0'};
  char   call_loc_pow[CALL_LOC_POW_SIZE] = {'\0'};
  char   call[CALL_SIZE] = {'\0'};
  char   loc[LOC_SIZE]   = {'\0'};
  char   pwr[PWR_SIZE]   = {'\0'};
  char   allcalls[ALL_RESULTS][CALL_SIZE] = {{'\0'}};
  double allfreqs[ALL_RESULTS] = {0};


  // Decoder is running
  rx_state.decoder_flag = True;

  /* Setup/Load hash tables */
  FILE *fhash;
  int   nh;
  char  hashtab[HASH_SIZE * CALL_SIZE] = { 0 };
  char   loctab[HASH_SIZE * LOC_SIZE]  = { 0 };

  if( wspr_rc.decoder_options.use_hashtable )
  {
    char hash_file[FILE_NAME_SIZE];
    Strlcpy( hash_file, wspr_rc.wspr_dir, FILE_NAME_SIZE );
    Strlcat( hash_file, "hashtable.txt",  FILE_NAME_SIZE );
    fhash = fopen( hash_file, "r+" );
    if( fhash != NULL )
    {
      char line[80];
      while( fgets(line, sizeof(line), fhash) != NULL )
      {
        char hcall[CALL_SIZE], hgrid[LOC_SIZE];
        hgrid[0] = '\0';
        sscanf( line, "%8d %12s %4s", &nh, hcall, hgrid );
        strcpy( hashtab + nh * CALL_SIZE, hcall );
        if( strlen(hgrid) > 0) strcpy(loctab + nh * LOC_SIZE, hgrid );
      }
    }
    else
    {
      fhash = fopen( hash_file, "w+" );
    }

    if( fhash != NULL ) Close_File( &fhash );
  } // if( options.usehashtable )

  // Allocate FFT buffers
  if( (fftin_i == NULL) || (fftin_q == NULL)  )
  {
    Mem_Alloc( (void **)&fftin_i, sizeof(double) * FFT_IO_SIZE );
    Mem_Alloc( (void **)&fftin_q, sizeof(double) * FFT_IO_SIZE );
  }

  /* Hann function */
  double hann[FFT_IO_SIZE];
  double dPhi = M_PI / FFT_IO_SIZE;
  for( int i = 0; i < FFT_IO_SIZE; i++ )
  {
    hann[i] = sin( dPhi * i );
  }

  /* FFT output alloc */
  const int blocks = (int)( 4 * floor(samples / FFT_IO_SIZE) ) - 1;
  if( ps == NULL )
  {
    Mem_Alloc( (void **)&ps, FFT_IO_SIZE * sizeof(double *) );
    for( int i = 0; i < FFT_IO_SIZE; i++ )
    {
      ps[i] = NULL;
      Mem_Alloc( (void **)&ps[i], sizeof(double) * (size_t)blocks );
    }
  }

  /* Main loop starts here */
  for( int ipass = 0; ipass < wspr_rc.decoder_options.npasses; ipass++ )
  {
    if( rx_state.exit_flag )
    {
      rx_state.decoder_flag = False;
      return( 0 );
    }

    if( ipass == 1 && uniques == 0 )
    {
      break;
    }

    if( ipass < 2 )
    {
      maxdrift = 4;
      minsync2 = 0.12;
    }

    if( ipass == 2 )
    {
      maxdrift = 0;    // no drift for smaller frequency estimator variance
      minsync2 = 0.10;
    }

    /* Compute FFT over 2 symbols, stepped by half symbols */
    for( int i = 0; i < blocks; i++ )
    {
      /* Load samples */
      for( int j = 0; j < FFT_IO_SIZE; j++ )
      {
        int k = i * 128 + j;
        fftin_i[j] = idat[k] * hann[j];
        fftin_q[j] = qdat[k] * hann[j];
      }

      cFFT( FFT_IO_SIZE, FFT_ORDER, fftin_i, fftin_q );

      //double max = 0;
      //int J = 0;
      /* Recover frequencies */
      for( int j = 0; j < FFT_IO_SIZE; j++ )
      {
        int k = j + 256;
        if( k > FFT_IO_SIZE - 1 ) k = k - FFT_IO_SIZE;
        ps[j][i] = fftin_i[k] * fftin_i[k] + fftin_q[k] * fftin_q[k];
        //if( max < ps[j][i] )
        // {
        //   max = ps[j][i];
        //   J = j;
        // }
      }
      //printf( "%12.1f  %3d\n", max, J );
    }

    // Compute average spectrum
    double psavg[FFT_IO_SIZE] = { 0.0 };
    for( int i = 0; i < blocks; i++ )
    {
      for( int j = 0; j < FFT_IO_SIZE; j++ )
      {
        psavg[j] += ps[j][i];
      }
    }

    // Already restricted by previous FIR
    // Smooth with 7-point window and limit spectrum to +/-150 Hz
    const int32_t window[7] = { 1, 1, 1, 1, 1, 1, 1 };
    double smspec[SMSPEC_SIZE];
    for( int i = 0; i < SMSPEC_SIZE; i++ )
    {
      smspec[i] = 0.0;
      for( int j = -3; j <= 3; j++ )
      {
        int k = 256 - 205 + i + j;
        smspec[i] += window[j + 3] * psavg[k];
      }
    }

    // Sort spectrum values, then pick off noise level as a percentile
    double tmpsort[SMSPEC_SIZE];
    for( int j = 0; j < SMSPEC_SIZE; j++ )
    {
      tmpsort[j] = smspec[j];
    }
    qsort( tmpsort, SMSPEC_SIZE, sizeof(double), Double_Comp );

    // Noise level of spectrum is estimated as 123/411= 30'th percentile
    double noise_level = tmpsort[122];

    /* Renormalize spectrum so that (large) peaks represent an estimate of snr.
     * We know from experience that threshold snr is near -7dB in wspr bandwidth,
     * corresponding to -7-26.3=-33.3dB in 2500 Hz bandwidth.
     * The corresponding threshold is -42.3 dB in 2500 Hz bandwidth for WSPR-15. */

    double min_snr = pow( 10.0, -8.0 / 10.0 );  // this is min snr in wspr bw
    double snr_scaling_factor = 26.3;
    for( int j = 0; j < SMSPEC_SIZE; j++ )
    {
      smspec[j] = smspec[j] / noise_level - 1.0;
      if( smspec[j] < min_snr ) smspec[j] = 0.1 * min_snr;
    }

    // Find all local maxima in smoothed spectrum.
    for( int i = 0; i < CANDIDATES; i++ )
    {
      candidates[i].freq  = 0.0;
      candidates[i].snr   = 0.0;
      candidates[i].drift = 0.0;
      candidates[i].shift = 0;
      candidates[i].sync  = 0.0;
    }

    int npk = 0;
    for( int j = 1; j < SMSPEC_SIZE - 1; j++ )
    {
      unsigned char candidate =
        ( smspec[j] > smspec[j - 1] ) &&
        ( smspec[j] > smspec[j + 1] ) &&
        ( npk < CANDIDATES );
      if( candidate )
      {
        candidates[npk].freq = ( j - 205 ) * ( DF / 2.0 );
        candidates[npk].snr  = 10.0 * log10( smspec[j] ) - snr_scaling_factor;
        npk++;
      }
    }

    // Don't waste time on signals outside of the range [fmin,fmax].
    int i = 0;
    for( int j = 0; j < npk; j++ )
    {
      if( (candidates[j].freq >= fmin) && (candidates[j].freq <= fmax) )
      {
        candidates[i] = candidates[j];
        i++;
      }
    }
    npk = i;

    // bubble sort on snr, bringing freq along for the ride
    struct cand tmp;
    for( int pass = 1; pass <= npk - 1; pass++ )
    {
      for( int k = 0; k < npk - pass; k++ )
      {
        if( candidates[k].snr < candidates[k + 1].snr )
        {
          tmp = candidates[k];
          candidates[k] = candidates[k + 1];
          candidates[k + 1] = tmp;
        }
      }
    }

    /* Make coarse estimates of shift (DT), freq, and drift
     * Look for time offsets up to +/- 8 symbols (about +/- 5.4 s) relative
     * to nominal start time, which is 2 seconds into the file
     * Calculates shift relative to the beginning of the file
     * Negative shifts mean that signal started before start of file
     * The program prints DT = shift-2 s
     * Shifts that cause sync vector to fall off of either end of the data
     * vector are accommodated by "partial decoding", such that missing
     * symbols produce a soft-decision symbol value of 128
     * The frequency drift model is linear, deviation of +/- drift/2 over the
     * span of 162 symbols, with deviation equal to 0 at the center of the
     * signal vector.
     */
    for( int j = 0; j < npk; j++ )  // For each candidate...
    {
      double sync = 0, sync_max = -1.0E30;
      int if0 = (int)( candidates[j].freq / (DF / 2.0) ) + NSPERSYM;

      for( int ifr = if0 - 1; ifr <= if0 + 1; ifr++ )  // Freq search
      {
        for( int k0 = -10; k0 < 22; k0++ )  // Time search
        {
          for( int idrift = -maxdrift; idrift <= maxdrift; idrift++ )  // Drift search
          {
            double ss  = 0.0;
            double pow = 0.0;
            for( int k = 0; k < NSYM; k++ )  // Sum over symbols
            {
              int ifd = (int)( (double)ifr + ((double)k - NBITS) / NBITS * (double)idrift / DF );
              int kindex = k0 + 2 * k;
              if( kindex < blocks )
              {
                double p0 = sqrt( ps[ifd - 3][kindex] );
                // FIXME something here causes segfault on quitting
                // wspr during the decoding period (after even minute)
                double p1 = sqrt( ps[ifd - 1][kindex] );
                double p2 = sqrt( ps[ifd + 1][kindex] );
                double p3 = sqrt( ps[ifd + 3][kindex] );

                ss = ss + ( 2 * pr3vector[k] - 1) * ((p1 + p3) - (p0 + p2) );
                pow = pow + p0 + p1 + p2 + p3;
                sync = ss / pow;
              }
            }

            if( sync > sync_max )  // Save coarse parameters
            {
              sync_max = sync;
              candidates[j].shift = 128 * ( k0 + 1 );
              candidates[j].drift = idrift;
              candidates[j].freq  = ( ifr - NSPERSYM ) * ( DF / 2.0 );
              candidates[j].sync  = sync;
            }
          }
        }
      }
    }

    /* Refine the estimates of freq, shift using sync as a metric.
     * Sync is calculated such that it is a double taking values in the range
     * [0.0,1.0].
     *
     * Function sync_and_demodulate has three modes of operation
     * mode is the last argument:
     *
     * 0 = no frequency or drift search. find best time lag.
     * 1 = no time lag or drift search. find best frequency.
     * 2 = no frequency or time lag search. Calculate soft-decision
     * symbols using passed frequency and shift.
     *
     * NB: best possibility for OpenMP may be here: several worker threads
     * could each work on one candidate at a time. */

    for( int j = 0; j < npk; j++ )
    {
      memset( callsign,     '\0', sizeof(char) * CALL_SIZE );
      memset( call_loc_pow, '\0', sizeof(char) * CALL_LOC_POW_SIZE );
      memset( call,         '\0', sizeof(char) * CALL_SIZE );
      memset( loc,          '\0', sizeof(char) * LOC_SIZE );
      memset( pwr,          '\0', sizeof(char) * PWR_SIZE );

      double freq  = candidates[j].freq;
      double drift = candidates[j].drift;
      double sync  = candidates[j].sync;
      int   shift  = candidates[j].shift;

      // Search for best sync lag (mode 0)
      fstep = 0.0;
      ifmin = 0;
      ifmax = 0;
      lagmin = shift - 128;
      lagmax = shift + 128;
      lagstep = 8;
      if( wspr_rc.decoder_options.quick_mode ) lagstep = 16;
      Sync_And_Demodulate(
          idat, qdat, samples, symbols, &freq,
          ifmin, ifmax, fstep, &shift, lagmin,
          lagmax, lagstep, &drift, symfac, &sync, 0 );

      // Search for frequency peak (mode 1)
      fstep = 0.1;
      ifmin = -2;
      ifmax =  2;
      Sync_And_Demodulate(
          idat, qdat, samples, symbols, &freq,
          ifmin, ifmax, fstep, &shift, lagmin,
          lagmax, lagstep, &drift, symfac, &sync, 1 );

      candidates[j].freq  = freq;
      candidates[j].shift = shift;
      candidates[j].drift = drift;
      candidates[j].sync  = sync;

      if( sync > minsync1 )
      {
        worth_a_try = 1;
      }
      else
      {
        worth_a_try = 0;
      }

      int idt = 0, ii = 0;
      int not_decoded = 1;
      while( worth_a_try && (not_decoded && idt <= (128 / iifac)) )
      {
        ii = ( idt + 1 ) / 2;
        if( idt % 2 == 1 ) ii = -ii;
        ii = iifac * ii;
        int jiggered_shift = shift + ii;

        // Use mode 2 to get soft-decision symbols
        Sync_And_Demodulate(
            idat, qdat, samples, symbols, &freq,
            ifmin, ifmax, fstep, &jiggered_shift,
            lagmin, lagmax, lagstep, &drift, symfac, &sync, 2 );

        double sq = 0.0;
        for( i = 0; i < NSYM; i++ )
        {
          double y = (double)symbols[i] - 128.0;
          sq += y * y;
        }
        double rms = sqrt( sq / NSYM );

        if( (sync > minsync2) && (rms > minrms) )
        {
          Deinterleave( symbols );
          not_decoded = Fano(
              &metric, &cycles, &maxnp, decdata, symbols,
              NBITS, mettab, delta, (unsigned int)maxcycles );
        }
        idt++;
        if( wspr_rc.decoder_options.quick_mode ) break;
      }

      if( worth_a_try && !not_decoded )
      {
        for( i = 0; i < 11; i++ )
        {
          if( decdata[i] > 127 )
          {
            message[i] = (int8_t)( decdata[i] - 256 );
          }
          else
          {
            message[i] = (int8_t)decdata[i];
          }
        }

        // Unpack the decoded message, update the hashtable, apply
        // sanity checks on grid and power, and return
        // call_loc_pow string and also callsign (for de-duping).
        BOOLEAN noprint =
          Unpk_( message, hashtab, loctab, call_loc_pow, call, loc, pwr, callsign );

        if( wspr_rc.decoder_options.subtraction && (ipass == 0) && !noprint )
        {
          unsigned char channel_symbols[NSYM];

          if( Get_Wspr_Channel_Symbols(call_loc_pow, hashtab, loctab, channel_symbols) )
          {
            Subtract_Signal2( idat, qdat, samples, freq, shift, drift, channel_symbols );
          }
          else
          {
            break;
          }
        }

        // Avoid this incorrect pattern
        if( strcmp(loc, "A000AA") == 0 ) break;

        // Remove dupes (same callsign and freq within 3 Hz)
        int32_t dupe = 0;
        for( i = 0; i < uniques; i++ )
        {
          if( !strcmp(callsign, allcalls[i]) && (fabs(freq - allfreqs[i]) < 3.0) )
            dupe = 1;
        }

        if( !dupe && !noprint )
        {
          strcpy( allcalls[uniques], callsign );
          allfreqs[uniques] = freq;
          uniques++;

          int16_t idx = uniques - 1;
          double dialfreq = (double)wspr_rc.decoder_options.center_freq / 1.0E06;
          double freq_print = dialfreq + freq / 1.0E06;

          decodes[idx].sync   = candidates[j].sync;
          decodes[idx].snr    = candidates[j].snr;
          decodes[idx].dt     = shift * DT - 2.0;
          decodes[idx].freq   = freq_print;
          decodes[idx].drift  = drift;
          decodes[idx].cycles = (int)cycles;
          decodes[idx].jitter = ii;
          Strlcpy( decodes[idx].message, call_loc_pow, CALL_LOC_POW_SIZE );
          Strlcpy( decodes[idx].call, call, CALL_SIZE );
          Strlcpy( decodes[idx].loc,  loc,  LOC_SIZE );
          Strlcpy( decodes[idx].pwr,  pwr,  PWR_SIZE );
        }
      }
    }

  } //for( int ipass = 0; ipass < wspr_rc.decoder_options.npasses; ipass++ )

  if( rx_state.exit_flag )
  {
    rx_state.decoder_flag = False;
    return( 0 );
  }

  /* Sort the result */
  decoder_results_t temp;
  for( int j = 1; j < uniques; j++ )
  {
    for( int k = 0; k < uniques - j; k++ )
    {
      if( decodes[k].snr < decodes[k + 1].snr )
      {
        temp = decodes[k];
        decodes[k] = decodes[k + 1];
        decodes[k + 1] = temp;
      }
    }
  }

  /* Return number of spots to the calling fct */
  *n_results = uniques;

  if( wspr_rc.decoder_options.use_hashtable )
  {
    char hash_file[FILE_NAME_SIZE];
    Strlcpy( hash_file, wspr_rc.wspr_dir, FILE_NAME_SIZE );
    Strlcat( hash_file, "hashtable.txt",  FILE_NAME_SIZE );
    fhash = fopen( hash_file, "w" );
    if( fhash == NULL )
    {
      rx_state.decoder_flag = False;
      return( -1 );
    }

    for( int i = 0; i < HASH_SIZE; i++ )
    {
      if( strncmp(hashtab + i * 13, "\0", 1) != 0 )
      {
        fprintf( fhash, "%5d %s %s\n", i, hashtab + i * 13, loctab + i * 5 );
      }
    }
    Close_File( &fhash );
  }

  rx_state.decoder_flag = False;
  return( 0 );
} // Wspr_Decode()

//------------------------------------------------------------------------

// Frees resources used in Wspr_Decode()
  void
Free_Wspr_Decoder( void )
{
  if( fftin_i != NULL )
    Mem_Free( (void **)&fftin_i );
  if( fftin_q != NULL )
    Mem_Free( (void **)&fftin_q );

  if( ps != NULL )
  {
    for( int i = 0; i < FFT_IO_SIZE; i++ )
      Mem_Free( (void **)&ps[i] );
    Mem_Free( (void **)&ps );
  }
} // Free_Wspr_Decoder()

//------------------------------------------------------------------------

