/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

#include "whisper.h"
#include "shared.h"
#include "utils.h"
#include "wsprd.h"
//#include "wsprsim_utils.h"
#include "xplanet.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/display.h"
#include "../Hermes2/sound.h"
#include <curl/curl.h>
#include <math.h>
#include <gtk/gtk.h>
#include <sys/time.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdio.h>

//------------------------------------------------------------------

#define FIR_TAPS            33

static void *Decoder( void *arg );

//------------------------------------------------------------------

/* Wspr_Run()
 *
 * Receives IQ samples from the Receiver and processes
 * using a CIC decimator and FIR compensation filter
 * before supplying the decimated samples to the Decoder
 */
  static void *
Wspr_Run( void *data )
{
  // Wait for sample buffers full
  static BOOLEAN wait;

  // Digimode buffer index
  static uint32_t digi_buf_idx = 0;

  /* CIC buffers/vars */
  static double Iy1, Qy1, Iy2, Qy2, It1y, Qt1y, It2y, Qt2y;
  static double Ix1  = 0, Ix2  = 0, Qx1  = 0, Qx2  = 0;
  static double It1z = 0, Qt1z = 0, It2z = 0, Qt2z = 0;

  static uint32_t decimationIndex = 0;

  // FIR compensation filter coefs
  const static double zCoef[33] =
  {
     3.93027027027E-7,  -2.017250293772E-6,  6.973384253819E-6, -2.185493537015E-5,
     5.917508813161E-5, -1.502062279671E-4,  3.477978848414E-4, -7.558695652174E-4,
     1.522761457109E-3, -2.941451233843E-3,  5.523425381904E-3, -1.031915393655E-2,
     1.957620446533E-2, -3.904858989424E-2,  8.739482961222E-2, -2.516539365452E-1,
     1.381010575793,
    -2.516539365452E-1,  8.739482961222E-2, -3.904858989424E-2,  1.957620446533E-2,
    -1.031915393655E-2,  5.523425381904E-3, -2.941451233843E-3,  1.522761457109E-3,
    -7.558695652174E-4,  3.477978848414E-4, -1.502062279671E-4,  5.917508813161E-5,
    -2.185493537015E-5,  6.973384253819E-6, -2.017250293772E-6,  3.93027027027E-7
  };

  /* FIR compensation filter buffers */
  static double firI[FIR_TAPS] = { 0.0 }, firQ[FIR_TAPS] = { 0.0 };

  // Used in 2min synchronization
  static uint32_t uwait_prev = 120000000;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  /* Prepare a low priority param for the decoder thread */
  struct sched_param param;
  pthread_attr_init( &decState.tattr );
  pthread_attr_setschedpolicy( &decState.tattr, SCHED_RR );
  pthread_attr_getschedparam( &decState.tattr, &param );
  param.sched_priority = 90;  // = sched_get_priority_min();
  pthread_attr_setschedparam( &decState.tattr, &param );

  // Keep looping till reception is stopped
  Flag[GUEST_RECEIVING] = True;
  while( !Flag[GUEST_QUIT] && !rx_state.exit_flag )
  {
    // Reset if signaled by data as a flag
    if( *((BOOLEAN *)data) == True )
    {
      Ix1 = 0; Ix2  = 0; Qx1  = 0; Qx2 = 0;
      Iy1 = 0; It1y = 0; It1z = 0; Qy1 = 0; Qt1y = 0; Qt1z = 0;
      Iy2 = 0; It2y = 0; It2z = 0; Qy2 = 0; Qt2y = 0; Qt2z = 0;

      digi_buf_idx    = 0;
      decimationIndex = 0;
      uwait_prev = 120000000; // 2min in uSec

      digi_buf_cnt = 0;
      rx_state.bufferIndex   = 0;
      rx_state.iqIndex[0]    = 0;
      rx_state.iqIndex[1]    = 0;
      wait = True;

      *((BOOLEAN *)data) = False;
    }

    // Wait for digimode buffer to fill with audio samples from demodulator.
    if( wait && !rx_state.exit_flag )
    {
      sem_wait( &digimode_semaphore );
      wait = False;
    }

    // CIC decimator (N = 2)
    for( digi_buf_idx = 0; digi_buf_idx < DIGIMODE_BUFFER_SIZE; digi_buf_idx++ )
    {
      // Integrator stages (N = 2). Ix negated for right sense of delta F
      Ix1 += -demod_id_cpy[digi_buf_idx] / 0xFFFF;  // Scaled to avoid huge values
      Qx1 += demod_qd_cpy[digi_buf_idx]  / 0xFFFF;
      Ix2 += Ix1;
      Qx2 += Qx1;

      /* Decimation stage */
      decimationIndex++;
      if( decimationIndex <= DOWNSAMPLING ) continue;
      decimationIndex = 0;

      /* 1st Comb */
      Iy1  = Ix2 - It1z;
      It1z = It1y;
      It1y = Ix2;
      Qy1  = Qx2 - Qt1z;
      Qt1z = Qt1y;
      Qt1y = Qx2;

      /* 2nd Comb */
      Iy2  = Iy1 - It2z;
      It2z = It2y;
      It2y = Iy1;
      Qy2  = Qy1 - Qt2z;
      Qt2z = Qt2y;
      Qt2y = Qy1;

      // FIR compensation filter
      double Isum = 0.0, Qsum = 0.0;
      for( uint32_t j = 0; j < FIR_TAPS; j++ )
      {
        Isum += firI[j] * zCoef[j];
        Qsum += firQ[j] * zCoef[j];

        if( j < FIR_TAPS - 1 )
        {
          firI[j] = firI[j + 1];
          firQ[j] = firQ[j + 1];
        }
      }

      firI[FIR_TAPS - 1] = Iy2;
      firQ[FIR_TAPS - 1] = Qy2;
      Isum += firI[FIR_TAPS - 1] * zCoef[FIR_TAPS - 1];
      Qsum += firQ[FIR_TAPS - 1] * zCoef[FIR_TAPS - 1];

      /* Save the result in the buffer */
      uint8_t idx = rx_state.bufferIndex;
      if( rx_state.iqIndex[idx] < SAMPLES_BUF_SIZE )
      {
        rx_state.iSamples[idx][rx_state.iqIndex[idx]] = Isum;
        rx_state.qSamples[idx][rx_state.iqIndex[idx]] = Qsum;
        rx_state.iqIndex[idx]++;
      }

    } // for( digi_buf_idx = 0; digi_buf_idx < DIGIMODE_BUFFER_SIZE; digi_buf_idx++ )

    // Increment current audio buffer output index
    digi_buf_cnt++;
    digi_buf_cnt %= NUM_DIGIMODE_BUFFERS;

    /* If sync between ring buffer input and output indices is lost,
     * sem_wait is not enabled, to allow the input index to catch up */
    if( digi_buf_input == digi_buf_cnt )
      wait = True;

    /* Date-time info & alignment */
    struct timeval lTime;
    uint32_t sec;
    uint32_t usec;
    uint32_t uwait;

    /* Wait for time Sync on 2 mins */
    gettimeofday( &lTime, NULL );
    sec   = (uint32_t)( lTime.tv_sec % 120 );
    usec  = (uint32_t)( sec * 1000000 + lTime.tv_usec );
    uwait = 120000000 - usec;

    /* Switch to the other buffer and trigger the decoder */
    if( uwait_prev <= uwait )
    {
      rx_state.bufferIndex = ( rx_state.bufferIndex + 1 ) % 2;
      rx_state.iqIndex[rx_state.bufferIndex] = 0;

      // Create a thread and stuff for separate decoding
      if( !Pthread_Create(
            &decState.thread, &decState.tattr, Decoder, NULL,
            _("Failed to create WSPR Decoder thread")) )
        return( NULL );
    }

    uwait_prev = uwait;
  } // while( Transceiver[Indices.TRx_Index]->receive_active ...

  // In case of exit by !Transceiver[Indices.TRx_Index]->receive_active
  rx_state.exit_flag = TRUE;

  // Clear Guest Receive flags
  Flag[GUEST_RECEIVING] = False;
  Flag[GUEST_DEMOD_WVR_DATA]  = False;
  Init_Semaphore( &digimode_semaphore, False );
  Wspr_Cleanup();

  return( NULL );
} // Wspr_Run()

//------------------------------------------------------------------

/* Print_Spots()
 *
 * Prints WSPR spot results in the spots textview
 */
  gboolean
Print_Spots( gpointer data )
{
  gchar str[128];
  double bearing, distance, home_lat, home_long, remote_lat, remote_long;
  int16_t n_results = *( (int16_t *)data );


  /* Signal to initialize */
  if( n_results == -1 )
  {
    gtk_text_buffer_get_iter_at_offset(
        wspr_gui.text_buffer, &wspr_gui.text_buffer_iter, 0 );
    gtk_text_buffer_create_tag(
        wspr_gui.text_buffer, "mono", "font", "mono", NULL);
    return( FALSE );
  }

  // No spot results
  if( (n_results == 0) && !rx_state.exit_flag )
  {
    Set_Label_Text( wspr_gui.wspr_status_label, "No WSPR Spots Detected" );
    snprintf( str, sizeof(str), " --- %02d-%02d-%04d  %02d:%02d",
        rx_state.gmt->tm_mday,
        rx_state.gmt->tm_mon  + 1,
        rx_state.gmt->tm_year + 1900,
        rx_state.gmt->tm_hour,
        rx_state.gmt->tm_min );

    /* Print message */
    gtk_text_buffer_insert_with_tags_by_name(
        wspr_gui.text_buffer, &wspr_gui.text_buffer_iter, str, -1, "mono", NULL );
    gtk_text_buffer_insert( wspr_gui.text_buffer, &wspr_gui.text_buffer_iter, "\n", -1 );

    /* Scroll Text View to bottom */
    gtk_text_view_scroll_mark_onscreen(
        GTK_TEXT_VIEW(wspr_gui.text_view), gtk_text_buffer_create_mark(
          wspr_gui.text_buffer, "end", &wspr_gui.text_buffer_iter, FALSE) );

    return( FALSE );
  } // if( (n_results == 0) && !rx_state.exit_flag )

  if( (n_results != 0) && !rx_state.exit_flag )
  {
    // Home station position
    Gridloc_to_Position( wspr_rc.decoder_options.rx_loc, &home_lat, &home_long );

    // Spots detected = n_results
    gchar lbl[32];
    snprintf( lbl, sizeof(lbl), "WSPR Spots Detected: %d", n_results );
    gtk_label_set_text( wspr_gui.wspr_status_label, lbl );
    for( int16_t idx = 0; idx < n_results; idx++ )
    {
      // Calculate bearing and distance to WSPR station
      if( strncmp(decoder_results[idx].loc, "XXXX", 4) != 0 )
      {
        Gridloc_to_Position( decoder_results[idx].loc, &remote_lat, &remote_long );
        Bearing_Distance( &bearing, &distance, home_lat, home_long, remote_lat, remote_long );
      }
      else
      {
        bearing  = 0;
        distance = 0;
      }

      // Print spot results to string
      snprintf( str, sizeof(str),
          " %2d  %02d-%02d-%04d  %02d:%02d  %5.1f  %5.2f %10.6f  %2d  %12s  %6s  %2s  %3d  %5d",
          idx + 1,
          rx_state.gmt->tm_mday,
          rx_state.gmt->tm_mon  + 1,
          rx_state.gmt->tm_year + 1900,
          rx_state.gmt->tm_hour,
          rx_state.gmt->tm_min,
          decoder_results[idx].snr,
          decoder_results[idx].dt,
          decoder_results[idx].freq,
          (int)decoder_results[idx].drift,
          decoder_results[idx].call,
          decoder_results[idx].loc,
          decoder_results[idx].pwr,
          (uint16_t)bearing,
          (uint16_t)distance );

      /* Print message */
      gtk_text_buffer_insert_with_tags_by_name(
          wspr_gui.text_buffer, &wspr_gui.text_buffer_iter, str, -1, "mono", NULL );
      gtk_text_buffer_insert( wspr_gui.text_buffer, &wspr_gui.text_buffer_iter, "\n", -1 );

      /* Scroll Text View to bottom */
      gtk_text_view_scroll_mark_onscreen(
          GTK_TEXT_VIEW(wspr_gui.text_view), gtk_text_buffer_create_mark(
            wspr_gui.text_buffer, "end", &wspr_gui.text_buffer_iter, FALSE) );
    }
  } // if( (n_results != 0) && !rx_state.exit_flag )

  return( FALSE );
} // Print_Spots()

//------------------------------------------------------------------

// Curl error handler
static char errbuf[CURL_ERROR_SIZE] = { '\0' };
static CURL *curl;

  static void
Curl_Error( CURLcode res, const char *messg )
{
  Set_Label_Text( wspr_gui.wspr_status_label, messg );
  size_t len = strlen( errbuf );
  fprintf( stderr, "\nlibcurl: (%d) ", res );
  if( len )
    fprintf( stderr, "%s%s", errbuf, ((errbuf[len - 1] != '\n') ? "\n" : "") );
  else
    fprintf( stderr, "%s\n", curl_easy_strerror(res) );
  curl_easy_cleanup( curl );
} // Curl_Error()

//------------------------------------------------------------------

/* Post_Spots()
 *
 * Posts WSPR results to wsprnet.org using CURL
 */
  static void
Post_Spots( int16_t n_results )
{
  CURLcode res;
  char url[256], dat[35], utim[5];

  if( !n_results && rx_state.exit_flag )
    return;

  // Date and time of start of spots
  snprintf( dat, sizeof(dat), "%02d%02d%02d",
      rx_state.gmt->tm_year - 100, rx_state.gmt->tm_mon + 1, rx_state.gmt->tm_mday );
  snprintf( utim, sizeof(utim), "%02d%02d",
      rx_state.gmt->tm_hour, rx_state.gmt->tm_min );

  // Post results to WSPR website using curl
  for( int16_t i = 0; i < n_results; i++ )
  {
    snprintf( url, sizeof(url),
        "http://wsprnet.org/post?function=wspr&rcall=%s&rgrid=%s"
        "&rqrg=%.6f&date=%s&time=%s&sig=%.0f&dt=%.1f&tqrg=%.6f"
        "&tcall=%s&tgrid=%s&dbm=%s&version=0.1_wsprd&mode=2",
        wspr_rc.decoder_options.rx_call, wspr_rc.decoder_options.rx_loc,
        decoder_results[i].freq,         dat, utim,
        decoder_results[i].snr,          decoder_results[i].dt,
        decoder_results[i].freq,         decoder_results[i].call,
        decoder_results[i].loc,          decoder_results[i].pwr );

    // Using curl API to send posts to wsprnet.org
    curl = curl_easy_init();
    if( curl )
    {
      res = curl_easy_setopt( curl, CURLOPT_ERRORBUFFER, errbuf );
      if( res != CURLE_OK )
      {
        Curl_Error( res, "curl_easy_setopt() Failed" );
        return;
      }

      res = curl_easy_setopt( curl, CURLOPT_URL, url );
      if( res != CURLE_OK )
      {
        Curl_Error( res, "curl_easy_setopt() Failed" );
        return;
      }

      res = curl_easy_setopt( curl, CURLOPT_NOBODY, 1 );
      if( res != CURLE_OK )
      {
        Curl_Error( res, "curl_easy_setopt() Failed" );
        return;
      }

      res = curl_easy_perform( curl );
      if( res != CURLE_OK )
      {
        Curl_Error( res, "Failed to Post Results" );
        return;
      }

      curl_easy_cleanup( curl );
    }
  } // for( int16_t i = 0; i < n_results; i++ )

  if( n_results )
    Set_Label_Text( wspr_gui.wspr_status_label, "Posted Results to wsprnet.org" );

} // Post_Spots()

//------------------------------------------------------------------

/* Decoder()
 *
 * Thread used for the WSPR Decoder
 */
  static void *
Decoder( void *arg )
{
  static int16_t n_results = 0;


  // Exit if exit flag set
  if( rx_state.exit_flag ) return( NULL );

  Flag[WSPR_DECODER_THREAD] = True;

  /* Select the previous transmission / other buffer */
  uint32_t prevBuffer = (rx_state.bufferIndex + 1) % 2;
  if( rx_state.iqIndex[prevBuffer] < ( (SIGNAL_LENGTH - 3) * SIGNAL_SAMPLE_RATE ) )
  {
    Set_Label_Text( wspr_gui.wspr_status_label, "Signal too short, skipping ..." );
    return( NULL );
  }

  /* Delete any previous samples tail */
  for( uint32_t i = rx_state.iqIndex[prevBuffer]; i < SAMPLES_BUF_SIZE; i++ )
  {
    rx_state.iSamples[prevBuffer][i] = 0.0;
    rx_state.qSamples[prevBuffer][i] = 0.0;
  }

  // Normalize the sample @-3dB
  double maxSig = 1.0E-24;
  for( int i = 0; i < SAMPLES_BUF_SIZE; i++)
  {
    double absI = fabs( rx_state.iSamples[prevBuffer][i] );
    double absQ = fabs( rx_state.qSamples[prevBuffer][i] );

    if( absI > maxSig ) maxSig = absI;
    if( absQ > maxSig ) maxSig = absQ;
  }

  maxSig = 2.0 / maxSig;
  for( int i = 0; i < SAMPLES_BUF_SIZE; i++ )
  {
    rx_state.iSamples[prevBuffer][i] *= maxSig;
    rx_state.qSamples[prevBuffer][i] *= maxSig;
  }

  /* Get the date at the beginning of last recording session with
   * 1 second margin added, just to be sure to be on this even minute
   */
  time_t unixtime;
  time( &unixtime );
  unixtime = unixtime - 120 + 1;
  rx_state.gmt = gmtime( &unixtime );

  /* Search & decode the signal */
  Set_Label_Text( wspr_gui.wspr_status_label, "Decoding WSPR Signals ..." );
  Wspr_Decode(
      rx_state.iSamples[prevBuffer],
      rx_state.qSamples[prevBuffer],
      SAMPLES_BUF_SIZE,
      decoder_results, &n_results );

  // Print results to text viewer and post to wsprnet.org
  if( !rx_state.exit_flag )
  {
    g_idle_add( Print_Spots, (gpointer)&n_results );
    while( g_main_context_iteration(NULL, FALSE) );
  }

  // Post spots to WSPR website
  if( wspr_rc.post && !rx_state.exit_flag )
    Post_Spots( n_results );

  // Display Spots in Xplanet window
  if( wspr_rc.xplanet && n_results && !rx_state.exit_flag )
    Wspr_Display_Location( n_results );

  // Enter any new frequency to the center frequency
  wspr_rc.decoder_options.center_freq = wspr_rc.decoder_options.new_freq;

  Flag[WSPR_DECODER_THREAD] = False;
  return( NULL );
} // Decoder()

//------------------------------------------------------------------

/* Wspr_Start()
 *
 * Starts/Stops WSPR Receiving
 */
  void
Wspr_Start( GtkToggleButton *togglebutton )
{
  //  Decoder_Self_Test();
  //  exit(0);

  // Abort if in exit state but decoder still running
  if( rx_state.exit_flag && rx_state.decoder_flag )
    return;

  // Start the WSPR listening and decoding
  if( gtk_toggle_button_get_active(togglebutton) &&
      Transceiver[Indices.TRx_Index]->receive_active )
  {
    Set_Label_Text( wspr_gui.wspr_status_label, "Synchronising with 2min mark ..." );

    // Allocate demodulator buffers
    Alloc_Demod_Buf_Copies( SOUND_OP_BUF_SIZE );

    // Initialize Digimode semaphore
    if( !Init_Semaphore(&digimode_semaphore, True) )
      return;

    // Enter any new frequency to the center frequency
    wspr_rc.decoder_options.center_freq = wspr_rc.decoder_options.new_freq;

    // Start receiving and decoding signals
    Flag[GUEST_DEMOD_WVR_DATA] = True;
    rx_state.exit_flag         = False;

    // Start WSPR Receive / Decode functions
    static BOOLEAN reset;
    reset = True;
    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Wspr_Run, &reset,
          _("Failed to create Receive thread")) )
      return;

    rx_state.run_flag = True;
  } //if( gtk_toggle_button_get_active(togglebutton) )
  else
  {
    // Set Run Exit flag, Wspr_Run() and Decoder() threads will exit
    rx_state.exit_flag = True;
    rx_state.run_flag  = False;
    Set_Label_Text( wspr_gui.wspr_status_label, "Stopping WSPR Decoder" );
  }

} // Wspr_Start()

//------------------------------------------------------------------

/* Wspr_Band_Combobox_Changed()
 *
 * Handles the wspr_bands_combobox_changed callback
 */
  void
Wspr_Bands_Combobox_Changed( GtkComboBoxText *combobox )
{
  // Current transceiver object
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get frequency from combobox text
  gchar *txt = gtk_combo_box_text_get_active_text( combobox );
  uint32_t center_freq = (uint32_t)( Atof(txt) * 1.0E6 );
  g_free( txt );
  wspr_rc.decoder_options.new_freq = center_freq;

  // Initialize on start up
  if( wspr_rc.decoder_options.center_freq == 0 )
    wspr_rc.decoder_options.center_freq = center_freq;

  // Set up new frequency on change
  if( TRx->rx_frequency != center_freq )
  {
    // Set the Rx Band combobox first
    Set_Rx_Band( TRx, center_freq );
    TRx->rx_frequency = center_freq;
    Update_Spin_Dial( TRx, RX_FLAG );
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );
  }

  // Set WSPR modulation mode
  Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_WSPR );

} // Wspr_Bands_Combobox_Changed()

//------------------------------------------------------------------

