/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "shared.h"
#include "wsprd.h"
#include "../Hermes2/callback_func.h"
#include "../common/shared.h"
#include "../common/utils.h"

//------------------------------------------------------------------

/* Wspr_Cleanup()
 *
 * Cleans up before quitting
 */
  void
Wspr_Cleanup( void )
{
  // Free the digimode semaphore
  Init_Semaphore( &digimode_semaphore, False );

  // Clear all relevant flags
  Flag[GUEST_QUIT]           = False;
  Flag[GUEST_DEMOD_WVR_DATA] = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

  // Free decoder resources
  Free_Wspr_Decoder();

  // Kill xplanet and remove files
  if( wspr_rc.xplanet )
    Kill_Xplanet(
        wspr_rc.xplanet_config_file,
        wspr_rc.xplanet_marker_file,
        wspr_rc.xplanet_arc_file );
  wspr_rc.xplanet = False;

} // Wspr_Cleanup()

//------------------------------------------------------------------------

/* Wspr_Read_Config()
 *
 * Loads the wsprrc configuration file
 */
  gboolean
Wspr_Read_Config( gpointer data )
{
  char
    rc_fpath[FILE_NAME_SIZE],  // File path to wsprrc
    line[READ_LINE_BUF_SIZE];  // Buffer for Read_Line

  // Config file pointer
  FILE *wsprrc;

  // Setup file path to wsprrc and create the wspr config file name
  Strlcpy( rc_fpath, wspr_rc.wspr_dir, sizeof(rc_fpath) );
  Strlcat( rc_fpath, "wspr.config",    sizeof(rc_fpath) );

  // Open wsprrc file
  wsprrc = fopen( rc_fpath, "r" );
  if( wsprrc == NULL )
  {
    perror( rc_fpath );
    Error_Dialog(
        _("Failed to open wsprrc file\n"\
          "Quit WSPR and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station Callsign, abort if EOF
  if( Read_Line(line, wsprrc, _("Station Callsign")) != SUCCESS )
    return( FALSE );
  Strlcpy( wspr_rc.decoder_options.rx_call, line, sizeof(wspr_rc.decoder_options.rx_call) );

  // Read Station Locator, abort if EOF
  if( Read_Line(line, wsprrc, _("Station Locator")) != SUCCESS )
    return( FALSE );
  Strlcpy( wspr_rc.decoder_options.rx_loc, line, sizeof(wspr_rc.decoder_options.rx_loc) );

  // Read Home Name
  if( Read_Line(line, wsprrc, _("Home Name") ) != SUCCESS )
    return( FALSE );
  Strlcpy( wspr_rc.home_name, line, sizeof(wspr_rc.home_name) );

  // Read Bands Combobox Active Index, abort if EOF
  wspr_rc.decoder_options.center_freq = 0; // Init to 0
  if( Read_Line(line, wsprrc, _("Bands Combobox Active Index")) != SUCCESS )
    return( FALSE );
  gint idx = atoi( line );
  gtk_combo_box_set_active( GTK_COMBO_BOX(wspr_gui.bands_combobox), idx );

  // Read Use Hashtable, abort if EOF
  if( Read_Line(line, wsprrc, _("Use Hashtable")) != SUCCESS )
    return( FALSE );
  wspr_rc.decoder_options.use_hashtable = (BOOLEAN)atoi( line );

  // Read Number of Passes, abort if EOF
  if( Read_Line(line, wsprrc, _("Number of Passes")) != SUCCESS )
    return( FALSE );
  wspr_rc.decoder_options.npasses = (uint8_t)atoi( line );

  // Read Subtraction Flag, abort if EOF
  if( Read_Line(line, wsprrc, _("Subtraction Flag")) != SUCCESS )
    return( FALSE );
  wspr_rc.decoder_options.subtraction = (BOOLEAN)atoi( line );

  // Read Quick Mode, abort if EOF
  if( Read_Line(line, wsprrc, _("Quick Mode")) != SUCCESS )
    return( FALSE );
  wspr_rc.decoder_options.quick_mode = (BOOLEAN)atoi( line );

  // Read Post Results flag, abort if EOF
  if( Read_Line(line, wsprrc, _("Post Results")) != SUCCESS )
    return( FALSE );
  wspr_rc.post = (BOOLEAN)atoi( line );
  if( wspr_rc.post )
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(wspr_gui.post_checkbutton), TRUE );
  else
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(wspr_gui.post_checkbutton), FALSE );

  // Read Xplanet Display flag, abort if EOF
  if( Read_Line(line, wsprrc, _("Xplanet Display")) != SUCCESS )
    return( FALSE );
  wspr_rc.xplanet = (BOOLEAN)atoi( line );
  if( wspr_rc.xplanet )
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(wspr_gui.xplanet_checkbutton), TRUE );
  else
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(wspr_gui.xplanet_checkbutton), FALSE );

  // Read and set Window position
  gint x, y;
  idx = 0;

  // Read Main Window position, abort if EOF
  if( Read_Line(line, wsprrc, _("Main Window position") ) != SUCCESS )
    return( FALSE );
  x = atoi( line );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
  y = atoi( &line[idx + 1] );
  gtk_window_move( GTK_WINDOW(wspr_gui.window), x, y );

  // Read Main Window size, abort if EOF
  idx = 0;
  if( Read_Line(line, wsprrc, _("Main Window size") ) != SUCCESS )
    return( FALSE );
  x = atoi( line );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
  y = atoi( &line[idx + 1] );
  gtk_window_resize( GTK_WINDOW(wspr_gui.window), x, y );

  // Read Xplanet Command Options
  if( Read_Line(wspr_rc.xplanet_options, wsprrc, _("Xplanet Options") ) != SUCCESS )
    return( FALSE );
  Strlcpy( wspr_rc.xplanet_command,
      "xplanet -window -projection rectangular ", sizeof(wspr_rc.xplanet_command) );
  Strlcat( wspr_rc.xplanet_command, wspr_rc.xplanet_options,
      sizeof(wspr_rc.xplanet_command) );

  // Read Xplanet Config file
  if( Read_Line(wspr_rc.xplanet_config_file, wsprrc, _("Xplanet config file") ) != SUCCESS )
    return( FALSE );

  // Read Xplanet Marker file
  if( Read_Line(wspr_rc.xplanet_marker_file, wsprrc, _("Xplanet marker file") ) != SUCCESS )
    return( FALSE );

  // Read Xplanet Arc file
  if( Read_Line(wspr_rc.xplanet_arc_file, wsprrc, _("Xplanet arc file") ) != SUCCESS )
    return( FALSE );

  // Finished reading config data
  Close_File( &wsprrc );

  return( FALSE );
} // End of Wspr_Read_Config()

//------------------------------------------------------------------

/* Wspr_Write_Config()
 *
 * Loads the wsprrc configuration file
 */
  void
Wspr_Write_Config( void )
{
  // File path to wsprrc
  char rc_fpath[FILE_NAME_SIZE];

  // Config file pointer
  FILE *wsprrc;

  // Setup file path to wsprrc and create the wspr config file name
  Strlcpy( rc_fpath, wspr_rc.wspr_dir, sizeof(rc_fpath) );
  Strlcat( rc_fpath, "wspr.config",    sizeof(rc_fpath) );

  // Open wsprrc file
  wsprrc = fopen( rc_fpath, "w" );
  if( wsprrc == NULL )
  {
    perror( rc_fpath );
    Error_Dialog(
        _("Failed to open wsprrc file\n"\
          "Quit WSPR and correct"), HIDE_OK );
    return;
  }

  // Write Station Callsign
  fprintf( wsprrc, _("# Station Callsign. Maximum 12 characters\n%s\n"),
      wspr_rc.decoder_options.rx_call );

  // Write Station Locator
  fprintf( wsprrc, _("# Station Locator. Maximum 6 characters\n%s\n"),
      wspr_rc.decoder_options.rx_loc );

  // Write Station Home Name
  fprintf( wsprrc, _("# Station Home Name. Maximum 80 characters\n%s\n"),
      wspr_rc.home_name );

  // Write Bands Combobox Index
  fprintf( wsprrc, _("# Write Bands Combobox Index\n%d\n"),
      gtk_combo_box_get_active(GTK_COMBO_BOX(wspr_gui.bands_combobox)) );

  // Write Use Hashtable
  fprintf( wsprrc, _("# Use Hashtable \n%d\n"),
      wspr_rc.decoder_options.use_hashtable );

  // Write Number of Passes
  fprintf( wsprrc, _("# Number of Passes \n%d\n"),
      wspr_rc.decoder_options.npasses );

  // Write Subtraction flag
  fprintf( wsprrc, _("# Subtraction Flag \n%d\n"),
      wspr_rc.decoder_options.subtraction );

  // Write Quick Mode
  fprintf( wsprrc, _("# Quick Mode \n%d\n"),
      wspr_rc.decoder_options.quick_mode );

  // Write Post Results Flag
  fprintf( wsprrc, _("# Post Results Flag \n%d\n"), wspr_rc.post );

  // Write Xplanet Display Flag
  fprintf( wsprrc, _("# Xplanet Display Flag \n%d\n"), wspr_rc.xplanet );

  // Get and save Window position
  gint x, y;
  gtk_window_get_position( GTK_WINDOW(wspr_gui.window), &x, &y );
  fprintf( wsprrc, _("# Main Window position\n%d,%d\n"), x, y );

  // Get and save Window size
  gtk_window_get_size( GTK_WINDOW(wspr_gui.window), &x, &y );
  fprintf( wsprrc, _("# Main Window size\n%d,%d\n#"), x, y );

  // Write Xplanet Options
  fprintf( wsprrc, _("# Xplanet Options\n%s\n"), wspr_rc.xplanet_options );

  // Write Xplanet Config file
  fprintf( wsprrc, _("# Xplanet Config file\n%s\n"), wspr_rc.xplanet_config_file );

  // Write Xplanet Marker file
  fprintf( wsprrc, _("# Xplanet Marker file\n%s\n"), wspr_rc.xplanet_marker_file );

  // Write Xplanet Arc file
  fprintf( wsprrc, _("# Xplanet Arc file\n%s\n#"), wspr_rc.xplanet_arc_file );

  // Finished Writing config data
  Close_File( &wsprrc );

} // End of Wspr_Write_Config()

//------------------------------------------------------------------

