/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "wefax.h"
#include "bookmarks.h"
#include "detect.h"
#include "display.h"
#include "utils.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/display.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

//------------------------------------------------------------------------

// Pixel value threshold for bilevel (0/255) image
#define BILEVEL_THRESHOLD       160

/* Minimum in-image sync pulse level
 * before correction is applied */
#define INIMAGE_SYNC_THRESHOLD  -150

// Sync correction allowed error range
#define SYNC_CORRECT_RANGE      5

#define INIMAGE_PHASING_RANGE   80

/* Value needed to position in-image
 * phasing pulse at beginning of line */
#define WEFAX_PHASING_PULSE_REF   40

//------------------------------------------------------------------------

// Receive and Standby status indicators
#define RECEIVE \
  _("<span background=\"green\" foreground=\"white\"> RECEIVE </span>")
#define STANDBY \
  _("<span background=\"lightgrey\" foreground=\"black\"> STANDBY </span>")

//------------------------------------------------------------------

enum
{
  ENHANCE_NONE = 0,
  ENHANCE_CONTRAST,
  ENHANCE_BILEVEL
};

//------------------------------------------------------------------------

/* Receive_Error()
 *
 * Displays error conditions on receive
 */
  static void
Receive_Error( void )
{
  Wefax_Show_Message( _("Error - Stopping Reception"), "red" );
  Wefax_Set_Indicators( WEFAX_ICON_START_NO );
  Wefax_Set_Indicators( WEFAX_ICON_SYNC_NO );
  Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
} // Receive_Error()

//------------------------------------------------------------------------

static uint8_t *image_buffer = NULL;

// Saves WEFAX images to disk
  static void
Save_Image( void )
{
  // File name string for saving images
  static char file_name_jpg[FILE_NAME_SIZE], file_name_pgm[FILE_NAME_SIZE];

  // File pointer for above
  FILE *fp = NULL;

  static BOOLEAN first_call = True;


  // Abort if image save not enabled
  if( !Flag[WEFAX_SAVE_IMAGE] ) return;

  // Initialize on first call
  if( first_call )
  {
    // Make a file name for the WEFAX image
    Wefax_File_Name( file_name_jpg, "jpg" );
    Wefax_File_Name( file_name_pgm, "pgm" );
    first_call = False;
  }

  // Open file and save WEFAX PGM image
  if( Flag[WEFAX_SAVE_IMAGE_PGM] )
  {
    if( Open_File(&fp, file_name_pgm, "w") == SUCCESS )
    {
      Wefax_Show_Message( _("Saving Decoded PGM Image File ..."), "black" );
      if( !Wefax_Save_Image_PGM( &fp, "P5", wefax_rc.pixels_per_line,
            wefax_display.line_count, 255, image_buffer) )
      {
        Wefax_Show_Message( _("Failed to save PGM Image File"), "red" );
        Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
      }
    }
    else
    {
      Wefax_Show_Message( _("Failed to open PGM Image File"), "red" );
      Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
    }

    return;
  } // if( Flag[WEFAX_SAVE_IMAGE_PGM) )

  // Open file and save WEFAX JPEG image
  if( Flag[WEFAX_SAVE_IMAGE_JPG] )
  {
    Wefax_Show_Message( _("Saving Decoded JPG Image File ..."), "black" );
    if( Open_File(&fp, file_name_jpg, "w") == SUCCESS )
    {
      if( !Wefax_Save_Image_JPEG( &fp, wefax_rc.pixels_per_line,
            wefax_display.line_count, image_buffer) )
      {
        Wefax_Show_Message( _("Failed to save JPG Image File"), "red" );
        Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
      }
    }
    else
    {
      Wefax_Show_Message( _("Failed to open JPG Image File"), "red" );
      Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
    }
  } // if( Flag[WEFAX_SAVE_IMAGE_JPG) )

  return;
} // Save_Images()

//------------------------------------------------------------------------

/* Wefax_Decode()
 *
 * Function that decodes Wefax signals into images
 */
  static BOOLEAN
Wefax_Decode( void )
{
  // First call of function flag
  static BOOLEAN first_call = True;

  // Buffer for creating a PGM image file
  uint32_t image_buffer_idx;  // Index to above
  size_t buf_size;

  // Detector output
  uint8_t discr_op = 0;

  // Detector output average
  static double discr_op_ave = 0;

  int16_t
    discr_op_max  = 0, // Detector output maximum
    discr_max_idx = 0; // Detector output max's index

  // Index to current pixel in image line
  static uint16_t pixel_idx;

  // Count of up or down sync error directions
  static int16_t sync_correct;

  // Stop Tone received flag
  static BOOLEAN stop = False;

  // Reset on new params
  if( Flag[WEFAX_START_NEW_IMAGE] )
  {
    first_call = True;
    Flag[WEFAX_START_NEW_IMAGE] = False;
  }

  // *** Initialize ***
  if( first_call )
  {
    Wefax_Set_Indicators( WEFAX_ICON_DECODE_YES );
    if( Flag[WEFAX_SAVE_IMAGE] )
      Wefax_Set_Indicators( WEFAX_ICON_SAVE_YES );

    // Fill pixbuf with background color
    gdk_pixbuf_fill( wefax_display.pixbuf, PIXBUF_BACKGND );
    Queue_Draw( wefax_gui.drawingarea );

    // Initialize statics
    pixel_idx = 0;
    wefax_display.line_count = 0;
    discr_op_ave = 0.0;
    first_call = False;
    stop = False;

    // Allocate image buffer
    Mem_Free( (void **) &image_buffer );
    buf_size = (size_t)wefax_rc.pixels_per_line;
    Mem_Alloc( (void **) &image_buffer, buf_size );

  } // if( first_call )

  // Stop on user request
  if( Flag[WEFAX_RECEIVE_STOP] &&
      wefax_display.line_count )
  {
    Save_Image();
    first_call   = True;
    wefax_action = WEFAX_ACTION_STOP;
    return( True );
  } // if( Flag[WEFAX_RECEIVE_STOP) )

  // Skip Image decoding
  if( Flag[WEFAX_SKIP_ACTION] )
  {
    // Re-initialize line buffer indices
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    Wefax_Show_Message( _("Skipping Image Decode"), "orange" );
    Flag[WEFAX_SKIP_ACTION] = False;

    if( wefax_display.line_count )
      Save_Image();

    first_call   = True;
    wefax_action = WEFAX_ACTION_BEGIN;
    return( True );
  } // if( Flag[WEFAX_SKIP_ACTION) )

  /* Fill line buffer with pixel values with signal
   * samples from FM detector. If it returns false,
   * wait for it to collect more sound samples */
  if( !FM_Detector(&discr_op) )
  {
    if( !FM_Detector(&discr_op) )
      return( False );
  }

  // Display detector output
  Wefax_Display_Signal( discr_op );

  // Detect stop pulse
  stop |= Stop_Tone_Detect( discr_op );

  // Current position in image buffer to save pixel value
  wefax_display.line_buffer[wefax_display.linebuff_input] = discr_op;
  wefax_display.linebuff_input++;
  if( wefax_display.linebuff_input >= wefax_rc.line_buffer_size )
    wefax_display.linebuff_input = 0;

  // Return to control function at each pixel
  pixel_idx++;
  if( pixel_idx < wefax_rc.pixels_per_line )
    return( True );

  // Copy line buffer to currrent image buffer line
  discr_max_idx = 0;
  discr_op_max  = -256;
  image_buffer_idx = wefax_display.line_count * wefax_rc.pixels_per_line;
  for( pixel_idx = 0; pixel_idx < wefax_rc.pixels_per_line; pixel_idx++ )
  {
    /* Try to sync image if enabled by finding
     * the position of sync pulse maximum */
    if( Flag[WEFAX_INIMAGE_PHASING] &&
        (pixel_idx < INIMAGE_PHASING_RANGE) )
    {
      // Average negated line buffer pixel values
      discr_op_ave *= WEFAX_PHASING_PUSLE_WIN - 1.0;
      discr_op_ave -=
        (double)wefax_display.line_buffer[wefax_display.linebuff_output];
      discr_op_ave /= WEFAX_PHASING_PUSLE_WIN;

      // Record maximum value of detector output
      if( discr_op_max < (int16_t)discr_op_ave )
      {
        discr_op_max  = (int16_t)discr_op_ave;
        discr_max_idx = (int16_t)pixel_idx;
      }
    } // if( Flag[WEFAX_INIMAGE_PHASING) )

    // Make image bi-level if enabled
    if( wefax_rc.image_enhance == ENHANCE_BILEVEL )
    {
      if( wefax_display.line_buffer[wefax_display.linebuff_output] > BILEVEL_THRESHOLD )
        wefax_display.line_buffer[wefax_display.linebuff_output] = 255;
      else
        wefax_display.line_buffer[wefax_display.linebuff_output] = 0;
    }

    // Copy line buffer to currrent image buffer line
    image_buffer[image_buffer_idx + pixel_idx] =
      wefax_display.line_buffer[wefax_display.linebuff_output];
    wefax_display.linebuff_output++;
    if( wefax_display.linebuff_output >= wefax_rc.line_buffer_size )
      wefax_display.linebuff_output = 0;
  } // for( pixel_idx = 0; pixel_idx < wefax_rc.pixels_per ...

  if( !Flag[WEFAX_INIMAGE_PHASING] )
    sync_correct = 0;

  // Correct sync error one pixel at a time if enabled
  if( Flag[WEFAX_INIMAGE_PHASING] &&
      (discr_op_max > INIMAGE_SYNC_THRESHOLD) )
  {
    // Position ref of sync pulse, tracks input idx
    // static int16_t sync_pos_ref;

    // Distance in pix of sync pulse from its required position
    static int16_t sync_error;

    // Try to set the sync pulse at start of line
    sync_error = discr_max_idx - WEFAX_PHASING_PULSE_REF;

    // Count up or down error conditions
    if( sync_error > 0 ) sync_correct--;
    if( sync_error < 0 ) sync_correct++;

    /* Keep sync correction inside the
     * sync correct range to avoid hunting */
    if( sync_correct >= SYNC_CORRECT_RANGE )
    {
      wefax_display.linebuff_input++;
      // sync_pos_ref++;
      sync_correct = 0;
    }
    else if( sync_correct <= -SYNC_CORRECT_RANGE )
    {
      wefax_display.linebuff_input--;
      // sync_pos_ref--;
      sync_correct = 0;
    }

    // Keep buffer index within bounds
    if( wefax_display.linebuff_input >= wefax_rc.line_buffer_size )
      wefax_display.linebuff_input -= wefax_rc.line_buffer_size;
    else if( wefax_display.linebuff_input < 0 )
      wefax_display.linebuff_input += wefax_rc.line_buffer_size;

  } // if( Flag[WEFAX_INIMAGE_PHASING) )

  /* Normalize image line for better contrast.
   * Leave behind the pixels of phasing pulse. */
  if( wefax_rc.image_enhance == ENHANCE_CONTRAST )
  {
    static uint32_t
      norm_idx,   // Index to image buffer for normalization
      norm_len;   // Length of image line to be normalized
    norm_idx = image_buffer_idx + WEFAX_PHASING_PULSE_SIZE;
    norm_len = wefax_rc.pixels_per_line - WEFAX_PHASING_PULSE_SIZE;
    Normalize( &image_buffer[norm_idx], norm_len );
  }

  // Fill pixels of display buffer from the image line
  for( pixel_idx = 0; pixel_idx < wefax_rc.pixels_per_line; pixel_idx++ )
  {
    // Pixbuf's pixel pointer
    guchar *pixel =
      wefax_display.pixel_buf +
      wefax_display.n_channels * pixel_idx +
      wefax_display.line_count * wefax_display.rowstride;

    pixel[0] = image_buffer[image_buffer_idx + pixel_idx];
    pixel[1] = image_buffer[image_buffer_idx + pixel_idx];
    pixel[2] = image_buffer[image_buffer_idx + pixel_idx];
  } // for( pixel_idx = 0; pixel_idx < wefax_rc.pixels_per ...

  // Draw the (partial) image
  Queue_Draw( wefax_gui.drawingarea );

  // Wait for GTK to complete its tasks
  while( g_main_context_iteration(NULL, FALSE) );

  /* Make sure that the buffer input
   * index stays ahead of output index */
  int diff = wefax_display.linebuff_input - wefax_display.linebuff_output;
  if( (diff < 0) && (diff >= -wefax_rc.pixels_per_line) )
    wefax_display.linebuff_input += wefax_rc.pixels_per_line;
  else if( diff > wefax_rc.pixels_per_line )
    wefax_display.linebuff_input -= wefax_rc.pixels_per_line;

  if( wefax_display.linebuff_input >= wefax_rc.line_buffer_size )
    wefax_display.linebuff_input -= wefax_rc.line_buffer_size;
  else if( wefax_display.linebuff_input < 0 )
    wefax_display.linebuff_input += wefax_rc.line_buffer_size;

  // End image decode on stop tone
  if( stop )
  {
    Wefax_Show_Message( _("Ending WEFAX Decode ..."), "green" );
    Wefax_Set_Indicators( WEFAX_ICON_DECODE_APPLY );
  }

  /* End image decode if gone for
   * too long (missed stop tone?) */
  wefax_display.line_count++;
  if( wefax_display.line_count >= wefax_rc.image_lines )
  {
    Wefax_Show_Message( _("Ending Decode-Missed Stop Tone?"), "orange" );
    Wefax_Set_Indicators( WEFAX_ICON_DECODE_SKIP );
    stop = True;
  }

  // End image decode and save
  if( stop && wefax_display.line_count )
  {
    Save_Image();
    first_call   = True;
    wefax_action = WEFAX_ACTION_BEGIN;
    return( True );
  } // if( stop && line_count )

  // Re-allocate image buffer per line
  buf_size = (size_t)( (wefax_display.line_count + 1) * wefax_rc.pixels_per_line );
  Mem_Realloc( (void **) &image_buffer, buf_size );

  pixel_idx = 0;
  return( True );
} // Wefax_Decode()

//------------------------------------------------------------------------

// Free resources
  void
Wefax_Free_Image( void )
{
  Mem_Free( (void **) &image_buffer );
}

//------------------------------------------------------------------------

/* Scroll_to_Top()
 *
 * Scrolls the Image window to the top
 */
  static gboolean
Scroll_to_Top( gpointer data )
{
  // Adjustments for setting the position of scroller
  GtkAdjustment *adjm;
  GtkScrolledWindow *scrollwin;

  scrollwin = GTK_SCROLLED_WINDOW(
      Builder_Get_Object(wefax_gui.window_builder, "wefax_image_scrolledwindow") );
  adjm = gtk_scrolled_window_get_vadjustment( scrollwin );
  gtk_adjustment_set_value( adjm, 0.0 );

  return( FALSE );
} // Scroll_to_Top()

//------------------------------------------------------------------------

/* Wefax_Control()
 *
 * Central control function that directs Wefax decoding functions
 */
  static void *
Wefax_Control( void *data )
{
  // Continue till user requests stop
  Flag[GUEST_RECEIVING] = True;
  while( !Flag[GUEST_QUIT] && (wefax_action != WEFAX_ACTION_STOP) )
  {
    /* Direct program flow according
     * to currently selected action */
    switch( wefax_action )
    {
      case WEFAX_ACTION_BEGIN: // Begin decoding process
        Wefax_Show_Message( _("Listening for Start Tone ..."), "black" );
        Wefax_Set_Indicators( WEFAX_ICON_START_YES );
        Wefax_Set_Indicators( WEFAX_ICON_SYNC_NO );
        Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );

        // Move scroller to top of image window
        g_idle_add( Scroll_to_Top, NULL );
        while( g_main_context_iteration(NULL, FALSE) );

        wefax_action = WEFAX_ACTION_START;
        break;

      case WEFAX_ACTION_START: // Looking for WEFAX start tone
        if( !Start_Tone_Detect() )
        {
          Receive_Error();
          return( NULL );
        }
        break;

      case WEFAX_ACTION_PHASING: // Sync with WEFAX phasing pulses
        if( !Phasing_Detect() )
        {
          Receive_Error();
          return( NULL );
        }
        break;

      case WEFAX_ACTION_DECODE: // Decode WEFAX images
        if( !Wefax_Decode() )
        {
          Receive_Error();
          Flag[GUEST_RECEIVING] = False;
          return( NULL );
        }
        break;
    } // switch( wefax_action )

  } // while(!Flag[GUEST_QUIT] && ... )

  // Function below hangs on window delete event ??
  if( !Flag[GUEST_QUIT] )
    Wefax_Show_Message( _("Stopping Reception"), "black" );
  Flag[GUEST_QUIT] = False;

  // Destroy digimode semaphore
  sem_post( &digimode_semaphore );
  Init_Semaphore(&digimode_semaphore, False );
  Flag[GUEST_RECEIVING] = False;

  return( NULL );
} // Wefax_Control()

//------------------------------------------------------------------------

/* Wefax_Drawingarea_Button_Press()
 *
 * Handles button press event on wefax drawingarea
 */
  gboolean
Wefax_Drawingarea_Button_Press( const GdkEventButton  *event )
{
  // Popup main menu
  if( event->button == 3 )
  {
    gtk_menu_popup_at_pointer( GTK_MENU(wefax_gui.popup_menu), NULL );
    return( TRUE );
  }

  /* Adjust the line buffer input index to bring the
   * column of button press to the beginnig of line */
  if( event->button == 1 )
  {
    wefax_display.linebuff_input -= (int)(event->x + 0.5);
    if( wefax_display.linebuff_input < 0 )
      wefax_display.linebuff_input += wefax_rc.line_buffer_size;
  }

  return( TRUE );
} // Wefax_Drawingarea_Button_Press()

//------------------------------------------------------------------------

/* Wefax_Start_Button_Toggled()
 *
 * Handles toggled event of START toggle button
 * Sets menu item sensitivity and various indicators
 */
  void
Wefax_Start_Button_Toggled( GtkToggleButton *togglebutton )
{
  GtkLabel *lbl = GTK_LABEL(
      Builder_Get_Object(wefax_gui.window_builder, "rcve_status") );
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    // Initialize digimode semaphore
    if( !Init_Semaphore(&digimode_semaphore, True) )
      return;

    // Start receiving and decoding signals
    Flag[GUEST_DEMOD_IQ_DATA]   = True;
    Flag[WEFAX_RECEIVE_STOP]    = False;
    gtk_label_set_markup( lbl, RECEIVE );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_NO );
    Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;
    wefax_action = WEFAX_ACTION_BEGIN;

    // The conditional is there to silence Coverity Scan warnings
    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Wefax_Control, NULL,
          "Failed to create Wefax_Control() thread") )
      return;
  }
  else
  {
    wefax_action = WEFAX_ACTION_STOP;
    Flag[GUEST_DEMOD_IQ_DATA]   = False;
    Flag[WEFAX_RECEIVE_STOP]    = True;

    // Destroy digimode semaphore
    sem_post( &digimode_semaphore );
    Init_Semaphore(&digimode_semaphore, False );

    Wefax_Set_Indicators( WEFAX_ICON_START_NO );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_NO );
    Wefax_Set_Indicators( WEFAX_ICON_DECODE_NO );
    gtk_label_set_markup( lbl, STANDBY );
  }
} // Wefax_Start_Button_Toggled()

//------------------------------------------------------------------------

