/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "detect.h"
#include "display.h"
#include "jpeg.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

//------------------------------------------------------------------

#define WEFAX_CONFIG_FILE  "/.hermes2/wefax/wefax.config"
#define WEFAX_DIRECTORY    "/.hermes2/wefax/"
#define WEFAX_BMK_FILE     "/.hermes2/wefax/bookmarks"

//------------------------------------------------------------------

/* Wefax_Read_Config()
 *
 * Loads the wefax.config configuration file
 */
  gboolean
Wefax_Read_Config( gpointer data )
{
  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];

  // Config file pointer
  FILE *wefaxrc;

  // Setup file path to wefaxrc
  Strlcpy( wefax_rc.rc_fpath, getenv("HOME"),    sizeof(wefax_rc.rc_fpath) );
  Strlcat( wefax_rc.rc_fpath, WEFAX_CONFIG_FILE, sizeof(wefax_rc.rc_fpath) );

  // Open wefaxrc file
  wefaxrc = fopen( wefax_rc.rc_fpath, "r" );
  if( wefaxrc == NULL )
  {
    perror( wefax_rc.rc_fpath );
    Wefax_Show_Message(
        _("Failed to open wefax.config file\n"\
          "Quit wefax and correct"), "red" );
    Error_Dialog(
        _("Failed to open wefax.config file\n"\
          "Quit wefax and correct"), HIDE_OK );
    return( FALSE );
  }

  // *** Runtime configuration data ***
  // Read default main window height, abort if EOF
  if( Read_Line(line, wefaxrc, _("Main Window Height")) != SUCCESS )
    return( FALSE );
  wefax_rc.window_height = (uint16_t)atoi( line );

  // Read default number of lines to decode, abort if EOF
  if( Read_Line(line, wefaxrc, _("Maximum Lines to Decode")) != SUCCESS )
    return( FALSE );
  wefax_rc.image_lines = (uint16_t)atoi( line );
  if( (wefax_rc.image_lines < 120) || (wefax_rc.image_lines > 3000) )
  {
    Close_File( &wefaxrc );
    Wefax_Show_Message(
        _("Specified Maximum Number of Lines"\
          "to Decode seems unreasonable\n"\
          "Quit and correct wefaxrc"), "red" );
    Error_Dialog(
        _("Specified Maximum Number of Lines"\
          "to Decode seems unreasonable\n"\
          "Quit and correct wefaxrc"), HIDE_OK );
    return( FALSE );
  }

  // Read default lines per minute, abort if EOF
  if( Read_Line(line, wefaxrc, _("Lines per Minute")) != SUCCESS )
    return( FALSE );
  wefax_rc.lines_per_min = atof( line );
  if( (wefax_rc.lines_per_min < 60.0) || (wefax_rc.lines_per_min > 1000.0) )
  {
    Close_File( &wefaxrc );
    Wefax_Show_Message(
        _("Error reading RPM (Lines/Minute)\n"\
          "Quit and correct wefaxrc"), "red" );
    Error_Dialog(
        _("Error reading RPM (Lines/Minute)\n"\
          "Quit and correct wefaxrc"), HIDE_OK );
    return( FALSE );
  }

  // Read default image resolution in pixels, abort if EOF
  if( Read_Line(line, wefaxrc, _("Wefax Image Resolution")) != SUCCESS )
    return( FALSE );
  wefax_rc.pixels_per_line = (uint16_t)atoi( line );
  if( (wefax_rc.pixels_per_line < 120) || (wefax_rc.pixels_per_line > 1200) )
  {
    Close_File( &wefaxrc );
    Wefax_Show_Message(
        _("Error reading Image Resolution\n"\
          "(Pixels per Image Line)\n"\
          "Quit and correct wefaxrc"), "red" );
    Error_Dialog(
        _("Error reading Image Resolution\n"\
          "(Pixels per Image Line)\n"\
          "Quit and correct wefaxrc"), HIDE_OK );
    return( FALSE );
  }

  // Read default IOC value, abort if EOF
  if( Read_Line(line, wefaxrc, _("Default IOC Value")) != SUCCESS )
    return( FALSE );
  wefax_rc.ioc_value = (uint16_t)atoi( line );
  if( (wefax_rc.ioc_value != 288) && (wefax_rc.ioc_value != 576) )
  {
    Close_File( &wefaxrc );
    Wefax_Show_Message(
        _("Error reading IOC Value\n"\
          "Quit and correct wefaxrc"), "red" );
    Error_Dialog(
        _("Error reading IOC Value\n"\
          "Quit and correct wefaxrc"), HIDE_OK );
    return( FALSE );
  }

  // Set the default start tone
  if( wefax_rc.ioc_value == IOC576 )
    wefax_rc.start_tone = IOC576_START_TONE;
  else
    wefax_rc.start_tone = IOC288_START_TONE;

  // Read default number of phasing lines, abort if EOF
  if( Read_Line(line, wefaxrc, _("Number of Phasing Lines")) != SUCCESS )
    return( FALSE );
  wefax_rc.phasing_lines = (uint8_t)atoi( line );
  if( (wefax_rc.phasing_lines < 10) || (wefax_rc.phasing_lines > 60) )
  {
    Close_File( &wefaxrc );
    Wefax_Show_Message(
        _("Error reading Default\n"
          "Number of Phasing Lines\n"\
          "Quit and correct wefaxrc"), "red" );
    Error_Dialog(
        _("Error reading Default\n"
          "Number of Phasing Lines\n"\
          "Quit and correct wefaxrc"), HIDE_OK );
    return( FALSE );
  }

  // Read default image enhancement algorithm, abort if EOF
  if( Read_Line(line, wefaxrc, _("Wefax Image Enhancement")) != SUCCESS )
    return( FALSE );
  wefax_rc.image_enhance = (uint8_t)atoi( line );

  // Read main Window's position and place it
  if( Read_Line(line, wefaxrc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(wefax_gui.window), x, y );

  // Form the wefax home directory
  Strlcpy( wefax_rc.wefax_dir, getenv("HOME"),  sizeof(wefax_rc.wefax_dir) );
  Strlcat( wefax_rc.wefax_dir, WEFAX_DIRECTORY, sizeof(wefax_rc.wefax_dir) );

  // Form the wefax stations file name
  Strlcpy( wefax_rc.bookmarks_file, getenv("HOME"), sizeof(wefax_rc.bookmarks_file) );
  Strlcat( wefax_rc.bookmarks_file, WEFAX_BMK_FILE, sizeof(wefax_rc.bookmarks_file) );

  // Initialize wefax runtime config
  wefax_rc.gui_init = True;
  Wefax_Configure();
  Wefax_Set_Menu_Items();
  Close_File( &wefaxrc );
  Alloc_Demod_Buf_Copies( SOUND_OP_BUF_SIZE );
  Flag[GUEST_DEMOD_IQ_DATA] = True;
  FM_Detector = FM_Detect_IQ;
  strncpy( wefax_rc.station_sideband, "USB", sizeof(wefax_rc.station_sideband) );

  return( FALSE );
} // End of Wefax_Read_Config()

//------------------------------------------------------------------

/* New_Lines_Per_Min()
 *
 * Initializes parameters on new RPM selected by user
 */
  void
New_Lines_Per_Min( void )
{
  const uint8_t rpm[NUM_RPM] =
  { RPM60, RPM90, RPM100, RPM120, RPM180, RPM240 };

  char name[8];
  uint8_t idx;

  // Find active RPM menu item
  for( idx = 0; idx < NUM_RPM; idx++ )
  {
    snprintf( name, sizeof(name), "rpm%d", rpm[idx] );
    name[7] = '\0';
    if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(wefax_gui.popup_menu_builder, name))) )
      break;
  }
  if( idx == NUM_RPM ) return;

  // Enter user selected value of RPM
  wefax_rc.lines_per_min = (double)rpm[idx];
  Wefax_Configure();

} // New_Lines_Per_Min()

//------------------------------------------------------------------

/* New_Pixels_Per_Line()
 *
 * Initializes parameters on new line resolution
 * (pixels per line) selected by the user
 */
  void
New_Pixels_Per_Line( void )
{
  const uint16_t pix[NUM_PIX] = { PIX600, PIX1200 };

  char name[9];
  uint8_t idx;

  // Find active pixels per line menu item
  for( idx = 0; idx < NUM_PIX; idx++ )
  {
    snprintf( name, sizeof(name), "pix%d", pix[idx] );
    name[7] = '\0';
    if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(wefax_gui.popup_menu_builder, name))) )
      break;
  }
  if( idx == NUM_PIX ) return;

  // Enter user selected value of resolution
  wefax_rc.pixels_per_line = pix[idx];
  Wefax_Configure();

} // New_Pixels_Per_Line()

//------------------------------------------------------------------

/* New_IOC()
 *
 * Initializes parameters on new
 * IOC value selected by the user
 */
  void
New_IOC( void )
{
  const uint16_t ioc[NUM_IOC] = { IOC288, IOC576 };
  const uint16_t stn[NUM_IOC] = { IOC288_START_TONE, IOC576_START_TONE };

  char name[10];
  uint8_t idx;

  // Find active IOC menu item
  for( idx = 0; idx < NUM_IOC; idx++ )
  {
    snprintf( name, sizeof(name), "ioc%d", ioc[idx] );
    name[7] = '\0';
    if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(wefax_gui.popup_menu_builder, name))) )
      break;
  }
  if( idx == NUM_IOC ) return;

  // Enter user selected value of IOC
  wefax_rc.ioc_value  = ioc[idx];
  wefax_rc.start_tone = stn[idx];

  // Period of start and stop tones in pixels
  double temp = wefax_rc.lines_per_min / 60.0; // lines/sec
  wefax_rc.start_tone_period =
    temp * (double)wefax_rc.pixels_per_line / (double)wefax_rc.start_tone;
  wefax_rc.stop_tone_period =
    temp * (double)wefax_rc.pixels_per_line / WEFAX_STOP_TONE;

} // New_IOC()

//------------------------------------------------------------------

/* New_Phasing_Lines()
 *
 * Initializes parameters on new number
 * of phasing lines selected by the user
 */
  void
New_Phasing_Lines( void )
{
  const uint8_t phl[NUM_PHL] = { PHL10, PHL20, PHL40, PHL60 };

  char name[10];
  uint8_t idx;

  // Find active phasing lines menu item
  for( idx = 0; idx < NUM_PHL; idx++ )
  {
    snprintf( name, sizeof(name), "phl%d", phl[idx] );
    name[7] = '\0';
    if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(wefax_gui.popup_menu_builder, name))) )
      break;
  }
  if( idx == NUM_PHL ) return;

  // Enter user selected value of phasing lines
  wefax_rc.phasing_lines = phl[idx];

} // New_Phasing_Lines()

//------------------------------------------------------------------

/* New_Image_Enhance()
 *
 * Initializes parameters on new image
 * enhancement algorithm selected by the user
 */
  void
New_Image_Enhance( void )
{
  const uint8_t ime[NUM_IME] = { IME0, IME1, IME2 };

  char name[8];
  uint8_t idx;

  // Find active phasing lines menu item
  for( idx = 0; idx < NUM_IME; idx++ )
  {
    snprintf( name, sizeof(name), "ime%d", ime[idx] );
    name[7] = '\0';
    if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(wefax_gui.popup_menu_builder, name))) )
      break;
  }
  if( idx == NUM_IME ) return;

  // Enter user selected value of phasing lines
  wefax_rc.image_enhance = ime[idx];

} // New_Image_Enhance()

//------------------------------------------------------------------

/* Wefax_Configure()
 *
 * Initializes wefax after change of parameters
 */
  void
Wefax_Configure( void )
{
  static uint16_t pixels_per_line = 0;
  static double lines_per_min     = 0.0;
  BOOLEAN new_lpm = False,  new_ppl = False;

  // (Re)-initialize when needed
  if( wefax_rc.gui_init )
  {
    pixels_per_line = 0;
    lines_per_min   = 0.0;
    new_lpm = False;
    new_ppl = False;
    wefax_rc.gui_init = False;
  }

  // Initialize on change of resolution
  if( pixels_per_line != wefax_rc.pixels_per_line )
  {
    pixels_per_line = wefax_rc.pixels_per_line;
    wefax_rc.pixels_per_line2 = wefax_rc.pixels_per_line / 2;
    new_ppl = True;
  }

  // Initialize on change of RPM
  if( (lines_per_min < wefax_rc.lines_per_min) ||
      (lines_per_min > wefax_rc.lines_per_min) )
  {
    lines_per_min = wefax_rc.lines_per_min;
    new_lpm = True;
  }

  // Initialize on change of resolution
  if( new_ppl )
  {
    /* We need a triple-sized buffer to avoid over-
     * running pixels after the buffer's output index */
    wefax_rc.line_buffer_size = 2 * pixels_per_line;

    // Allocate line buffer
    Mem_Realloc(
        (void **) &wefax_display.line_buffer,
        (size_t)wefax_rc.line_buffer_size );
    memset( wefax_display.line_buffer, 0, (size_t)wefax_rc.line_buffer_size );

    // Create pixbuff for WEFAX images
    if( wefax_display.pixbuf != NULL )
    {
      g_object_unref( wefax_display.pixbuf );
      wefax_display.pixbuf = NULL;
    }
    wefax_display.pixbuf = gdk_pixbuf_new(
        GDK_COLORSPACE_RGB, FALSE, 8,
        pixels_per_line, wefax_rc.image_lines );

    // Error, not enough memory
    if( wefax_display.pixbuf == NULL)
    {
      Wefax_Show_Message(
          _("Failed to Allocate Memory to Pixbuf\n"
            "Please Quit and correct"), "red" );
      Error_Dialog(
          _("Failed to Allocate Memory to Pixbuf\n"
            "Please Quit and correct"), HIDE_OK );
      return;
    }

    // Fill pixbuf with background color
    gdk_pixbuf_fill( wefax_display.pixbuf, PIXBUF_BACKGND );

    // Get details of pixbuf
    wefax_display.pixel_buf  = gdk_pixbuf_get_pixels( wefax_display.pixbuf );
    wefax_display.rowstride  = gdk_pixbuf_get_rowstride( wefax_display.pixbuf );
    wefax_display.n_channels = gdk_pixbuf_get_n_channels( wefax_display.pixbuf );

    // Globalize drawingarea to be displayed
    wefax_gui.drawingarea =
      Builder_Get_Object( wefax_gui.window_builder, "wefax_drawingarea" );
    gtk_widget_set_size_request(
        wefax_gui.drawingarea,
        pixels_per_line,
        wefax_rc.image_lines );
    gtk_widget_show( wefax_gui.drawingarea );

    // Set window size as required
    // The scrolled window image container
    GtkWidget *image_scroller =
      Builder_Get_Object( wefax_gui.window_builder, "wefax_image_scrolledwindow" );
    gtk_widget_set_size_request(
        image_scroller, -1, wefax_rc.window_height );
    gtk_window_resize( GTK_WINDOW(wefax_gui.window), 10, 10 );

    // Re-initialize line buffer indices
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    // Signal WEFAX decoder to reset
    Flag[WEFAX_START_NEW_IMAGE] = True;
  } // if( new_ppl )

  // Initialize on change of RPM or resolution
  if( new_lpm || new_ppl )
  {
    // Length (duration) of an image pixel in DSP samples
    double temp = wefax_rc.lines_per_min / 60.0; // lines/sec
    if( temp != 0.0 ) temp = SND_DSP_RATE / temp; // samples/line

    // Add sync slant correction to pixel length
    wefax_rc.pixel_len = temp;
    temp = (double)wefax_rc.pixels_per_line + wefax_rc.sync_slant;

    // Samples/pixel as a float
    if( temp != 0.0 )
      wefax_rc.pixel_len = wefax_rc.pixel_len / temp;

    // Period of start and stop tones in pixels
    temp = lines_per_min / 60.0; // lines/sec
    wefax_rc.start_tone_period =
      temp * (double)pixels_per_line / (double)wefax_rc.start_tone;
    wefax_rc.stop_tone_period =
      temp * (double)pixels_per_line / WEFAX_STOP_TONE;

  } // if( if( new_lpm || new_ppl )

} // Wefax_Configure()

//------------------------------------------------------------------------

/* Wefax_File_Name()
 *
 * Prepare a file name, using date and time
 */
  void
Wefax_File_Name( char *file_name, const char *extn )
{
  size_t len; // String length of file_name

  // Variables for reading time (UTC)
  time_t tp;
  struct tm utc;

  // Prepare a file name as UTC date-time.
  // Default paths are images/ and record/
  time( &tp );
  utc = *gmtime( &tp );
  Strlcpy( file_name, wefax_rc.wefax_dir, FILE_NAME_SIZE - 28 );
  len = strlen( file_name );
  strftime( &file_name[len], 27, "images/%d%b%Y-%H%MUTC.", &utc );
  len = strlen( file_name );
  Strlcat( &file_name[len], extn, 4 );

} // Wefax_File_Name()

//------------------------------------------------------------------------

/* Wefax_Save_Image_PGM()
 *
 * Write an image buffer to file
 */
  BOOLEAN
Wefax_Save_Image_PGM(
    FILE **fp,
    const char *type,
    uint16_t width,
    uint16_t height,
    uint16_t max_val,
    const uint8_t *buffer )
{
  size_t size;

  // Write header in Ch-A output PGM files
  if( fprintf(*fp, "%s\n%s\n%d %d\n%d\n",
        type, _("# Created by wefax"), width, height, max_val) < 0 )
  {
    Close_File( fp );
    perror( "wefax: Error writing Image to file" );
    Wefax_Set_Indicators( WEFAX_ICON_SAVE_NO );
    Wefax_Show_Message( _("Error writing Image to file"), "red" );
    Error_Dialog( _("Error writing Image to file"), HIDE_OK );
    return( False );
  }

  // P6 type (PGM) files are 3* size in pixels
  if( strcmp(type, "P6") == 0 )
    size = (size_t)(3 * width * height);
  else
    size = (size_t)(width * height);

  // Write image buffer to file, abort on error
  if( fwrite(buffer, 1, size, *fp) != size )
  {
    Close_File( fp );
    perror( "wefax: Error writing Image to file" );
    Wefax_Set_Indicators( WEFAX_ICON_SAVE_NO );
    Wefax_Show_Message( _("Error writing Image to file"), "red" );
    Error_Dialog( _("Error writing Image to file"), HIDE_OK );
    return( False );
  }

  Wefax_Set_Indicators( WEFAX_ICON_SAVE_APPLY );
  Wefax_Show_Message( _("Image File saved OK"), "green" );
  Close_File( fp );

  return( True );
} // Wefax_Save_Image_PGM()

//------------------------------------------------------------------------

/* Wefax_Save_Image_JPEG()
 *
 * Save an image buffer to file in jpeg format
 */
  BOOLEAN
Wefax_Save_Image_JPEG(
    FILE **fp,
    int width,
    int height,
    const uint8_t *buffer )
{
  int len;
  int wdt, hgt;
  size_t siz;
  uint8_t *buff = NULL;

  // Make the buffer size as multiple of 8x8 block
  wdt = ( width - 1 ) / 8;
  wdt = ( wdt + 1 ) * 8;
  hgt = ( height - 1 ) / 8;
  hgt = ( hgt + 1 ) * 8;

  // Allocate temp buffer to multiple of 8x8
  siz = (size_t)wdt * (size_t)hgt * sizeof(uint8_t);
  Mem_Alloc( (void **) &buff, siz );

  // Clear temp buffer and copy image data
  memset( buff, 0xff, siz );
  siz = (size_t)width * (size_t)height * sizeof(uint8_t);
  memcpy( buff, buffer, siz );

  // Create a jpeg encoder
  jpec_enc_t *enc = jpec_enc_new( buff, wdt, hgt );

  // Run encoder to create jpeg image
  const uint8_t *jpeg = jpec_enc_run( enc, &len );
  siz = (size_t)len;

  // Write image buffer to file, abort on error
  if( fwrite(jpeg, sizeof(uint8_t), siz, *fp) != siz )
  {
    Close_File( fp );
    Mem_Free( (void **) &buff );
    perror( _("mlrpt: Error writing image to file") );
    Wefax_Show_Message( _("Error writing image to file"), "red" );
    return( False );
  }

  // Clean up and close file
  jpec_enc_del( enc );
  Mem_Free( (void **) &buff );
  Wefax_Show_Message( _("Image File saved OK"), "green" );
  Close_File( fp );

  return( True );
} // Wefax_Save_Image_JPEG()

//------------------------------------------------------------------------

typedef struct _WEFAX_MESG
{
  char *mesg;
  char *attr;
} wefax_mesg_t;

/* _Wefax_Show_Message()
 *
 * Idle callback to print message strings in the Text View scroller
 */
  static gboolean
_Wefax_Show_Message( gpointer data )
{
  static GtkTextIter iter;

  // (Re)-initialize
  if( wefax_rc.mesg_init )
  {
    gtk_text_buffer_get_iter_at_offset( wefax_gui.text_buffer, &iter, 0 );
    wefax_rc.mesg_init = False;
  }

  // Print message, don't use color tags for black text
  if( strncmp(((wefax_mesg_t *)data)->attr, "black",
        strlen(((wefax_mesg_t *)data)->attr)) == 0 )
  {
    gtk_text_buffer_insert(
        wefax_gui.text_buffer, &iter, ((wefax_mesg_t *)data)->mesg, -1 );
  }
  else
  {
    gtk_text_buffer_insert_with_tags_by_name(
        wefax_gui.text_buffer, &iter,
        ((wefax_mesg_t *)data)->mesg, -1,
        ((wefax_mesg_t *)data)->attr, NULL );
  }
  gtk_text_buffer_insert( wefax_gui.text_buffer, &iter, "\n", -1 );

  // Scroll Text View to bottom
  gtk_text_view_scroll_mark_onscreen(
      wefax_gui.text_view, gtk_text_buffer_create_mark(
        wefax_gui.text_buffer, "end", &iter, FALSE) );

  return( FALSE );
} // _Wefax_Show_Message()

// Locks the thread that invokes the idle callback
static sem_t mesg_sem;

// Unlocks the thread above when the idle callback completes
  static void
Unlock_Mesg( gpointer data )
{
  int sval;
  sem_getvalue( &mesg_sem, &sval );
  if( !sval ) sem_post( &mesg_sem );
}

/* Wefax_Show_Message()
 *
 * Uses the Wefax_Show_Message() idle callback above to show messages
 */
  void
Wefax_Show_Message( char *mesg, char *attr )
{
  static wefax_mesg_t message;

  message.mesg = mesg;
  message.attr = attr;

  // Create semaphore to lock idle add thread
  Init_Semaphore( &mesg_sem, True );

  g_idle_add_full(
      G_PRIORITY_HIGH_IDLE,
      _Wefax_Show_Message,
      (gpointer)&message,
      Unlock_Mesg );

  // Wait for idle callback function to exit
  while( g_main_context_iteration(NULL, FALSE) );
  sem_wait( &mesg_sem );

  // Free semaphore
  Init_Semaphore( &mesg_sem, False );

} // Wefax_Show_Message()

//------------------------------------------------------------------------

