/*
 * Copyright (c) 2014 Moodstocks SAS
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef WEFAX_JPEG_H
#define WEFAX_JPEG_H   1

#include <stdint.h>

#define JPEC_BUFFER_INIT_SIZ    65536
#define JPEG_ENC_DEF_QUAL       80  // default quality factor
#define JPEC_ENC_HEAD_SIZ       330 // header typical size in bytes
#define JPEC_ENC_BLOCK_SIZ      30  // 8x8 entropy coded block typical size in bytes

//* Structure used to hold and process an image 8x8 block
typedef struct _JPEG_BLOCK
{
  float dct[64];    // DCT coefficients
  int quant[64];    // Quantization coefficients
  int zz[64];       // Zig-Zag coefficients
  int len;          // Length of Zig-Zag coefficients
} jpec_block_t;

//* Extensible byte buffer
typedef struct _JPEG_BUF
{
  uint8_t *stream; // byte buffer
  int len;         // current length
  int siz;         // maximum size
} jpec_buffer_t;

//* Skeleton for an Huffman entropy coder
typedef struct _JPEG_HUFF_SKEL
{
  void *opq;
  void (*del)(void *opq);
  void (*encode_block)(void *opq, jpec_block_t *block, jpec_buffer_t *buf);
} jpec_huff_skel_t;

//* Entropy coding data that hold state along blocks
typedef struct _JPEG_HUFF_STATE
{
  int32_t buffer;       // bits buffer
  int nbits;            // number of bits remaining in buffer
  int dc;               // DC coefficient from previous block (or 0)
  jpec_buffer_t *buf;  // JPEG global buffer
} jpec_huff_state_t;

//* Type of an Huffman JPEG encoder
typedef struct _JPEG_HUFF
{
  jpec_huff_state_t state;    // state from previous block encoding
} jpec_huff_t;

//* JPEG encoder
typedef struct _JPEC_ENC
{
  //* Input image data
  const uint8_t *img;    // image buffer
  int w;            // image width
  int h;            // image height
  int w8;           // w rounded to upper multiple of 8

  //* JPEG extensible byte buffer
  jpec_buffer_t *buf;

  //* Compression parameters
  int qual;       // JPEG quality factor
  int dqt[64];    // scaled quantization matrix

  //* Current 8x8 block
  int bmax;             // maximum number of blocks (N)
  int bnum;             // block number in 0..N-1
  int bx;          // block start X
  int by;          // block start Y
  jpec_block_t block;   // block data

  //* Huffman entropy coder
  jpec_huff_skel_t *hskel;
} jpec_enc_t;

// jpeg.c
jpec_enc_t *jpec_enc_new(const uint8_t *img, int w, int h);
void jpec_enc_del(jpec_enc_t *e);
const uint8_t *jpec_enc_run(jpec_enc_t *e, int *len);

#endif

