/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/display.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/sound.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

#define BLACK_CUT_OFF   5 // Black cut-off percentile for normalization
#define WHITE_CUT_OFF  40 // White cut-off percentile for normalization

// Colors used in the level gauge
#define GAUGE_WHITE     1.0, 1.0, 1.0
#define GAUGE_RED       0.8, 0.0, 0.0
#define GAUGE_YELLOW    0.8, 0.8, 0.0
#define GAUGE_GREEN     0.0, 0.8, 0.0
#define GAUGE_GREY      0.3, 0.3, 0.3

// iFFT parameter definitions
#define WEFAX_IFFT_UPPER_FREQ   2400 // Frequency at upper end of FFT display
#define WEFAX_IFFT_LOWER_FREQ   1200 // Frequency at lower end of FFT display

//------------------------------------------------------------------

/* Wefax_Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */
  void
Wefax_Display_Waterfall( void )
{
  uint16_t
    idh, idv,  // Index to hor. and vert. position in warterfall
    idf, idb,  // Indices to ifft arrays
    temp;

  // Constants needed to draw white lines in waterfall
  static uint16_t
    white_line = 0,
    black_line = 0;

  static uint16_t
    white_freq = 0, // To initialize the function
    black_freq = 0;

  // Pointer to current pixel
  static guchar *pix;

  // Constants needed to draw white lines in waterfall
  if( (white_freq != WEFAX_WHITE_FREQ) ||
      (black_freq != WEFAX_BLACK_FREQ) )
  {
    uint16_t delta_f = 0;
    white_freq = WEFAX_WHITE_FREQ;
    black_freq = WEFAX_BLACK_FREQ;

    delta_f = white_freq - WEFAX_IFFT_LOWER_FREQ;
    white_line =
      ( delta_f * (uint16_t)wefax_wfall.width ) /
      ( WEFAX_IFFT_UPPER_FREQ - WEFAX_IFFT_LOWER_FREQ );

    delta_f = black_freq - WEFAX_IFFT_LOWER_FREQ;
    black_line =
      ( delta_f * (uint16_t)wefax_wfall.width ) /
      ( WEFAX_IFFT_UPPER_FREQ - WEFAX_IFFT_LOWER_FREQ );
  }

  // Draw a vertical white line in waterfall at detector's freq.
  pix = wefax_wfall.pixels + wefax_wfall.rowstride + wefax_wfall.n_channels;
  pix += wefax_wfall.n_channels * white_line;
  pix[0] = pix[1] = pix[2] = 0xff;
  pix = wefax_wfall.pixels + wefax_wfall.rowstride + wefax_wfall.n_channels;
  pix += wefax_wfall.n_channels * black_line;
  pix[0] = pix[1] = pix[2] = 0xff;

  // Copy each line of waterfall to next one
  temp = (uint16_t)wefax_wfall.height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
    pix = wefax_wfall.pixels +
      wefax_wfall.rowstride * idv + wefax_wfall.n_channels;

    for( idh = 0; idh < wefax_wfall.width; idh++ )
    {
      pix[0] = pix[- wefax_wfall.rowstride];
      pix[1] = pix[1 - wefax_wfall.rowstride];
      pix[2] = pix[2 - wefax_wfall.rowstride];
      pix   += wefax_wfall.n_channels;
    }
  }

  // Got to top left of pixbuf
  pix = wefax_wfall.pixels + wefax_wfall.rowstride;

  // Do the DFT on input array
  idb = 0;
  iFFT_Real( &wefax_ifft_data );
  for( idf = wefax_ifft_data.data_len / 2; idf < wefax_ifft_data.data_len; idf += 2 )
  {
    // Calculate power of signal at each freq. ("bin")
    uint16_t pixel_val = iFFT_Bin_Value( &wefax_ifft_data, idf, False );

    // Calculate average bin values
    wefax_ifft_data.bin_ave[idb++] = pixel_val;

    // Color code signal strength
    Colorize( pix, (uint8_t)pixel_val );
    pix += wefax_wfall.n_channels;

  } // for( idf = wefax_ifft_data.data_len / 2; idf < wefax_ifft_data.data_len; idf += 2+ )

  // Reset function
  iFFT_Bin_Value( &wefax_ifft_data, 0, True );

  // At last draw waterfall
  Queue_Draw( wefax_gui.spectrum_drawingarea );

} // Wefax_Display_Waterfall()

//------------------------------------------------------------------------

// Points to plot
static GdkPoint *points = NULL;

/* Wefax_Display_Signal()
 *
 * Updates the signal scope display
 */
  void
Wefax_Display_Signal( uint8_t plot )
{
  static uint16_t
    height = 0,     // Height of scope window
    limit,          // Limit plotting to inside of scope margin
    points_idx = 0; // Index to points array

  // Initialize on first call
  if( points == NULL )
  {
    Mem_Alloc( (void **) &points, sizeof(GdkPoint) * (size_t)wefax_scope_width );
  }

  // Initialize on parameter change
  if( height != wefax_scope_height )
  {
    height = (uint16_t)wefax_scope_height;
    limit = height - SCOPE_CLEAR;
  }

  // Save values to be plotted
  plot = (uint8_t)( ((uint16_t)plot * limit ) / 255 );
  points[points_idx].y = limit - (gint)plot;
  if( points[points_idx].y < SCOPE_CLEAR )
    points[points_idx].y = SCOPE_CLEAR;
  points[points_idx].x = points_idx;

  // Recycle buffer idx when full and plot
  points_idx++;
  if( points_idx >= wefax_scope_width )
  {
    Flag[GUEST_ENABLE_SCOPE] = True;
    Queue_Draw( wefax_gui.scope_drawingarea );
    points_idx = 0;
  } // if( points_idx == wfall_pixbuf.width )

} // Wefax_Display_Signal( void )

//------------------------------------------------------------------------

/* Wefax_Draw_Signal()
 *
 * Draws the signal detector's output
 */
  void
Wefax_Draw_Signal( cairo_t *cr )
{
  uint16_t idx;

  // Draw scope backgrounds
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)wefax_scope_width,
      (double)wefax_scope_height );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_source_rgb( cr, SCOPE_FOREGND );

  cairo_move_to( cr,
      (double)points[0].x,
      (double)points[0].y );
  for( idx = 1; idx < wefax_scope_width; idx++ )
    cairo_line_to( cr,
        (double)points[idx].x,
        (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );

  Flag[GUEST_ENABLE_SCOPE] = False;

} // Wefax_Draw_Signal()

//------------------------------------------------------------------------

/* Wefax_Set_Indicators()
 *
 * Sets the indicator icons ("LEDs") in the Control frame
 */
  void
Wefax_Set_Indicators( uint8_t flag )
{
  GtkWidget   *icon = NULL;
  const gchar *name = NULL;

  // Act according to icon select flag
  switch( flag )
  {
    case WEFAX_ICON_START_YES:
      icon = wefax_gui.start_icon;
      name = "gtk-yes";
      break;

    case WEFAX_ICON_SYNC_YES:
      icon = wefax_gui.sync_icon;
      name = "gtk-yes";
      break;

    case WEFAX_ICON_DECODE_YES:
      icon = wefax_gui.decode_icon;
      name = "gtk-yes";
      break;

    case WEFAX_ICON_SAVE_YES:
      icon = wefax_gui.save_icon;
      name = "gtk-yes";
      break;

    case WEFAX_ICON_START_NO:
      icon = wefax_gui.start_icon;
      name = "gtk-no";
      break;

    case WEFAX_ICON_SYNC_NO:
      icon = wefax_gui.sync_icon;
      name = "gtk-no";
      break;

    case WEFAX_ICON_DECODE_NO:
      icon = wefax_gui.decode_icon;
      name = "gtk-no";
      break;

    case WEFAX_ICON_SAVE_NO:
      icon = wefax_gui.save_icon;
      name = "gtk-no";
      break;

    case WEFAX_ICON_START_SKIP:
      icon = wefax_gui.start_icon;
      name = "gtk-cancel";
      break;

    case WEFAX_ICON_SYNC_SKIP:
      icon = wefax_gui.sync_icon;
      name = "gtk-cancel";
      break;

    case WEFAX_ICON_DECODE_SKIP:
      icon = wefax_gui.decode_icon;
      name = "gtk-cancel";
      break;

    case WEFAX_ICON_SAVE_SKIP:
      icon = wefax_gui.save_icon;
      name = "gtk-cancel";
      break;

    case WEFAX_ICON_START_APPLY:
      icon = wefax_gui.start_icon;
      name = "gtk-apply";
      break;

    case WEFAX_ICON_SYNC_APPLY:
      icon = wefax_gui.sync_icon;
      name = "gtk-apply";
      break;

    case WEFAX_ICON_DECODE_APPLY:
      icon = wefax_gui.decode_icon;
      name = "gtk-apply";
      break;

    case WEFAX_ICON_SAVE_APPLY:
      icon = wefax_gui.save_icon;
      name = "gtk-apply";
      break;

    default: return;

  } // switch( flag )

  Set_Icon( GTK_IMAGE(icon), name, GTK_ICON_SIZE_BUTTON );

} // Wefax_Set_Indicators()

//------------------------------------------------------------------------

/* Wefax_Set_Menu_Items()
 *
 * Sets the appropriate menu items to reflect
 * the settings in the configuration file
 */
  void
Wefax_Set_Menu_Items( void )
{
  GtkCheckMenuItem *item;

  const uint8_t  rpm[NUM_RPM] = { RPM60, RPM90, RPM100, RPM120, RPM180, RPM240 };
  const uint16_t pix[NUM_PIX] = { PIX600, PIX1200 };
  const uint16_t ioc[NUM_IOC] = { IOC288, IOC576 };
  const uint8_t  phl[NUM_PHL] = { PHL10, PHL20, PHL40, PHL60 };
  const uint8_t  ime[NUM_IME] = { IME0, IME1, IME2 };

#define NAME_SIZE    8
  char name[NAME_SIZE+1];
  uint8_t idx;

  // Find current RPM value
  for( idx = 0; idx < NUM_RPM; idx++ )
    if( rpm[idx] == (int)wefax_rc.lines_per_min )
      break;
  if( idx == NUM_RPM ) return;

  // Set active RPM menu item
  snprintf( name, sizeof(name), "rpm%d", rpm[idx] );
  name[NAME_SIZE] = '\0';
  item = GTK_CHECK_MENU_ITEM(
      Builder_Get_Object(wefax_gui.popup_menu_builder, name) );
  gtk_check_menu_item_set_active( item, TRUE );

  // Find current resolution value
  for( idx = 0; idx < NUM_PIX; idx++ )
    if( pix[idx] == wefax_rc.pixels_per_line )
      break;
  if( idx == NUM_PIX ) return;

  // Set active resolution menu item
  snprintf( name, sizeof(name), "pix%d", pix[idx] );
  name[NAME_SIZE] = '\0';
  item = GTK_CHECK_MENU_ITEM(
      Builder_Get_Object(wefax_gui.popup_menu_builder, name) );
  gtk_check_menu_item_set_active( item, TRUE );

  // Find current IOC value
  for( idx = 0; idx < NUM_IOC; idx++ )
    if( ioc[idx] == wefax_rc.ioc_value )
      break;
  if( idx == NUM_IOC ) return;

  // Set active IOC menu item
  snprintf( name, sizeof(name), "ioc%d", ioc[idx] );
  name[NAME_SIZE] = '\0';
  item = GTK_CHECK_MENU_ITEM(
      Builder_Get_Object(wefax_gui.popup_menu_builder, name) );
  gtk_check_menu_item_set_active( item, TRUE );

  // Find current phasing lines value
  for( idx = 0; idx < NUM_PHL; idx++ )
    if( phl[idx] == wefax_rc.phasing_lines )
      break;
  if( idx == NUM_PHL ) return;

  // Set active phasing lines menu item
  snprintf( name, sizeof(name), "phl%d", phl[idx] );
  name[NAME_SIZE] = '\0';
  item = GTK_CHECK_MENU_ITEM(
      Builder_Get_Object(wefax_gui.popup_menu_builder, name) );
  gtk_check_menu_item_set_active( item, TRUE );

  // Find current image enhancement setting
  for( idx = 0; idx < NUM_IME; idx++ )
    if( ime[idx] == wefax_rc.image_enhance )
      break;
  if( idx == NUM_IME ) return;

  // Set active image enhancement menu item
  snprintf( name, sizeof(name), "ime%d", ime[idx] );
  name[NAME_SIZE] = '\0';
  item = GTK_CHECK_MENU_ITEM(
      Builder_Get_Object(wefax_gui.popup_menu_builder, name) );
  gtk_check_menu_item_set_active( item, TRUE );

} // Wefax_Set_Menu_Items()

//------------------------------------------------------------------------

/* Normalize()
 *
 * Does histogram (linear) normalization of a WEFAX line
 */
  void
Normalize( uint8_t *line_buf, uint32_t line_len )
{
  int32_t
    hist[256],  // Intensity histogram of pgm image file
    blk_cutoff, // Count of pixels for black cutoff value
    wht_cutoff, // Count of pixels for white cutoff value
    pixel_cnt;  // Total pixels counter for cut-off point

  uint32_t idx; // Index for loops etc

  int16_t
    black_val,  // Black cut-off pixel intensity value
    white_val,  // White cut-off pixel intensity value
    val_range;  // Range of intensity values in image

  // Clear histogram
  memset( (void *)hist, 0, sizeof(hist) );

  // Build image intensity histogram
  for( idx = 0; idx < line_len; idx++ )
    hist[line_buf[idx]] += 1;

  // Determine black/white cut-off counts
  blk_cutoff = (line_len * BLACK_CUT_OFF) / 100;
  wht_cutoff = (line_len * WHITE_CUT_OFF) / 100;

  // Find black cut-off intensity value
  pixel_cnt = 0;
  for( black_val = 0; black_val <= 255; black_val++ )
  {
    pixel_cnt += hist[black_val];
    if( pixel_cnt > blk_cutoff ) break;
  }

  // Find white cut-off intensity value
  pixel_cnt = 0;
  for( white_val = 255; white_val  >= 0; white_val-- )
  {
    pixel_cnt += hist[white_val];
    if( pixel_cnt > wht_cutoff ) break;
  }

  // Rescale pixels in image for full intensity range
  val_range = white_val - black_val;
  if( val_range <= 0 ) return;

  // Perform histogram normalization on images
  for( pixel_cnt = 0; pixel_cnt < (int32_t)line_len; pixel_cnt++ )
  {
    int32_t pixel_val = line_buf[pixel_cnt];
    pixel_val = ( (pixel_val - black_val) * 255 ) / val_range;

    pixel_val = ( pixel_val < 0 ? 0 : pixel_val );
    pixel_val = ( pixel_val > 255 ? 255 : pixel_val );
    line_buf[pixel_cnt] = (uint8_t)pixel_val;
  }

} // End of Normalize()

//------------------------------------------------------------------------

/* Wefax_Spectrum_Size_Allocate()
 *
 * Handles the size_allocate callback on spectrum drawingarea
 */
  void
Wefax_Spectrum_Size_Allocate( uint16_t width, uint16_t height )
{
  // Destroy existing pixbuff
  if( wefax_wfall.pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(wefax_wfall.pixbuf) );
    wefax_wfall.pixbuf = NULL;
  }

  // Create waterfall pixbuf
  wefax_wfall.pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8, width, height );
  if( wefax_wfall.pixbuf == NULL ) return;

  wefax_wfall.pixels = gdk_pixbuf_get_pixels( wefax_wfall.pixbuf );
  wefax_wfall.width  = (uint16_t)gdk_pixbuf_get_width ( wefax_wfall.pixbuf );
  wefax_wfall.height = (uint16_t)gdk_pixbuf_get_height( wefax_wfall.pixbuf );
  wefax_wfall.rowstride  = (uint16_t)gdk_pixbuf_get_rowstride( wefax_wfall.pixbuf );
  wefax_wfall.n_channels = (uint16_t)gdk_pixbuf_get_n_channels( wefax_wfall.pixbuf );
  gdk_pixbuf_fill( wefax_wfall.pixbuf, 0 );

  /* Calculate dft stride to put upper freq at end of DFT
   * display and allow a freq range max/min ratio of 2:1 */
  wefax_ifft_data.ifft_stride = SND_DSP_RATE / WEFAX_IFFT_UPPER_FREQ / 2;

  /* Initialize Ifft and config. ifft size is twice
   * the waterfall size because we only need to
   * display the upper half, from 1200 to 2400 Hz */
  wefax_ifft_data.ifft_width = (uint16_t)wefax_wfall.width * 2;
  Initialize_iFFT( &wefax_ifft_data );

} // Wefax_Spectrum_Size_Allocate()

//------------------------------------------------------------------------

/* Set_Sync_Slant()
 *
 * Sets the value of parameters used to deslant WEFAX image
 */
  void
Set_Sync_Slant( double sync_slant )
{
  // Pixels/1000 lines sync slant correction
  wefax_rc.sync_slant = sync_slant / 1000.0;

  // Length (duration) of an image pixel in DSP samples
  double temp = wefax_rc.lines_per_min / 60.0; // lines/sec
  if( temp != 0.0 )
    temp = SND_DSP_RATE / temp; // samples/line

  // Add sync slant correction to pixel length
  wefax_rc.pixel_len = temp;
  temp = (double)wefax_rc.pixels_per_line + wefax_rc.sync_slant;

  // Samples/pixel as a float
  if( temp != 0.0 ) wefax_rc.pixel_len = wefax_rc.pixel_len / temp;

} // Set_Sync_Slant()

//------------------------------------------------------------------------

/* Wefax_Display_Level_Gauge()
 *
 * Displays a level gauge for signal
 * strength | stop train | start train
 */
  void
Wefax_Display_Level_Gauge( cairo_t *cr )
{
  // Width and height of gauge widget
  int width = 0, height = 0, level;

  // Width and height of gauge bar
  static double bar_width = 0, bar_height = 0;

  /* It happens if the call is
   * from expose event callback */
  if( gauge_input < 0 ) gauge_input = 0;

  // Create level gauge
  if( wefax_rc.gauge_init )
  {
    // Create a Cairo context
    GtkAllocation alloc;
    gtk_widget_get_allocation( wefax_gui.level_gauge, &alloc );
    height = alloc.height;
    width  = alloc.width;
    bar_width  = (double)width  - 1.0;
    bar_height = (double)height - 1.0;
    wefax_rc.gauge_init = False;
  }

  // Create red bar graph of input level
  level = gauge_input;
  if( level > gauge_level1 ) level = gauge_level1;
  cairo_set_source_rgb( cr, GAUGE_RED );
  cairo_rectangle( cr,
      1.0, bar_height - (double)level,
      bar_width, (double)level );
  cairo_fill( cr );

  // Create yellow bar graph of input level
  level = gauge_input;
  if( level > gauge_level1 )
  {
    if( level > gauge_level2 ) level = gauge_level2;
    cairo_set_source_rgb( cr, GAUGE_YELLOW );
    cairo_rectangle( cr,
        1.0, bar_height - (double)level, bar_width,
        (double)level - (double)gauge_level1 );
    cairo_fill( cr );
  }

  // Create green bar graph of input level
  level = gauge_input;
  if( level > gauge_level2 )
  {
    if( level > (int)bar_height ) level = (int)bar_height;
    cairo_set_source_rgb( cr, GAUGE_GREEN );
    cairo_rectangle( cr,
        1.0, bar_height - (double)level,
        bar_width, (double)level - (double)gauge_level2 );
    cairo_fill( cr );
  }

  // Clear rest of bar graph
  cairo_set_source_rgb( cr, GAUGE_WHITE );
  cairo_rectangle( cr,
      1.0, 1.0, bar_width,
      bar_height - (double)level );
  cairo_fill( cr );

  // Create bar graph outline of sync level
  cairo_set_source_rgb( cr, GAUGE_GREY );
  cairo_rectangle( cr,
      0.0, 0.0,
      (double)width,
      (double)height );
  cairo_stroke( cr );

} // Wefax_Display_Level_Gauge()

//------------------------------------------------------------------------

/* Wefax_Tune_TRx()
 *
 * Tunes the transceiver to the frequency of the strongest
 * signal near a mouse click in the waterfall window
 */
  void
Wefax_Tune_TRx( double x )
{
  int32_t
    bin_max,    // Max bin value found in this range
    audio_freq; // Audio frequency in waterfal window

  int
    max_idx,    // dft idx at which the max is found
    from, to,   // Range to scan for a max bin value
    ifft_idx;   // Idx used to search ifft bin values

  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  /* Calculate dft index corresponding
   * to pointer x in waterfall window */
  ifft_idx = (int)(x - 0.5);

  // Look above and below click point for max bin val
  from = ifft_idx - 10;
  if( from < 0 ) from = 0;
  to = ifft_idx + 10;
  if( to >= wefax_wfall.width )
    to = wefax_wfall.width - 1;

  // Find max bin value around click point
  bin_max = 0;
  max_idx = ifft_idx;
  for( ifft_idx = from; ifft_idx < to; ifft_idx++ )
    if( bin_max < wefax_ifft_data.bin_ave[ifft_idx] )
    {
      bin_max = wefax_ifft_data.bin_ave[ifft_idx];
      max_idx = ifft_idx;
    }

  // Audio freq. corresponding to dft index
  audio_freq  = ( WEFAX_IFFT_UPPER_FREQ - WEFAX_IFFT_LOWER_FREQ ) * max_idx;
  audio_freq /= wefax_wfall.width;
  audio_freq += WEFAX_IFFT_LOWER_FREQ;

  // Add correction to TCVR frequency
  TRx->rx_frequency += (uint32_t)( audio_freq - WEFAX_WHITE_FREQ );
  Update_Spin_Dial( TRx, RX_FLAG );
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

} // Wefax_Tune_TRx()

//-------------------------------------------------------------------------

// Free resources

  void
Wefax_Free_Display( void )
{
  Mem_Free( (void **) &points );
  Free_iFFT( &wefax_ifft_data );
}

//------------------------------------------------------------------------

