/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/sound.h"
#include "../Hermes2/demodulate.h"
#include <gtk/gtk.h>
#include <math.h>
#include <semaphore.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

/* Number of Start/Stop tone periods * 60 sec
 * used in the Goertzel detector */
#define TONE_PERIOD_MULT    3600.0

// Length of Start/Stop tone averaging window
#define TONE_LEVEL_AVE_WIN  8

// Length of signal averaging window
#define SIG_AVE_WINDOW      20.0

// Scale factors for displaying signal level in the Gauge
#define SIG_GAUGE_SCALE     200
#define SIG_GAUGE_LEVEL1    32
#define SIG_GAUGE_LEVEL2    80

// Scale factors for displaying stop/start level in the Gauge
#define STOP_GAUGE_SCALE    2500
#define START_GAUGE_SCALE   2500

/* Count of calls to gauge display function
 * before the gauge display is refreshed */
#define GAUGE_COUNT         255

// To keep values of detected signal in reasonable limits
#define BILEVEL_SCALE_FACTOR    750.0

// Detection thresholds for start and stop tones
#define START_TONE_UP       300000
#define START_TONE_DOWN     100000
#define STOP_TONE_UP        300000
#define STOP_TONE_DOWN      100000

// Parameters to scale Zero Crossing detector into 0-255
#define ZERO_DET_FLOOR      500.0
#define ZERO_DET_SCALE      3.0

// Parameters to scale IQ Detect detector into 0-255
#define IQ_DETECT_FLOOR     400.0
#define IQ_DETECT_SCALE     3.14

//------------------------------------------------------------------------

/* Queue_Draw_Gauge()
 *
 * Queues the level gauge for drawing
 */
  static void
Queue_Draw_Gauge( void )
{
  static uint8_t cnt = 0;

  if( cnt++ == GAUGE_COUNT )
  {
    Queue_Draw( wefax_gui.level_gauge );
    cnt = 0;
  }
} // Queue_Draw_Gauge()

//------------------------------------------------------------------------

// Circular signal samples buffer for Goertzel detector
static int16_t *signal_buff = NULL;

//------------------------------------------------------------------------

/* FM_Detect_IQ()
 *
 * Estimates the frequency of the incoming WEFAX signal by calculating
 * the instantaneous phase of the carrier over one Sound sample duration
 * (1/48kHz). The difference in phase between subsequent samples gives
 * the frequency:
 * phi = atan2(i, q); dphi = phi-last-phi; freq = dphi * sample rate / 2pi
 */
  BOOLEAN
FM_Detect_IQ( uint8_t *signal_level )
{
  static double
    discrim_output,  // Output of FM detector (0-255)
    signal_freq;     // Measured frequency of incoming WEFAX signal

  /* Count of Audio samples used. It is a float as
   * for some modes, like SSTV, the pixel length
   * is not an integer number of Audio samples */
  static double samples_used_cnt = 0.0;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( False );

  // Wait for digimode buffer to fill with audio samples from demodulator.
  if( wait )
  {
    sem_wait( &digimode_semaphore );
    wait = False;
  }

  // Average frequency measurements over the length of a pixel
  while( samples_used_cnt < wefax_rc.pixel_len )
  {
    // Digimode buffer index
    static uint32_t digi_buf_idx = 0;

    // Get a audio sample from the digimodes buffer
    int16_t signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];

    // Save the sample to iFFT buffer
    if( iFFT_Data(signal_sample, &wefax_ifft_data) )
      Wefax_Display_Waterfall();

    /* Ring buffer for calculated phase angles. It seems that the effects of noise
     * induced jitter on delta phi (dphi) can be reduced if it is calculated from
     * values of phi separated by a few samples (here we use 5 stage ring buffer) */
    static double phi_buf[PHI_BUF_LENGTH];
    static uint8_t buf_idx = 0;
    double phi  = atan2( demod_id_cpy[digi_buf_idx], demod_qd_cpy[digi_buf_idx] );
    double dphi = phi - phi_buf[buf_idx];
    phi_buf[buf_idx] = phi;
    buf_idx++;
    if( buf_idx >= PHI_BUF_LENGTH ) buf_idx = 0;

    // Correct for discontinuity at +-180 deg phase angle
    CLAMP_PI( dphi );

    // Calculate signal frequency corresponding to delta phi
    double freq = dphi * SND_DSP_RATE / M_2PI / PHI_BUF_LENGTH;

    // Summate freequency over one pixel, for averaging
    signal_freq += freq;

    // Count DSP samples
    samples_used_cnt += 1.0;

    // Wait for digimode buffer to be filled
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      // Increment current audio buffer output index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      /* If sync between ring buffer input and output indices is lost,
       * sem_wait is not enabled, to allow the input index to catch up */
      if( digi_buf_input == digi_buf_cnt )
        wait = True;

      digi_buf_idx = 0;
      return( False );
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
  } // while( samples_used_cnt < wefax_rc.pixel_len )

  // Reset the samples index
  samples_used_cnt -= wefax_rc.pixel_len;

  // Scale and floor frequency to give a value 0-255
  signal_freq /= wefax_rc.pixel_len;
  discrim_output = ( signal_freq + IQ_DETECT_FLOOR ) / IQ_DETECT_SCALE;
  signal_freq = 0;

  // Limit disriminator output in right range
  if( discrim_output > 255.0 ) discrim_output = 255.0;
  if( discrim_output < 0.0 )   discrim_output = 0.0;
  *signal_level = (unsigned char)discrim_output;

  return( True );
} // FM_Detect_IQ()

//------------------------------------------------------------------------

/* FM_Detect_Zero_Crossing()
 *
 * Estimates the frequency of the incoming WEFAX audio signal by
 * counting Audio samples, up for +ve and down for -ve, up to a
 * maximum count of +- 1/4 of a cycle (of the highest (white)
 * frequency). This results in the count passing through zero
 * between the +ve and -ve half cycles of the signal and thus
 * a measure of the length of half a signal cycle is obtained.
 * From this the instantaneous signal frequency is calculated.
 */
  BOOLEAN
FM_Detect_Zero_Crossing( uint8_t *signal_level )
{
  static double
    discrim_output,           // Output of FM detector (0-255)
    zero_cross_interp = 0.0;  // Interpolation of zero crossing point

  /* Count of Audio samples used. It is a float as
   * for some modes, like SSTV, the pixel length
   * is not an integer number of Audio samples */
  static double samples_used_cnt = 0.0;

  static double
    zeros_period = 0.0, // Time elapsed between zeros, in Audio samples
    signal_freq  = 0.0, // Measured frequency of incoming WEFAX signal
    new_average  = 0.0, // New Audio samples average
    last_average = 0.0; // Last Audio samples average

  static int
    period_cnt_incr = 0,  // Number of increments to period counter
    pixel_num_zeros = 0;  // Number of zero crossings in a pixel

  /* These two variables are used to limit the effects of noise
   * by imposing a minimum count of Audio samples between zeros
   */
  // Minimum length of WEFAX signal 1/3 cycle in Audio samples
  static int min_cycle3 = SND_DSP_RATE / WEFAX_WHITE_FREQ / 3;

  // Count of samples between zero crossings
  static int inter_zero_samples = 0;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( False );

  // Wait for digimode buffer to fill with audio samples from demodulator.
  if( wait )
  {
    sem_wait( &digimode_semaphore );
    wait = False;
  }

  /* Look for a zero crossing of the WEFAX audio
   * signal over the duration of each image pixel */
  while( samples_used_cnt < wefax_rc.pixel_len )
  {
    // Digimode buffer index
    static uint32_t digi_buf_idx = 0;

    // Get a audio sample from the digimodes buffer
    int16_t signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];

    // Save the sample to iFFT buffer
    if( iFFT_Data(signal_sample, &wefax_ifft_data) )
      Wefax_Display_Waterfall();

    // Sliding widow average of DSP signal samples
    new_average  = new_average * ( SIG_AVE_WINDOW - 1.0 );
    new_average += (double)signal_sample;
    new_average /= SIG_AVE_WINDOW;

    // This gives us a zero crossing of the input waveform
    inter_zero_samples++;
    if( (new_average * last_average <= 0.0) &&
        (inter_zero_samples >= min_cycle3) )
    {
      // Signal frequency is 1/2 DSP rate / length of half cycle
      // Interpolate point of zero crossing
      if( (last_average - new_average) != 0.0 )
        zero_cross_interp = new_average / ( last_average - new_average );
      if( zero_cross_interp < -1.0 ) zero_cross_interp = -1.0;
      if( zero_cross_interp >  1.0 ) zero_cross_interp =  1.0;

      pixel_num_zeros++;
      period_cnt_incr    = 0;
      inter_zero_samples = 0;
    } // if( (new_average * last_average) < 0.0 )

    // Save current signal average
    last_average = new_average;

    // Count number of signal samples between zero crossings
    zeros_period += 1.0;
    period_cnt_incr++;

    // Count DSP samples
    samples_used_cnt += 1.0;

    // Wait for digimode buffer to be filled
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      // Increment current audio buffer output index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      /* If sync between ring buffer input and output indices is lost,
       * sem_wait is not enabled, to allow the input index to catch up */
      if( digi_buf_input == digi_buf_cnt )
        wait = True;

      digi_buf_idx = 0;
      return( False );
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
  } // while( samples_used_cnt < wefax_rc.pixel_len )

  // Add extrapolation of zero crossing
  if( pixel_num_zeros )
  {
    // Half the Audio sample rate, as a double
    static double sample_rate2 = SND_DSP_RATE / 2;

    // Calculate signal frequency from half cycle period
    zeros_period += zero_cross_interp;
    double half_cycle =
      ( zeros_period - (double)period_cnt_incr ) / (double)pixel_num_zeros;
    if( half_cycle != 0.0 )
      signal_freq = sample_rate2 / half_cycle;

    /* Prepares zeros_period to properly count
     * signal samples to next zero crossing */
    zeros_period = (double)period_cnt_incr - zero_cross_interp;
    period_cnt_incr = 0;
  }
  pixel_num_zeros = 0;

  // Reset the samples index
  samples_used_cnt -= wefax_rc.pixel_len;

  // Scale and floor frequency to give a value 0-255
  discrim_output = signal_freq / ZERO_DET_SCALE - ZERO_DET_FLOOR;

  // Limit disriminator output in right range
  if( discrim_output > 255.0 ) discrim_output = 255.0;
  if( discrim_output < 0.0 )   discrim_output = 0.0;
  *signal_level = (unsigned char)discrim_output;

  return( True );
} // FM_Detect_Zero_Crossing()

//------------------------------------------------------------------------

/* FM_Detect_Bilevel()
 *
 * Estimates the WEFAX input signal's frequency by comparing
 * the output of a Goertzel detector on the black frequency
 * (1500 Hz) and one on the white frequency (2300 Hz).
 */
  BOOLEAN
FM_Detect_Bilevel( uint8_t *signal_level )
{
  static BOOLEAN first_call = True;

  static uint16_t
    det_period, // Integration period of Goertzel detector
    signal_idx; // Signal samples buffer index

  static uint16_t
    black_level, // Level of the Black signal
    white_level, // Level of the White signal
    idx;

  // Variables for the Goertzel algorithm
  static double scale;
  double black_q1, black_q2;
  double white_q1, white_q2;

  /* Index of DSP samples used. It is a float as
   * for some modes, like SSTV, the pixel length
   * is not an integer number of DSP samples */
  static double samples_used_cnt = 0.0;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  static double white_coeff, black_coeff;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( False );

  // Initialize on first call
  if( first_call )
  {
    double w;
    double white_cosw, black_cosw;

    // Omega for the white frequency
    w = M_2PI / SND_DSP_RATE * WEFAX_WHITE_FREQ;
    white_cosw  = cos( w );
    white_coeff = 2.0 * white_cosw;

    // Omega for the black frequency
    w = M_2PI / SND_DSP_RATE * WEFAX_BLACK_FREQ;
    black_cosw  = cos( w );
    black_coeff = 2.0 * black_cosw;

    det_period = SND_DSP_RATE / (WEFAX_WHITE_FREQ - WEFAX_BLACK_FREQ);

    // To keep values of detected signal in reasonable limits
    scale = (double)det_period * BILEVEL_SCALE_FACTOR;

    // Allocate samples buffer and clear
    size_t len = sizeof(int16_t) * (size_t)det_period;
    Mem_Alloc( (void **) &signal_buff, len );
    memset( signal_buff, 0, len );
    signal_idx = 0;

    first_call = False;
  } // if( first_call... )

  // Wait for digimode buffer to fill with audio samples from demodulator.
  if( wait )
  {
    sem_wait( &digimode_semaphore );
    wait = False;
  }

  // Save samples for detector
  while( samples_used_cnt < wefax_rc.pixel_len )
  {
    static uint32_t digi_buf_idx = 0;

    // Get a audio sample from the digimodes buffer
    int16_t signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];
    signal_buff[signal_idx] = signal_sample;

    // Save the sample to iFFT buffer
    if( iFFT_Data(signal_sample, &wefax_ifft_data) )
      Wefax_Display_Waterfall();

    // Increment/reset circular buffer's index
    signal_idx++;
    if( signal_idx >= det_period ) signal_idx = 0;

    // Count DSP samples
    samples_used_cnt += 1.0;

    // Wait for digimode buffer to be filled
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      // Increment current audio sample output buffer index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      /* If sync between ring buffer input and output indices is lost,
       * sem_wait is not enabled, to allow the input index to catch up */
      if( digi_buf_input == digi_buf_cnt )
        wait = True;

      digi_buf_idx = 0;
      return( False );
    }
  } // while( samples_used_cnt < wefax_rc.pixel_len )

  // Reset the samples index
  samples_used_cnt -= wefax_rc.pixel_len;

  /* Calculate signal level of black and white
   * tone frequencies using Goertzel algorithm */
  black_q1 = black_q2 = 0.0;
  white_q1 = white_q2 = 0.0;
  for( idx = 0; idx < det_period; idx++ )
  {
    double black_q0 =
      black_coeff * black_q1 - black_q2 + (double)signal_buff[signal_idx];
    black_q2 = black_q1;
    black_q1 = black_q0;

    double white_q0 =
      white_coeff * white_q1 - white_q2 + (double)signal_buff[signal_idx];
    white_q2 = white_q1;
    white_q1 = white_q0;

    // Increment/reset circular buffers' index
    signal_idx++;
    if( signal_idx >= det_period ) signal_idx = 0;

  } // for( idx = 0; idx < det_period; idx++ )

  // Magnitude of black tone scaled by dot size and tone freq
  black_q1 /= scale;
  black_q2 /= scale;
  black_level = (uint16_t)
    ((black_q1 * black_q1 + black_q2 * black_q2 -
      black_q1 * black_q2 * black_coeff));

  // Magnitude of white tone scaled by dot size and tone freq
  white_q1 /= scale;
  white_q2 /= scale;
  white_level = (uint16_t)
    ( (white_q1 * white_q1 + white_q2 * white_q2 -
       white_q1 * white_q2 * white_coeff) );

  /* Calculate signal level according to ratio between
   * black and white Goertzel tone detector outputs */
  if( black_level > 8 * white_level ) *signal_level = 0;
  else if( black_level > 4 * white_level )
    *signal_level = 64;
  else if( white_level < 4 * black_level )
    *signal_level = 128;
  else if( white_level < 8 * black_level )
    *signal_level = 196;
  else *signal_level = 255;

  return( True );
} // FM_Detect_Bilevel()

//------------------------------------------------------------------------

/* Phasing_Detect()
 *
 * Detects phasing pulses using Goertzel's algorithm.
 */
  BOOLEAN
Phasing_Detect( void )
{
  static uint16_t
    pixels_per_line = 0, // WEFAX RPM or lines per minute
    pixel_idx       = 0, // Index to pixels in Wefax line
    phasing_cnt     = 0, // Count of phasing pulse lines examined
    pulse_max_idx   = 0; // Fragment count where maximum pulse level occurs

  // Average of phasing pulse over sliding window
  static double phasing_pulse_ave = 0.0;

  static int16_t
    phasing_error = 0, // The distance of phasing pulse from line middle
    phasing_ref   = 0, // Reference for initial position of phasing pulse
    error_limit,       // Max value of phasing pulse position error
    phasing_pulse_max; // Maximum level of above over the length of a line

  // Output from FM detector
  uint8_t discrim_output = 0;

  // Initialize on change of parameters
  if( pixels_per_line != wefax_rc.pixels_per_line )
  {
    /* We need to look for a phasing pulse maximum
     * over the length (in pixesls) of 1 WEFAX line */
    pixels_per_line = wefax_rc.pixels_per_line;

    /* This puts the pahsing pulse in the middle
     * of the image line during the syncing process */
    phasing_ref = pixels_per_line / 2 + WEFAX_PHASING_PULSE_SIZE;

    pixel_idx = 0;
  } // if( pixels_per_line != wefax_rc.pixels_per_line )

  // Stop on user request
  if( Flag[WEFAX_RECEIVE_STOP] )
  {
    pixel_idx     = 0;
    phasing_cnt   = 0;
    phasing_error = 0;
    wefax_action  = WEFAX_ACTION_STOP;
    return( True );
  }

  // Skip looking for phasing pulses
  if( Flag[WEFAX_SKIP_ACTION] )
  {
    // Re-initialize line buffer indices
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    Wefax_Show_Message( _("Skipping Phasing Pulse Sync"), "orange" );
    Wefax_Show_Message( _("Starting WEFAX Image Decoder ..."), "black" );
    Wefax_Show_Message( _("Listening for Stop Tone ..."), "black" );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_SKIP );

    Flag[WEFAX_SKIP_ACTION] = False;
    pixel_idx     = 0;
    phasing_cnt   = 0;
    phasing_error = 0;
    wefax_action  = WEFAX_ACTION_DECODE;
    return( True );
  }

  /* Fill line buffer with pixel values with signal
   * samples from FM detector. If it returns false,
   * wait for it to collect more sound samples */
  if( !FM_Detector(&discrim_output) )
  {
    if( !FM_Detector(&discrim_output) )
      return( False );
  }

  // Display the phasing pulse level
  Wefax_Display_Signal( discrim_output );

  // Fill line buffer with pixel values
  wefax_display.line_buffer[wefax_display.linebuff_input] = discrim_output;

  // Advance line buffer input index
  wefax_display.linebuff_input++;
  if( wefax_display.linebuff_input >= pixels_per_line )
    wefax_display.linebuff_input = 0;

  pixel_idx++;
  if( pixel_idx < pixels_per_line ) return( True );

  /* Look for a phasing pulse maximum over a line.
   * We begin the index from -phasing_error to dump
   * the number of pixels needed to center the pulse */
  phasing_pulse_max = -256;
  phasing_pulse_ave = 0.0;

  // Look for phasing pulse maximum
  for( pixel_idx = 0; pixel_idx < pixels_per_line; pixel_idx++ )
  {
    // Average line buffer pixel values
    phasing_pulse_ave *= WEFAX_PHASING_PUSLE_WIN - 1.0;
    phasing_pulse_ave += (double)wefax_display.line_buffer[pixel_idx];
    phasing_pulse_ave /= WEFAX_PHASING_PUSLE_WIN;

    // Record the input buffer index where max occurs
    if( phasing_pulse_max < (int)phasing_pulse_ave )
    {
      phasing_pulse_max = (int16_t)phasing_pulse_ave;
      pulse_max_idx     = pixel_idx;
    }
  } // for( pixel_idx = 0; pixel_idx < pixels_per_line; pixel_idx++

  // Count phasing pulse lines
  phasing_cnt++;

  /* Limit of pulse position error is
   * reduced progressively with line count */
  error_limit   = pixels_per_line / 2 / phasing_cnt;
  phasing_error = (int16_t)pulse_max_idx - phasing_ref;

  // Limit value of phasing error to avoid big jumps
  if( phasing_error > error_limit )
    phasing_error = error_limit;
  else if( phasing_error < -error_limit )
    phasing_error = -error_limit;

  // Adjust the line buffer index by the phasing error
  wefax_display.linebuff_input -= phasing_error;
  if( wefax_display.linebuff_input >= pixels_per_line )
    wefax_display.linebuff_input -= pixels_per_line;
  else if( wefax_display.linebuff_input < 0 )
    wefax_display.linebuff_input += pixels_per_line;

  // Look for phasing pulses over most of phasing lines
  if( phasing_cnt > wefax_rc.phasing_lines )
  {
    /* Point the line buffer output index to middle
     * of the lines buffer, this puts the phasing
     * pulse at the beginning of image lines */
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    Wefax_Show_Message( _("Phasing Pulse Synching ended"), "green" );
    Wefax_Show_Message( _("Starting WEFAX Image Decoder ..."), "black" );
    Wefax_Show_Message( _("Listening for Stop Tone ..."), "black" );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_APPLY );
    pixel_idx     = 0;
    phasing_cnt   = 0;
    phasing_error = 0;
    wefax_action  = WEFAX_ACTION_DECODE;
  }

  pixel_idx = 0;
  return( True );
} // Phasing_Detect()

//------------------------------------------------------------------------

/* Tone_Detect()
 *
 * Detects audio tones used for start and stop signaling.
 * The tone period is in pixels and input in pixel levels.
 */
  static void
Tone_Detect(
    double tone_period,
    uint8_t input,
    uint32_t *tone_level )
{
  static uint16_t
    input_cnt,       // Count of pixel level inputs
    detector_period; // Integration period of Goertzel detector

  // Variables for the Goertzel algorithm
  static double coeff, period;
  static double q0 = 0.0, q1 = 0.0, q2 = 0.0;

  // Sliding window average of tone level
  static uint32_t level_ave = 0;

  // Stop on user request
  if( Flag[WEFAX_RECEIVE_STOP] )
  {
    input_cnt = 0;
    period    = 0.0;
    level_ave = 0;
    wefax_action = WEFAX_ACTION_STOP;
    return;
  } // if( Flag[WEFAX_RECEIVE_STOP) )

  // Skip looking for start/stop tones
  if( Flag[WEFAX_SKIP_ACTION] )
  {
    // Re-initialize line buffer indices
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    input_cnt = 0;
    period    = 0.0;
    level_ave = 0;
    return;
  }

  // Initialize on new parameters
  if( (period < tone_period) || (period > tone_period) )
  {
    period = tone_period;
    detector_period = (uint16_t)
      ( period * TONE_PERIOD_MULT / wefax_rc.lines_per_min + 0.5 );

    double w = M_2PI / period;
    coeff = 2.0 * cos( w );

    // Reset variables
    input_cnt = 0;
    q0 = q1 = q2 = 0.0;
    level_ave = 0;
  } // if( period != tone_period )

  // Calculate Start/Stop level using Goertzel algorithm
  q0 = coeff * q1 - q2 + (double)input - 127.0;
  q2 = q1;
  q1 = q0;

  // Compute tone level and reset after detector_period inputs
  if( input_cnt++ >= detector_period )
  {
    // Reduce the magnitude to reasonable levels
    q1 /= period;
    q2 /= period;
    uint32_t level = (uint32_t)( q1 * q1 + q2 * q2 - q1 * q2 * coeff );

    // Compute sliding average of tone level and return
    level_ave *= TONE_LEVEL_AVE_WIN - 1;
    level_ave += level;
    level_ave /= TONE_LEVEL_AVE_WIN;

    // Reset variables
    q0 = q1 = q2 = 0.0;
    input_cnt = 0;
  } // if( input_cnt++ >= detector_period )

  *tone_level = level_ave;
  return;
} // Tone_Detect()

//------------------------------------------------------------------------

/* Start_Tone_Detect()
 *
 * Listens for and detects the Start tone
 */
  BOOLEAN
Start_Tone_Detect( void )
{
  // Detector output
  uint8_t discrim_output = 0;
  static uint32_t tone_level = 0;
  static BOOLEAN tone_up = False;

  /* Take signal samples from FM detector for the Start Tone detector.
   * If it returns false, wait for it to collect sound samples */
  if( !FM_Detector(&discrim_output) )
  {
    if( !FM_Detector(&discrim_output) )
      return( False );
  }
  Tone_Detect( wefax_rc.start_tone_period, discrim_output, &tone_level );

  // Display detector output and level gauge
  Wefax_Display_Signal( discrim_output );
  gauge_input  = tone_level / START_GAUGE_SCALE;
  gauge_level1 = START_TONE_DOWN / START_GAUGE_SCALE;
  gauge_level2 = START_TONE_UP   / START_GAUGE_SCALE;
  Queue_Draw_Gauge();

  // Skip looking for start tones
  if( Flag[WEFAX_SKIP_ACTION] )
  {
    // Re-initialize line buffer indices
    wefax_display.linebuff_input  = 0;
    wefax_display.linebuff_output =
      wefax_rc.line_buffer_size - wefax_rc.pixels_per_line2;

    Flag[WEFAX_SKIP_ACTION] = False;
    Wefax_Show_Message( _("Skipping Start Tone Detection"), "orange" );
    Wefax_Show_Message( _("Synchronizing Phasing Pulses ..."), "black" );
    Wefax_Set_Indicators( WEFAX_ICON_START_SKIP );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_YES );
    wefax_action = WEFAX_ACTION_PHASING;
    tone_up = False;
    return( True );
  }

  // Record the rise of start tone level
  if( tone_level > START_TONE_UP )
    tone_up = True;

  // Go to searching for Phasing Pulses when tone goes down
  if( (tone_level < START_TONE_DOWN) && tone_up )
  {
    Wefax_Show_Message( _("Start Tone Detected"), "green" );
    Wefax_Show_Message( _("Synchronizing Phasing Pulses ..."), "black" );
    Wefax_Set_Indicators( WEFAX_ICON_START_APPLY );
    Wefax_Set_Indicators( WEFAX_ICON_SYNC_YES );
    tone_up = False;
    wefax_action = WEFAX_ACTION_PHASING;
  }

  return( True );
} // Start_Tone_Detect()

//------------------------------------------------------------------------

/* Stop_Tone_Detect()
 *
 * Listens for and detects the Stop tone
 */
  BOOLEAN
Stop_Tone_Detect( uint8_t discrim_output )
{
  // Detector output
  static uint32_t tone_level = 0;
  static BOOLEAN tone_up = False;

  // Get the stop tone level
  Tone_Detect( wefax_rc.stop_tone_period, discrim_output, &tone_level );

  // Display detector output and level gauge
  gauge_input  = tone_level / STOP_GAUGE_SCALE;
  gauge_level1 = STOP_TONE_DOWN / STOP_GAUGE_SCALE;
  gauge_level2 = STOP_TONE_UP   / STOP_GAUGE_SCALE;
  Queue_Draw_Gauge();

  // Record the rise of start tone level
  if( tone_level > STOP_TONE_UP )
    tone_up = True;

  // Go to searching for Phasing Pulses when tone goes down
  if( (tone_level < STOP_TONE_DOWN) && tone_up )
  {
    Wefax_Show_Message( _("Stop Tone Detected"), "green" );
    tone_up = False;
    return( True );
  }

  return( False );
} // Stop_Tone_Detect()

//------------------------------------------------------------------------

// Free resources
  void
Wefax_Free_Detect( void )
{
  Mem_Free( (void **) &signal_buff );
  Mem_Free( (void **) &wefax_display.line_buffer );
  Free_Demod_Buf_Copies();
}

//------------------------------------------------------------------------

