/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "bookmarks.h"
#include "detect.h"
#include "display.h"
#include "interface.h"
#include "shared.h"
#include "utils.h"
#include "wefax.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/demodulate.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------

// All the callback functions of the WEFAX client

  void
on_wefax_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  Flag[GUEST_QUIT] = True;
  Flag[GUEST_DEMOD_IQ_DATA] = False;
  Flag[WEFAX_RECEIVE_STOP]  = True;
  wefax_action = WEFAX_ACTION_STOP;
  g_object_unref( wefax_gui.window_builder );
  g_object_unref( wefax_gui.popup_menu_builder );

  if( wefax_gui.bmk_window )
  {
    gtk_widget_destroy( wefax_gui.bmk_window );
    wefax_gui.bmk_window = NULL;
  }

  // Wait for Rx mode to exit
  if( Flag[GUEST_RECEIVING] )
    pthread_join( hermes2_rc.guest_rx_thread, NULL );

  Flag[WEFAX_SKIP_ACTION]     = False;
  Flag[WEFAX_INIMAGE_PHASING] = False;
  Flag[WEFAX_SAVE_IMAGE_PGM]  = False;
  Flag[WEFAX_SAVE_IMAGE_JPG]  = False;
  Flag[WEFAX_BMK_LIST_OK]     = False;
  Flag[GUEST_ENABLE_SCOPE]    = False;
  Flag[WEFAX_START_NEW_IMAGE] = False;
  Flag[WEFAX_SAVE_IMAGE]      = False;

  Wefax_Free_Detect();
  Wefax_Free_Display();
  Wefax_Free_Image();

  wefax_gui.window = NULL;
  hermes2_gui.guest_window = NULL;
}

//------------------------------------------------------------------

  gboolean
on_wefax_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Save_Window_Position( wefax_gui.window, wefax_rc.rc_fpath );
  Flag[GUEST_QUIT] = True;
  Flag[GUEST_DEMOD_IQ_DATA] = False;
  Flag[WEFAX_RECEIVE_STOP]  = True;
  wefax_action = WEFAX_ACTION_STOP;

  while( g_main_context_iteration(NULL, FALSE) );
  Guest_Quit_Activate( wefax_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------

  void
on_wefax_scope_drawingarea_configure(
    GtkWidget       *widget,
    GdkEventConfigure *event,
    gpointer         user_data)
{
  wefax_scope_width  = (uint16_t)event->width;
  wefax_scope_height = (uint16_t)event->height;
}

//------------------------------------------------------------------

  void
on_wefax_spectrum_drawingarea_configure(
    GtkWidget         *widget,
    const GdkEventConfigure *event,
    gpointer           user_data)
{
  Wefax_Spectrum_Size_Allocate( (uint16_t)event->width, (uint16_t)event->height );
}

//------------------------------------------------------------------

  gboolean
on_wefax_spectrum_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( wefax_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, wefax_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------

  gboolean
on_wefax_spectrum_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  Wefax_Tune_TRx( event->x );
  return( TRUE );
}

//------------------------------------------------------------------

  gboolean
on_wefax_scope_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  // Draw scope background
  if( !Flag[GUEST_ENABLE_SCOPE] )
  {
    cairo_set_source_rgb( cr, SCOPE_BACKGND );
    cairo_rectangle( cr, 0.0, 0.0,
        (double)wefax_scope_width, (double)wefax_scope_height );
    cairo_fill( cr );
  }
  else Wefax_Draw_Signal( cr );

  return( TRUE );
}

//------------------------------------------------------------------

  void
on_wefax_start_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( ddv->transmit_on ) return;

  if( Transceiver[Indices.TRx_Index]->receive_active )
    Wefax_Start_Button_Toggled( togglebutton );
}

//------------------------------------------------------------------

  void
on_wefax_skip_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Flag[WEFAX_SKIP_ACTION] = True;
}

//------------------------------------------------------------------

  void
on_wefax_rpm_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  New_Lines_Per_Min();
}

//------------------------------------------------------------------

  void
on_wefax_pix_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  New_Pixels_Per_Line();
}

//------------------------------------------------------------------

  void
on_wefax_ioc_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  New_IOC();
}

//------------------------------------------------------------------

  void
on_wefax_phl_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  New_Phasing_Lines();
}

//------------------------------------------------------------------

  void
on_wefax_enhance_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  New_Image_Enhance();
}

//------------------------------------------------------------------

  void
on_wefax_deslant_spinbutton_value_changed(
    GtkSpinButton *spinbutton,
    gpointer       user_data)
{
  double sync_slant = gtk_spin_button_get_value( spinbutton );
  Set_Sync_Slant( sync_slant );
}

//------------------------------------------------------------------

  void
on_wefax_in_image_phasing_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[WEFAX_INIMAGE_PHASING] = True;
  else
    Flag[WEFAX_INIMAGE_PHASING] = False;
}

//------------------------------------------------------------------

  gboolean
on_wefax_drawingarea_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer         user_data)
{
  return( Wefax_Drawingarea_Button_Press(event) );
}

//------------------------------------------------------------------

  gboolean
on_wefax_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( wefax_display.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, wefax_display.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------

  void
on_wefax_bmk_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( wefax_gui.bmk_window == NULL )
    Wefax_List_Bookmarks();
}

//------------------------------------------------------------------

  void
on_wefax_bmk_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  Flag[WEFAX_BMK_LIST_OK]  = False;
  wefax_gui.bmk_list_store = NULL;
  wefax_gui.bmk_treeview   = NULL;
  wefax_gui.bmk_window     = NULL;
}

//------------------------------------------------------------------

  void
on_wefax_down_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Wefax_Bmk_Select_Treeview_Row( TREEVIEW_DOWN );
}

//------------------------------------------------------------------

  void
on_wefax_up_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Wefax_Bmk_Select_Treeview_Row( TREEVIEW_UP );
}

//------------------------------------------------------------------

  void
on_wefax_save_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Wefax_Save_Bookmarks_File( wefax_rc.bookmarks_file );
}

//------------------------------------------------------------------

  void
on_wefax_delete_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Wefax_Delete_Bookmarks_Row();
}

//------------------------------------------------------------------

  void
on_wefax_new_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Wefax_New_Bookmarks_Row();
}

//------------------------------------------------------------------

  void
on_wefax_bmk_treeview_cursor_changed(
    GtkTreeView *treeview,
    gpointer     user_data)
{
  Wefax_Bmk_Cursor_Changed( treeview );
}

//------------------------------------------------------------------

  gboolean
on_wefax_bmk_treeview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 2 )
  {
    Wefax_Bmk_Treeview_Button_Press();
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------

  void
on_wefax_jpeg_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[WEFAX_SAVE_IMAGE_JPG] = True;
    Flag[WEFAX_SAVE_IMAGE_PGM] = False;
  }
}

//------------------------------------------------------------------

  void
on_wefax_pgm_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[WEFAX_SAVE_IMAGE_PGM] = True;
    Flag[WEFAX_SAVE_IMAGE_JPG] = False;
  }
}

//------------------------------------------------------------------

  void
on_wefax_both_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[WEFAX_SAVE_IMAGE_JPG] = True;
    Flag[WEFAX_SAVE_IMAGE_PGM] = True;
  }
}

//------------------------------------------------------------------

  void
on_wefax_iqdetect_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Alloc_Demod_Buf_Copies( SOUND_OP_BUF_SIZE );
  Flag[GUEST_DEMOD_IQ_DATA] = True;
  FM_Detector = FM_Detect_IQ;
}

//------------------------------------------------------------------

  void
on_wefax_zerocross_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_DEMOD_IQ_DATA] = False;
  FM_Detector = FM_Detect_Zero_Crossing;
}

//------------------------------------------------------------------

  void
on_wefax_bilevel_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_DEMOD_IQ_DATA] = False;
  FM_Detector = FM_Detect_Bilevel;
}

//------------------------------------------------------------------

  gboolean
on_wefax_gauge_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  Wefax_Display_Level_Gauge( cr );
  return( TRUE );
}

//------------------------------------------------------------------

  void
on_wefax_save_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    Wefax_Set_Indicators( WEFAX_ICON_SAVE_YES );
    Flag[WEFAX_SAVE_IMAGE] = True;
  }
  else
  {
    Wefax_Set_Indicators( WEFAX_ICON_SAVE_SKIP);
    Flag[WEFAX_SAVE_IMAGE] = False;
  }
}

//------------------------------------------------------------------

