/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "bookmarks.h"
#include "interface.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/display.h"
#include <gtk/gtk.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

#define WEFAX_BMK_LINE_LENGTH   75  // Bookmarks file line length
#define WEFAX_NAME_WIDTH        32  // Bookmarks file width in char of Name
#define WEFAX_FREQ_WIDTH        11  // Bookmarks file width in char of Freq
#define WEFAX_SDB_WIDTH         4   // Bookmarks file width in char of Sideband
#define WEFAX_RPM_WIDTH         4   // Bookmarks file width in char of RPM
#define WEFAX_RESOL_WIDTH       5   // Bookmarks file width in char of Resol
#define WEFAX_IOC_WIDTH         4   // Bookmarks file width in char of IOC
#define WEFAX_PHL_WIDTH         3   // Bookmarks file width in char of Phasing Lines
#define WEFAX_SLANT_WIDTH       4   // Bookmarks file width in char of Deslant

//------------------------------------------------------------------------

// Bookmark list columns
enum
{
  WEFAX_NAME_COL = 0,
  WEFAX_FREQ_COL,
  WEFAX_SDB_COL,
  WEFAX_RPM_COL,
  WEFAX_RESOL_COL,
  WEFAX_IOC_COL,
  WEFAX_PHL_COL,
  WEFAX_SLANT_COL,
  WEFAX_LIST_NUM_COLS
};

//------------------------------------------------------------------------

/* cell_edited_callback()
 *
 * Text cell edited callback
 */
  static void
cell_edited_callback(
    GtkCellRendererText *cell,
    gchar               *path,
    gchar               *new_text,
    gpointer             user_data )
{
  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  guint column;

  column = GPOINTER_TO_UINT( g_object_get_data(G_OBJECT(cell), "column") );
  selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(user_data) );
  gtk_tree_selection_get_selected( selection, &model, &iter );
  gtk_list_store_set( GTK_LIST_STORE(model), &iter, column, new_text, -1 );

} // cell_edited_callback()

//------------------------------------------------------------------------

/* Insert_Columns()
 *
 * Inserts columns in a list store
 */
  static void
Insert_Columns( GtkListStore* store, uint8_t ncols, char *colname[] )
{
  uint8_t idx;
  GtkTreeModel *model;

  // Create renderer and insert columns to treeview
  for( idx = 0; idx < ncols; idx++ )
  {
    GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "editable", TRUE, NULL);
    g_signal_connect(
        renderer, "edited", (GCallback)cell_edited_callback, wefax_gui.bmk_treeview );
    g_object_set_data( G_OBJECT(renderer), "column", GUINT_TO_POINTER(idx) );
    gtk_tree_view_insert_column_with_attributes(
        wefax_gui.bmk_treeview, -1, colname[idx], renderer, "text", idx, NULL );
  }

  // Create a model to insert list store
  model = GTK_TREE_MODEL( store );
  gtk_tree_view_set_model( wefax_gui.bmk_treeview, model );

  // Destroy model automatically with view
  g_object_unref( model );

} // Insert_Columns()

//------------------------------------------------------------------------

/* Create_List_Store()
 *
 * Create stores needed for the treeview
 */
  static void
Create_List_Store( void )
{
  // Bookmark list column names
  char *list_col_name[WEFAX_LIST_NUM_COLS] =
  {
    _("STATION NAME"),
    _("FREQUENCY"),
    _(" S/B"),
    _("RPM"),
    _("PIX/L"),
    _(" IOC"),
    _("PHL"),
    _("SLANT  ")
  };

  // Create list stores only if needed
  if( wefax_gui.bmk_list_store != NULL ) return;

  // Create bookmarks list store
  wefax_gui.bmk_list_store = gtk_list_store_new(
      WEFAX_LIST_NUM_COLS,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING );

  // Insert bookmark list columns
  Insert_Columns( wefax_gui.bmk_list_store, WEFAX_LIST_NUM_COLS, list_col_name );

} // Create_List_Store()

//------------------------------------------------------------------------

/* Wefax_List_Bookmarks()
 *
 * Reads bookmarks data from file and lists in tree view
 */
  void
Wefax_List_Bookmarks( void )
{
  gboolean ret;          // Return status variable
  FILE *fp = NULL;       // Bookmarks file pointer
  GtkTreeIter iter;      // Treeview list iteration
  GtkTreeSelection *sel; // Selected treeview row
  GtkTreePath *path;     // Path to ead data from treeview

  // Buffers one line of bookmark data
  char line_buf[READ_LINE_BUF_SIZE];

  // Create bookmarks window
  wefax_gui.bmk_window =
    Create_Wefax_Bmk_Window( &wefax_gui.bmk_window_builder );
  gtk_widget_show( wefax_gui.bmk_window );

  // Create treeview pointer
  wefax_gui.bmk_treeview = GTK_TREE_VIEW(
      Builder_Get_Object(wefax_gui.bmk_window_builder, "wefax_bmk_treeview") );
  g_object_unref( wefax_gui.bmk_window_builder );
  wefax_gui.bmk_window_builder = NULL;

  // Create the list store
  Create_List_Store();

  // Open bookmarks file
  if( Open_File(&fp, wefax_rc.bookmarks_file, "r") == FAILURE )
  {
    Wefax_Show_Message( "Failed to open Bookmarks File\n"
        "in ~/.hermes2/wefax/", "red" );
    Error_Dialog( "Failed to open Bookmarks File\n"
        "in ~/.hermes2/wefax/", SHOW_OK );
    return;
  }

  // Clear all tree view
  gtk_list_store_clear( wefax_gui.bmk_list_store );

  // Get new row if available
  ret = gtk_tree_model_get_iter_first(
      GTK_TREE_MODEL(wefax_gui.bmk_list_store), &iter);

  // Fill bookmark list
  while( !ret )
  {
    // Start and end character of strings in treeview
    uint8_t start, end;

    // Read a line and separate entries
    int eof = Read_Line( line_buf, fp, "Bookmarks File" );
    if( eof == EOF ) break;
    if( eof == ERROR )
    {
      Wefax_Show_Message( "Error reading Bookmarks File\n"
          "~/.hermes2/wefax/bookmarks", "red" );
      Error_Dialog( "Error reading Bookmarks File \n"
          "~/.hermes2/wefax/bookmarks", SHOW_OK );
      return;
    }

    // Append a row and fill in bookmark data
    gtk_list_store_append( wefax_gui.bmk_list_store, &iter );

    /* Separate entries in the line by string
     * terminator and set data to list store */
    // Bookmark Name
    start = 0;
    end = WEFAX_NAME_WIDTH;
    line_buf[end] = '\0';
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_NAME_COL, &line_buf[start], -1 );

    // Frequency - Hz
    end += WEFAX_FREQ_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_NAME_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_FREQ_COL, &line_buf[start], -1 );

    // Sideband - USB or LSB
    end += WEFAX_SDB_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_FREQ_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_SDB_COL, &line_buf[start], -1 );

    // RPM - lines/min
    end += WEFAX_RPM_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_SDB_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_RPM_COL, &line_buf[start], -1 );

    // Resolution - pix/line
    end += WEFAX_RESOL_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_RPM_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_RESOL_COL, &line_buf[start], -1 );

    // IOC Value
    end += WEFAX_IOC_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_RESOL_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_IOC_COL, &line_buf[start], -1 );

    // Phasing Lines Value
    end += WEFAX_PHL_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_IOC_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_PHL_COL, &line_buf[start], -1 );

    // Slant Value
    end += WEFAX_SLANT_WIDTH + 1;
    line_buf[end] = '\0';
    start += WEFAX_PHL_WIDTH + 1;
    gtk_list_store_set( wefax_gui.bmk_list_store,
        &iter, WEFAX_SLANT_COL, &line_buf[start], -1 );

    // Get new row if available
    ret = gtk_tree_model_iter_next(
        GTK_TREE_MODEL(wefax_gui.bmk_list_store), &iter);

  } // while()

  // Select the first entry
  sel = gtk_tree_view_get_selection( wefax_gui.bmk_treeview );
  path = gtk_tree_path_new_from_string( "0" );
  gtk_tree_selection_select_path( sel, path );
  gtk_tree_path_free( path );

  Close_File( &fp );
  Flag[WEFAX_BMK_LIST_OK] = True;

} // Wefax_List_Bookmarks()

//------------------------------------------------------------------------

/* Wefax_Save_Bookmarks_File()
 *
 * Saves the data in bookmarks treeview to a given filename
 */
  BOOLEAN
Wefax_Save_Bookmarks_File( char *bookmarks_file )
{
  FILE *fp = NULL;      // Bookmarks file pointer
  GtkTreeIter iter;     // Iteration to bookmark treeview

  // Buffer for one file line
  char file_line[WEFAX_BMK_LINE_LENGTH];

  gboolean ret = FALSE; // Return value of functions
  char *string; // String created from Wefax parameters
  uint8_t idx;

  /* Do not save if treeview is not
   * valid or save flag not set */
  if( wefax_gui.bmk_treeview == NULL )
    return( False );

  // Bookmarks treeview model
  GtkTreeModel *model = GTK_TREE_MODEL(wefax_gui.bmk_list_store);

  // Get the first iteration
  ret = gtk_tree_model_get_iter_first( model, &iter);
  if( !ret )
  {
    Wefax_Show_Message( "Bookmarks Treeview empty", "red" );
    Error_Dialog( "Bookmarks Treeview empty", SHOW_OK );
    return( False );
  }

  // Open bookmarks file for writing
  if( (Open_File(&fp, (char *)bookmarks_file, "w") == FAILURE) )
  {
    Wefax_Show_Message( "Failed to open Bookmarks File\n"
        "in ~/.hermes2/wefax/", "red" );
    Error_Dialog( "Failed to open Bookmarks File\n"
        "in ~/.hermes2/wefax/", SHOW_OK );
    return( False );
  }

  // Write treeview data
  Wefax_Show_Message( "Saving Bookmarks File ...", "black" );
  while( ret )
  {
    // Length and its limit of above string
    size_t len, limit;
    uint32_t value;

    // Clear file line
    for( idx = 0; idx < WEFAX_BMK_LINE_LENGTH; idx++ )
      file_line[idx] = ' ';

    // Get the bookmark name
    gtk_tree_model_get( model, &iter, WEFAX_NAME_COL, &string, -1 );
    len = strlen( string );
    limit = WEFAX_NAME_WIDTH;
    if( len > limit ) len = limit;

    // Remove leading spaces
    idx = 0;
    while( (idx < (int)len) && (string[idx] == ' ') )
    {
      idx++;
      len--;
    }

    // Enter the Station name
    snprintf( file_line, len, "%s", &string[idx] );
    file_line[len-1] = ' ';
    idx = WEFAX_NAME_WIDTH + 1;
    g_free( string );

    // Get the Station frequency
    gtk_tree_model_get( model, &iter, WEFAX_FREQ_COL, &string, -1 );
    value = (uint32_t)atoi( string );
    g_free( string );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_FREQ_WIDTH + 1, "%11u", value );
    idx += WEFAX_FREQ_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get and write the Station Sideband
    gtk_tree_model_get( model, &iter, WEFAX_SDB_COL, &string, -1 );
    snprintf( &file_line[idx], WEFAX_SDB_WIDTH + 1, "%s", string );
    g_free( string );
    idx += WEFAX_SDB_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get the Station RPM
    gtk_tree_model_get( model, &iter, WEFAX_RPM_COL, &string, -1 );
    value = (uint32_t)atoi( string );
    g_free( string );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_RPM_WIDTH + 1, "%4u", value );
    idx += WEFAX_RPM_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get the Station resolution
    gtk_tree_model_get( model, &iter, WEFAX_RESOL_COL, &string, -1 );
    value = (uint32_t)atoi( string );
    g_free( string );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_RESOL_WIDTH + 1, "%5u", value );
    idx += WEFAX_RESOL_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get the Station ioc
    gtk_tree_model_get( model, &iter, WEFAX_IOC_COL, &string, -1 );
    value = (uint32_t)atoi( string );
    g_free( string );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_IOC_WIDTH + 1, "%4u", value );
    idx += WEFAX_IOC_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get the Station phasing lines
    gtk_tree_model_get( model, &iter, WEFAX_PHL_COL, &string, -1 );
    value = (uint32_t)atoi( string );
    g_free( string );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_PHL_WIDTH + 1, "%3u", value );
    idx += WEFAX_PHL_WIDTH + 1;
    file_line[idx-1] = ' ';

    // Get the Station sync slant
    gtk_tree_model_get( model, &iter, WEFAX_SLANT_COL, &string, -1 );

    // Format value before writing to file
    snprintf( &file_line[idx], WEFAX_SLANT_WIDTH + 1, "%4d", atoi(string) );
    g_free( string );

    // Write line to bookmarks file
    file_line[WEFAX_BMK_LINE_LENGTH - 1] = '\0';
    fprintf( fp, "%s\n", file_line );

    ret = gtk_tree_model_iter_next( model, &iter);
  } // while( ret )
  fprintf( fp, "[END]\n#" );

  Wefax_Show_Message( "Bookmarks File saved OK", "green" );
  Close_File( &fp );

  return( False );
} // Wefax_Save_Bookmarks_File()

//------------------------------------------------------------------------

/* Wefax_Bmk_Select_Treeview_Row()
 *
 * Selects a bookmarks treeview row on user click
 * on up or down buttons in bookmarks window
 */
  void
Wefax_Bmk_Select_Treeview_Row( uint8_t direction )
{
  static int8_t row_num = 0;
  char path_str[4];      // String to create a path from
  GtkTreeSelection *sel; // Selected row in treeview
  GtkTreePath *path;     // A path into the selected row

  /* Increment or decrement row number
   * and clamp between 0 and 99 */
  if( direction == TREEVIEW_DOWN )
  {
    row_num ++;
    if( row_num > 99 ) row_num = 99;
  }
  else if( direction == TREEVIEW_UP )
  {
    row_num--;
    if( row_num < 0 ) row_num = 0;
  }

  // Create a selection object
  sel = gtk_tree_view_get_selection( wefax_gui.bmk_treeview );

  // Create a string for new path and select row
  snprintf( path_str, sizeof(path_str), "%3u", (uint8_t)row_num );
  path = gtk_tree_path_new_from_string( path_str );
  gtk_tree_selection_select_path( sel, path );
  gtk_tree_path_free( path );

  // Simulate a button press on treeview
  Wefax_Bmk_Cursor_Changed( wefax_gui.bmk_treeview );

} // Wefax_Bmk_Select_Treeview_Row()

//------------------------------------------------------------------------

/* Wefax_Bmk_Cursor_Changed()
 *
 * Handles cursor changed event on bookmarks treeview.
 * Reads the selected row and sets menu items with the
 * new values of wefax parameters and tunes the receiver
 */
  void
Wefax_Bmk_Cursor_Changed( GtkTreeView *treeview )
{
  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  GtkLabel *label;  // Set the Wefax image label to bookmark name
  gchar    *name;   // Station name

  // Remembers above name
  static gchar current[WEFAX_NAME_WIDTH + 1];

  // Value from treeview strings
  gchar *value;

  // Slant correction factor
  int slant;

  // Transceiver frequency and mode
  uint32_t center_freq;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get the selected row
  if( !Flag[WEFAX_BMK_LIST_OK] ) return;
  selection = gtk_tree_view_get_selection( treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter );

  // Get the Station name
  gtk_tree_model_get( model, &iter, WEFAX_NAME_COL, &name, -1 );

  /* If bookmarks row has not changed
   * or name not valid, do nothing */
  current[WEFAX_NAME_WIDTH] = '\0';
  if( (strncmp(current, name, WEFAX_NAME_WIDTH) == 0) || (strstr(name, "--")) )
  {
    g_free( name );
    return;
  }

  // Save current Station name
  strncpy( current, name, WEFAX_NAME_WIDTH );

  // Remove trailing spaces
  int8_t idx, len;
  len = (int8_t)strlen( name ) - 1;
  for( idx = len; idx >= 0; idx-- )
    if( name[idx] != ' ' ) break;
  name[idx + 1] = '\0';

  // Set the label of wefax image frame
  label = GTK_LABEL( Builder_Get_Object(wefax_gui.window_builder, "wefax_image_label") );
  gtk_label_set_text( label, name );
  g_free( name );

  // Get the RX frequency
  gtk_tree_model_get( model, &iter, WEFAX_FREQ_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  center_freq = (uint32_t)atoi( value );
  g_free( value );

  // Set up new frequency on change
  if( TRx->rx_frequency != center_freq )
  {
    // Set the Rx Band combobox first
    Set_Rx_Band( TRx, center_freq );
    TRx->rx_frequency = center_freq;
    Update_Spin_Dial( TRx, RX_FLAG );
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );
  }

  // Get the Station Sideband
  gtk_tree_model_get( model, &iter, WEFAX_SDB_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  if( strstr(value, "USB") )
    Rx_Modulation_Mode_Changed(Indices.TRx_Index, RX_MODE_RFAXU );
  else
    Rx_Modulation_Mode_Changed(Indices.TRx_Index, RX_MODE_RFAXL );
  g_free( value );

  // Get the RPM
  gtk_tree_model_get( model, &iter, WEFAX_RPM_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  wefax_rc.lines_per_min = atof( value );
  g_free( value );

  // Get the resolution
  gtk_tree_model_get( model, &iter, WEFAX_RESOL_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  wefax_rc.pixels_per_line = (uint16_t)atoi( value );
  g_free( value );

  // Get the IOC
  gtk_tree_model_get( model, &iter, WEFAX_IOC_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  wefax_rc.ioc_value = (uint16_t)atoi( value );
  g_free( value );

  // Period of start and stop tones in pixels
  if( wefax_rc.ioc_value == IOC576 )
    wefax_rc.start_tone = IOC576_START_TONE;
  else if( wefax_rc.ioc_value == IOC288 )
    wefax_rc.start_tone = IOC288_START_TONE;
  double temp = wefax_rc.lines_per_min / 60.0; // lines/sec
  wefax_rc.start_tone_period =
    temp * (double)wefax_rc.pixels_per_line / (double)wefax_rc.start_tone;
  wefax_rc.stop_tone_period =
    temp * (double)wefax_rc.pixels_per_line / WEFAX_STOP_TONE;

  // Get the phasing lines
  gtk_tree_model_get( model, &iter, WEFAX_PHL_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  wefax_rc.phasing_lines = (uint8_t)atoi( value );
  g_free( value );

  // Get the slant and set spin button
  gtk_tree_model_get( model, &iter, WEFAX_SLANT_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  slant = atoi( value );
  g_free( value );
  GtkSpinButton *spinb = GTK_SPIN_BUTTON(
      Builder_Get_Object(wefax_gui.window_builder, "wefax_deslant_spinbutton") );
  gtk_spin_button_set_value( spinb, slant );

  // Configure wefax for new station
  Wefax_Configure();

} // Wefax_Bmk_Cursor_Changed()

//------------------------------------------------------------------------

/* Wefax_Bmk_Treeview_Button_Press()
 *
 * Handles the bookmarks treeview button press callback
 */
  void
Wefax_Bmk_Treeview_Button_Press( void )
{
  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  int sync_slant = 0;

  // Data from bookmarks treeview
  gchar
    freq[WEFAX_FREQ_WIDTH+1],
    sdb[WEFAX_SDB_WIDTH+1],
    rpm[WEFAX_RPM_WIDTH+1],
    resol[WEFAX_RESOL_WIDTH+1],
    ioc[WEFAX_IOC_WIDTH+1],
    phl[WEFAX_PHL_WIDTH+1],
    slant[WEFAX_SLANT_WIDTH+1];

    const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

    // Get the selected row
    selection = gtk_tree_view_get_selection( wefax_gui.bmk_treeview );
    gtk_tree_selection_get_selected( selection, &model, &iter );
    sync_slant = (int)( wefax_rc.sync_slant * 1000.0 );

    // Enter station data
    snprintf( freq,  sizeof(freq),  "%*u",
        WEFAX_FREQ_WIDTH, TRx->rx_frequency );
    if( TRx->rx_modulation_mode == RX_MODE_RFAXL )
      snprintf( sdb,  sizeof(sdb),  "%*s", WEFAX_SDB_WIDTH, "LSB" );
    else
      snprintf( sdb,  sizeof(sdb),  "%*s", WEFAX_SDB_WIDTH, "USB" );
    snprintf( rpm,   sizeof(rpm),   "%*u",
        WEFAX_RPM_WIDTH, (uint8_t)wefax_rc.lines_per_min );
    snprintf( resol, sizeof(resol), "%*u",
        WEFAX_RESOL_WIDTH, wefax_rc.pixels_per_line );
    if( wefax_rc.ioc_value > 999 ) wefax_rc.ioc_value = 999;
    snprintf( ioc,   sizeof(ioc),   "%*u",
        WEFAX_IOC_WIDTH, wefax_rc.ioc_value );
    snprintf( phl,   sizeof(phl),   "%*u",
        WEFAX_PHL_WIDTH, wefax_rc.phasing_lines );
    snprintf( slant, sizeof(slant), "%*d",
        WEFAX_SLANT_WIDTH, sync_slant );

    // Set data to list store
    gtk_list_store_set(
        wefax_gui.bmk_list_store, &iter,
        WEFAX_FREQ_COL,  freq,
        WEFAX_SDB_COL,   sdb,
        WEFAX_RPM_COL,   rpm,
        WEFAX_RESOL_COL, resol,
        WEFAX_IOC_COL,   ioc,
        WEFAX_PHL_COL,   phl,
        WEFAX_SLANT_COL, slant,
        -1 );

} // Wefax_Bmk_Treeview_Button_Press()

//------------------------------------------------------------------------

/* Wefax_New_Bookmarks_Row()
 *
 * Adds a new row to the bookmarks treeview
 */
  void
Wefax_New_Bookmarks_Row( void )
{
  // Bookmarks treeview selection objects
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter, sibling;
  int ncols, idx;

  if( wefax_gui.bmk_treeview == NULL )
    return;

  // Find selected row and add new after
  selection = gtk_tree_view_get_selection( wefax_gui.bmk_treeview );
  if( ! gtk_tree_selection_get_selected(selection, &model, &sibling) )
  {
    // Empty tree view case
    model = gtk_tree_view_get_model( wefax_gui.bmk_treeview );
    gtk_list_store_insert( GTK_LIST_STORE(model), &iter, 0 );
  }
  else gtk_list_store_insert_after(
      GTK_LIST_STORE(model), &iter, &sibling);

  // Prime columns of new row
  gtk_tree_selection_select_iter( selection, &iter);
  ncols = gtk_tree_model_get_n_columns( model );
  for( idx = 0; idx < ncols; idx++ )
    gtk_list_store_set(
        GTK_LIST_STORE(model), &iter, idx, "--", -1 );

} // Wefax_New_Bookmarks_Row()

//------------------------------------------------------------------------

/* Wefax_Delete_Bookmarks_Row()
 *
 * Deletes a selected row in the bookmarks treeview
 */
  void
Wefax_Delete_Bookmarks_Row( void )
{
  // Bookmarks treeview selection objects
  GtkTreeModel *model;
  GtkTreeIter   iter;
  GtkTreeSelection *selection;

  if( wefax_gui.bmk_treeview == NULL )
    return;

  selection = gtk_tree_view_get_selection( wefax_gui.bmk_treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter);
  gtk_list_store_remove( GTK_LIST_STORE(model), &iter );

} // Wefax_Delete_Bookmarks_Row()

//------------------------------------------------------------------------

