/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "time_signal.h"
#include "display.h"
#include "../common/common.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include <math.h>
#include <stdint.h>
#include <stdlib.h>
#include <time.h>

//----------------------------------------------------------------------

// I/Q Samples buffers for the LP Filters
static double *demod_id = NULL, *demod_qd = NULL;

//----------------------------------------------------------------------

/* Scale factor to reduce averaged I/Q
 * samples to a reasonable value */
#include <stdlib.h>
#define AVERAGE_IQ_SCALE  1.0E6

// Prime minimum variables to search for minima
#define MINIMUM_PRIME  1000000000

// One minute in seconds
#define ONE_MINUTE_SECS  60

//------------------------------------------------------------------------

/* DCF77 sample rate. Same as sound card only for
 * future compatibility, in case CW decoding of the
 * carrier is implemented from audio signal */
#define DCF77_SAMPLE_RATE  48000

// DCF77 pulse width (0.1 sec) in ADC samples
#define DCF77_PULSE_WIDTH    4800
#define DCF77_SECOND_PULSES  10

/* 1/10 of DCF77 pulse (10 msec) in ADC samples. This
 * is used to refine the search for DCF77 time pulses */
#define DCF77_TICK_WIDTH   480
#define DCF77_PULSE_TICKS  10

// Number of trailing pulses saved in FIFO
#define DCF77_TRAIL_PULSES  2

// Number of ticks per second
#define DCF77_SECOND_TICKS  100

// Number of pulse buffers in ring samples buffer
#define DCF77_SAMPLES_BUFFERS  3

// DCF77 data bit array index for Parity Bits
#define DCF77_P1  28
#define DCF77_P2  35
#define DCF77_P3  58

//------------------------------------------------------------------------

/* ALL162 sample rate. Same as sound card only for
 * future compatibility, in case CW decoding of the
 * carrier is implemented for audible monitoring */
#define ALL162_SAMPLE_RATE  48000

// Number of buffers in ring samples buffer
#define ALL162_SAMPLES_BUFFERS  3

// ALL162 data bit width (0.1sec) in ADC samples
#define ALL162_PULSE_WIDTH  4800

// Upscale factor to allow use of int's
#define ALL162_UPSCALE  1000.0

// Sliding window average of pulse levels
#define ALL162_PULSE_AVE_MUL  49
#define ALL162_PULSE_AVE_DIV  50

/* Points of phase reversal during data bits 0 or 1.
 * 1/4 of data bit has +ve phase change, 1/2 has
 * negative and last 1/4 has positive phase change.
 * Measures are in samples, fractions of data bit length */
#define ALL162_RAMP_A  1200
#define ALL162_RAMP_B  3600
#define ALL162_RAMP_C  4800

// Allouis ALL162 'ticks' (10 msec) per data bit (0.1 sec)
#define ALL162_PULSE_TICKS  10

// ALL162 number of pulses per second
#define ALL162_SECOND_PULSES  10

// Number of ticks per second
#define ALL162_SECOND_TICKS  100

// Number of trailing pulses saved in FIFO
#define ALL162_TRAIL_PULSES  2

// Allouis ALL162 'tick' width in samples
#define ALL162_TICK_WIDTH  480

// Offset from seconds marker to look for data bits
#define ALL162_SEC_MARKER_OFFSET  11

// Offset from minutes marker to zero seconds
#define ALL162_MIN_MARKER_OFFSET  30

// Factors for sliding window average of sec markers
#define ALL162_SEC_MARKER_MUL  9
#define ALL162_SEC_MARKER_DIV  10

//------------------------------------------------------------------------

/* RBU66 sample rate. Same as sound card only for
 * future compatibility, in case CW decoding of the
 * carrier is implemented for audible monitoring */
#define RBU66_SAMPLE_RATE  48000

// RBU66 pulse width (0.1 sec) in ADC samples
#define RBU66_PULSE_WIDTH  4800

// RBU66 'ticks' (10 msec) per pulse (0.1 sec)
#define RBU66_PULSE_TICKS  10

// Number of trailing pulses saved in FIFO
#define RBU66_TRAIL_PULSES  4

// Number of data bit combinations (00 to 11)
#define RBU66_DATA_BITS  4

// RBU66 'tick' width in samples
#define RBU66_TICK_WIDTH  480

// RBU66 number of pulses per second
#define RBU66_SECOND_PULSES  10

// PM tone frequencies (Hz) for 0 and 1 bits
#define RBU66_TONE_0  100.0
#define RBU66_TONE_1  312.5

// Factors for sliding window averaging
#define RBU66_AVERAGE_MUL  1999.0
#define RBU66_AVERAGE_DIV  2000.0

// Number of buffers in ring samples buffer
#define RBU66_SAMPLES_BUFFERS  3

//------------------------------------------------------------------------

/* MSF60 sample rate. Same as sound card only for
 * future compatibility, in case CW decoding of the
 * carrier is implemented for audible monitoring */
#define MSF60_SAMPLE_RATE  48000

// MSF60 pulse width (0.1 sec) in ADC samples
#define MSF60_PULSE_WIDTH  4800

// MSF60 'ticks' (10 msec) per pulse (0.1 sec)
#define MSF60_PULSE_TICKS  10

// MSF60 'tick' width in samples
#define MSF60_TICK_WIDTH   480

// Number of trailing pulses saved in FIFO
#define MSF60_TRAIL_PULSES  4

// Number of data bit combinations (00 to 11)
#define MSF60_DATA_BITS  4

// MSF60 number of pulses per second
#define MSF60_SECOND_PULSES  10

// Factors for sliding window averaging
#define MSF60_AVERAGE_MUL  1999.0
#define MSF60_AVERAGE_DIV  2000.0

// Number of buffers in ring samples buffer
#define MSF60_SAMPLES_BUFFERS  3

//------------------------------------------------------------------------

// DCF77 data bit array index for other Flags
enum DCF77_OTHER
{
  DCF77_R_ST = 15,
  DCF77_A1,
  DCF77_Z1,
  DCF77_Z2,
  DCF77_A2
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Minutes
enum DCF77_MIN
{
  DCF77_MIN1 = 21,
  DCF77_MIN2,
  DCF77_MIN4,
  DCF77_MIN8,
  DCF77_MIN10,
  DCF77_MIN20,
  DCF77_MIN40
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Hours
enum DCF77_HRS
{
  DCF77_HOUR1 = 29,
  DCF77_HOUR2,
  DCF77_HOUR4,
  DCF77_HOUR8,
  DCF77_HOUR10,
  DCF77_HOUR20
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Day of the Month
enum DCF77_DATE
{
  DCF77_MDAY1 = 36,
  DCF77_MDAY2,
  DCF77_MDAY4,
  DCF77_MDAY8,
  DCF77_MDAY10,
  DCF77_MDAY20
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Day of the Week
enum DCF77_WEEKDAY
{
  DCF77_WDAY1 = 42,
  DCF77_WDAY2,
  DCF77_WDAY4
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Month of the Year
enum DCF77_MONTH
{
  DCF77_MON1 = 45,
  DCF77_MON2,
  DCF77_MON4,
  DCF77_MON8,
  DCF77_MON10
};

//------------------------------------------------------------------------

// DCF77 data bit array index for Year in Century
enum DCF77_YEAR
{
  DCF77_YEAR1 = 50,
  DCF77_YEAR2,
  DCF77_YEAR4,
  DCF77_YEAR8,
  DCF77_YEAR10,
  DCF77_YEAR20,
  DCF77_YEAR40,
  DCF77_YEAR80
};

//------------------------------------------------------------------------

// RBU66 data bits for DUT1
#define RBU66_PDUT1  1
#define RBU66_MDUT1  9
#define RBU66_NDUT1  8
#define RBU66_WDUT1  0.1

//------------------------------------------------------------------------

// RBU66 data bits for dUT1
#define RBU66_PdUT1  3
#define RBU66_MdUT1  11
#define RBU66_NdUT1  5
#define RBU66_WdUT1  0.02

//------------------------------------------------------------------------

// Parity bit indices
#define RBU66_P1  49
#define RBU66_P2  50
#define RBU66_P3  53
#define RBU66_P4  54
#define RBU66_P5  55
#define RBU66_P6  56
#define RBU66_P7  57
#define RBU66_P8  58

//------------------------------------------------------------------------

// Default DUT of Moscow (3 hrs)
#define DUT_MOSCOW  3

//------------------------------------------------------------------------

// RBU66 data bit array index for Truncated Julian Day
enum RBU66_JULIAN
{
  RBU66_TJD8000 = 18,
  RBU66_TJD4000,
  RBU66_TJD2000,
  RBU66_TJD1000,
  RBU66_TJD800,
  RBU66_TJD400,
  RBU66_TJD200,
  RBU66_TJD100,
  RBU66_TJD80,
  RBU66_TJD40,
  RBU66_TJD20,
  RBU66_TJD10,
  RBU66_TJD8,
  RBU66_TJD4,
  RBU66_TJD2,
  RBU66_TJD1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Moscow Time - UTC (DeltaUT)
enum RBU66_DELTA
{
  RBU66_DLTUTS = 18,
  RBU66_DLTUT10,
  RBU66_DLTUT8,
  RBU66_DLTUT4,
  RBU66_DLTUT2,
  RBU66_DLTUT1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Year in Century
enum RBU66_YEAR
{
  RBU66_YEAR80 = 25,
  RBU66_YEAR40,
  RBU66_YEAR20,
  RBU66_YEAR10,
  RBU66_YEAR8,
  RBU66_YEAR4,
  RBU66_YEAR2,
  RBU66_YEAR1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Month of the Year
enum RBU66_MONTH
{
  RBU66_MONTH10 = 33,
  RBU66_MONTH8,
  RBU66_MONTH4,
  RBU66_MONTH2,
  RBU66_MONTH1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Day of the Week
enum RBU66_WEEKDAY
{
  RBU66_WDAY4 = 38,
  RBU66_WDAY2,
  RBU66_WDAY1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Day of the Month
enum RBU66_DATE
{
  RBU66_MDAY20 = 41,
  RBU66_MDAY10,
  RBU66_MDAY8,
  RBU66_MDAY4,
  RBU66_MDAY2,
  RBU66_MDAY1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Hours
enum RBU66_HRS
{
  RBU66_HOUR20 = 47,
  RBU66_HOUR10,
  RBU66_HOUR8,
  RBU66_HOUR4,
  RBU66_HOUR2,
  RBU66_HOUR1
};

//------------------------------------------------------------------------

// RBU66 data bit array index for Minutes
enum RBU66_MIN
{
  RBU66_MIN40 = 53,
  RBU66_MIN20,
  RBU66_MIN10,
  RBU66_MIN8,
  RBU66_MIN4,
  RBU66_MIN2,
  RBU66_MIN1
};

//------------------------------------------------------------------------

// MSF60 data bits for DUT1
#define MSF60_PDUT1  1
#define MSF60_MDUT1  9
#define MSF60_NDUT1  8
#define MSF60_WDUT1  0.1

// Parity bit indices
#define MSF60_P1  54
#define MSF60_P2  55
#define MSF60_P3  56
#define MSF60_P4  57

// Summer Time Flags
#define MSF60_ST_WARN    53
#define MSF60_ST_EFFECT  58

//------------------------------------------------------------------------

// MSF60 data bit array index for Year in Century
enum MSF60_YEAR
{
  MSF60_YEAR80 = 17,
  MSF60_YEAR40,
  MSF60_YEAR20,
  MSF60_YEAR10,
  MSF60_YEAR8,
  MSF60_YEAR4,
  MSF60_YEAR2,
  MSF60_YEAR1
};

//------------------------------------------------------------------------

// MSF60 data bit array index for Month of the Year
enum MSF60_MONTH
{
  MSF60_MONTH10 = 25,
  MSF60_MONTH8,
  MSF60_MONTH4,
  MSF60_MONTH2,
  MSF60_MONTH1
};

//------------------------------------------------------------------------

// MSF60 data bit array index for Day of the Month
enum MSF60_DATE
{
  MSF60_MDAY20 = 30,
  MSF60_MDAY10,
  MSF60_MDAY8,
  MSF60_MDAY4,
  MSF60_MDAY2,
  MSF60_MDAY1
};

//------------------------------------------------------------------------

// MSF60 data bit array index for Day of the Week
enum MSF60_WEEKDAY
{
  MSF60_WDAY4 = 36,
  MSF60_WDAY2,
  MSF60_WDAY1
};

//------------------------------------------------------------------------

// MSF60 data bit array index for Hours
enum MSF60_HRS
{
  MSF60_HOUR20 = 39,
  MSF60_HOUR10,
  MSF60_HOUR8,
  MSF60_HOUR4,
  MSF60_HOUR2,
  MSF60_HOUR1
};

//------------------------------------------------------------------------

// MSF60 data bit array index for Minutes
enum MSF60_MIN
{
  MSF60_MIN40 = 45,
  MSF60_MIN20,
  MSF60_MIN10,
  MSF60_MIN8,
  MSF60_MIN4,
  MSF60_MIN2,
  MSF60_MIN1
};

//------------------------------------------------------------------------

/* Free_Time_Buffers()
 *
 * Frees file I/Q buffers used in demodulators
 */
  void
Free_Time_Buffers( void )
{
  Mem_Free( (void **) &demod_id );
  Mem_Free( (void **) &demod_qd );
} // Free_Time_Buffers()

//----------------------------------------------------------------------

/* Decode_DCF77()
 *
 * Decodes the time signals from DCF77 station
 */
  static void
Decode_DCF77( time_data_t *time_data )
{
  // DCF77 data bits saved over one minute
  static uint8_t data_bits[ONE_MINUTE_SECS], data_bit;

  // Used to save decoded DCF77 time/date data
  struct tm lcl;

  // Enter Seconds
  lcl.tm_sec  = time_data->second;
  time_data->tim.tm_sec = lcl.tm_sec;

  // Decode data bits on Minute (zero second)
  data_bit = time_data->data_bit_1;
  if( time_data->second == 0 )
  {
    // DCF77 data bit parity count
    uint8_t parity, idx;

    // Check Even Parity Bits of Minutes
    parity = 0;
    for( idx = DCF77_MIN1; idx <= DCF77_P1; idx++ )
      parity += data_bits[idx];
    if( parity % 2 )
      time_data->status |= MINUTE_ERROR;
    else
      time_data->status &= ~MINUTE_ERROR;

    // Check Even Parity Bits of Hours
    parity = 0;
    for( idx = DCF77_HOUR1; idx <= DCF77_P2; idx++ )
      parity += data_bits[idx];
    if( parity % 2 )
      time_data->status |= HOUR_ERROR;
    else
      time_data->status &= ~HOUR_ERROR;

    // Check Even Parity Bits of Date
    parity = 0;
    for( idx = DCF77_MDAY1; idx <= DCF77_P3; idx++ )
      parity += data_bits[idx];
    if( parity % 2 )
      time_data->status |= DATE_ERROR;
    else
      time_data->status &= ~DATE_ERROR;

    // Set Transmitter Status flag
    if( data_bits[DCF77_R_ST] )
      time_data->status |= DCF77_TX_ABNL;
    else
      time_data->status &= ~DCF77_TX_ABNL;

    // Check the CET/CEST flags, should be complementary
    if( data_bits[DCF77_Z1] == data_bits[DCF77_Z2] )
      time_data->status |= DCF77_Z1_Z2_ERROR;
    else
      time_data->status &= ~DCF77_Z1_Z2_ERROR;

    // Set Summer Time flags
    if( data_bits[DCF77_Z1] )
      time_data->status |= DCF77_Z1_CEST;
    else
      time_data->status &= ~DCF77_Z1_CEST;
    if( data_bits[DCF77_Z2] )
      time_data->status |= DCF77_Z2_CET;
    else
      time_data->status &= ~DCF77_Z2_CET;

    // Set Summer Time and Leap sec Announcement flags
    if( data_bits[DCF77_A1] )
      time_data->status |= DCF77_A1_CEST;
    else
      time_data->status &= ~DCF77_A1_CEST;
    if( data_bits[DCF77_A2] )
      time_data->status |= DCF77_A2_LEAP_SEC;
    else
      time_data->status &= ~DCF77_A2_LEAP_SEC;

    // Decode time of day - Minutes
    lcl.tm_min =
      (data_bits[DCF77_MIN10]) + (data_bits[DCF77_MIN20] << 1) +
      (data_bits[DCF77_MIN40] << 2);
    lcl.tm_min *= 10;
    lcl.tm_min +=
      (data_bits[DCF77_MIN1])      + (data_bits[DCF77_MIN2] << 1) +
      (data_bits[DCF77_MIN4] << 2) + (data_bits[DCF77_MIN8] << 3);

    // Decode time of day - Hours
    lcl.tm_hour  = data_bits[DCF77_HOUR10] + (data_bits[DCF77_HOUR20] << 1);
    lcl.tm_hour *= 10;
    lcl.tm_hour +=
      (data_bits[DCF77_HOUR1])      + (data_bits[DCF77_HOUR2] << 1) +
      (data_bits[DCF77_HOUR4] << 2) + (data_bits[DCF77_HOUR8] << 3);
    lcl.tm_hour -= data_bits[DCF77_Z1] + 1; // DCF77 is +1 hour to UTC
    if( lcl.tm_hour < 0 ) lcl.tm_hour += 24;

    // Decode Day of the Month
    lcl.tm_mday  = data_bits[DCF77_MDAY10] + (data_bits[DCF77_MDAY20] << 1);
    lcl.tm_mday *= 10;
    lcl.tm_mday +=
      (data_bits[DCF77_MDAY1])      + (data_bits[DCF77_MDAY2] << 1) +
      (data_bits[DCF77_MDAY4] << 2) + (data_bits[DCF77_MDAY8] << 3);

    // Decode Day of the Week
    lcl.tm_wday =
      data_bits[DCF77_WDAY1] + (data_bits[DCF77_WDAY2] << 1) +
      (data_bits[DCF77_WDAY4] << 2);
    //lcl.tm_wday -= 1; // To comply with struct tm format

    // Decode Month of the Year
    lcl.tm_mon  = 10 * data_bits[DCF77_MON10];
    lcl.tm_mon +=
      (data_bits[DCF77_MON1])      + (data_bits[DCF77_MON2] << 1) +
      (data_bits[DCF77_MON4] << 2) + (data_bits[DCF77_MON8] << 3);
    lcl.tm_mon -= 1; // To comply with struct tm format

    /* Decode Year in Century (00-99).
     * !!NOTE: Not compatible with struct tm after 2099!! */
    lcl.tm_year =
      (data_bits[DCF77_YEAR10])      + (data_bits[DCF77_YEAR20] << 1) +
      (data_bits[DCF77_YEAR40] << 2) + (data_bits[DCF77_YEAR80] << 3);
    lcl.tm_year *= 10;
    lcl.tm_year +=
      (data_bits[DCF77_YEAR1])      + (data_bits[DCF77_YEAR2] << 1) +
      (data_bits[DCF77_YEAR4] << 2) + (data_bits[DCF77_YEAR8] << 3);

    // To comply with struct tm format. !!VALID ONLY TILL 2099!!
    lcl.tm_year += 100;

    // Convert to local time
    lcl.tm_isdst = -1;
    time_t utc = timegm( &lcl );
    lcl = *localtime( &utc );
    time_data->tim.tm_min   = lcl.tm_min;
    time_data->tim.tm_hour  = lcl.tm_hour;
    time_data->tim.tm_mday  = lcl.tm_mday;
    time_data->tim.tm_wday  = lcl.tm_wday;
    time_data->tim.tm_mon   = lcl.tm_mon;
    time_data->tim.tm_year  = lcl.tm_year;
    time_data->tim.tm_isdst = lcl.tm_isdst;

  } // if( second_idx == 0 )
  else   // Save data bits to decode on second zero
    data_bits[time_data->second] = data_bit;

  return;
} // Decode_DCF77()

//----------------------------------------------------------------------

/* Decode_RBU66()
 *
 * Decodes the time signals from RBU66 station
 */
  static void
Decode_RBU66( time_data_t *time_data )
{
  // RBU66 data bits saved over one minute
  static uint8_t
    data_bits_1[ONE_MINUTE_SECS],
    data_bits_2[ONE_MINUTE_SECS],
    data_bit_1, data_bit_2;

    // Used to save decoded RBU66 time/date data
    struct tm lcl;

    // Enter Seconds
    lcl.tm_sec  = time_data->second;
    time_data->tim.tm_sec = lcl.tm_sec;

    // Decode data bits on Minute (zero second)
    data_bit_1 = time_data->data_bit_1;
    data_bit_2 = time_data->data_bit_2;
    if( time_data->second == 0 )
    {
      // RBU66 data bit parity count
      uint8_t parity, idx;

      // Check for Even Parity Bits of TJD
      parity = 0;
      for( idx = RBU66_TJD8000; idx <= RBU66_TJD100; idx++ )
        parity += data_bits_2[idx];
      parity += data_bits_2[RBU66_P1];
      if( parity % 2 )
        time_data->status |= RBU66_TJD_ERROR;
      else
        time_data->status &= ~RBU66_TJD_ERROR;

      // Check for Even Parity Bits of TJD
      parity = 0;
      for( idx = RBU66_TJD80; idx <= RBU66_TJD1; idx++ )
        parity += data_bits_2[idx];
      parity += data_bits_2[RBU66_P2];
      if( parity % 2 )
        time_data->status |= RBU66_TJD_ERROR;
      else if( !(time_data->status & RBU66_TJD_ERROR) )
        time_data->status &= ~RBU66_TJD_ERROR;

      // Check for Even Parity Bits of DeltaUT
      parity = 0;
      for( idx = RBU66_DLTUTS; idx <= RBU66_DLTUT1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P3];
      if( parity % 2 )
        time_data->status |= RBU66_DUT_ERROR;
      else
        time_data->status &= ~RBU66_DUT_ERROR;

      // Check for Even Parity Bits of Years
      parity = 0;
      for( idx = RBU66_YEAR80; idx <= RBU66_YEAR1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P4];
      if( parity % 2 )
        time_data->status |= YEAR_ERROR;
      else
        time_data->status &= ~YEAR_ERROR;

      // Check for Even Parity Bits of Month/DoW
      parity = 0;
      for( idx = RBU66_MONTH10; idx <= RBU66_WDAY1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P5];
      if( parity % 2 )
        time_data->status |= MONTH_ERROR;
      else
        time_data->status &= ~MONTH_ERROR;

      // Check for Even Parity Bits of Days
      parity = 0;
      for( idx = RBU66_MDAY20; idx <= RBU66_MDAY1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P6];
      if( parity % 2 )
        time_data->status |= MDAY_ERROR;
      else
        time_data->status &= ~MDAY_ERROR;

      // Set the date error flag
      if( time_data->status & (YEAR_ERROR | MONTH_ERROR | MDAY_ERROR) )
        time_data->status |= DATE_ERROR;
      else
        time_data->status &= ~DATE_ERROR;

      // Check for Even Parity Bits of Hours
      parity = 0;
      for( idx = RBU66_HOUR20; idx <= RBU66_HOUR1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P7];
      if( parity % 2 )
        time_data->status |= HOUR_ERROR;
      else
        time_data->status &= ~HOUR_ERROR;

      // Check for Even Parity Bits of Minutes
      parity = 0;
      for( idx = RBU66_MIN40; idx <= RBU66_MIN1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[RBU66_P8];
      if( parity % 2 )
        time_data->status |= MINUTE_ERROR;
      else
        time_data->status &= ~MINUTE_ERROR;

      // Decode DUT1 +ve values
      time_data->DUT1 = 0.0;
      for( idx = 0; idx < RBU66_NDUT1; idx++ )
        if( data_bits_2[RBU66_PDUT1 + idx] )
          time_data->DUT1 = RBU66_WDUT1 * (double)(idx + 1);

      // Decode DUT1 -ve values
      for( idx = 0; idx < RBU66_NDUT1; idx++ )
        if( data_bits_2[RBU66_MDUT1 + idx] )
          time_data->DUT1 = -RBU66_WDUT1 * (double)(idx + 1);

      // Decode dUT1 +ve values
      double dUT1 = 0.0;
      for( idx = 0; idx < RBU66_NdUT1; idx++ )
        if( data_bits_1[RBU66_PdUT1 + idx] )
          dUT1 = RBU66_WdUT1 * (double)(idx + 1);

      // Decode dUT1 -ve values
      for( idx = 0; idx < RBU66_NdUT1; idx++ )
        if( data_bits_1[RBU66_MdUT1 + idx] )
          dUT1 = -RBU66_WdUT1 * (double)(idx + 1);
      time_data->DUT1 += dUT1;

      // Decode Truncated Julian Time
      time_data->tjd =
        (data_bits_2[RBU66_TJD1000])      + (data_bits_2[RBU66_TJD2000] << 1) +
        (data_bits_2[RBU66_TJD4000] << 2) + (data_bits_2[RBU66_TJD8000] << 3);
      time_data->tjd *= 10;
      time_data->tjd +=
        (data_bits_2[RBU66_TJD100])      + (data_bits_2[RBU66_TJD200] << 1) +
        (data_bits_2[RBU66_TJD400] << 2) + (data_bits_2[RBU66_TJD800] << 3);
      time_data->tjd *= 10;
      time_data->tjd +=
        (data_bits_2[RBU66_TJD10])      + (data_bits_2[RBU66_TJD20] << 1) +
        (data_bits_2[RBU66_TJD40] << 2) + (data_bits_2[RBU66_TJD80] << 3);
      time_data->tjd *= 10;
      time_data->tjd +=
        (data_bits_2[RBU66_TJD1])      + (data_bits_2[RBU66_TJD2] << 1) +
        (data_bits_2[RBU66_TJD4] << 2) + (data_bits_2[RBU66_TJD8] << 3);

      /* Decode Year in Century (00-99).
       * !!NOTE: Not compatible with struct tm after 2099!! */
      lcl.tm_year =
        (data_bits_1[RBU66_YEAR10])      + (data_bits_1[RBU66_YEAR20] << 1) +
        (data_bits_1[RBU66_YEAR40] << 2) + (data_bits_1[RBU66_YEAR80] << 3);
      lcl.tm_year *= 10;
      lcl.tm_year +=
        (data_bits_1[RBU66_YEAR1])      + (data_bits_1[RBU66_YEAR2] << 1) +
        (data_bits_1[RBU66_YEAR4] << 2) + (data_bits_1[RBU66_YEAR8] << 3);

      // To comply with struct tm format. !!VALID ONLY TILL 2099!!
      lcl.tm_year += 100;

      // Decode Month of the Year
      lcl.tm_mon  = 10 * data_bits_1[RBU66_MONTH10];
      lcl.tm_mon +=
        (data_bits_1[RBU66_MONTH1])      + (data_bits_1[RBU66_MONTH2] << 1) +
        (data_bits_1[RBU66_MONTH4] << 2) + (data_bits_1[RBU66_MONTH8] << 3);
      lcl.tm_mon -= 1; // To comply with struct tm format

      // Decode Day of the Week
      lcl.tm_wday =
        data_bits_1[RBU66_WDAY1] + (data_bits_1[RBU66_WDAY2] << 1) +
        (data_bits_1[RBU66_WDAY4] << 2);
      //lcl.tm_wday -= 1; // To comply with struct tm format

      // Decode Day of the Month
      lcl.tm_mday  = data_bits_1[RBU66_MDAY10] + (data_bits_1[RBU66_MDAY20] << 1);
      lcl.tm_mday *= 10;
      lcl.tm_mday +=
        (data_bits_1[RBU66_MDAY1])      + (data_bits_1[RBU66_MDAY2] << 1) +
        (data_bits_1[RBU66_MDAY4] << 2) + (data_bits_1[RBU66_MDAY8] << 3);

      // Decode time of day - Hours
      lcl.tm_hour  = data_bits_1[RBU66_HOUR10] + (data_bits_1[RBU66_HOUR20] << 1);
      lcl.tm_hour *= 10;
      lcl.tm_hour +=
        (data_bits_1[RBU66_HOUR1])      + (data_bits_1[RBU66_HOUR2] << 1) +
        (data_bits_1[RBU66_HOUR4] << 2) + (data_bits_1[RBU66_HOUR8] << 3);

      // Compensate for DeltaUT (Moscow Time - UTC) if no parity error
      uint16_t dltut;
      BOOLEAN sign;
      if( time_data->status & RBU66_DUT_ERROR )
      {
        sign  = 1;
        dltut = DUT_MOSCOW; // Default DUT of Moscow (+3hrs)
      }
      else
      {
        sign  = data_bits_1[RBU66_DLTUTS];
        dltut = data_bits_1[RBU66_DLTUT10] * 10;
        dltut +=
          (data_bits_1[RBU66_DLTUT1])      + (data_bits_1[RBU66_DLTUT2] << 1) +
          (data_bits_1[RBU66_DLTUT4] << 2) + (data_bits_1[RBU66_DLTUT8] << 3);
      }
      if( sign )
        lcl.tm_hour += dltut;
      else
        lcl.tm_hour -= dltut;
      if( lcl.tm_hour < 0 ) lcl.tm_hour += 24;
      time_data->DUT = dltut;

      // Decode time of day - Minutes
      lcl.tm_min =
        (data_bits_1[RBU66_MIN10]) + (data_bits_1[RBU66_MIN20] << 1) +
        (data_bits_1[RBU66_MIN40] << 2);
      lcl.tm_min *= 10;
      lcl.tm_min +=
        (data_bits_1[RBU66_MIN1])      + (data_bits_1[RBU66_MIN2] << 1) +
        (data_bits_1[RBU66_MIN4] << 2) + (data_bits_1[RBU66_MIN8] << 3);

      // Convert to local time
      lcl.tm_isdst = -1;
      time_t utc = timegm( &lcl );
      lcl = *localtime( &utc );
      time_data->tim.tm_min   = lcl.tm_min;
      time_data->tim.tm_hour  = lcl.tm_hour;
      time_data->tim.tm_mday  = lcl.tm_mday;
      time_data->tim.tm_wday  = lcl.tm_wday;
      time_data->tim.tm_mon   = lcl.tm_mon;
      time_data->tim.tm_year  = lcl.tm_year;
      time_data->tim.tm_isdst = lcl.tm_isdst;

    } // if( second_idx == 0 )
    else   // Save data bits to decode on second zero
    {
      data_bits_1[time_data->second] = data_bit_1;
      data_bits_2[time_data->second] = data_bit_2;
    }

    return;
} // Decode_RBU66()

//----------------------------------------------------------------------

/* Decode_MSF60()
 *
 * Decodes the time signals from MSF60 station
 */
  static void
Decode_MSF60( time_data_t *time_data )
{
  // MSF60 data bits saved over one minute
  static uint8_t
    data_bits_1[ONE_MINUTE_SECS],
    data_bits_2[ONE_MINUTE_SECS],
    data_bit_1, data_bit_2;

    // Used to save decoded MSF60 time/date data
    struct tm lcl;

    // Enter Seconds
    lcl.tm_sec  = time_data->second;
    time_data->tim.tm_sec = lcl.tm_sec;

    // Decode data bits on Minute (zero second)
    data_bit_1 = time_data->data_bit_1;
    data_bit_2 = time_data->data_bit_2;
    if( time_data->second == 0 )
    {
      // MSF60 data bit parity count
      uint8_t parity, idx;

      // Check for Odd Parity Bits of Year
      parity = 0;
      for( idx = MSF60_YEAR80; idx <= MSF60_YEAR1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[MSF60_P1];
      if( parity & 1 )
        time_data->status &= ~YEAR_ERROR;
      else
        time_data->status |= YEAR_ERROR;

      // Check for Odd Parity Bits of Month/Day of Month
      parity = 0;
      for( idx = MSF60_MONTH10; idx <= MSF60_MDAY1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[MSF60_P2];
      if( parity & 1 )
        time_data->status &= ~MONTH_ERROR;
      else
        time_data->status |= MONTH_ERROR;

      // Check for Odd Parity Bits of Day of Week
      parity = 0;
      for( idx = MSF60_WDAY4; idx <= MSF60_WDAY1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[MSF60_P3];
      if( parity & 1 )
        time_data->status &= ~MDAY_ERROR;
      else
        time_data->status |= MDAY_ERROR;

      // Set the Date error flag
      if( time_data->status & (MONTH_ERROR | MDAY_ERROR) )
        time_data->status |= DATE_ERROR;
      else
        time_data->status &= ~DATE_ERROR;

      // Check for Odd Parity Bits of Time
      parity = 0;
      for( idx = MSF60_HOUR20; idx <= MSF60_MIN1; idx++ )
        parity += data_bits_1[idx];
      parity += data_bits_2[MSF60_P4];
      if( parity & 1 )
        time_data->status &= ~(HOUR_ERROR | MINUTE_ERROR);
      else
        time_data->status |= (HOUR_ERROR | MINUTE_ERROR);

      // Check the summer time flags, should be complementary
      if( data_bits_2[MSF60_ST_WARN] &&
          data_bits_2[MSF60_ST_EFFECT] )
        time_data->status |= MSF60_SMTM_ERROR;
      else
        time_data->status &= ~MSF60_SMTM_ERROR;

      // Set Summer Time Effective flag
      if( data_bits_2[MSF60_ST_EFFECT] )
        time_data->status |= MSF60_SMTM_EFFC;
      else
        time_data->status &= ~MSF60_SMTM_EFFC;

      // Set Summer Time Warn flag
      if( data_bits_2[MSF60_ST_WARN] )
        time_data->status |= MSF60_SMTM_WARN;
      else
        time_data->status &= ~MSF60_SMTM_WARN;

      // Decode DUT1 +ve values
      time_data->DUT1 = 0.0;
      for( idx = 0; idx < MSF60_NDUT1; idx++ )
        if( data_bits_2[MSF60_PDUT1 + idx] )
          time_data->DUT1 = MSF60_WDUT1 * (double)(idx + 1);

      // Decode DUT1 -ve values
      for( idx = 0; idx < MSF60_NDUT1; idx++ )
        if( data_bits_2[MSF60_MDUT1 + idx] )
          time_data->DUT1 = -MSF60_WDUT1 * (double)(idx + 1);

      /* Decode Year in Century (00-99).
       * !!NOTE: Not compatible with struct tm after 2099!! */
      lcl.tm_year =
        (data_bits_1[MSF60_YEAR10])      + (data_bits_1[MSF60_YEAR20] << 1) +
        (data_bits_1[MSF60_YEAR40] << 2) + (data_bits_1[MSF60_YEAR80] << 3);
      lcl.tm_year *= 10;
      lcl.tm_year +=
        (data_bits_1[MSF60_YEAR1])      + (data_bits_1[MSF60_YEAR2] << 1) +
        (data_bits_1[MSF60_YEAR4] << 2) + (data_bits_1[MSF60_YEAR8] << 3);

      // To comply with struct tm format. !!VALID ONLY TILL 2099!!
      lcl.tm_year += 100;

      // Decode Month of the Year
      lcl.tm_mon  = 10 * data_bits_1[MSF60_MONTH10];
      lcl.tm_mon +=
        (data_bits_1[MSF60_MONTH1])      + (data_bits_1[MSF60_MONTH2] << 1) +
        (data_bits_1[MSF60_MONTH4] << 2) + (data_bits_1[MSF60_MONTH8] << 3);
      lcl.tm_mon -= 1; // To comply with struct tm format

      // Decode Day of the Week
      lcl.tm_wday =
        data_bits_1[MSF60_WDAY1] + (data_bits_1[MSF60_WDAY2] << 1) +
        (data_bits_1[MSF60_WDAY4] << 2);
      //lcl.tm_wday -= 1; // To comply with struct tm format

      // Decode Day of the Month
      lcl.tm_mday  = data_bits_1[MSF60_MDAY10] + (data_bits_1[MSF60_MDAY20] << 1);
      lcl.tm_mday *= 10;
      lcl.tm_mday +=
        (data_bits_1[MSF60_MDAY1])      + (data_bits_1[MSF60_MDAY2] << 1) +
        (data_bits_1[MSF60_MDAY4] << 2) + (data_bits_1[MSF60_MDAY8] << 3);

      // Decode time of day - Hours
      lcl.tm_hour  = data_bits_1[MSF60_HOUR10] + (data_bits_1[MSF60_HOUR20] << 1);
      lcl.tm_hour *= 10;
      lcl.tm_hour +=
        (data_bits_1[MSF60_HOUR1])      + (data_bits_1[MSF60_HOUR2] << 1) +
        (data_bits_1[MSF60_HOUR4] << 2) + (data_bits_1[MSF60_HOUR8] << 3);
      lcl.tm_hour -= data_bits_2[MSF60_ST_EFFECT];
      if( lcl.tm_hour < 0 ) lcl.tm_hour += 24;

      // Decode time of day - Minutes
      lcl.tm_min =
        (data_bits_1[MSF60_MIN10]) + (data_bits_1[MSF60_MIN20] << 1) +
        (data_bits_1[MSF60_MIN40] << 2);
      lcl.tm_min *= 10;
      lcl.tm_min +=
        (data_bits_1[MSF60_MIN1])      + (data_bits_1[MSF60_MIN2] << 1) +
        (data_bits_1[MSF60_MIN4] << 2) + (data_bits_1[MSF60_MIN8] << 3);

      // Convert to local time
      lcl.tm_isdst = -1;
      time_t utc = timegm( &lcl );
      lcl = *localtime( &utc );
      time_data->tim.tm_min   = lcl.tm_min;
      time_data->tim.tm_hour  = lcl.tm_hour;
      time_data->tim.tm_mday  = lcl.tm_mday;
      time_data->tim.tm_wday  = lcl.tm_wday;
      time_data->tim.tm_mon   = lcl.tm_mon;
      time_data->tim.tm_year  = lcl.tm_year;
      time_data->tim.tm_isdst = lcl.tm_isdst;

    } // if( second_idx == 0 )
    else   // Save data bits to decode on second zero
    {
      data_bits_1[time_data->second] = data_bit_1;
      data_bits_2[time_data->second] = data_bit_2;
    }

    return;
} // Decode_MSF60()

//----------------------------------------------------------------------

/* Receive_DCF77()
 *
 * Demodulates DCF77 77.5kHz Time Signals
 */
  void
Receive_DCF77( Transceiver_t *TRx )
{
  static int
    minute_marker,        // Marks the DCF77 pulse count where minute marker occurs
    min_pulse_count = 10, // Count of DCF77 pulse periods (10/sec)
    seconds_refer   = 0;  // The offset in seconds to actual minute mark

  static uint32_t
    seconds_count   = 0,  // Count of seconds for finding minute marker
    demod_input_idx = 0,  // Points to demod I/Q buffer's point of in-putting data
    demod_buf_idx   = 0,  // Points to demod I/Q buffer's part in use
    samples_count   = 0,  // Count of samples input to demodulation buffer
    hermes2_buf_idx  = 0,  // Index to hermes2's ADC samples buffer
    ovrsmpl_count   = 0,  // Count of ADC samples to sum into one demod sample
    bandwidth       = 0;  // Saves current bandwidth value

  // Length of samples ring buffer for demodulator
  static uint32_t demod_buf_len = DCF77_PULSE_WIDTH * DCF77_SAMPLES_BUFFERS;

  static double
    trail_pulse[DCF77_TRAIL_PULSES * DCF77_PULSE_TICKS], // Saved trailing pulses
    pulse_minimum = MINIMUM_PRIME,  // Minimum of pulse (bit) detector
    pulse_maximum = 0.0;            // Pulse maximum value

  /* Average value of I and Q samples
   * over one DCF77 pulse  width (0.1sec) */
  static double average_i = 0.0, average_q = 0.0;

  /* Current (leading) value of signal detector
   * and Data bit (0 or 1) detector levels */
  double lead_pulse, data_bit_0, data_bit_1;

  // Demodulator filter data structs for samples buffers
  static filter_data_t demod_filter_i, demod_filter_q;

  // DCF77 decoded time
  static time_data_t time_data;

  /* Allocate I/Q samples (3) ring buffers. We need 3x the
   * DCF77 pulse width to refine the search for pulse edges */
  if( (demod_id == NULL) || (demod_qd == NULL) )
  {
    size_t req = (size_t)demod_buf_len * sizeof(double);
    Mem_Alloc( (void **) &demod_id, req );
    Mem_Alloc( (void **) &demod_qd, req );
    memset( demod_id, 0, req );
    memset( demod_qd, 0, req );
    average_i = 0.0;
    average_q = 0.0;
  }

  // Prepare LP filters on Bandwidth change
  if( bandwidth != TRx->rx_bandwidth )
  {
    double cutoff;

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    cutoff  = (double)TRx->rx_bandwidth;
    cutoff /= DCF77_SAMPLE_RATE * 2.0;

    // Initialize Demodulator I filter
    demod_filter_i.cutoff = cutoff;
    demod_filter_i.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_i.npoles = RX_CW_FILTER_POLES;
    demod_filter_i.type   = FILTER_LOWPASS;
    demod_filter_i.ring_idx = 0;
    demod_filter_i.samples_buffer_len = DCF77_PULSE_WIDTH;
    demod_filter_i.init_filter = True;

    // Initialize Demodulator Q filter
    demod_filter_q.cutoff = cutoff;
    demod_filter_q.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_q.npoles = RX_CW_FILTER_POLES;
    demod_filter_q.type   = FILTER_LOWPASS;
    demod_filter_q.ring_idx = 0;
    demod_filter_q.samples_buffer_len = DCF77_PULSE_WIDTH;
    demod_filter_q.init_filter = True;

    bandwidth = TRx->rx_bandwidth;
    return;
  } // if( bandwidth != TRx->rx_bandwidth )

  // Fill buffer from SDR receiver device
  while( TRx->timedec_active )
  {
    static uint32_t
      pulse_lead_idx  = 0,  // Demodulation buffer index for leading edge of pulses
      pulse_trail_idx = 0;  // Demodulation buffer index for trailing edge of pulses

    while( samples_count < DCF77_PULSE_WIDTH )
    {
      /* Use TRx->sound_decimate samples to
       * produce one carrier amplitude sample */
      demod_id[demod_input_idx] = 0.0;
      demod_qd[demod_input_idx] = 0.0;

      while( ovrsmpl_count < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( hermes2_buf_idx >= hermes2_rc.ddc_buffer_length )
        {
          hermes2_buf_idx = 0;
          return;
        }

        // I/Q values as floats
        demod_id[demod_input_idx] +=
          TRx->roof_filter_i.samples_buffer[hermes2_buf_idx];
        demod_qd[demod_input_idx] +=
          TRx->roof_filter_q.samples_buffer[hermes2_buf_idx];

        ovrsmpl_count++;
        hermes2_buf_idx++;
      } // while( ovrsmpl_count < TRx->sound_decimate )

      // Normalize sound_decimate
      demod_id[demod_input_idx] /= (double)ovrsmpl_count;
      demod_qd[demod_input_idx] /= (double)ovrsmpl_count;

      ovrsmpl_count = 0;
      samples_count++;
      demod_input_idx++;
    } // while( samples_count < DCF77_PULSE_WIDTH )
    samples_count = 0;

    // Filter current I/Q buffers
    demod_filter_i.samples_buffer =
      demod_id + demod_buf_idx * DCF77_PULSE_WIDTH;
    demod_filter_q.samples_buffer =
      demod_qd + demod_buf_idx * DCF77_PULSE_WIDTH;
    DSP_Filter( &demod_filter_i );
    DSP_Filter( &demod_filter_q );

    // Samples buffer index for trailing edge of a pulse
    demod_buf_idx++;
    if( demod_buf_idx >= DCF77_SAMPLES_BUFFERS )
      demod_buf_idx = 0;
    pulse_trail_idx = demod_buf_idx * DCF77_PULSE_WIDTH;

    /* Samples buffer index inputing new samples.
     * Points to next buffer after the last filled */
    demod_input_idx = pulse_trail_idx;

    // Samples buffer index for leading edge of a pulse
    pulse_lead_idx = pulse_trail_idx + DCF77_PULSE_WIDTH;
    if( pulse_lead_idx >= demod_buf_len )
      pulse_lead_idx = 0;

    // Look for DCF77 data bits over 'ticks' (10) in one pulse width
    for( uint16_t ticks_count = 0; ticks_count < DCF77_PULSE_TICKS; ticks_count++ )
    {
      // Add new samples to I/Q average from leading edge index
      for( uint16_t samples_idx = 0; samples_idx < DCF77_TICK_WIDTH; samples_idx++ )
      {
        average_i += demod_id[pulse_lead_idx];
        average_q += demod_qd[pulse_lead_idx];

        /* Subtract old samples from I/Q
         * average from trailing edge index */
        average_i -= demod_id[pulse_trail_idx];
        average_q -= demod_qd[pulse_trail_idx];

        pulse_lead_idx++;
        pulse_trail_idx++;
      }

      // Scale down average I/Q values to usable values
      double i = average_i / AVERAGE_IQ_SCALE;
      double q = average_q / AVERAGE_IQ_SCALE;

      // Current (lead) Signal power is the sum of I/Q averages squared
      lead_pulse = i * i + q * q;

      /* We use three consecutive pulse periods (of 0.1sec) to estimate
       * the value of a data bit (0 or 1). Full carrier is denoted as '+'
       * and reduced carrier as '-'. A zero bit is signaled as +-+ and a
       * 1 bit as +-- so that we need to have the current value of the
       * carrier, over one pulse width (0.1sec), and its value over the
       * past two pulses to determine the data bit value. Below t1 points
       * to the previously saved carrier value and t2 to the one before it */
      uint16_t t1 = ticks_count;
      uint16_t t2 = t1 + DCF77_PULSE_TICKS;

      /* Below, data_bit_0 will reach maximum value when the evaluation of the
       * three pulses is done over the period when the carier level goes +-+
       * The value of the leading and trailing pulse is halved to normalize the
       * summation and make it compatible with the summation below for data_bit_1 */
      data_bit_0 = lead_pulse / 2.0 - trail_pulse[t1] + trail_pulse[t2] / 2.0;

      /* Below, data_bit_0 will reach maximum value when the evaluation of the
       * three pulses is done over the period when the carier level goes +-- */
      data_bit_1 = -lead_pulse / 2.0 - trail_pulse[t1] / 2.0 + trail_pulse[t2];

      /* New lead pulse saved as the trailing pulses
       * for the next calculation of data bit values */
      trail_pulse[t2] = trail_pulse[t1];
      trail_pulse[t1] = lead_pulse;

      // Save the maximum value of detetors
      if( pulse_maximum < data_bit_0 )
      {
        pulse_maximum = data_bit_0;
        time_data.data_bit_1 = 0;
      }
      if( pulse_maximum < data_bit_1 )
      {
        pulse_maximum = data_bit_1;
        time_data.data_bit_1 = 1;
      }

    } // for( ticks_count = 0; ticks_count < DCF77_PULSE_TICKS; ticks_count++ )

    // Every second
    if( (min_pulse_count % DCF77_SECOND_PULSES) == 0 )
    {
      /* Record the minimum value of the edge detector.
       * This is needed to detect the minute marker of
       * DCF77, which has no data pulse */
      if( pulse_minimum > pulse_maximum )
      {
        pulse_minimum = pulse_maximum;
        minute_marker = min_pulse_count / DCF77_SECOND_PULSES;
      }

      // Reset for the next 1 sec interval
      pulse_maximum = 0.0;

      /* Count up seconds to 1 min
       * to find minute marker */
      seconds_count++;

      // Reset for the next 1 min interval
      if( seconds_count >= ONE_MINUTE_SECS )
      {
        /* Subtracted from pulse count to
         * give actual second of minute */
        seconds_refer   = minute_marker + 1;
        seconds_count   = 0;
        min_pulse_count = 0;
        pulse_minimum   = MINIMUM_PRIME;
      }

      // The actual current second of the minute
      time_data.second = min_pulse_count / DCF77_SECOND_PULSES - seconds_refer;

      // Clamp second of minute index
      if( time_data.second < 0 )
        time_data.second += ONE_MINUTE_SECS;
      if( time_data.second >= ONE_MINUTE_SECS )
        time_data.second -= ONE_MINUTE_SECS;

      // Decode and display time and date
      time_data.station = DCF77;
      Decode_DCF77( &time_data );
      gdk_threads_add_idle( Hermes2_Display_Time, &time_data );

    } // if( (min_pulse_count % DCF77_SECOND_PULSES) == 0 )

    // Count up DCF77 pulses (0.1sec) over 1 minute
    min_pulse_count++;

    // Free buffers and exit on new time station
    if( Flag[TIME_NEW_STATION] )
    {
      Flag[TIME_NEW_STATION] = False;
      break;
    }

  } // while( TRx->timedec_active )

  Free_Time_Buffers();
}  // Receive_DCF77()

//----------------------------------------------------------------------

/* Receive_ALL162()
 *
 * Demodulates Allouis 162kHz Time Signals
 */
  void
Receive_ALL162( Transceiver_t *TRx )
{
  static int
    dphi_trail_idx = 0, // Index to saved (trailing) values of dphi
    ramp_i_cnt     = 0, // Counters for determining the direction of dphi
    ramp_q_cnt     = ALL162_RAMP_A;

  static uint32_t
    min_ticks_count = 0, // Count of ALL162 tick periods (100/sec) over 1 min
    minute_tick     = 0, // Index to ticks during a minute, to find minute marker
    minute_marker   = 0, // Marks the ALL162 ticks count at the minute
    demod_input_idx = 0, // Points to demod I/Q buffer's point of in-putting data
    demod_buf_idx   = 0, // Points to demod I/Q buffer's part in use
    sec_pulse_count = 0, // Count of ALL162 data bit periods (10/sec) over 1 sec
    seconds_count   = 0, // Count of seconds in a minute, to find minute marker
    hermes2_buf_idx  = 0, // Index to hermes2's ADC samples buffer
    ovrsmpl_count   = 0, // Count of ADC samples to sum into one demod sample
    samples_count   = 0, // Count of samples input to demodulation buffer
    bandwidth       = 0; // Saves current bandwidth value

  // Length of samples ring buffer for demodulator
  static uint32_t demod_buf_len = ALL162_PULSE_WIDTH * ALL162_SAMPLES_BUFFERS;

  int
    lead_pulse, // Current (leading) value of phase shift modulation
    data_bit_0, // Weighted sum of three phase shift periods to detect a 0 bit
    data_bit_1; // Weighted sum of three phase shift periods to detect a 1 bit

  static int
    pulse_average = 0, // Average level of 0.1 sec pulses
    trail_pulse[ALL162_TRAIL_PULSES * ALL162_PULSE_TICKS], // Saved trailing pulses
    pulse_minimum = MINIMUM_PRIME;  // Minimum of pulse edge (bit) detector

  /* Phase angle of past I/Q samples, sum of phase
   * change over one ALL162 data bit width (0.1sec) */
  static double
    phi0 = 0.0,
    dphi_average_i,
    dphi_average_q,
    dphi_i_trail[ALL162_PULSE_WIDTH],
    dphi_q_trail[ALL162_PULSE_WIDTH];

    /* Phase angle of current I/Q samples,
     * lead pulse, data bit (0 or 1) detector */
  double phi1, dphi;

    // Demodulator filter data structs for samples buffers
  static filter_data_t demod_filter_i, demod_filter_q;

    // ALL162 decoded time
  static time_data_t time_data;

  static int
    sec_marker[ALL162_SECOND_TICKS], // Array of ave modulation transitions marking seconds
    sec_marker_idx     = 0, // Index to current marker in array above
    sec_marker_max_idx = 0; // Index to marker in array with max value

  /* Allocate I/Q samples (3) ring buffers. We need 3x the
   * ALL162 pulse width to refine the search for pulse edges */
  if( (demod_id == NULL) || (demod_qd == NULL) )
  {
    size_t req = (size_t)demod_buf_len * sizeof(double);
    Mem_Alloc( (void **) &demod_id, req );
    Mem_Alloc( (void **) &demod_qd, req );
    memset( demod_id, 0, req );
    memset( demod_qd, 0, req );

    // Also re-initialize some variables
    phi0 = 0.0;
    for( int idx = 0; idx < ALL162_SECOND_TICKS; idx++ )
      sec_marker[idx] = 0;
  }

  // Prepare LP filters on Bandwidth change
  if( bandwidth != TRx->rx_bandwidth )
  {
    double cutoff;

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    cutoff  = (double)TRx->rx_bandwidth;
    cutoff /= ALL162_SAMPLE_RATE * 2.0;

    // Initialize Demodulator I filter
    demod_filter_i.cutoff = cutoff;
    demod_filter_i.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_i.npoles = RX_CW_FILTER_POLES;
    demod_filter_i.type   = FILTER_LOWPASS;
    demod_filter_i.ring_idx = 0;
    demod_filter_i.samples_buffer_len = ALL162_PULSE_WIDTH;
    demod_filter_i.init_filter = True;

    // Initialize Demodulator Q filter
    demod_filter_q.cutoff = cutoff;
    demod_filter_q.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_q.npoles = RX_CW_FILTER_POLES;
    demod_filter_q.type   = FILTER_LOWPASS;
    demod_filter_q.ring_idx = 0;
    demod_filter_q.samples_buffer_len = ALL162_PULSE_WIDTH;
    demod_filter_q.init_filter = True;

    bandwidth = TRx->rx_bandwidth;
    return;
  } // if( bandwidth != TRx->rx_bandwidth )

  // Fill buffer from hermes2 device
  while( TRx->timedec_active )
  {
    // Demodulation buffer index for leading edge of pulses
    static uint32_t pulse_lead_idx  = 0;

    while( samples_count < ALL162_PULSE_WIDTH )
    {
      /* Use TRx->sound_decimate samples to
       * produce one carrier amplitude sample */
      demod_id[demod_input_idx] = 0.0;
      demod_qd[demod_input_idx] = 0.0;

      while( ovrsmpl_count < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( hermes2_buf_idx >= hermes2_rc.ddc_buffer_length )
        {
          hermes2_buf_idx = 0;
          return;
        }

        // I/Q values as floats
        demod_id[demod_input_idx] +=
          TRx->roof_filter_i.samples_buffer[hermes2_buf_idx];
        demod_qd[demod_input_idx] +=
          TRx->roof_filter_q.samples_buffer[hermes2_buf_idx];

        ovrsmpl_count++;
        hermes2_buf_idx++;
      } // while( ovrsmpl_count < TRx->sound_decimate )

      // Normalize sound_decimate
      demod_id[demod_input_idx] /= (double)ovrsmpl_count;
      demod_qd[demod_input_idx] /= (double)ovrsmpl_count;

      ovrsmpl_count = 0;
      samples_count++;
      demod_input_idx++;
    } // while( samples_count < ALL162_PULSE_WIDTH )
    samples_count = 0;

    // Filter current I/Q buffers
    demod_filter_i.samples_buffer =
      demod_id + demod_buf_idx * ALL162_PULSE_WIDTH;
    demod_filter_q.samples_buffer =
      demod_qd + demod_buf_idx * ALL162_PULSE_WIDTH;
    DSP_Filter( &demod_filter_i );
    DSP_Filter( &demod_filter_q );

    /* Samples buffer index inputing new samples.
     * Points to next buffer after the last filled */
    demod_buf_idx++;
    if( demod_buf_idx >= ALL162_SAMPLES_BUFFERS )
      demod_buf_idx = 0;
    demod_input_idx = demod_buf_idx * ALL162_PULSE_WIDTH;

    // Samples buffer index for leading edge of a pulse
    pulse_lead_idx = demod_input_idx + ALL162_PULSE_WIDTH;
    if( pulse_lead_idx >= demod_buf_len )
      pulse_lead_idx = 0;

    // Look for ALL162 phasing over 'ticks' (10) in one data bit width
    for( uint16_t ticks = 0; ticks < ALL162_PULSE_TICKS; ticks++ )
    {
      // Add new dphi to summation from leading edge index
      for( uint16_t samples_idx = 0; samples_idx < ALL162_TICK_WIDTH; samples_idx++ )
      {
        // Phase angle of current sampled IQ signal
        phi1 = atan2( demod_qd[pulse_lead_idx], demod_id[pulse_lead_idx] );

        // Phase difference from previous to current I/Q sample
        dphi = phi1 - phi0;

        // Save current phase as the next previous
        phi0 = phi1;

        // Correct for discontinuity at +-180 deg phase angle
        CLAMP_PI( dphi );

        // Summate phase changes over a period of one pulse (0.1sec)
        dphi_average_i -= dphi_i_trail[dphi_trail_idx]; // FIFO
        if( (ramp_i_cnt > ALL162_RAMP_A) &&
            (ramp_i_cnt <= ALL162_RAMP_B) )
        {
          dphi_average_i += -dphi;
          dphi_i_trail[dphi_trail_idx] = -dphi; // Save in FIFO
        }
        else
        {
          dphi_average_i += dphi;
          dphi_i_trail[dphi_trail_idx] = dphi; // Save in FIFO
        }

        ramp_i_cnt++;
        if( ramp_i_cnt >= ALL162_RAMP_C )
          ramp_i_cnt = 0;

        // Summate phase changes over a period of one pulse (0.1sec)
        dphi_average_q -= dphi_q_trail[dphi_trail_idx]; // FIFO
        if( (ramp_q_cnt > ALL162_RAMP_A) &&
            (ramp_q_cnt <= ALL162_RAMP_B) )
        {
          dphi_average_q += -dphi;
          dphi_q_trail[dphi_trail_idx] = -dphi; // Save in FIFO
        }
        else
        {
          dphi_average_q += dphi;
          dphi_q_trail[dphi_trail_idx] = dphi; // Save in FIFO
        }

        ramp_q_cnt++;
        if( ramp_q_cnt >= ALL162_RAMP_C )
          ramp_q_cnt = 0;

        // Save phase changes for the length of one pulse
        dphi_trail_idx++;
        if( dphi_trail_idx >= ALL162_PULSE_WIDTH )
          dphi_trail_idx = 0;

        pulse_lead_idx++;
      } // for( samples_idx = 0; samples_idx < ALL162_TICK_WIDTH; samples_idx++ )

      /* Phase modulation over the leading
       * ALL162 data bit period (0.1s) */
      lead_pulse = (int)(ALL162_UPSCALE * hypot(dphi_average_i, dphi_average_q) );

      /* We use three consecutive databit periods (of 0.1sec) to estimate
       * the value of a data bit (0 or 1). Plain carrier is denoted as '-'
       * and PSK'd carrier as '+'. A zero bit is signaled as -+- and a
       * 1 bit as -++ so that we need to have the current value of the
       * phase shift, over one databit width (0.1sec), and its value over the
       * past two databits to determine the data bit value. Below t1 points
       * to the previously saved carrier value and t2 to the one before it */
      uint16_t t1 = ticks;
      uint16_t t2 = t1 + ALL162_PULSE_TICKS;

      // Minute zero follows the minimum of pulse average
      if( (min_ticks_count + ALL162_MIN_MARKER_OFFSET) == minute_marker )
        time_data.second = 59;

      /* Time data transmission completes one pulse (10 ticks) after
       * the second marker. In practice it does so after 11 ticks */
      int data_ready_mark = sec_marker_max_idx + ALL162_SEC_MARKER_OFFSET;
      if( data_ready_mark >= ALL162_SECOND_TICKS )
        data_ready_mark -= ALL162_SECOND_TICKS;

      // This happens once a second if all is well (no static QRM)
      if( sec_marker_idx == data_ready_mark )
      {
        /* Look for a 0|1 data bit at the end of the first phase shifting
         * sequence. This appears to happen 11 ticks after the sec marker.
         * Below, data_bit_0 will reach maximum value when the evaluation of the
         * three databits is done over the period when the carier level goes -+- */
        data_bit_0 = -lead_pulse / 2 + trail_pulse[t1] - trail_pulse[t2] / 2;

        /* Below, data_bit_1 will reach maximum value when the evaluation of the
         * three pulses is done over the period when the carier level goes -++
         * The value of the lead and trail databit is halved to normalize the
         * summation and make it compatible with the summation for data_bit_0 */
        data_bit_1 = lead_pulse / 2 + trail_pulse[t1] / 2 - trail_pulse[t2];

        // Determine status of time data bit
        if( data_bit_1 < data_bit_0 )
          time_data.data_bit_1 = 0;
        else
          time_data.data_bit_1 = 1;

        // The actual current second of the minute
        time_data.second++;

        // Clamp second of minute index
        if( time_data.second >= ONE_MINUTE_SECS )
          time_data.second -= ONE_MINUTE_SECS;

        // Decode and display time and date (same as DCF77)
        time_data.station = ALL162;
        Decode_DCF77( &time_data );
        gdk_threads_add_idle( Hermes2_Display_Time, &time_data );

        seconds_count++;
      } // if( sec_marker_idx == data_ready_mark )

      // Sliding window average of second markers
      sec_marker[sec_marker_idx] *= ALL162_SEC_MARKER_MUL;
      sec_marker[sec_marker_idx] += lead_pulse - trail_pulse[t1];
      sec_marker[sec_marker_idx] /= ALL162_SEC_MARKER_DIV;
      sec_marker_idx++;
      if( sec_marker_idx >= ALL162_SECOND_TICKS )
        sec_marker_idx = 0;

      // Sliding window average of pulse levels
      pulse_average *= ALL162_PULSE_AVE_MUL;
      pulse_average += lead_pulse;
      pulse_average /= ALL162_PULSE_AVE_DIV;

      /* Record the minimum value of the pulse detector.
       * This is needed to detect the minute marker of
       * ALL162, which has no phase shifts */
      if( pulse_minimum > pulse_average )
      {
        pulse_minimum = pulse_average;
        minute_tick   = min_ticks_count;
      }

      // Count ticks over one minute
      min_ticks_count++;

      /* New lead pulse saved as the trailing pulses
       * for the next calculation of data bit values */
      trail_pulse[t2] = trail_pulse[t1];
      trail_pulse[t1] = lead_pulse;

    } // for( ticks = 0; ticks < ALL162_PULSE_TICKS;... )

    // Every second
    if( sec_pulse_count >= ALL162_SECOND_PULSES )
    {
      // Record index of maxed seconds marker
      int mark_max = 0;
      for( int idx = 0; idx < ALL162_SECOND_TICKS; idx++ )
      {
        if( mark_max < sec_marker[idx] )
        {
          mark_max = sec_marker[idx];
          sec_marker_max_idx = idx;
        }
      }

      sec_pulse_count = 0;
    } // if( sec_pulse_count == ALL162_SECOND_PULSES )

    // Count up ALL162 pulses (0.1sec) over 1 minute
    sec_pulse_count++;

    // Every minute
    if( seconds_count >= ONE_MINUTE_SECS )
    {
      minute_marker   = minute_tick;
      pulse_minimum   = MINIMUM_PRIME;
      seconds_count   = 0;
      min_ticks_count = 0;
    }

    // Free buffers and exit on new time station
    if( Flag[TIME_NEW_STATION] )
    {
      Flag[TIME_NEW_STATION] = False;
      break;
    }

  } // while( Flag[TIME_RECV_START) )

  Free_Time_Buffers();
}  // Receive_ALL162()

//----------------------------------------------------------------------

/* Receive_RBU66()
 *
 * Demodulates RBU 66.667kHz Time Signals
 */
  void
Receive_RBU66( Transceiver_t *TRx )
{
  static int
    min_pulse_count = 10, // Count of RBU66 pulse periods (10/sec) over 1 min
    seconds_refer   = 0,  // The offset in seconds to actual minute mark
    minute_marker;        // Marks the RBU66 pulse count where minute marker occurs

  static uint32_t
    databit_max,          // Index to data bit array that is maximum
    seconds_count   = 0,  // Count of seconds for finding minute marker
    demod_input_idx = 0,  // Points to demod I/Q buffer's point of in-putting data
    demod_buf_idx   = 0,  // Points to demod I/Q buffer's part in use
    samples_count   = 0,  // Count of samples input to demodulation buffer
    hermes2_buf_idx  = 0,  // Index to hermes2's ADC samples buffer
    ovrsmpl_count   = 0,  // Count of ADC samples to sum into one demod sample
    bandwidth       = 0;  // Saves current bandwidth value

  // Length of samples ring buffer for demodulator
  static uint32_t demod_buf_len = RBU66_PULSE_WIDTH * RBU66_SAMPLES_BUFFERS;

  // Saved sequence of tone values
  static double trail_pulse[RBU66_TRAIL_PULSES * RBU66_PULSE_TICKS];

  // Current tone pulse detector level
  double lead_pulse;

  static double
    detect_max = 0.0,     // Maximum level of tone detectors
    minute_train,         // Tone pulses train marking minute
    minute_train_max = 0, // Max of above marker
    data_bits[RBU66_DATA_BITS]; // Level of data bit combination detectors

  static double
    W0  = 0.0,  // Phase and delta phase of 0 tone
    dW0 = M_2PI * RBU66_TONE_0 / RBU66_SAMPLE_RATE,
    W1  = 0.0,  // Phase and delta phase of 1 tone
    dW1 = M_2PI * RBU66_TONE_1 / RBU66_SAMPLE_RATE;

  // I/Q samples of data tones, shifted to 0 frequency
  static double
    bit0_id, bit0_qd,
    bit1_id, bit1_qd;

  // Demodulator filter data structs for samples buffers
  static filter_data_t demod_filter_i, demod_filter_q;

  // RBU66 decoded time
  static time_data_t time_data;

  /* Allocate I/Q samples (3) ring buffers. We need 3x the
   * RBU66 pulse width to refine the search for pulse edges */
  if( (demod_id == NULL) || (demod_qd == NULL) )
  {
    size_t req = (size_t)demod_buf_len * sizeof(double);
    Mem_Alloc( (void **) &demod_id, req );
    Mem_Alloc( (void **) &demod_qd, req );
    memset( demod_id, 0, req );
    memset( demod_qd, 0, req );
  }

  // Prepare LP filters on Bandwidth change
  if( bandwidth != TRx->rx_bandwidth )
  {
    double cutoff;

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    cutoff  = (double)TRx->rx_bandwidth;
    cutoff /= RBU66_SAMPLE_RATE * 2.0;

    // Initialize Demodulator I filter
    demod_filter_i.cutoff = cutoff;
    demod_filter_i.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_i.npoles = RX_CW_FILTER_POLES;
    demod_filter_i.type   = FILTER_LOWPASS;
    demod_filter_i.ring_idx = 0;
    demod_filter_i.samples_buffer_len = RBU66_PULSE_WIDTH;
    demod_filter_i.init_filter = True;

    // Initialize Demodulator Q filter
    demod_filter_q.cutoff = cutoff;
    demod_filter_q.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_q.npoles = RX_CW_FILTER_POLES;
    demod_filter_q.type   = FILTER_LOWPASS;
    demod_filter_q.ring_idx = 0;
    demod_filter_q.samples_buffer_len = RBU66_PULSE_WIDTH;
    demod_filter_q.init_filter = True;

    bandwidth = TRx->rx_bandwidth;
    return;
  } // if( bandwidth != TRx->rx_bandwidth )

  // Fill buffer from hermes2 device
  while( TRx->timedec_active )
  {
    // Demodulation buffer index for leading edge of pulses
    static uint32_t pulse_lead_idx  = 0;

    while( samples_count < RBU66_PULSE_WIDTH )
    {
      /* Use TRx->sound_decimate samples to
       * produce one carrier amplitude sample */
      demod_id[demod_input_idx] = 0.0;
      demod_qd[demod_input_idx] = 0.0;

      while( ovrsmpl_count < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( hermes2_buf_idx >= hermes2_rc.ddc_buffer_length )
        {
          hermes2_buf_idx = 0;
          return;
        }

        // I/Q values as floats
        demod_id[demod_input_idx] +=
          TRx->roof_filter_i.samples_buffer[hermes2_buf_idx];
        demod_qd[demod_input_idx] +=
          TRx->roof_filter_q.samples_buffer[hermes2_buf_idx];

        ovrsmpl_count++;
        hermes2_buf_idx++;
      } // while( ovrsmpl_count < TRx->sound_decimate )

      // Normalize sound_decimate
      demod_id[demod_input_idx] /= (double)ovrsmpl_count;
      demod_qd[demod_input_idx] /= (double)ovrsmpl_count;

      ovrsmpl_count = 0;
      samples_count++;
      demod_input_idx++;
    } // while( samples_count < RBU66_PULSE_WIDTH )
    samples_count = 0;

    // Filter current I/Q buffers
    demod_filter_i.samples_buffer =
      demod_id + demod_buf_idx * RBU66_PULSE_WIDTH;
    demod_filter_q.samples_buffer =
      demod_qd + demod_buf_idx * RBU66_PULSE_WIDTH;
    DSP_Filter( &demod_filter_i );
    DSP_Filter( &demod_filter_q );

    // Samples buffer index for trailing edge of a pulse
    demod_buf_idx++;
    if( demod_buf_idx >= RBU66_SAMPLES_BUFFERS )
      demod_buf_idx = 0;

    /* Samples buffer index inputing new samples.
     * Points to next buffer after the last filled */
    demod_input_idx = demod_buf_idx * RBU66_PULSE_WIDTH;

    // Samples buffer index for trailing edge of a pulse
    pulse_lead_idx = demod_input_idx + RBU66_PULSE_WIDTH;
    if( pulse_lead_idx >= demod_buf_len )
      pulse_lead_idx = 0;

    // Look for RBU66 tones over 'ticks' (10) in one data bit width
    for( uint16_t ticks_count = 0; ticks_count < RBU66_PULSE_TICKS; ticks_count++ )
    {
      for( uint16_t samples_idx = 0; samples_idx < RBU66_TICK_WIDTH; samples_idx++ )
      {
        /* Average of I/Q samples of data bit 0
         * tone, frequency shifted to 0 Hz */
        bit0_id *= RBU66_AVERAGE_MUL;
        bit0_id +=
          demod_id[pulse_lead_idx] * cos( W0 ) +
          demod_qd[pulse_lead_idx] * sin( W0 );
        bit0_id /= RBU66_AVERAGE_DIV;

        bit0_qd *= RBU66_AVERAGE_MUL;
        bit0_qd +=
          demod_qd[pulse_lead_idx] * cos( W0 ) -
          demod_id[pulse_lead_idx] * sin( W0 );
        bit0_qd /= RBU66_AVERAGE_DIV;

        W0 += dW0;
        CLAMP_2PI( W0 );

        /* Average of I/Q samples of data bit 0
         * tone, frequency shifted to 0 Hz */
        bit1_id *= RBU66_AVERAGE_MUL;
        bit1_id +=
          demod_id[pulse_lead_idx] * cos( W1 ) +
          demod_qd[pulse_lead_idx] * sin( W1 );
        bit1_id /= RBU66_AVERAGE_DIV;

        bit1_qd *= RBU66_AVERAGE_MUL;
        bit1_qd +=
          demod_qd[pulse_lead_idx] * cos( W1 ) -
          demod_id[pulse_lead_idx] * sin( W1 );
        bit1_qd /= RBU66_AVERAGE_DIV;

        W1 += dW1;
        CLAMP_2PI( W1 );

        pulse_lead_idx++;
      } // for( samples_idx = 0; samples_idx < RBU66_TICK_WIDTH; samples_idx++ )

      // Level of 0 and 1 tone pulses
      double tone_0 = bit0_id * bit0_id + bit0_qd * bit0_qd;
      double tone_1 = bit1_id * bit1_id + bit1_qd * bit1_qd;

      /* Lead pulse is diff of 1 and 0 tone
       * levels, so it can be -ve or +ve */
      lead_pulse = tone_1 - tone_0;

      // Indices to sequencial levels of lead pulse
      uint16_t t1 = ticks_count;
      uint16_t t2 = t1 + RBU66_PULSE_TICKS;
      uint16_t t3 = t2 + RBU66_PULSE_TICKS;
      uint16_t t4 = t3 + RBU66_PULSE_TICKS;

      /* Data bit tones train summation is found by adding or subtracting
       * weighted values of the lead pulse and previous (4) lead pulse
       * levels, according to the sequence of tones transmitted by RBU66.
       * This is, starting at the 700 msec mark, 001xx, where xx are the
       * two data bits, bit 1 and bit 2. Only the 001xx train is unique */
      data_bits[0] = -lead_pulse / 4.0 - // 00100
        trail_pulse[t1] / 4.0 + trail_pulse[t2] -
        trail_pulse[t3] / 4.0 - trail_pulse[t4] / 4.0;
      data_bits[1] = -lead_pulse / 3.0 + // 00110
        trail_pulse[t1] / 2.0 + trail_pulse[t2] / 2.0 -
        trail_pulse[t3] / 3.0 - trail_pulse[t4] / 3.0;
      data_bits[2] = lead_pulse / 2.0 - // 00101
        trail_pulse[t1] / 3.0 + trail_pulse[t2] / 2.0 -
        trail_pulse[t3] / 3.0 - trail_pulse[t4] / 3.0;
      data_bits[3] = lead_pulse / 3.0 + // 00111
        trail_pulse[t1] / 3.0 + trail_pulse[t2] / 3.0 -
        trail_pulse[t3] / 2.0 - trail_pulse[t4] / 2.0;

      // Find maximum of minute pulse train 0000011111
      double minute = lead_pulse +
        trail_pulse[t1] + trail_pulse[t2] +
        trail_pulse[t3] + trail_pulse[t4];
      if( minute_train < minute ) minute_train = minute;

      /* New lead pulse saved as the trailing pulses
       * for the next calculation of data bit values */
      trail_pulse[t4] = trail_pulse[t3];
      trail_pulse[t3] = trail_pulse[t2];
      trail_pulse[t2] = trail_pulse[t1];
      trail_pulse[t1] = lead_pulse;

      // Find which summation of tone levels is max
      for( uint32_t idx = 0; idx < RBU66_DATA_BITS; idx++ )
      {
        if( detect_max < data_bits[idx] )
        {
          detect_max = data_bits[idx];
          databit_max = idx;
        }
      }

    } // for( ticks_count = 0; ticks_count < RBU66_PULSE_TICKS;... )

    // Every second
    if( (min_pulse_count % RBU66_SECOND_PULSES) == 0 )
    {
      time_data.data_bit_1 = (uint8_t)(  databit_max & 0x01 );
      time_data.data_bit_2 = (uint8_t)( (databit_max & 0x02) >> 1 );

      /* Record the maximum value of the minute train.
       * This is needed to detect the minute marker of
       * RBU66, which has five consecutive 1 bits. */
      if( minute_train_max < minute_train )
      {
        minute_train_max = minute_train;
        minute_marker = min_pulse_count / RBU66_SECOND_PULSES;
      }

      // Reset for the next 1 sec interval
      detect_max   = 0.0;
      minute_train = 0.0;

      /* Count up seconds to 1 min
       * to find minute marker */
      seconds_count++;

      // Reset for the next 1 min interval
      if( seconds_count >= ONE_MINUTE_SECS )
      {
        /* Subtracted from pulse count to
         * give actual second of minute */
        seconds_refer = minute_marker;
        seconds_count = 0;
        min_pulse_count  = 0;
        minute_train_max = 0;
      }

      // The actual current second of the minute
      time_data.second =
        min_pulse_count / RBU66_SECOND_PULSES - seconds_refer;

      // Clamp second of minute index
      if( time_data.second < 0 )
        time_data.second += ONE_MINUTE_SECS;
      if( time_data.second >= ONE_MINUTE_SECS )
        time_data.second -= ONE_MINUTE_SECS;

      // Decode and display time and date
      time_data.station = RBU66;
      Decode_RBU66( &time_data );
      gdk_threads_add_idle( Hermes2_Display_Time, &time_data );

    } // if( (min_pulse_count % RBU66_SECOND_PULSES) == 0 )

    // Count up RBU66 pulses (0.1sec) over 1 minute
    min_pulse_count++;

    // Free buffers and exit on new time station
    if( Flag[TIME_NEW_STATION] )
    {
      Flag[TIME_NEW_STATION] = False;
      break;
    }

  } // while( Flag[TIME_RECV_START) )

  Free_Time_Buffers();
} // Receive_RBU66()

//----------------------------------------------------------------------

/* Receive_MSF60()
 *
 * Demodulates MSF 60kHz Time Signals
 */
  void
Receive_MSF60( Transceiver_t *TRx )
{
  static int
    min_pulse_count = 10, // Count of MSF60 pulse periods (10/sec) over 1 min
    seconds_refer   = 0,  // The offset in seconds to actual minute mark
    minute_marker;        // Marks the MSF60 pulse count where minute marker occurs

  static uint32_t
    databit_max,          // Index to data bit array that is maximum
    demod_input_idx = 0,  // Points to demod I/Q buffer's point of in-putting data
    demod_buf_idx   = 0,  // Points to demod I/Q buffer's part in use
    seconds_count   = 0,  // Count of seconds for finding minute marker
    samples_count   = 0,  // Count of samples input to demodulation buffer
    hermes2_buf_idx  = 0,  // Index to hermes2's ADC samples buffer
    ovrsmpl_count   = 0,  // Count of ADC samples to sum into one demod sample
    bandwidth       = 0;  // Saves current bandwidth value

  // Length of samples ring buffer for demodulator
  static uint32_t demod_buf_len = MSF60_PULSE_WIDTH * MSF60_SAMPLES_BUFFERS;

  /* Average value of I and Q samples
   * over one MSF60 pulse  width (0.1sec) */
  static double average_i = 0.0, average_q = 0.0;

  // Saved sequence of tone values
  static double trail_pulse[MSF60_TRAIL_PULSES * MSF60_PULSE_TICKS];

  // Current tone pulse detector level
  double lead_pulse;

  static double
    detect_max = 0.0,                 // Maximum level of tone detectors
    data_bits[MSF60_DATA_BITS],       // Saved levels of previous data bits
    minute_train     = MINIMUM_PRIME, // Tone pulses train marking minute
    minute_train_min = MINIMUM_PRIME; // Minimum of above marker

  // Demodulator filter data structs for samples buffers
  static filter_data_t demod_filter_i, demod_filter_q;

  // MSF60 decoded time
  static time_data_t time_data;

  /* Allocate I/Q samples (3) ring buffers. We need 3x the
   * MSF60 pulse width to refine the search for pulse edges */
  if( (demod_id == NULL) || (demod_qd == NULL) )
  {
    size_t req = (size_t)demod_buf_len * sizeof(double);
    Mem_Alloc( (void **) &demod_id, req );
    Mem_Alloc( (void **) &demod_qd, req );
    memset( demod_id, 0, req );
    memset( demod_qd, 0, req );
    average_i = 0.0;
    average_q = 0.0;
  }

  // Prepare LP filters on Bandwidth change
  if( bandwidth != TRx->rx_bandwidth )
  {
    double cutoff;

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    cutoff  = (double)TRx->rx_bandwidth;
    cutoff /= MSF60_SAMPLE_RATE * 2.0;

    // Initialize Demodulator I filter
    demod_filter_i.cutoff = cutoff;
    demod_filter_i.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_i.npoles = RX_CW_FILTER_POLES;
    demod_filter_i.type   = FILTER_LOWPASS;
    demod_filter_i.ring_idx = 0;
    demod_filter_i.samples_buffer_len = MSF60_PULSE_WIDTH;
    demod_filter_i.init_filter = True;

    // Initialize Demodulator Q filter
    demod_filter_q.cutoff = cutoff;
    demod_filter_q.ripple = RX_CW_FILTER_RIPPLE;
    demod_filter_q.npoles = RX_CW_FILTER_POLES;
    demod_filter_q.type   = FILTER_LOWPASS;
    demod_filter_q.ring_idx = 0;
    demod_filter_q.samples_buffer_len = MSF60_PULSE_WIDTH;
    demod_filter_q.init_filter = True;

    bandwidth = TRx->rx_bandwidth;
    return;
  } // if( bandwidth != TRx->rx_bandwidth )

  // Fill buffer from hermes2 device
  while( TRx->timedec_active )
  {
    static uint32_t
      pulse_lead_idx  = 0,  // Demodulation buffer index for leading edge of pulses
      pulse_trail_idx = 0;  // Demodulation buffer index for trailing edge of pulses

    while( samples_count < MSF60_PULSE_WIDTH )
    {
      /* Use TRx->sound_decimate samples to
       * produce one carrier amplitude sample */
      demod_id[demod_input_idx] = 0.0;
      demod_qd[demod_input_idx] = 0.0;

      while( ovrsmpl_count < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( hermes2_buf_idx >= hermes2_rc.ddc_buffer_length )
        {
          hermes2_buf_idx = 0;
          return;
        }

        // I/Q values as floats
        demod_id[demod_input_idx] +=
          TRx->roof_filter_i.samples_buffer[hermes2_buf_idx];
        demod_qd[demod_input_idx] +=
          TRx->roof_filter_q.samples_buffer[hermes2_buf_idx];

        ovrsmpl_count++;
        hermes2_buf_idx++;
      } // while( ovrsmpl_count < TRx->sound_decimate )

      // Normalize sound_decimate
      demod_id[demod_input_idx] /= (double)ovrsmpl_count;
      demod_qd[demod_input_idx] /= (double)ovrsmpl_count;

      ovrsmpl_count = 0;
      samples_count++;
      demod_input_idx++;
    } // while( samples_count < MSF60_PULSE_WIDTH )
    samples_count = 0;

    // Filter current I/Q buffers
    demod_filter_i.samples_buffer =
      demod_id + demod_buf_idx * MSF60_PULSE_WIDTH;
    demod_filter_q.samples_buffer =
      demod_qd + demod_buf_idx * MSF60_PULSE_WIDTH;
    DSP_Filter( &demod_filter_i );
    DSP_Filter( &demod_filter_q );

    // Samples buffer index for trailing edge of a pulse
    demod_buf_idx++;
    if( demod_buf_idx >= MSF60_SAMPLES_BUFFERS )
      demod_buf_idx = 0;
    pulse_trail_idx = demod_buf_idx * MSF60_PULSE_WIDTH;

    /* Samples buffer index inputing new samples.
     * Points to next buffer after the last filled */
    demod_input_idx = demod_buf_idx * MSF60_PULSE_WIDTH;

    // Samples buffer index for leading edge of a pulse
    pulse_lead_idx = pulse_trail_idx + MSF60_PULSE_WIDTH;
    if( pulse_lead_idx >= demod_buf_len )
      pulse_lead_idx = 0;

    // Look for MSF60 tones over 'ticks' (10) in one data bit width
    for( uint32_t ticks_count = 0; ticks_count < MSF60_PULSE_TICKS; ticks_count++ )
    {
      for( uint32_t samples_idx = 0; samples_idx < MSF60_TICK_WIDTH; samples_idx++ )
      {
        average_i += demod_id[pulse_lead_idx];
        average_q += demod_qd[pulse_lead_idx];

        /* Subtract old samples from I/Q
         * average from trailing edge index */
        average_i -= demod_id[pulse_trail_idx];
        average_q -= demod_qd[pulse_trail_idx];

        pulse_lead_idx++;
        pulse_trail_idx++;
      } // for( samples_idx = 0; samples_idx < MSF60_TICK_WIDTH; samples_idx++ )

      // Scale down average I/Q values to usable values
      double i = average_i / AVERAGE_IQ_SCALE;
      double q = average_q / AVERAGE_IQ_SCALE;

      // Signal power is the sum of I/Q averages squared
      lead_pulse = i * i + q * q;

      // Indices to sequencial levels of lead pulse
      uint32_t t1 = ticks_count;
      uint32_t t2 = t1 + MSF60_PULSE_TICKS;
      uint32_t t3 = t2 + MSF60_PULSE_TICKS;
      uint32_t t4 = t3 + MSF60_PULSE_TICKS;

      /* Data bit tones train summation is found by adding or subtracting
       * weighted values of the lead pulse and previous (4) lead pulse
       * levels, according to the sequence of tones transmitted by MSF60.
       * This is, starting at the 800 msec mark, 11xxx, where xxx are the
       * data bit pulses, bit A and bit B. Only the 11xxx train is unique */
      data_bits[0] = lead_pulse / 4.0 +  // 11011
        trail_pulse[t1] / 4.0 - trail_pulse[t2] +
        trail_pulse[t3] / 4.0 + trail_pulse[t4] / 4.0;
      data_bits[1] = lead_pulse / 3.0 -  // 11001
        trail_pulse[t1] / 2.0 - trail_pulse[t2] / 2.0 +
        trail_pulse[t3] / 3.0 + trail_pulse[t4] / 3.0;
      data_bits[2] = -lead_pulse / 2.0 + // 11010
        trail_pulse[t1] / 3.0 - trail_pulse[t2] / 2.0 +
        trail_pulse[t3] / 3.0 + trail_pulse[t4] / 3.0;
      data_bits[3] = -lead_pulse / 3.0 - // 11000
        trail_pulse[t1] / 3.0 - trail_pulse[t2] / 3.0 +
        trail_pulse[t3] / 2.0 + trail_pulse[t4] / 2.0;

      // Find minimum of minute pulse train 00000
      double minute = lead_pulse +
        trail_pulse[t1] + trail_pulse[t2] +
        trail_pulse[t3] + trail_pulse[t4];
      if( minute_train > minute ) minute_train = minute;

      /* New lead pulse saved as the trailing pulses
       * for the next calculation of data bit values */
      trail_pulse[t4] = trail_pulse[t3];
      trail_pulse[t3] = trail_pulse[t2];
      trail_pulse[t2] = trail_pulse[t1];
      trail_pulse[t1] = lead_pulse;

      // Find which summation of tone levels is max
      for( uint32_t idx = 0; idx < 4; idx++ )
      {
        if( detect_max < data_bits[idx] )
        {
          detect_max  = data_bits[idx];
          databit_max = idx;
        }
      }

    } // for( ticks_count = 0; ticks_count < MSF60_PULSE_TICKS;... )

    // Every second
    if( (min_pulse_count % MSF60_SECOND_PULSES) == 0 )
    {
      time_data.data_bit_1 = (uint8_t)(  databit_max & 0x01 );
      time_data.data_bit_2 = (uint8_t)( (databit_max & 0x02) >> 1 );

      /* Record the maximum value of the minute train.
       * This is needed to detect the minute marker of
       * MSF60, which has five consecutive 1 bits. */
      if( minute_train_min > minute_train )
      {
        minute_train_min = minute_train;
        minute_marker = min_pulse_count / MSF60_SECOND_PULSES;
      }

      // Reset for the next 1 sec interval
      detect_max   = 0.0;
      minute_train = MINIMUM_PRIME;

      /* Count up seconds to 1 min
       * to find minute marker */
      seconds_count++;

      // Reset for the next 1 min interval
      if( seconds_count >= ONE_MINUTE_SECS )
      {
        /* Subtracted from pulse count to
         * give actual second of minute */
        seconds_refer = minute_marker;
        seconds_count = 0;
        min_pulse_count  = 0;
        minute_train_min = MINIMUM_PRIME;
      }

      // The actual current second of the minute
      time_data.second =
        min_pulse_count / MSF60_SECOND_PULSES - seconds_refer;

      // Clamp second of minute index
      if( time_data.second < 0 )
        time_data.second += ONE_MINUTE_SECS;
      if( time_data.second >= ONE_MINUTE_SECS )
        time_data.second -= ONE_MINUTE_SECS;

      // Decode and display time and date
      time_data.station = MSF60;
      Decode_MSF60( &time_data );
      gdk_threads_add_idle( Hermes2_Display_Time, &time_data );

    } // if( (min_pulse_count % MSF60_SECOND_PULSES) == 0 )

    // Count up MSF60 pulses (0.1sec) over 1 minute
    min_pulse_count++;

    // Free buffers and exit on new time station
    if( Flag[TIME_NEW_STATION] )
    {
      Flag[TIME_NEW_STATION] = False;
      break;
    }

  } // while( Flag[TIME_RECV_START) )

  Free_Time_Buffers();
} // Receive_MSF60()

//----------------------------------------------------------------------

