/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "interface.h"
#include "time_signal.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../hpsdr/hpsdr_init.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/interface.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

//----------------------------------------------------------------------

static BOOLEAN init = True;

//----------------------------------------------------------------------

/* Hermes2_Display_Time()
 *
 * Displays Time and Data as decoded from (VLF) time stations
 */
  gboolean
Hermes2_Display_Time( gpointer data )
{
  const time_data_t *time_data = (time_data_t *) data;

  // Time Display Window's widgets
  static GtkWidget
    *time_label = NULL,
    *date_label = NULL,
    *hour_parity_image   = NULL,
    *minute_parity_image = NULL,
    *date_parity_image   = NULL,
    *year_parity_image   = NULL,
    *month_parity_image  = NULL,
    *mday_parity_image   = NULL,

    *dcf77_cest_image    = NULL,
    *dcf77_cet_image     = NULL,
    *dcf77_txstat_image  = NULL,
    *dcf77_sumtim_image  = NULL,
    *dcf77_leapsec_image = NULL,

    *rbu66_dut_parity_image = NULL,
    *rbu66_tjd_parity_image = NULL,
    *rbu66_dut_label        = NULL,
    *rbu66_dut1_label       = NULL,
    *rbu66_tjd_label        = NULL,

    *msf60_dut1_label     = NULL,
    *msf60_stwarn_image   = NULL,
    *msf60_steffect_image = NULL;


  // Initialize on first call
  if( init )
  {
    dcf77_cest_image    = Builder_Get_Object( time_window_builder, "dcf77_cest_image" );
    dcf77_cet_image     = Builder_Get_Object( time_window_builder, "dcf77_cet_image" );
    dcf77_sumtim_image  = Builder_Get_Object( time_window_builder, "dcf77_sumtim_image" );
    dcf77_leapsec_image = Builder_Get_Object( time_window_builder, "dcf77_leapsec_image" );
    dcf77_txstat_image  = Builder_Get_Object( time_window_builder, "dcf77_txstat_image" );

    rbu66_dut1_label       = Builder_Get_Object( time_window_builder, "rbu66_dut1_label" );
    rbu66_tjd_parity_image = Builder_Get_Object( time_window_builder, "rbu66_tjd_parity_image" );
    rbu66_tjd_label        = Builder_Get_Object( time_window_builder, "rbu66_tjd_label" );
    rbu66_dut_parity_image = Builder_Get_Object( time_window_builder, "rbu66_dut_parity_image" );
    rbu66_dut_label        = Builder_Get_Object( time_window_builder, "rbu66_dut_label" );

    msf60_dut1_label     = Builder_Get_Object( time_window_builder, "msf60_dut1_label" );
    msf60_stwarn_image   = Builder_Get_Object( time_window_builder, "msf60_stwarn_image" );
    msf60_steffect_image = Builder_Get_Object( time_window_builder, "msf60_steffect_image" );

    time_label          = Builder_Get_Object( time_window_builder, "time_label" );
    date_label          = Builder_Get_Object( time_window_builder, "date_label" );
    hour_parity_image   = Builder_Get_Object( time_window_builder, "hour_parity_image" );
    minute_parity_image = Builder_Get_Object( time_window_builder, "minute_parity_image" );
    date_parity_image   = Builder_Get_Object( time_window_builder, "date_parity_image" );
    year_parity_image   = Builder_Get_Object( time_window_builder, "year_parity_image" );
    month_parity_image  = Builder_Get_Object( time_window_builder, "month_parity_image" );
    mday_parity_image   = Builder_Get_Object( time_window_builder, "mday_parity_image" );
    date_parity_image   = Builder_Get_Object( time_window_builder, "date_parity_image" );

    init = False;
  } // if( init )

  // Format and Display Time
  char time_date[96];
  strftime( time_date, sizeof(time_date),
      _("<span size=\"large\" weight=\"bold\" font=\"mono\">%H:%M:%S</span>"),
      &(time_data->tim) );
  gtk_label_set_markup( GTK_LABEL(time_label), time_date );

  // These checks and displays done on the minute
  if( time_data->tim.tm_sec == 0 )
  {
    // Clear non-common indicators
    gtk_image_set_from_icon_name(
        GTK_IMAGE(year_parity_image),
        "gtk-close", GTK_ICON_SIZE_BUTTON );
    gtk_image_set_from_icon_name(
        GTK_IMAGE(month_parity_image),
        "gtk-close", GTK_ICON_SIZE_BUTTON );
    gtk_image_set_from_icon_name(
        GTK_IMAGE(mday_parity_image),
        "gtk-close", GTK_ICON_SIZE_BUTTON );

    // Minutes Parity Error
    if( time_data->status & MINUTE_ERROR )
      gtk_image_set_from_icon_name(
          GTK_IMAGE(minute_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
    else
      gtk_image_set_from_icon_name(
          GTK_IMAGE(minute_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

    // Hours Parity Error
    if( time_data->status & HOUR_ERROR )
      gtk_image_set_from_icon_name(
          GTK_IMAGE(hour_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
    else
      gtk_image_set_from_icon_name(
          GTK_IMAGE(hour_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

    // General Date Parity Error
    if( time_data->status & DATE_ERROR )
      gtk_image_set_from_icon_name(
          GTK_IMAGE(date_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
    else
      gtk_image_set_from_icon_name(
          GTK_IMAGE(date_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );


    // These are provided by DCF77 and ALL162
    if( (time_data->station == DCF77) ||
        (time_data->station == ALL162) )
    {
      // If Z1 and Z2 (CEST/CET) are same, its an error condition
      if( time_data->status & DCF77_Z1_Z2_ERROR )
      {
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_cet_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_cest_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      }
      else // Display Summer Time status
      {
        // CEST Summer Time
        if( time_data->status & DCF77_Z1_CEST )
          gtk_image_set_from_icon_name(
              GTK_IMAGE(dcf77_cest_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
        else
          gtk_image_set_from_icon_name(
              GTK_IMAGE(dcf77_cest_image), "gtk-close", GTK_ICON_SIZE_BUTTON );

        // CET Regular (winter) Time
        if( time_data->status & DCF77_Z2_CET )
          gtk_image_set_from_icon_name(
              GTK_IMAGE(dcf77_cet_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
        else
          gtk_image_set_from_icon_name(
              GTK_IMAGE(dcf77_cet_image), "gtk-close", GTK_ICON_SIZE_BUTTON );
      } // if( time_data->status & Z1_Z2_ERROR )

      // Transmitter operational status
      if( time_data->status & DCF77_TX_ABNL )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_txstat_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_txstat_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Summer Time Imminent Announcement Flag
      if( time_data->status & DCF77_A1_CEST )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_sumtim_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_sumtim_image), "gtk-close", GTK_ICON_SIZE_BUTTON );

      // Leap Second Imminent Announcement Flag
      if( time_data->status & DCF77_A2_LEAP_SEC )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_leapsec_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(dcf77_leapsec_image), "gtk-close", GTK_ICON_SIZE_BUTTON );
    } // if( (time_data->station == DCF77) || ...

    // These are provided by RBU66
    if( time_data->station == RBU66 )
    {
      // Year parity error
      if( time_data->status & YEAR_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(year_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(year_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Month parity error
      if( time_data->status & MONTH_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(month_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(month_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Day of Month parity error
      if( time_data->status & MDAY_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(mday_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(mday_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // DeltaUT parity error
      if( time_data->status & RBU66_DUT_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(rbu66_dut_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(rbu66_dut_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Tr. Julian Day parity error
      if( time_data->status & RBU66_TJD_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(rbu66_tjd_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(rbu66_tjd_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Display RBU66 DUT1, TJD and DeltaUT data
      snprintf( time_date, sizeof(time_date), _("DUT1: %+05.2f"), time_data->DUT1 );
      gtk_label_set_text( GTK_LABEL(rbu66_dut1_label), time_date );
      snprintf( time_date, sizeof(time_date), _("T. Julian Day: %04d"), time_data->tjd );
      gtk_label_set_text( GTK_LABEL(rbu66_tjd_label), time_date );
      snprintf( time_date, sizeof(time_date), _("DUT: %+02d"), time_data->DUT );
      gtk_label_set_text( GTK_LABEL(rbu66_dut_label), time_date );

    } // if( time_data->station == RBU66 )

    // These are provided by MSF60
    if( time_data->station == MSF60 )
    {
      // Year parity error
      if( time_data->status & YEAR_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(year_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(year_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // Month & Day of Month parity error
      if( time_data->status & MDAY_ERROR )
        gtk_image_set_from_icon_name(
            GTK_IMAGE(mday_parity_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      else
        gtk_image_set_from_icon_name(
            GTK_IMAGE(mday_parity_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );

      // If S.T. Warn and Effective are same, we have an error condition
      if( time_data->status & MSF60_SMTM_ERROR )
      {
        gtk_image_set_from_icon_name(
            GTK_IMAGE(msf60_stwarn_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
        gtk_image_set_from_icon_name(
            GTK_IMAGE(msf60_steffect_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      }
      else
      {
        // Summer Time Imminent warning
        if( time_data->status & MSF60_SMTM_WARN )
          gtk_image_set_from_icon_name(
              GTK_IMAGE(msf60_stwarn_image), "dialog-warning", GTK_ICON_SIZE_BUTTON );
        else
          gtk_image_set_from_icon_name(
              GTK_IMAGE(msf60_stwarn_image), "gtk-close", GTK_ICON_SIZE_BUTTON );

        // Summer Time Effective flag
        if( time_data->status & MSF60_SMTM_EFFC )
          gtk_image_set_from_icon_name(
              GTK_IMAGE(msf60_steffect_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
        else
          gtk_image_set_from_icon_name(
              GTK_IMAGE(msf60_steffect_image), "gtk-close", GTK_ICON_SIZE_BUTTON );
      } // if( time_data->status & MSF_SMTM_ERR )

      // Display MSF60 DUT1
      snprintf( time_date, sizeof(time_date), _("DUT1: %+05.2f"), time_data->DUT1 );
      gtk_label_set_text( GTK_LABEL(msf60_dut1_label), time_date );

    } // if( time_data->station == MSF60 )

    // Format and display date
    strftime( time_date, sizeof(time_date),
        "<span size=\"large\" weight=\"bold\" font=\"mono\">%A %B %d %Y</span>",
        &(time_data->tim) );
    gtk_label_set_markup( GTK_LABEL(date_label), time_date );

  } // if( tim.tm_sec == 0 )

  return( FALSE );
} // Hermes2_Display_Time()

//----------------------------------------------------------------------

/* Close_Time_Display()
 *
 * Frees time display widgets and resets time display
 */
  void
Close_Time_Display( void )
{
  Transceiver[Indices.Time_TRx_Idx]->timedec_active = False;
  g_object_unref( time_window_builder );
  gtk_widget_destroy( time_window );

  // Stop the Radio stream
  Device_Start_Stop( STOP_RADIO_STREAM );
  Flag[TIME_NEW_STATION] = False;
  init = True;

} // Close_Time_Display()

//----------------------------------------------------------------------

