/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "callback_func.h"
#include "common.h"
#include "decode.h"
#include "detect.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/sound.h"

//------------------------------------------------------------------

/* Sstv_Cleanup()
 *
 * Cleans up before quitting
 */
  void
Sstv_Cleanup( void )
{
  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] )
    Save_QSO_Record( &qso_record );

  // Close open files
  Close_File( &(qso_record.qso_record_fp) );
  Close_File( &(qso_record.adif_log_fp) );
  Close_File( &(qso_record.station_log_fp) );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Close some GUI objects
  if( sstv_gui.cairo_surface != NULL )
    cairo_surface_destroy( sstv_gui.cairo_surface );
  g_object_unref( sstv_gui.window_builder );
  g_object_unref( sstv_gui.tx_window_builder );
  sstv_gui.window_builder    = NULL;
  sstv_gui.tx_window_builder = NULL;
  sstv_gui.window            = NULL;
  sstv_gui.cairo_surface     = NULL;
  hermes2_gui.guest_window   = NULL;

  ////g_object_unref( sstv_gui.popup_menu_builder );
  ////sstv_gui.popup_menu_builder = NULL;

  // Clear General flags
  Flag[GUEST_RECORD_QSO]        = False;
  Flag[GUEST_RECEIVE_MODE]      = False;
  Flag[GUEST_TRANSMIT_MODE]     = False;
  Flag[GUEST_TRANSMIT_TAG]      = False;
  Flag[GUEST_TRANSMIT_MACRO]    = False;
  Flag[GUEST_TRANSMIT_KEYBD]    = False;
  Flag[GUEST_KEYBD_BUSY]        = False;
  Flag[GUEST_CAPITALIZE]        = False;
  Flag[GUEST_SAVE_RECORD]       = False;
  Flag[GUEST_QUIT]              = False;
  Flag[HERMES2_SEND_DUC_PACKET] = False;
  Flag[TRANSMIT_MORSE_MESG]     = False;
  Flag[GUEST_RECEIVING]         = False;
  Flag[GUEST_TRANSMITTING]      = False;

  Sstv_Free_Display();

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

} // Sstv_Cleanup()

//------------------------------------------------------------------------

/* New_Sstv_Mode()
 *
 * Carries out actions needed for a new SSTV mode
 */
  void
New_Sstv_Mode( uint8_t mode, BOOLEAN reset )
{
  static uint8_t md = 255;
  if( reset ) md = 255;

  if( mode == DEFAULT_MODE )
  {
    // Set the SSTV Mode label to default
    Set_Label_Text( GTK_LABEL(sstv_gui.sstv_mode_label), DEFAULT_LABEL );
    mode = M1;  // *** DEFAULT_MODE is Martin M1 ***
  }
  else
  {
    // Set the SSTV Mode label to selected
    Set_Label_Text( GTK_LABEL(
          sstv_gui.sstv_mode_label), sstv_mode_params[mode].mode_name );
  }

  // Set the SSTV status Mode index and the SSTV status VIS code
  sstv_status.sstv_mode_num   = sstv_mode_params[mode].vis_mode_num;
  sstv_status.sstv_mode_index = mode;

  // Set a default Line Sync period
  sstv_status.line_sync_period = sstv_mode_params[mode].line_time * SND_DSP_RATE;

  // Signal various functions to initialize at start
  sstv_status.tone_detectors_init = True;
  sstv_status.line_sync_det_init  = True;
  sstv_status.image_decoder_init  = True;
  sstv_status.line_sync_proc_init = True;
  sstv_status.scottie_offset      = 0;

  // Set up the appropriate SSTV Image decoder
  if( mode <= M2 )
    Sstv_Image_Decoder = Decode_Martin_Mode;
  else if( mode <= SDX )
  {
    Sstv_Image_Decoder = Decode_Scottie_Mode;

    /* If the mode is Scottie, the Sync pulse is between the Blue and
     * Red scans so the image index must be corrected accordingly to
     * compensate. This is two Scottie separator times and two Scottie
     * color scan times. Because the search for the SSTV Mode is done
     * with a default mode of M1, we add a Martin sync pulse to the offset */
    double offset =
      2.0 * sstv_mode_params[mode].separator +
      2.0 * sstv_mode_params[mode].scan_time +
      sstv_mode_params[mode].line_sync;

    // Scottie offset in number of freq buffer locations (freq samples)
    sstv_status.scottie_offset = (int32_t)( offset * SND_DSP_RATE );
  }
  else if( mode <= P7 )
    Sstv_Image_Decoder = Decode_Pasokon_Mode;
  else if( mode <= PD290 )
    Sstv_Image_Decoder = Decode_PD_Mode;
  else if( mode == SC2_180 )
    Sstv_Image_Decoder = Decode_SC2180_Mode;

  // No action for the same mode
  if( md == mode ) return;
  md = mode;

  // Set the size request for the image drawingarea
  Resize_Image_Pixbuf(
      (gint)sstv_mode_params[mode].pix_per_line,
      (gint)sstv_mode_params[mode].num_of_lines,
      reset );

} // New_Sstv_Mode()

//------------------------------------------------------------------------

typedef struct _SAVE_PIXBUF
{
  GdkPixbuf *pixbuf;
  char *handle;
  const char *ext;
} save_pixbuf_t;

// Idle add function to save a pixbuf
  static gboolean
_Sstv_Save_Pixbuf( gpointer data )
{
  // Save pixbuf to file
  GError *error = NULL;
  if( gdk_pixbuf_save(
        ((save_pixbuf_t *)data)->pixbuf,
        ((save_pixbuf_t *)data)->handle,
        ((save_pixbuf_t *)data)->ext,
        &error, NULL) )
  {
    // Set the Rx Status Label
    gtk_label_set_text( GTK_LABEL(sstv_gui.rx_status_label),
        "SSTV Image Saved OK" );
  }
  else
  {
    // Set the Rx Status Label
    gtk_label_set_text( GTK_LABEL(sstv_gui.rx_status_label),
        "Failed to Save SSTV Image" );
  }

  return( FALSE );
} // _Sstv_Save_Pixbuf()

/* Sstv_Save_Pixbuf()
 *
 * Prepares a file name with "SSTV Mode + date + time . ext" as format
 */
  void
Sstv_Save_Pixbuf( GdkPixbuf *pixbuf, const char *mode, const char *ext )
{
  // Variables for reading time (UTC)
  time_t tp;
  struct tm ltm;
  char tim[16], mte[32];
  static char handle[FILE_NAME_SIZE];
  static save_pixbuf_t data;

  // Format UTC date-time
  time( &tp );
  ltm = *localtime( &tp );
  strftime( tim, sizeof(tim), "%F_%H%M", &ltm );

  // Prepare file name by adding SSTV Mode name and extension
  Strlcpy( handle, getenv("HOME"), sizeof(handle) );
  Strlcat( handle, "/.hermes2/sstv/received_images/", sizeof(handle) );
  snprintf( mte, sizeof(mte), "%s_%s.%s", mode, tim, ext );
  Strlcat( handle, mte, sizeof(handle) );

  // Save pixbuf to file
  data.pixbuf = pixbuf;
  data.handle = handle;
  data.ext    = ext;
  g_idle_add( _Sstv_Save_Pixbuf, (gpointer)&data );

} // Sstv_Save_Pixbuf()

/*------------------------------------------------------------------------*/

/* Sstv_Read_Config()
 *
 * Loads the sstvrc configuration file
 */
  gboolean
Sstv_Read_Config( gpointer data )
{
  char
    rc_fpath[FILE_NAME_SIZE],  // File path to sstvrc
    line[READ_LINE_BUF_SIZE];  // Buffer for Read_Line

  // Config file pointer
  FILE *sstvrc;


  // Setup file path to sstvrc and create the sstv config file name
  Strlcpy( rc_fpath, sstv_rc.sstv_dir, sizeof(rc_fpath) );
  Strlcat( rc_fpath, "sstv.config",    sizeof(rc_fpath) );

  // Open sstvrc file
  sstvrc = fopen( rc_fpath, "r" );
  if( sstvrc == NULL )
  {
    perror( rc_fpath );
    Error_Dialog(
        _("Failed to open sstvrc file\n"\
          "Quit SSTV and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station Callsign, abort if EOF
  if( Read_Line(line, sstvrc, _("Station Callsign")) != SUCCESS )
    return( FALSE );
  Strlcpy( sstv_rc.callsign, line, sizeof(sstv_rc.callsign) );

  // Read Detector Sensitivity, abort if EOF
  if( Read_Line(line, sstvrc, _("Detector Sensitivity")) != SUCCESS )
    return( FALSE );
  sstv_status.detector_sens = Strtod( line );

  // Read Auto Listen Status, abort if EOF
  if( Read_Line(line, sstvrc, _("Auto Listen Status")) != SUCCESS )
    return( FALSE );
  sstv_status.auto_listen_enable = (BOOLEAN)atoi( line );

  // Read Auto Save Status, abort if EOF
  if( Read_Line(line, sstvrc, _("Auto Save Status")) != SUCCESS )
    return( FALSE );
  sstv_status.auto_save_enable = (BOOLEAN)atoi( line );

  // Read Tx CW ID Status, abort if EOF
  if( Read_Line(line, sstvrc, _("Tx CW ID Status")) != SUCCESS )
    return( FALSE );
  sstv_status.cw_id_enable = (BOOLEAN)atoi( line );

  // Read Tx FSK ID Status, abort if EOF
  if( Read_Line(line, sstvrc, _("Tx FSK ID Status")) != SUCCESS )
    return( FALSE );
  sstv_status.fsk_id_enable = (BOOLEAN)atoi( line );

  // Read SSTV Mode, abort if EOF
  if( Read_Line(line, sstvrc, _("SSTV Mode")) != SUCCESS )
    return( FALSE );
  uint8_t mode = (uint8_t)atoi( line );
  Sstv_Mode_Menu_Item_Activate( NULL, mode );

  // Read SSTV Tx Image Name
  if( Read_Line(line, sstvrc, _("SSTV Tx Image Name")) != SUCCESS )
    return( FALSE );
  Strlcpy( sstv_status.tx_image_name, line, TX_IMAGE_NAME_SIZE );

  // Read Tx Font Face
  if( Read_Line(line, sstvrc, _("SSTV Tx Font Face")) != SUCCESS )
    return( FALSE );
  Strlcpy( sstv_rc.tx_font_face, line, FONT_FACE_SIZE );

  // Read Tx Brush Size
  if( Read_Line(line, sstvrc, _("SSTV Tx Brush Size")) != SUCCESS )
    return( FALSE );
  sstv_rc.tx_brush_size = Atof( line );

  // Read Tx Brush/Font Color and set to color chooser
  uint8_t idx = 0;
  if( Read_Line(line, sstvrc, _("# SSTV Tx Brush/Font Color"))  != SUCCESS )
    return( FALSE );
  sstv_rc.tx_brush_font_color.red = Atof( line );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 4) && (line[idx] != ',') );
  sstv_rc.tx_brush_font_color.green = Atof( &line[idx + 1] );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 4) && (line[idx] != ',') );
  sstv_rc.tx_brush_font_color.blue = Atof( &line[idx + 1] );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 4) && (line[idx] != ',') );
  sstv_rc.tx_brush_font_color.alpha = Atof( &line[idx + 1] );
  GtkWidget *cc = Builder_Get_Object( sstv_gui.tx_window_builder, "sstv_tx_pick_color" );
  gtk_color_chooser_set_rgba( GTK_COLOR_CHOOSER(cc), &sstv_rc.tx_brush_font_color );

  // Set up GUI widgets
  GtkWidget *rng = Builder_Get_Object( sstv_gui.window_builder, "sstv_sensitivity_hscale" );
  gtk_range_set_value( GTK_RANGE(rng), sstv_status.detector_sens * 10.0 );

  GtkWidget *chk = Builder_Get_Object( sstv_gui.window_builder, "sstv_auto_listen_checkbutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(chk), sstv_status.auto_listen_enable );

  chk = Builder_Get_Object( sstv_gui.window_builder, "sstv_autosave_checkbutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(chk), sstv_status.auto_save_enable );

  chk = Builder_Get_Object( sstv_gui.window_builder, "sstv_tx_cw_id_checkbutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(chk), sstv_status.cw_id_enable );

  chk = Builder_Get_Object( sstv_gui.window_builder, "sstv_tx_fsk_id_checkbutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(chk), sstv_status.fsk_id_enable );

  gtk_spin_button_set_value(
      GTK_SPIN_BUTTON(sstv_gui.sstv_tx_brush_size), sstv_rc.tx_brush_size );

  GtkWidget *fc = Builder_Get_Object( sstv_gui.tx_window_builder, "sstv_tx_font_chooser" );
  gtk_font_chooser_set_font( GTK_FONT_CHOOSER(fc), sstv_rc.tx_font_face );

  // Read and set Window positions
  gint x, y;
  idx = 0;

  // Read Main Window position, abort if EOF
  if( Read_Line(line, sstvrc, _("Main Window position") ) != SUCCESS )
    return( FALSE );
  x = atoi( line );
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
  y = atoi( &line[idx + 1] );
  gtk_window_move( GTK_WINDOW(sstv_gui.window), x, y );

  // Read Rx Image Window position, abort if EOF
  if( Read_Line(line, sstvrc, _(" Rx Image Window position") ) != SUCCESS )
    return( FALSE );

  x = atoi( line );
  idx = 0;
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
  y = atoi( &line[idx + 1] );
  gtk_window_move( GTK_WINDOW(sstv_gui.rx_image_window), x, y );

  // Read Tx Image Window position, abort if EOF
  if( Read_Line(line, sstvrc, _(" Tx Image Window position") ) != SUCCESS )
    return( FALSE );
  x = atoi( line );
  idx = 0;
  do idx++;
  while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
  y = atoi( &line[idx + 1] );
  gtk_window_move( GTK_WINDOW(sstv_gui.tx_image_window), x, y );

  // Load default image file and display in Tx Image window
  Sstv_Display_Stock_Image( sstv_status.tx_image_name );

  // Finished reading config data
  Close_File( &sstvrc );

  return( FALSE );
} // End of Sstv_Read_Config()

//------------------------------------------------------------------

/* Sstv_Write_Config()
 *
 * Loads the sstvrc configuration file
 */
  void
Sstv_Write_Config( void )
{
  // File path to sstvrc
  char rc_fpath[FILE_NAME_SIZE];

  // Config file pointer
  FILE *sstvrc;


  // Setup file path to sstvrc and create the sstv config file name
  Strlcpy( rc_fpath, sstv_rc.sstv_dir, sizeof(rc_fpath) );
  Strlcat( rc_fpath, "sstv.config",    sizeof(rc_fpath) );

  // Open sstvrc file
  sstvrc = fopen( rc_fpath, "w" );
  if( sstvrc == NULL )
  {
    perror( rc_fpath );
    Error_Dialog(
        _("Failed to open sstvrc file\n"\
          "Quit SSTV and correct"), HIDE_OK );
    return;
  }

  // Write Station Callsign
  fprintf( sstvrc, _("# Station Callsign. Maximum 23 characters\n%s\n"), sstv_rc.callsign );

  // Write Detector Sensitivity, abort if EOF
  fprintf( sstvrc, _("# Detector Sensitivity\n%3.1f\n"), sstv_status.detector_sens );

  // Write Auto Listen Status, abort if EOF
  fprintf( sstvrc, _("# Auto Listen Status\n%d\n"), sstv_status.auto_listen_enable );

  // Write Auto Save Status, abort if EOF
  fprintf( sstvrc, _("# Auto Save Status\n%d\n"), sstv_status.auto_save_enable );

  // Write Tx CW ID Status, abort if EOF
  fprintf( sstvrc, _("# Tx CW ID Status\n%d\n"), sstv_status.cw_id_enable );

  // Write Tx FSK ID Status, abort if EOF
  fprintf( sstvrc, _("# Tx FSK ID Status\n%d\n"), sstv_status.fsk_id_enable );

  // Write SSTV Mode Index
  if( sstv_status.sstv_mode_auto )
    fprintf( sstvrc, _("# SSTV Mode Index\n%d\n"), AUTO );
  else
    fprintf( sstvrc, _("# SSTV Mode Index\n%d\n"), sstv_status.sstv_mode_index );

  // Write SSTV Tx Image Name
  fprintf( sstvrc, _("# SSTV Tx Image Name\n%s\n"), sstv_status.tx_image_name );

  // Write Tx Font Face
  fprintf( sstvrc, _("# SSTV Tx Font Face\n%s\n"), sstv_rc.tx_font_face );

  // Write Tx Brush Size
  fprintf( sstvrc, _("# SSTV Tx Brush Size\n%5.2f\n"), sstv_rc.tx_brush_size );

  // Write Tx Brush/Font Color
  fprintf( sstvrc, _("# SSTV Tx Brush/Font Color\n%f, %f, %f, %f\n"),
      sstv_rc.tx_brush_font_color.red,
      sstv_rc.tx_brush_font_color.green,
      sstv_rc.tx_brush_font_color.blue,
      sstv_rc.tx_brush_font_color.alpha );

  // Get and save Window positions
  gint x, y;
  gtk_window_get_position( GTK_WINDOW(sstv_gui.window), &x, &y );
  fprintf( sstvrc, _("# Main Window position\n%d,%d\n"), x, y );
  gtk_window_get_position( GTK_WINDOW(sstv_gui.rx_image_window), &x, &y );
  fprintf( sstvrc, _("# Rx Image Window position\n%d,%d\n"), x, y );
  gtk_window_get_position( GTK_WINDOW(sstv_gui.tx_image_window), &x, &y );
  fprintf( sstvrc, _("# Tx Image Window position\n%d,%d\n#"), x, y );

  // Finished Writing config data
  Close_File( &sstvrc );

} // End of Sstv_Write_Config()

//------------------------------------------------------------------

